/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file implements a lookaside list.  It is mainly intended to allow
   efficient memory allocation of small structures in kernels where the
   minimum memory allocation is a page, but is not kernel-specific. */

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_internal.h"
#include "gm_malloc_debug.h"
#include "gm_struct_lock.h"

#define GM_DEBUG_LOOKASIDE 0

#if GM_DEBUG_LOOKASIDE
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

#if !GM_STRUCT_ALIGNMENT
#define GM_STRUCT_ALIGNMENT 8
#endif

struct gm_lookaside_free_entry
{
  struct gm_lookaside_free_entry *next;
};

struct gm_lookaside_segment
{
  struct gm_lookaside_segment *next;
  struct gm_lookaside_segment *prev;
  struct gm_lookaside_free_entry *free_entries;
  struct gm_lookaside *lookaside;
  gm_size_t alloced_entry_cnt;
#if (4*GM_SIZEOF_VOID_P + GM_SIZEOF_SIZE_T) % GM_STRUCT_ALIGNMENT
  char pad[GM_STRUCT_ALIGNMENT
	   - (4 * GM_SIZEOF_VOID_P + GM_SIZEOF_SIZE_T) % GM_STRUCT_ALIGNMENT];
#endif
  char data[1];
};

GM_TOP_LEVEL_ASSERT (sizeof (gm_size_t) == GM_SIZEOF_SIZE_T);
#define GM_LOOKASIDE_SEGMENT_DATA_OFFSET			\
	    GM_OFFSETOF (struct gm_lookaside_segment, data[0])


struct gm_lookaside
{
  unsigned long entry_len;
  unsigned long entry_cnt;
  unsigned long min_entry_cnt;
  /* List of segments.  Segments with free entries are at the front. */
  struct gm_lookaside_segment_list
  {
    struct gm_lookaside_segment *next;
    struct gm_lookaside_segment *prev;
  }
  segment_list;
  struct gm_lookaside_free_entry *delayed_free_entry;
  GM_STRUCT_LOCK_DECL
};

/* count of user lookaside tables */

static unsigned long _gm_lookaside_cnt;

/* An internal lookaside, from which lookaside structures are allocated. */

static struct gm_lookaside _gm_internal_lookaside;

/* Return the number of entries that will fit in a page with a little
   room to spare, or 1, whichever is larger. */

static unsigned long
_gm_lookaside_entries_per_seg (struct gm_lookaside *l)
{
  unsigned long ret;

  gm_assert (l->entry_len);
  gm_assert (GM_PAGE_LEN - GM_STRUCT_ALIGNMENT
	     >  (unsigned long) GM_LOOKASIDE_SEGMENT_DATA_OFFSET);
  ret = GM_STATIC_CAST (unsigned long, (((GM_PAGE_LEN - GM_STRUCT_ALIGNMENT)
	  - GM_LOOKASIDE_SEGMENT_DATA_OFFSET) / l->entry_len));

  return (ret ? ret : 1);
}

static unsigned long
_gm_lookaside_segment_alloc_len (struct gm_lookaside *l)
{
  return GM_STATIC_CAST (unsigned long, (GM_LOOKASIDE_SEGMENT_DATA_OFFSET
	  + l->entry_len * _gm_lookaside_entries_per_seg (l)));
}

/************
 * segment list manipulation
 ************/

/* remove segment from segment list. */

static gm_inline void
_gm_lookaside_segment_list_remove (struct gm_lookaside_segment *seg)
{
  struct gm_lookaside_segment *next, *prev;

  next = seg->next;
  prev = seg->prev;
  next->prev = prev;
  prev->next = next;
}

/* Prepend segment to segment list */

static gm_inline void
_gm_lookaside_segment_list_prepend (struct gm_lookaside *l,
				    struct gm_lookaside_segment *seg)
{
  _gm_lookaside_segment_list_remove (seg);

  seg->next = l->segment_list.next;
  seg->prev = (struct gm_lookaside_segment *) &l->segment_list;
  seg->next->prev = seg;
  l->segment_list.next = seg;
}

/* Append segment to segment list */

static gm_inline void
_gm_lookaside_segment_list_append (struct gm_lookaside *l,
				   struct gm_lookaside_segment *seg)
{
  _gm_lookaside_segment_list_remove (seg);

  seg->next = (struct gm_lookaside_segment *) &l->segment_list;
  seg->prev = l->segment_list.prev;
  l->segment_list.prev = seg;
  seg->prev->next = seg;
}

/************
 * Utilities
 ************/

static gm_status_t
_gm_lookaside_grow (struct gm_lookaside *l)
{
  char *e, *limit;
  struct gm_lookaside_segment *seg;
  unsigned long alloc_len;

  /* allocate another segment to create more entries. */

  alloc_len = _gm_lookaside_segment_alloc_len (l);
  seg = (struct gm_lookaside_segment *) gm_alloc_pages (alloc_len);
  if (!seg)
    return GM_FAILURE;
  gm_assert (GM_PAGE_ALIGNED (seg));

  /* init segment */
  gm_bzero (seg, sizeof (*seg));	/* OK to not zero entries. */
  seg->lookaside = l;
  seg->next = seg->prev = seg;

  /* Add each entry to free list. */

  e = (char *) seg + GM_LOOKASIDE_SEGMENT_DATA_OFFSET;
  limit = (char *) seg + alloc_len;
  do
    {
      struct gm_lookaside_free_entry *free_entry;

      free_entry = (struct gm_lookaside_free_entry *) e;
      free_entry->next = seg->free_entries;
      seg->free_entries = free_entry;
      l->entry_cnt++;
      e += l->entry_len;
    }
  while (e < limit);

  /* Prepend segment to segment list, since it has free entries. */
  _gm_lookaside_segment_list_prepend (l, seg);
  return GM_SUCCESS;
}

/* finalize an initialized lookaside table. */

static gm_status_t
_gm_lookaside_finalize (struct gm_lookaside *l)
{
  struct gm_lookaside_segment *seg, *next;
  unsigned long residual_cnt = 0;

  GM_CALLED ();
  
  /* free all the segment in the segment list, using null termination HACK. */

  l->segment_list.prev->next = 0;
  for (seg = l->segment_list.next; seg; seg = next)
    {
      residual_cnt++;
      gm_assert (GM_PAGE_ALIGNED (seg));
      next = seg->next;
      gm_free_pages (seg, _gm_lookaside_segment_alloc_len (l));
    }

  if (residual_cnt)
    {
      GM_PRINT (GM_DEBUG_LOOKASIDE,
		("Automatically freeing %lu residual buffers in lookaside "
		 "list.\n", residual_cnt));
    }

  GM_RETURN_STATUS (GM_SUCCESS);
}

/* initialize a preallocated lookaside table. */

static gm_status_t
_gm_lookaside_init (struct gm_lookaside *l,
		    gm_size_t entry_len, gm_size_t min_entry_cnt)
{

  GM_CALLED ();
  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("lookaside = 0x%p, len = 0x%lx,cnt =%ld \n", l,
	     entry_len, min_entry_cnt));

  /* Increase entry size to the appropriate granularity */

  entry_len = GM_ROUNDUP (size, entry_len, GM_STRUCT_ALIGNMENT);

  /* Initialize fields. */

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("initialize fields \n"));
  gm_bzero (l, sizeof (*l));
  l->entry_len = GM_STATIC_CAST (unsigned long, entry_len);
  l->min_entry_cnt = GM_STATIC_CAST (unsigned long, min_entry_cnt);
  l->segment_list.next
    = l->segment_list.prev = (struct gm_lookaside_segment *) &l->segment_list;

  /* Preallocate the minimum number of entries. */

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("growing\n"));
  while (l->entry_cnt < l->min_entry_cnt)
    {
      if (_gm_lookaside_grow (l) != GM_SUCCESS)
	goto abort_with_alloced_entries;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 10, ("grew\n"));
  GM_STRUCT_LOCK (*l);
  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_alloced_entries:
  _gm_lookaside_finalize (l);
  GM_RETURN_STATUS (GM_FAILURE);
}

/************
 * Entry points
 ************/

/* Allocate an entry from the lookaside table. */

static void *
_gm_lookaside_alloc (struct gm_lookaside *l)
{
  struct gm_lookaside_segment *seg;
  struct gm_lookaside_free_entry *free_entry;

  GM_CALLED_WITH_ARGS (("%p", l));

  GM_PRINT (GM_PRINT_LEVEL >= 10, (" lookaside buffer = 0x%p\n", l));

  if (l->delayed_free_entry)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 10, (" free buffer = %p\n", l));
      /* If an entry was schedule to be freed, simply use that entry. */

      free_entry = l->delayed_free_entry;
      l->delayed_free_entry = 0;
    }
  else
    {
      /* Arrange for there to be free entries. */

      GM_PRINT (GM_PRINT_LEVEL >= 10, (" Arrange for free entries\n"));
      seg = l->segment_list.next;
      if (seg == (struct gm_lookaside_segment *) &l->segment_list
	  || !seg->free_entries)
	{
	  if (_gm_lookaside_grow (l) != GM_SUCCESS)
	    GM_RETURN_PTR (0);
	  seg = l->segment_list.next;
	  gm_assert (seg != (struct gm_lookaside_segment *) &l->segment_list
		     && seg->free_entries);
	}

      /* Use a free entry. */

      GM_PRINT (GM_PRINT_LEVEL >= 10, (" set free entries \n"));
      free_entry = seg->free_entries;
      seg->free_entries = free_entry->next;
      seg->alloced_entry_cnt++;
      if (!seg->free_entries)
	_gm_lookaside_segment_list_append (l, seg);
    }
  GM_PRINT (GM_PRINT_LEVEL >= 10, (" return from gm_lookaside_alloc \n"));
  GM_RETURN_PTR ((void *) free_entry);
}

/* Allocate an entry from the lookaside table, with debugging */

GM_ENTRY_POINT void *
gm_lookaside_alloc (struct gm_lookaside *l)
{
  void *ret;

  GM_CALLED ();
  GM_STRUCT_UNLOCK (*l);

  ret = _gm_lookaside_alloc (l);
  if (ret)
    {
      ret = GM_MDEBUG_CLIENT_PTR (ret);
      GM_MDEBUG_RECORD_CLIENT_PTR (ret, gm_lookaside_free);
    }

  GM_STRUCT_LOCK (*l);
  GM_RETURN_PTR (ret);
}

/* Allocate and clear an entry */

GM_ENTRY_POINT void *
gm_lookaside_zalloc (struct gm_lookaside *l)
{
  void *ret;

  GM_CALLED ();
  GM_STRUCT_UNLOCK (*l);

  ret = _gm_lookaside_alloc (l);
  if (ret)
    {
      ret = GM_MDEBUG_CLIENT_PTR (ret);
      GM_MDEBUG_RECORD_CLIENT_PTR (ret, gm_lookaside_free);
      gm_bzero (ret, GM_MDEBUG_CLIENT_LEN (l->entry_len));
    }

  GM_STRUCT_LOCK (*l);
  GM_RETURN_PTR (ret);
}

/* Immediately free an entry allocated from the lookaside list. */

static void
_gm_lookaside_free (struct gm_lookaside_free_entry *free_entry)
{
  struct gm_lookaside_segment *seg;
  struct gm_lookaside *l;

  GM_CALLED_WITH_ARGS (("%p", free_entry));

  /* Infer the segment and lookaside for the entry. */

  gm_assert (!GM_PAGE_ALIGNED (free_entry));
  seg = (struct gm_lookaside_segment *) GM_PAGE_ROUND_DOWN (ptr, free_entry);
  gm_assert (seg);
  gm_assert (GM_PAGE_ALIGNED (seg));
  l = seg->lookaside;
  gm_assert (l);

  /* free the entry */

  free_entry->next = seg->free_entries;
  seg->free_entries = free_entry;
  gm_assert (seg->alloced_entry_cnt);
  seg->alloced_entry_cnt--;
  _gm_lookaside_segment_list_prepend (l, seg);

  /* free the segment if all the entries in the segment have been
     freed and doing so would not cause there to be less than the
     minimum number of entries. */

  if (!seg->alloced_entry_cnt)
    {
      unsigned long per_seg;

      per_seg = _gm_lookaside_entries_per_seg (l);
      if (l->entry_cnt - per_seg >= l->min_entry_cnt)
	{
	  _gm_lookaside_segment_list_remove (seg);
	  gm_free_pages (seg, _gm_lookaside_segment_alloc_len (l));
	  l->entry_cnt -= per_seg;
	}
    }

  GM_RETURN_NOTHING ();
}

/* Schedule an allocated entry to be freed, and actually peform any
   scheduled free. */

GM_ENTRY_POINT void
gm_lookaside_free (void *ptr)
{
  struct gm_lookaside_segment *seg;
  struct gm_lookaside_free_entry *free_entry;
  struct gm_lookaside *l;

  GM_CALLED_WITH_ARGS (("%p", ptr));

  GM_MDEBUG_REMOVE_CLIENT_PTR (ptr, gm_lookaside_free);
  ptr = GM_MDEBUG_INTERNAL_PTR (ptr);

  /* infer free entry, segment, and lookaside list for the buffer. */

  free_entry = (struct gm_lookaside_free_entry *) ptr;
  seg = (struct gm_lookaside_segment *) GM_PAGE_ROUND_DOWN (ptr, ptr);
  gm_assert (GM_PAGE_ALIGNED (seg));
  l = seg->lookaside;
  GM_STRUCT_UNLOCK (*l);

  /* perform previously delayed free. */

  if (l->delayed_free_entry)
    _gm_lookaside_free (l->delayed_free_entry);

  /* delay the current free */

  l->delayed_free_entry = free_entry;

  GM_STRUCT_LOCK (*l);
  GM_RETURN_NOTHING ();
}

/* create (alloc and init) a lookaside table. */

GM_ENTRY_POINT struct gm_lookaside *
gm_create_lookaside (gm_size_t entry_len, gm_size_t min_entry_cnt)
{
  struct gm_lookaside *ret;

  GM_CALLED ();

  /* Initialize the lookaside package, if needed. */

  if (_gm_lookaside_cnt == 0)
    {
      gm_always_assert (GM_PAGE_LEN != 0);
      if (_gm_lookaside_init (&_gm_internal_lookaside,
			      GM_MDEBUG_INTERNAL_LEN (sizeof
						      (struct gm_lookaside)),
			      0) != GM_SUCCESS)
	{
	  goto abort_with_nothing;
	}
    }

  /* Allocate the lookaside */

  ret = gm_lookaside_alloc (&_gm_internal_lookaside);
  if (!ret)
    goto abort_with_internal_lookaside;
  _gm_lookaside_cnt++;

  /* Initialize the user's lookaside table. */

  if (_gm_lookaside_init (ret, GM_MDEBUG_INTERNAL_LEN (entry_len),
			  min_entry_cnt) != GM_SUCCESS)
    goto abort_with_lookaside;

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("There are %lx lookasides after create.\n",
				   _gm_lookaside_cnt));
  GM_RETURN_PTR (ret);

abort_with_lookaside:
  gm_lookaside_free (ret);
  _gm_lookaside_cnt--;
abort_with_internal_lookaside:
  if (_gm_lookaside_cnt == 0)
    _gm_lookaside_finalize (&_gm_internal_lookaside);
abort_with_nothing:
  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("GM_create_lookaside FAILED, cnt = 0x%lx\n", _gm_lookaside_cnt));
  GM_RETURN_PTR (0);
}

/* destroy (finalize and free) a lookaside table. */

GM_ENTRY_POINT void
gm_destroy_lookaside (struct gm_lookaside *l)
{
  GM_CALLED ();

  if (!l)
    {
      GM_WARN (("caller tried to free NULL lookaside.\n"));
      GM_RETURN_NOTHING ();
    }

  /* destroy the user's lookaside. */

  GM_STRUCT_UNLOCK (*l);
  _gm_lookaside_finalize (l);
  gm_lookaside_free (l);
  _gm_lookaside_cnt--;

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("There are %lx lookasides after destroy.\n", _gm_lookaside_cnt));

  /* If all lookasides have been destroyed, destroy the internal
     lookaside list. */

  if (_gm_lookaside_cnt == 0)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 10, ("finalizing lookaside support\n"));
      _gm_lookaside_finalize (&_gm_internal_lookaside);
    }

  GM_STRUCT_LOCK (*l);
  GM_RETURN_NOTHING ();
}

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
