/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 2000 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

#include "gm_config.h"
#include "gm.h"
#include "gm_debug.h"

static
unsigned char *
append_blank (unsigned char *where, gm_size_t pos)
{
  if ((pos & 3) == 0)
    *where++ = ' ';
  if ((pos & 7) == 0)
    *where++ = ' ';
  *where++ = '.';
  *where++ = '.';
  *where++ = ' ';
  return where;
}

static
unsigned char *
append_byte (unsigned char *where, gm_size_t pos)
{
  if ((pos & 3) == 0)
    *where++ = ' ';
  if ((pos & 7) == 0)
    *where++ = ' ';
  *where++ = '%';
  *where++ = '0';
  *where++ = '2';
  *where++ = 'x';
  *where++ = ' ';
  return where;
}

/* A line represents 16 bytes aligned on a 16-byte boundary.  Print
   the line containing "ptr".  Do not print past "limit".  Bytes that
   the user did not request to be printed are printed as "..".  Return
   a pointer to the first byte in the next line or limit, whichever is
   less. */

static
void *
hex_dump_line (const unsigned char *ptr, const unsigned char *limit)
{
  unsigned char template[128], *end_template = template;
  gm_u64_t *ptr64;
  gm_u64_t line64[2];
  gm_size_t i, istart, ilimit;
  /* Shortcut to refer to line64 a byte at a time. */
  unsigned char * const line8 = (unsigned char *) line64;

  if (limit <= ptr)
    {
      return (void *) limit;
    }

  /* istart <- line offset of first byte */
  istart = (gm_size_t) ptr & 0xf;
  /* ptr64 <- start of line in-situ */
  ptr64 = (gm_u64_t *) (ptr - istart);
  /* limit <- last byte to print */
  if (limit > (unsigned char *) ptr - istart + 0x10)
    limit = ptr - istart + 0x10;
  /* ilimit <- line offset of limit */
  ilimit = limit - (const unsigned char *) ptr64;
  
  /* copy the bytes on this line into line[].  We do this to avoid any
     partword programmed I/O in case ptr is in I/O space. */

  line64[0] = ptr64[0];
  line64[1] = ptr64[1];

  /* Shift the bytes to be printed to the bottom of line8.  This
     allows the following template-building mechanism to print the
     right bytes. */
  
  for (i = 0; &ptr[i] < limit; i++)
    {
      line8[i] = line8[istart + i];
    }
  
  /* generate a template for the line */

  *end_template++ = '%';
  *end_template++ = 'p';
  *end_template++ = ':';
  for (i = 0; i < istart; i++)
    {
      end_template = append_blank (end_template, i);
    }
  for (i = istart; i < ilimit; i++)
    {
      end_template = append_byte (end_template, i);
    }
  for (i = ilimit; i < 0x10; i++)
    {
      end_template = append_blank (end_template, i);
    }
  *end_template++ = '\n';
  *end_template = 0;
  
  /* print the entire line at once.  This helps insure that odd print
     environments (like kernel printing and MCP printing) do not break
     the line. */

  GM_PRINT (1, ((const char *)template, ptr64,
		(unsigned int) line8[0x0],
		(unsigned int) line8[0x1],
		(unsigned int) line8[0x2],
		(unsigned int) line8[0x3],
		(unsigned int) line8[0x4],
		(unsigned int) line8[0x5],
		(unsigned int) line8[0x6],
		(unsigned int) line8[0x7],
		(unsigned int) line8[0x8],
		(unsigned int) line8[0x9],
		(unsigned int) line8[0xa],
		(unsigned int) line8[0xb],
		(unsigned int) line8[0xc],
		(unsigned int) line8[0xd],
		(unsigned int) line8[0xe],
		(unsigned int) line8[0xf]));
	    
  return (void *) limit;
}

GM_ENTRY_POINT
void
gm_hex_dump (const void *_ptr, gm_size_t len)
{
  const unsigned char *ptr, *limit;

  if (len > 1024)
    {
      GM_PRINT (1, (__GM_FUNCTION__ "(): buffer too long\n"));
      return;
    }
  
  ptr = _ptr;
  limit = ptr + len;
  while (ptr < limit)
    {
      ptr = hex_dump_line (ptr, limit);
    }
}
