/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

/* This module implements a generic hash table.  It uses lookaside
   lists to ensure efficient memory allocation in the kernel. */

/* TODO:

   Make hash table shrink as it gets empty, with hysteresis to prevent
   rehashing thrashing.

   Support GM_HASH_SMOOTH flag, for non-bursty hash table resizing. */

#include "gm_call_trace.h"
#include "gm_compiler.h"
#include "gm_debug.h"
#include "gm_internal.h"

#define GM_DEBUG_HASH 0

/************************************************************************
 * Structs
 ************************************************************************/

typedef struct gm_hash_entry
{
  struct gm_hash_entry *next;
  void *key;
  void *data;
  struct gm_hash_entry *next_for_key;
}
gm_hash_entry_t;

typedef struct gm_hash_segment
{
  struct gm_hash_segment *next;
  unsigned long bin_cnt;
  gm_hash_entry_t **bin;
}
gm_hash_segment_t;

typedef struct gm_hash
{
  unsigned long (*hash) (struct gm_hash * h, void *key);
  long (*compare) (struct gm_hash * h, void *key1, void *key2);
    gm_status_t (*store_key) (struct gm_hash * h, void *key, void **where);
  void (*copy_key) (struct gm_hash * h, void *key, void **where);
  void (*unstore_key) (void **where);
  void *(*reference_key) (void **where);
    gm_status_t (*store_data) (struct gm_hash * h, void *key, void **where);
  void (*unstore_data) (void **where);
  void *(*reference_data) (void **where);
  long (*user_compare) (void *key1, void *key2);
  unsigned long (*user_hash) (void *key1);
  struct gm_hash_segment *first_segment;
  struct gm_lookaside *entry_lookaside;
  struct gm_lookaside *key_lookaside;
  struct gm_lookaside *data_lookaside;
  unsigned long key_len;
  unsigned long data_len;
  unsigned long entry_cnt;
  unsigned long bin_cnt;
  int flags;
}
gm_hash_t;

/************************************************************************
 * Globals
 ************************************************************************/

static struct gm_lookaside *gm_hash_segment_lookaside = 0;
static struct gm_lookaside *gm_hash_hash_lookaside = 0;
#ifdef SAB
#endif
static unsigned long gm_hash_cnt = 0;

/************************************************************************
 * internal functions
 ************************************************************************/

/* Finalize a hash table (i.e.: free its resources) */
static gm_status_t _gm_hash_grow (gm_hash_t * hash);
gm_status_t _gm_lookaside_finalize (struct gm_lookaside *l);

/************
 * Handling externally managed keys
 ************/

static unsigned long
_gm_hash_hash_ptr_key (struct gm_hash *h, void *key)
{
  return h->user_hash (key);
}

static long
_gm_hash_compare_ptr_keys (struct gm_hash *h, void *key1, void *key2)
{
  return h->user_compare (key1, key2);
}

static gm_status_t
_gm_hash_store_ptr_key (struct gm_hash *h, void *key, void **where)
{
  GM_PARAMETER_MAY_BE_UNUSED (h);

  *where = key;
  return GM_SUCCESS;
}

#define _gm_hash_store_ptr_data _gm_hash_store_ptr_key

static void
_gm_hash_unstore_ptr (void **where)
{
  GM_PARAMETER_MAY_BE_UNUSED (where);

  return;
}

static void *
_gm_hash_reference_ptr (void **ptr)
{
  return *ptr;
}

static void
_gm_hash_copy_ptr_key (struct gm_hash *h, void *key, void **where)
{
  GM_PARAMETER_MAY_BE_UNUSED (h);
  GM_PARAMETER_MAY_BE_UNUSED (key);

  *where = key;
}

/************
 * Handling keys no larger than a void*.
 ************/

static unsigned long
_gm_hash_hash_small_key (struct gm_hash *h, void *key)
{
  return h->user_hash ((void *) key);
}

static long
_gm_hash_compare_small_keys (struct gm_hash *h, void *key1, void *key2)
{
  return h->user_compare ((void *) key1, (void *) key2);
}

static gm_status_t
_gm_hash_store_small_key (struct gm_hash *h, void *key, void **where)
{
  *where = 0;
  gm_bcopy (key, where, h->key_len);
  return GM_SUCCESS;
}

static gm_status_t
_gm_hash_store_small_data (struct gm_hash *h, void *data, void **where)
{
  *where = 0;
  gm_bcopy (data, where, h->data_len);
  return GM_SUCCESS;
}

#define _gm_hash_unstore_small _gm_hash_unstore_ptr

static void *
_gm_hash_reference_small (void **ptr)
{
  return (void *) ptr;
}

static void
_gm_hash_copy_small_key (struct gm_hash *h, void *key, void **where)
{
  *where = 0;
  gm_bcopy (key, where, h->key_len);
}

/************
 * Handing keys larger than a void*
 ************/

#define _gm_hash_hash_large_key  _gm_hash_hash_ptr_key

#define _gm_hash_compare_large_keys _gm_hash_compare_ptr_keys

static gm_status_t
_gm_hash_store_large_key (struct gm_hash *h, void *key, void **where)
{
  *where = gm_lookaside_alloc (h->entry_lookaside);
  if (!*where)
    return GM_FAILURE;
  gm_bcopy (key, *where, h->key_len);
  return GM_SUCCESS;
}

static gm_status_t
_gm_hash_store_large_data (struct gm_hash *h, void *data, void **where)
{
  *where = gm_lookaside_alloc (h->data_lookaside);
  if (!*where)
    return GM_FAILURE;
  gm_bcopy (data, *where, h->data_len);
  return GM_SUCCESS;
}

static void
_gm_hash_unstore_large (void **where)
{
  gm_lookaside_free (*where);
}

#define _gm_hash_reference_large _gm_hash_reference_ptr

static void
_gm_hash_copy_large_key (struct gm_hash *h, void *what, void **where)
{
  gm_assert (*where);
  gm_bcopy (what, *where, h->key_len);
}

/************
 * misc internal functions
 ************/

static void
_gm_hash_finalize (struct gm_hash *h)
{
  struct gm_hash_segment *seg, *next;

  GM_CALLED ();

  /* Free all of the hash table segments. */

  for (seg = h->first_segment; seg; seg = next)
    {
      next = seg->next;
      gm_always_assert (seg->bin);
      gm_free (seg->bin);
      gm_lookaside_free (seg);
    }

  /* Free all of the hash table keys and entries */
  if (h->key_lookaside)
    gm_destroy_lookaside (h->key_lookaside);
  if (h->data_lookaside)
    gm_destroy_lookaside (h->data_lookaside);
  gm_destroy_lookaside (h->entry_lookaside);

  GM_RETURN_NOTHING ();
}

/* Initialize a hash table. */

static gm_status_t
_gm_hash_init (struct gm_hash *h,
	       long (*gm_user_compare) (void *key1, void *key2),
	       unsigned long (*gm_user_hash) (void *key1),
	       gm_size_t key_len,
	       gm_size_t data_len, gm_size_t min_cnt, int flags)
{
  gm_bzero (h, sizeof (*h));
  h->user_compare = gm_user_compare;
  h->user_hash = gm_user_hash;
  h->key_len = GM_STATIC_CAST (unsigned long, key_len);
  h->data_len = GM_STATIC_CAST (unsigned long, data_len);
  h->flags = flags;

  /* Precompute internal function to be used to manipulate user keys. */

  if (key_len == 0)
    {
      h->hash = _gm_hash_hash_ptr_key;
      h->compare = _gm_hash_compare_ptr_keys;
      h->store_key = _gm_hash_store_ptr_key;
      h->unstore_key = _gm_hash_unstore_ptr;
      h->reference_key = _gm_hash_reference_ptr;
      h->copy_key = _gm_hash_copy_ptr_key;
    }
  else if (key_len <= sizeof (void *))
    {
      h->hash = _gm_hash_hash_small_key;
      h->compare = _gm_hash_compare_small_keys;
      h->store_key = _gm_hash_store_small_key;
      h->unstore_key = _gm_hash_unstore_small;
      h->reference_key = _gm_hash_reference_small;
      h->copy_key = _gm_hash_copy_small_key;
    }
  else
    {
      h->hash = _gm_hash_hash_large_key;
      h->compare = _gm_hash_compare_large_keys;
      h->store_key = _gm_hash_store_large_key;
      h->unstore_key = _gm_hash_unstore_large;
      h->reference_key = _gm_hash_reference_large;
      h->copy_key = _gm_hash_copy_large_key;
      h->key_lookaside = gm_create_lookaside (key_len, min_cnt);
      if (!h->key_lookaside)
	{
	  GM_NOTE (("Could not initialize hash key lookaside list.\n"));
	  goto abort_with_nothing;
	}
    }

  /* Precompute internal functions to use to manipulate user data. */

  if (data_len == 0)
    {
      h->store_data = _gm_hash_store_ptr_data;
      h->unstore_data = _gm_hash_unstore_ptr;
      h->reference_data = _gm_hash_reference_ptr;
    }
  else if (data_len <= sizeof (void *))
    {
      h->store_data = _gm_hash_store_small_data;
      h->unstore_data = _gm_hash_unstore_small;
      h->reference_data = _gm_hash_reference_small;
    }
  else
    {
      h->store_data = _gm_hash_store_large_data;
      h->unstore_data = _gm_hash_unstore_large;
      h->reference_data = _gm_hash_reference_large;
      h->data_lookaside = gm_create_lookaside (data_len, min_cnt);
      if (!h->data_lookaside)
	{
	  GM_NOTE (("Could not initialize hash data lookaside table.\n"));
	  goto abort_with_key_lookaside;
	}
    }

  /* Allocate storage for minimum number of entries. */

  h->entry_lookaside =
    gm_create_lookaside (sizeof (gm_hash_entry_t), min_cnt);
  if (!h->entry_lookaside)
    {
      GM_NOTE (("Could not initialize hash entry lookaside list.\n"));
      goto abort_with_data_lookaside;
    }

  /* ensure there are some bins and enough bins for MIN_CNT */

  do
    {
      if (_gm_hash_grow (h) != GM_SUCCESS)
	goto abort_with_growth;
    }
  while (h->bin_cnt < min_cnt);

  return GM_SUCCESS;

abort_with_growth:
  _gm_hash_finalize (h);	/* shortcut */
  return GM_FAILURE;
abort_with_data_lookaside:
  if (h->data_lookaside)
    gm_destroy_lookaside (h->data_lookaside);
abort_with_key_lookaside:
  if (h->key_lookaside)
    gm_destroy_lookaside (h->key_lookaside);
abort_with_nothing:
  return GM_FAILURE;
}

/* Find linked list of entries (a hash bin) that might contain the
   key.  This has O(1) average performance, and O(log(N)) worst-case
   performance, since it scans from biggest to smallest segment. */

static gm_hash_entry_t **
_gm_hash_where (gm_hash_t * hash, void *key)
{
  unsigned long bin_num;
  gm_hash_segment_t *seg;
  gm_hash_entry_t **where = 0;

  GM_CALLED_WITH_ARGS (("%p,%p", hash, key));

  gm_assert (GM_POWER_OF_TWO (hash->bin_cnt));
  bin_num = hash->hash (hash, key) & (hash->bin_cnt - 1);

  seg = hash->first_segment;
  gm_assert (seg);

  /* find the bin for the key. */

  while (seg)
    {
      if (bin_num < seg->bin_cnt)
	{
	  where = &seg->bin[bin_num];
	  break;
	}
      bin_num -= seg->bin_cnt;
      seg = seg->next;
    }
  gm_assert (where);

  /* Scan for a matching entry. */

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("searching\n"));
  while (*where)
    {
      if (!hash->compare (hash, hash->reference_key (&(*where)->key), key))
	GM_RETURN_PTR (where);
      where = &(*where)->next;
      GM_PRINT (GM_PRINT_LEVEL >= 10, (".\n"));
    }
  /* There is no matching entry in the bin, so return a pointer to the
     the end of the bin chain.  This makes *where be 0, indicating
     that no match was found. */

  GM_RETURN_PTR (where);
}

static gm_status_t
_gm_hash_grow (gm_hash_t * hash)
{
  struct gm_hash_segment *seg;

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("Growing hash table.\n"));

  /* alloc new segment */

  seg = gm_lookaside_alloc (gm_hash_segment_lookaside);
  if (!seg)
    goto abort_with_nothing;

  /* alloc new bins for segment */

  seg->bin_cnt = (hash->bin_cnt
		  ? hash->bin_cnt
		  : GM_PAGE_LEN / sizeof (gm_hash_entry_t **));
  seg->bin = gm_malloc (seg->bin_cnt * sizeof (gm_hash_entry_t **));
  if (!seg->bin)
    goto abort_with_new_seg;
  gm_bzero (seg->bin, seg->bin_cnt * sizeof (gm_hash_entry_t **));

  /* Commit to using new segment */

  hash->bin_cnt += seg->bin_cnt;
  seg->next = hash->first_segment;
  hash->first_segment = seg;

  /* Rehash all the entries in the table.  (There is none in the first
     (new) segment.) */

  for (seg = seg->next; seg; seg = seg->next)
    {
      unsigned long bin;

      for (bin = 0; bin < seg->bin_cnt; bin++)
	{
	  gm_hash_entry_t *e, *next, **where;

	  /* remove all the entries in this bin */

	  e = seg->bin[bin];
	  seg->bin[bin] = 0;

	  /* reinsert all the entries removed from the bin.  Note that we
	     leave the chains of entries for the same key intact. */

	  while (e)
	    {
	      where = _gm_hash_where (hash, hash->reference_key (&e->key));
	      gm_assert (!*where);
	      *where = e;
	      next = e->next;
	      e->next = 0;
	      e = next;
	    }
	}
    }

  return GM_SUCCESS;

abort_with_new_seg:
  gm_lookaside_free (seg);
abort_with_nothing:
  return GM_FAILURE;
}

/************************************************************************
 * entry points
 ************************************************************************/

/* create (alloc and init) a hash table. */

GM_ENTRY_POINT struct gm_hash *
gm_create_hash (long (*gm_user_compare) (void *key1, void *key2),
		unsigned long (*gm_user_hash) (void *key),
		gm_size_t key_len,
		gm_size_t data_len, gm_size_t min_cnt, int flags)
{
  struct gm_hash *ret;

  /* Initialize this package, if needed. */

  GM_CALLED ();

  if (gm_hash_cnt == 0)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 10,
		("Call  gm_hash_hash_lookaside, cnt = 0x%lx\n", gm_hash_cnt));

      gm_hash_hash_lookaside
	= gm_create_lookaside (sizeof (struct gm_hash), 0);
      GM_PRINT (GM_PRINT_LEVEL >= 10,
		("AFTER Call, gm_hash_hash_lookaside = %p\n",
		 gm_hash_hash_lookaside));
      if (!gm_hash_hash_lookaside)
	goto abort_with_nothing;

      gm_hash_segment_lookaside
	= gm_create_lookaside (sizeof (struct gm_hash_segment), 0);
      if (!gm_hash_segment_lookaside)
	goto abort_with_hash_lookaside;
    }

  /* allocate storage for the user's hash table. */

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("Call  gm_lookaside_alloc, gm_hash_hash_lookaside = %p\n",
	     gm_hash_hash_lookaside));
  ret = gm_lookaside_alloc (gm_hash_hash_lookaside);
  if (!ret)
    goto abort_with_initialization;
  gm_hash_cnt++;

  /* initialize the hash table. */

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("Call  gm_hash_init, cnt = 0x%lx\n",
				   gm_hash_cnt));
  if (_gm_hash_init (ret, gm_user_compare, gm_user_hash, key_len, data_len,
		     min_cnt, flags) != GM_SUCCESS)
    {
      goto abort_with_alloc;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("There are %lx hash tables after gm_create_hash.\n",
	     gm_hash_cnt));
  GM_RETURN_PTR (ret);

abort_with_alloc:
  gm_lookaside_free (ret);
  gm_hash_cnt--;
abort_with_initialization:
  if (gm_hash_cnt == 0)
    {
      gm_destroy_lookaside (gm_hash_segment_lookaside);
      gm_hash_segment_lookaside = 0;
    abort_with_hash_lookaside:
      gm_destroy_lookaside (gm_hash_hash_lookaside);
      gm_hash_hash_lookaside = 0;
    }
abort_with_nothing:
  GM_PRINT (GM_PRINT_LEVEL >= 10, ("HASH_DREATE FAILURE\n"));
  GM_RETURN_PTR ((void *) 0);
}

/* Destroy (finalize and free) a hash table. */

GM_ENTRY_POINT void
gm_destroy_hash (struct gm_hash *hash)
{
  GM_CALLED ();

  if (!hash)
    {
      GM_WARN (("caller tried to destroy NULL hash\n"));
      GM_RETURN_NOTHING ();
    }

  _gm_hash_finalize (hash);
  gm_lookaside_free (hash);
  gm_hash_cnt--;
  if (gm_hash_cnt == 0)
    {
      GM_PRINT (GM_DEBUG_HASH, ("Finalizing hash table support.\n"));
      gm_destroy_lookaside (gm_hash_segment_lookaside);
      gm_hash_segment_lookaside = 0;
      gm_destroy_lookaside (gm_hash_hash_lookaside);
      gm_hash_hash_lookaside = 0;
    }
  GM_PRINT (GM_DEBUG_HASH,
	    ("There are %lx hash tables after gm_destroy_hash.\n",
	     gm_hash_cnt));

  GM_RETURN_NOTHING ();
}

/* Remove an entry from the hash table, returning a pointer to the
   associated data.  If the data was copied into the hash table, then
   it is only guaranteed valid until the next call to a gm_hash
   function operating on the same hash table. */

GM_ENTRY_POINT void *
gm_hash_remove (gm_hash_t * hash, void *key)
{
  gm_hash_entry_t **e, *entry;
  void *data;

  e = _gm_hash_where (hash, key);
  if (!*e)
    return 0;

  /* record pointer to entry we wish to remove and splice it out. */

  if ((*e)->next_for_key)
    {
      entry = (*e)->next_for_key;
      e = &(*e)->next_for_key;
      *e = (*e)->next_for_key;
    }
  else
    {
      entry = *e;
      *e = (*e)->next;
    }

  /* Free the entry and its associated resources. */

  data = hash->reference_data (&entry->data);
  hash->unstore_data (&entry->data);
  hash->unstore_key (&entry->key);
  gm_lookaside_free (entry);

  hash->entry_cnt--;
  return data;
}

/* Find the data for a particular key */

GM_ENTRY_POINT void *
gm_hash_find (gm_hash_t * hash, void *key)
{
  gm_hash_entry_t **e;

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("gm_hash_find(%p, %p)\n", hash, key));

  e = _gm_hash_where (hash, key);
  if (!*e)
    return 0;
  return hash->reference_data (&(*e)->data);
}

/* Replace a key in the table, updating all entries using that key. */

GM_ENTRY_POINT void
gm_hash_rekey (gm_hash_t * hash, void *old_key, void *new_key)
{
  gm_hash_entry_t **e, *entry;

  for (e = _gm_hash_where (hash, old_key);
       *e; e = _gm_hash_where (hash, old_key))
    {
      entry = *e;		/* Save the entry */
      *e = entry->next;		/* splice out the entry */

      /* Copy the new key over the old. */
      hash->copy_key (hash, new_key, &entry->key);

      /* Insert the entry in its new location. */
      e = _gm_hash_where (hash, new_key);
      entry->next = *e;
      *e = entry;
    }
}

/* Insert an entry in the hash table. */

GM_ENTRY_POINT gm_status_t
gm_hash_insert (gm_hash_t * hash, void *key, void *data)
{
  gm_hash_entry_t *entry;
  gm_status_t status;
  gm_hash_entry_t **where_to_store;

  /* Grow table if needed. */

  if (hash->bin_cnt <= hash->entry_cnt)
    {
      if (_gm_hash_grow (hash) != GM_SUCCESS)
	return GM_FAILURE;
    }

  /* allocate storage for the entry */

  entry = gm_lookaside_alloc (hash->entry_lookaside);
  if (!entry)
    {
      status = GM_OUT_OF_MEMORY;
      goto abort_with_nothing;
    }

  /* store the key in the entry */

  status = hash->store_key (hash, key, &entry->key);
  if (status != GM_SUCCESS)
    goto abort_with_entry;

  status = hash->store_data (hash, data, &entry->data);
  if (status != GM_SUCCESS)
    goto abort_with_stored_key;

  where_to_store = _gm_hash_where (hash, hash->reference_key (&entry->key));
  if (*where_to_store)
    {
      entry->next = 0;
      entry->next_for_key = (*where_to_store)->next_for_key;
      (*where_to_store)->next_for_key = entry;
    }
  else
    {
      entry->next = *where_to_store;
      entry->next_for_key = 0;
      *where_to_store = entry;
    }

  hash->entry_cnt++;
  return GM_SUCCESS;

abort_with_stored_key:
  hash->unstore_key (&entry->key);
abort_with_entry:
  gm_lookaside_free (entry);
abort_with_nothing:
  return GM_FAILURE;
}

/************************************************************************
 * common comparison and hash functions
 ************************************************************************/

/************
 * string utility functions
 ************/

GM_ENTRY_POINT long
gm_hash_compare_strings (void *key1, void *key2)
{
  return gm_strcmp ((char *) key1, (char *) key2);
}

GM_ENTRY_POINT unsigned long
gm_hash_hash_string (void *key)
{
  return gm_crc_str ((char *) key);
}

/************
 * long integer utility functions
 ************/

GM_ENTRY_POINT long
gm_hash_compare_longs (void *key1, void *key2)
{
  return *(long *) key1 - *(long *) key2;
}

GM_ENTRY_POINT unsigned long
gm_hash_hash_long (void *key)
{
  return gm_crc ((char *) key, sizeof (long));
}

/************
 * integer utility functions
 ************/

GM_ENTRY_POINT long
gm_hash_compare_ints (void *key1, void *key2)
{
  return *(int *) key1 - *(int *) key2;
}

GM_ENTRY_POINT unsigned long
gm_hash_hash_int (void *key)
{
  return gm_crc ((char *) key, sizeof (int));
}

/************
 * pointer utility functions
 ************/

GM_ENTRY_POINT long
gm_hash_compare_ptrs (void *key1, void *key2)
{
  return key1 != key2;
}

GM_ENTRY_POINT unsigned long
gm_hash_hash_ptr (void *key)
{
  return gm_crc (&key, sizeof (key));
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
