/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_debug_malloc.h"
#include "gm_internal.h"
#include "gm_malloc_debug.h"

#if !GM_KERNEL
#include <stdlib.h>
#endif

#if GM_DEBUG_MALLOC
  /* Verify that an appropriate pointer is being freed. */
  
void
_gm_mdebug_remove_client_ptr (void *client_ptr, void *free_func, void *caller)
{
  struct gm_mdebug **dbg;
  GM_PARAMETER_MAY_BE_UNUSED (caller);
  
  /* find the best matching allocation record */

  for (dbg = &gm_mdebug_root; *dbg; dbg = &((*dbg)->next))
    {
      if (*dbg == GM_MDEBUG_INTERNAL_PTR (client_ptr))
	break;
    }

  /* Report error for frees of non-allocated ptrs. */

  if (!*dbg)
    {
      _GM_PANIC (("Free function %p called from %p tried to free %p,\n",
		  free_func, caller, client_ptr));
      GM_PANIC (("    which was never allocated."));
    }

  /* Report frees of allocated pointers with the wrong free function */

  if ((*dbg)->free_function && (*dbg)->free_function != free_func)
    {
      _GM_PANIC (("Free function %p called from %p tried to free %p,\n",
		  free_func, caller, client_ptr));
      GM_PANIC (("    which was allocated by %p and should have been "
		 "freed by %p.\n",
		 (*dbg)->alloc_caller, (*dbg)->free_function));
    }

  /* Splice the match out of the allocation list. */
     
  *dbg = (*dbg)->next;
}
#endif

GM_ENTRY_POINT
void
gm_free (void *ptr)
{
  GM_CALLED_WITH_ARGS (("%p", ptr));

  GM_MDEBUG_REMOVE_CLIENT_PTR (ptr, gm_free);
  ptr = GM_MDEBUG_INTERNAL_PTR (ptr);

  /* Free the pointer */

#if GM_KERNEL
  gm_arch_kernel_free (ptr);
#else
  free (ptr);
#endif

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("%p freed\n", ptr));

  GM_RETURN_NOTHING ();
}

#if GM_DEBUG_MALLOC
/* Report memory leaks */

void
gm_malloc_residual (void)
{
  struct gm_mdebug *ptr;

  for (ptr = gm_mdebug_root; ptr; ptr = ptr->next)
    {
      GM_NOTE (("Block not freed:0x%p allocated at 0x%p\n",
		(char *) ptr + sizeof (struct gm_mdebug), ptr->alloc_caller));
    }
}
#endif


