/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_debug.h"
#include "gm_enable_trace.h"
#include "gm_internal.h"
#include "gm_trace.h"

#if !GM_KERNEL
#  include <stdio.h>
   
				/* close() */
#  ifdef HAVE_UNISTD_H
#    include <unistd.h>
#  endif
   
				/* munmap() */
#  ifdef HAVE_SYS_TYPES_H
#    include <sys/types.h>
#  endif
#  ifdef HAVE_SYS_MMAN_H
#    include <sys/mman.h>
#  endif
#endif


/* Free all the resources for the port. */

GM_ENTRY_POINT void
gm_close (gm_port_t *p)
{
  int i;
  void *addr;
  unsigned long len;

  /****
   * Remove the port from the list of open ports.
   ****/

  if (!p)
    return;

  _gm_open_ports_remove (p);

  /****
   * Free resources allocated since gm_open()
   ****/

  /* Free the buffer allocated in _gm_get_kernel_build_id */

  if (p->kernel_build_id)
    gm_free (p->kernel_build_id);

  /* Free the allocated DMA memory zones and unmap the associated
     memory. */
  
  for (i=((int) p->num_zones)-1; i>=0; i--)
    {
      addr = p->dma_zone_base[i];
      gm_assert (addr);
      len = GM_STATIC_CAST (unsigned long, p->dma_zone_len[i]);
      gm_assert (len);
      gm_assert (GM_POWER_OF_TWO (len));
      GM_PRINT (GM_PRINT_LEVEL >= 1, ("Freeing zone %p.\n", p->dma_zone[i]));
      gm_zone_destroy_zone (p->dma_zone[i]);
#if GM_CAN_REGISTER_MEMORY
      /* the driver would be able to do it anyway, but better to do at once */
      gm_deregister_memory (p, addr, len);
      gm_free_pages (addr, len);
#else
      _gm_munmap (p, addr, len);
#endif
    }
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Done freeing zones.\n"));

  /****
   * Free resources allocated in gm_open_common(). 
   ****/

  if (p->RTC)
    _gm_munmap (p, (void *)p->RTC,
		GM_STATIC_CAST (unsigned long, p->mappings->RTC.len));
  if (p->recv_queue_start)
    _gm_munmap (p, (void *)p->recv_queue_start,
		GM_STATIC_CAST (unsigned long, p->mappings->recv_queue.len));
  if (p->send_queue_start)
    _gm_munmap (p, (void *)p->send_queue_start,
		GM_STATIC_CAST (unsigned long, p->mappings->send_queue.len));
  if (p->mappings)
    gm_free (p->mappings);

#if GM_KERNEL
  gm_port_state_close (p->kernel_port_state);
  gm_minor_free (p->kernel_port_state->minor);
#else 
#  ifdef _WIN32
  CloseHandle (p->fd);
#  else
  close (p->fd);
#  endif
#endif
  gm_free (p);
  gm_finalize ();
}
