/*******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_compiler.h"
#include "gm_debug.h"
#include "gm_internal.h"

#if GM_OS_LINUX && !GM_KERNEL
#include <sys/ioctl.h>
#endif

#if !GM_KERNEL
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif

#define GM_DEBUG_USER_IOCTL 0

#if GM_DEBUG_USER_IOCTL
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

/* Internal function to make IOCTLs look the same in all environments
   (unix, NT, VxWorks, inside the kernel, etc.).  Behaves like Unix
   ioctl() except that it requires BUFSIZE, and any non-null BUF must
   point to a real buffer of bufsize.
   
   Returns GM_SUCCESS on success. */

static
gm_status_t
__gm_user_ioctl (gm_port_t *p, unsigned int cmd, void *buf,
		 gm_size_t bufsize)
{
  GM_CALLED_WITH_ARGS (("%p,%u,%p,"GM_U64_TMPL, p, cmd, buf,
			GM_U64_ARG (bufsize)));
#if GM_KERNEL
  GM_PARAMETER_MAY_BE_UNUSED (p);
  GM_PARAMETER_MAY_BE_UNUSED (cmd);
  GM_PARAMETER_MAY_BE_UNUSED (buf);
  GM_PARAMETER_MAY_BE_UNUSED (bufsize);
  GM_RETURN_STATUS (GM_FAILURE);
#elif defined WIN32
  {
    DWORD trash;
    
    gm_assert (p->fd != INVALID_HANDLE_VALUE);
  
    /* Perform the Ioctl on that port */

    if (DeviceIoControl (p->fd, cmd, buf,
			 GM_STATIC_CAST (unsigned long, bufsize), buf,
			 GM_STATIC_CAST (unsigned long, bufsize), &trash, NULL)
	== 0)
      {
	GM_RETURN_STATUS (GM_FAILURE);
      }
  }
  GM_RETURN_STATUS (GM_SUCCESS);
#else /* WIN32 not defined */
  gm_assert (p->fd);
  GM_PARAMETER_MAY_BE_UNUSED (bufsize);
  GM_RETURN_STATUS ((gm_status_t) ioctl (p->fd, cmd, buf));
#endif  /* !GM_KERNEL */
}

/* This function abstract the various ioctl()-like functions found
   on various operating systems.  Note that it may temporarily modify
   the contents of BUF during the call, so BUF must always be writable. */

GM_ENTRY_POINT
gm_status_t
_gm_user_ioctl (gm_port_t *p, unsigned int cmd, void *buf,
		gm_size_t bufsize)
{
  gm_status_t ret;

  GM_CALLED_WITH_ARGS (("%p,0x%x,%p,0x%lx", p, cmd, buf, bufsize));

  /* We don't trust the OS to not mangle the error code.  Therefore,
     we pass the error code back in the beginning of the copy buffer
     if there is an error.  If the copy buffer is too small or
     nonexisitant, we substitute an adequate copy buffer. */

#if GM_OS_OSF1
  /* osf1 can't return error status if the ioctl returns failure */
  GM_RETURN_STATUS (__gm_user_ioctl (p, cmd, buf, bufsize));
#endif

  
  if (bufsize >= sizeof (ret))
    {
      gm_status_t save;		/* space to save potentially trashed bytes */

      gm_bcopy (buf, (void *) &save, sizeof (save)); /* save bytes */
      if (__gm_user_ioctl (p, cmd, buf, bufsize) == GM_SUCCESS)
	{
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
      gm_bcopy (buf, (void *)&ret, sizeof (ret)); /* copy error code */
      gm_bcopy ((void *) &save, buf, sizeof (save)); /* restore bytes */
      GM_RETURN_STATUS (ret);
    }
  else if (buf)
    {
      if (__gm_user_ioctl (p, cmd, (void *)&ret, sizeof (ret)) == GM_SUCCESS)
	{
	  gm_bcopy ((void *) &ret, buf, bufsize); /* copy results */
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
      GM_RETURN_STATUS (ret);
    }
  else
    {
      if (__gm_user_ioctl (p, cmd, (void *)&ret, sizeof (ret)) == GM_SUCCESS)
	ret = GM_SUCCESS;
    }

  GM_RETURN_STATUS (ret);
}
