/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 			 *
 * Copyright (c) 1999 by Myricom, Inc.					 				 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

#include "gm_debug.h"
#include "gm_internal.h"

#if GM_KERNEL
#include "gm_enable_security.h"
#include "gm_pio.h"
#endif

#define GM_DEBUG_SET_ROUTE 0

GM_ENTRY_POINT
gm_status_t
_gm_set_route (gm_port_t *p, unsigned int target_node_id, unsigned int length,
	       char *route)
{
#if GM_KERNEL
  gm_port_state_t *ps = p->kernel_port_state;
  gm_instance_state_t *is = ps->instance;

  GM_PRINT (GM_DEBUG_SET_ROUTE,
			("_gm_set_route (%p, id=%d len=%d route=%p)\n",
			 p, target_node_id, length,route));

  if (GM_ENABLE_SECURITY && !ps->privileged)
    return GM_PERMISSION_DENIED;
  if (target_node_id > is->max_node_id
      || length > GM_MAX_NETWORK_DIAMETER)
    return GM_INVALID_PARAMETER;

  if (!is->lanai.running) {
    GM_PRINT (GM_DEBUG_SET_ROUTE, ("oops: lanai.running false\n"));
    return GM_UNATTACHED;
  }
  
#if (PRINT_LEVEL >= 5)
  {
    unsigned int i;
    GM_PRINT (GM_DEBUG_SET_ROUTE, ("setting route to node %d of length %d to:",
								   length, target_node_id));
    for (i=0; i<length; i++) {
      GM_PRINT (GM_DEBUG_SET_ROUTE, (" %02x", route[i]));
    }
    GM_PRINT (GM_DEBUG_SET_ROUTE, ("\n"));
  }
#endif

  /* No need to synchronize with LANai since momentary short routes are
     harmless. */

  /* clear the route */

  gm_write_lanai_global_u8 (is, connection[target_node_id].has_route, 0);
  gm_write_lanai_global_u8 (is, connection[target_node_id].route_len, 0);
  GM_STBAR ();
  
  /* set the route */

  if (length != GM_NO_ROUTE)
    {
      gm_bcopy_to_lanai_globals
	(is, route,
	 connection[target_node_id].route[GM_MAX_NETWORK_DIAMETER-length],
	 length);
      GM_STBAR ();
      gm_write_lanai_global_u8 (is, connection[target_node_id].route_len,
				(gm_u8_t) length);
      gm_write_lanai_global_u8 (is, connection[target_node_id].has_route,
				(gm_u8_t) -1);
    }
  
  /* now set the config reply message to have this node's ID */
  /* so that gm_get_mapper_unique_id works */
  {
    gm_u8_t my_id[8];
    if (gm_unique_id (p, (char *) my_id) == GM_SUCCESS)
      {
	gm_write_lanai_global_byte_array (is, my_id, mapper_state.unique_id);
      }
  }

  return GM_SUCCESS;
#else
  gm_route_info_t ri;
  unsigned int i;

  ri.length = (gm_u8_t) length;
  ri.target_node_id = target_node_id;
  if (length && (length != GM_NO_ROUTE))
    {
      for (i=0; i<length; i++)
        ri.route[i] = route[i];
    }

  return _gm_user_ioctl (p, GM_SET_ROUTE, &ri, sizeof (ri));
#endif
}


/*
  This file uses feldy indentation:

  Local Variables:
  tab-width:4
  c-file-style:"bsd"
  End:
*/
