/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_internal.h"
#include "gm_enable_galvantech_workaround.h"

#define GM_DEBUG_IP_CHECKSUM 0

#if GM_DEBUG_IP_CHECKSUM
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

/* Return an IP checksum.  Be careful not to read past the ends of the
   buffer, since this code can run on the host, and doing so may cause
   segfaults.

   This uses tricks from RFC 1071, including the fact that the checksum
   can be computed while disregarding endianness.  However, it
   is critical that the checksum is properly verified. */

gm_u16_t __gm_ip_checksum (void *_message, unsigned long _len)
{
  register unsigned long accum = 0;
  register void *message;
  register unsigned long len;
  
  gm_assert (sizeof (accum) >= 4);

  GM_CALLED_WITH_ARGS (("%p,0x%lx", _message, _len));

  /* Compute checksum conventionally up to a 32 byte boundary. */

  message = _message;
  len = _len;
  if (((long) message & 1) && (len >= 1))
    {
      register gm_u8_t *temp8;

      temp8 = (gm_u8_t *) message;
      accum += __gm_hton_u16 (*temp8++);
      message = (void *) temp8;
      len -= 1;
    }
  gm_assert (GM_ALIGNED (message, 2));
  while (((long) message & 31) && (len >= 2))
    {
      register gm_u16_t *temp16;

      temp16 = (gm_u16_t *) message;
      accum += *temp16++;
      message = (void *) temp16;
      len -= 2;
    }

  /* compute core in unrolled loop that should pipeline well on 32 and
     64 bit machines. */

  if (len >= 33)
    {
      register gm_u64_t a, b;

      gm_assert (GM_ALIGNED (message, 32));
      do
	{
	  register gm_u64_t *temp64;

	  temp64 = (gm_u64_t *) message;

	  a = *temp64++;
	  b = *temp64++;
	  accum += (unsigned long) ((a >> 48) & 0xffff);
	  accum += (unsigned long) ((a >> 32) & 0xffff);
	  accum += (unsigned long) ((a >> 16) & 0xffff);
	  accum += (unsigned long) (a & 0xffff);
	  a = *temp64++;
	  accum += (unsigned long) ((b >> 48) & 0xffff);
	  accum += (unsigned long) ((b >> 32) & 0xffff);
	  accum += (unsigned long) ((b >> 16) & 0xffff);
	  accum += (unsigned long) (b & 0xffff);
	  b = *temp64++;
	  accum += (unsigned long) ((a >> 48) & 0xffff);
	  accum += (unsigned long) ((a >> 32) & 0xffff);
	  accum += (unsigned long) ((a >> 16) & 0xffff);
	  accum += (unsigned long) (a & 0xffff);
	  accum += (unsigned long) ((b >> 48) & 0xffff);
	  accum += (unsigned long) ((b >> 32) & 0xffff);
	  accum += (unsigned long) ((b >> 16) & 0xffff);
	  accum += (unsigned long) (b & 0xffff);

	  message = (void *) temp64;

	  /* Roll any carries back into the checksum */
	  accum = (accum & 0xffff) + (accum >> 16);

	  /* advance */
	  len -= 32;
	}
      while (len >= 33);
    }

  /* Handle the small remainder of the packet conventionally. */

  gm_assert (GM_ALIGNED (message, 2));
  while (len >= 2)
    {
      gm_u16_t d;
      register gm_u16_t *temp16;

      temp16 = (gm_u16_t *) message;
      d = *temp16++;
      message = (void *) temp16;
      accum += d;
      /* advance */
      len -= 2;
    }
  if (len)
    {
      accum += __gm_hton_u16 (*(gm_u8_t *) message);
      /* no need to advance, since we are done */
    }

  /* feldy - add in the length also to help catch other sorts of errors */
  accum += (gm_u16_t) _len;

  /* Roll any carries back into the checksum */

  accum = (accum & 0xffff) + (accum >> 16);
  accum = (accum & 0xffff) + (accum >> 16);

  /* correct for misaligned start by swapping the bytes. */

  if ((long) _message & 1)
    {
      accum = ((accum >> 8) & 0xff) + ((accum << 8) & 0xff00);
    }

  gm_assert (accum == (accum & 0xffff));
  GM_PRINT (GM_DEBUG_IP_CHECKSUM, ("returning checksum 0x%04lx\n", accum));
  GM_RETURN (__gm_ntoh_u16 ((gm_u16_t) accum));
}

/* Verify a received IP checksum. */

gm_status_t
_gm_ip_checksum_verify (void *message, unsigned long len,
			gm_u16_t recvd_checksum)
{
#if 1
  gm_u32_t accum;

  GM_PRINT (GM_DEBUG_IP_CHECKSUM, ("recvd IP checksum=0x%04x\n",
				   (unsigned int) recvd_checksum));

  /* compute checksum over data and received checksum */

  accum = __gm_ip_checksum (message, len);
  GM_PRINT (GM_DEBUG_IP_CHECKSUM,
	    ("computed ones complement checksum=0x%04x\n",
	     (unsigned int) accum));
  accum += recvd_checksum;
  accum = (accum & 0xffff) + (accum >> 16);
  accum = (accum & 0xffff) + (accum >> 16);
  if (accum == 0xffff)
    return GM_SUCCESS;
  else
    return GM_FAILURE;
#else
  if (__gm_ip_checksum (message, len) == recvd_checksum)
    {
      GM_PRINT (GM_DEBUG_IP_CHECKSUM,
		("checksum_verify: recvd IP checksum=0x%04x -- verified\n",
		 (unsigned int) recvd_checksum));
      return GM_SUCCESS;
    }
  else
    {
      GM_PRINT (GM_DEBUG_IP_CHECKSUM,
		("checksum_verify: recvd IP checksum=0x%04x -- ** FAILED\n",
		 (unsigned int) recvd_checksum));
      return GM_FAILURE;
    }
#endif
}

/* _gm_ip_checksum() entry point with verification.  The real work is done
   in __gm_ip_checksum() above. */

gm_u16_t _gm_ip_checksum (void *message, unsigned long len)
{
  gm_u16_t ret;

  ret = __gm_ip_checksum (message, len) ^ 0xffff;
  gm_assert (_gm_ip_checksum_verify (message, len, ret) == GM_SUCCESS);
  return ret;
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
