/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997, 1998 by Myricom, Inc.                       *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_lanai.h"
#include "gm_bsp_support.h"

#include <sys/types.h>

#include "vxWorks.h"
#include "stdlib.h"
#include "string.h"
#include "vmLib.h"
#include "stdio.h"
#include "time.h"
#include "taskLib.h"
/*#include "cacheLib.h"*/

extern int sysClkRateGet ();
/*
   #define INUM_TO_IVEC(intNum)    ((VOIDFUNCPTR *) (intNum))
 */

#include "gm_internal.h"
#include "gm_arch.h"
#include "gm_call_trace.h"
#include "gm_page_hash.h"

#ifdef USE_ZLIB
#include "zlib.h"
#endif

#if GM_SUPPORT_PCI_REV_1 || GM_SUPPORT_PCI_REV_2 || GM_SUPPORT_PCI_REV_3
#define GM_PCI 1
#include "drv/pci/pciLocalBus.h"
#include "drv/pci/pciConfigLib.h"
#else /* no PCI */
#define GM_PCI 0
#endif /* no PCI */

extern int sysClkRateGet ();

#ifndef NULL
#define NULL (void *)0
#endif


/***********************************************************************
 * Globals
 ***********************************************************************/
#include <boardCsr.h>		/* SAB */

/*
 * linked list of GM devices
 */
static gm_instance_state_t *dev_head = 0;
static gm_instance_state_t *gm_instances[GM_ARCH_MAX_INSTANCE];
int gm_instance_initialized[GM_ARCH_MAX_INSTANCE] = { 0, 0, 0, 0 };
static int gm_num_instance = 0;
static int gm_max_user_locked_pages;
static int gm_skip_init;

#if GM_DEBUG
int gm_print_level = GM_PRINT_LEVEL;
static gm_instance_state_t *debug_is;
#endif

#ifdef IBM_405_MMU_HACK
/****************
 * IBM 405 HACK!!!
 * The vxworks doesn't have MMU support yet.
 * so we fake vmBasePageSizeGet.
 ****************/
static int vmBasePageSizeGet(void) { return 4096; }
#endif /* IBM_405_MMU_HACK */

/************
 * gm_port_state initialization 
 ************/

/* This is called just after the port state is created (in gm_minor.c)
   to perform architecture-specific initialization. */

gm_status_t gm_arch_port_state_init (gm_port_state_t * ps)
{
/*
   printf("\ngm_arch_port_state_init:  ps = %p   is = %p\n\n",
   ps, ps->instance);
 */
  return GM_SUCCESS;
}

/* This is called just before the port state is destroyed (in
   gm_minor.c) to perform architecture-specific finalization. */

void
gm_arch_port_state_fini (gm_port_state_t * ps)
{
  return;
}

/************
 * gm_port_state initialization 
 ************/

/* This is called at the end of gm_port_state_open() to perform architecture-
   specific initialization. */

gm_status_t gm_arch_port_state_open (gm_port_state_t * ps)
{
  return GM_SUCCESS;
}

/* This is called at the start of gm_port_state_close to perform
   architecture-specific finalization. */

void
gm_arch_port_state_close (gm_port_state_t * ps)
{
  return;
}

/****************
 * gm_abort
 ****************/

void
gm_arch_abort (void)
{
  printf("gm_arch_abort called\n");
}

/***********************************************************************
 * Utility functions
 ***********************************************************************/

/* These are required so the gm_ioctl( ) can copy its arguments
   and results. */

gm_status_t
gm_arch_copyin (gm_port_state_t * ps,
		void *what, void *where, gm_size_t amount)
{
  return GM_FAILURE;
}

gm_status_t
gm_arch_copyout (gm_port_state_t * ps,
		 void *what, void *where, gm_size_t amount)
{
  GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_arch_copyout(%p, %p, %p, %ld)\n",
				  ps, what, where, amount));
  bcopy (what, where, amount);
  return GM_SUCCESS;
}

/* Utility functions for generic error printing */

#if (defined CSPI_MAP26xx)
static char *buffer_cspi = (char *) 0;
static int index_cspi = 0;
static char *topMem = (char *) 0x00000000;
extern char *sysMemTop ();
extern char *sysPhysMemTop ();
static cspi_count = 0;

void
find_cspi_index ()
{
  cspi_count += 1;
  if (buffer_cspi == 0)
    {
      buffer_cspi = (char *) sysMemTop ();
      topMem = sysPhysMemTop ();
      if (buffer_cspi <= topMem)
	{
	  bzero (buffer_cspi, (long) topMem - (long) &buffer_cspi[0]);
	}
    }
}

#if PLEASE_DO_NOT_REENABLE_WITHOUT_TALKING_TO_GLENN_FIRST
int
gm_printf (char *format, ...)
{
  va_list ap;
  int ret;
  long *temp = (long *) 0x4400;
  /* long R1 = sysR1();
     long R2; */

  va_start (ap, format);


#ifdef SAB
  temp[0] = 0xfab1e;
  temp[1] = 0xfab1e;
  temp[3] = 0xfab1e;
  temp[4] = 0xfab1e;
#endif
  temp[2] = 0xfab1e;
  temp[5] = 0xfab1e;
  temp[6] = 0xfab1e;
  temp[7] = 0xfab1e;
  temp[8] = 0xfab1e;

  find_cspi_index ();
  if (&buffer_cspi[index_cspi + 256] < topMem)
    {
      ret = vsprintf (&buffer_cspi[index_cspi], format, ap);
      index_cspi = ((index_cspi + ret + 15) & 0xfffffff0);
    }

#ifdef SAB
  temp[0] = 0xc0bb1e;
  temp[1] = 0xc0bb1e;
  temp[3] = 0xc0bb1e;
  temp[4] = 0xc0bb1e;
#endif
  temp[2] = 0xc0bb1e;
  temp[5] = 0xc0bb1e;
  temp[6] = 0xc0bb1e;
  temp[7] = 0xc0bb1e;
  temp[8] = 0xc0bb1e;

  va_end (ap);
  return GM_SUCCESS;
}
#endif


void
gm_vxworks_print (char *format, ...)
{
  va_list ap;
  int ret;
  va_start (ap, format);

  find_cspi_index ();
  if (&buffer_cspi[index_cspi + 256] < topMem)
    {
      ret = vsprintf (&buffer_cspi[index_cspi], format, ap);
      index_cspi = ((index_cspi + ret + 15) & 0xfffffff0);
    }

  va_end (ap);
}

void
gm_vxworks_warn (char *format, ...)
{
  va_list ap;
  int ret;
  va_start (ap, format);

  find_cspi_index ();
  if (&buffer_cspi[index_cspi + 256] < topMem)
    {
      ret = vsprintf (&buffer_cspi[index_cspi], format, ap);
      index_cspi = ((index_cspi + ret + 15) & 0xfffffff0);
    }
  va_end (ap);
}

void
gm_vxworks_info (char *format, ...)
{
  va_list ap;
  int ret;
  va_start (ap, format);

  find_cspi_index ();
  if (&buffer_cspi[index_cspi + 256] < topMem)
    {
      ret = vsprintf (&buffer_cspi[index_cspi], format, ap);
      index_cspi = ((index_cspi + ret + 15) & 0xfffffff0);
    }

  va_end (ap);
}

void
gm_vxworks_note (char *format, ...)
{
  va_list ap;
  int ret;
  va_start (ap, format);

  find_cspi_index ();
  if (&buffer_cspi[index_cspi + 256] < topMem)
    {
      ret = vsprintf (&buffer_cspi[index_cspi], format, ap);
      index_cspi = ((index_cspi + ret + 15) & 0xfffffff0);
    }

  va_end (ap);
}


void
gm_vxworks_panic (char *format, ...)
{
  va_list ap;
  int ret;
  va_start (ap, format);

  find_cspi_index ();
  if (&buffer_cspi[index_cspi + 256] < topMem)
    {
      ret = vsprintf (&buffer_cspi[index_cspi], format, ap);
      index_cspi = ((index_cspi + ret + 15) & 0xfffffff0);
    }

  va_end (ap);
  while (1)
    {
    }
}
#endif /* defined CSPI_MAP26xx */

static void *
gm_kmalloc (unsigned int size, int priority)
{
  void *ptr;
  ptr = malloc (size);
  if (ptr)
    {
      bzero (ptr, size);
    }
  printf ("****** GM_KMALLOC returning  %p\n", ptr);
  return (ptr);
}

static void
gm_kfree (void *obj)
{
  printf ("****** GM_KFREE freeing %p\n", obj);
  free (obj);
}

static void *
gm_vmalloc (unsigned long size)
{
  void *ptr;
  ptr = malloc (size);
  if (ptr)
    {
      bzero (ptr, size);
    }
  printf ("****** GM_VMALLOC returning  %p\n", ptr);
  return (ptr);
}

static void
gm_vfree (void *obj)
{
  printf ("****** GM_VFREE freeing %p\n", obj);
  free (obj);
}

static void *
gm_ioremap (unsigned long offset, unsigned long size)
{
  return NULL;
}

static void
gm_iounmap (void *obj)
{
}

gm_status_t _gm_kernel_recv_queue_update (gm_port_t * p)
{
  return GM_FAILURE;
}





#if GM_PCI

/************
 * PCI configuration space access functions.
 ************/

gm_status_t
gm_arch_read_pci_config_32 (gm_instance_state_t * is,
			    gm_offset_t offset, gm_u32_t * valueptr)
{
  pciConfigInLong (is->arch.pcibusno, is->arch.pcideviceno,
		   is->arch.pcifuncno, offset, valueptr);
  return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_32 (gm_instance_state_t * is,
			     gm_offset_t offset, gm_u32_t value)
{
  pciConfigOutLong (is->arch.pcibusno, is->arch.pcideviceno,
		    is->arch.pcifuncno, offset, value);
  return GM_SUCCESS;
}

/* 16-bit accesses */

gm_status_t
gm_arch_read_pci_config_16 (gm_instance_state_t * is,
			    gm_offset_t offset, gm_u16_t * valueptr)
{
  pciConfigInWord (is->arch.pcibusno, is->arch.pcideviceno,
		   is->arch.pcifuncno, offset, valueptr);
  return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_16 (gm_instance_state_t * is,
			     gm_offset_t offset, gm_u16_t value)
{
  pciConfigOutWord (is->arch.pcibusno, is->arch.pcideviceno,
		    is->arch.pcifuncno, offset, value);
  return GM_SUCCESS;
}

/* 8-bit accesses */

gm_status_t
gm_arch_read_pci_config_8 (gm_instance_state_t * is,
			   gm_offset_t offset, gm_u8_t * valueptr)
{
  pciConfigInByte (is->arch.pcibusno, is->arch.pcideviceno,
		   is->arch.pcifuncno, offset, valueptr);
  return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_8 (gm_instance_state_t * is,
			    gm_offset_t offset, gm_u8_t value)
{
  pciConfigOutByte (is->arch.pcibusno, is->arch.pcideviceno,
		    is->arch.pcifuncno, offset, value);
  return GM_SUCCESS;
}

#endif /* GM_PCI */

/****************************
 * Synchronization functions
 ****************************/

void
gm_test_sync (gm_arch_sync_t * s)
{
  if (s)
    {
      if ((s->topline == 0x12345678) && (s->botline == 0x87654321))
	{
	  return;
	}
      else
	{
	  GM_PANIC (
		    ("gm_test_sync(%p) FAILED 0x%x 0x%x\n", s, s->topline,
		     s->botline));
	}
    }
}

gm_arch_sync_t *firstsync = NULL;

void
gm_arch_sync_init (gm_arch_sync_t * s, gm_instance_state_t * is)
{
  gm_test_sync (firstsync);

  s->semSync = semBCreate (SEM_Q_FIFO, SEM_EMPTY);
  s->semMutex = semBCreate (SEM_Q_PRIORITY, SEM_FULL);
  s->topline = 0x12345678;
  s->botline = 0x87654321;
  GM_PRINT (GM_PRINT_LEVEL >= 8,
	    ("gm_arch_sync_init(%p, %p)  sync = 0x%x  mutex = 0x%x\n", s, is,
	     s->semSync, s->semMutex));
  if (!s->semSync || !s->semMutex)
    {
      GM_PANIC (
		("gm_arch_sync_init(%p, %p)  couldn't allocate sync = 0x%x  mutex = 0x%x\n",
		 s, is, (int)(s->semSync), (int)(s->semMutex)));
    }
  if (!firstsync)
    {
      firstsync = s;
    }
}

void
gm_arch_sync_reset (gm_arch_sync_t * s)
{
  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("gm_arch_sync_reset(%p)  sync = 0x%x \n", s, s->semSync));
  gm_test_sync (firstsync);
  gm_test_sync (s);
}

void
gm_arch_sync_destroy (gm_arch_sync_t * s)
{
  gm_test_sync (firstsync);
  GM_PRINT (GM_PRINT_LEVEL >= 8,
	    ("gm_arch_sync_destroy(%p)  sync = 0x%x  mutex = 0x%x\n", s,
	     s->semSync, s->semMutex));
  gm_assert (s->topline == 0x12345678);
  gm_assert (s->botline == 0x87654321);
  semDelete (s->semSync);
  semDelete (s->semMutex);
  bzero ((char *) s, sizeof (gm_arch_sync_t));
  if (s == firstsync)
    {
      firstsync = NULL;
    }
}

void
gm_arch_mutex_enter (gm_arch_sync_t * s)
{
  gm_test_sync (firstsync);

  GM_PRINT (GM_PRINT_LEVEL >= 9,
	    ("gm_arch_mutex_enter1(%p) top = 0x%x  bot = 0x%x\n", s,
	     s->topline, s->botline));
  gm_assert (s->topline == 0x12345678);
  gm_assert (s->botline == 0x87654321);


  semTake (s->semMutex, WAIT_FOREVER);
  GM_PRINT (GM_PRINT_LEVEL >= 9,
	    ("gm_arch_mutex_enter2(%p) top = 0x%x  bot = 0x%x\n", s,
	     s->topline, s->botline));
}

void
gm_arch_mutex_exit (gm_arch_sync_t * s)
{
  gm_test_sync (firstsync);

/*
	GM_PRINT (GM_PRINT_LEVEL >= 999, ("gm_arch_mutex_exit1(%p) top = 0x%x  bot = 0x%x\n", s, s->topline, s->botline));
*/
  gm_assert (s->topline == 0x12345678);
  gm_assert (s->botline == 0x87654321);

  semGive (s->semMutex);
/*
	GM_PRINT (GM_PRINT_LEVEL >= 999, ("gm_arch_mutex_exit2(%p) top = 0x%x  bot = 0x%x\n", s, s->topline, s->botline));
*/
}

/* Sleep functions. Return 0 on wake, -1 on timeout, and 1 on signal. */

/* Wake the thread sleeping on the synchronization variable. */
void
gm_arch_wake (gm_arch_sync_t * s)
{
  gm_test_sync (firstsync);

/*
   logMsg("gm_arch_wake(%p)n", s, 0, 0, 0, 0, 0);
 */
  gm_assert (s->topline == 0x12345678);
  gm_assert (s->botline == 0x87654321);

  semGive (s->semSync);
}

/* sleep until awakend or timeout */
gm_arch_sleep_status_t gm_arch_timed_sleep (gm_arch_sync_t * s, int seconds)
{
  STATUS rv;

  gm_test_sync (firstsync);

  GM_PRINT (GM_PRINT_LEVEL >= 8,
	    ("gm_arch_timed_sleep(%p, %d secs)  sync = 0x%x mutex = 0x%x \n",
	     s, seconds, s->semSync, s->semMutex));

  gm_test_sync (s);

  rv = semTake (s->semSync, (CLOCKS_PER_SEC * seconds) + 1);

  if ((rv == ERROR) && (errno == S_objLib_OBJ_ID_ERROR))
    {
      printf ("***gm_arch_timed_sleep(%p, %d secs) bad sync ID?\n",
	      s, seconds);
      return GM_FAILURE;
    }

  if ((rv == ERROR) && (errno == S_objLib_OBJ_TIMEOUT))
    {
      /* timed out */
      return GM_SLEEP_TIMED_OUT;
    }


  return GM_WOKE;
}

/* sleep until awakened or get a signal */
gm_arch_sleep_status_t gm_arch_signal_sleep (gm_arch_sync_t * s)
{
  STATUS rv;

  gm_test_sync (firstsync);

  GM_PRINT (GM_PRINT_LEVEL >= 8,
	    ("gm_arch_signal_sleep(%p) sync = 0x%x \n", s, s->semSync));

  gm_test_sync (s);

  rv = semTake (s->semSync, WAIT_FOREVER);

  if ((rv == ERROR) && (errno == S_objLib_OBJ_ID_ERROR))
    {
      printf ("***gm_arch_signal_sleep(%p) bad sync ID?\n", s);
      return GM_FAILURE;
    }

  return GM_WOKE;
}

/*********************************************************************
 * DMA region functions
 *********************************************************************/

static void
gm_arch_lock_page (unsigned long phys)
{
}

static void
gm_arch_unlock_page (unsigned long phys)
{
}




gm_status_t
gm_arch_lock_user_buffer_page (gm_instance_state_t * is, gm_up_t in,
			       gm_dp_t * dma_addr, gm_arch_page_lock_t * lock)
{
#if GM_CAN_REGISTER_MEMORY
  gm_dp_t my_dma_addr;

  my_dma_addr = (gm_dp_t) in;
  my_dma_addr |= GM_ARCH_DMA_BITS_TO_SET;
  my_dma_addr &= ~GM_ARCH_DMA_BITS_TO_CLEAR;
  *dma_addr = my_dma_addr;
  return GM_SUCCESS;
#else
  return GM_FAILURE;
#endif
}


void
gm_arch_unlock_user_buffer_page (gm_arch_page_lock_t * lock)
{
}



static gm_dp_t
gm_vxworks_region_dma_addr (gm_arch_dma_region_t * r, void *addr)
{
  gm_dp_t dma_addr;

  dma_addr = (gm_dp_t) addr;
  dma_addr |= GM_ARCH_DMA_BITS_TO_SET;
  dma_addr &= ~GM_ARCH_DMA_BITS_TO_CLEAR;
  return (gm_dp_t) dma_addr;
}


/* Allocate LEN bytes of DMA memory that is contiguous in kernel space
   by possibly segmented in DMA space.

   If r->register_function is non-null, call r->register_page_function
   (r, dma_addr) for each page. */

gm_status_t
gm_arch_dma_region_alloc (gm_instance_state_t * is,
			  gm_arch_dma_region_t * r,
			  gm_size_t len, gm_u32_t flags,
			  gm_register_page_function_t register_page_func,
			  void *arg)
{
  int page_num, pages;
  void *addr;

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("gm_arch_dma_region_alloc(%p, %p, %d, 0x%x, %p)\n", is, r, len,
	     flags, arg));

#ifdef IBM_405_MMU_HACK
  len = GM_PAGE_ROUNDUP (u32, len + GM_PAGE_LEN);
#else
  len = GM_PAGE_ROUNDUP (u32, len);
#endif
  
  pages = len / GM_PAGE_LEN;

  r->is = is;
  r->len = len;
  r->flags = flags;
  r->original_addr =
#ifdef NON_CACHED
    cacheDmaMalloc (len);
#elif defined (IBM_405_MMU_HACK)
    malloc(len);
#else
    valloc (len);
#endif
  if (!r->original_addr)
    {
      printf
	("**** gm_arch_dma_region_alloc: couldn't valloc() size %ld bytes \n",
	 len);
      return GM_FAILURE;
    }

  if (flags & GM_ARCH_DMA_CONSISTENT)
    {
#ifdef DO_VM_STUFF
      STATUS rv;
      rv =
	vmStateSet (NULL, r->original_addr, len, VM_STATE_MASK_CACHEABLE,
		    VM_STATE_CACHEABLE_NOT);
      if (rv == ERROR)
	{
	  printf ("**** vmStateSet (%p, %d)\n", r->original_addr, len);
	}
#endif
    }

  bzero (r->original_addr, len);
  r->addr = (void *)GM_PAGE_ROUNDUP(up, r->original_addr);

  r->addr2 = r->addr;

  if (register_page_func)
    {
      page_num = 0;
      addr = r->addr;
      while (page_num < pages)
	{
	  gm_dp_t bus_addr = gm_vxworks_region_dma_addr (r, addr);
	  register_page_func (arg, bus_addr, page_num);
	  page_num++;
	  addr += GM_PAGE_LEN;
	}
    }

  return GM_SUCCESS;
}

void
gm_arch_dma_region_free (gm_arch_dma_region_t * r)
{
#ifdef NON_CACHED
  cacheDmaFree (r->original_addr);
#else
  free (r->original_addr);
#endif
}

void *
gm_arch_dma_region_kernel_addr (gm_arch_dma_region_t * r)
{
  return r->addr;
}

gm_s32_t gm_arch_dma_region_status (gm_arch_dma_region_t * r)
{
  return 0xf;
}


gm_dp_t gm_arch_dma_region_dma_addr (gm_arch_dma_region_t * r)
{
  return gm_vxworks_region_dma_addr (r, r->addr);
}

gm_dp_t gm_arch_dma_region_dma_addr_advance (gm_arch_dma_region_t * r)
{
  void *previous = r->addr2;
  r->addr2 += GM_PAGE_LEN;
  return gm_vxworks_region_dma_addr (r, previous);
}

gm_status_t gm_arch_dma_region_sync (gm_arch_dma_region_t * r, int command)
{
  /* FIX */
  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("****** gm_arch_dma_region_sync called - NOT IMPLEMENTED\n"));
  return GM_SUCCESS;
}


/*********************************************************************
 * kernel memory allocation functions 
 *********************************************************************/

void *
gm_arch_kernel_malloc (unsigned long len, int flags)
{
  void *ptr;
  ptr = malloc (len);
  if (ptr)
    {
      bzero (ptr, len);
    }
  return (ptr);
}

void
gm_arch_kernel_free (void *ptr)
{
  if (ptr)
    {
      free (ptr);
    }
}

/*********************************************************************
 * memory mapping (into kernel space)
 *********************************************************************/

gm_status_t
gm_arch_map_io_space (gm_instance_state_t * is, gm_u32_t offset, gm_u32_t len,
		      void **kaddr)
{
  UINT32 dev_addr;

  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("gm_arch_map_io_space(%p, 0x%x, 0x%x, .. )\n", is, offset, len));
#ifdef CSPI_MAP26xx
  *kaddr = (void *) offset;
#else
  pciConfigInLong (is->arch.pcibusno, is->arch.pcideviceno,
		   is->arch.pcifuncno, PCI_CFG_BASE_ADDRESS_0, &dev_addr);

  dev_addr &= ~0xff;		/* clear prefetch and other bits */

  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("gm_arch_map_io_space: dev_addr from card  = 0x%x\n", dev_addr));
  dev_addr = GM_BOARD_PCI_TO_CPU (dev_addr + offset);
  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("gm_arch_map_io_space: dev_addr translated = 0x%x\n", dev_addr));

  *kaddr = (void *) dev_addr;
#endif /* CSPI_MAP26xx */

  /* FIX - do I need to do any kind of registering with vxworks ? */
  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("gm_arch_map_io_space: mapped board at address = %p for len = %d bytes\n",
	     *kaddr, len));

  return GM_SUCCESS;
}

void
gm_arch_unmap_io_space (gm_instance_state_t * is, gm_u32_t offset,
			gm_u32_t len, void **kaddr)
{
  /* FIX - do I need to do any kind of UNregistering with vxworks ? */

  return;
}

/* needed for memory registration? */

gm_status_t
gm_arch_mmap_contiguous_segment (gm_port_state_t * ps,
				 void *kaddr,
				 unsigned long blockSize, gm_up_t * vaddr)
{
  GM_NOT_IMP();
  return GM_FAILURE;
}

void
gm_arch_munmap_contiguous_segments (gm_port_state_t * ps)
{
  GM_NOT_IMP();
}

/*********************************************************************
 * miscellaneous functions
 *********************************************************************/

gm_status_t gm_arch_page_len (unsigned long *result)
{
  *result = (unsigned long) vmBasePageSizeGet ();
  return GM_SUCCESS;
}

gm_status_t gm_arch_physical_pages (gm_u32_t * result)
{
  return GM_FAILURE;
}


extern int gethostname (char *name, int nameLen);
gm_status_t gm_arch_gethostname (char *ptr, int len)
{
  int rv;
  rv = gethostname (ptr, len);
  if (rv == OK)
    {
      ptr[len - 1] = 0;
      return GM_SUCCESS;
    }
  return GM_FAILURE;
}

void
gm_arch_spin (gm_instance_state_t * is, gm_u32_t usecs)
{
  unsigned int ticks;

  ticks = (usecs * sysClkRateGet () / 1000000) + 1;

  taskDelay (ticks);
}

gm_status_t gm_arch_get_page_len (unsigned long *page_len)
{
  *page_len = vmBasePageSizeGet ();
  return GM_SUCCESS;
}

gm_dp_t gm_arch_no_VM_dma_bits_to_set (gm_instance_state_t * is)
{
  return GM_ARCH_DMA_BITS_TO_SET;
}


gm_dp_t gm_arch_no_VM_dma_bits_to_clear (gm_instance_state_t * is)
{
  return GM_ARCH_DMA_BITS_TO_CLEAR;
}

#if GM_PCI
/****************************************************************
 * Stuff for scanning and configuring the PCI bus
 ****************************************************************/

/* if you have more pci slots, increase this number, add more
   descriptions to the slot_desc below, and add the
   definitions in gm_bsp_support.h to match the new number
   number of slots */

#define GM_ARCH_MAX_PCI_SLOT_NUM       4

typedef struct
{
  UINT32 addr;			/* address of the card */
  UINT32 span;			/* amount of memory */
  unsigned char intline;	/* interrupt line */
  UINT32 pcibusno;		/* PCI Bus Number */
  UINT32 pcideviceno;		/* PCI Device Number */
  UINT32 pcifuncno;		/* PCI Function Number */
}
MYRINET_CARD;

static UINT32 myrinet_units;

static MYRINET_CARD slot_desc[GM_ARCH_MAX_PCI_SLOT_NUM] = {
  {GM_SLOT_0_ADDR, GM_SLOT_0_SPAN, GM_SLOT_0_INT_LINE,
   GM_SLOT_0_BUSNO, GM_SLOT_0_DEVNO, GM_SLOT_0_FUNCNO},
  {GM_SLOT_1_ADDR, GM_SLOT_1_SPAN, GM_SLOT_1_INT_LINE,
   GM_SLOT_1_BUSNO, GM_SLOT_1_DEVNO, GM_SLOT_1_FUNCNO},
  {GM_SLOT_2_ADDR, GM_SLOT_2_SPAN, GM_SLOT_2_INT_LINE,
   GM_SLOT_2_BUSNO, GM_SLOT_2_DEVNO, GM_SLOT_2_FUNCNO},
  {GM_SLOT_3_ADDR, GM_SLOT_3_SPAN, GM_SLOT_3_INT_LINE,
   GM_SLOT_3_BUSNO, GM_SLOT_3_DEVNO, GM_SLOT_3_FUNCNO}
};

static MYRINET_CARD myrinet_cards[GM_ARCH_NUM_BOARDS_SUPPORTED];


/****************************************************************
 * gm_arch_pci_span
 *    Returns the amount of memory used by the pci card
 ****************************************************************/

unsigned int
_gm_arch_pci_span (int pcibusno, int pcideviceno, int pcifuncno)
{
  unsigned int bits;
  unsigned int save;

  if (pciConfigInLong (pcibusno, pcideviceno, pcifuncno,
		       PCI_CFG_BASE_ADDRESS_0, &save) != OK)
    return 0;
  if (pciConfigOutLong (pcibusno, pcideviceno, pcifuncno,
			PCI_CFG_BASE_ADDRESS_0, 0xffffffff) != OK)
    return 0;
  if (pciConfigInLong (pcibusno, pcideviceno, pcifuncno,
		       PCI_CFG_BASE_ADDRESS_0, &bits) != OK)
    return 0;
  if (pciConfigOutLong (pcibusno, pcideviceno, pcifuncno,
			PCI_CFG_BASE_ADDRESS_0, save) != OK)
    return 0;

  return ~(bits & 0xfffffff0) + 1;
}

/****************************************************************
 * gm_arch_pci_find_next_unit
 *    Finds the next Myrinet card and fills in pcibusno,
 *      pcideviceno, and pcifuncno.
 *    Arguments:
 *      reset_find - if true resets the find to look for
 *       the first device.
 *    Returns OK if a board was found, ERROR if there were
 *      no more boards, or if we have already found the
 *      maximum number of boards supported.
 ****************************************************************/

STATUS
_gm_arch_pci_find_next_unit (BOOL reset_find, int *pcibusno,
			     int *pcideviceno, int *pcifuncno)
{
  static int unit = 0;
  static int units_found = 0;

  if (reset_find)
    {
      unit = 0;
      units_found = 0;
    }


  if (units_found >= GM_ARCH_NUM_BOARDS_SUPPORTED)
    {
      GM_NOTE (("Trying to find one more unit than the max allowed\n"));
      return (ERROR);
    }

  for (; unit < GM_ARCH_NUM_BOARDS_SUPPORTED * 3; unit++)
    {
      if (unit < GM_ARCH_NUM_BOARDS_SUPPORTED)
	{
	  if (pciFindDevice (GM_PCI_VENDOR_MYRICOM,
			     GM_PCI_DEVICE_MYRINET,
			     unit % GM_ARCH_NUM_BOARDS_SUPPORTED,
			     pcibusno, pcideviceno, pcifuncno) == OK)
	    break;
	}
      else if (unit < GM_ARCH_NUM_BOARDS_SUPPORTED * 2)
	{
	  if (pciFindDevice (GM_PCI_VENDOR_MYRICOM2,
			     GM_PCI_DEVICE_MYRINET,
			     unit % GM_ARCH_NUM_BOARDS_SUPPORTED,
			     pcibusno, pcideviceno, pcifuncno) == OK)
	    break;
	}
      else if (unit < GM_ARCH_NUM_BOARDS_SUPPORTED * 3)
	{
	  if (pciFindDevice (GM_PCI_VENDOR_MYRICOM2,
			     0,
			     unit % GM_ARCH_NUM_BOARDS_SUPPORTED,
			     pcibusno, pcideviceno, pcifuncno) == OK)
	    break;
	}
    }

  if (unit < GM_ARCH_NUM_BOARDS_SUPPORTED * 3)
    {
      unit++;
      units_found++;
      return (OK);
    }
  else
    return (ERROR);
}

/****************************************************************
 * gm_arch_pci_fill_card_info
 *   Finds the slot described by the busno, devno, and funcno
 *    and fills the myrinet_cards structure for unit with
 *    the rest of the information about that slot.
 *   Returns ERROR if no slot description matches the busno,
 *    devno, and funcno that was supplied.
 ****************************************************************/

STATUS _gm_arch_pci_fill_unit_info (unit, busno, devno, funcno)
{
  int i;

  for (i = 0; i < GM_ARCH_MAX_PCI_SLOT_NUM; i++)
    {
      if (busno == slot_desc[i].pcibusno
	  && devno == slot_desc[i].pcideviceno
	  && funcno == slot_desc[i].pcifuncno)
	{
	  myrinet_cards[unit].pcibusno = slot_desc[i].pcibusno;
	  myrinet_cards[unit].pcideviceno = slot_desc[i].pcideviceno;
	  myrinet_cards[unit].pcifuncno = slot_desc[i].pcifuncno;
	  myrinet_cards[unit].addr = slot_desc[i].addr;
	  myrinet_cards[unit].span = slot_desc[i].span;
	  myrinet_cards[unit].intline = slot_desc[i].intline;
	  return (OK);
	}
    }
  return (ERROR);
}

/****************************************************************
 * gm_arch_pci_find_and_configure
 *    Finds and configures each myrinet pci card found
 *    Returns: number of units found, or -1 for error.
 ****************************************************************/

int
_gm_arch_pci_find_and_configure (void)
{
  int pcibusno = 0;
  int pcideviceno = 0;
  int pcifuncno = 0;
  UINT32 dev_addr;

  int unit;
  BOOL reset_find;

  /* initialize the global number of units */
  myrinet_units = 0;

  GM_INFO (("Maximum Myrinet Cards Supported = %d\n",
	    GM_ARCH_NUM_BOARDS_SUPPORTED));
  GM_PRINT (1, ("Scanning PCI bus\n"));
  fflush (stdout);

  /* find the pci devices and put them in our myrinet_cards table */
  for (unit = 0; unit < GM_ARCH_NUM_BOARDS_SUPPORTED; unit++)
    {
      GM_PRINT (1, ("Looking for unit %d\n", unit));
      fflush (stdout);

      if (unit == 0)
	reset_find = TRUE;
      else
	reset_find = FALSE;

      if (_gm_arch_pci_find_next_unit (reset_find, &pcibusno, &pcideviceno,
				       &pcifuncno) == OK)
	{

#if !GM_SYSTEM_HAS_BIOS
	  /* we don't have a bios, so look up the information in our
	     slot description table */
	  if (_gm_arch_pci_fill_unit_info (myrinet_units, pcibusno,
					   pcideviceno, pcifuncno) == OK)
	    {
	      myrinet_units++;
	    }
	  else
	    {
	      GM_INFO (("*******************************************\n"));
	      _GM_INFO (("* Myrinet Card found in slot described by:\n"));
	      _GM_INFO (("* busno  = %2d\n", pcibusno));
	      _GM_INFO (("* devno  = %2d\n", pcideviceno));
	      _GM_INFO (("* funcno = %2d\n", pcifuncno));
	      _GM_INFO (("* Maybe you need to add this information \n"));
	      _GM_INFO (("* to gm_bsp_support.h ???\n"));
	      _GM_INFO (("*******************************************\n"));
	    }
#else /* GM_SYSTEM_HAS_BIOS */
	  myrinet_cards[unit].pcibusno = pcibusno;
	  myrinet_cards[unit].pcideviceno = pcideviceno;
	  myrinet_cards[unit].pcifuncno = pcifuncno;

	  myrinet_cards[unit].span =
	    _gm_arch_pci_span (pcibusno, pcideviceno, pcifuncno);
	  pciConfigInLong (pcibusno, pcideviceno, pcifuncno,
			   PCI_CFG_BASE_ADDRESS_0, &dev_addr);
	  dev_addr &= PCI_MEMBASE_MASK;
	  myrinet_cards[unit].addr = dev_addr;

	  pciConfigInByte (pcibusno, pcideviceno, pcifuncno,
			   PCI_CFG_DEV_INT_LINE,
			   &myrinet_cards[unit].intline);
	  myrinet_units++;
#endif /* GM_SYSTEM_HAS_BIOS */
	}
    }

  /* now that we have found all the devices, configure them */
  for (unit = 0; unit < myrinet_units; unit++)
    {

      if (pciDevConfig (myrinet_cards[unit].pcibusno,
			myrinet_cards[unit].pcideviceno,
			myrinet_cards[unit].pcifuncno,
			NULL, myrinet_cards[unit].addr,
			(PCI_CMD_MASTER_ENABLE | PCI_CMD_MEM_ENABLE)) != OK)
	{
	  GM_NOTE (("Failed to configure PCI device\n"));
	  fflush (stdout);
	  return (-1);
	}

      pciConfigOutLong (pcibusno, pcideviceno, pcifuncno,
			PCI_CFG_BASE_ADDRESS_0, myrinet_cards[unit].addr);
      /* write zero to Base Adress 1 to make 64 bit addressable cards
         work under vxworks */
      pciConfigOutLong (pcibusno, pcideviceno, pcifuncno,
			PCI_CFG_BASE_ADDRESS_1, 0x0);
      pciConfigOutByte (pcibusno, pcideviceno, pcifuncno,
			PCI_CFG_DEV_INT_LINE, myrinet_cards[unit].intline);

      GM_INFO (("=============================\n"));
      _GM_INFO (("| Myrinet PCI conf unit %2d  |\n", unit));
      _GM_INFO (("=============================\n"));
      _GM_INFO (("| pcifuncno    :   %8d |\n", myrinet_cards[unit].pcifuncno));
      _GM_INFO (("| pcibusno     :   %8d |\n", myrinet_cards[unit].pcibusno));
      _GM_INFO (("| pcideviceno  :   %8d |\n", myrinet_cards[unit].pcideviceno));
      _GM_INFO (("| pciaddr      : 0x%08x |\n", myrinet_cards[unit].addr));
      _GM_INFO (("| span         : 0x%08x |\n", myrinet_cards[unit].span));
      _GM_INFO (("| intline      :       0x%02x |\n", myrinet_cards[unit].intline));
      _GM_INFO (("=============================\n"));

#if GM_CPU_i386
      /* Add the pci space to the memory map */
      if (sysMmuMapAdd ((void *) myrinet_cards[unit].addr,
			myrinet_cards[unit].span,
			VM_STATE_MASK_VALID | VM_STATE_MASK_WRITABLE
			| VM_STATE_MASK_CACHEABLE,
			VM_STATE_VALID | VM_STATE_WRITABLE
			| VM_STATE_CACHEABLE_NOT) == ERROR)
	{
	  GM_NOTE (("Failed to add myrinet card to memory map\n"));
	  return (-1);
	}
#endif /* GM_CPU_i368 */
    }
  return (myrinet_units);
}
#endif /* GM_SUPPORT_PCI */
/****************************************************************
 * gmVxWorksInit
 *    Initializes gm under vxWorks
 ****************************************************************/
STATUS gmVxWorksInit (void)
{

  gm_instance_state_t *is;

  int intline;
  int units;
  int status;
  int i;

  GM_INFO (("GM driver version %s build %s\n", _gm_version, _gm_build_id));
  fflush (stdout);

  for (i = 0; i < GM_ARCH_MAX_INSTANCE; i++)
    {
      gm_instance_initialized[i] = 0;
    }

#if GM_PCI

  GM_PRINT (1, ("Looking for Myrinet PCI cards\n"));
  fflush (stdout);

  units = _gm_arch_pci_find_and_configure ();

  GM_INFO (("Found %d Myrinet PCI cards\n", units));
  fflush (stdout);

  if (units == 0)
    {
      GM_INFO (("No myrinet cards found\n"));
      fflush (stdout);
      goto abort_with_nothing;
    }
  else if (units < 0)
    {
      GM_NOTE (("Error initializing Myrinet cards\n"));
      fflush (stdout);
      goto abort_with_nothing;
    }

#if GM_CPU_i386
  /* We now need to reinitialize the basic mmu */
  if (vmBaseGlobalMapInit (&sysPhysMemDesc[0],
			   sysPhysMemDescNumEnt, TRUE) == NULL)
    {
      GM_NOTE (("Failed to reinitialize the memory map\n"));
      goto abort_with_nothing;
    }
#endif /* GM_CPU_i386 */

#else /* !GM_PCI */
  /* we don't have pci, assume we have only 1 lanai card */
  units = 1;

  GM_INFO (("PCI not defined. Skipped PCI init\n"));
  fflush (stdout);
#endif /* !GM_PCI */

  /* set the page length ASAP, as it is used all over. */
#if !GM_PAGE_LEN
  if (!GM_PAGE_LEN)
    GM_PAGE_LEN = vmBasePageSizeGet ();
#endif /*GM_PAGE_LEN */

  /* now initialize the instances */

  for (i = 0; i < units; i++)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("Inititializing instance %d\n", gm_num_instance));
      gm_instances[gm_num_instance] = malloc (sizeof (gm_instance_state_t));
      if (gm_instances[gm_num_instance] == NULL)
	{
	  GM_NOTE (("malloc of GM instance struct failed\n"));
	  goto abort_with_nothing;
	}
      is = gm_instances[gm_num_instance];
      bzero ((unsigned char *) is, sizeof (gm_instance_state_t));
      GM_PRINT (1, ("Instance state ptr = %p\n", is));

#if GM_PCI
      is->arch.pcibusno = myrinet_cards[gm_num_instance].pcibusno;
      is->arch.pcideviceno = myrinet_cards[gm_num_instance].pcideviceno;
      is->arch.pcifuncno = myrinet_cards[gm_num_instance].pcifuncno;
#endif
      is->id = gm_num_instance;


      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("GM_PAGE_LEN before call to instance_init is %d\n",
		 GM_PAGE_LEN));
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("Calling gm_instance_init(%p, %d, %d)\n", is,
		 gm_num_instance, GM_SYSTEM_BUS_TYPE));
      if (gm_instance_init (is, gm_num_instance, GM_SYSTEM_BUS_TYPE) !=
	  GM_SUCCESS)
	{
	  GM_INFO (("Could not initialize instance %d.\n", gm_num_instance));
	  gm_instance_initialized[gm_num_instance] = 0;
	  goto abort_with_instance;
	}
      gm_instance_initialized[gm_num_instance] = 1;

#if GM_PCI
      /* Set up the interrupt handler and enable interrupts */
      intline = myrinet_cards[gm_num_instance].intline;
#elif CSPI_MAP26xx
      intline = GM_CSPI_MAP26xx_INT_LINE;
#else
      #error Dont know how to set up the interrupt line for this system
#endif
      
#if GM_CPU_i386
      GM_PRINT (1, ("calling pciIntConnect(%p, %p, %p)\n",
		    INUM_TO_IVEC (INT_NUM_IRQ0 + intline), gm_intr, is));
      status =
	pciIntConnect (INUM_TO_IVEC (INT_NUM_IRQ0 + intline),
		       (void *) gm_intr, (int) is);
      if (status != OK)
	{
	  GM_NOTE (("pciIntConnect failed status = 0x%x\n", status));
	  goto abort_with_instance;
	}

      GM_PRINT (1, ("calling sysIntEnablePIC(0x%x)\n", intline));
      status = sysIntEnablePIC (intline);
      if (status != OK)
	{
	  GM_NOTE (("sysIntEnablePIC failed status = 0x%x\n", status));
	  goto abort_with_interrupt;
	}

#elif GM_CPU_powerpc
      GM_PRINT (1, ("calling intConnect(%p, %p, %p)\n",
		    INUM_TO_IVEC (intline), gm_intr, is));
      status =
	intConnect (INUM_TO_IVEC (intline), (void *) gm_intr, (int) is);
      if (status != OK)
	{
	  GM_NOTE (("intConnect returned bad status = 0x%x\n", status));
	  goto abort_with_instance;
	}

      GM_PRINT (1, ("calling intEnable(0x%x)\n", intline));
      status = intEnable (intline);

      if (status == ERROR)
	{
	  GM_NOTE (("intEnable returned bad status = 0x%x\n", status));
	  goto abort_with_interrupt;
	}

#endif /* GM_CPU_powerpc */

      gm_enable_interrupts (gm_instance_for_id (gm_num_instance));
      gm_instance_initialized[gm_num_instance] = 1;

      GM_INFO (("Init Complete for instance %d.\n", gm_num_instance));
      fflush (stdout);

      gm_num_instance++;
    }

  return (OK);

abort_with_interrupt:
  GM_PRINT (1, ("abort_with_interrupt\n"));
#if GM_CPU_i386
  sysIntDisablePIC (intline);
#elif GM_CPU_powerpc
  intDisable (intline);
#endif /* GM_CPU_PPC */

abort_with_instance:
  GM_PRINT (1, ("abort_with_instance\n"));
  if (gm_instances[gm_num_instance])
    {
      free (gm_instances[gm_num_instance]);
      gm_instances[gm_num_instance] = 0;
    }
abort_with_nothing:
  GM_PRINT (1, ("abort_with_nothing\n"));
  gm_instance_initialized[gm_num_instance] = 1;
  return (ERROR);
}


/*********************************************************************
 * other stuff
 *********************************************************************/


/********** FIX - MAJOR HACKS TO GET PROGRAMS TO RUN ***************/
int
__eabi (void)
{
  GM_PANIC (("\n******** __eabi called\n"));
  return (0);
}

void
_gm_always_assertion_failed (char *assertion, int line, char *file)
{
  printf ("\n******** _gm_always_assertion_failed called\n");
  printf ("********    file = %s   line = %d  assert = %s\n",
	  file, line, assertion);
}

#ifdef DELETE
void
gm_perror (const char *message, gm_status_t gm_status)
{
  printf ("gm_perror: %s status=%d\n", message, gm_status);
}


void
gm_dump_chunks (int unit)
{
  gm_instance_state_t *is;
  unsigned char *cptr;
  is = gm_instance_for_id (unit);

  printf ("lanai.globals = %p\n", is->lanai.globals);

  printf ("_PORT in globals = 0x%x\n",
	  gm_ntok_lp (is, &is->lanai.globals->_PORT));

  printf ("send_chunks\n");
  cptr = (unsigned char *) &is->lanai.globals->send_chunk[0];
  gm_kpio_hex_dump (cptr, 128);

  cptr = (unsigned char *) &is->lanai.globals->send_chunk[1];
  gm_kpio_hex_dump (cptr, 128);

  printf ("recv_chunks\n");
  cptr = (unsigned char *) &is->lanai.globals->recv_chunk[0];
  gm_kpio_hex_dump (cptr, 128);

  cptr = (unsigned char *) &is->lanai.globals->recv_chunk[1];
  gm_kpio_hex_dump (cptr, 128);

  printf ("port 1\n");
  cptr = (unsigned char *) &is->lanai.globals->port[1];
  gm_kpio_hex_dump (cptr, 128);

/*
	printf("port 2\n");
	cptr = (unsigned char *)&is->lanai.globals->port[2];
	gm_hex_dump(cptr,1024);
*/

}
#endif /* DELETE */

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
