/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997, 1998, 1999 by Myricom, Inc.                 *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/*
    gd.c
    source file for gm ifnet driver for digital unix
    finucane@myri.com

    this code depends on the following #defines being defined or not:

    INSIST
    GM_EMULATE_BYTE_DMAS
    GD_USE_DMA_MAP
    GD_USE_IPL  (must be 1)
    
*/

#define INSIST 1
#define GD_USE_DMA_MAP 1 
#define GD_USE_IPL 1

#include "gd.h"

#if (GM_DEBUG > 0 ) || defined (INSIST)
#define insist(e) do { if (!(e)) {printf ("assertion failed in file %s, line %d (%s)\n",__FILE__,__LINE__, #e); goto exception;}} while (0)
#else
#define insist(e) do {if (0) goto exception;} while (0);
#endif


struct controller*gm_osf1_get_controller (int unit);

void gd_printMbuf (struct mbuf*m)
{
  int i, np, length;
  struct mbuf*t;

  printf ("mbuf length is %d\n", m->m_pkthdr.len);
  
  np = 0;
  
  for (t = m; t; t = t->m_next)
  {
    if (t->m_len == 0)
      continue;

    printf ("piece %d\n", np++);

    for (i = 0; i < t->m_len; i++)
      printf ("%02x%c",0xff & ((char*)(t->m_data))[i], i % 8 == 7 ? '\n' : ' ');
    printf ("\n");
  }
  exception: return;
}

int gd_if_done (struct ifnet*ifp)
{
  return(0);
}

static gd_Device*gd_devices [GD_MAX_DEVICES];

int gd_isEmpty (gd_Queue*q)
{
  insist (q);
  insist (q->size >= 0);
  insist (q->head && q->size || !q->head && !q->size);
  return !q->head;
  exception: return 0;
}

int gd_put (gd_Queue*q, gd_Item*p)
{
  insist (q && p);
  insist (q->size >= 0);
  
  p->next = 0;
  if (!q->tail)
  {
    insist (!q->head && q->size == 0);
    q->tail = q->head = p;
  }
  else
  {
    insist (q->head && q->size);
    q->tail->next = p;
    q->tail = p;
  }
  q->size++;
  return 1;
  
  exception: return 0;
}

gd_Item*gd_get (gd_Queue*q)
{
  gd_Item*p;
  insist (q);
  insist (q->size >= 0);
  
  if (!q->head)
  {
    insist (!q->tail && q->size == 0);
    return 0;
  }
  p = q->head;
  q->head = q->head->next;
  if (!q->head)
  {
    insist (p == q->tail && q->size == 1);
    q->tail = 0;
  }
  q->size--;
  return p;
  exception: return 0;
}

gd_Item*gd_newItem (gd_Device*d)
{
  gd_Item*p = 0;
  insist (d);
  
  MALLOC (p, gd_Item*, sizeof (gd_Item), M_DEVBUF, M_NOWAIT | M_ZERO);
  insist (p);
  p->device = d;

  return p;
  exception: return 0;
}

void gd_freeItem (gd_Item*p)
{
  insist (p);
  FREE (p, M_DEVBUF);
  exception:;
}

gd_Item*gd_newBufferItem (gd_Device*d, int size)
{
  gd_Item*p;
  sg_entry_t entry;
  int i, j;
  
  insist (d);
  insist (d->_controller);
  
  p = gd_newItem (d);
  insist (p);
   
  p->osize = size;
  MALLOC (p->ma, char*, size, M_DEVBUF, M_WAITOK);
  if (!p->ma) {
    printf("gm: gd_newBufferItem MALLOC(size=%d) failed\n", size);
  }
  insist (p->ma);

  p->va = GD_ROUND_POINTER (p->ma);
  insist (p->va);

  if (!d->useDmaMap)
  {
    p->pa = GD_VTOP (p->va);
  }
  else
  {
    if (!dma_map_alloc (size, d->_controller, &p->dmaHandle, DMA_ALL | DMA_IN | DMA_OUT | DMA_CONTIG))
    {
      GM_NOTE (("gd_newBufferItem: dma_map_alloc failed. size was %d\n", size));

      FREE (p->ma, M_DEVBUF);
      gd_freeItem (p);
      return 0;
    } 
    if (!dma_map_load (size, (unsigned long) p->va, 0, d->_controller, &p->dmaHandle, 0, 
			DMA_ALL | DMA_IN | DMA_OUT | DMA_CONTIG))
    {
      GM_NOTE (("gd_newBufferItem: dma_map_load failed. size was %d\n", size));
      
      dma_map_dealloc (p->dmaHandle);
      FREE (p->ma, M_DEVBUF);
      gd_freeItem (p);
      return 0;
    }
  
    if (!(entry = dma_get_curr_sgentry (p->dmaHandle)))
    {
      GM_NOTE (("gd_newBufferItem: dma_get_curr_sgentry failed\n"));
      
      dma_map_unload (0, p->dmaHandle);
      dma_map_dealloc (p->dmaHandle);

      FREE (p->ma, M_DEVBUF);
      gd_freeItem (p);
      return 0;
    }
    
    if (entry->bc < size)
    {
	GM_NOTE(("gd_newBufferItem: got multiple entries from dma_map_load??\n"));

	dma_map_unload (0, p->dmaHandle);
	dma_map_dealloc (p->dmaHandle);

	FREE (p->ma, M_DEVBUF);
	gd_freeItem (p);
	return 0;
    }

    p->pa = entry->ba;
    
    for (i = 0; i < GM_MAX_ETHERNET_GATHER_CNT; i++)
    {
      if (!dma_map_alloc (size, d->_controller, &p->pieces [i], DMA_ALL | DMA_IN | DMA_OUT | DMA_CONTIG))
      {
	GM_NOTE (("gd_newBufferItem: dma_map_alloc failed\n"));
	
	dma_map_unload (0, p->dmaHandle);
	dma_map_dealloc (p->dmaHandle);
	
	for (j = 0; j < i; j++)
	  dma_map_dealloc (p->pieces [j]);
	
	FREE (p->ma, M_DEVBUF);
	gd_freeItem (p);
	return 0;
      }
    }
  }
  insist (p->pa);
  return p;
  exception: return 0;
}

int gd_dmaUnloadPieces (gd_Item*p)
{
  int i;
  insist (p && p->device);
  insist (p->numPieces >= 0 && p->numPieces < GM_MAX_ETHERNET_GATHER_CNT);

  if (p->device->useDmaMap)
  {
    for (i = 0; i < p->numPieces; i++)
      dma_map_unload (0, p->pieces [i]);
  }
  p->numPieces = 0;
  return 1;
  exception: return 0;
}

int gd_dmaLoadPieces (gd_Item*p, struct mbuf*m, gm_ethernet_segment_descriptor_t*pieces)
{
  struct mbuf*t;
  sg_entry_t entry;

  insist (p && p->device);
  insist (p->numPieces == 0);
  insist (m);

  p->m = m;
  p->numPieces = 0;
  
  for (t = m; t; t = t->m_next)
  {
    if (t->m_len == 0)
      continue;
    
    insist (p->numPieces >= 0 && p->numPieces < GM_MAX_ETHERNET_GATHER_CNT);

    if (p->device->useDmaMap)
    {
      int rv;
      insist (p->dmaHandle);
      
      if (!dma_map_load (t->m_len, (unsigned long) t->m_data, 0, p->device->_controller, 
		&p->pieces [p->numPieces], 0, DMA_ALL | DMA_IN | DMA_OUT | DMA_CONTIG))
      {
	GM_NOTE (("gd_dmaLoadPieces: dma_map_load failed on virtual address 0x%p, size was %d\n", 
		  t->m_data, t->m_len));
	return 0;
      }
      if (!(entry = dma_get_curr_sgentry (p->pieces [p->numPieces])))
      {
	GM_NOTE (("gd_dmaLoadPieces: dma_get_curr_sgentry failed on %d-th piece.", p->numPieces));
	dma_map_unload (0, p->pieces [p->numPieces]);
	return 0;
      }

     if (entry->bc < t->m_len)
     {
	GM_NOTE(("gd_dmaLoadPieces: got multiple entries from dma_map_load??\n"));
	dma_map_unload (0, p->pieces [p->numPieces]);
	return 0;
     }

      pieces [p->numPieces].ptr = gm_hton_dp ((gm_dp_t) entry->ba);
      pieces [p->numPieces].len.n =  htonl /*gm_hton_u32*/ (t->m_len);
    }
    else
    {
      pieces [p->numPieces].ptr = gm_hton_dp ((gm_dp_t) GD_VTOP (t->m_data));
      pieces [p->numPieces].len.n = htonl /*gm_hton_u32*/ (t->m_len);
    }
   
    GM_PRINT (GM_PRINT_LEVEL >= 5,("dmaload:  pieces[%d] ==> 0x%x  %d\n", p->numPieces, 
		GD_VTOP (t->m_data), t->m_len));

    p->numPieces++;
  }
  return p->numPieces;
  exception: return 0;
}

void gd_freeBufferItem (gd_Item*p)
{
  int i;

  insist (p);
  
  insist (p->va);
  FREE (p->ma, M_DEVBUF);

  if (p->device->useDmaMap)
  {  
    dma_map_dealloc (p->dmaHandle);
    dma_map_unload (0, p->dmaHandle);
    
    for (i = 0; i < GM_MAX_ETHERNET_GATHER_CNT; i++)
      dma_map_dealloc (p->pieces [i]);
  }
  gd_freeItem (p);
  exception:;
}

int gd_feedGm (gd_Device*d)
{
  gd_Item*p;
  gm_ethernet_segment_descriptor_t piece;
  
  insist (d && d->opened);

  while (d->pendingReceives.size < GM_NUM_ETHERNET_RECV_TOKENS && (p = gd_get (&d->freeReceives)))
  {
    p = gd_newMbufItem (d, p);
    insist (p);
    
    piece.ptr = gm_hton_dp ((gm_dp_t) p->pa);
    piece.len.n = htonl /*gm_hton_u32*/ (p->osize);
    
    gd_put (&d->pendingReceives, p);

    insist(d->port);
    _gm_provide_ethernet_scatter_list (d->port, 1, &piece);
  }
  return 1;
  exception: return 0;
}

void gd_reclaimReceiveMbufItem (caddr_t address, u_long length, caddr_t argument)
{
  gd_Item*p = (gd_Item*) argument;
#if GD_USE_IPL
  int s;
#endif
 
  if (!p)
    return;
  
  insist (address && argument);
  insist (p && p->device);
  insist (p->m);

#if GD_USE_IPL
  s = splimp ();
#endif
  simple_lock (&p->device->lock);
  
  gd_put (&p->device->freeReceives, p);
 
  if (p->device->opened)
    gd_feedGm (p->device);

  simple_unlock (&p->device->lock);
#if GD_USE_IPL
  splx (s);
#endif

  exception:;
}

gd_Item*gd_newMbufItem (gd_Device*d, gd_Item*p)
{
  insist (d && p);
  
  MGETHDR (p->m, M_DONTWAIT, MT_DATA);
  insist (p->m);
  
  p->m->m_next = 0;
  p->m->m_nextpkt = 0;
  p->m->m_flags |= M_EXT;
  p->m->m_ext.ext_buf = p->m->m_data = p->va;
  p->m->m_ext.ext_size = p->osize;
  p->m->m_ext.ext_free = gd_reclaimReceiveMbufItem;
  p->m->m_ext.ext_arg = (void*)p;
  p->m->m_ext.ext_ref.forw = p->m->m_ext.ext_ref.back = &p->m->m_ext.ext_ref;

  return p;

  exception: return 0;
}

void gd_freeMbuf (gd_Item*p)
{
  struct mbuf*m;
  insist (p);
  insist (p && p->m);
  
  p->m->m_ext.ext_arg = 0;
  p->m->m_flags = 0;
  MFREE (p->m, m);

  exception:;
}

void gd_freeMbufItem (gd_Item*p)
{
  insist (p && p->m);
  gd_freeMbuf (p);
  gd_freeItem (p);
  exception:;
}

gd_Device*gd_newDevice (struct controller*_controller)
{
  gd_Device*d = 0;
  int i;
 
  insist (_controller);

  MALLOC (d, gd_Device*, sizeof (gd_Device), M_DEVBUF, M_NOWAIT | M_ZERO);
  insist (d);
 
  d->port = NULL; 
  d->_controller = _controller;

#ifdef GD_USE_DMA_MAP
  d->useDmaMap = 1;
#else
  d->useDmaMap = 0;
#endif

  insist (gd_isEmpty (&d->pendingSends));
  insist (gd_isEmpty (&d->pendingReceives));
  
  strncpy (d->name, GD_NAME, GD_NAME_LEN);  
  
  for (i = 0; i < GD_NUM_SEND_BUFFERS; i++)
    gd_put (&d->freeSends, gd_newBufferItem (d, GD_BUFFER_SIZE));
  
  for (i = 0; i < GD_NUM_RECEIVE_BUFFERS; i++)
    gd_put (&d->freeReceives, gd_newBufferItem (d, GD_BUFFER_SIZE));
  
  insist (d->freeReceives.size == GD_NUM_RECEIVE_BUFFERS);
  insist (d->freeSends.size == GD_NUM_SEND_BUFFERS);
  
  return d;
  exception: return 0;
}

int gd_if_init (int unit)
{
  return ESUCCESS;
}

int gd_if_start (struct ifnet*ifp)
{
  struct mbuf*m,*t;
#if GD_USE_IPL
  int s;
#endif
  int r=0;
  gd_Device*d = (gd_Device*) ifp;
  
  insist (ifp);

#if GD_USE_IPL
  s = splimp ();
#endif
  
  GM_PRINT (GM_PRINT_LEVEL >= 5,("gd_if_start(%p) called\n",ifp));

  if (!simple_lock_try (&d->lock))
  {
    GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_start(%p) returning, couldn't get simple_lock\n",ifp));
#if GD_USE_IPL
    splx (s);
#endif
    return 0;
  }
  
  if (!(m = ifp->if_snd.ifq_head)) 
  {
    GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_start(%p) returning, nothing in send Q\n",ifp));
    simple_unlock (&d->lock);
#if GD_USE_IPL
    splx (s);
#endif
    return 0;
  }
  
  while (1) 
  {
     IF_DEQUEUE (&ifp->if_snd, m);

     if (m)
     {
       r = gd_send (d, m);
       if (!r)
       {
         break;
       }
     }
     else 
     {
       break;
     }
  }
  
  simple_unlock (&d->lock);
#if GD_USE_IPL
  splx (s);
#endif
  
  return r ? ESUCCESS : ENOBUFS;
  
  exception: return 0; 
}

int gd_if_watchdog (int unit)
{
  return(0);
}

int gd_if_reset (int unit, int dc)
{
  insist (unit >= 0 && unit < GD_MAX_DEVICES);
  insist (gd_devices [unit]);
  
  gd_devices [unit]->gd_if.if_flags |= IFF_RUNNING;
  
  return ESUCCESS;
  exception: return 0;
}

int gd_if_ioctl (struct ifnet*ifp, unsigned command, caddr_t data)
{
#if GD_USE_IPL
  int s;
#endif
  gd_Device*d = (gd_Device*) ifp;
  struct ifdevea*ifd = (struct ifdevea*) data;
  struct ctrreq*ctr = (struct ctrreq*) data;
  struct ifreq*ifr = (struct ifreq*) data;
  u_short ifmtu;
  int status = ESUCCESS;
  int do_start = 0;
  
  insist (d);

  GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) called  (d = %p)\n",
		ifp,command,data,d));
  
#if GD_USE_IPL
  s = splimp ();
#endif
  simple_lock (&d->lock);

  switch (command)
  {
    case SIOCENABLBACK:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCENABLBACK\n",ifp,command,data));
      ifp->if_flags |= IFF_LOOPBACK;
      break;

    case SIOCDISABLBACK:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCDISABLBACK\n",ifp,command,data));
      ifp->if_flags &= ~IFF_LOOPBACK;
      break;

    case SIOCRPHYSADDR:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCRPHYSADDR \n",ifp,command,data));
      bcopy (d->gd_address, ifd->current_pa, 6);
      bcopy (d->gd_address, ifd->default_pa, 6);
      break;

    case SIOCRDCTRS:
    case SIOCRDZCTRS:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCRDxCTRS\n",ifp,command,data));
      ctr->ctr_ether = d->gd_ctrblk;
      ctr->ctr_type = CTR_ETHER;
      ctr->ctr_ether.est_seconds = (time.tv_sec - d->gd_ztime) > 0xfffe ? 0xffff : (time.tv_sec - d->gd_ztime);
      if (command == SIOCRDZCTRS)
      {
	d->gd_ztime = time.tv_sec;
	bzero (&d->gd_ctrblk, sizeof (struct estat));
      }
      break;

    case SIOCSIFADDR:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCSIFADDR\n",ifp,command,data));
#ifdef DELETE_FELDY
      /* why is this here? - feldy */  ifp->if_flags |= IFF_UP;
      if (!d->opened) 
      {
        gd_open(d);
        do_start = 1;
      }
#endif /* DELETE_FELDY */
      break;

    case SIOCSIFFLAGS:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCSIFFLAGS\n",ifp,command,data));
      if (ifr->ifr_flags & IFF_UP)
      {
        GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) IFF_UP - call gd_open\n",
		ifp,command,data));
	gd_open (d);
	do_start = 1;
      }
      else 
      {
        GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) IFF_DOWN - call gd_close \n",
		ifp,command,data));
	gd_close (d);
      }
      break;

    case SIOCSIPMTU:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCSIPMTU\n",ifp,command,data));
      bcopy (ifr->ifr_data, (u_char*) &ifmtu, sizeof (u_short));
      
      if (ifmtu > GM_IP_MTU || ifmtu < 1)
	status = EINVAL;
      else
	ifp->if_mtu = ifmtu;
      break;

    case SIOCADDMULTI:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) SIOCADDMULTI\n",ifp,command,data));
      status = ESUCCESS;
      break;
      
    default:
      GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) default??\n",ifp,command,data));
      status = ESUCCESS;
  }
 
  simple_unlock (&d->lock);
#if GD_USE_IPL
  splx (s);
#endif
 
  /* we do this here after unlocking to prevent a double-lock in gd_if_start */
  if (do_start) 
  {
     GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) IFF_UP - call gd_if_start\n",
		ifp,command,data));
     gd_if_start (ifp);
  }

  GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_if_ioctl(%p, 0x%x, %p) returning status = 0x%x\n",
		ifp,command,data,status)); 
  return status;


  exception:return 0;
}

int gd_detach (int unit)
{
  gd_Device*d;
  insist (unit >= 0 && unit < GD_MAX_DEVICES);
  d = gd_devices [unit];
  insist (d);
  
  gd_freeDevice (d);
  gd_devices [unit] = 0;
  
  return 1;
  exception: return 0;
}

int gd_attach (int unit)
{
  int i, r;
  struct ifnet*ifp;
  struct sockaddr_in*sin;
  gd_Device*d;

  d  = gd_newDevice (gm_osf1_get_controller (unit));
  insist (d);
  
  d->unit = unit;
  
#if GM_EMULATE_BYTE_DMAS
  d->forceAlignment = 0;
#else
  d->forceAlignment = 1;
#endif
/* feldy */
/*
  GM_PRINT (1,("gd_attach - forcing copies on all packets\n"));
  d->forceAlignment = 1;
*/

  r = gd_open (d);
  insist (r);
  
  ifp = &d->gd_if;
  
  d->gd_ac.ac_bcastaddr = (u_char *) etherbroadcastaddr;
  d->gd_ac.ac_arphrd = ARPHRD_ETHER;

  ifp->if_addrlen = 6;
  bcopy (d->address, d->gd_address, 6);
  ifp->if_hdrlen = sizeof (struct ether_header);
  ifp->if_unit = d->unit;
  ifp->if_name = d->name;
  ifp->if_mediamtu = GM_ETHERNET_MTU;
  ifp->if_mtu = GM_IP_MTU;
  ifp->if_type = IFT_ETHER;
  ifp->if_flags = IFF_NOTRAILERS | IFF_SIMPLEX | IFF_BROADCAST /*| IFF_MULTICAST */ ;
  ((struct arpcom *) ifp)->ac_flag = 0;
  sin = (struct sockaddr_in *) &ifp->if_addr;
  sin->sin_family = AF_INET;
  ifp->if_init = gd_if_init;
  ifp->if_output = ether_output;
  ifp->if_start = gd_if_start;
  ifp->if_ioctl = gd_if_ioctl;
  ifp->if_timer = 0;
  ifp->if_done = gd_if_done;
  ifp->if_watchdog = gd_if_watchdog;
  ifp->if_reset = gd_if_reset;
  ifp->if_trustgrp = IFG_ANY;
  
  ifp->if_baudrate = 1280000000;
  ifp->if_version = d->name;
  attachpfilter (&(d->ed));

  simple_lock_init (&d->lock);

  ifp->if_affinity = NETMASTERCPU;  /* NETALLCPU or NETMASTERCPU to funnel */
  ifp->lk_softc = &d->lock;

  /*
   * Initialize the if_snd queue lock and maxlen fields.  This is
   * unique to pcmcia since pcmcia is probed so late (it misses the
   * the common ifnet initialization).  This is also the case for
   * dynamically configured LAN drivers.
   */
  IFQ_LOCKINIT_SPLIMP(&ifp->if_snd);

  if_attach (ifp);

  gd_devices [unit] = d;
  
  ifp->if_snd.ifq_maxlen = IFQ_MAXLEN;

  ifp->if_flags &= ~IFF_OACTIVE;
  ifp->if_flags |= IFF_RUNNING;
 
  r = gd_close (d);
  insist (r);
 
   
  return 1;
  exception: return 0;
}

int gd_close (gd_Device*d)
{  
  gd_Item*p;
  
  insist (d);
  /* insist (d->opened); */
  
  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_close(%p) called\n",d));

  if (!d->opened)
  {
    GM_WARN (("gd_close(%p) called, but !opened??\n",d));
  }

  if (d->port)
  {
	gm_ethernet_set_sent_intr_callback (d->port, 0, 0);
  	gm_ethernet_set_recv_intr_callback (d->port, 0, 0);

  	gm_close (d->port);
  	d->port = NULL;
  	d->opened = 0;
  }

  while (p = gd_get (&d->pendingReceives))
    gd_put (&d->freeReceives, p);
  
  while (p = gd_get (&d->pendingSends))
  {
    if (p->m)
    {
      m_freem (p->m);
      p->m = 0; 
    }
    gd_put (&d->freeSends, p);
  }

  ((struct ifnet*) d)->if_flags &= ~IFF_UP;

  return 1;
  exception: return 0;
}

int gd_open (gd_Device*d)
{
  int r;
  
  insist (d);
  insist (!d->opened);
  /*insist (d->freeSends.size == GD_NUM_SEND_BUFFERS);*/
  insist (d->freeReceives.size == GD_NUM_RECEIVE_BUFFERS);
  
  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_open(%p) called\n",d));
  r = gm_open (&d->port, d->unit, GM_ETHERNET_PORT_ID, d->name, GM_API_VERSION_1_0);
  insist (r == GM_SUCCESS);
  insist(d->port);
  
  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_open(%p) called  port = %p\n",d,d->port));

  d->opened = 1;
  
  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_open(%p): calling gm_get_unique_board_id \n",d));
  r = gm_get_unique_board_id (d->port, d->address);
  insist (r == GM_SUCCESS);

  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_open(%p): setting callbacks\n",d));
  gm_ethernet_set_sent_intr_callback (d->port, gd_sendDone, d);
  gm_ethernet_set_recv_intr_callback (d->port, gd_receiveDone, d);

  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_open(%p): feeding\n",d));
  gd_feedGm (d);

  ((struct ifnet*) d)->if_flags |= IFF_UP;

  return 1;
  exception: return 0;
}

void gd_freeDevice (gd_Device*d)
{
  gd_Item*p;
  
  insist (gd_isEmpty (&d->pendingSends));
  insist (gd_isEmpty (&d->pendingReceives));

  while (p = gd_get (&d->freeSends))
    gd_freeBufferItem (p);
  
  while (p = gd_get (&d->freeReceives))
    gd_freeBufferItem (p);

  while (p = gd_get (&d->pendingReceives))
    gd_freeBufferItem (p);

  insist (gd_isEmpty (&d->freeSends));
  insist (gd_isEmpty (&d->freeReceives));

  FREE (d, M_DEVBUF);  
  exception:;
}

int gd_send (gd_Device*d, struct mbuf*m)
{
  struct mbuf *t,*nextPacket;
  gd_Item*p = 0 ;
  char*offset;
  unsigned char*address;
  int numPieces;
  int i, aligned, broadcast;
  gm_ethernet_segment_descriptor_t pieces [GM_MAX_ETHERNET_GATHER_CNT];
  
  insist (d && m);
 
  /* catch any sends that happen after port close */ 
  if (!d->opened) 
  {
    d->gd_if.if_snd.ifq_drops++;
    GM_NOTE (("gd_send: dropping packet %d, GM port not open\n",
	      d->gd_if.if_snd.ifq_drops));
    m_freem (m);
    return(0);
  }

  if (d->pendingSends.size >= GM_NUM_SEND_TOKENS || !(p = gd_get (&d->freeSends)))
  {
    GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_send: dropping packet %d, no free tokens\n",
		d->gd_if.if_snd.ifq_drops));
    d->gd_if.if_snd.ifq_drops++;
    m_freem (m);
    return 0;
  }
  insist (p);
  
  numPieces = p->size = 0;
  aligned = 1;
  
  for (t = m; t; t = t->m_next)
  {
    if (t->m_len == 0)
      continue;
    
    insist (t->m_data);
    
    numPieces++;
    p->size += t->m_len;
    
    aligned = aligned && GD_ALIGNED ((unsigned)t->m_data) && GD_ALIGNED (t->m_len);
  }

  if (p->size > (GM_IP_MTU + sizeof (struct ether_header) + 2))
  {
    d->gd_if.if_oerrors++;
    GM_NOTE (("gd_send: dropping packet %d  len=%d (too big).\n",
	      d->gd_if.if_snd.ifq_drops, p->size));

    d->gd_if.if_snd.ifq_drops++;
    m_freem (m);
    gd_put (&d->freeSends, p);
    return 0;
  } 

  address = (unsigned char*) m->m_data;  
  broadcast = address [0] & 0x1;

  GM_PRINT (GM_PRINT_LEVEL >= 3,("gd_send: numPieces=%d  size=%d  aligned=%d\n",
			numPieces,p->size,aligned));

  if ((numPieces >= GM_MAX_ETHERNET_GATHER_CNT) || (!aligned && d->forceAlignment))
  {
    GM_PRINT (GM_PRINT_LEVEL >= 3,("gd_send: copying to single buffer num=%d align=%d\n",
			numPieces,aligned));
    p->m = 0;
    
    offset = p->va;
    for (t = m; t; t = t->m_next)
    {
      if (t->m_len == 0)
	continue;      
     
      bcopy (t->m_data, offset, t->m_len);
      offset += t->m_len;
    }
    insist (offset ==  p->va + p->size);
    
    pieces [0].ptr = gm_hton_dp ((gm_dp_t) p->pa);
    pieces [0].len.n = htonl /*gm_hton_u32*/ (GD_ROUND_INT (p->size));
    
    numPieces = 1;
    m_freem (m);
  }
  else
  {
    numPieces = gd_dmaLoadPieces (p, m, pieces);
  }

  insist (numPieces <= GM_MAX_ETHERNET_GATHER_CNT);
 
  gd_put (&d->pendingSends, p);
  insist (d->pendingSends.size <= GM_NUM_SEND_TOKENS);
  
  insist(d->port);

  if (broadcast)
  {
    gm_ethernet_broadcast (d->port, numPieces, pieces, 1);
    GM_PRINT (GM_PRINT_LEVEL >= 3,("gm_ethernet_broadcast called numPieces=%d\n",numPieces));
  }
  else
  {
    gm_ethernet_send (d->port, numPieces, pieces, address, 1);
    GM_PRINT (GM_PRINT_LEVEL >= 3,("gm_ethernet_send called numPieces=%d\n",numPieces));
  }

  return 1;
  exception: return 0;
}

void gd_sendDone (void *ptr)
{
  gd_Device* d = (gd_Device*) ptr;
  gd_Item*p;
  struct mbuf*m;
  
  insist (d);

  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_sendDone called\n"));

  /*interrupt already locked by GM, so we don't need splimps and splxes*/
  simple_lock (&d->lock);

  p = gd_get (&d->pendingSends);
  insist (p);
  insist ((p->size > 0) && (p->size <= (GM_IP_MTU+sizeof(struct ether_header)+2)));
  
  d->gd_if.if_opackets++;
  d->gd_ctrblk.est_bloksent++;
  d->gd_if.if_obytes += p->size;
  d->gd_ctrblk.est_bytesent += p->size;
  
  if (p->m)
  {
    gd_dmaUnloadPieces (p);
    m_freem (p->m);
    p->m = 0; 
  }
  
  gd_put (&d->freeSends, p);

  while (1) 
  {
     struct mbuf *m;
     int r;
     int count=0;

     IF_DEQUEUE (&d->gd_if.if_snd, m);

     if (m)
     {
       GM_PRINT (GM_PRINT_LEVEL >= 1,("doing send in sendDone count=%d\n",count));
       r = gd_send (d, m); 
       if (!r) 
       {
         break;
       }
     }
     else 
     {
       break;
     }
  }

  simple_unlock (&d->lock);
  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_sendDone completed opkt = %d\n",d->gd_if.if_opackets));
  
  exception: return;
}

void gd_receiveDone (void *ptr, unsigned length, gm_u16_t csum)
{
  gd_Device *d = (gd_Device *)ptr;
  gd_Item*p;
  struct ether_header *eh;
  
  insist (d);
  insist ((length > 0));
  insist(length <= GD_BUFFER_SIZE);

  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_receiveDone called len = %d\n",length));

  /*interrupt already locked by GM, so we don't need splimps and splixes*/
  simple_lock (&d->lock);

  p = gd_get (&d->pendingReceives);
  insist (p);

  if (d->pendingReceives.size < GD_MIN_RECEIVE_BUFFERS)
  {
    gd_feedGm (d);
  }

  if (d->pendingReceives.size < GD_MIN_RECEIVE_BUFFERS)
  {
    GM_PRINT (GM_PRINT_LEVEL >= 2,("gd_receiveDone: dropping size<MIN  (%d < %d)\n",
		d->pendingReceives.size,GD_MIN_RECEIVE_BUFFERS));
    gd_freeMbuf (p);
    gd_put (&d->freeReceives, p);
    gd_feedGm (d);
    d->gd_if.if_snd.ifq_drops++;
  }
  else
  {
    insist (p->m->m_ext.ext_buf == p->va);
    insist (p->m);
    insist (p->va == p->m->m_data);
    insist (p->m->m_ext.ext_buf == p->va);
    insist (p->m->m_ext.ext_size == GD_BUFFER_SIZE);
    insist (p->m->m_ext.ext_free == gd_reclaimReceiveMbufItem);
    insist (p->m->m_ext.ext_arg == (char*) p);
    insist (p->m->m_next == 0);

    eh = (struct ether_header*) (p->va + 2);
    eh->ether_type = htons (eh->ether_type);

    p->m->m_data = p->va + sizeof (struct ether_header) + 2;
    p->m->m_pkthdr.len = p->m->m_len = length - sizeof (struct ether_header) - 2;

    p->m->m_pkthdr.rcvif = &d->gd_if;    
    d->gd_ctrblk.est_bytercvd += length;
    d->gd_ctrblk.est_blokrcvd++;
    d->gd_if.if_ipackets++;
    d->gd_if.if_ibytes += length;    
    
    ether_input (&d->gd_if, eh, p->m);
  }
  
  simple_unlock (&d->lock);
  GM_PRINT (GM_PRINT_LEVEL >= 1,("gd_receiveDone called ipkts = %d\n",d->gd_if.if_ipackets));

  return;

  exception:;
}


