/*
 * @DEC_COPYRIGHT@
 */
/*
 * HISTORY
 * $Log: if_skeleton.c,v $
 * Revision 1.3  1999/12/15 22:39:52  feldy
 * *** empty log message ***
 *
 * Revision 1.2  1999/09/29 22:53:13  feldy
 * Bob's merge of main_with_explicit
 *
 * Revision 1.1.2.1  1999/09/13 23:18:10  feldy
 * *** empty log message ***
 *
 * Revision 1.1.2.2  1997/04/29  18:22:38  Kirt_Gillum
 *  Initial creation.
 *  [1997/04/29  18:09:40  Kirt_Gillum]
 *
 * $EndLog$
 */
#pragma ident "@(#)$RCSfile: if_skeleton.c,v $ $Revision: 1.3 $ (DEC) $Date: 1999/12/15 22:39:52 $"
/*
 * ++
 * Skeleton LAN driver
 * -------------------
 * 
 * This is a template (or skeleton) LAN driver that can be used to 
 * initiate development of new LAN interface device drivers.   Generally,
 * you need to change all symbol references of 'sk' to a two or more letter
 * identifier for your driver, and then you're ready to do real work.
 * 
 * Comments that have #CHOOSE# included provide you with a choice of options
 * of which you'll have to pick the correct one(s) for your supported bus 
 * and/or platform.  Otherwise, there will be comments that will hopefully
 * direct you to plug in the correct values/operations.
 * 
 * Author:
 *  Kirt Gillum (gillum@zk3.dec.com)
 *  Digital UNIX, LAN Driver Development
 *  Digital Equipment Corporation
 *  12/8/95
 * 
 * --
 */

/* 
 * Driver description
 * ==================
 * 
 * Give a brief description of what the driver controls, reference the
 * specification you're using, describe the general nature of the device
 * (programmed I/O, dma, shared memory, etc).
 * 
 * It's also generally useful to describe quirks associated with the
 * device up front (so a reader will be ready for workarounds when getting
 * to the appropriate section/routine).  
 */

/*
 * Include files
 * =============
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/vmmac.h>
#include <vm/vm_kern.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/time.h>
#include <sys/kernel.h>
#include <sys/proc.h>			/* Needed for nonsymmetric drivers. */
#include <sys/sysconfig.h>
#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/if_ether.h>
#include <net/ether_driver.h>
#include <io/common/devdriver.h>
#include <hal/cpuconf.h>
#include <kern/thread.h>		/* For thread access */
#include <kern/sched_prim.h>
#include <kern/lock.h>			/* For locking */

/*
 * #CHOOSE# your bus support include files.   Including all of those that 
 * your device runs on.
 */
#include <io/dec/tc/tc.h>		/* TURBOchannel */
#include <io/dec/eisa/eisa.h>	/* EISA -- Covers ISA bus too */
#include <io/dec/pci/pci.h>		/* PCI */
#include <io/dec/pcmcia/pcmcia.h>	/* PCMCIA */
#include <io/dec/pcmcia/cardinfo.h>		/* PCMCIA */

/*
 * #CHOOSE# the name of your private include file.  Typically this 
 * is something like <io/dec/netif/if_skreg.h>.   The convention of
 * calling this file a 'reg' file relates to the contents of the file
 * being CSR 'register' definitions.  However, feel free to call this
 * anything you want (and at any directory you choose to build your driver
 * out of).
 */
/*#include <io/dec/netif/if_skreg.h> */

/* 
 * Softc structure definition
 * ==========================
 * 
 * In a departure from the statically configured drivers, dynamically loaded
 * drivers must allocate their softc structures without knowing how many
 * supported devices are on any given system.  The system configuration 
 * process loads/calls this driver as needed.
 * 
 * The following fields are common in all (most) LAN driver softcs.   The
 * ether_driver struct must be first in the list.
 * 
 * Multicasting tracking could be made common, but, for now, every driver
 * does it on their own.  This template driver will reference this mechanism
 * in a couple of places throughout the driver.
 * 
 * WARNING:
 *    Counters for medias other than Ethernet are a little more difficult.
 *    See net/if.h, union ctr_ctrs.  Mostly, all counters, characteristics,
 *    and status are stored in this softc.   This ctr_ctrs structure is not
 *    in the 'ether_driver' struct, so you have to accomodate these counters
 *    elsewhere.
 */

struct sk_softc {
	struct ether_driver is_ed;	/* driver */
#define	is_ac	is_ed.ess_ac	/* common part */
#define	ztime	is_ed.ess_ztime	/* Time since last zeroed */
#define	is_if	is_ac.ac_if		/* network-visible interface */
#define	is_addr	is_ac.ac_enaddr	/* hardware address */
	struct lan_media lan_media;	/* Media state/values */
#define lm_media_mode  lan_media.lan_media_mode
#define lm_media_state lan_media.lan_media_state
#define lm_media       lan_media.lan_media
	struct lan_multi is_multi;	/* Multicast table */
#define lan_nmulti     is_multi.lan_nmulti
	vm_offset_t basereg;		/* base register */
	int debug;					/* Set if IFF_DEBUG is set */

	/*
	 * Interrupt handler id
	 */
	ihandler_id_t *hid;			/* set from handler_add */

	/*
	 * Put driver specific fields here.  This includes things like;
	 *      ring pointers
	 *      csr pointers
	 *      rings
	 *      resource maps
	 *      flags
	 *      status
	 *      state
	 */

	/*
	 * NETSYNC lock definition.  All LAN drivers must use this lock (look
	 * in probe routine for initialization).
	 */
	    decl_simple_lock_data(, sk_softc_lock)
};

/*
 * LOCKS
 * =====
 * 
 * Locks help provide exclusive access to data structures on multiprocessor
 * systems.  You must also use IPL to synchronize with interrupt level code
 * within the driver ('s = splimp()') on both uni and multiprocessor systems.
 * 
 * LAN drivers must use a single primitive simple (spin) lock to access the
 * softc. Since network layer code above the driver accesses the ifnet struct
 * defined within the softc structure, the driver communicates the address
 * of the softc-lock to the upper layer via the ifnet structure.
 * 
 * LAN drivers can also be made multi-processor safe by funneling (forcing
 * the driver to always run on the master CPU). To do so, the affinity
 * field within the ifnet structure must be intialized in the attach routine
 * as follows:
 * 
 *  ifp->if_affinity = NETMASTERCPU;
 * 
 * No locks need to be used within the driver if this scheme is used.
 * 
 * To put lock structures in, you enter a field like the following in
 * the softc structure (one per device);
 *      decl_simple_lock (, sk_softc_lock);
 * Declare simple lock information prior to the 'probe' routine via (one
 * per driver);
 *      decl_simple_lock_info(, sk_lock_info);
 * Then you initialize the structure in the 'attach' routine;
 *      simple_lock_setup(&sc->sk_softc_lock, sk_lock_info);
 * To use the lock;
 *      simple_lock(&sc->sk_softc_lock);
 * To release the lock;
 *      simple_unlock(&sc->sk_softc_lock);
 * 
 * The following is an example of the lock information definition.
 */
decl_simple_lock_info(, sk_lock_info);

/*
 * Forward/External references
 * ===========================
 */
extern struct timeval time;
int sk_probe(), sk_attach(), sk_intr(), sk_reset();
int sk_init(), sk_start(), sk_ioctl(), sk_watch();
int sk_rint(), sk_tint();
int sk_start_locked(), sk_init_locked(), sk_reset_locked();

	/*
	 * Implmentation specific functions
	 */
void sk_shutdown();
int sk_create_controller();
void sk_callback_create_controller();

/*
 * Softc and controller arrays
 * ===========================
 * 
 * Unfortunately, these arrays have to be static.   These are used to find
 * out the driver's softc and controller structure when the driver is called
 * with just an 'ifnet' structure.  The ifnet's if_unit field is used as an
 * index into these arrays.  The unfortunate side effect of this is that 
 * there's now a fixed max limit to the number of devices you can have...
 * 
 * These arrays are initialized at probe (only when the driver knows that
 * the probe is going to suceed).
 */
#define  sk_MAXDEV 10
struct sk_softc *sk_softc[sk_MAXDEV] =
{0};
struct controller *sk_info[sk_MAXDEV] =
{0};

int sk_cfg_state = 0;

/*
 * The "driver" structure
 * ======================
 * 
 * See the Digital UNIX driver writer's manual for more information about
 * these fields.
 */
struct driver skdriver =
{
	sk_probe,					/* probe */
	0,							/* slave */
	sk_attach,					/* cattach */
	0,							/* dattach */
	0,							/* go */
	0,							/* addr_list */
	0,							/* dev_name */
	0,							/* dev_list */
	"sk",						/* ctlr_name */
	sk_info						/* ctlr_list */
};

/*
 * Macros, etc.
 * ============
 * 
 * Any driver specific macros, constants, variables (CSR read/write, reset, 
 * etc.).
 */

/*
 * Broadcast address
 * =================
 * 
 * Used in a couple areas of the driver.
 * 
 * etherbroadcastaddr contains this address.  Only uncomment the following
 * if you need to use a local copy.
 */

/*
 * static u_char sk_bcast[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
 */

/*
 * Configuration data
 *
 * See the driver manual associated with the bus(es) you're operating on
 * to get bus specific configuration requirements.  The example listed 
 * here will most likely NOT work for any real driver.
 * 
 * The configuration parameters in this structure are;
 *              name = ascii name of the entry
 *      type = What the data is.  One of;
 *          CFG_ATTR_STRTYPE - null terminated string
 *          CFG_ATTR_INTTYPE - 32-bit integer
 *          CFG_ATTR_UINTTYPE - 32-bit unsigned integer
 *          CFG_ATTR_LONGTYPE - 64-bit integer
 *          CFG_ATTR_ULONGTYPE - 64-bit unsigned integer
 *          CFG_ATTR_BINTYPE - Array of bytes
 *          operation = How the data can be handled.  One of;
 *          CFG_OP_QUERY - means the param is queryable
 *          CFG_OP_CONFIGURE - means the param is setable at load
 *          CFG_OP_RECONFIGURE - means the param can be reset at
 *                       run time
 *      addr = address where the data is stored
 *      min_val = minimum acceptable size of param
 *      max_val = maximum acceptable size of param
 *      val_size = size of param
 * 
 */
unsigned char sk_sd[CFG_ATTR_NAME_SZ] = "sk LAN Device Driver";
unsigned char sk_mcn[CFG_ATTR_NAME_SZ] = "sk";
unsigned char sk_mt[CFG_ATTR_NAME_SZ] = "";
unsigned char sk_pci_optiondata[300] = "";
unsigned char sk_isa_optiondata[300] = "";
unsigned char sk_pcmcia_optiondata[400] = "";	/* Some strings are long */
unsigned char sk_eisa_optiondata[300] = "";
cfg_subsys_attr_t sk_attributes[] =
{
	/*
	 * BUS specific parameter areas (remove all that don't apply)
	 */
	{"PCI_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE | CFG_OP_QUERY,
	 (caddr_t) sk_pci_optiondata, 0, 300, 0},
	{"ISA_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE | CFG_OP_QUERY,
	 (caddr_t) sk_isa_optiondata, 0, 300, 0},
	{"PCMCIA_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE | CFG_OP_QUERY,
	 (caddr_t) sk_pcmcia_optiondata, 0, 400, 0},
	{"EISA_Option", CFG_ATTR_STRTYPE, CFG_OP_CONFIGURE | CFG_OP_QUERY,
	 (caddr_t) sk_eisa_optiondata, 0, 300, 0},
	/*
	 * Description of the subsystem (What this driver drives). (OPTIONAL)
	 */
	{"Subsystem_Description", CFG_ATTR_STRTYPE, CFG_OP_QUERY,
	 (caddr_t) sk_sd, 2, CFG_ATTR_NAME_SZ,
	 CFG_ATTR_NAME_SZ},
	/*
	 * Name of the device. (OPTIONAL) 
	 */
	{"Module_Config_Name", CFG_ATTR_STRTYPE, CFG_OP_QUERY,
	 (caddr_t) sk_mcn, 2, CFG_ATTR_NAME_SZ, 2},
	  /*
	   * Mark the end of the list.
	   */
	{"", 0, 0, 0, 0, 0, 0}
};

/*
 * sk_configure()
 *      Configures, Unconfigures, and returns status based on requests
 *  from cfgmgr.  
 * 
 * Arguments:
 *  op      Operation type (CFG_OP_CONFIGURE, CFG_OP_UNCONFIGURE,
 *                      CFG_OP_QUERY).
 *  indata      Input data structure
 *  indatalen   Number of cfg_attr_t entries
 * 
 * Return Value:
 *  0       success
 *  EINVAL      Invalid parameter
 *  ENOPEN      not applicable to LAN drivers
 *  EBUSY       Device is active
 *  ESRCH       not applicable to LAN drivers
 */
sk_configure(cfg_op_t op,		/* Operation */
			 cfg_attr_t * indata,	/* Input data structure */
			 size_t indatalen,	/* Size of input data structure */
			 cfg_attr_t * outdata,	/* Output data structure */
			 size_t outdatalen)
{								/* Size of output data structure */
	int i;

	switch (op) {
		 /*
		  * Configure a driver
		  */
	 case CFG_OP_CONFIGURE:
		 /*
		  * Make sure we have a module config name.
		  */
		 if (strcmp(sk_mcn, "") == 0) {
			 strcpy(sk_mcn, "sk");
		 }

		 /*
		  * Determine if we're statically or dynamically configured.
		  * 
		  * sk_cfg_state will be either SUBSYSTEM_DYNAMICALLY_CONFIGURED
		  * or SUBSYSTEM_STATICALLY_CONFIGURED.
		  */
		 if (cfgmgr_get_state(sk_mcn, &sk_cfg_state) != ESUCCESS) {
			 printf("sk.mod: cfgmgr_get_state failed\n");
			 return (EINVAL);
		 }

		 /*
		  * Allocate a controller structure.  At config time, we allocate
		  * only one controller struct.  This allows the driver to be called
		  * at probe when the associated hardware unit is found.  At probe,
		  * another controller structure is created thereby allowing another
		  * unit to be probed.  This has the effect that there'll always be
		  * one more controller struct defined than devices you'll actually 
		  * have.
		  */
		 if (sk_cfg_state == SUBSYSTEM_DYNAMICALLY_CONFIGURED) {

			 /* 
			  * Subsytem is dynamically configured.   We may create the
			  * controller structure immediately, and, since we're
			  * called pass the autoconfiguraiton point, we must call
			  * the configure_driver interface to cause our new controller's
			  * probe routines to be called.
			  */
			 if (sk_create_controller() != ESUCCESS) {
				 printf("sk.mod: Cannot create controller structure\n");
				 return (EINVAL);
			 }

			 /*
			  * Now configure the driver.
			  */
			 i = configure_driver(sk_mcn, DRIVER_WILDNUM, "*", &skdriver);
			 if (i != ESUCCESS) {
				 printf("sk.mod: Cannot configure driver\n");
				 return (i);
			 }
		 }
		 else {

			 /*
			  * Subsystem is statically configured.  We must register a
			  * callback to create the controller structs once the calls
			  * are available to perform this operation.   After controller
			  * structs are created, the autoconfigure code will probe 
			  * each controller.
			  */
			 if (register_callback(sk_callback_create_controller,
						CFG_PT_PRECONFIG, CFG_ORD_NOMINAL, 0) != ESUCCESS) {
				 printf("sk.mod: Cannot register PRECONFIG callback\n");
				 return (EINVAL);
			 }
		 }
		 break;

		 /*
		  * Query parameters.  Basically a no-op.
		  */
	 case CFG_OP_QUERY:
		 return (0);
		 break;

		 /*
		  * UNconfigure a driver.  Note, LAN drivers are not unloadable
		  * at this time since there's no if_unattach.  Therefore, we 
		  * fail the unconfigure request.
		  */
	 case CFG_OP_UNCONFIGURE:
		 return (EINVAL);
		 break;
	}
}

/*
 * sk_probe()
 *  Allocate and initialize the softc for this device (if it's really
 *  on this system).  The following other actions must occur here;
 * 
 *      set the ethernet/fddi/token ring hardware address
 *      perform bus specific initialization
 *      register the shutdown routine
 *      initialize the hardware (reset)
 *
 * Arguments:
 *  io_handle   io-handle for this device as determined by caller.
 *  ctlr        pointer to controller struct for the interface.
 *
 * Return Value:
 *  Success:    ~0
 *  Failure:    0
 */
sk_probe(register io_handle_t io_handle, register struct controller *ctlr)
{
	struct sk_softc *sc;
	int unit = ctlr->ctlr_num, success = 1;
	struct handler_intr_info sk_intr_info;
	ihandler_t sk_ihandle;

	/*
	 * Perform bus specific intialization.
	 * 
	 * This should also verify that the device you're called with is what 
	 * you think it is.  Return 0 if not recognized (this is the actual 
	 * function of the probe routine).  Otherwise continue.
	 */
	switch (ctlr->bus_hd->bus_type) {
	 case BUS_PCI:
		 /*
		  * For PCI, the pci configuration header comes in through the
		  * io_handle parameter.   You define the pointer to the header 
		  * in the following manner;
		  * 
		  *  struct pci_config_hdr *pci_cfg_hdr = (void *)io_handle;
		  * 
		  * Take a look at io/dec/pci/pci.h for more information about
		  * what you get in this structure.
		  */
		 break;
	 case BUS_ISA:
		 /*
		  * For ISA, you get information set by the console command
		  * 'isacfg' via the get_config routine.  This routine is 
		  * rather well documented in the driver reference, but here are 
		  * some simple examples.
		  * 
		  *  struct e_port port_sel;
		  *  struct bus_mem mem_sel;
		  *  struct irq irq_sel;
		  *  struct dma dma_sel;
		  * 
		  * get_config(ctlr, RES_PORT, NULL, &port_sel, 0);
		  *  ioaddress = port_sel.base_address;
		  * get_config(ctlr, RES_IRQ, NULL, &irq_sel, 0);
		  *      irq_val = irq_sel.channel;
		  * get_config(ctlr, RES_DMA, NULL, &dma_sel, 0);
		  *  dma_chan = dma_sel.channel;
		  * get_config(ctlr, EISA_MEM, NULL, &mem_sel, 0);
		  *  start = mem_sel.start_addr;
		  * 
		  * The isacfg handle is accessable through the isa_slot_to_name
		  * function.
		  * 
		  *  char *isa_handle;
		  * 
		  * isa_handle = isa_slot_to_name(ctlr->slot);
		  */
		 break;
	 case BUS_PCMCIA:
		 /*
		  * For PCMCIA, you determine your I/O Address in the following
		  * manner;
		  * 
		  *  struct card_info *card_infop = 
		  *                         (struct card_info *)(ctlr->card_info_ptr);
		  * 
		  *  ioaddress = io_handle+card_infop->io_addr[0];
		  * 
		  * You must also register a card removal event routine.
		  * 
		  *  pcmcia_register_event_callback(card_infop->socket_vnum,
		  *                     CARD_REMOVAL_EVENT,
		  *                     (caddr_t)sk_card_remove,
		  *                     (caddr_t)sc);
		  * 
		  * Great care must be used when writing a PCMCIA driver since 
		  * your adapter could disappear at any time (and your reads to io
		  * space will be junk).
		  * 
		  * If you need to get at any of the CIS stuff on the card, several
		  * routines exist for doing this.  See the PCMCIA driver manual for
		  * more information (most LAN devices don't care).
		  */
		 break;
	 case BUS_EISA:
		 /*
		  * Your base address comes in as the io_handle parameter.
		  * 
		  * Also uses the same get_config routine calls as described in 
		  * BUS_ISA...
		  */
		 break;
	 default:
		 printf("sk%d: Unrecognized bus type\n", unit);
		 return (0);			/* Fail */
	}

	/*
	 * Make sure we haven't exceeded the maximum number of units that this
	 * driver supports.
	 */
	if (unit > sk_MAXDEV) {
		printf("sk%d: sk_probe: unit exceeds max supported devices\n",
			   unit);
		return (0);
	}

	/*
	 * Allocate space for the softc.  Note that you should use
	 * MALLOC_VAR if the size is not a constant.
	 */
	MALLOC(sc, void *, sizeof(struct sk_softc), M_DEVBUF, M_NOWAIT);
	if (!sc) {
		printf("sk%d: sk_probe: failed to get memory for softc\n",
			   unit);
		return (0);
	}
	bzero(sc, sizeof(struct sk_softc));

	/*
	 * Fill in softc structure.
	 */
	if (!sucess) {
		FREE(sc, M_DEVBUF);
		return (0);
	}

	/*
	 * Initialize software (rings, etc.).
	 */
	if (!sucess) {
		FREE(sc, M_DEVBUF);
		return (0);
	}

	/*
	 * Initialize hardware (reset, etc.).
	 */
	if (!sucess) {
		FREE(sc, M_DEVBUF);
		return (0);
	}

	/*
	 * Read and save the 48-bit physical address of the device into the
	 * sc->is_addr field.
	 */
	if (!sucess) {
		FREE(sc, M_DEVBUF);
		return (0);
	}

	/*
	 * Register interrupt routine (enable in attach).
	 */
	sk_intr_info.configuration_st = (caddr_t) ctlr;
	sk_intr_info.intr = sk_intr;
	sk_intr_info.param = (caddr_t) unit;
	sk_intr_info.config_type = CONTROLLER_CONFIG_TYPE;

	/*
	 * If driver is capable of sharing interrupts, indicate
	 * this via the following;
	 * 
	 *     sk_intr_info.config_type |= SHARED_INTR_CAPABLE;
	 * 
	 */

	sk_ihandle.ih_bus = ctlr->bus_hd;
	sk_ihandle.ih_bus_info = (char *) &sk_intr_info;

	sc->hid = handler_add(&sk_ihandle);
	if (sc->hid == (ihandler_id_t *) (NULL)) {
		printf("sk%d: interrrupt handler add failed\n", unit);
		FREE(sc, M_DEVBUF);
		return (0);
	}

	/*
	 * Save controller and softc pointers in our tables.
	 */
	sk_softc[unit] = sc;
	sk_info[unit] = ctlr;

	/*
	 * Check if dynamically configured or PCMCIA.  For some reason, 
	 * ifnet structs don't get initialized properly when they're dynamically
	 * brought into the kernel.  Therefore, we do manual initialization here.
	 */
	if ((sk_cfg_state == SUBSYSTEM_DYNAMICALLY_CONFIGURED) ||
			(ctlr->bus_hd->bus_type == BUS_PCMCIA)) {
		/*
		 * Initialize the if_snd queue lock and maxlen fields.  This is
		 * unique to pcmcia since pcmcia is probed so late (it misses the
		 * the common ifnet initialization).  This is also the case for
		 * dynamically configured LAN drivers.
		 */
		IFQ_LOCKINIT_SPLIMP(&sc->is_if.if_snd);
		sc->is_if.if_snd.ifq_maxlen = IFQ_MAXLEN;
	}

	/*
	 * Try to allocate another controller to support an additional unit.
	 */
	if (sk_create_controller() != ESUCCESS) {
		printf("sk%d: WARNING: create_controller failed\n", unit);
	}

	/*
	 * Register shutdown routine.  This is required to make the device
	 * stop all activity as a result of a system crash/shutdown.
	 */
	drvr_register_shutdown(sk_shutdown, (void *) ctlr, DRVR_REGISTER);
	return (~0);
}

/*
 * sk_attach()
 *  Fill in ifnet structure and attach to the network layers.  
 *
 * Arguments:
 *  ctlr        pointer to controller struct for the interface.
 *
 * Return Value:
 *  None.
 */
sk_attach(struct controller * ctlr)
{
	register int unit = ctlr->ctlr_num;
	register struct sk_softc *sc = sk_softc[unit];
	register struct ifnet *ifp = &sc->is_if;
	register struct sockaddr_in *sin;

	/*
	 * Setup address/header information for this media's type.
	 */
	ifp->if_addrlen = 6;		/* media address len */
	ifp->if_hdrlen =			/* Max datalink header size */
		sizeof(struct ether_header) + 8;	/* src+dst+pid/size+pid */

	/*
	 * Setup media specific portions
	 */
	sc->is_ac.ac_bcastaddr = (u_char *) etherbroadcastaddr;
	sc->is_ac.ac_arphrd = ARPHRD_ETHER;
	ifp->if_unit = unit;
	ifp->if_name = "sk";
	ifp->if_mtu = ETHERMTU;		/* FDDIMTU, TRNx_RFC1042_*_MTU see if_trn.h */
	ifp->if_mediamtu = ETHERMTU;
	ifp->if_type = IFT_ETHER;	/* IFT_FDDI, IFT_ISO88025 */
	ifp->if_flags = IFF_BROADCAST | IFF_MULTICAST | IFF_NOTRAILERS | IFF_SIMPLEX;
	((struct arpcom *) ifp)->ac_flag = 0;
	sin = (struct sockaddr_in *) &ifp->if_addr;
	sin->sin_family = AF_INET;

	/*
	 * Setup dispatch routines  --  used by network layer to call driver
	 */
	ifp->if_init = sk_init;		/* init routine */
	ifp->if_output = ether_output;	/* Common for all LAN drivers */
	ifp->if_start = sk_start;	/* start (output, transmit) routine */
	ifp->if_ioctl = sk_ioctl;	/* "I" octal */
	ifp->if_timer = 0;			/* Secs until if_watchdog is called */
	ifp->if_watchdog = sk_watch;	/* Optional */
	ifp->if_reset = sk_reset;	/* reset routine */
	ifp->if_sysid_type = 0;		/* Optional -- See DNA spec for 
								   MOP number */
	ifp->if_version = "XYZ FANCY Ethernet Interface";

	/*
	 * Set baudrate (used by SNMP).
	 * 
	 *  Other baudrates are;
	 *      ETHER_BANDWIDTH_100MB (coming soon)
	 *      FDDI_BANDWIDTH_100MB
	 *      TRN_BANDWIDTH_4MB
	 *      TRN_BANDWIDTH_16MB
	 */
	ifp->if_baudrate = ETHER_BANDWIDTH_10MB;

	/*
	 * Setup lock info.  It's no longer necessary to #if NETSYNC_LOCK since
	 * it's always defined now.  However, it's the LAN driver convention
	 * to use the symbol.
	 */
#if NETSYNC_LOCK
	ifp->if_affinity = NETALLCPU;
	ifp->lk_softc = &sc->sk_softc_lock;
#endif
	simple_lock_setup(&sc->sk_softc_lock, sk_lock_info);

	/*
	 * Print banner indicating that the device is loaded and print
	 * the hardware address (this code assumes that you've already
	 * stored it in sc->is_addr).
	 */
	printf("sk%d: %s, hardware address: %s\n", unit,
		   ifp->if_version, ether_sprintf(sc->is_addr));

	/*
	 * Attach the packet filter.
	 */
	attachpfilter(&(sc->is_ed));

	/*
	 * Attach the network layer.
	 */
	if_attach(ifp);

	/*
	 * Enable the interrupt handler (interrupts may now occur).
	 */
	handler_enable(sc->hid);
}

/* 
 * sk_init_locked()
 *  Initialize interface.   This routines assumes that the 
 *  softc locked is already owned and that SPL == IMP.
 *
 * Arguments:
 *  sc      The softc pointer
 *  ifp     The interface pointer
 *  unit        The unit number of the interface
 *
 * Return Value:
 *  - EIO
 *  - ENOMEM
 *  - ENOBUFS
 *  - ESUCCESS
 */
sk_init_locked(struct sk_softc *sc, struct ifnet *ifp, int unit)
{
	struct controller *ctlr = sk_info[unit];

	/*
	 * Set debug if IFF_DEBUG is set.  This flag can be used to dynamically
	 * put the driver into debug mode (very useful during development).  It
	 * can be turned on by 'ifconfig ifx debug' and off by 
	 * 'ifconfig ifx -debug'... 
	 */
	if (ifp->if_flags & IFF_DEBUG)
		sc->debug++;
	else
		sc->debug = 0;

	/*
	 * This could be called as a 're-init'.  There could be buffers 
	 * left on the ring that are pending.   Pull off all pending transmits
	 * and prepend them to the if_snd queue.
	 * 
	 * To put them on the if_snd queue, use the following algorithm;
	 * 
	 *      IFQ_LOCK(&ifp->if_snd);
	 *      do {
	 *             IF_PREPEND_NOLOCK(&ifp->if_snd, m);
	 *          } while (!done);
	 *      IFQ_UNLOCK(&ifp->if_snd);
	 */

	/*
	 * Clear internal counters/indices (if any) and initialize 
	 * rings/structures.
	 */

	/*
	 * Initialize the device to use the rings.
	 */

	/*
	 * Process any special flags in the ifp->if_flags field.  The most
	 * notable flags are;
	 * 
	 *  IFF_LOOPBACK
	 *  IFF_ALLMULTI
	 *  IFF_PROMISC
	 */

	/*
	 * Start the device.
	 */

	/*
	 * Mark the device as running and mark that there is no output
	 * outstanding.
	 */
	ifp->if_flags |= IFF_RUNNING;
	ifp->if_flags &= ~IFF_OACTIVE;

	/*
	 * If there are any pending packets, start transmit of them now.
	 */
	if (ifp->if_snd.ifq_head)
		sk_start_locked(sc, ifp);

	return ESUCCESS;
}

/* 
 * sk_init()
 *  Jacket to Initialize interface.
 *
 * Arguments:
 *  unit        The unit number of the interface
 *
 * Return Value:
 *  - EIO
 *  - ENOMEM
 *  - ENOBUFS
 *  - ESUCCESS
 */
sk_init(int unit)
{
	register struct sk_softc *sc = sk_softc[unit];
	register struct ifnet *ifp = &sc->is_if;
	int i, s;

	/*
	 * Set IPL, and attain lock.
	 */
	s = splimp();
	simple_lock(&sc->sk_softc_lock);

	/*
	 * Call the initialize interface routine.
	 */
	i = sk_init_locked(sc, ifp, unit);

	/*
	 * Unlock, reset IPL.
	 */
	simple_unlock(&sc->sk_softc_lock);
	splx(s);

	/*
	 * Exit with status returned from initialization routine.
	 */
	return (i);
}

/*
 * sk_start_locked()  
 *  Interface start routine - start output on the interface 
 *
 * Arguments:
 *  sc  The softc pointer
 *  ifp The ifnet pointer, pointing to the intf. to o/p on.
 *
 * Return Value:
 *  None.
 */
sk_start_locked(struct sk_softc * sc, struct ifnet * ifp)
{
	register int unit = ifp->if_unit;
	struct ether_header *eh;
	struct mbuf *m;

	/*
	 * Remove packets from the pending queue and have the device
	 * transmit the packet.
	 */
	while (1) {
		IF_DEQUEUE(&ifp->if_snd, m);
		if (m /* && there's room on the device */ ) {
			/*
			 * Transmit the buffer.  Account for the bytes going out
			 * either here or in transmit completion (up to your particular
			 * device's behaviour).  If you wish to count at interrupt
			 * completion, you will have to queue the mbuf somewhere until
			 * the transmit gets processed.  After the transmit is counted,
			 * you must m_freem the buffer (chain).
			 * 
			 * Quick synopsis of mbuf fields used in transmitting.
			 *      m_len = length of segment
			 *      m_data = data in segment
			 *      m_next = pointer to next segment (if non-NULL)
			 * In the first mbuf (the "header" mbuf).
			 *      m_pkthdr.len = total size of packet
			 * 
			 * The following example shows how to count the packet and free
			 * the buffer(s).
			 * 
			 *      ADD_XMIT_PACKET(ifp, sc, m->m_pkthdr.len);
			 *      eh = mtod(m, struct ether_header *);
			 *      if (eh->ether_dhost[0] & 0x1) {
			 *        ADD_XMIT_MPACKET(ifp, sc, m->m_pkthdr.len);
			 *      }
			 *      m_freem(m);
			 * 
			 */

			/*
			 * Indicate that the output process is active (indicates to 
			 * the network layer that subsequent transmits should be queued 
			 * rather than directly passed to this routine).
			 * 
			 * This flag gets cleared in the interrupt completion routine 
			 * (and in reset/init calls).
			 * 
			 * This flag is a basic flow control mechanism.  If used
			 * properly, it prevents a thrashing situation where the 
			 * interrupt and network threads are combating for the same 
			 * transmit resource.
			 */
			ifp->if_flags |= IFF_OACTIVE;
		}
		else if (m /* && there's no room on the device */ ) {
			IF_PREPEND(&ifp->if_snd, m);
			break;
		}
		else
			break;
	}

	/*
	 * Tell device to process the transmit(s).
	 */

	/*
	 * Start the watchdog (optional, but a real good idea).
	 */
	ifp->timer = 3;
}

/*
 * sk_start()
 *  Jacket to sk_start_locked.   Simply sets IPL and locks the softc.
 * 
 * Arguments:
 *  ifp The ifnet pointer, pointing to the intf. to o/p on.
 *
 * Return Value:
 *  None.
 */
sk_start(register struct ifnet *ifp)
{
	register int unit = ifp->if_unit, s;
	register struct sk_softc *sc = sk_softc[unit];

	/* 
	 * Set IPL, and take lock.  If the lock can't be attained, just exit
	 * (most likely the transmit will be picked up at a later time).
	 */
	s = splimp();
	if (!simple_lock_try(&sc->sk_softc_lock)) {
		splx(s);
		return;
	}

	/*
	 * Call the interface start routine.
	 */
	sk_start_locked(sc, ifp);

	/*
	 * Unlock, reset IPL.
	 */
	simple_unlock(&sc->sk_softc_lock);
	splx(s);
}

/*
 * sk_watch() 
 *  Watchdog timer routine - goes off whenever a transmit times-out.
 *
 * Arguments:
 *  unit        The unit number of the interface
 *
 * Return Value:
 *  None.
 */
sk_watch(int unit)
{
	register struct sk_softc *sc = sk_softc[unit];
	register struct ifnet *ifp = &sc->is_if;
	int s;

	s = splimp();
	simple_lock(&sc->sk_softc_lock);

	/*
	 * m_freem(m) the offending transmit
	 */

	/*
	 * Clear the timer and reset the unit.
	 */
	ifp->if_timer = 0;
	sk_reset_locked(sc, ifp, unit);

	simple_unlock(&sc->sk_softc_lock);
	splx(s);
}

/*
 * sk_reset_locked()
 *  Reset the interface.
 *
 * Argument
 *  sc      The softc pointer
 *  ifp     The interface pointer
 *  unit        The unit number of the interface
 *
 * Return Value 
 *  none
 */
sk_reset_locked(struct sk_softc *sc, struct ifnet *ifp, int unit)
{
	register struct controller *ctlr = sk_info[unit];
	int s;

	/*
	 * Stop the hardware
	 */

	/*
	 * Indicate that the device is no longer running
	 */
	ifp->if_flags &= ~IFF_RUNNING;

	/*
	 * Initialize the device
	 */
	sk_init_locked(sc, ifp, unit);
}

/*
 * sk_reset()
 *  Jacket routine into the Reset interface.
 * 
 * Argument
 *  unit        The unit number of the interface
 *
 * Return Value 
 *  none
 */
sk_reset(int unit)
{
	struct sk_softc *sc = sk_softc[unit];
	struct ifnet *ifp = &sc->is_if;
	int s;

	/*
	 * Set IPL, and attain lock.
	 */
	s = splimp();
	simple_lock(&sc->sk_softc_lock);

	/*
	 * Call the initialize interface routine.
	 */
	sk_reset_locked(sc, ifp, unit);

	/*
	 * Unlock, reset IPL.
	 */
	simple_unlock(&sc->sk_softc_lock);
	splx(s);
}

/*
 * sk_ioctl()
 *  Process an ioctl request. In doing so, the interface is
 *  restarted and reinitialized (in most cases).
 *
 * Arguments:
 *  ifp     Pointer to the ifnet structure
 *  cmd     The ioctl to be performed
 *  data        The corresponding data for the ioctl.
 *
 * Return Value:
 *  - EIO
 *  - EINVAL
 *  - ENOMEM
 *  - ENOBUFS
 *  - ESUCCESS
 */
sk_ioctl(register struct ifnet *ifp, register u_int cmd, register caddr_t data)
{
	register struct sk_softc *sc = sk_softc[ifp->if_unit];
	register struct controller *ctlr = sk_info[ifp->if_unit];
	register unit = ifp->if_unit;
	struct ifreq *ifr = (struct ifreq *) data;
	struct ifdevea *ifd = (struct ifdevea *) data;
	struct ctrreq *ctr = (struct ctrreq *) data;
	struct ifaddr *ifa = (struct ifaddr *) data;
	struct ifchar *ifc = (struct ifchar *) data;
	int s, i, j, lock_on = 1, status = ESUCCESS;

	s = splimp();
	simple_lock(&sc->sk_softc_lock);

	switch (cmd) {

		 /*
		  * Enable Loopback mode
		  */
	 case SIOCENABLBACK:
		 ifp->if_flags |= IFF_LOOPBACK;
		 if (ifp->if_flags & IFF_RUNNING)
			 sk_reset_locked(sc, ifp, unit);
		 break;

		 /*
		  * Disable Loopback mode
		  */
	 case SIOCDISABLBACK:
		 ifp->if_flags &= ~IFF_LOOPBACK;
		 if (ifp->if_flags & IFF_RUNNING)
			 sk_reset_locked(sc, ifp, unit);
		 break;

		 /*
		  * Read current physical and hardware addresses
		  */
	 case SIOCRPHYSADDR:
		 /*
		  * Copy current address
		  */
		 bcopy(sc->is_addr, ifd->current_pa, 6);
		 /*
		  * Copy hardware address (defined in a ROM somewhere)
		  */
		 /*
		  * bcopy(sc->, ifd->default_pa, 6);
		  */
		 break;

		 /*
		  * Set locally administered MAC address.   If not supported, change
		  * status to a meaningful errno, and break.
		  */
	 case SIOCSPHYSADDR:
		 /*
		  * Copy to ifnet struct
		  */
		 bcopy(ifr->ifr_addr.sa_data, sc->is_addr, 6);

		 /*
		  * Notify the packet filter
		  */
		 pfilt_newaddress(sc->is_ed.ess_enetunit, sc->is_addr);

		 /*
		  * Notify the hardware.  This may require a reset of the device
		  * if not sufficiently intelligent.
		  */
		 if (ifp->if_flags & IFF_RUNNING) {
			 /*
			  * Only change the hardware if it's running.
			  */
		 }

		 /*
		  * Drop lock and ipl.   The functions used from here on out don't
		  * require synch.
		  */
		 simple_unlock(&sc->sk_softc_lock);
		 splx(s);
		 lock_on = 0;

		 /* 
		  * If an IP address has been configured then an ARP 
		  * packet must be broadcast to tell all hosts which 
		  * currently have our address in their ARP tables to 
		  * update their information.
		  */
		 if (((struct arpcom *) ifp)->ac_flag & AC_IPUP) {
			 rearpwhohas((struct arpcom *) ifp);
		 }

		 /*
		  * Notify about possible change in af_link address.
		  */
		 if_sphyaddr(ifp, ifr);
		 break;

		 /*
		  * Delete a multicast
		  */
	 case SIOCDELMULTI:
		 /*
		  * This routine will differ from device to device depending on
		  * how it handles multicasts.  If the address is found in the
		  * multicast table, it's removed.  Each multicast address has 
		  * a reference count (see lan_common.c) associated with it.  
		  * When the reference count goes down to zero (signalling actual
		  * deletion), we must update the device.  Otherwise, we just
		  * return.
		  * 
		  * NOTE: All Digital UNIX LAN device drivers must always have 
		  * the broadcast address enabled.
		  */
		 if (bcmp(ifr->ifr_addr.sa_data, etherbroadcastaddr, 6) != 0) {
			 switch (lan_del_multi(&sc->is_multi,
								 (unsigned char *) ifr->ifr_addr.sa_data)) {
			  case LAN_MULTI_CHANGED:
				  /*
				   * This means that there's been a change to the list of
				   * currently active multicasts.  Action may be required
				   * on the device to remove the selected multicast from 
				   * the device's table/list/bitmasque.
				   * 
				   * For nominal cases;  a simple call to sk_reset_locked
				   * is all that is needed to update the devices multicast
				   * list.  This would be done as follows;
				   * 
				   *  if (ifp->if_flags & IFF_RUNNING)
				   *     sk_reset_locked(sc, ifp, unit);
				   */
				  break;
			  case LAN_MULTI_NOT_CHANGED:
				  /* 
				   * This means that the remove succeeded, but that the
				   * multicast address is still in use.  Just break in this
				   * case.
				   */
				  break;
			  case LAN_MULTI_FAILED:
			  default:
				  /*
				   * This means that the address was not found.
				   */
				  status = EINVAL;
				  break;
			 }
		 }

		 /*
		  * The following debug code shows how to use the LAN_GET_MULTI
		  * macros (in lan_common.h).
		  */
		 if (sc->debug) {
			 j = 0;
			 printf("sk%d: Dump of multicast table after DEL (%d entries)\n",
					unit, sc->lan_nmulti);
			 for (i = 0; i < sc->lan_nmulti; i++) {
				 unsigned char *maddr;

				 LAN_GET_MULTI(&sc->is_multi, maddr, j);
				 printf("   %d  %s (muse==%d)\n", i + 1, ether_sprintf(maddr),
						sc->is_multi.lan_mtable[j - 1].muse);
			 }
		 }
		 break;

		 /*
		  * Add a multicast
		  */
	 case SIOCADDMULTI:
		 /*
		  * This routine will differ from device to device depending on
		  * how it handles multicasts.  If the address isn't currently
		  * enabled, it's added to the list (see lan_common.c).  If the
		  * address is already enabled, then the reference count is 
		  * incremented, and status is returned indicating that no additonal
		  * action is required.
		  * 
		  * NOTE: All Digital UNIX LAN device drivers must always have 
		  * the broadcast address enabled.
		  */
		 if (bcmp(ifr->ifr_addr.sa_data, etherbroadcastaddr, 6) != 0) {
			 switch (lan_add_multi(&sc->is_multi,
								 (unsigned char *) ifr->ifr_addr.sa_data)) {
			  case LAN_MULTI_CHANGED:
				  /*
				   * This means that there's been a change to the list of
				   * currently active multicasts.  Action may be required
				   * on the device to add the selected multicast to
				   * the device's table/list/bitmasque.
				   * 
				   * For nominal cases;  a simple call to sk_reset_locked
				   * is all that is needed to update the devices multicast
				   * list.  This would be done as follows;
				   * 
				   *  if (ifp->if_flags & IFF_RUNNING)
				   *     sk_reset_locked(sc, ifp, unit);
				   */
				  break;
			  case LAN_MULTI_NOT_CHANGED:
				  /* 
				   * This means that the add succeeded, but that the
				   * multicast address was already in use.  Just break in this
				   * case.
				   */
				  break;
			  case LAN_MULTI_FAILED:
			  default:
				  /*
				   * This means that we're out of room to put multicast
				   * addresses.
				   */
				  status = EINVAL;
				  break;
			 }
		 }

		 /*
		  * Another debug example of the dump of the multicast table using
		  * the LAN_GET_MULTI macro.
		  */
		 if (sc->debug) {
			 j = 0;
			 printf("sk%d: Dump of multicast table after ADD (%d entries)\n",
					unit, sc->lan_nmulti);
			 for (i = 0; i < sc->lan_nmulti; i++) {
				 unsigned char *maddr;

				 LAN_GET_MULTI(&sc->is_multi, maddr, j);
				 printf("   %d  %s (muse==%d)\n", i + 1, ether_sprintf(maddr),
						sc->is_multi.lan_mtable[j - 1].muse);
			 }
		 }
		 break;

		 /*
		  * Read & Read/Zero counters.  This is a little tricky.
		  * 
		  * In addition to the regular counters that need to be recorded, 
		  * FDDI and Token Ring have to respond to the MIB (SNMP) requests for
		  * counter information.  The MIB information is the same thing but
		  * different as the standard counters.
		  * 
		  * The SIOCRDZCTRS functions zeroes the counters AFTER returning
		  * the values currently set.
		  * 
		  * Remove the example portions of the medias that you don't support
		  * from the following.
		  */
	 case SIOCRDCTRS:
	 case SIOCRDZCTRS:

		 /*
		  * ================
		  * ETHERNET EXAMPLE
		  * ================
		  */
		 ctr->ctr_ether = sc->ctrblk;	/* Copy current counters */
		 ctr->ctr_type = CTR_ETHER;		/* These are Ethernet Counters */
		 /*
		  * Return time since last zeroed
		  */
		 ctr->ctr_ether.est_seconds = (time.tv_sec - sc->ztime) > 0xfffe ?
			 0xffff : (time.tv_sec - sc->ztime);
		 /*
		  * Zero the counters if requested to 
		  */
		 if (cmd == SIOCRDZCTRS) {
			 sc->ztime = time.tv_sec;
			 bzero(&sc->ctrblk, sizeof(struct estat));
		 }
		 /*break; */
		 /*
		  * ============
		  * FDDI EXAMPLE
		  * ============
		  */

		 /*
		  * If the device is initializing (or not running), set status to 
		  * EACCES and break.
		  */

		 /*
		  * See what type of counters are requested
		  */
		 switch (ctr->ctr_type) {

		  case FDDIMIB_SMT:	/* Return FMIB stats */
		  case FDDIMIB_MAC:
		  case FDDIMIB_PATH:
		  case FDDIMIB_PORT:
		  case FDDIMIB_PORT_B:
		  case FDDIMIB_ATTA:
			  /*
			   * Look in if.h for all the structures you need to return.
			   *      FDDIMIB_SMT == struct fddismt
			   *      FDDIMIB_MAC == struct fddimac
			   *      FDDIMIB_PATH == struct fddipath
			   *      FDDIMIB_PORT == struct fddiport
			   *      FDDIMIB_PORT_B == struct fddiport (set port_index = 2)
			   *      FDDIMIB_ATTA == struct fddiatta
			   * 
			   * goes into &ctr->fmib_smt, fmib_mac, fmib_path, fmib_port, 
			   *       fmib_port, fmib_atta
			   */
			  break;

		  case FDDISMT_MIB_SMT:
		  case FDDISMT_MIB_MAC:
		  case FDDISMT_MIB_PORT:
		  case FDDISMT_MIB_PORT_B:
		  case FDDISMT_MIB_PATH:
			  /*
			   * Again, if.h contains these structures
			   *      FDDISMT_MIB_SMT == struct smtmib_smt
			   *      FDDI_MIB_MAC == struct smtmib_mac
			   *      FDDI_MIB_PORT == struct smtmib_port
			   *      FDDI_MIB_PORT_B == struct smtmib_port (port_index = 2)
			   *      FDDISMT_MIB_PATH == struct smtmib_path
			   * 
			   * goes into &ctr->smib_smt, smib_mac, smib_port, smib_port,
			   *           smib_path
			   */
			  break;

		  case FDDIDECEXT_MIB:
			  /*
			   * If you're feeling ambitious, this is struct decext_mib.  This
			   * is a digital extension.  EINVAL should be OK if you don't
			   * feel like supporting this.
			   * 
			   * goes into &ctr->decmib_ext
			   */
			  break;

		  case FDDI_STATUS:
			  /*
			   * struct fddiDNA_status goes into &ctr->ctr_ctrs.dna_status_fddi
			   */
			  break;

		  default:
		  case CTR_FDDI:
			  /*
			   * The standard case
			   * 
			   *  If the device keeps the counters on board, you need to 
			   * create a routine that retrieves the counters from the 
			   * device.
			   */
			  ctr->ctr_fddi = sc->ctrblk;	/* Copy current counters */
			  ctr->ctr_type = CTR_FDDI;
			  break;
		 }
		 /*break; */
		 /* 
		  * ==================
		  * TOKEN RING EXAMPLE
		  * ==================
		  */

		 /*
		  * Select type of counters requested
		  */
		 switch (ctr->ctr_type) {
		  case TRN_CHAR:
			  /*
			   * Results go into &ctr->ctr_ctrs.trnchar
			   */
			  break;
		  case TRN_MIB_ENTRY:
			  /*
			   * Results go into &ctr->ctr_ctrs.dot5Entry
			   */
			  break;
		  case TRN_MIB_STAT_ENTRY:
			  /*
			   * Results go into &ctr->ctr_ctrs.dot5StatsEntry
			   */
			  break;
		  case TRN_MIB_TIMER_ENTRY:	/* This is optional and not supported */
			  status = ENOTSUP;
			  break;
		  default:
		  case CTR_TRN:
			  /*
			   * Results go into &ctr->ctr_ctrs.trncount
			   * 
			   * struct trncount is defined in netinet/if_trnstat.h
			   * 
			   * There's no real easy way to do these counters.   It's best
			   * you do what you think is right and to verify it with 
			   * 'netstat -s -I<devname>'...
			   */
			  ctr->ctr_type = CTR_TRN;
			  break;
		 }
		 break;

		 /*
		  * Bring UP the device
		  */
	 case SIOCSIFADDR:
		 ifp->if_flags |= IFF_UP;
		 sk_reset_locked(sc, ifp, unit);

		 /*
		  * Set counter cleared time (used by DECnet/netstat/clusters/etc.)
		  */
		 if (sc->ztime == 0)
			 sc->ztime = time.tv_sec;
		 break;

		 /*
		  * Use currently set flags (only if running)
		  */
	 case SIOCSIFFLAGS:
		 if (ifp->if_flags & IFF_RUNNING)
			 sk_reset_locked(sc, ifp, unit);
		 break;

		 /*
		  * Set IP MTU
		  */
	 case SIOCSIPMTU:
		 bcopy(ifr->ifr_data, (u_char *) & ifmtu, sizeof(u_short));
		 /*
		  * Compare what is passed to the media's max.
		  *      FDDIMTU
		  *      TRN_4_RFC1042_IP_MTU
		  *      TRN_16_RFC1042_IP_MTU
		  */
		 if (ifmtu > ETHERMTU || ifmtu < IP_MINMTU)
			 status = EINVAL;
		 else
			 ifp->if_mtu = ifmtu;
		 break;

		 /*
		  * Set Media speed  --  4/16 (Token Ring), 10/100 (Ethernet)
		  *          Typically  20 means full duplex Ethernet
		  *                                200 means full duplex Fast Ethernet
		  * 
		  * Note:  This function is duplicated by the ifc_speed field in 
		  *    the SIOCIFSETCHAR function.
		  */
	 case SIOCSMACSPEED:
		 bcopy(ifr->ifr_data, (u_char *) & speed, sizeof(u_short));
		 break;

		 /*
		  * Reset -- reset the device
		  */
	 case SIOCIFRESET:
		 sk_reset_locked(sc, ifp, unit);
		 break;

		 /*
		  * Set Characteristics
		  */
	 case SIOCIFSETCHAR:
		 /*
		  * Parameters to set are retrieved from 'ifc'
		  * 
		  *  Common paramters (should be supported by all drivers).
		  *      ifc_auto_sense      -1 (don't change) 
		  *                  LAN_AUTOSENSE_ENABLE,
		  *                  LAN_AUTOSENSE_DISABLE.
		  *      ifc_media_type      Values from lan_common.h 
		  *                  LAN_MEDIA_UTP, LAN_MEDIA_BNC,
		  *                  LAN_MEDIA_STP, LAN_MEDIA_FIBER,
		  *                  LAN_MEDIA_AUI, LAN_MEDIA_4PAIR,
		  *                  -1 means don't change.
		  *      ifc_media_speed     driver dependent same as in
		  *                  SIOCSMACSPEED. -1 means don't
		  *                  change.
		  *      ifc_full_duplex_mode    -1 means don't change. 
		  *                  LAN_FDX_ENABLE, LAN_FDX_DISABLE
		  * 
		  *  FDDI parameters
		  *      ifc_treq, ifc_tvx, ifc_rtoken, ifc_ring_purger,
		  *      ifc_cnt_interval, ifc_lem
		  */
		 break;

		 /*
		  * Default case
		  */
	 default:
		 status = EINVAL;
	}

	/*
	 * If already unlocked and splx'ed, don't do it again.
	 */
	if (lock_on) {
		simple_unlock(&sc->sk_softc_lock);
		splx(s);
	}
	return (status);
}

/*
 * sk_intr()
 *  Interface interrupt routine.
 *
 * Arguments:
 *  unit        The unit number of the interface
 *
 * Return Value:
 *  None.
 */
sk_intr(register int unit)
{
	register u_int s, status;
	register struct sk_softc *sc = sk_softc[unit];
	register struct ifnet *ifp = &sc->is_if;

	/*
	 * If no interrupt is pending, return the status that says we
	 * didn't do anything.  This is used for shared interrupt support
	 * (see documentation for the handler_add routine).
	 */
	/*
	 * if ((CSR & INTERRUPT) == 0)
	 *     return INTR_NOT_SERVICED;
	 */

	s = splimp();
	simple_lock(&sc->sk_softc_lock);

	/*
	 * Process receive and transmit rings.
	 */
	sk_rint(sc, ifp);
	sk_tint(sc, ifp);

	/*
	 * Check to see if there are any transmits pending and that there
	 * is room on the device for more transmits.  If so, transmit.
	 */
	if (ifp->if_snd.ifq_head) {
		sk_start_locked(sc, ifp);
	}
	else {
		ifp->if_timer = 0;
	}

	/*
	 * Release lock and resume IPL
	 */
	simple_unlock(&sc->sk_softc_lock);
	splx(s);

	/*
	 * Indicate that an interrupt was serviced
	 */
	return INTR_SERVICED;
}

/*
 * sk_rint()
 *  Interface receive interrupt completion routine.  Calls ether_input
 *  with completed buffers.  
 * 
 * Arguments:
 *  sc      The unit's softc structure
 *  ifp     The unit's ifnet structure
 *
 * Return Value:
 *  None.
 */
sk_rint(struct sk_softc * sc, struct ifnet * ifp)
{
	struct ether_header eh;

	/*
	 * The call to ether_input and counter functions requires the following
	 * (Ethernet examples only).
	 * 
	 * Setting of the mbuf fields.  m_len and m_pkthdr.len are the length 
	 * of the packet minus the header and trailing CRC (if the device delivers 
	 * it).  m_pkthdr.rcvif is the ifp of the receiving device. 
	 *      m->m_pkthdr.len = m->m_len = len - sizeof(struct ether_header);
	 *      m->m_pkthdr.rcvif = ifp;
	 * 
	 * IMPORTANT -- 
	 *    m_data (after the header) must be a longword aligned address.  IP 
	 *    depends on the IP header being aligned.   Care must be taken to
	 *    insure that the mbuf data area is filled in such a manner that
	 *    the packet data (past the datalink header) is longword aligned.
	 * 
	 * Creation of the Ethernet header structure (this is used by 
	 * ether_input for filtering which user is to receive the packet).
	 *      eh = *(mtod(m, struct ether_header *));
	 *      eh.ether_type = ntohs((unsigned short)eh.ether_type);
	 *      m->m_data += sizeof(struct ether_header);
	 * 
	 * Add counters for the packet received.  This accounts for 
	 *      ADD_RECV_PACKET(ifp, sc, m->m_pkthdr.len);
	 *      if (eh.ether_dhost[0] & 0x1) {
	 *         ADD_RECV_MPACKET(ifp, sc, m->m_pkthdr.len);
	 *          }
	 *      
	 *      ether_input(ifp, &eh, m);
	 */
}

/*
 * sk_tint()
 *  Interface transmit interrupt completion routine.   Removes 
 *  pending transmit context, and frees any mbufs that were 
 *  retained (via m_freem(m)).
 *
 * Arguments:
 *  sc      The unit's softc structure
 *  ifp     The unit's ifnet structure
 *
 * Return Value:
 *  None.
 */
sk_tint(struct sk_softc *sc, struct ifnet *ifp)
{

	/*
	 * Clear output active flag so that other transmits may be queued.
	 */
	ifp->if_flags &= ~IFF_OACTIVE;
}

/*
 * sk_create_controller
 *  Create a controller struct.  
 * 
 * Arguments:
 *  none
 * 
 * Return Value:
 *  Success == ESUCCESS, ~Success == some failure code
 */
int
sk_create_controller()
{
	struct controller_config ccfg, *ccfgp = &ccfg;
	int status;

	/*
	 * Setup the constant stuff.
	 */
	ccfgp->revision = CTLR_CONFIG_REVISION;
	strcpy(ccfgp->subsystem_name, sk_mcn);
	strcpy(ccfgp->bus_name, "*");	/* Any bus */
	ccfgp->devdriver = &skdriver;

	/*
	 * Create the controller.
	 */
	status = create_controller_struct(ccfgp);
	if (status != ESUCCESS) {
		printf("sk.mod: FAILED to create controller structs during init\n");
		return (status);
	}
	return (ESUCCESS);
}

/*
 * sk_callback_create_controller
 *  Callback to create a controller structure.  This simply is a jacket
 *  to the sk_create_controller routine, but it also responsible for
 *  reporting errors to the "framework".
 * 
 * Arguments:    (none are really used)
 *  point       Configuration point
 *  order       Priority at callback
 *  args        user arguments
 *  event_args  ???
 * 
 * Return Value:
 *  none
 */
void
sk_callback_create_controller(int point, int order,
							  unsigned long args, unsigned long event_args)
{
	int status;

	/*
	 * Create a controller.
	 */
	status = sk_create_controller();

	/*
	 * Return error status if there was an error.
	 */
	if (status != ESUCCESS)
		cfgmgr_set_status(sk_mcn);
}
