#include <assert.h>
#include <stdio.h> 
#include <stdlib.h>
#include <tchar.h>

#include <windows.h>  
#include <setupapi.h>
#include <newdev.h>

#include <cfgmgr32.h>
#include <devguid.h>
#include <initguid.h>

typedef struct install_data
{
  LPTSTR path;
  int num_found;
} install_data;

typedef void (*process_buffer_func) (HDEVINFO device_info_set,
				     SP_DEVINFO_DATA *info,
				     PVOID buffer,
				     DWORD buffer_size,
				     BOOL *keep_traversing,
				     void *data);

/* Hardware IDs for Myrinet Adapters.  Change this to an array of strings if
   a lot more IDs get get added. */
_TCHAR *vendor_id1 = _TEXT("PCI\\VEN_14C1&DEV_8043");
_TCHAR *vendor_id2 = _TEXT("PCI\\VEN_10E8&DEV_8043");

/* The name of the inf file for the GM driver. */
_TCHAR *fname = "oemsetup";
_TCHAR *ext = "inf";

/* install_device attempts to install the GM driver if the hardware id passed
   in matches that of a Myrinet adapter. */
void install_device (HDEVINFO device_info_set,
		     SP_DEVINFO_DATA *info,
		     PVOID buffer,
		     DWORD buffer_size,
		     BOOL *keep_traversing,
		     void *data)
{
  LPTSTR p;
  install_data *id;
  BOOL reboot_required;
  
  id = data;
  p = buffer;
  
  if (p != NULL) {
    if (_tcsncmp (p, vendor_id1, _tcslen (vendor_id1)) == 0 ||
	_tcsncmp (p, vendor_id2, _tcslen (vendor_id2)) == 0) {
      
      id->num_found++;
      reboot_required = FALSE;
      if (UpdateDriverForPlugAndPlayDevices (NULL, p, id->path,
					     INSTALLFLAG_FORCE,
					     &reboot_required)) {
	_tprintf (_TEXT ("Automatic driver installation was successful.\n"));
	if (reboot_required == TRUE) {
	  _tprintf (_TEXT ("Please reboot the computer.\n"));
	}
      }
      else {
	switch (GetLastError ()) {
	case ERROR_FILE_NOT_FOUND:
	  _tprintf (_TEXT ("The  %s.%s file was not found.\n"), fname, ext);
	  break;
	default:
	  _tprintf (_TEXT ("Automatic driver installation"
			   " was not successful.\n"));
	  _tprintf (_TEXT ("Use the Device Manager"
			   " to install the GM driver.\n"));
	  break;
	}
      }
      *keep_traversing = FALSE;
    }
  }
}

/* remove_device attempts to uninstall the GM driver if the hardware id passed
   in matches that of a Myrinet adapter */
void remove_device (HDEVINFO device_info_set,
		    SP_DEVINFO_DATA *info,
		    PVOID buffer,
		    DWORD buffer_size,
		    BOOL *keep_traversing,
		    void *data)
{
  LPTSTR p;
  int *num_found;
  
  num_found = data;
  p = buffer;

  if (p != NULL) {
    if (_tcsncmp (p, vendor_id1, _tcslen (vendor_id1)) == 0 ||
	_tcsncmp (p, vendor_id2, _tcslen (vendor_id2)) == 0) {
      *num_found += 1;
      _tprintf (_TEXT ("Removing Myrinet device.\n"));
      if (!SetupDiCallClassInstaller (DIF_REMOVE, device_info_set, info)) {
	_tprintf (_TEXT ("Unable to remove Myrinet device.\n"));
      }
    }
  }
}

/* traverse_devices loops through all the devices in the system, using a
   callback funtion to process each device. */
void traverse_devices (DWORD property, process_buffer_func f, void *data)
{
  HDEVINFO device_info_set;
  SP_DEVINFO_DATA device_info_data;
  DWORD i;
  DWORD data_type;
  BYTE buffer[4096];
  DWORD buffer_size;
  BOOL keep_traversing;

  device_info_set = SetupDiGetClassDevs (NULL, NULL, NULL,
					 DIGCF_ALLCLASSES | DIGCF_PRESENT);
  if (device_info_set == INVALID_HANDLE_VALUE) {
    _tprintf (_TEXT ("Unable to enumerate hardware devices.\n"));
    goto abort_with_nothing;
  }

  keep_traversing = TRUE;
  device_info_data.cbSize = sizeof (device_info_data);

  for (i = 0; SetupDiEnumDeviceInfo (device_info_set, i, &device_info_data);
       ++i) {
    if (SetupDiGetDeviceRegistryProperty (device_info_set, &device_info_data,
					  property, &data_type,
					  buffer, sizeof (buffer),
					  &buffer_size)) {
      f (device_info_set, &device_info_data, buffer, buffer_size,
	 &keep_traversing, data);
    }
    
    if (keep_traversing == FALSE) {
      break;
    }
  }
  SetupDiDestroyDeviceInfoList (device_info_set);
  
 abort_with_nothing:
  ;
}

void print_usage (void)
{
  _tprintf (_TEXT ("usage: gm_setup --install\tinstall driver\n"));
  _tprintf (_TEXT ("       gm_setup --remove\tremove driver\n"));
}

int __cdecl _tmain(int argc, _TCHAR **argv, _TCHAR **envp)
{
  _TCHAR path[_MAX_PATH];
  _TCHAR drive[_MAX_DRIVE];
  _TCHAR dir[_MAX_DIR];

  if (argc != 2) {
    print_usage ();
    goto abort_with_nothing;
  }

  if (GetModuleFileName (NULL, path, _MAX_PATH) == 0) {
    _tprintf (_TEXT ("Unable to determine path to %s.%s\n"), fname, ext);
    goto abort_with_nothing;
  }

  _tsplitpath (path, drive, dir, NULL, NULL); 
  _tmakepath (path, drive, dir, fname, ext);
  
  if (_tcscmp (argv[1], _TEXT ("--install")) == 0) {
    install_data id;
    
    id.path = path;
    id.num_found = 0;

    traverse_devices (SPDRP_HARDWAREID, install_device, &id);

    if (id.num_found == 0) {
      _tprintf (_TEXT ("No Myrinet adapters were detected.\n"));
      _tprintf (_TEXT ("Please use the Device Manager to install the GM"
		       " driver.\n"));
    }
  }
  else if (_tcscmp (argv[1], _TEXT ("--remove")) == 0) {
    int num_found;
    
    num_found = 0;

    traverse_devices (SPDRP_HARDWAREID, remove_device, &num_found);

    if (num_found == 0) {
      _tprintf (_TEXT ("No Myrinet adapters were detected.\n"));
      _tprintf (_TEXT ("Please use the Device Manager to remove the GM"
		       " driver.\n"));
    }
  }
  else {
    print_usage ();
    goto abort_with_nothing;
  }
  
  return EXIT_SUCCESS;

 abort_with_nothing:
  return EXIT_FAILURE;
}
