/* this files contains function to associate a registered page with an
appropriate mapped area structure (the vm_area_struct) that has
appropriate semantics for fork */

#if GM_LINUX_FULL_MM

#include "gm_internal.h"

#include <asm/page.h>
#include <linux/mm.h>
#ifdef __SMP__
#undef current
#define current gm_current
#endif
#ifdef __alpha__
#define get_new_asn_and_reload wrap_get_new_asn_and_reload
#define tbi gm_tbi
#endif
#include <asm/pgtable.h>

#include "gm_arch_def.h"

/* this variable determines if we modify the vm_area structs (that
   describes all mappings of a process) when registering memory, this
   is necessary to have correct semantics against fork, and eventually
   mremap, munmap,... , but some patches to the kernel (exporting some
   memory mgmt related functions) are necessary to do this, their
   avaibility is checked at runtime */
int gm_full_mm = 1;

extern void (*gm_merge_segments)(struct mm_struct *, unsigned long, unsigned long);
extern void (*gm_insert_vm_struct)(struct mm_struct *, struct vm_area_struct *);

/* this function is called for registered memory area inside the fork call
   we copy the child and the parent */
static void reg_vm_open(struct vm_area_struct *vma) 
{
  unsigned long addr;
  pgd_t *src_pgd, *dst_pgd;
  
#if 0
    /* FIXME:  need to track a ref count, to avoid open/close function be called after removall */
  ps->arch.ref_count += 1;
#endif
  if (vma->vm_mm == gm_current->mm) {
    GM_PRINT (GM_PRINT_LEVEL >= 4,("reg_vm_open called outside of fork: probably a fixup or user called mlock?\n"));
    return;
  }
  if (!(vma->vm_flags & VM_LOCKED)) {
    GM_WARN(("user unlocked a register area!!!!\n"));
  }
  /* OK we must be probably in a fork call */
  GM_PRINT (GM_PRINT_LEVEL >= 1,("fork was called inside a GM application\n"));
  addr = vma->vm_start;
  src_pgd = pgd_offset(gm_current->mm, addr);
  dst_pgd = pgd_offset(vma->vm_mm, addr);

  /* loop over pgd entries */
  while (addr < vma->vm_end) {
    unsigned long pmd_off = addr & ~PGDIR_MASK;
    unsigned long pmd_end = pmd_off + vma->vm_end - addr;
    pmd_t *src_pmd,*dst_pmd;
    /* check  valid pgd entries */
    gm_always_assert(!pgd_none(*src_pgd) && ! pgd_bad(*src_pgd));
    gm_always_assert(!pgd_none(*dst_pgd) && ! pgd_bad(*dst_pgd));
    src_pmd = pmd_offset(src_pgd,pmd_off);
    dst_pmd = pmd_offset(dst_pgd,pmd_off);
    if (pmd_end > PGDIR_SIZE)
      pmd_end = PGDIR_SIZE;
    
    /* loop over pmd entries */
    while (pmd_off < pmd_end) {
      unsigned long pte_off = pmd_off & ~PMD_MASK;
      unsigned long pte_end = pte_off + pmd_end - pmd_off;
      pte_t *src_pte, *dst_pte;
      /* check valid pmd entries */
      gm_always_assert(!pmd_none(*src_pmd) && ! pmd_bad(*src_pmd));
      gm_always_assert(!pmd_none(*dst_pmd) && ! pmd_bad(*dst_pmd));
      src_pte = pte_offset(src_pmd,pte_off);
      dst_pte = pte_offset(dst_pmd,pte_off);
      if (pte_end > PMD_SIZE)
        pte_end = PMD_SIZE;
      
      /* loop over pte entries */
      while (pte_off < pte_end) {
        /* see code inside linux/mm/memory.c in cow case : do_wp_page */
        unsigned long addr_base  = (addr & PGDIR_MASK) + (pmd_off & PMD_MASK) + pte_off;
        unsigned long new_page = __get_free_page(GFP_KERNEL);
        unsigned long old_page = pte_page(*src_pte);
        gm_always_assert(pte_present(*src_pte) && pte_present(*dst_pte) && 
                  !pte_write(*src_pte) && !pte_write(*dst_pte) && 
                  pte_page(*dst_pte) == old_page);
        gm_always_assert(mem_map[MAP_NR(old_page)].count >= 2); /* 1 in parent, 1 in child, 1 in page_hash */
        if (mem_map[MAP_NR(old_page)].count != 3) {
          GM_PRINT (1,("after_fork:mem_map[MAP_NR(old_page)].count(%d) != 3 \n",mem_map[MAP_NR(old_page)].count));
          continue;
        }
#if 0
        gm_always_assert(__pa((void*)old_page) == kvirt_to_phys(is,addr_base,0));
#endif
        
        /* copy new_page */
	memcpy((void *) new_page, (void *) old_page, PAGE_SIZE);
        flush_page_to_ram(old_page);
        flush_page_to_ram(new_page);
        flush_cache_page(vma, addr_base);
        set_pte(dst_pte,pte_mkwrite(pte_mkdirty(mk_pte(new_page,vma->vm_page_prot))));
        free_page(old_page); /* remove child reference */
        /* put again original page in rw mode */
        set_pte(src_pte,pte_mkwrite(*src_pte));
        gm_flush_tlb_page(vma,addr_base);
        pte_off += PAGE_SIZE;
        src_pte++;
        dst_pte++;
      } /* end pte loop */
      pmd_off = (pmd_off + PMD_SIZE) & PMD_MASK;
      src_pmd++;
      dst_pmd++;
    } /* end pmd loop */
    addr = (addr + PGDIR_SIZE) & PGDIR_MASK;
    src_pgd++;
    dst_pgd++;
  } /* end pgd loop */
  vma->vm_ops = NULL; /* the child now has anonymous memory */
  vma->vm_pte = 0; 
}

static void reg_vm_close(struct vm_area_struct *vma) 
{
#if 0
  gm_port_state_t *ps;
  ps = (gm_port_state_t*)vma->vm_pte;
  ps->arch.ref_count -= 1;
  gm_always_assert(ps->arch.ref_count > 1);
#endif
}

static struct vm_operations_struct gm_reg_vm_ops = {
        reg_vm_open,			/* open */
	reg_vm_close,			/* close */
	NULL,			/* unmap */
	NULL,			/* protect */
	NULL,			/* sync */
	NULL,			/* advise */
	NULL,      		/* nopage */
	NULL,			/* wppage */
	NULL,			/* swapout */
	NULL,			/* swapin */
};

int gm_linux_reg_page(gm_arch_page_lock_t *lock,unsigned long addr)
{
  struct vm_area_struct *vma;
  
  /* first arrange for some hook to be called on fork() */
  lock->vma = addr;
  vma = gm_vm_fixup(vma,lock->vma,lock->vma+GM_PAGE_LEN);
  if (!vma)
    return GM_OUT_OF_MEMORY;
  if (vma->vm_flags & VM_LOCKED) {
    GM_WARN(("gm_lock_user_buffer_page: page is already locked\n"));
  }
  gm_always_assert(vma->vm_ops == NULL);
  vma->vm_flags |= VM_LOCKED; /* disabled swapping on this page */
  vma->vm_ops = &gm_reg_vm_ops;
  gm_merge_segments(vma->vm_mm,vma->vm_start,vma->vm_end);
}

void gm_linux_unreg_page(gm_arch_page_lock_t *lock)
{
  struct vm_area_struct *vma = find_vma(gm_current->mm,lock->vma);
  if (vma->vm_ops == &gm_reg_vm_ops && (vma->vm_flags & VM_LOCKED) &&
      vma->vm_start <= lock->vma && vma->vm_end > lock->vma) {
    vma = gm_vm_fixup(vma,lock->vma,lock->vma+GM_PAGE_LEN);
    vma->vm_ops = 0;
    vma->vm_flags &= ~VM_LOCKED;
    gm_merge_segments(vma->vm_mm,vma->vm_start,vma->vm_end);
  } else {
    GM_WARN(("unlock_user_buffer_page: page 0x%lx has been manipulated in an unexpected way\n",lock->vma));
  }
}


#ifdef __alpha__
void (*gm_tbi) (long type,...);
void (*gm_get_new_asn_and_reload) (struct task_struct *, struct mm_struct *);
/* we need a wrapper here because function is declared and used in the same header */
void 
wrap_get_new_asn_and_reload(struct task_struct *task, struct mm_struct *mm)
{
	gm_get_new_asn_and_reload(task, mm);
}
#endif

void (*gm_merge_segments) (struct mm_struct *, unsigned long, unsigned long);
void (*gm_insert_vm_struct) (struct mm_struct *, struct vm_area_struct *);


void gm_mm_init(void)
{
#ifdef __alpha__
#ifdef HAS_GET_MODULE_SYMBOL
  gm_tbi = get_module_symbol(0, "tbi");
  gm_get_new_asn_and_reload = get_module_symbol(0, "get_new_asn_and_reload");
#endif
  if (!gm_tbi || !gm_get_new_asn_and_reload)
    gm_full_mm = 0;
#endif
  gm_smp_flush_tlb = get_module_symbol(0, "smp_flush_tlb");
  if (!gm_smp_flush_tlb)
    gm_full_mm = 0;
#ifdef HAS_GET_MODULE_SYMBOL
  gm_merge_segments = get_module_symbol(0, "merge_segments");
  gm_insert_vm_struct = get_module_symbol(0, "insert_vm_struct");
#endif
  if (!gm_merge_segments || !gm_insert_vm_struct)
    gm_full_mm = 0;
  if (!gm_full_mm) {
    GM_WARN(("LIMITED FUNCTIONALITY: fork, mremap,... will not work correctly in GM applications\n"));
  }
}

#endif /* GM_LINUX_FULL_MM */
