/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: maxstern@myri.com */

/*
 $Id: gm_irix.c,v 1.36 2000/12/20 23:54:07 maxstern Exp $
 */

/************************************************************************
 * This file includes the IRIX-specific driver code, specifically, the data
 * and functions which interface the gm driver to the IRIX kernel.
 *
 * This driver supports IRIX 6.5 and up *only*.  The "bible" for IRIX 6.5
 * device driver implementation is "IRIX Device Driver Programmer's
 * Guide," SGI Document Number 007-0911-130.
 *
 * NOTE: For a Table of Contents of the subdirectory containing this source
 *       file, see ./README_TOC.
 *
 ************************************************************************/


/************************************************************************
 *
 * Table of Contents of this source file (Data and Function Directory);
 * subroutines shown indented:
 * -------------------------------------------------------------------
 * #include directives
 * Type and Macro definitions
 * "Forward" function prototype declarations
 * Global Data
 *    External Linkage Data
 *    Static (Internal Linkage) Data
 * IP Support Functions
 *    gm_irix_get_vertex()
 *    gm_irix_port_use_A64_dma()
 * IRIX Kernel Required Functions
 *    <pfx>init()                 Driver initialization
 *    <pfx>reg()                  Register the driver with IRIX
 *    <pfx>attach()               Device initialization
 *    <pfx>open()                 
 *       gm_irix_open_from_ioconfig()
 *          gm_irix_intr_alloc() 
 *       gm_irix_open() 
 *          gm_irix_clone_open() 
 *          gm_irix_minor_open() 
 *    <pfx>ioctl()
 *       gm_irix_clone_ioctl()
 *       gm_irix_minor_ioctl()
 *    <pfx>map()
 *       gm_irix_clone_map()
 *       gm_irix_minor_map()
 *    <pfx>unmap()
 *       gm_irix_clone_unmap()
 *       gm_irix_minor_unmap()
 *    <pfx>close()
 *       gm_irix_clone_close()
 *       gm_irix_minor_close()
 *    <pfx>unload()               Driver termination
 *    <pfx>detach()               Device termination
 *    <pfx>unreg()                Driver de-registration
 * Utility functions to support the above
 *    gm_irix_unloadme()          Utility function to prepare for reload
 *    gm_irix_unload_gm()
 *    gm_irix_reloadme()          Utility function to support reload
 *    gm_irix_reload_gm()
 *    gm_irix_setup_clone_master_pstates()
 *    gm_irix_make_master_pstate()
 *    gm_irix_delete_clone_master_pstates()
 *    gm_irix_delete_minor()
 *    gm_irix_device_add()
 *    gm_irix_edge_add()
 *    gm_irix_edge_remove()
 *    gm_irix_device_destroy()
 *    gm_irix_record_new_vhdl()
 *    gm_irix_find_old_vhdl()
 *    gm_irix_minor_for_vhdl()
 *    gm_irix_device_hash_compare()
 *    gm_irix_device_hash_hash()
 *    gm_irix_minor_hash_compare()
 *    gm_irix_minor_hash_hash()
 *    gm_irix_enable_interrupts()
 *    gm_irix_disable_interrupts()
 *    gm_irix_intr_free() 
 *    gm_irix_intr()              Interrupt handler declared to PCI
 *    gm_irix_ehandlr()           Error handler declared to PCI
 *    gm_irix_init_dev_lock()
 *    gm_irix_load_config_values()
 * Debugging support
 *    gm_irix_display_global_parms()
 *    gm_irix_test_basic_functions()
 *    gm_irix_display_dev_desc()
 *    gm_irix_display_config_values()
 *    gm_irix_display_config_space()
 *    gm_irix_display_instance_values()
 *    gm_irix_display_interrupt_handle()
 *    gm_irix_parse_devt()
 *    gm_irix_displ_invent()
 *    gm_irix_display_setpn_args()
 *    __C_runtime_error()
 * 
 ************************************************************************/


/************************************************************************
 * TO DO:
 *
 *  * See the file ./README.todo for a (hopefully) comprehensive list of
 *    remaining implementation and cleanup tasks.
 *
 *  * I have used "////" strings in this and other source files to call
 *    attention to specific code sequences where work is still needed.
 *
 ************************************************************************/


/************************************************************************
 * The following #include's are only those directly needed to support the
 * statements in this program text file.  For more information on this
 * practice, see "README.manifesto" in this subdirectory.
 ************************************************************************/

/* ////// the work implied by the above comment has not been done yet ///// */

         /* *** IRIX header files (in collating sequence)              *** */
#include <assert.h>
#include <fcntl.h>
#include <limits.h>        /* needed *only* for INT_MAX                */
#include <stdarg.h>        /* needed for va_start, etc.                */

#include <ksys/ddmap.h>    /* i/o space mapping                        */

#include <sys/buf.h>       /* buf_t, etc.                              */
#include <sys/cmn_err.h>   /* error handling                           */
#include <sys/conf.h>
#include <sys/cred.h>
#include <sys/dmamap.h>    /* DMA stuff                                */
#include <sys/debug.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/graph.h>     /* graph_error_t, etc.                      */
#include <sys/hwgraph.h>   /* dev_to_vhdl, etc.                        */
#include <sys/invent.h>    /* device_inventory_add() etc.              */
#include <sys/iobus.h>     /* intr_arg_t etc.                          */
#include <sys/ioerror.h>   /* error_handler_f etc.                     */
#include <sys/kmem.h>      /* kmem_ func's etc. ////// not needed ??   */
#include <sys/ksynch.h>    /* locks                                    */
#include <sys/mkdev.h>     /* mkdev, dev_t, etc.                       */
#include <sys/mload.h>     /* M_VERSION, etc.                          */
#include <sys/mman.h>      /* PROT_READ, etc.                          */
#include <sys/open.h>      /* OTYP_CHR, etc.                           */
#include <sys/sema.h>      /* locks & synchs                           */
#include <sys/systm.h>     /* sprintf, vsprintf, etc.    */
#include <sys/types.h>     /* basic, incl. vertex_hdl_t                */

#include <netinet/in.h>

#include <sys/PCI/pciio.h> /* PCI                                      */

/* The IRIX man page intro(D5) says this header file "must always be   */
/* the last header file included":                                     */
#include <sys/ddi.h>       /* DDI driver stuff                         */

         /* *** GM header files (approximately in dependency sequence    *** */
#include "gm_arch_local.h" /* declarations needed only in this subdirectory  */
#include "gm_call_trace.h"
#include "gm_compiler.h"
#include "gm_impl.h"       /* platform-independent, needed by gm.c           */
#include "gm_instance.h"   /* platform-independent */
#include "gm_internal.h"   /* platform-independent */
#include "gm_lanai.h"      /* platform-independent */
#include "gm_debug_host_dma.h"
#include "gm_debug_instance_state.h"
#include "gm_debug_mem_register.h"
#include "gm_debug_port_state.h"

#if GM_IRIX_IFNET_READY
extern int gx_attach (int unit);
#endif

/***********************************************************************
 * Type and Macro definitions used only in this source module
 ***********************************************************************/

/* See also ./gm_arch_local.h. */

/*
 * Device hash: key:     {GM unit, GM minor number}
 *              returns: IRIX devt
 */
typedef struct gm_irix_device_hash_key
{
   unsigned int    unit;	/* istate->id */
   gm_arch_minor_t minor;
} gm_irix_device_hash_key_t;


/*
 * Minor hash: key:     IRIX devt
 *             returns: GM minor number
 */
typedef dev_t gm_irix_minor_hash_key_t;


/***********************************************************************
 * Forward declarations
 ***********************************************************************/

/*
 * In some OS's, a list of forward declarations (function prototypes) is
 * needed here, for use (e.g.) in function tables that form part of the
 * interface between the driver and the kernel.
 *
 * IRIX does not have this requirement, so we put forwards here *only* for
 * internal utility subroutines as needed.
 *
 * Note that the organization used in this module is to place subroutines
 * lexically following their callers; this layout, while logical to the
 * reader, makes it necessary to use forward declarations.  Here they are:
 */

GM_STATIC gm_status_t gm_irix_open_from_ioconfig(gm_instance_state_t *the_is,
                                                 dev_t               *the_devp);

GM_STATIC gm_status_t gm_irix_intr_alloc(gm_instance_state_t *the_is);

GM_STATIC gm_status_t gm_irix_open(gm_arch_minor_t the_minor);

GM_STATIC gm_irix_u_func_t   gm_irix_clone_open,
                             gm_irix_minor_open;

GM_STATIC gm_irix_io_func_t  gm_irix_clone_ioctl,
                             gm_irix_minor_ioctl;

GM_STATIC gm_irix_map_func_t gm_irix_clone_map,
                             gm_irix_minor_map;

GM_STATIC gm_irix_u_func_t   gm_irix_clone_unmap,
                             gm_irix_minor_unmap;

GM_STATIC gm_irix_u_func_t   gm_irix_clone_close,
                             gm_irix_minor_close;

GM_STATIC pciio_iter_f gm_irix_unloadme;
GM_STATIC pciio_iter_f gm_irix_reloadme;

GM_STATIC void gm_irix_unload_gm(gm_instance_state_t *the_is);

#define GM_IRIX_ENABLE_RELOAD 0
#if GM_IRIX_ENABLE_RELOAD
GM_STATIC void gm_irix_reload_gm(gm_instance_state_t *the_is);
#endif /* GM_IRIX_RELOAD_GM */

GM_STATIC gm_status_t gm_irix_setup_clone_master_pstates(gm_instance_state_t
                                                                      *the_is);

GM_STATIC gm_status_t gm_irix_make_master_pstate(gm_instance_state_t *the_is,
				                 char               privileged,
                                                 gm_port_state_t    **the_ps);

GM_STATIC void gm_irix_delete_clone_master_pstates(gm_instance_state_t
                                                                      *the_is);

GM_STATIC void gm_irix_delete_minor(gm_port_state_t *the_ps);

GM_STATIC gm_status_t gm_irix_device_add(vertex_hdl_t from,
				         char         *name,
				         vertex_hdl_t *new_vhdl,
                                         int          priv_flag);

GM_STATIC gm_status_t gm_irix_edge_add(vertex_hdl_t vhdl, char *name);

GM_STATIC void        gm_irix_edge_remove(vertex_hdl_t *vhdl, char *name);

GM_STATIC void        gm_irix_device_destroy(vertex_hdl_t vhdl);

GM_STATIC gm_status_t gm_irix_record_new_vhdl(unsigned int    the_unit,
                                              gm_arch_minor_t the_gm_minor,
                                              vertex_hdl_t    the_vhdl);

GM_STATIC vertex_hdl_t gm_irix_find_old_vhdl(unsigned int    the_unit,
                                             gm_arch_minor_t the_gm_minor);

GM_STATIC int          gm_irix_minor_for_vhdl(vertex_hdl_t    the_vhdl,
                                              gm_arch_minor_t *the_minor);

GM_STATIC long          gm_irix_device_hash_compare(void *a, void *b);

GM_STATIC unsigned long gm_irix_device_hash_hash(void *a);

GM_STATIC long          gm_irix_minor_hash_compare(void *a, void *b);

GM_STATIC unsigned long gm_irix_minor_hash_hash(void *a);

GM_STATIC gm_status_t gm_irix_enable_interrupts(gm_instance_state_t *the_is);

GM_STATIC void        gm_irix_disable_interrupts(gm_instance_state_t *the_is);

GM_STATIC void gm_irix_intr_free(gm_instance_state_t *the_is);

GM_STATIC intr_func_f  gm_irix_intr;

GM_STATIC error_handler_f gm_irix_ehandlr;

GM_STATIC void gm_irix_load_config_values(gm_instance_state_t *the_is,
					  gm_pci_config_t     *cfg_data);

GM_STATIC void gm_irix_init_dev_lock(gm_instance_state_t *the_is);

#if GM_IRIX_DISPLAY_GLOBAL_PARMS
GM_STATIC void gm_irix_display_global_parms(void);
#endif

#if GM_DEBUG
GM_STATIC void gm_irix_test_basic_functions(gm_instance_state_t *the_is);
#endif


/* FOLLOWING DEFINE is a temporary hack due to the unsolved 64-bit PIO
 * bug (Problem index number 01-003).
 */
#define GM_DONT_TRY_64_BIT_BAR_DIE_DIE 0


/* HERE we are experimenting to find the right locking strategy.
 */
#define GM_IRIX_USE_OPEN_AND_CLOSE_SYNC 0


/***********************************************************************
 * Global data
 ***********************************************************************/

/*
 * External global data required by IRIX 6.5 kernel
 */

int  GM_PFXNM_DEVFLAG   = D_MP;      /* specify driver attributes to lboot */
char *GM_PFXNM_MVERSION = M_VERSION; /* mload ver. to make driver loadable */


/*******************************************************************
 * NOTE: Global data used throughout irix/gm is defined in gm_arch.c
 *******************************************************************/


/*
 * Global data internal to this module
 *
 * NOTE: ***ANY*** global data, whether declared here or anywhere else in
 *       the driver, must ***NOT*** be used across reloads!  Any data that
 *       is to survive an unload/reload must reside in the instance_state.
 */

/*
 * The open_and_close sync structure is used to protect calls to the GM
 * infrastructure that allocate and deallocate minor numbers, port_states,
 * et al.  The GM infrastructure is not MP-safe with respect to these
 * resources.
 */
#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
GM_STATIC gm_arch_sync_t *gm_irix_open_and_close_sync;
#endif

GM_STATIC int            gm_irix_state = GM_SUCCESS;
GM_STATIC int            gm_irix_instance_count = 0;
GM_STATIC struct gm_hash *gm_irix_device_hash = (struct gm_hash *)NULL;
GM_STATIC struct gm_hash *gm_irix_minor_hash  = (struct gm_hash *)NULL;

GM_STATIC gm_arch_sync_t *gm_irix_device_hash_sync;
GM_STATIC gm_arch_sync_t *gm_irix_minor_hash_sync;


/***********************************************************************
 * Support functions for the ip driver
 ***********************************************************************/

vertex_hdl_t
gm_irix_get_vertex(gm_port_t*p)
{
  return p &&
	 p->kernel_port_state &&
	 p->kernel_port_state->instance ?
    p->kernel_port_state->instance->arch.conn : 0;
}

gm_boolean_t
gm_irix_port_use_A64_dma(gm_port_t*p)
{
#if GM_IRIX_64BIT_DP_READY
   gm_instance_state_t *is;


   is = (p && p->kernel_port_state) ? p->kernel_port_state->instance : 0;
   if (is)
      return(is->flags & GM_INSTANCE_64BIT_DMA_ADDR_OK);
   else
      return(0);
#else
   GM_PARAMETER_MAY_BE_UNUSED (p);
   return(0);
#endif
}


/************************************************************************
 * IRIX kernel-required functions
 *
 * Description:
 *
 * These are the functions that are required by the IRIX 6.5 kernel.  For
 * more information about the macro-generated function names, see gm_arch.h.
 * For more information about the function specifications, see the IRIX Device
 * Driver Programmer's Guide.
 *
 * The function definitions are given in roughly logical sequence.
 ************************************************************************/

/*
 * <pfx>init()
 *
 * Description:  This is the first function that the IRIX kernel calls upon
 *               loading the driver, making this the driver-initialization
 *               entry point; for the device-initialization entry point, see
 *               <pfx>attach.
 *
 * Note:         Because this is a loadable and unloadable driver, it is
 *               possible for the driver to be loaded multiple times while
 *               IRIX continues to run.  Therefore, we provide the ability to
 *               unlink and relink the interrupt and error handlers.  The
 *               algorithm used here is based on the example PCI driver in the
 *               IRIX device driver programmer's guide.
 *
 *               In particular, this function will also be called when the
 *               driver is *unregistered*; IRIX intends that the driver be
 *               given the opportunity to prepare itself for the <pfx>unreg()
 *               call.
 *
 * NOTE:         It has been reported (by Rod Freier at Utah) that interrupt
 *               handling doesn't work for non-loadable drivers.  We have not
 *               run into this problem with this driver, since it was designed
 *               from the beginning to be loadable.  We note Rod's report here
 *               only as a warning to the unaware.
 */

#define GM_FNAME GM_PFXNM_INIT

void
GM_PFXNM_INIT(void)
{
   char *usage_device_hash    = "device_hash";
   char *usage_minor_hash     = "minor_hash";
#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
   char *usage_open_and_close = "open_and_close";
#endif

/* Must initialize this mutex before doing any prints                */ 
   MUTEX_INIT(&gm_irix_print_mu, MUTEX_DEFAULT, "GM print mutex");

   gm_need_kvtophys_analysis = 1;
#if GM_IRIX_USE_MONSTER
   bzero(&gm_monster_debug_buf, GM_MONSTER_DEBUG_BUF_SIZ + GM_MONSTER_TAG_SIZ);
   gm_monster_current = &(gm_monster_debug_buf[GM_MONSTER_DEBUG_BUF_SIZ]);
   strcpy(gm_monster_current, ">>>> END OF THE MONSTER DEBUG BUFFER <<<<\n\0");
   gm_monster_offset     = 0;
   gm_monster_wrap_tally = 0;
   gm_monster_current = &(gm_monster_debug_buf[0]);
#endif

   GM_ANNOUNCE_FNAME_ENTRY(1);

   GM_NOTE (("GM driver version: %s\n", _gm_version));

   _GM_NOTE (("    build: %s\n", _gm_build_id));

#if GM_IRIX_DISPLAY_GLOBAL_PARMS
   gm_irix_display_global_parms();
#endif

   /* set the page length global variable ASAP, as it is used all over. */

   if (GM_PAGE_LEN)            /* if already set (how could that happen?) */
      gm_always_assert(GM_PAGE_LEN == ptob(1)); /* Anyway, it better be correct! */
   else                        /*set it now */
      GM_PAGE_LEN = ptob(1);

#if GM_IRIX_DISPLAY_GLOBAL_PARMS
   GM_PRINT (1, ("page size is %d\n", GM_PAGE_LEN));
   GM_PRINT (1, ("Some macro-derived values:\n"));
   _GM_PRINT (1, ("    GM_RECV_QUEUE_SLOTS_PER_PAGE %d\n",
                  GM_RECV_QUEUE_SLOTS_PER_PAGE));
   _GM_PRINT (1, ("    GM_NUM_RECV_QUEUE_SLOTS      %d\n",
                  GM_NUM_RECV_QUEUE_SLOTS));
#endif

#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
   gm_irix_open_and_close_sync = (gm_arch_sync_t *)gm_create_mutex();
   if (!gm_irix_open_and_close_sync)
      GM_PANIC (("Couldn't allocate open_and_close sync\n"));

   if (strlen(usage_open_and_close) < GM_IRIX_SYNC_USAGE_LEN)
      strcpy(gm_irix_open_and_close_sync->usage, usage_open_and_close);
#endif

   gm_init();                        /* initialize GM */

   gm_assert(gm_irix_device_hash == (struct gm_hash *)NULL);
   gm_irix_device_hash = gm_create_hash(gm_irix_device_hash_compare,
					gm_irix_device_hash_hash,
					sizeof(gm_irix_device_hash_key_t),
					sizeof(vertex_hdl_t),
					16, 0);
   if (gm_irix_device_hash == (struct gm_hash *)NULL)
   {
      GM_WARN(("Could not allocate device hash table\n"));
      gm_always_assert(0);
   }
   GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
            ("Allocated device hash table at 0x%p\n", gm_irix_device_hash));
   gm_irix_device_hash_sync = (gm_arch_sync_t *)gm_create_mutex();
   if (!gm_irix_device_hash_sync)
      GM_PANIC (("Couldn't allocate device_hash sync\n"));

   if (strlen(usage_device_hash) < GM_IRIX_SYNC_USAGE_LEN)
      strcpy(gm_irix_device_hash_sync->usage, usage_device_hash);

   gm_assert(gm_irix_minor_hash == (struct gm_hash *)NULL);
   gm_irix_minor_hash = gm_create_hash(gm_irix_minor_hash_compare,
				       gm_irix_minor_hash_hash,
				       sizeof(gm_irix_minor_hash_key_t),
				       sizeof(gm_arch_minor_t),
				       16, 0);
   if (gm_irix_minor_hash == (struct gm_hash *)NULL)
   {
      GM_WARN(("Could not allocate minor hash table\n"));
      gm_always_assert(0);
   }
   GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
            ("Allocated minor hash table at 0x%p\n", gm_irix_minor_hash));
   gm_irix_minor_hash_sync = (gm_arch_sync_t *)gm_create_mutex();
   if (!gm_irix_minor_hash_sync)
      GM_PANIC (("Couldn't allocate minor_hash sync\n"));

   if (strlen(usage_minor_hash) < GM_IRIX_SYNC_USAGE_LEN)
      strcpy(gm_irix_minor_hash_sync->usage, usage_minor_hash);

#if GM_CAN_REGISTER_MEMORY
   gm_irix_setup_memory_registration();
#endif

   pciio_iterate(GM_PREFIX_STRING, gm_irix_reloadme);  /* in case of reload */

   GM_ANNOUNCE_FNAME_EXIT(1);
}
#undef GM_FNAME


/*
 * <pfx>reg()
 *
 * Description:  This hook allows the driver to register itself with the IRIX
 *               kernel.  This entry point supports loadability of the driver.
 *
 * NOTE:         The IRIX Device Driver Guide states that the <pfx>attach()
 *               function can be called even before pciio_driver_register()
 *               has returned to this function.  That doesn't seem to be a
 *               hazard, as long as we take care not to modify any global data
 *               while we're in this hook.
 */

#define GM_FNAME GM_PFXNM_REG
int
GM_PFXNM_REG(void)
{
   register int ret;


   GM_ANNOUNCE_FNAME_ENTRY(1);

   GM_PRINT (GM_PRINT_LEVEL >= 1, ("Registering driver as:\n"));
   _GM_PRINT (GM_PRINT_LEVEL >= 1,
              ("    Vendor = 0x%x, Device = 0x%x, Prefix = %s\n",
               GM_PCI_VENDOR_MYRICOM,GM_PCI_DEVICE_MYRINET,GM_PREFIX_STRING));

   /*    Register, identifying our card and the associated prefix string   */
   ret = pciio_driver_register(GM_PCI_VENDOR_MYRICOM,
                               GM_PCI_DEVICE_MYRINET, 
                               GM_PREFIX_STRING,
                               0);

   if (ret)
   {
      GM_WARN(("pciio_driver_register(GM_PCI_VENDOR_MYRICOM...) returned %d\n",
               ret));
      GM_ANNOUNCE_FNAME_RETURN(1, ret);
   }

   GM_PRINT (GM_PRINT_LEVEL >= 1, ("Registering driver as:\n"));
   _GM_PRINT (GM_PRINT_LEVEL >= 1,
              ("    Vendor = 0x%x, Device = 0x%x, Prefix = %s\n",
               GM_PCI_VENDOR_MYRICOM2,GM_PCI_DEVICE_MYRINET,GM_PREFIX_STRING));

   /*    Register, identifying our card and the associated prefix string   */
   ret = pciio_driver_register(GM_PCI_VENDOR_MYRICOM2,
                               GM_PCI_DEVICE_MYRINET, 
                               GM_PREFIX_STRING,
                               0);

   if (ret)
     GM_WARN(("pciio_driver_register(GM_PCI_VENDOR_MYRICOM2...) returned %d\n",
              ret));

   GM_ANNOUNCE_FNAME_RETURN(1, ret);
}
#undef GM_FNAME


/*
 * <pfx>attach()
 *
 * Description:
 *    IRIX has found a myrinet card; it calls the <pfx>attach() function
 *    to attach the driver to the device.  In other words, <pfx>attach is
 *    the *device-instance* initialization point, as contrasted with
 *    <pfx>init,  which is the *driver* initialization point.
 *
 *    Instance-initialization includes the following steps:
 *
 *    o  Allocate memory for the instance_state data structure
 *    o  Set up PIO mapping for the IO space of the PCI bus
 *    o  Create vertex /hw/.../gm<n> in the hwgraph
 *    o  Call gm_instance_init() to do platform-independent initialization
 *    o  Add vertexes to the IRIX hwgraph
 *    o  Save the instance_state data structure in the hardware graph vertex
 *    o  Allocate, connect, and enable the interrupt handler
 *    o  Create the minor devices
 *
 * NOTES: 
 *    1. This driver supports *only* PCI connections (GM_MYRINET_BUS_PCI).
 *
 *    2. Currently, we are only creating the /hw/gm<n> vertex.  It is
 *       likely possible to create a /dev/gm<n> "file" which is a symlink
 *       to the corresponding /hw/gm<n> "file".  We don't think it is
 *       *necessary* to do this, so we are not attempting it yet.
 *
 *    3. See the NOTE under <pfx>reg() above regarding asynchrony between
 *       that function and this one.
 *
 *    4. A hazardous area here is the initialization of the instance_state.
 *       Before we can call gm_xxx() functions, some of its fields must be
 *       filled in, but others of them are fillable only after such calls.
 *       For example, we need the vhdl filled in before we can call
 *       gm_instance_init(), because it's needed by pci_config_get().  Our
 *       strategy is: (a) fill in fields as soon as we can, and (b) try to
 *       do things in an order which recognizes such dependencies.
 *
 *    5. Similarly, it is essential not to save pointers in the instance_
 *       state which may become invalid when the driver is unloaded and
 *       reloaded.  An early bug involved storing the device_desc_t pointer
 *       in the instance_state, on the principle that we need it various
 *       places in the code.  NOT.  Since the driver is loaded and unloaded
 *       at system boot, and then reloaded after all drivers have had their
 *       shot, the kernel's device_desc locations are not in the same place
 *       on the second time around.  The solution: always call
 *       default_device_desc_get() when we need the device_desc_t.
 *
 *    6. Don't be misled by the function name "device_inventory_add."  This
 *       name does not mean that the function adds a *device* to the inventory.
 *       It means that the function adds *information* to the inventory.
 */

#define GM_FNAME GM_PFXNM_ATTACH
int
GM_PFXNM_ATTACH(vertex_hdl_t conn_vhdl)
{
   gm_instance_state_t  *gm_is;
   vertex_hdl_t         nonp_vhdl,
                        priv_vhdl;


   GM_ANNOUNCE_FNAME_ENTRY(1);

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(conn_vhdl);
#endif

/* First add our non-privileged and privileged clone device vertexes
 * to the hwgraph
 */

   if (gm_irix_device_add(conn_vhdl, GM_IRIX_NONP_HWNAME, &nonp_vhdl, 0)
         != GM_SUCCESS)
      goto abort_with_nothing; 

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(nonp_vhdl);
#endif

   if (gm_irix_device_add(conn_vhdl, GM_IRIX_PRIV_HWNAME, &priv_vhdl, 1)
         != GM_SUCCESS)
      goto abort_with_nonp_device;

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(priv_vhdl);
#endif

/* Now we can get a gm_instance data structure                  */

   gm_is = (gm_instance_state_t *)gm_irix_kmemzalloc(sizeof(*gm_is),
                                                   KM_NOSLEEP+KM_CACHEALIGN);
   if (!gm_is)  /* if allocation failed */
   {
      GM_WARN( ("Failed to allocate instance_state memory\n"));
      goto abort_with_priv_device;
   }

/*
 *    Immediately, fill in as much of the structure as we can (see NOTE
 *    #4 in the function prolog above), and add it to the hwgraph vertex.
 */

/* We won't know the controller number until gm_irix_open_from_ioconfig() */ 
   gm_is->arch.controller      = -1;
   gm_is->arch.controller_susp = -1;
   gm_is->arch.conn         = conn_vhdl;
   gm_is->arch.pci_acc_vhdl = conn_vhdl;
   gm_is->arch.kernel_abi   = get_current_abi();  /* this will equal
                                              ABI_IRIX5_64 for 64-bit kernel */
   gm_is->arch.conn_devt    = vhdl_to_dev(conn_vhdl);
   gm_is->arch.nonpriv_vhdl = nonp_vhdl;
   gm_is->arch.priv_vhdl    = priv_vhdl;
   gm_is->arch.nonpriv_devt = vhdl_to_dev(nonp_vhdl);
   gm_is->arch.priv_devt    = vhdl_to_dev(priv_vhdl);
   strcpy(gm_is->arch.clone_edge_name, "");
   strcpy(gm_is->arch.p_clone_edge_name, "");
   /*
    * Initialize our device sync
    */
   gm_irix_init_dev_lock(gm_is);

   /*  save our instance_state structure in both vertexes */
   GM_IRIX_INSTANCE_SET_VH(nonp_vhdl, (void *)gm_is);
   GM_IRIX_INSTANCE_SET_VH(priv_vhdl, (void *)gm_is);

   /* =========================
    * PCI PIO Mapping will be done later, in gm_arch_map_io_space()
    * ========================= */

#if 0
/* Experimental disablement of BUS MASTER bit */
   if (gm_disable_PCI_bus_master_DMA(gm_is) != GM_SUCCESS)
      GM_PANIC (("Couldn't disable bus master DMA\n"));
#if GM_IRIX_DISPLAY_CONFIG_VALUES
   gm_irix_display_config_space(gm_is, "after disabling bus master DMA");
#endif
#endif

/* This call, by adding the indicated attributes to the device, ensures
 * that ioconfig will call <pfx>open() with our new dev_t, so that we can
 * learn our stable controller (instance) number.  See the file
 * ../myrigm.ioconfig for more information.  The value of INV_NET_MYRINET
 * must match the INV_TYPE value specified in myrigm.ioconfig.
 *
 * NOTE: The symbol INV_NET_MYRINET has been introduced by SGI/IRIX effective
 *       with IRIX 6.5.8.  The preprocessor logic below ensures that GM will
 *       work correctly with IRIX releases prior to that one.
 *
 *       See additional notes in ../myrigm.ioconfig.
 */

#ifdef INV_NET_MYRINET
#if (INV_NET_MYRINET != 11)
#error Error: INV_NET_MYRINET has wrong value - notify help@myri.com
#stop  (force compilation to quit)
#endif
#else
#define INV_NET_MYRINET 11
#endif

   GM_PRINT (GM_PRINT_LEVEL >= 1,
             ("Calling device_inventory_add() for vhdl %d\n", nonp_vhdl)); 
   device_inventory_add(vhdl_to_dev(nonp_vhdl), /* device (dev_t) */
                        INV_NETWORK,            /* class          */
                        INV_NET_MYRINET,        /* type           */
                        100,                    /* major (ctrlr)  */
                        -1,                     /* minor (unit)   */
                        0                   );  /* state          */

#if GM_IRIX_DISPLAY_INVENTORY
   gm_irix_displ_invent(vhdl_to_dev(conn_vhdl), "after device_inventory_add");
   gm_irix_displ_invent(vhdl_to_dev(nonp_vhdl), "after device_inventory_add");
#endif

/*** Set up error handling for the device                        ***/
   pciio_error_register(conn_vhdl, &gm_irix_ehandlr, gm_is);
   gm_is->arch.ehandlr_registered = 1;

/* We will do all interrupt-handling setup in the open routine.           */

/* NOTE: We don't use GM_SUCCESS as our return code, because we are
         communicating with the IRIX kernel, not with GM or the end user. */

#if GM_DEBUG_INSTANCE_STATE
   gm_irix_display_instance_values(gm_is, "at " GM_FNAME_STR " exit");
   delay((long)drv_usectohz(1000000)); /* Hack to allow display to complete */
#endif
#if GM_IRIX_DISPLAY_CONFIG_VALUES
   gm_irix_display_config_space(gm_is, "at " GM_FNAME_STR " exit");
   delay((long)drv_usectohz(1000000)); /* Hack to allow display to complete */
#endif
   
   GM_ANNOUNCE_FNAME_RETURN(1, 0);


/********************* Error handling -- unroll what we've done ***********/

abort_with_priv_device:
   gm_irix_device_destroy(priv_vhdl);
abort_with_nonp_device:
   gm_irix_device_destroy(nonp_vhdl);
abort_with_nothing:
/* Hack: try to make sure we unregister ourselves with IRIX in <pfx>unload() */
   gm_irix_state = GM_FAILURE;
   /* GM_UNATTACHED not really apt, because it implies the lanai is not
    * running.
    */
   GM_ANNOUNCE_FNAME_RETURN(1, ENOENT);

} /* <pfx>attach() */

#undef GM_FNAME

/*
 * <pfx>open()
 *
 * Description:  There are two possibilities.
 *
 *    1. "open from ioconfig".
 *
 *    At system startup time, ioconfig has determined that our device needs
 *    to be assigned a controller number.  It assigns the number and calls
 *    <pfx>open() to give us a chance to recognize this number.
 *
 *    2. "Clone open", or port open.
 *
 *    A user process has done an 'open' on our device special file
 *    ("clone" device).  IRIX has determined that this driver is responsible
 *    for the device; it calls this function to perform driver-specific
 *    open-time actions.  We create a new hwgraph vertex, and we allocate
 *    a new minor number and uninitialized gm_port_state.  Later, a
 *    GM_SET_PORT_NUM ioctl will allow the gm_port_state to be fully
 *    initialized, and we expect the user process to close the clone device
 *    and open the newly created minor device, which belongs to it alone.
 *
 * Specifications:
 *
 *    Function return: 0, indicating success; or,
 *                     a failure code chosen from sys/errno.h (we don't use a
 *                     GM error code because we are returning to IRIX, not GM).
 *
 *    Input parms: devp  points to a dev_t identifying the device
 *                 oflag bit flags defining the mode of open (we expect
 *                       F_READ and F_WRITE)
 *                 otyp  code defining the type of open (we expect OTYP_CHR)
 *                 crp   handle for this device when communicating with the
 *                       kernel about access permissions
 *
 *    Output parm: NONE  Returning a new dev_t through the *devp
 *                       parameter is not legal in IRIX for a character
 *                       driver, but *only* for a clone driver, which
 *                       must be a STREAMS driver.
 *
 * NOTE: ioconfig assigns controller numbers *within PCI class*.  We declare
 *       our PCI class in the device_inventory_add call in <pfx>attach(). 
 *
 * NOTE: The term "clone" is used above in the GM sense, and not in the narrow
 *       IRIX sense; the latter refers to a driver which is dependent on the
 *       IRIX "clone" driver.
 *
 * NOTE: Following the <pfx>open() call, ioconfig also calls <pfx>close().
 *
 */

#define GM_FNAME GM_PFXNM_OPEN
int
GM_PFXNM_OPEN(dev_t *devp, int oflag, int otyp, cred_t *crp)
{
   gm_instance_state_t *gm_is;       /* instance-state for this device       */
   gm_status_t         ret_code;     /* GM status code, to be translated     */


   GM_PARAMETER_MAY_BE_UNUSED (oflag);                    /* If not GM_DEBUG */
   GM_PARAMETER_MAY_BE_UNUSED (crp);                      /* If not GM_DEBUG */
   GM_PRINT (GM_PRINT_LEVEL >= 1,
             (GM_FNAME_STR " entered with %p->%d, 0x%x, 0x%x,\n",
              devp, *devp, oflag, otyp));
   _GM_PRINT (GM_PRINT_LEVEL >= 1, ("    %p\n", crp));

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(*devp);
#endif
#if GM_IRIX_DISPLAY_INVENTORY
   gm_irix_displ_invent(*devp, "at entry to " GM_FNAME_STR);
#endif

   if (otyp != OTYP_CHR)
   {
      GM_WARN((GM_FNAME_STR "() called with bad otyp %d\n", otyp));
      GM_ANNOUNCE_FNAME_RETURN(1, EINVAL);
   }

   /* Find our instance_state structure for this device */

   GM_IRIX_INSTANCE_GET_DV(*devp, gm_is);  /* Ask kernel for our instance */

   if (!gm_is)
      /* OOPS! Should have been there!!??!! */
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);

#if GM_DEBUG_INSTANCE_STATE
   gm_irix_display_instance_values(gm_is, "in " GM_FNAME_STR);
#endif
#if GM_IRIX_DISPLAY_DEV_DESC
   gm_irix_display_dev_desc(0, gm_is, "in " GM_FNAME_STR);
#endif

#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC && 0
   gm_mutex_enter((struct gm_mutex *)gm_irix_open_and_close_sync);
#endif
   gm_mutex_enter((struct gm_mutex *)&gm_is->arch.device_lock);

   if (gm_is->arch.controller < 0)
      ret_code = gm_irix_open_from_ioconfig(gm_is, devp);
   else
   {
      gm_arch_minor_t minor_num;

      if (*devp == gm_is->arch.nonpriv_devt)
         minor_num = gm_is->clone_minor;
      else if (*devp == gm_is->arch.priv_devt)
         minor_num = gm_is->privileged_clone_minor;
      else if (!gm_irix_minor_for_vhdl(dev_to_vhdl(*devp), &minor_num))
         GM_PANIC(("Unknown devp %d in " GM_FNAME_STR, *devp));

      ret_code = gm_irix_open(minor_num);
   }

#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC && 0
   gm_mutex_exit((struct gm_mutex *)gm_irix_open_and_close_sync);
#endif
   gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);

   GM_ANNOUNCE_FNAME_RETURN(1, gm_irix_localize_status(ret_code));

} /* <pfx>open() */

#undef GM_FNAME

 /*
 * First-open processing.  See <pfx>open() for more details.
 *
 * NOTE:  No matter how many devices we have created at our connect vertex,
 *        ioconfig calls <pfx>open() only once per instance, which suits
 *        our purposes perfectly.
 */
#define GM_FNAME gm_irix_open_from_ioconfig

GM_STATIC gm_status_t
GM_FNAME(gm_instance_state_t *the_is, dev_t *the_devp)
{
   vertex_hdl_t        our_vhdl;     /* hwgraph vertex handle for device */
   int                 ctrl_num;     /* Our controller number */
   gm_status_t         ret_code;     /*  */
   char                clone_name[GM_IRIX_CLONE_NAME_SZ], /*  */
                       p_clone_name[GM_IRIX_PCLONE_NAME_SZ]; /*  */
#if GM_DEBUG
   int                 seq_bit;
   gm_status_t         gm_status;    /*  */
#endif

   GM_PRINT (GM_PRINT_LEVEL >= 1,
             (GM_FNAME_STR " entered with is = %p, dev_t = %d\n",
              the_is, *the_devp));

#if GM_IRIX_DISPLAY_DEV_DESC
   gm_irix_display_dev_desc(0, the_is,
                            "before call to device_controller_num_get");
#endif

   our_vhdl = dev_to_vhdl(*the_devp);

   if (*the_devp == the_is->arch.nonpriv_devt)

      GM_PRINT (GM_PRINT_LEVEL >= 1,
                ("open from ioconfig for non-priv device %d (0x%x)\n",
                 *the_devp, *the_devp));

   else if (*the_devp == the_is->arch.priv_devt)

      GM_PRINT (GM_PRINT_LEVEL >= 1,
                ("open from ioconfig for privileged device %d (0x%x)\n",
	         *the_devp, *the_devp));

   else
   {
      GM_WARN(("Unexpected dev_t %d (0x%x) in " GM_FNAME_STR "\n",
	       *the_devp, *the_devp));
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_nothing;
   }

   ctrl_num = device_controller_num_get(our_vhdl);
   if (ctrl_num < 0)
   {
      GM_WARN(("Vertex missing controller number\n"));
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_nothing;
   }

   GM_PRINT (GM_IRIX_DEBUG_DEVNAME,
             ("Controller number for dev %d is %d\n", *the_devp, ctrl_num));

   the_is->arch.controller_susp = ctrl_num;

   gm_always_assert(ctrl_num <= GM_IRIX_MAX_CLONE_NUM);
   sprintf(clone_name, GM_IRIX_CLONE_NAME, ctrl_num);

   if (gm_irix_edge_add(the_is->arch.nonpriv_vhdl, clone_name) != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_nothing;
   }
   strcpy(the_is->arch.clone_edge_name, clone_name);

   sprintf(p_clone_name, GM_IRIX_PCLONE_NAME, ctrl_num);

   if (gm_irix_edge_add(the_is->arch.priv_vhdl, p_clone_name) != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_nonp_edge;
   }
   strcpy(the_is->arch.p_clone_edge_name, p_clone_name);

   GM_PRINT (GM_IRIX_DEBUG_DEVNAME, ("created hwgraph convenience edges\n"));

/*
 *    gm_initialize the interface, including mapping the board regions,
 *    copying eeprom, and starting the LANai control program (with
 *    interrupts disabled.)
 */
#if GM_IRIX_DISPLAY_CONFIG_VALUES
   gm_irix_display_config_space(the_is,
                                "from cfg space just before loading into is");
#endif
   gm_irix_load_config_values(the_is, &(the_is->ifc.pci.config));
#if GM_IRIX_DISPLAY_CONFIG_VALUES
   gm_irix_display_config_values(&(the_is->ifc.pci.config),
			        "from is->ifc just before gm_instance_init");
#endif

#if GM_DONT_TRY_64_BIT_BAR_DIE_DIE
   if (the_is->ifc.pci.config.Base_Addresses_Registers[1])
   {
      GM_WARN((
            "Bailing out - cannot support 64-bit BAR yet (problem 01-003)\n"));
      ret_code = GM_UNSUPPORTED_DEVICE;
      goto abort_with_priv_edge;
   }
#endif
#if 0
   gm_irix_programmed_panic("just before gm_instance_init");
#endif

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("before gm_instance_init");
#endif

   if ((GM_IRIX_INSTANCE_LIMIT) &&
       (gm_irix_instance_count >= GM_IRIX_INSTANCE_LIMIT))
   {
      GM_WARN(("Not initializing unit %d because limit %d has been reached\n",
	       ctrl_num, GM_IRIX_INSTANCE_LIMIT));
      goto abort_with_priv_edge;
   }

   if (gm_instance_init(the_is, ctrl_num, GM_MYRINET_BUS_PCI) != GM_SUCCESS)
   {
      GM_WARN(("Could not initialize unit [%d]\n", ctrl_num));
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_priv_edge;
   }
   GM_PRINT (GM_PRINT_LEVEL >= 1, ("Unit %d initialized\n", ctrl_num));
   the_is->arch.gm_instance_init = 1;
   gm_irix_instance_count++;

   gm_always_assert(the_is->id == ctrl_num);

#if GM_DEBUG
   gm_status = gm_read_correct_sequence_bit(the_is, &seq_bit);
   if (gm_status == GM_SUCCESS)
      GM_PRINT (GM_PRINT_LEVEL >= 1,
                ("gm_read_correct_sequence_bit() returned SUCCESS"
                  " with value %d\n", seq_bit));
   else
   {
      GM_PRINT (GM_PRINT_LEVEL >= 0,
                ("gm_read_correct_sequence_bit() returned FAILURE (%d)"
                  " with value %d\n", gm_status, seq_bit));
/*    Note that failure is normal for PCI REV 1, but a problem for REV 2  */
      if (the_is->ifc.pci.config.Revision_ID == 1)
         GM_NOTE (("This result is as expected for this REV 1 card\n"));
      else
      {
         GM_WARN(("This is probably a problem for this PCI REV %d card\n",
	          the_is->ifc.pci.config.Revision_ID));
         ret_code = GM_INTERNAL_ERROR;
         goto abort_with_priv_edge;
      }
   }
#endif

#if GM_DEBUG_INSTANCE_STATE
   gm_irix_display_instance_values(the_is, "after gm_instance_init");
#endif
#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("after gm_instance_init");
#endif

  /************
   * Allocate the "permanent clone master" minor port states
   ************/

   if (gm_irix_setup_clone_master_pstates(the_is) != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_initialized_instance;
   }

/*
   ////// The Device Driver Programmer's Guide suggests registering the
   ////// interrupt handler in the <pfx>attach() entry point.  However, it
   ////// seems to make more sense to wait until now, when we are ready to
   ////// call gm_enable_interrupts().  It might be a good idea, nevertheless.
   ////// to review this design decision with SGI et al.
 */
   if (gm_irix_intr_alloc(the_is) != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_pstates; 
   }

   if (gm_irix_enable_interrupts(the_is) != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_allocated_intr;
   }

#if GM_DEBUG
   gm_irix_test_basic_functions(the_is);
#endif

#if GM_IRIX_DISPLAY_DEV_DESC
   gm_irix_display_dev_desc(0, the_is, "after gm_irix_enable_interrupts");
#endif
#if GM_IRIX_DISPLAY_INTERRUPT_HANDLE
   gm_irix_display_interrupt_handle(the_is->arch.intr_obj,
                                    "after gm_irix_enable_interrupts");
#endif
#if GM_DEBUG_INSTANCE_STATE
   gm_irix_display_instance_values(the_is, "after gm_irix_enable_interrupts");
#endif

  /* success, so commit */

   GM_PRINT (GM_PRINT_LEVEL >= 1,
             ("Opened device with device number %d (0x%x)\n",
              *the_devp, *the_devp));

#if GM_IRIX_IFNET_READY
   GM_PRINT (GM_PRINT_LEVEL >= 0, ("Attaching the IP interface\n"));
   {
     int ipok = 0;
     
     ipok = gx_attach (ctrl_num);
     if (!ipok)
       {
	 GM_WARN(("IP interface attach FAILED\n"));
       }
     else
       {
	 GM_PRINT (GM_PRINT_LEVEL >= 1, ("IP interface attach ok\n"));
       }
   }
#else
    GM_NOTE (("NOT attaching the IP interface - not yet functional\n"));
#endif
   
   GM_ANNOUNCE_FNAME_RETURN(5, GM_SUCCESS);


abort_with_allocated_intr:
   gm_irix_intr_free(the_is);
abort_with_pstates:
   gm_irix_delete_clone_master_pstates(the_is);
abort_with_initialized_instance:
   gm_instance_finalize(the_is);
   the_is->arch.gm_instance_init = 0;
abort_with_priv_edge:
   gm_irix_edge_remove(&(the_is->arch.priv_vhdl), p_clone_name);
abort_with_nonp_edge:
   gm_irix_edge_remove(&(the_is->arch.nonpriv_vhdl), clone_name);
abort_with_nothing:
   GM_ANNOUNCE_FNAME_RETURN(1, ret_code);

} /* gm_irix_open_from_ioconfig() */

#undef GM_FNAME

/*
 * gm_irix_intr_alloc()
 *
 * Allocate interrupt "object" for this instance.
 */
#define GM_FNAME gm_irix_intr_alloc

GM_STATIC gm_status_t
GM_FNAME(gm_instance_state_t *the_is)
{
   gm_u8_t           int_pin;      /* Interrupt_Pin from config space  */ 
   pciio_intr_line_t int_line; 
   device_desc_t     dev_desc;
   pciio_intr_t      intr_object = (pciio_intr_t)NULL;


   if (GM_IRIX_DEBUG_WAKE)
      GM_ANNOUNCE_FNAME_ENTRY(0);

/* Although the field (the_is->pci.config.Interrupt_Pin) theoretically holds
 * the value we seek, it doesn't look like gm_instance_init() fills this in
 * for us, so we go grab the config space register here for ourselves.
 */
   gm_arch_read_pci_config_8(the_is,
                             GM_OFFSETOF(gm_pci_config_t, Interrupt_Pin),
                             &int_pin);
   switch(int_pin)
   {
   case 1:
      int_line = PCIIO_INTR_LINE_A;
      break;
   case 2:
      int_line = PCIIO_INTR_LINE_B;
      break;
   case 3:
      int_line = PCIIO_INTR_LINE_C;
      break;
   case 4:
      int_line = PCIIO_INTR_LINE_D;
      break;
   default:
      GM_WARN(("Unanticipated Interrupt_Pin value %d\n", int_pin));
      GM_ANNOUNCE_FNAME_RETURN(1, GM_INTERNAL_ERROR);
   }

   dev_desc = device_desc_default_get(the_is->arch.conn);

#if GM_IRIX_DISPLAY_DEV_DESC
   gm_irix_display_dev_desc(dev_desc, the_is,
                            "before call to pciio_intr_alloc");
#endif

   GM_PRINT (GM_IRIX_DEBUG_WAKE,
             ("Calling pciio_intr_alloc(%d, %p, %d, %d)\n",
            the_is->arch.conn, dev_desc, int_line, the_is->arch.nonpriv_vhdl));

#if 0
   cmn_err(CE_PANIC,"Panic so we can see our state here\n");
#endif

   intr_object =
      pciio_intr_alloc(the_is->arch.conn,          /* Connection vertex      */
                       dev_desc,                   /* device descriptor      */
                       int_line,                   /* which interrupt line   */
                       the_is->arch.nonpriv_vhdl); /* Device "owner" vertex  */
   if (intr_object == (pciio_intr_t)NULL)
   {
      GM_WARN(("Unable to allocate interrupt object\n"));
      GM_ANNOUNCE_FNAME_RETURN(1, GM_INTERNAL_ERROR);
   }

#if GM_IRIX_DISPLAY_INTERRUPT_HANDLE
   gm_irix_display_interrupt_handle(intr_object, "after pciio_intr_alloc()");
#endif
   
   the_is->arch.intr_obj = intr_object;

   return GM_SUCCESS;
}

#undef GM_FNAME

/*
 * Respond to user request to open "/dev/gm[p]N", the clone master device,
 * or "/hw/gm[p]NmM", a cloned device.  The arch.open_func for the
 * portstate will point to the appropriate open function.
 *
 * What happens is this:
 *
 *  1. User calls gm_open(); gm_open selects /dev/gmN or /dev/gmpN as
 *     appropriate, and calls open().  These are the clone master devices.
 *  2. IRIX kernel calls <pfx>open().
 *  3. <pfx>open() recognizes that this open is from a user (rather than
 *     from ioconfig) and calls gm_irix_open() (this function).
 *  4. The port_state for the clone master device has its arch.open_func
 *     set to point to gm_irix_clone_open(), so that function is called.
 *  5. gm_irix_clone_open() returns success to the kernel, which returns
 *     to gm_open(), which now holds an fd for the clone master device.
 *  6. gm_open() enters IRIX-specific code, which calls _gm_user_ioctl()
 *     with GM_SET_PORT_NUM and a special args structure.
 *  7. _gm_user_ioctl calls ioctl().
 *  8. IRIX kernel calls <pfx>ioctl().
 *  9. <pfx>ioctl(), still dealing with the clone_master device, calls
 *     gm_irix_clone_ioctl().
 * 10. gm_irix_clone_ioctl() does the following:
 *     a. Calls gm_minor_alloc() to reserve a gm_minor number <m>;
 *     b. If node "myrinet<m>" doesn't yet exist, creates it and its
 *        convenience vertex, /hw/gm<n>m<m>;
 *     c. Changes the permissions on "myrinet<m>" to 666;
 *     d. Saves the IRIX minor number assigned by IRIX in the port state;
 *     e. Calls gm_ioctl to do the platform-independent GM_SET_PORT_NUM
 *        processing;
 *     f. Returns to the kernel, having copied the new edge name into the
 *        special args structure.
 * 11. Control returns to the IRIX-specific code in gm_open().  Here, we
 *     close the clone master device and open the new device name that
 *     we received via the special args structure.  Voila - the user has
 *     a device known only to it, with its own dev_t and associated
 *     portstate.  Now all other user ioctl's and mmaps can proceed
 *     correctly.
 */
#define GM_FNAME gm_irix_open

GM_STATIC gm_status_t
GM_FNAME(gm_arch_minor_t the_minor)
{
   gm_port_state_t *p_state;    /*  */


   GM_ANNOUNCE_FNAME_ENTRY(1);
   GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
	    ("Calling gm_minor_get_port_state(%d)\n", the_minor));

   p_state = gm_minor_get_port_state(the_minor);

   gm_assert(p_state);
   gm_assert(p_state->arch.open_func);

   return (p_state->arch.open_func(p_state));
     
} /* gm_irix_open() */

#undef GM_FNAME

/*
 * Open one of our clone master devices.  See <pfx>open() and
 * gm_irix_open() for more details.
 */
#define GM_FNAME gm_irix_clone_open

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps)
{
   gm_instance_state_t *istate;


   GM_ANNOUNCE_FNAME_ENTRY(1);

   istate = the_ps->instance;
   gm_assert(istate);

   GM_ANNOUNCE_FNAME_RETURN(5, GM_SUCCESS);
     
} /* gm_irix_clone_open() */

#undef GM_FNAME



/*
 * Open a port device.  See <pfx>open() and gm_irix_open() for more details.
 */
#define GM_FNAME gm_irix_minor_open

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps)
{
   GM_PARAMETER_MAY_BE_UNUSED (the_ps);
  
   GM_ANNOUNCE_FNAME_ENTRY(1);

   GM_ANNOUNCE_FNAME_RETURN(5, GM_SUCCESS);

} /* gm_irix_minor_open() */

#undef GM_FNAME


/*
 * <pfx>ioctl()
 *
 * Description: If this is the clone ioctl, we have a lot of work to do here.
 *              If it's the user minor device ioctl, simply call gm_ioctl().
 */

#define GM_FNAME GM_PFXNM_IOCTL

int
GM_PFXNM_IOCTL(dev_t  dev,    /* Device code                    */
               int    cmd,    /* IOCTL command code             */
               void   *arg,   /* IOCTL command arg, if any      */
               int    mode,   /* Mode the file was opened with  */
               cred_t *crp,   /* Opaque authentication struct.  */
               int    *rvalp) /* Return value in case of error  */
{
   gm_status_t         gm_status;
   gm_instance_state_t *gm_is;       /* instance-state for this device       */
   gm_port_state_t     *gm_ps;
   gm_arch_minor_t     minor_num;


   GM_PARAMETER_MAY_BE_UNUSED (rvalp);
   GM_PARAMETER_MAY_BE_UNUSED (crp);
   
   GM_PRINT (GM_PRINT_LEVEL >= 5,
             (GM_FNAME_STR " entered with cmd 0x%x (%s) for dev %d\n",
              cmd, _gm_ioctl_cmd_name(cmd), dev));

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(dev);
#endif

   GM_IRIX_INSTANCE_GET_DV(dev, gm_is);  /* Ask kernel for our instance */
   if (!gm_is)
      /* OOPS! Should have been there!!??!! */
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);

   gm_mutex_enter((struct gm_mutex *)&gm_is->arch.device_lock);

   /* Find our gm minor number for this device */

   if (!gm_irix_minor_for_vhdl(dev_to_vhdl(dev), &minor_num))
   {
      /* OOPS! Should have been there!!??!! */
      gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);
   }

   gm_ps = gm_minor_get_port_state(minor_num);
   gm_assert(gm_ps);
   gm_assert(gm_ps->arch.ioct_func);
   gm_assert(gm_ps->instance == gm_is);

   gm_status = (gm_ps->arch.ioct_func(gm_ps, cmd, arg, mode));

   gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);

   GM_ANNOUNCE_FNAME_RETURN(5, gm_irix_localize_status(gm_status));

} /* <pfx>ioctl() */

#undef GM_FNAME


/*
 * gm_irix_clone_ioctl()
 *
 * Description: respond to the user's GM_SET_PORT_NUM ioctl by creating or
 *              reusing a new device (/hw/.../myrinetK) and a convenience
 *              vertex (/hw/gm[p]NmK) pointing to it.  K is the gm_minor
 *              number, and  N is the instance (AKA "unit" or "controller")
 *              number.  The convenience vertex name is passed back to the
 *              caller in the arg parameter to be used for the re-open.
 *
 *              Any other ioctl code received by this function is an error.
 */

#define GM_FNAME gm_irix_clone_ioctl

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps,  /* our port_state                 */
         int             cmd,      /* IOCTL command code             */
         void            *arg,     /* IOCTL command arg; REQUIRED    */
         int             mode)     /* Mode the file was opened with  */

{
   gm_status_t         ret_code;
   gm_status_t         ret_code_gm;
   gm_instance_state_t *istate;
   char                new_device_name[GM_IRIX_DEVNAME_SZ+1],
                       new_edge_name[GM_IRIX_EDGE_NAME_MAX_SZ+1],
                       new_open_name[GM_IRIX_OPEN_NAME_MAX_SZ+1];
   vertex_hdl_t        new_vhdl;
   gm_arch_minor_t     minor_num;
   gm_port_state_t     *new_ps;

   gm_irix_set_port_num_arg_t *ioctl_args;


   GM_ANNOUNCE_FNAME_ENTRY(1);

   gm_assert(the_ps);
   istate = the_ps->instance;
   gm_assert(istate);

   if (cmd != GM_SET_PORT_NUM)
      return GM_INVALID_COMMAND;
   if (!arg)
   {
      GM_WARN (("GM_SET_PORT_NUM IOCTL with no arg parameter\n"));
      return GM_INVALID_PARAMETER;
   }

   ioctl_args = (gm_irix_set_port_num_arg_t *)arg;

#if GM_IRIX_DISPLAY_SETPN_ARGS
   gm_irix_display_setpn_args(ioctl_args, "at entry to " GM_FNAME_STR);
#endif

#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
   gm_mutex_enter((struct gm_mutex *)gm_irix_open_and_close_sync);
#endif
   ret_code_gm = gm_minor_alloc(istate, &minor_num);
   if (ret_code_gm != GM_SUCCESS)
   {
      GM_WARN (("Could not allocate new gm_minor number\n"));
      ret_code = GM_INTERNAL_ERROR;
      goto abort_with_nothing;
   }

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.minor_alloc_cnt++;
#endif
   GM_PRINT (GM_IRIX_DEBUG_DEVNAME, ("Using minor number %d for unit %d\n",
                                     minor_num, istate->id));
   new_ps = gm_minor_get_port_state(minor_num);
   gm_assert(new_ps);
   gm_assert(new_ps->instance == istate);       /* set by gm_minor_alloc() */
  
   /* Record the privileges from the master port state and set arch.x fields */
   new_ps->privileged            = the_ps->privileged;
   new_ps->arch.is_master_pstate = GM_FALSE;
   new_ps->arch.open_func = &gm_irix_minor_open;
   new_ps->arch.ioct_func = &gm_irix_minor_ioctl;
   new_ps->arch.clos_func = &gm_irix_minor_close;
   new_ps->arch.mmap_func = &gm_irix_minor_map;
   new_ps->arch.umap_func = &gm_irix_minor_unmap;

   /* the following is a crude attempt to avoid overlarge result of the
    * sprintf().  Do something better.  //////
    */
   GM_PRINT (GM_PRINT_LEVEL >= 8, ("Checking that %d < %d\n",
		 (istate->id+1)*minor_num, GM_IRIX_MINOR_NAME_MAX_NUMBER));
   gm_always_assert((istate->id+1)*minor_num<GM_IRIX_MINOR_NAME_MAX_NUMBER);

   sprintf(new_device_name, GM_IRIX_DEV_NAME, minor_num);
   sprintf(new_open_name, GM_IRIX_OPEN_NAME, istate->id, minor_num);
   if (strlen(new_open_name) > (size_t)ioctl_args->name_size)
   {
      GM_WARN(("Cannot create device name %s; length %d > maximum %d\n",
               new_open_name, strlen(new_open_name), ioctl_args->name_size));
      ret_code = GM_INTERNAL_ERROR;
      goto abort_with_minor_number;
   }

   new_vhdl = gm_irix_find_old_vhdl(istate->id, minor_num);

   if (new_vhdl)		/* We can reuse a device vertex */
   {
      hwgraph_chmod(new_vhdl, 0666);  /* Anyone can open                    */
   }

   else				/* We need to create a new vertex */

   {
      if (gm_irix_device_add(istate->arch.conn, new_device_name,
                                  &new_vhdl, new_ps->privileged) != GM_SUCCESS)
      {
         ret_code = GM_FAILURE;
         goto abort_with_minor_number;
      }
      
      GM_IRIX_INSTANCE_SET_VH(new_vhdl, (void *)istate);

      sprintf(new_edge_name, GM_IRIX_EDGE_NAME, istate->id, minor_num);
      if (gm_irix_edge_add(new_vhdl, new_edge_name) != GM_SUCCESS)
      {
         ret_code = GM_INTERNAL_ERROR;
         goto abort_with_created_minor;
      }

      if (gm_irix_record_new_vhdl(istate->id, minor_num, new_vhdl) !=
                                                                    GM_SUCCESS)
      {
         ret_code = GM_INTERNAL_ERROR;
         goto abort_with_created_minor;
      }
   }

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(new_vhdl);
#endif

   new_ps->arch.vhdl       = new_vhdl;
   new_ps->arch.irix_minor = geteminor(new_vhdl);
   strcpy(new_ps->arch.edge_name, new_edge_name);

   /* Let gm_ioctl() do SET_PORT_NUM processing                     */
   gm_mutex_enter((struct gm_mutex *)&new_ps->sync);

   /* Call the portable IOCTL. */
   new_ps->arch.ddi_copyxx_context = mode;
   ret_code = gm_ioctl(new_ps,
                       GM_SET_PORT_NUM,
                       (void *)&(ioctl_args->port_id),
                       sizeof(ioctl_args->port_id),
                       (void *)&(ioctl_args->port_id),
                       sizeof(ioctl_args->port_id),
                       0                               );
   gm_mutex_exit((struct gm_mutex *)&new_ps->sync);

   if (ret_code != GM_SUCCESS)
      goto abort_with_created_minor;

   gm_arch_copyout(new_ps,           new_open_name,
                   ioctl_args->name, strlen(new_open_name)+1);

#if GM_IRIX_DISPLAY_SETPN_ARGS
   gm_irix_display_setpn_args(ioctl_args, "at exit from " GM_FNAME_STR);
#endif

   GM_ANNOUNCE_FNAME_RETURN(1, GM_SUCCESS);

abort_with_created_minor:
   gm_irix_delete_minor(new_ps); /* Also calls gm_minor_free() !!! */
   goto abort_with_nothing;	 /* ... so skip around redundant call */

abort_with_minor_number:
   gm_minor_free(minor_num);
#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.minor_free_cnt++;
#endif

abort_with_nothing:
#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
   gm_mutex_exit((struct gm_mutex *)gm_irix_open_and_close_sync);
#endif
   GM_ANNOUNCE_FNAME_RETURN(1, ret_code);

} /* gm_irix_clone_ioctl() */

#undef GM_FNAME


/*
 * gm_irix_minor_ioctl()
 *
 * Description:  Simply call gm_ioctl.
 */

#define GM_FNAME gm_irix_minor_ioctl

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps,  /* our port_state                 */
         int             cmd,      /* IOCTL command code             */
         void            *arg,     /* IOCTL command arg, if any      */
         int             mode)     /* Mode the file was opened with  */
{
   gm_status_t     ret_code;
   
   GM_ANNOUNCE_FNAME_ENTRY(5);

   if (cmd == GM_SET_PORT_NUM)
      GM_ANNOUNCE_FNAME_RETURN(1, GM_INVALID_COMMAND); /* Not legal here */

   /* Hand it off to gm_ioctl()                */
   gm_mutex_enter((struct gm_mutex *)&the_ps->sync);

   /* Call the portable IOCTL. */
   the_ps->arch.ddi_copyxx_context = mode;
   ret_code = gm_ioctl(the_ps,
                       (unsigned int)cmd,
                       (void *)arg,
                       INT_MAX,
                       (void *)arg,
                       INT_MAX,
                       0           );
   gm_mutex_exit((struct gm_mutex *)&the_ps->sync);

   GM_ANNOUNCE_FNAME_RETURN(5, ret_code);

} /* gm_irix_minor_ioctl() */

#undef GM_FNAME


/*
 * <pfx>map()
 *
 * Description: A process has called mmap() for one of our devices.
 *              We call the clone or minor map function as set in the
 *              portstate.
 */

#define GM_FNAME GM_PFXNM_MAP

int
GM_PFXNM_MAP(dev_t dev, vhandl_t *vt, off_t off, int len, int prot)
{
   gm_status_t         gm_status;
   gm_instance_state_t *gm_is;       /* instance-state for this device       */
   gm_port_state_t     *gm_ps;
   gm_arch_minor_t     minor_num;


   GM_ANNOUNCE_FNAME_ENTRY(5);

   GM_IRIX_VHANDL_DISP(vt);

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(dev);
#endif

   GM_IRIX_INSTANCE_GET_DV(dev, gm_is);  /* Ask kernel for our instance */
   if (!gm_is)
      /* OOPS! Should have been there!!??!! */
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);

   gm_mutex_enter((struct gm_mutex *)&gm_is->arch.device_lock);

   /* Find our gm minor number for this device */

   if (!gm_irix_minor_for_vhdl(dev_to_vhdl(dev), &minor_num))
   {
      /* OOPS! Should have been there!!??!! */
      gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);
   }

   gm_ps = gm_minor_get_port_state(minor_num);
   gm_assert(gm_ps);
   gm_assert(gm_ps->arch.mmap_func);
   gm_assert(gm_ps->instance == gm_is);

   gm_status = (gm_ps->arch.mmap_func(gm_ps, vt, off, len, prot));

   gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);
   GM_ANNOUNCE_FNAME_RETURN(5, gm_irix_localize_status(gm_status));
}

#undef GM_FNAME


/*
 * gm_irix_clone_map()
 *
 * Description: A process has called mmap() for our clone device.
 *              This is an illegal call.
 */
#define GM_FNAME gm_irix_clone_map

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps, vhandl_t *vt, off_t off, int len, int prot)
{
   GM_PARAMETER_MAY_BE_UNUSED (the_ps);
   GM_PARAMETER_MAY_BE_UNUSED (vt);
   GM_PARAMETER_MAY_BE_UNUSED (off);
   GM_PARAMETER_MAY_BE_UNUSED (len);
   GM_PARAMETER_MAY_BE_UNUSED (prot);
  
   GM_ANNOUNCE_FNAME_RETURN(1, GM_INVALID_COMMAND); /* Not legal here */
}

#undef GM_FNAME


/*
 * gm_irix_minor_map()
 *
 * Description: A process has called mmap() for its minor device.

   //// adapt the following description from Solaris to IRIX ////

   Map interface memory into the process.  This can be used to map the
   LANai control register, special registers, SRAM, copy block, etc. into
   the user space.  The offsets and sizes of these various regions can
   be obtained using the GM_GET_MAPPING_SPECS ioctl, which returns
   a description of the mappings in a "gm_mapping_specs" structure.

   The pages of the copy block must be mapped in order and without
   skipping any offsets.

   gm_segmap ensures that all offsets passed to gm_mmap are
   legitimate, so there is no need to range check offsets here.
   However, checks must be performed here for "promiscuous" mappings
   of the device registers and SRAM areas other than the "send queue"
   and "activation" pages.
 */

#define GM_FNAME gm_irix_minor_map

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps, vhandl_t *vt, off_t off, int len, int prot)
{
   gm_status_t   gm_status;
   int           func_ret;
   void          *resource_addr;
   size_t        user_len;
   unsigned int  gm_permissions;


   GM_ANNOUNCE_FNAME_ENTRY(5);

   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("Mapping offset 0x%x for minor %d, portstate %p\n",
              off, the_ps->minor, the_ps));
   GM_IRIX_VHANDL_DISP(vt);

   user_len  = v_getlen(vt);

   GM_PRINT (GM_PRINT_LEVEL >= 6, ("VM (user) address = %p, length = %d\n",
		v_getaddr(vt), user_len));
   GM_PRINT (GM_PRINT_LEVEL >= 8,
             ("Verifying %d (map length) >= %d (user memory length)\n",
		len, user_len));
   if (user_len < len)
   {
      GM_WARN(("User memory length %d too small for map length %d\n",
                  user_len, len));
      gm_status = GM_INVALID_COMMAND;
      goto exit_with_gm_status;
   }

   gm_permissions = 0;
   if (prot & PROT_READ)
      gm_permissions |= GM_MAP_READ;
   if (prot & PROT_WRITE)
      gm_permissions |= GM_MAP_WRITE;
   if (prot & PROT_EXEC)
   {
      GM_WARN(("User program attempted EXEC access in mmap call"));
      gm_status = GM_INVALID_PARAMETER;
      goto exit_with_gm_status;
   }

   GM_PRINT (GM_PRINT_LEVEL >= 6, ("Calling gm_prepare_to_mmap()\n"));
   gm_status = gm_prepare_to_mmap(the_ps, off, len, gm_permissions);
   if (gm_status != GM_SUCCESS)
   {
      GM_WARN(("gm_prepare_to_mmap returned error %d\n", gm_status));
      goto exit_with_gm_status;
   }

   GM_PRINT (GM_PRINT_LEVEL >= 6, ("Calling gm_mmap()\n"));
   gm_status = gm_mmap(the_ps, (gm_u32_t)off, &resource_addr);
   if (gm_status != GM_SUCCESS)
   {
      GM_WARN(("gm_mmap returned error %d\n", gm_status));
      goto exit_with_gm_status;
   }
   gm_assert(resource_addr);

   GM_PRINT (GM_PRINT_LEVEL >= 8,
             ("Calling v_mapphys(%p, %p, 0x%x)\n", vt, resource_addr, len));
   func_ret = v_mapphys(vt, resource_addr, len);
   if (func_ret)
   {
      if (func_ret == ENOMEM)	/* ENOMEM is the only error return expected */
      {
         GM_WARN((GM_FNAME_STR ": v_mapphys returned error ENOMEM\n"));
         gm_status = GM_OUT_OF_MEMORY;
      }
      else
      {
         GM_WARN((GM_FNAME_STR ": v_mapphys returned (undescribed) error 0x%x\n",
                  func_ret));
         gm_status = GM_INTERNAL_ERROR;
      }
      goto exit_with_gm_status;
   }

   gm_always_assert(gm_status == GM_SUCCESS);

exit_with_gm_status:
   GM_ANNOUNCE_FNAME_RETURN(5, gm_status);

} /* gm_irix_minor_map() */

#undef GM_FNAME


/*
 * <pfx>unmap()
 *
 * Description:  All user processes have released their maps for a specific
 *               mapped region.
 *
 * NOTE that IRIX does *not* call this on every user unmap() call, but only
 * when the fully-unmapped state been attained.  The Programmer's Guide
 * also says:
 *
 *      On entry, the kernel has completed unmapping the object from the user
 *      process address space. This entry point does not need to do anything to
 *      affect the user address space; it only needs to release any resources
 *      that were allocated to support the mapping...
 *
 *      If the driver allocated no resources to support a mapping, no action is
 *      needed here; the entry point can consist of a "RETURN(0)" statement.
 *
 *      When the driver does allocate memory or a PIO map to support a mapping,
 *      and supports multiple mappings, the driver needs to identify the
 *      resource associated with this particular mapping in order to release
 *      it. The v_gethandle() function returns a unique number based on the
 *      vt argument; this can be used to identify resources.
 *
 *  /////  analysis and implementation (if any) along the above lines has
 *  /////  *NOT* yet been done.  Note that Rod Freier has offered to share
 *  /////  example code for this.
 */

#define GM_FNAME GM_PFXNM_UNMAP

int
GM_PFXNM_UNMAP(dev_t dev, vhandl_t *vt)
{
   gm_status_t         gm_status;
   gm_instance_state_t *gm_is;       /* instance-state for this device       */
   gm_port_state_t     *gm_ps;
   gm_arch_minor_t     minor_num;

   GM_PARAMETER_MAY_BE_UNUSED (vt);
   
   GM_ANNOUNCE_FNAME_ENTRY(5);

   GM_IRIX_VHANDL_DISP(vt);

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(dev);
#endif

   GM_IRIX_INSTANCE_GET_DV(dev, gm_is);  /* Ask kernel for our instance */
   if (!gm_is)
      /* OOPS! Should have been there!!??!! */
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);

   gm_mutex_enter((struct gm_mutex *)&gm_is->arch.device_lock);

   /* Find our gm minor number for this device */

   if (!gm_irix_minor_for_vhdl(dev_to_vhdl(dev), &minor_num))
   {
      /* OOPS! Should have been there!!??!! */
      gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);
   }

   gm_ps = gm_minor_get_port_state(minor_num);
   gm_assert(gm_ps);
   gm_assert(gm_ps->arch.umap_func);
   gm_assert(gm_ps->instance == gm_is);

   gm_status = (gm_ps->arch.umap_func(gm_ps));

   gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);

   GM_ANNOUNCE_FNAME_RETURN(5, gm_irix_localize_status(gm_status));

} /* <pfx>unmap() */

#undef GM_FNAME


/*
 * gm_irix_clone_unmap()
 *
 * Description: mmap() has been called for our clone device.
 *              This is an illegal call.
 */

#define GM_FNAME gm_irix_clone_unmap

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps)
{
   GM_PARAMETER_MAY_BE_UNUSED (the_ps);
  
   GM_ANNOUNCE_FNAME_RETURN(1, GM_INVALID_COMMAND); /* Not legal here */

} /* gm_irix_clone_unmap() */

#undef GM_FNAME



/*
 * gm_irix_minor_unmap()
 */

#define GM_FNAME gm_irix_minor_unmap

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps)
{
  GM_PARAMETER_MAY_BE_UNUSED (the_ps);
  
  GM_ANNOUNCE_FNAME_ENTRY(5);
  GM_ANNOUNCE_FNAME_RETURN(5, GM_SUCCESS);

} /* gm_irix_minor_unmap() */

#undef GM_FNAME


/*
 * <pfx>close()
 *
 * Description:  ioconfig or a user process has called close() for a GM device
 *               and this was the *last* close; i.e., the device is now held
 *               by no process whatsoever.
 */

#define GM_FNAME GM_PFXNM_CLOSE

int
GM_PFXNM_CLOSE(dev_t dev, int flags, int otyp, cred_t *credp)
{
   gm_status_t         gm_status;
   gm_instance_state_t *gm_is;
   gm_port_state_t     *gm_ps;
   gm_arch_minor_t     minor_num;


   GM_PARAMETER_MAY_BE_UNUSED (flags);                 /* If not GM_DEBUG */
   GM_PARAMETER_MAY_BE_UNUSED (otyp);                  /* If not GM_DEBUG */
   GM_PARAMETER_MAY_BE_UNUSED (credp);                 /* If not GM_DEBUG */
   GM_PRINT (GM_PRINT_LEVEL >= 1,
             (GM_FNAME_STR " entered with %d, 0x%x, 0x%x, %p\n",
              dev, flags, otyp, credp));

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(dev);
#endif

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("at entry to " GM_FNAME_STR);
#endif

   /* Find our instance_state structure for this device */

   GM_IRIX_INSTANCE_GET_DV(dev, gm_is);  /* Ask kernel for our instance */
   gm_assert(gm_is);

   GM_PRINT (GM_PRINT_LEVEL >= 6, ("Got instance state %p for dev %d (0x%x)\n",
	                           gm_is, dev, dev));

   gm_mutex_enter((struct gm_mutex *)&gm_is->arch.device_lock);

   if (gm_is->arch.controller < 0)
   {
      GM_PRINT (GM_PRINT_LEVEL >= 5, (" - detected call from ioconfig\n"));
      gm_always_assert(gm_is->arch.controller_susp >= 0);
      gm_is->arch.controller = gm_is->arch.controller_susp;
      gm_status = GM_SUCCESS;
      goto exit_with_gm_status;
   }

/* NOT the call from ioconfig, so do close processing for our device */

   if (!gm_irix_minor_for_vhdl(dev_to_vhdl(dev), &minor_num))
   {
      /* OOPS! Should have been there!!??!! */
      gm_status = GM_INTERNAL_ERROR;
      goto exit_with_gm_status;
   }

#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
   gm_mutex_enter((struct gm_mutex *)gm_irix_open_and_close_sync);
#endif

   gm_ps = gm_minor_get_port_state(minor_num);
   gm_assert(gm_ps);
   gm_assert(gm_ps->arch.clos_func);

   gm_status = (gm_ps->arch.clos_func(gm_ps));

#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
   gm_mutex_exit((struct gm_mutex *)gm_irix_open_and_close_sync);
#endif

exit_with_gm_status:

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("at exit from " GM_FNAME_STR);
#endif

   gm_mutex_exit((struct gm_mutex *)&gm_is->arch.device_lock);

   GM_ANNOUNCE_FNAME_RETURN(1, gm_irix_localize_status(gm_status));

} /* <pfx>close() */

#undef GM_FNAME


/*
 * gm_irix_clone_close()
 *
 * Description: The clone master is to be closed.  Nothing to do here.
 */

#define GM_FNAME gm_irix_clone_close

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps)
{
   GM_PARAMETER_MAY_BE_UNUSED (the_ps);
  
   GM_ANNOUNCE_FNAME_RETURN(1, GM_SUCCESS);

} /* gm_irix_clone_close() */

#undef GM_FNAME



/*
 * gm_irix_minor_close()
 *
 * Description: The user's minor device is to be closed.
 */

#define GM_FNAME gm_irix_minor_close

GM_STATIC gm_status_t
GM_FNAME(gm_port_state_t *the_ps)
{
   GM_ANNOUNCE_FNAME_ENTRY(1);

   gm_irix_delete_minor(the_ps);
   
   GM_ANNOUNCE_FNAME_RETURN(1, GM_SUCCESS);

} /* gm_irix_minor_close() */

#undef GM_FNAME

/*
 * <pfx>unload()
 *
 * Description:  Called by the kernel when dynamically unloading the
 * driver, to give us a last chance to clean up.  The things that we
 * must do include:
 *
 *  * Make sure there are no active callback functions, interrupt   
 *    handlers, or other driver entry points active.
 *
 *  * Make sure there are no pending kernel timeout functions.
 *
 *  * Make sure there are no pointers to driver static data, etc., in
 *    IRIX's device vertex data structures.  (The pointers will persist,
 *    but they will be invalid upon reload of the driver).
 *
 *  * If, subsequent to the above steps, there is still any reason
 *    that the driver should not be unloaded, we can tell the kernel
 *    not to unload us by returning EBUSY instead of zero.
 *
 * Note that the sample pci driver supplied by SGI suggests using a
 *      "driver-busy" flag (actually, a counter of number of devices open)
 *      in the driver module, and testing this flag before unloading:
 * 
 *         if (psamp_inuse)
 *            return EBUSY;
 *
 * There appears to be no simple way of testing whether this GM driver is
 * busy, analogous to the above.
 *
 * IMPORTANT NOTE: according to the man page for mload(4), "An unload
 *                 routine should be treated as an interrupt routine and
 *                 should not call any routines that would cause it to
 *                 sleep, such as: biowait(), sleep(), psema() or delay()."
 *                 Note that we call the platform-independent function
 *                 gm_finalize() here, over the contents of which we have
 *                 no direct control.  It appears clear that gm_finalize()
 *                 currently adheres to the rule given above.
 */

#define GM_FNAME GM_PFXNM_UNLOAD

int
GM_PFXNM_UNLOAD(void)
{
   GM_ANNOUNCE_FNAME_ENTRY(1);

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("at entry to " GM_FNAME_STR);
#endif

   pciio_iterate(GM_PREFIX_STRING, gm_irix_unloadme);

#if GM_CAN_REGISTER_MEMORY
   gm_irix_teardown_mem_registration();
#endif

   if (gm_irix_device_hash)
   {
      gm_destroy_hash(gm_irix_device_hash);
      gm_irix_device_hash = NULL;
   }
   if (gm_irix_minor_hash)
   {
      gm_destroy_hash(gm_irix_minor_hash);
      gm_irix_minor_hash = NULL;
   }

   if (gm_irix_minor_hash_sync)
      gm_destroy_mutex((struct gm_mutex *)gm_irix_minor_hash_sync);
   if (gm_irix_device_hash_sync)
      gm_destroy_mutex((struct gm_mutex *)gm_irix_device_hash_sync);
#if GM_IRIX_USE_OPEN_AND_CLOSE_SYNC
   if (gm_irix_open_and_close_sync)
      gm_destroy_mutex((struct gm_mutex *)gm_irix_open_and_close_sync);
#endif

   gm_finalize();            /* finalize GM -- ////// maybe the wrong place */

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("at exit from " GM_FNAME_STR);
#endif

   if (gm_irix_state != GM_SUCCESS)
      pciio_driver_unregister(GM_PREFIX_STRING);

/* NOTE: We don't use GM_SUCCESS as our return code, because we are
         communicating with the IRIX kernel, not with GM or the end user. */

   GM_ANNOUNCE_FNAME_RETURN(1, 0);
}

#undef GM_FNAME

/*
 * <pfx>detach()
 *
 * Description:  IRIX has determined that this device must be detached;
 * it calls this function to undo as much as possible of the actions
 * previously performed by <pfx>attach().  We unroll these actions in
 * reverse sequence.
 *
 * NOTE:  The IRIX 6.4 Myrinet (non-GM) driver myri_detach() entry point
 *        is a no-op, simply returning 0.  This is because that driver
 *        is not loadable.
 */

#define GM_FNAME GM_PFXNM_DETACH

int
GM_PFXNM_DETACH(vertex_hdl_t conn_vhdl)
{
   gm_instance_state_t *gm_is;
   graph_error_t       hwgraph_ret;  /* hwgraph function status */
   vertex_hdl_t        our_vhdl; 
   char                edge_name[GM_IRIX_EDGE_NAME_MAX_SZ+1];
   vertex_hdl_t        rem_vhdl;


   GM_ANNOUNCE_FNAME_ENTRY(1);

   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("- detaching for connection handle %d\n", conn_vhdl)); 

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("at entry to " GM_FNAME_STR);
#endif

/* First, we disconnect the privileged vertex */
   hwgraph_ret = hwgraph_traverse(conn_vhdl, GM_IRIX_PRIV_HWNAME, &our_vhdl);
   if (hwgraph_ret != GRAPH_SUCCESS)
   {
      GM_WARN(("hwgraph_traverse(0x%x, %s, &vhdl) returned %d\n",
               conn_vhdl, GM_IRIX_PRIV_HWNAME, hwgraph_ret));
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);
   }

   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("hwgraph_traverse() found privileged vertex w/handle %d\n",
              our_vhdl));

   GM_IRIX_INSTANCE_GET_VH(our_vhdl, gm_is);

   if (!gm_is)
      /* OOPS! Should have been there!!??!! */
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);

#if GM_DEBUG_INSTANCE_STATE
   gm_irix_display_instance_values(gm_is, "from priv vertex in " GM_FNAME_STR);
#endif

/* 
 * The gm interfaces have probably been torn down already, but just in case
 * they haven't, do it now.
 */
   if (gm_is->arch.gm_instance_init)
   {
      gm_irix_unload_gm(gm_is);
      if (gm_is->arch.piomap)
      {
         pciio_piomap_free(gm_is->arch.piomap);
         gm_is->arch.piomap = (pciio_piomap_t)NULL;
      }
   }

   gm_irix_intr_free(gm_is);

/*
 * Remove the vertex's convenience edge, then the vertex itself.
 */
   if (strlen(gm_is->arch.p_clone_edge_name))
   {
      rem_vhdl = our_vhdl;
      strcpy(edge_name, gm_is->arch.p_clone_edge_name);
      gm_irix_edge_remove(&rem_vhdl, edge_name);
   }

   GM_IRIX_INSTANCE_UNSET_VH(our_vhdl);

   gm_irix_device_destroy(our_vhdl);

/* OK, now we disconnect the non-privileged vertex */
   hwgraph_ret = hwgraph_traverse(conn_vhdl, GM_IRIX_NONP_HWNAME, &our_vhdl);
   if (hwgraph_ret != GRAPH_SUCCESS)
   {
      GM_WARN(("hwgraph_traverse(0x%x, %s, &vhdl) returned %d\n",
               conn_vhdl, GM_IRIX_NONP_HWNAME, hwgraph_ret));
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);
   }

   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("hwgraph_traverse() found nonprivileged vertex w/handle %d\n",
              our_vhdl));

   GM_IRIX_INSTANCE_GET_VH(our_vhdl, gm_is);

   if (!gm_is)
      /* OOPS! Should have been there!!??!! */
      GM_ANNOUNCE_FNAME_RETURN(1, ENXIO);

#if GM_DEBUG_INSTANCE_STATE
   gm_irix_display_instance_values(gm_is,
                                   "from non-priv vertex in " GM_FNAME_STR);
#endif

   if (strlen(gm_is->arch.clone_edge_name))
   {
      rem_vhdl = our_vhdl;
      strcpy(edge_name, gm_is->arch.clone_edge_name);
      gm_irix_edge_remove(&rem_vhdl, edge_name);
   }

   GM_IRIX_INSTANCE_UNSET_VH(our_vhdl);

   gm_irix_device_destroy(our_vhdl);

/* Now we are ready to finish tearing down the instance */

   gm_irix_kmemfree(gm_is, sizeof(gm_instance_state_t));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_memleak_report("at exit from " GM_FNAME_STR);
#endif

/* NOTE: We don't use GM_SUCCESS as our return code, because we are
         communicating with the IRIX kernel, not with GM or the end user. */
   GM_ANNOUNCE_FNAME_RETURN(1, 0);

} /* <pfx>detach() */

#undef GM_FNAME


/************************************************************************
 * Dynamic loading entry points.
 ************************************************************************/

/*
 * <pfx>unreg()
 *
 * Description:  Called by the kernel prior to calling <pfx>unload(),
 * to allow the driver to "unregister" itself.
 */

#define GM_FNAME GM_PFXNM_UNREG

int
GM_PFXNM_UNREG(dev_t dev, int flags, int otyp, cred_t * credp)
{
   GM_PARAMETER_MAY_BE_UNUSED (dev);
   GM_PARAMETER_MAY_BE_UNUSED (flags);
   GM_PARAMETER_MAY_BE_UNUSED (otyp);
   GM_PARAMETER_MAY_BE_UNUSED (credp);
  
   GM_ANNOUNCE_FNAME_ENTRY(1);
   pciio_driver_unregister(GM_PREFIX_STRING);

/* NOTE: We don't use GM_SUCCESS as our return code, because we are
         communicating with the IRIX kernel, not with GM or the end user. */

   GM_ANNOUNCE_FNAME_RETURN(1, 0);
}

#undef GM_FNAME

/************************************************************************
 * Utility functions for the IRIX-required functions above
 ************************************************************************/

/*
 * gm_irix_unloadme()
 *
 * Description:  Called by <pfx>unload(), via pciio_iterate(), to "disconnect"
 *               each attached device before the driver becomes unloaded.
 *
 * NOTE:         See commentary at <pfx>unload for details on what must be
 *               done here.
 *
 *               /////  Review all those items to make sure we are kosher /////
 */
#define GM_FNAME gm_irix_unloadme

GM_STATIC void
GM_FNAME(vertex_hdl_t conn_point)
{
   vertex_hdl_t         vhdl;         /* handle for end vertex */
   gm_instance_state_t  *istate;      /* instance-specific data */
   graph_error_t        hwgraph_ret;  /* hwgraph function status */


   GM_ANNOUNCE_FNAME_ENTRY(1);

   GM_PRINT (GM_PRINT_LEVEL >= 1,
             ("- unloading for vertex_hdl_t %d\n", conn_point)); 

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(vhdl_to_dev(conn_point));
#endif

/* It should be safe to assume that the non-priv vertex contains all the info
 * that we need, because the PRIV_HWNAME is linked to the same instance.
 */
   hwgraph_ret = hwgraph_traverse(conn_point, GM_IRIX_NONP_HWNAME, &vhdl);
   if (hwgraph_ret != GRAPH_SUCCESS)
   {
      GM_WARN(("hwgraph_traverse(0x%x, %s, &vhdl) returned %d\n",
               conn_point, GM_IRIX_NONP_HWNAME, hwgraph_ret));
      GM_ANNOUNCE_FNAME_EXIT(1);
      return;
   }

/* found a vertex                              */
   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("hwgraph_traverse() found vertex with handle %d\n", vhdl));

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(vhdl);
#endif
   
   GM_IRIX_INSTANCE_GET_VH(vhdl, istate);
   gm_assert(istate);

   if (istate->arch.gm_instance_init)
      gm_irix_unload_gm(istate);
   
   /*
    * Disconnect our error handler
    */

   if (istate->arch.ehandlr_registered)
   {
      GM_PRINT (GM_PRINT_LEVEL >= 6, ("un-registering error handler\n"));
      pciio_error_register(conn_point, 0, 0);
      istate->arch.ehandlr_registered = 0;
   }

   GM_ANNOUNCE_FNAME_EXIT(1);
}

#undef GM_FNAME

/*
 * gm_irix_unload_gm()
 */
#define GM_FNAME gm_irix_unload_gm

GM_STATIC void
GM_FNAME(gm_instance_state_t *the_is)
{
   gm_irix_disable_interrupts(the_is);
   gm_irix_delete_clone_master_pstates(the_is);
   gm_instance_finalize(the_is);
   the_is->arch.gm_instance_init = 0;
   the_is->arch.controller = the_is->arch.controller_susp = -1;
}

#undef GM_FNAME

/*
 * gm_irix_reloadme()
 *
 * Description:  Called by <pfx>init, via pciio_iterate(), to "reconnect"
 *               each connection point when the driver has been reloaded.
 */
#define GM_FNAME gm_irix_reloadme

GM_STATIC void
GM_FNAME(vertex_hdl_t conn_point)
{
   vertex_hdl_t         vhdl;    /* handle for end vertex */
   gm_instance_state_t  *istate; /* instance-specific data */
   graph_error_t        hwgraph_ret;  /* hwgraph function status */


   GM_ANNOUNCE_FNAME_ENTRY(1);

   GM_PRINT (GM_PRINT_LEVEL >= 1,
             ("- reloading for vertex_hdl_t %d\n", conn_point)); 

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(vhdl_to_dev(conn_point));
#endif

/* It should be safe to assume that this vertex contains all the info
 * that we need, because the PRIV_HWNAME is linked to the same instance.
 */
   hwgraph_ret = hwgraph_traverse(conn_point, GM_IRIX_NONP_HWNAME, &vhdl);
   if (hwgraph_ret != GRAPH_SUCCESS)
   {
      GM_WARN(("hwgraph_traverse(0x%x, %s, &vhdl) returned %d\n",
               conn_point, GM_IRIX_NONP_HWNAME, hwgraph_ret));
      GM_ANNOUNCE_FNAME_EXIT(1);
      return;
   }

/* found a vertex                              */
   GM_PRINT (GM_PRINT_LEVEL >= 1,
             ("hwgraph_traverse() found vertex with handle %d\n", vhdl));

#if GM_IRIX_DISPLAY_DEVT
   gm_irix_parse_devt(vhdl);
#endif
   
   GM_IRIX_INSTANCE_GET_VH(vhdl, istate);
   gm_assert(istate);

#if 0
   gm_irix_reload_gm(istate);   /* I think this function is superfluous */
#endif

   /*
    * Reinitialize our device sync
    */
   gm_irix_init_dev_lock(istate);

   /*
    * Reconnect our error handler
    */
   GM_PRINT (GM_PRINT_LEVEL >= 6, ("re-registering error handler\n"));
   pciio_error_register(conn_point,
                        &gm_irix_ehandlr,
                        istate);

   istate->arch.ehandlr_registered = 1;

   GM_ANNOUNCE_FNAME_EXIT(1);
}

#undef GM_FNAME


#if GM_IRIX_ENABLE_RELOAD
/*
 * gm_irix_reload_gm()
 */
#define GM_FNAME gm_irix_reload_gm

GM_STATIC void
GM_FNAME(gm_instance_state_t *the_is)
{
   /*
    * Reinitialize the instance
    */

   if (gm_instance_init(the_is, the_is->id, GM_MYRINET_BUS_PCI) != GM_SUCCESS)
   {
      GM_WARN(("[%d]: Could not re-initialize unit\n", the_is->id));
      return;
   }
   the_is->arch.gm_instance_init = 1;

   /*
    * Re-establish our permanent minor portstates
    */
   if (gm_irix_setup_clone_master_pstates(the_is) != GM_SUCCESS)
   {
      GM_WARN(("Unable to re-establish permanent portstates\n"));
      return;
   }

   if (the_is->arch.intr_obj)
   {
      GM_PRINT (GM_PRINT_LEVEL >= 6, ("re-connecting interrupt handler\n"));
      pciio_intr_connect(the_is->arch.intr_obj,    /* Connection vertex     */
                         gm_irix_intr,             /* The handler function  */
                         the_is,                   /* Args for handler      */
                         (void *)0              ); /* Thread pointer n/a    */
      the_is->arch.ihandlr = gm_irix_intr;
   }

#if GM_DEBUG
   gm_irix_test_basic_functions(the_is);
#endif

}
#endif /* GM_IRIX_ENABLE_RELOAD */

#undef GM_FNAME

/*
 * Setup the two "permanent" minor "clones", which will remain active
 * until the driver is unloaded and/or detached.
 */

#define GM_FNAME gm_irix_setup_clone_master_pstates

GM_STATIC gm_status_t
GM_FNAME(gm_instance_state_t *the_is)
{
   gm_port_state_t     *ps_for_nonpriv_clone,
                       *ps_for_priv_clone;
   gm_status_t         ret_code;


/* NOTE: When the permanent clones have been allocated, one of
 *       {clone_minor,privileged_clone_minor} can legally be 0, but not both.
 *       Here, they must BOTH be 0.
 */
   gm_assert(!(the_is->clone_minor));
   gm_assert(!(the_is->privileged_clone_minor));


   GM_ANNOUNCE_FNAME_ENTRY(4);

/*
 * Create clone-master port state for non-privileged cloning, corresponding
 * to device name /dev/gm<n> (/hw/gm<n>).
 */
   if (gm_irix_make_master_pstate(the_is,
                                  GM_FALSE,	/* Is not privileged */
                                  &ps_for_nonpriv_clone)
   != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_nothing;
   }

   ps_for_nonpriv_clone->arch.vhdl = the_is->arch.nonpriv_vhdl;
   the_is->clone_minor = ps_for_nonpriv_clone->minor;
   if (gm_irix_record_new_vhdl(the_is->id,
                               the_is->clone_minor,
                               the_is->arch.nonpriv_vhdl) != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR;
      goto abort_with_non_priv_minor;
   }

/*
 * Create clone-master port state for privileged cloning, corresponding
 * to device name /dev/gmp<n> (/hw/gmp<n>).
 */

   if (gm_irix_make_master_pstate(the_is,
                                  GM_TRUE,	/* Is privileged  */
                                  &ps_for_priv_clone)
   != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR; 
      goto abort_with_non_priv_minor;
   }

   ps_for_priv_clone->arch.vhdl = the_is->arch.priv_vhdl;
   the_is->privileged_clone_minor = ps_for_priv_clone->minor;
   if (gm_irix_record_new_vhdl(the_is->id,
                               the_is->privileged_clone_minor,
                               the_is->arch.priv_vhdl         ) != GM_SUCCESS)
   {
      ret_code = GM_INTERNAL_ERROR;
      goto abort_with_priv_minor;
   }

   GM_ANNOUNCE_FNAME_RETURN(4, GM_SUCCESS);
   
abort_with_priv_minor:
   gm_irix_delete_minor(ps_for_priv_clone);
abort_with_non_priv_minor:
   gm_irix_delete_minor(ps_for_nonpriv_clone);
abort_with_nothing:
   GM_ANNOUNCE_FNAME_RETURN(1, ret_code);
}

#undef GM_FNAME

/*
 * Create minor: 
 */

#define GM_FNAME gm_irix_make_master_pstate

GM_STATIC gm_status_t
GM_FNAME(gm_instance_state_t *the_is,
         char                privileged,
         gm_port_state_t     **the_ps      )
{
   gm_port_state_t *pstate;
   gm_status_t     gm_status;
   gm_arch_minor_t minor_num;


   GM_ANNOUNCE_FNAME_ENTRY(5);

   GM_PRINT (GM_PRINT_LEVEL >= 6, (" - the_is=%p,\n"
                "       privileged=%d, the_ps=%p\n",
	        the_is, privileged, the_ps));

   gm_assert(the_ps);

   gm_status = gm_minor_alloc(the_is, &minor_num);
   if (gm_status)
      goto abort_with_nothing;

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.minor_alloc_cnt++;
#endif
   pstate = gm_minor_get_port_state(minor_num);
   gm_assert(pstate);
   gm_assert(pstate->instance == the_is);
   gm_assert(pstate->minor == minor_num);

   GM_PRINT (GM_IRIX_DEBUG_DEVNAME,
             ("Allocated %s minor port_state 0x%p for unit %d, minor %d\n",
              ((privileged) ? "privileged" : "non-privileged"),
              pstate, the_is->id, minor_num));

   pstate->privileged            = privileged;
   pstate->arch.is_master_pstate = GM_TRUE;

   pstate->arch.open_func = &gm_irix_clone_open;
   pstate->arch.ioct_func = &gm_irix_clone_ioctl;
   pstate->arch.clos_func = &gm_irix_clone_close;
   pstate->arch.mmap_func = &gm_irix_clone_map;
   pstate->arch.umap_func = &gm_irix_clone_unmap;

   *the_ps = pstate; 

   GM_ANNOUNCE_FNAME_RETURN(5, GM_SUCCESS);

abort_with_nothing:
   GM_ANNOUNCE_FNAME_RETURN(1, GM_FAILURE);
 
} /* gm_irix_make_master_pstate */

#undef GM_FNAME


/*
 * Delete the two "permanent" minor "clones".
 */

#define GM_FNAME gm_irix_delete_clone_master_pstates

GM_STATIC void
GM_FNAME(gm_instance_state_t *the_is)
{
   gm_port_state_t *p_state;

   if (GM_IRIX_DEBUG_DEVNAME)
      GM_ANNOUNCE_FNAME_ENTRY(0);

   p_state = gm_minor_get_port_state(the_is->clone_minor);
   if (p_state)
   {
      gm_irix_delete_minor(p_state);
      the_is->clone_minor = 0;
   }

   p_state = gm_minor_get_port_state(the_is->privileged_clone_minor);
   if (p_state)
   {
      gm_irix_delete_minor(p_state);
      the_is->privileged_clone_minor = 0;
   }

   GM_ANNOUNCE_FNAME_EXIT(4);
}

#undef GM_FNAME

/*
 * Delete minor: close its port state, de-access its hwgraph vertex, and free
 *               its gm_minor number
 */

#define GM_FNAME gm_irix_delete_minor

GM_STATIC void
GM_FNAME(gm_port_state_t *the_ps)
{
   gm_arch_minor_t minor_num;

 
   GM_ANNOUNCE_FNAME_ENTRY(5);

   GM_PRINT (GM_PRINT_LEVEL >= 6, (" - the_ps=%p\n", the_ps));

   gm_assert(the_ps);

   minor_num = the_ps->minor;

   if (the_ps->opened)
   {
      gm_port_state_close(the_ps);
#if GM_IRIX_DEBUG_MEMLEAKS
      gm_irix_mld_struct.pstate_close_cnt++;
#endif
   }
   
   hwgraph_chmod(the_ps->arch.vhdl, 0000);  /* Deny all access to hw vertex  */
   gm_minor_free(minor_num);
#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.minor_free_cnt++;
#endif

   GM_ANNOUNCE_FNAME_EXIT(5);
 
} /* gm_irix_delete_minor */

#undef GM_FNAME

/*
 * Add a device to the hwgraph
 */

#define GM_FNAME gm_irix_device_add

GM_STATIC gm_status_t
GM_FNAME(vertex_hdl_t from, char *name, vertex_hdl_t *new_vhdl, int priv_flag)
{
   graph_error_t   hwgraph_ret;  /* hwgraph function status */
   device_desc_t   device_desc;
   GM_PARAMETER_MAY_BE_UNUSED (priv_flag);
   
   GM_ANNOUNCE_FNAME_ENTRY(5);

   GM_PRINT (GM_IRIX_DEBUG_DEVNAME,
             ("calling hwgraph_char_device_add(0x%x, %s, %s, &new)\n",
                from, name, GM_PREFIX_STRING));

   hwgraph_ret =
      hwgraph_char_device_add(from,               /* starting vtx */
                              name,               /* path name    */
                              GM_PREFIX_STRING,   /* driver pfx   */
                              new_vhdl         ); /* new vertex   */
   if (hwgraph_ret != GRAPH_SUCCESS)
   {
      GM_WARN(("Could not add vertex %s; return = %d\n", name, hwgraph_ret));
      return GM_FAILURE;
   }

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.vertex_add_cnt++;
#endif

   GM_PRINT (GM_IRIX_DEBUG_DEVNAME,
             ("New vertex handle is %d (0x%x)\n", *new_vhdl, *new_vhdl));

/* Add identifying info to device descriptor */
#if GM_IRIX_DISPLAY_DEV_DESC
   gm_irix_display_dev_desc(device_desc_default_get(*new_vhdl), 0,
                            "before setting its name field");
#endif
/* NOTE: I saw code like this in the sample driver in the chapter
 *       on Network Device Drivers in the DDPG.  It doesn't seem to
 *       be very useful, but I'm leaving it here for now.
 */
   device_desc = device_desc_dup(*new_vhdl);
   device_desc_intr_name_set(device_desc, "Myrinet");
   device_desc_default_set(*new_vhdl, device_desc);
#if GM_IRIX_DISPLAY_DEV_DESC
   gm_irix_display_dev_desc(device_desc_default_get(*new_vhdl), 0,
                            "after setting its name field");
#endif

/* Set permissions for access to the devices  */
#if 0				/* Feldy says: give full access to all */
   if (priv_flag)
      hwgraph_chmod(*new_vhdl, 0660);  /* Only usr root and grp sys can open */
   else
#endif
      hwgraph_chmod(*new_vhdl, 0666);  /* Anyone can open                    */

   return GM_SUCCESS;

} /* gm_irix_device_add */

#undef GM_FNAME

/*
 * Add an edge to the hwgraph
 */

#define GM_FNAME gm_irix_edge_add

GM_STATIC gm_status_t
GM_FNAME(vertex_hdl_t vhdl, char *name)
{
   graph_error_t   hwgraph_ret;  /* hwgraph function status */


   GM_ANNOUNCE_FNAME_ENTRY(5);

/* sys/hwgraph.h defines hwgraph_root as 'extern vertex_hdl_t hwgraph_root'  */
   GM_PRINT (GM_IRIX_DEBUG_DEVNAME,
             ("Calling hwgraph_edge_add for %s\n", name)); 
   hwgraph_ret = hwgraph_edge_add(hwgraph_root, vhdl, name);

   if (hwgraph_ret != GRAPH_SUCCESS)
   {
      GM_WARN(("Could not create edge %s; return = %d\n", name, hwgraph_ret));
      return GM_FAILURE;
   }
#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.edge_add_cnt++;
#endif
   return GM_SUCCESS;

} /* gm_irix_edge_add */

#undef GM_FNAME



/*
 * Remove an edge from the hwgraph
 */

#define GM_FNAME gm_irix_edge_remove

GM_STATIC void
GM_FNAME(vertex_hdl_t *vhdl, char *name)
{
   graph_error_t   hwgraph_ret;  /* hwgraph function status */


   GM_ANNOUNCE_FNAME_ENTRY(5);

   GM_PRINT (GM_IRIX_DEBUG_DEVNAME,
             ("Calling hwgraph_edge_remove for %s, vhdl=%d\n", name, *vhdl)); 

/* sys/hwgraph.h defines hwgraph_root as 'extern vertex_hdl_t hwgraph_root'  */
   hwgraph_ret = hwgraph_edge_remove(hwgraph_root, name, vhdl);

   if (hwgraph_ret != GRAPH_SUCCESS)

      GM_WARN(("Could not remove edge %s; return = %d\n", name, hwgraph_ret));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.edge_remove_cnt++;
#endif

} /* gm_irix_edge_remove */

#undef GM_FNAME

/*
 * Remove a vertex from the hwgraph
 *
 * NOTE: We don't actually know how to do this.  The function
 *       hwgraph_vertex_destroy() leaves the hardware graph in a PANIC-able
 *       state, and the man page says this function is just for kernel-internal
 *       use, not for drivers.
 *
 *       As a substitute for destroying the vertex, we change its permissions
 *       to 000.
 */

#define GM_FNAME gm_irix_device_destroy

GM_STATIC void
GM_FNAME(vertex_hdl_t vhdl)
{
   GM_ANNOUNCE_FNAME_ENTRY(1);

   hwgraph_chmod(vhdl, 0000);  /* Now nobody can open */

   GM_NOTE (("No action taken to destroy vhdl=%d; only chmod 000\n", vhdl)); 

} /* gm_irix_device_destroy */

#undef GM_FNAME

/*
 * Record the existence of a vertex for a given unit/gm_minor pair,
 * so that we can reuse this vertex later.
 *
 * NOTE: It would be a good idea to add bulletproofing to avoid the situation
 *       where we would attempt to "reuse" a vertex that had not been disposed
 *       of, and was in fact still in use.  This would represent an internal
 *       error, since GM should not re-assign a given gm_minor number for a
 *       given instance when the previous port state has not been closed yet.
 */

#define GM_FNAME gm_irix_record_new_vhdl

GM_STATIC gm_status_t
GM_FNAME(unsigned int    the_unit,      /* Identifies the instance      */
         gm_arch_minor_t the_gm_minor,	/* Identifies the port state    */
         vertex_hdl_t    the_vhdl)	/* Identifies the device vertex */
{
   gm_irix_device_hash_key_t dev_hash_key;


   dev_hash_key.unit  = the_unit;
   dev_hash_key.minor = the_gm_minor;

   gm_mutex_enter((struct gm_mutex *)gm_irix_device_hash_sync);
   if (gm_hash_insert(gm_irix_device_hash, &dev_hash_key, &the_vhdl) !=
                                                                   GM_SUCCESS)
   {
      gm_mutex_exit((struct gm_mutex *)gm_irix_device_hash_sync);
      GM_WARN(("Could not insert device hash for vhdl %d (0x%x)\n",
               the_vhdl, the_vhdl));
      return GM_FAILURE;
   }
   gm_mutex_exit((struct gm_mutex *)gm_irix_device_hash_sync);
   GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
            ("Inserted device hash for vhdl %d (0x%x), unit %d, gm_minor %d\n",
             the_vhdl, the_vhdl, the_unit, the_gm_minor));

   gm_mutex_enter((struct gm_mutex *)gm_irix_minor_hash_sync);
   if (gm_hash_insert(gm_irix_minor_hash, &the_vhdl, &the_gm_minor) !=
                                                                   GM_SUCCESS)
   {
      gm_mutex_exit((struct gm_mutex *)gm_irix_minor_hash_sync);
      GM_WARN(("Could not insert minor hash for minor %d\n", the_gm_minor));
      return GM_FAILURE;
   }
   gm_mutex_exit((struct gm_mutex *)gm_irix_minor_hash_sync);
   GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
            ("Inserted minor hash for minor %d, vhdl %d (0x%x)\n",
             the_gm_minor, the_vhdl, the_vhdl));

   return GM_SUCCESS;
}

#undef GM_FNAME

/*
 * Find the vertex handle for the existing vertex for a given unit/gm_minor
 * pair, if any.  Returns v-handle, or zero if none has been created yet for
 * this unit/gm_minor pair.
 */

#define GM_FNAME gm_irix_find_old_vhdl

GM_STATIC vertex_hdl_t
GM_FNAME(unsigned int    the_unit,      /* Identifies the instance      */
         gm_arch_minor_t the_gm_minor)	/* Identifies the port state    */
{
   gm_irix_device_hash_key_t hash_key;
   vertex_hdl_t              *old_vhdl;


   hash_key.unit  = the_unit;
   hash_key.minor = the_gm_minor;

   old_vhdl = (vertex_hdl_t *)gm_hash_find(gm_irix_device_hash, &hash_key);

   if (old_vhdl)
   {
      GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
	       ("gm_hash_find returned vhdl %d for unit %d, gm_minor %d\n",
	        *old_vhdl, the_unit, the_gm_minor));
      return (*old_vhdl);
   }
   else
   {
      GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
	       ("gm_hash_find returned NULL for unit %d, gm_minor %d\n",
	        the_unit, the_gm_minor));
      return (0);
   }
}

#undef GM_FNAME


/*
 * Find the minor number for a given vertex_hdl_t.
 * This search should always succeed.
 * Returns 1 if success.
 */

#define GM_FNAME gm_irix_minor_for_vhdl

GM_STATIC int
GM_FNAME(vertex_hdl_t the_vhdl, gm_arch_minor_t *the_minor)
{
   gm_irix_minor_hash_key_t hash_key;
   gm_arch_minor_t          *minor_num;


   hash_key = the_vhdl;

   minor_num = (gm_arch_minor_t *)gm_hash_find(gm_irix_minor_hash, &hash_key);

   if (minor_num)
   {
      GM_PRINT(GM_IRIX_DEBUG_DEVNAME,
	       ("gm_hash_find returned minor %d for vhdl %d\n",
	        *minor_num, the_vhdl));
      *the_minor = *minor_num;
      return (1);
   }
   else
   {
      GM_PANIC(("gm_hash_find returned NULL for vhdl %d\n", the_vhdl));
      return (0);
   }
}

#undef GM_FNAME



GM_STATIC long
gm_irix_device_hash_compare(void *a, void *b)
{
   GM_PRINT(GM_IRIX_DEBUG_DEVHASH,
            ("gm_irix_device_hash_compare entered with {%d,%d}, {%d,%d}\n",
	     ((gm_irix_device_hash_key_t *)a)->unit,
             ((gm_irix_device_hash_key_t *)a)->minor,
             ((gm_irix_device_hash_key_t *)b)->unit,
	     ((gm_irix_device_hash_key_t *)b)->minor));

   return ((((gm_irix_device_hash_key_t *)a)->unit !=
                                  ((gm_irix_device_hash_key_t *)b)->unit) ||
           (((gm_irix_device_hash_key_t *)a)->minor !=
                                  ((gm_irix_device_hash_key_t *)b)->minor)   );
}



GM_STATIC unsigned long
gm_irix_device_hash_hash(void *a)
{
   gm_u64_t hash_value;
  

   GM_PRINT(GM_IRIX_DEBUG_DEVHASH,
            ("gm_irix_device_hash_hash entered with %p -> {%d,%d}\n",
             a, ((gm_irix_device_hash_key_t *)a)->unit,
                ((gm_irix_device_hash_key_t *)a)->minor));

   hash_value = ((gm_irix_device_hash_key_t *)a)->unit;
   hash_value = (hash_value << 32) +
	        ((gm_irix_device_hash_key_t *)a)->minor;

   return gm_crc((void *)&hash_value, sizeof(hash_value));
}



GM_STATIC long
gm_irix_minor_hash_compare(void *a, void *b)
{
   GM_PRINT(GM_IRIX_DEBUG_DEVHASH,
            ("gm_irix_minor_hash_compare entered with %d, %d\n",
             *((gm_irix_minor_hash_key_t *)a),
             *((gm_irix_minor_hash_key_t *)b)));

   return (*((gm_irix_minor_hash_key_t *)a) !=
           *((gm_irix_minor_hash_key_t *)b)    );
}



GM_STATIC unsigned long
gm_irix_minor_hash_hash(void *a)
{
   gm_u64_t hash_value;
  

   GM_PRINT(GM_IRIX_DEBUG_DEVHASH,
            ("gm_irix_minor_hash_hash entered with %p -> %d\n",
             a, *((gm_irix_minor_hash_key_t *)a) ));

   hash_value = *((gm_irix_minor_hash_key_t *)a);

   return gm_crc((void *)&hash_value, sizeof(hash_value));
}

/*
 * Set up interrupt handling, based on previously allocated "interrupt object"
 */

#define GM_FNAME gm_irix_enable_interrupts

GM_STATIC gm_status_t
GM_FNAME(gm_instance_state_t *the_is)
{
   int pci_ret;	     /* pci function status */


   GM_ANNOUNCE_FNAME_ENTRY(2);

   pci_ret =
      pciio_intr_connect(the_is->arch.intr_obj,   /* Int. resource handle */
                         gm_irix_intr,            /* The handler          */
                         the_is,                  /* Args for handler     */
                         (void *)0             ); /* Thread pointer n/a   */
   if (pci_ret)
   {
      GM_WARN(("Could not connect interrupt handler"));
      goto abort_with_nothing;
   }

   the_is->arch.ihandlr = gm_irix_intr;

   /* Enable interrupts.  NOTE: This must be done only after it is safe
      to call the interrupt handler for this device, meaning that
      gm_instance_init() has been called and all the above setup is done. */

   GM_PRINT (GM_PRINT_LEVEL >= 1,
             ("Enabling interrupts for unit %d\n", the_is->id));
   if (gm_enable_interrupts(the_is) != GM_SUCCESS)
   {
      GM_WARN(("Cannot enable interrupts\n"));
      goto abort_with_connected_intr;
   }

   the_is->arch.interrupts_enabled = 1;

   GM_ANNOUNCE_FNAME_RETURN(2, GM_SUCCESS);

abort_with_connected_intr:
   pciio_intr_disconnect(the_is->arch.intr_obj);
   the_is->arch.ihandlr = (intr_func_t)NULL;
abort_with_nothing:
   GM_ANNOUNCE_FNAME_RETURN(1, GM_INTERNAL_ERROR);
}

#undef GM_FNAME

/*
 * gm_irix_disable_interrupts()
 *
 * Description: Tear down interrupt handling
 */

#define GM_FNAME gm_irix_disable_interrupts

GM_STATIC void
GM_FNAME(gm_instance_state_t *the_is)
{
   GM_ANNOUNCE_FNAME_ENTRY(2);

   if (the_is->arch.interrupts_enabled)
   {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
                ("Disabling interrupts for unit %d\n", the_is->id));
   
      if (gm_disable_interrupts(the_is) != GM_SUCCESS)
	 GM_NOTE (("Cannot disable interrupts\n"));
   }

   if (the_is->arch.ihandlr)
   {
      pciio_intr_disconnect(the_is->arch.intr_obj);
      the_is->arch.ihandlr = (intr_func_t)NULL;
   }
}

#undef GM_FNAME

/*
 * gm_irix_intr_free()
 *
 * Description:  Deallocate the interrupt "object" for this instance
 */

GM_STATIC void
gm_irix_intr_free(gm_instance_state_t *the_is)
{
   if (the_is->arch.intr_obj)
   {
      pciio_intr_free(the_is->arch.intr_obj);
      the_is->arch.intr_obj = (pciio_intr_t)NULL;
   }
}

/*
 * gm_irix_intr()
 *
 * Description:  This is the interrupt handler declared to IRIX PCI via the
 *               pciio_intr_connect() function.
 *
 * NOTE:         Types specifying the function prototype are declared
 *               in <sys/iobus.h>.  We have specified, in the
 *               pciio_intr_connect() call, that the intr_arg_t is a pointer
 *               to the instance_state.
 *
 */
GM_STATIC void
gm_irix_intr(intr_arg_t arg)
{
   gm_u32_t  ret_code;


   GM_PRINT (GM_PRINT_LEVEL >= 9,
             ("gm_irix_intr() entered with 0x%x\n", arg)); 

/* If not GM_DEBUG, ret_code is set but never used.  Set it anyway, in case
 * its value might be useful in a crash dump.
 */
   ret_code = gm_intr((gm_instance_state_t *)arg);

#if GM_DEBUG		   /* In the interrupt path, so conditional compile */
   if (GM_PRINT_LEVEL >= 8)
     {
       char *ret_string;

       switch (ret_code)
	 {
	 case GM_ARCH_INTR_UNCLAIMED:
	   ret_string = "UNCLAIMED";
	   break;
	 case GM_ARCH_INTR_CLAIMED:
	   ret_string = "CLAIMED";
	   break;
	 default:
	   ret_string = "<unknown>"; 
	 };
       GM_PRINT (GM_PRINT_LEVEL >= 8, 
                 ("gm_intr() returned %d (%s)\n", ret_code, ret_string)  );
     }
#endif
}


/*
 * gm_irix_ehandlr()
 *
 * Description:  This is the error handler declared to IRIX PCI via the
 *               pciio_error_register() function.
 *
 * NOTE:         Types specifying the function prototype are declared
 *               in <sys/ioerror.h>.
 */
#define GM_FNAME gm_irix_ehandlr

GM_STATIC int
GM_FNAME (error_handler_arg_t arg,
	  int                 error_code,
	  ioerror_mode_t      mode,
	  ioerror_t           *ioerror   )
{
   gm_instance_state_t *is = (gm_instance_state_t *)arg;
   char                *mode_label;
   int                 err_code_copy;

#define return_code 1
#define GM_IRIX_DECODE_BIT(word,bit,label) \
   if (word & bit) {                       \
      _GM_PRINT(1, ("    " label "\n"));   \
      word &= ~(bit);                      \
   }


   GM_CALLED ();

   GM_WARN(("Error handler: unit %d, e_code %d (0x%x)\n",
            is->id, error_code, error_code));

   GM_NOTE(("Pausing 2 secs\n"));
   delay((long)drv_usectohz(2000000)); /* Wait for prints to complete */

   GM_NOTE(("Error code interpretation:\n"));
   switch(error_code)
   {
   case IOECODE_UNSPEC:
      _GM_PRINT(1, ("    <UNSPECIFIED>\n"));
      break;

   case IOECODE_PIO_READ:
      _GM_PRINT(1, ("    PIO Read\n"));
      break;

   case IOECODE_PIO_WRITE:
      _GM_PRINT(1, ("    PIO Write\n"));
      break;

   case IOECODE_DMA_READ:
      _GM_PRINT(1, ("    DMA Read\n"));
      break;

   case IOECODE_DMA_WRITE:
      _GM_PRINT(1, ("    DMA Write\n"));
      break;

   default:
      err_code_copy = error_code;
      GM_IRIX_DECODE_BIT(err_code_copy, IOECODE_PIO,   "PIO");
      GM_IRIX_DECODE_BIT(err_code_copy, IOECODE_DMA,   "DMA");
      GM_IRIX_DECODE_BIT(err_code_copy, IOECODE_READ,  "Read");
      GM_IRIX_DECODE_BIT(err_code_copy, IOECODE_WRITE, "Write");
      if (err_code_copy)
	 _GM_PRINT(1, ("    *** Unknown flags 0x%x\n", err_code_copy));
   }

   switch(mode)
   {
   case MODE_DEVPROBE:
      mode_label = "Probing mode";
      break;
   case MODE_DEVERROR:
      mode_label = "Error while system running";
      break;
   case MODE_DEVUSERERROR:
      mode_label = "Device error due to user mode access";
      break;
   case MODE_DEVREENABLE:
      mode_label = "Re-enable pass";
      break;
   default:
      mode_label = "<unknown code>";
   }
   GM_NOTE(("Error mode is %s\n", mode_label));

   GM_NOTE(("Pausing 1 sec, then calling ioerror_dump()\n"));
   delay((long)drv_usectohz(1000000)); /* Wait 1 sec for prints to complete */
   ioerror_dump(GM_PREFIX_STRING, error_code, mode, ioerror);

   GM_NOTE(("Pausing 1 sec, then returning %d\n", return_code));
   delay((long)drv_usectohz(1000000)); /* Wait 1 sec for prints to complete */
   GM_RETURN (return_code);		/* Tell IRIX we are helpless */
}

#undef GM_FNAME

/*
 * gm_irix_init_dev_lock()
 *
 * Description:  Initialize the device lock in the instance state.
 */
GM_STATIC void
gm_irix_init_dev_lock(gm_instance_state_t *the_is)
{
   char *usage_dev_lock = "device_lock";

   gm_arch_sync_init(&the_is->arch.device_lock, the_is);

   if (strlen(usage_dev_lock) < GM_IRIX_SYNC_USAGE_LEN)
      strcpy(the_is->arch.device_lock.usage, usage_dev_lock);
}


/*
 * gm_irix_load_config_values()
 *
 * Description:  Copy the configuration space values from the card to a host
 *               data structure.
 */
GM_STATIC void
gm_irix_load_config_values(gm_instance_state_t *the_is, 
                           gm_pci_config_t     *cfg_data)
{
   unsigned short  i;


   gm_arch_read_pci_config_16(the_is, 0, &(cfg_data->Vendor_ID)); 
   gm_arch_read_pci_config_16(the_is, 2, &(cfg_data->Device_ID));

   gm_arch_read_pci_config_16(the_is, 4, &(cfg_data->Command));
   gm_arch_read_pci_config_16(the_is, 6, &(cfg_data->Status ));

   gm_arch_read_pci_config_8(the_is,  8, &(cfg_data->Revision_ID));
   gm_arch_read_pci_config_8(the_is,  9, 
                                &(cfg_data->Class_Code_Programming_Interface));
   gm_arch_read_pci_config_8(the_is, 10, &(cfg_data->Class_Code_Subclass    ));
   gm_arch_read_pci_config_8(the_is, 11, &(cfg_data->Class_Code_Base_Class  ));

   gm_arch_read_pci_config_8(the_is, 12, &(cfg_data->Cache_Line_Size));
   gm_arch_read_pci_config_8(the_is, 13, &(cfg_data->Latency_Timer  ));
   gm_arch_read_pci_config_8(the_is, 14, &(cfg_data->Header_Type    ));
   gm_arch_read_pci_config_8(the_is, 15, &(cfg_data->bist           ));

   for (i=0; i<6; i++)
      gm_arch_read_pci_config_32(the_is, 16 + i*4,
                                    &(cfg_data->Base_Addresses_Registers[i]));

   gm_arch_read_pci_config_32(the_is, 40, &(cfg_data->Cardbus_CIS_Pointer));

   gm_arch_read_pci_config_16(the_is, 44, &(cfg_data->Subsystem_Vendor_ID)); 
   gm_arch_read_pci_config_16(the_is, 46, &(cfg_data->Subsystem_ID       )); 

   gm_arch_read_pci_config_32(the_is, 48,
                                      &(cfg_data->Expansion_ROM_Base_Address));

   gm_arch_read_pci_config_32(the_is, 52, &(cfg_data->Reserved[0]));
   gm_arch_read_pci_config_32(the_is, 56, &(cfg_data->Reserved[1]));

   gm_arch_read_pci_config_8(the_is, 60, &(cfg_data->Interrupt_Line));
   gm_arch_read_pci_config_8(the_is, 61, &(cfg_data->Interrupt_Pin ));
   gm_arch_read_pci_config_8(the_is, 62, &(cfg_data->Min_Gnt       ));
   gm_arch_read_pci_config_8(the_is, 63, &(cfg_data->Max_Lat       ));

} /* gm_irix_load_config_values() */


/**********************************************************************
 * DEBUGGING SUPPORT
 **********************************************************************/

#if GM_IRIX_DISPLAY_GLOBAL_PARMS
/* gm_irix_display_global_parms()
 *
 * Description: display various global variables, parameters, and
 *              defines, along with various sizes etc. for debugging.
 *
 * NOTE:  "GM_PRINT (1,(..." is used here because these prints are
 *        always and forever unconditional.
 */
GM_STATIC void
gm_irix_display_global_parms(void)
{
#if GM_DEBUG
   gm_boolean_t printed_debug_flag = 0;
   char         *monster_text =
#if GM_IRIX_USE_MONSTER
                             "IS";
#else
                             "is NOT";
#endif
#if 0				/* See references below */
   ulong_t   syst_time;		/* really a time_t value */
   struct tm local_time;
   char      *time_string;
#endif
#endif


#if 0				/* Have to find out how to do this */
   drv_getparm (TIME, &syst_time);
   local_time  = localtime(&syst_time);
   time_string = asctime(&localtime);
   GM_PRINT (1, ("System time of day is %s", time_string)); /* no '\n' needed */
#endif
   GM_PRINT (1, ("GM_DEBUG = %d\n", GM_DEBUG));
   GM_PRINT (1, ("Print level is %d\n", GM_PRINT_LEVEL));
   GM_PRINT (1, ("Monster debug buffer %s in use\n", monster_text));
   GM_PRINT (1, ("Sized types in the kernel:\n"));
   _GM_PRINT (1, ("    gm_s64_t   %d\n", sizeof(gm_s64_t )));
   _GM_PRINT (1, ("    gm_up_t    %d\n", sizeof(gm_up_t  )));
   _GM_PRINT (1, ("    gm_dp_t    %d\n", sizeof(gm_dp_t  )));
   _GM_PRINT (1, ("    paddr_t    %d\n", sizeof(paddr_t  )));
   _GM_PRINT (1, ("    uvaddr_t   %d\n", sizeof(uvaddr_t )));
   _GM_PRINT (1, ("    iopaddr_t  %d\n", sizeof(iopaddr_t)));
   _GM_PRINT (1, ("    alenaddr_t %d\n", sizeof(alenaddr_t)));
   _GM_PRINT (1, ("    long       %d\n", sizeof(long)));
   _GM_PRINT (1, ("    ulong_t    %d\n", sizeof(ulong_t)));
   _GM_PRINT (1, ("    pid_t      %d\n", sizeof(pid_t)));
   GM_PRINT (1, ("Some defines:\n"));
   _GM_PRINT(1, ("    GM_SIZEOF_VOID_P     %d\n", GM_SIZEOF_VOID_P));
   _GM_PRINT(1, ("    GM_SIZEOF_UP_T       %d\n", GM_SIZEOF_UP_T));
   _GM_PRINT(1, ("    GM_SIZEOF_DP_T       %d\n", GM_SIZEOF_DP_T));
   _GM_PRINT(1, ("    GM_NUM_PORTS         %d\n", GM_NUM_PORTS));
   _GM_PRINT(1, ("    GM_DMA_GRANULARITY   %d\n", GM_DMA_GRANULARITY));
   _GM_PRINT(1, ("    GM_RDMA_GRANULARITY  %d\n", GM_RDMA_GRANULARITY));
   _GM_PRINT(1, ("    GM_IP_MTU            %d\n", GM_IP_MTU));
   _GM_PRINT(1, ("    GM_MAX_ETHERNET_GATHER_CNT %d\n",
                 GM_MAX_ETHERNET_GATHER_CNT));
   GM_PRINT  (1, ("Some structure sizes:\n"));
   _GM_PRINT (1, ("    gm_port_protected_lanai_side_t  %d\n",
                  sizeof(gm_port_protected_lanai_side_t)));
   _GM_PRINT (1, ("    gm_pte_t                        %d\n",
                  sizeof(gm_pte_t)));
   _GM_PRINT (1, ("    struct gm_off_len_uvma          %d\n",
                  sizeof(struct gm_off_len_uvma)));
   _GM_PRINT (1, ("    gm_page_hash_cache_t            %d\n",
                  sizeof(gm_page_hash_cache_t)));
   _GM_PRINT (1, ("    gm_cached_pte_t                 %d\n",
                  sizeof(gm_cached_pte_t)));
   _GM_PRINT (1, ("    gm_recv_queue_slot_t            %d\n",
                  sizeof(gm_recv_queue_slot_t)));
   _GM_PRINT (1, ("    struct gm_lanai_globals         %d (0x%x)\n",
                  sizeof(struct gm_lanai_globals),
                  sizeof(struct gm_lanai_globals)));
   _GM_PRINT (1, ("    struct gm_connection            %d\n",
                  sizeof(struct gm_connection)));
   _GM_PRINT (1, ("    gm_ethernet_segment_descriptor_t %d\n",
                  sizeof(gm_ethernet_segment_descriptor_t)));
   _GM_PRINT (1, ("    struct gm_ethernet_send_event   %d\n",
                  sizeof(struct gm_ethernet_send_event)));
   GM_PRINT  (1, ("Some arch-dependent structure sizes:\n"));
   _GM_PRINT (1, ("    gm_instance_state_t     %d\n",
                  sizeof(gm_instance_state_t)));
   _GM_PRINT (1, ("    gm_port_state_t         %d\n",
                  sizeof(gm_port_state_t)));
   _GM_PRINT (1, ("    gm_arch_instance_info_t %d\n",
                  sizeof(gm_arch_instance_info_t)));
   _GM_PRINT (1, ("    gm_arch_port_info_t     %d\n",
                  sizeof(gm_arch_port_info_t)));
   _GM_PRINT (1, ("    gm_arch_dma_region_t    %d\n",
                  sizeof(gm_arch_dma_region_t)));
   _GM_PRINT (1, ("    gm_arch_sync_t          %d\n",
                  sizeof(gm_arch_sync_t)));

#ifdef WORDS_BIGENDIAN
   GM_PRINT (1, ("WORDS_BIGENDIAN     = %d\n", WORDS_BIGENDIAN));
#else
   GM_PRINT (1, ("WORDS_BIGENDIAN     is undefined\n"));
#endif

#ifdef GM_CPU_BIGENDIAN
   GM_PRINT (1, ("GM_CPU_BIGENDIAN    = %d\n", GM_CPU_BIGENDIAN));
#else
   GM_PRINT (1, ("GM_CPU_BIGENDIAN    is undefined\n"));
#endif

#ifdef GM_BYTE_FLIP_NEEDED
   GM_PRINT (1, ("GM_BYTE_FLIP_NEEDED = %d\n", GM_BYTE_FLIP_NEEDED));
#else
   GM_PRINT (1, ("GM_BYTE_FLIP_NEEDED is undefined\n"));
#endif

#ifdef GM_NEED_MEMSET
   GM_PRINT (1, ("GM_NEED_MEMSET = %d\n", GM_NEED_MEMSET));
#else
   GM_PRINT (1, ("GM_NEED_MEMSET is undefined\n"));
#endif

#ifdef GM_ENABLE_NEW_FEATURES
   GM_PRINT (1, ("GM_ENABLE_NEW_FEATURES = %d\n", GM_ENABLE_NEW_FEATURES));
#else
   GM_PRINT (1, ("GM_ENABLE_NEW_FEATURES is undefined\n"));
#endif

#ifdef GM_SUPPORT_L4
   GM_PRINT (1, ("GM_SUPPORT_L4 = %d\n", GM_SUPPORT_L4));
#else
   GM_PRINT (1, ("GM_SUPPORT_L4 is undefined\n"));
#endif

#ifdef GM_SUPPORT_L7
   GM_PRINT (1, ("GM_SUPPORT_L7 = %d\n", GM_SUPPORT_L7));
#else
   GM_PRINT (1, ("GM_SUPPORT_L7 is undefined\n"));
#endif

#ifdef GM_SUPPORT_L8
   GM_PRINT (1, ("GM_SUPPORT_L8 = %d\n", GM_SUPPORT_L8));
#else
   GM_PRINT (1, ("GM_SUPPORT_L8 is undefined\n"));
#endif

#ifdef GM_SUPPORT_L9
   GM_PRINT (1, ("GM_SUPPORT_L9 = %d\n", GM_SUPPORT_L9));
#else
   GM_PRINT (1, ("GM_SUPPORT_L9 is undefined\n"));
#endif

#ifdef GM_SUPPORT_PCI_REV_1
   GM_PRINT (1, ("GM_SUPPORT_PCI_REV_1 = %d\n", GM_SUPPORT_PCI_REV_1));
#else
   GM_PRINT (1, ("GM_SUPPORT_PCI_REV_1 is undefined\n"));
#endif

#ifdef GM_SUPPORT_PCI_REV_2
   GM_PRINT (1, ("GM_SUPPORT_PCI_REV_2 = %d\n", GM_SUPPORT_PCI_REV_2));
#else
   GM_PRINT (1, ("GM_SUPPORT_PCI_REV_2 is undefined\n"));
#endif

#ifdef GM_SUPPORT_PCI_REV_3
   GM_PRINT (1, ("GM_SUPPORT_PCI_REV_3 = %d\n", GM_SUPPORT_PCI_REV_3));
#else
   GM_PRINT (1, ("GM_SUPPORT_PCI_REV_3 is undefined\n"));
#endif

#define GM_IRIX_PRINT_DEBUG_FLAG(f)  \
   if (f) {                          \
      _GM_PRINT(1, ("   " #f "\n")); \
      printed_debug_flag = 1; }

   GM_PRINT (1, 
         ("The following debug flags (and possibly others) are in effect:\n"));
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_DEVHASH);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_DEVNAME);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_DMAMAPS);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_MEMLEAKS);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_MUTEX);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_PIO);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_SYNC);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_IRIX_DEBUG_WAKE);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_DEBUG_HOST_DMA);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_DEBUG_INSTANCE_STATE);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_DEBUG_MEM_REGISTER);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_DEBUG_PORT_STATE);
   GM_IRIX_PRINT_DEBUG_FLAG (GM_LOG_DISPATCHES);
   if (!printed_debug_flag)
      _GM_PRINT(1, ("    <NONE>\n"));

   if (GM_SIZEOF_DP_T == 8)
      if (GM_IRIX_64BIT_DP_READY)
         GM_PRINT (1, ("Driver will use A64 DMA whenever possible\n"));
      else
         GM_PRINT (1, ("Driver will NOT use A64 DMA, even when possible\n"));
   else
      GM_PRINT (1, ("Driver can NOT use A64 DMA; GM_SIZEOF_DP_T = %d\n",
		    GM_SIZEOF_DP_T));

   GM_PRINT (1,("Driver %s memory registration\n",
		((GM_CAN_REGISTER_MEMORY) ? "supports" : "does NOT support")));
   GM_PRINT (1, ("Driver %s the IP interface\n",
		 ((GM_IRIX_IFNET_READY) ? "supports" : "does NOT support")));
   if (GM_IRIX_INSTANCE_LIMIT)
      GM_PRINT (1, ("At most %d units will be initialized\n",
		    GM_IRIX_INSTANCE_LIMIT));

} /* gm_irix_display_global_parms() */

#endif /* GM_IRIX_DISPLAY_GLOBAL_PARMS */



#if GM_DEBUG
/* gm_irix_test_basic_functions()
 *
 * Description: perform a basic sanity test before we get too far.
 */
GM_STATIC void
gm_irix_test_basic_functions(gm_instance_state_t *the_is)
{
   unsigned int spin_secs = 10;


#if 0
/* I just want to make sure this function isn't causing a bug... */
   GM_PRINT (GM_PRINT_LEVEL >= 0, 
          ("Exiting gm_irix_test_basic_functions() without doing anything\n"));
   return;
#endif

   GM_PRINT (GM_PRINT_LEVEL >= 0,
             ("Test of spinning for %d seconds\n", spin_secs));
   gm_arch_spin(the_is, GM_SEC_TO_USEC(spin_secs));

#if 0
   GM_PRINT (GM_PRINT_LEVEL >= 0, ("Test of pausing\n"));
   gm_pause_lanai(the_is);

   GM_PRINT (GM_PRINT_LEVEL >= 0, ("Test of unpausing\n"));
   gm_unpause_lanai(the_is);
#endif

   GM_PRINT (GM_PRINT_LEVEL >= 0, ("Testing complete\n"));
}
#endif


#if GM_IRIX_DISPLAY_DEV_DESC
/*
 * gm_irix_display_dev_desc()
 *
 * Description:  Provide formatted display of an IRIX device_desc_t struct
 *               for debugging purposes.
 *
 *               device_desc_t is declared in <sys/iobus.h>.
 */
GM_STATIC void
gm_irix_display_dev_desc(device_desc_t       the_desc, 
                         gm_instance_state_t *the_is,
                         char                *the_comment)
{
   device_desc_t dev_desc;
   

   if (the_is)
   {
      dev_desc = device_desc_default_get(the_is->arch.conn);

      if ((the_desc) && (dev_desc != the_desc))
         GM_WARN(("Device Descriptor moved from %p to %p !\n",
      	          the_desc, dev_desc));
   }
   else
      dev_desc = the_desc;

   GM_PRINT (GM_PRINT_LEVEL >= 0,
             ("Device descriptor values %s:\n", the_comment));

   _GM_PRINT(1, ("    Structure location: %p\n", dev_desc));

   if (!dev_desc)
      return;

   _GM_PRINT(1, ("    intr_target  = %d (0x%x)\n",
                 dev_desc->intr_target, dev_desc->intr_target));

   _GM_PRINT(1, ("    intr_policy  = %d (0x%x)\n",
                 dev_desc->intr_policy, dev_desc->intr_policy));

   _GM_PRINT(1, ("    intr_swlevel = %d (0x%x)\n",
                 dev_desc->intr_swlevel, dev_desc->intr_swlevel));

   _GM_PRINT(1, ("    intr_name    = %s\n",
	         (dev_desc->intr_name) ? dev_desc->intr_name : "<NULL>" ));

   _GM_PRINT(1, ("    flags        = 0x%x)\n", dev_desc->flags));
}
#endif


#if GM_IRIX_DISPLAY_CONFIG_VALUES
/*
 * gm_irix_display_config_values()
 *
 * Description:  Provide formatted display of the entire configuration
 *               space header for debugging purposes.
 */
GM_STATIC void
gm_irix_display_config_values(gm_pci_config_t *cfg_data, char *the_comment)
{
   unsigned short  i;
   char            *type_string;
   

   GM_PRINT (1, ("Configuration values %s:\n", the_comment));
   _GM_PRINT(1, ("      Vendor ID = 0x%x, Device ID = 0x%x\n",
                 cfg_data->Vendor_ID, cfg_data->Device_ID));
   _GM_PRINT(1, ("      Command   = 0x%x, Status    = 0x%x\n",
                 cfg_data->Command, cfg_data->Status));
   _GM_PRINT(1, ("      Revision ID = 0x%x, Class Base = 0x%x,\n",
                 cfg_data->Revision_ID, cfg_data->Class_Code_Base_Class));
   _GM_PRINT(1, ("            Subclass = 0x%x, Programming Interface = 0x%x\n",
                 cfg_data->Class_Code_Subclass,
                 cfg_data->Class_Code_Programming_Interface));
   _GM_PRINT(1, ("      Cache Line Size = 0x%x, Latency Timer = 0x%x,\n",
                 cfg_data->Cache_Line_Size, cfg_data->Latency_Timer));
   _GM_PRINT(1, ("            Header Type = 0x%x, bist = 0x%x\n",
                 cfg_data->Header_Type, cfg_data->bist));

   for (i=0; i<6; i++)
      _GM_PRINT(1, ("      Base Address Register %d = 0x%x\n",
                    i, cfg_data->Base_Addresses_Registers[i]));

   _GM_PRINT(1, ("      Cardbus CIS Pointer = 0x%x\n",
                 cfg_data->Cardbus_CIS_Pointer));
   _GM_PRINT(1, ("      Subsystem Vendor ID = 0x%x, Subsystem ID = 0x%x\n",
                 cfg_data->Subsystem_Vendor_ID, cfg_data->Subsystem_ID));
   _GM_PRINT(1, ("      Expansion ROM Base Address = 0x%x\n",
                 cfg_data->Expansion_ROM_Base_Address));
   _GM_PRINT(1, ("      Reserved (at 0x34)         = 0x%x\n",
                 cfg_data->Reserved[0]));
   _GM_PRINT(1, ("      Reserved (at 0x38)         = 0x%x\n",
                 cfg_data->Reserved[1]));
   _GM_PRINT(1, ("      Interrupt Line = 0x%x, Interrupt Pin = 0x%x\n",
                 cfg_data->Interrupt_Line, cfg_data->Interrupt_Pin));
   _GM_PRINT(1, ("            Min_Gnt = 0x%x, Max_Lat = 0x%x\n",
                 cfg_data->Min_Gnt, cfg_data->Max_Lat));

   switch(GM_PCI_BASE_ADDRESS_TYPE(cfg_data->Base_Addresses_Registers[0]))
   {
   case GM_PCI_BASE_ADDRESS_32BIT_TYPE:
      type_string = "Simple 32-bit BAR";
      break;

   case GM_PCI_BASE_ADDRESS_20BIT_TYPE:
      type_string = "20-bit BAR";
      break;

   case GM_PCI_BASE_ADDRESS_64BIT_TYPE:
      type_string = "Simple 64-bit BAR";
      break;

   case GM_PCI_BASE_ADDRESS_RESERVED_TYPE:
      type_string = "*** RESERVED VALUE ***";
      break;

   default:
      type_string = "*** INTERNAL ERROR ***";
   }   
   GM_NOTE (("Type flag in BAR[0] indicates %s\n", type_string));
}

/*
 * gm_irix_display_config_space()
 *
 * Description:  Provide formatted display of the entire configuration
 *               space header for debugging purposes.
 */
GM_STATIC void
gm_irix_display_config_space(gm_instance_state_t *the_is, char *the_comment)
{
   gm_pci_config_t cfg_data;


   gm_irix_load_config_values(the_is, &cfg_data);

   gm_irix_display_config_values(&cfg_data, the_comment);   
}
#endif


#if GM_DEBUG_INSTANCE_STATE
/*
 * gm_irix_display_instance_values()
 *
 * Description:  Provide formatted display of selected parts of the
 *               instance_state structure for debugging
 */
GM_STATIC void
gm_irix_display_instance_values(gm_instance_state_t *the_is, char *the_comment)
{
   GM_PRINT (1, ("Selected Instance State struct values %s:\n", the_comment));

   gm_dump_instance(the_is);	/* dump platform independent stuff */

   GM_PRINT (1, ("  PCI Revision = %d\n", the_is->ifc.pci.config.Revision_ID));

   _GM_PRINT(1,
             ("  Vertex handles: Conn = %d (0x%x), PCI access = %d (0x%x),\n",
              the_is->arch.conn,         the_is->arch.conn,
              the_is->arch.pci_acc_vhdl, the_is->arch.pci_acc_vhdl));

   _GM_PRINT(1,
        ("                  npriv clone = %d (0x%x), priv clone = %d (0x%x)\n",
              the_is->arch.nonpriv_vhdl, the_is->arch.nonpriv_vhdl,
              the_is->arch.priv_vhdl,    the_is->arch.priv_vhdl   ));

   _GM_PRINT(1,
        ("  dev_t: Conn = %d (0x%x), nonpriv = %d (0x%x), priv = %d (0x%x)\n",
              the_is->arch.conn_devt,    the_is->arch.conn_devt,
              the_is->arch.nonpriv_devt, the_is->arch.nonpriv_devt,
              the_is->arch.priv_devt,    the_is->arch.priv_devt   ));

   _GM_PRINT(1, ("  piomap = %p, Interrupt hdl = %p\n",
                 the_is->arch.piomap, the_is->arch.intr_obj));

   _GM_PRINT(1, ("  Controller = %d, controller_susp = %d\n",
                 the_is->arch.controller, the_is->arch.controller_susp));
}
#endif


#if GM_IRIX_DISPLAY_INTERRUPT_HANDLE
/*
 * gm_irix_display_interrupt_handle()
 *
 * Description:  Provide formatted display of the interrupt handle for
 *               debugging.  Unfortunately, the interrupt handle is an
 *               opaque structure; pciio_intr_t is declared in
 *               <sys/PCI/pciio.h> as a pointer to "struct pciio_intr_s",
 *               and that struct is not declared anywhere in /usr/include.
 *               All we can do is display some words in hex.
 */
GM_STATIC void
gm_irix_display_interrupt_handle(pciio_intr_t the_handle, char *the_comment)
{
   GM_PRINT (GM_PRINT_LEVEL >= 0, ("Interrupt handle %s:\n", the_comment));

   if (!the_handle)
      _GM_PRINT(1, ("    Handle pointer is VOID\n"));
   else
   {
      _GM_PRINT(1, ("    Handle pointer is 0x%p\n", the_handle));
      gm_irix_hexdump((u_char *)the_handle, 128);
   }
}
#endif




#if GM_IRIX_DISPLAY_DEVT
/*
 * gm_irix_parse_devt()
 *
 * Description:  Show the parse of a dev_t value
 */
GM_STATIC void
gm_irix_parse_devt(dev_t the_devt)
{
   GM_PRINT (1, ("Parse of dev_t %d (0x%x):\n", the_devt, the_devt));
   _GM_PRINT(1, ("      getminor = %d (0x%x), geteminor = %d (0x%x)\n",
                 getminor(the_devt),  getminor(the_devt),
                 geteminor(the_devt), geteminor(the_devt)));
   _GM_PRINT(1, ("      getmajor = %d (0x%x), getemajor = %d (0x%x)\n",
                 getmajor(the_devt),  getmajor(the_devt),
                 getemajor(the_devt), getemajor(the_devt)));
   _GM_PRINT(1, ("      canonical vertex name =\n"));
   _GM_PRINT(1, ("        %v\n", the_devt));
}
#endif


#if GM_IRIX_DISPLAY_INVENTORY
/*
 * gm_irix_displ_invent()
 *
 * Description:  Read and report the device inventory
 */
GM_STATIC void
gm_irix_displ_invent(dev_t the_devt, char *the_comment)
{
   inventory_t *inv_ret;
   invplace_t  inv_place;
   int         rec_numb = 0;


   GM_PRINT (1, ("Dump of inventory data for device %d (0x%x) %s\n",
                 the_devt, the_devt, the_comment));

   bzero(&inv_place, sizeof(inv_place)); /* supposed to indicate "get first" */
   inv_ret = device_inventory_get_next(the_devt, &inv_place);

   if (!inv_ret)
   {
      _GM_PRINT(1, ("    *** FAILED ***\n"));
      return;
   }

   while (inv_ret)
   {
      _GM_PRINT(1, ("    Readout of inventory record no. %d:\n", ++rec_numb));
      _GM_PRINT(1, ("      class \t= %d (0x%x)\n",
                    inv_ret->inv_class,      inv_ret->inv_class));
      _GM_PRINT(1, ("      type \t= %d (0x%x)\n",
                    inv_ret->inv_type,       inv_ret->inv_type));
      _GM_PRINT(1, ("      major \t= %d (0x%x)\n",
                    inv_ret->inv_controller, inv_ret->inv_controller));
      _GM_PRINT(1, ("      minor \t= %d (0x%x)\n",
                    inv_ret->inv_unit,       inv_ret->inv_unit));
      _GM_PRINT(1, ("      state \t= %d (0x%x)\n",
                    inv_ret->inv_state,      inv_ret->inv_state));

      inv_ret = device_inventory_get_next(the_devt, &inv_place);
   }
}
#endif


#if GM_IRIX_DISPLAY_SETPN_ARGS
/*
 * gm_irix_display_setpn_args()
 *
 * Description:  Provide formatted display of the special args struct
 *               used for the GM_SET_PORT_NUM ioctl.
 */
GM_STATIC void
gm_irix_display_setpn_args(gm_irix_set_port_num_arg_t *the_args,
                           char                       *the_comment)
{
   GM_PRINT (1, ("set_port_num_arg struct %s:\n", the_comment));

   if (!the_args)
      _GM_PRINT(1, ("  Args pointer is VOID\n"));
   else
   {
      _GM_PRINT(1, ("  Args pointer is 0x%p\n", the_args));
      _GM_PRINT(1, ("  port_id   = %d\n", the_args->port_id));
      _GM_PRINT(1, ("  name_size = %d\n", the_args->name_size));
      _GM_PRINT(1, ("  name      = %s\n",
                    (the_args->name) ? the_args->name : "<NULL>"));
   }
}
#endif



/*
 * __C_runtime_error()
 *
 * Description: This function receives runtime errors raised in the presence of
 *              the -DEBUG: flag.  See the DEBUG_GROUP(5) manpage for details.
 *
 * NOTES:       We would like to predicate the inclusion of this function
 *              on a #define, but unfortunately this is not straightforward,
 *              as the compiler doesn't generate one based on the -DEBUG flag.
 *
 *              This facility seems to be designed for user-level, not
 *              kernel, programs.  We are experimenting to see how it works.
 */

void __C_runtime_error(int trap_code, char *name, int line_no, ...)
{
   switch (trap_code)
   {
   /* Subscript range violations: */
   case BRK_RANGE:
      if ( line_no == -1 )
         GM_WARN(("Runtime subscript range violation in %s\n", name));
      else
         GM_WARN(("Runtime subscript range violation in %s (line %d)\n",
                  name, line_no));
      break;

   /* Others (unknown trap codes): */
   default:
      if ( line_no == -1 )
         GM_WARN(("Runtime trap %d in %s\n", trap_code, name));
      else
         GM_WARN(("Runtime trap %d in %s (line %d)\n",
                  trap_code, name, line_no));
   }
}
 

/*********************************************************************
 **                                                                 **
 **   END of gm_irix.c                                              **
 **                                                                 **
 *********************************************************************/
