/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: maxstern@myri.com */

/*
 $Id: gm_arch.c,v 1.43 2000/12/20 23:54:30 maxstern Exp $
 */

/************************************************************************
 * This file includes the IRIX-specific driver code, specifically the data
 * and functions which are required to support the platform-independent parts
 * of the GM driver.
 *
 * NOTE: This driver supports IRIX 6.5 *only*.  The "bible" for IRIX 6.5
 * device driver implementation is "IRIX Device Driver Programmer's
 * Guide," SGI Document Number 007-0911-130.
 *
 * NOTE: This driver supports PCI interfaces ONLY.
 *
 * NOTE: For a Table of Contents of this subdirectory, see README_TOC.
 *
 * Table of Contents of this source file (Data and Function Directory);
 * subfunctions indicated by indentation:
 * -------------------------------------------------------------------
 *
 * Forward declarations for subfunctions
 * Global Data
 *    External Linkage Data
 *    Static (Internal Linkage) Data
 * Debugging Declarations and Functions (will change as needed)
 *    gm_irix_decode_sync_status
 *    gm_irix_debug_sync
 * GM Required Functions
 *    PIO access (driver reads and writes into the card)
 *      gm_arch_read_pci_config_32()
 *      gm_arch_write_pci_config_32()
 *      gm_arch_read_pci_config_16()
 *      gm_arch_write_pci_config_16()
 *      gm_arch_read_pci_config_8()
 *      gm_arch_write_pci_config_8()
 *      gm_arch_map_io_space()
 *      gm_arch_unmap_io_space()
 *    DMA access (LANai reads and writes into node memory)
 *      gm_arch_dma_region_alloc()
 *      gm_arch_dma_region_dma_addr()
 *      gm_arch_dma_region_dma_addr_advance()
 *        gm_irix_region_dma_addr()
 *      gm_arch_dma_region_kernel_addr()
 *      gm_arch_dma_region_sync()
 *      gm_arch_dma_region_status()
 *      gm_arch_dma_region_free()
 *    Kernel memory management
 *      gm_arch_kernel_malloc()
 *      gm_arch_kernel_free()
 *      gm_irix_kmemalloc()
 *      gm_irix_kmemzalloc()
 *      gm_irix_kmemfree()
 *    Synchronization and related functions
 *      gm_arch_sync_init()
 *      gm_arch_sync_reset()
 *      gm_arch_sync_destroy()
 *      gm_arch_mutex_enter()
 *      gm_arch_mutex_exit()
 *      gm_arch_signal_sleep()
 *      gm_arch_timed_sleep()
 *      gm_arch_wake()
 *      gm_arch_spin()
 *    Memory Registration support
 *      gm_irix_setup_memory_registration()
 *      gm_irix_teardown_mem_registration()
 *      gm_arch_lock_user_buffer_page()
 *      gm_arch_unlock_user_buffer_page()
 *      gm_irix_fast_dma_callback()
 *      gm_irix_unlock_page()
 *      gm_irix_page_lock_hash_compare()
 *      gm_irix_page_lock_hash_hash()
 *    Internal DMA support
 *      gm_irix_dmamap_alloc()
 *      gm_irix_dmamap_addr()
 *      gm_irix_dmamap_free()
 *    Port state support
 *      gm_arch_port_state_init()
 *      gm_arch_port_state_open()
 *      gm_arch_port_state_close()
 *      gm_arch_port_state_fini()
 *    Ioctl support
 *      gm_arch_copyin()
 *      gm_arch_copyout()
 *    Miscellaneous required functions
 *      gm_arch_gethostname()
 *      gm_arch_get_page_len()
 *    Diagnostic and debugging support
 *      gm_irix_localize_status()
 *      gm_irix_get_bridge_version()
 *      gm_irix_print()
 *      gm_irix_warn()
 *      gm_irix_note()
 *      gm_irix_panic()
 *      gm_arch_abort()
 *      gm_irix_memleak_report()
 *      gm_irix_area_is_0()
 *      gm_irix_hexdump()
 * 
 ************************************************************************/

/************************************************************************
 * The following #include's are only those directly needed to support the
 * statements in this program text file.  For more information on this
 * practice, see "README.manifesto" in this subdirectory.
 ************************************************************************/

/* /// the work implied by the above comment has not been completed yet //// */

        /* *** IRIX header files (in collating sequence except as noted) *** */
/*
 * the alenlist(D4X) man page prescribes that these three headers be included
 * in this sequence.
 */
#include <sys/types.h>     /* basic, incl. vertex_hdl_t  */
#include <sys/alenlist.h>  /* Address/Length Lists       */

#include <assert.h>
#include <fcntl.h>
#include <stdarg.h>        /* needed for va_start, etc.                */

#include <ksys/ddmap.h>    /* i/o space mapping  */

#include <sys/buf.h>       /* fast_userdma() et al. */
#include <sys/cmn_err.h>   /* error handling     */
#include <sys/conf.h>
#include <sys/cred.h>
#include <sys/debug.h>
#include <sys/dmamap.h>    /* DMA stuff          */
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/hwgraph.h>   /* dev_to_vhdl, etc.  */
#include <sys/kmem.h>      /* kmem_ func's etc.  */
#include <sys/ksynch.h>    /* locks              */
#include <sys/mkdev.h>     /* mkdev, dev_t, etc. */
#include <sys/mload.h>     /* M_VERSION, etc.    */
#include <sys/open.h>      /* OTYP_CHR, etc.     */
#include <sys/sema.h>      /* locks & synchs     */
#include <sys/systm.h>     /* add_exit_callback, sprintf, vsprintf, etc.    */
#include <sys/utsname.h>   /* utsname()          */

#include <sys/PCI/pciio.h> /* PCI                */

/* The IRIX man page intro(D5) says this header file "must always be   */
/* the last header file included":                                     */
#include <sys/ddi.h>       /* DDI driver stuff                         */

         /* *** GM header files (approximately in dependency sequence    *** */
#include "gm_impl.h"       /* platform-independent, needed by gm.c           */
#include "gm_compiler.h"
#include "gm_cpp.h"
#include "gm_debug.h"
#include "gm_internal.h"   /* platform-independent */
#include "gm_lanai.h"      /* platform-independent */
#include "gm_arch_types.h" /* IRIX-specific declarations needed by gm_impl.h */
#include "gm_arch.h"       /* general IRIX-specific exported declarations    */
#include "gm_arch_local.h" /* declarations needed only in this subdirectory  */
#include "gm_debug_mem_register.h"
#include "gm_debug_port_state.h"


/***********************************************************************
 * Forward declarations
 ***********************************************************************/

GM_STATIC pciio_dmamap_t gm_irix_dmamap_alloc(vertex_hdl_t the_vhdl,
                                              size_t       the_size,
                                              gm_u32_t     prefetch,
                                              gm_u32_t     dma_64_bit);

GM_STATIC iopaddr_t gm_irix_dmamap_addr(pciio_dmamap_t the_dmamap,
                                        paddr_t        the_addr,
                                        paddr_t        *the_physaddr,
#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
                                        iopaddr_t      *the_template,
#endif
                                        size_t         the_size,
                                        gm_u32_t       dma_64_bit);

GM_STATIC void gm_irix_dmamap_free(pciio_dmamap_t the_dma_map);

GM_STATIC gm_dp_t gm_irix_region_dma_addr(gm_arch_dma_region_t *r,
                                          int                  advance_flag);

GM_STATIC char gm_irix_get_bridge_version(gm_instance_state_t *is);

#if GM_CAN_REGISTER_MEMORY
void gm_irix_fast_dma_callback(void *the_parm);

GM_STATIC void gm_irix_unlock_page(gm_irix_page_info_t *page_info);

GM_STATIC long gm_irix_page_lock_hash_compare(void *a, void *b);

GM_STATIC unsigned long gm_irix_page_lock_hash_hash(void *a);
#endif

/***********************************************************************
 * Global data
 ***********************************************************************/


/*
 * External global data required by other parts of driver
 * (initialized in <pfx>init())
 */

mutex_t gm_irix_print_mu;	   /* Guards printing from MP overlaps */
int     gm_need_kvtophys_analysis; /* Print the kvtophys analysis only once */

/*********************
 * Global data internal to this module
 *********************/

#if GM_CAN_REGISTER_MEMORY
typedef struct
{
/* This struct is stored in a hash table, keyed on the physical address.
 * Its main use is to avoid redundant pinning and unpinning of pinned pages.
 */
  int                 pin_count,    /* For debugging */
                      unpin_count;
  int                 ref_count;    /* Active users of this page        */

} gm_irix_page_lock_hash_t;

GM_STATIC struct gm_hash *gm_irix_page_lock_hash = (struct gm_hash *)NULL;
GM_STATIC gm_arch_sync_t *gm_irix_page_lock_hash_sync;
#endif


#if GM_IRIX_DEBUG_DMAMAPS
/* BUG: Pages allocated / deallocated by gx.c aren't tallied ///// */
static int dma_pages_active = 0;
#endif

/* ////// To do: Figure out where and how to increment and decrement this
 * //////        tally.  Note that we'd like it to represent the current
 * //////        total number of pages locked down, not the number of active
 * //////        fastdma's.
 */
static int pages_locked = 0;

#if GM_IRIX_DEBUG_MEMLEAKS
/* poor man's memory leak detection */
/* Not 'static', so debuggers can see it */
struct gm_irix_mld gm_irix_mld_struct = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
#endif

/**********************************************************************
 * 
 * Debugging stuff
 *
 **********************************************************************/

/* The following array is an attempt to compensate for the unfriendly
 * behavior of some IRIX kernels in not successfully getting all the
 * printed output from cmn_err output to the console.  We haven't been
 * able to learn why this happens nor what can be done to force the
 * flushing, so we simply keep the last bunch of output lines in memory
 * so we can look at them with the debugger.
 *
 * NOTE: The above does _NOT_ refer to losing the last part of the output
 *       before a PANIC or other abnormal halt.  Rather, in the presence of
 *       this (apparent) IRIX bug, lines and pieces of lines are missing
 *       from the _midst_ of the output, usually associated with times of
 *       large output volume.
 *
 * NOTE: This appears to be necessary on the Octane, but _NOT_ necessary
 *       on the Origins.
 */
#if GM_IRIX_USE_MONSTER
   char          gm_monster_debug_buf[GM_MONSTER_DEBUG_BUF_SIZ +
                                                           GM_MONSTER_TAG_SIZ];
   unsigned int  gm_monster_offset;        /* Byte offset of current entry   */
   char          *gm_monster_current;      /* Pointer to current entry       */
   unsigned int  gm_monster_wrap_tally;    /* Count of times it has wrapped  */

#define GM_MONSTER_SLOT_TERMINATE                            \
   if (gm_monster_current[GM_MONSTER_LINESIZ - 1] != '\0') { \
      gm_monster_current[GM_MONSTER_LINESIZ - 2] = '\n';     \
      gm_monster_current[GM_MONSTER_LINESIZ - 1] = '\0'; }

#define GM_MONSTER_NEXT_SLOT {                                             \
   unsigned int cursize = (unsigned int)strlen(gm_monster_current);        \
   if ((gm_monster_offset+cursize+GM_MONSTER_LINESIZ) <                    \
                                                 GM_MONSTER_DEBUG_BUF_SIZ) \
      gm_monster_offset += cursize;                                        \
   else {                                                                  \
      gm_monster_offset = 0;                                               \
      gm_monster_wrap_tally++; }                                           \
   gm_monster_current = &(gm_monster_debug_buf[gm_monster_offset]);        \
   bzero((void *)gm_monster_current, GM_MONSTER_LINESIZ);                  \
   strcpy(gm_monster_current, ">>>> CURRENT MONSTER POSITION <<<<\n\0"); }

#endif

#if GM_IRIX_DEBUG_WAKE
GM_STATIC void
gm_irix_decode_sync_status(int the_status)
{
   char *decoded_status;


   switch(the_status)
   {
   case GM_ARCH_SYNC_STATUS_INITIAL:
      decoded_status = "Initial state";
      break;

   case GM_ARCH_SYNC_STATUS_ASLEEP:
      decoded_status = "Asleep";
      break;

   case GM_ARCH_SYNC_STATUS_WAKED_B4_WAIT:
      decoded_status = "Awakened before waiting/sleeping";
      break;

   case GM_ARCH_SYNC_STATUS_WAKED_BY_SIGNAL:
      decoded_status = "Awakened by signal";
      break;

   case GM_ARCH_SYNC_STATUS_WAKED_BY_INTERRUPT:
      decoded_status = "Awakened by interrupt";
      break;

   case GM_ARCH_SYNC_STATUS_WAKED_BY_TIMEOUT:
      decoded_status = "Awakened by timeout";
      break;

   default:
      decoded_status = "<illegal value>";
   }

   _GM_PRINT (1, ("status ==> %s\n", decoded_status));
}
#endif


#if GM_IRIX_DEBUG_SYNC || GM_IRIX_DEBUG_MUTEX
#define GM_IRIX_OWNER_MU(mu) \
   if (MUTEX_OWNED(&(mu)))   \
      available = "NOT";     \
   else                      \
      available = "IS";      \
   if (MUTEX_MINE(&(mu)))    \
      mine = "IS";           \
   else                      \
      mine = "NOT";

#if SEMAINFOP
#define GM_IRIX_PRINT_MU(mu)                                             \
   if ((mu.m_info) || (mu.m_bits) || (mu.m_queue))                       \
      _GM_PRINT (1, ("... m_info = %s, m_bits = 0x%x, m_queue = 0x%p\n", \
                   mu.m_info, mu.m_bits, mu.m_queue));                   \
   else                                                                  \
      _GM_PRINT (1, ("... struct contents are null\n"));
#else
#define GM_IRIX_PRINT_MU(mu)                                \
   if ((mu.m_bits) || (mu.m_queue))                         \
      _GM_PRINT (1, ("... m_bits = 0x%x, m_queue = 0x%p\n", \
                   mu.m_bits, mu.m_queue));                 \
   else                                                     \
      _GM_PRINT (1, ("... struct contents are null\n"));
#endif

GM_STATIC void
gm_irix_debug_sync(gm_arch_sync_t *the_sync, char *the_comment)
{
   char *available,
        *mine;


   GM_PRINT (1, ("Debugging gm_arch_sync_t at 0x%p\n", the_sync));
   _GM_PRINT (1, ("    %s:\n", the_comment));
   _GM_PRINT (1, ("... usage: \"%s\"\n", the_sync->usage));
   if (GM_PRINT_LEVEL >= 2)
   {
      if (!gm_irix_hexdump((u_char *)the_sync, sizeof(*the_sync)))
	 return;			/* The area was all zero */
   }
   else
      if (gm_irix_area_is_0((u_char *)the_sync, sizeof(*the_sync)))
      {
	 _GM_PRINT (1, ("... Area is all zero\n"));
         return;
      }

   GM_IRIX_OWNER_MU(the_sync->simple_mu);
   _GM_PRINT (1, ("simple_mu (%s available, %s mine):\n", available, mine));
   GM_IRIX_PRINT_MU(the_sync->simple_mu);
   _GM_PRINT (1, ("... entry count %d, exit count %d\n",
                  the_sync->sim_mu_entry_cnt, the_sync->sim_mu_exit_cnt));

#if GM_IRIX_DEBUG_SYNC
/* outer_mu and inner_mu used only in conjuction with synch/wake logic */
   GM_IRIX_OWNER_MU(the_sync->outer_mu);
   _GM_PRINT (1, ("outer_mu (%s available, %s mine):\n", available, mine));
   GM_IRIX_PRINT_MU(the_sync->outer_mu);

   GM_IRIX_OWNER_MU(the_sync->inner_mu);
   _GM_PRINT (1, ("inner_mu (%s available, %s mine):\n", available, mine));
   GM_IRIX_PRINT_MU(the_sync->inner_mu);

#if SEMAINFOP
   if ((the_sync->sv.sv_info) || (the_sync->sv.sv_queue))
      _GM_PRINT (1, ("sv_info = %s, sv_queue = 0x%p\n",
		     the_sync->sv.sv_info, the_sync->sv.sv_queue));
   else
      _GM_PRINT (1, ("sv struct contents are null\n"));
#else
   if (the_sync->sv.sv_queue)
      _GM_PRINT (1, ("sv_queue = 0x%p\n", the_sync->sv.sv_queue));
   else
      _GM_PRINT (1, ("sv struct contents are null\n"));
#endif

   _GM_PRINT (1, ("... wake_cnt = %d\n", the_sync->wake_cnt));

#if GM_IRIX_DEBUG_WAKE
   _GM_PRINT (1, ("... status = %d\n", the_sync->status));
   gm_irix_decode_sync_status(the_sync->status);
#endif
#endif
}
#endif

/**********************************************************************
 * GM OS-abstraction required functions.
 *
 * Description:
 *
 * These are the functions that the platform-independent part of the
 * driver uses.  The interface and semantics of these functions are
 * predetermined, but the implementation is platform-dependent.
 *
 * E.g., gm_arch_read_pci_config_32() is declared in gm_impl.h, and it
 * is called by gm_instance.c:gm_pci_board_span() (but only in driver
 * versions that -- like this one -- support the PCI bus).
 *
 * NOTE: Functions below with names like gm_irix_* are local to the IRIX-
 * specific part of the driver.  Functions with names like gm_arch_* are
 * exported to the platform_independent part of the driver.
 *
 **********************************************************************/

/* This macro says that we want I/O to be in byte order                    */
/* NOTE: To support the O2, this may not be sufficient                     */
#define GM_IRIX_ENDIANNESS PCIIO_BYTE_STREAM


/**********************************************************************
 * PIO access, comprising config space access and piomapped access
 **********************************************************************/

/************
 * PCI configuration space access functions.
 *
 * NOTE: The function type of pciio_config_get() is uint64_t.  I haven't
 * been able to find where this type is defined in any IRIX header file, so
 * I suspect it's a compiler built-in.  I have more research to do on this.
 * Anyway, the real point to be aware of is that all three "read" functions
 * below coerce this uint64_t return value into the appropriate GM "sized
 * type," either gm_u32_t, gm_u16_t, or gm_u8_t.  Hopefully, this is no
 * problem.
 ************/

/* 32-bit accesses */

gm_status_t
gm_arch_read_pci_config_32 (gm_instance_state_t *is,
			    gm_offset_t offset,
			    gm_u32_t *value)
{
  *value = (gm_u32_t) pciio_config_get (is->arch.pci_acc_vhdl,
					(unsigned) offset,
					(unsigned) sizeof (*value));
  return GM_SUCCESS;
}



gm_status_t
gm_arch_write_pci_config_32(gm_instance_state_t *is,
			    gm_offset_t offset,
			    gm_u32_t value)
{
  pciio_config_set (is->arch.pci_acc_vhdl,
		    (unsigned) offset,
		    (unsigned) sizeof (value),
		    (uint64_t) value);
  return GM_SUCCESS;
}

 
/* 16-bit accesses */

gm_status_t
gm_arch_read_pci_config_16(gm_instance_state_t *is,
			   gm_offset_t offset,
			   gm_u16_t *value)
{
  *value = (gm_u16_t) pciio_config_get (is->arch.pci_acc_vhdl,
					(unsigned) offset,
					(unsigned) sizeof (*value));
  return GM_SUCCESS;
}



gm_status_t
gm_arch_write_pci_config_16(gm_instance_state_t *is,
			    gm_offset_t offset,
			    gm_u16_t value)
{
  pciio_config_set (is->arch.pci_acc_vhdl,
		    (unsigned) offset,
		    (unsigned) sizeof (value),
		    (uint64_t) value);
  return GM_SUCCESS;
}



/* 8-bit accesses */

gm_status_t
gm_arch_read_pci_config_8(gm_instance_state_t *is,
			  gm_offset_t offset,
			  gm_u8_t *value)
{
  *value = (gm_u8_t) pciio_config_get (is->arch.pci_acc_vhdl,
					(unsigned) offset,
					(unsigned) sizeof (*value));
  return GM_SUCCESS;
}



gm_status_t
gm_arch_write_pci_config_8(gm_instance_state_t *is,
			   gm_offset_t offset,
			   gm_u8_t value)
{
  pciio_config_set (is->arch.pci_acc_vhdl,
		    (unsigned) offset,
		    (unsigned) sizeof (value),
		    (uint64_t) value);
  return GM_SUCCESS;
}

/************
 * I/O space (PIO) mapping
 ************/

gm_status_t
gm_arch_map_io_space(gm_instance_state_t *is,
                     gm_u32_t            offset,
                     gm_u32_t            len,
                     void                **kaddr)
{
/* This macro says that we map the memory space defined by Base Addr Reg 0 */
#define GM_IRIX_PCIIO_SPACE PCIIO_SPACE_WIN(0)


   pciio_piomap_t pio_map  = (pciio_piomap_t)NULL;
   device_desc_t  dev_desc = device_desc_default_get(is->arch.conn);
   void           *bus_addr;


   GM_PRINT (GM_IRIX_DEBUG_PIO,
             ("gm_arch_map_io_space entered with 0x%p, %d,\n", is, offset));
   _GM_PRINT (GM_IRIX_DEBUG_PIO,
              ("    0x%x, 0x%p\n", len, kaddr));

   GM_PRINT (GM_IRIX_DEBUG_PIO,
             ("Calling pciio_piomap_alloc(%d, 0x%p, %d, 0,\n",
              is->arch.conn, dev_desc, GM_IRIX_PCIIO_SPACE));
   _GM_PRINT (GM_IRIX_DEBUG_PIO,
              ("    0x%x, 0x%x, 0x%x)\n", len, len, GM_IRIX_ENDIANNESS));
#if GM_IRIX_DISPLAY_DEV_DESC
   gm_irix_display_dev_desc(dev_desc, is, "passed to pciio_piomap_alloc()");
#endif
   pio_map = pciio_piomap_alloc(is->arch.conn,       /* vertex handle       */
                                dev_desc,            /* device descriptor   */
                                GM_IRIX_PCIIO_SPACE, /* space definition    */
                                0,                   /* off. w/i that space */
                                len,                 /* total size to alloc */
                                len,                 /* max sz in 1 mapping */
                                GM_IRIX_ENDIANNESS); /* flags: OK to sleep  */

   if (pio_map)
   {
      GM_PRINT (GM_IRIX_DEBUG_PIO,
                ("Calling pciio_piomap_addr(0x%p, 0x%x, 0x%x)\n",
                 pio_map, offset, len));
      bus_addr = pciio_piomap_addr(pio_map, offset, len);
      is->arch.piomap = pio_map;
   }
   else
   {
/*    We really don't think there's any situation where the above call
 *    fails but this one succeeds, but SGI once advised us to try this --
 *    so, here goes...
 */
      GM_WARN(
  ("Unable to allocate PIO map for IO space; trying pciio_piotrans_addr()\n"));
      bus_addr = pciio_piotrans_addr(is->arch.conn,  /* vertex handle       */
                                dev_desc,            /* device descriptor   */
                                GM_IRIX_PCIIO_SPACE, /* space definition    */
                                offset,              /* off. w/i that space */
                                len,                 /* total size to map   */
                                GM_IRIX_ENDIANNESS); /* flags: OK to sleep  */

   }
   
   if (!(bus_addr))
   {
      gm_status_t err_code;
      char        bridge_version;

      bridge_version = gm_irix_get_bridge_version(is);
      if (bridge_version < 'D')
      {
         GM_WARN (("Unable to utilize unit %d; bridge is revision %c;\n",
                   is->id, bridge_version));
         GM_WARN (("Bridge revision D or later needed; see SGI Bug 482741\n"));
         err_code = GM_UNSUPPORTED_DEVICE;
      }
      else
      {
         GM_WARN (("Unable to allocate PIO map for unit %d; reason unknown\n",
                   is->id));
         err_code = GM_OUT_OF_MEMORY;
      }

      if (pio_map)
      {
         pciio_piomap_done(pio_map);
         pciio_piomap_free(pio_map);
         pio_map = (pciio_piomap_t)NULL;
      }

      return(err_code);
   }

   *kaddr = (void *)((gm_u64_t)bus_addr + offset);

   GM_PRINT (GM_IRIX_DEBUG_PIO,
             ("gm_arch_map_io_space() mapped %d (0x%x) bytes\n", len, len));
   _GM_PRINT (GM_IRIX_DEBUG_PIO,
              ("     to 0x%p for unit %d\n", *kaddr, is->id));
   return(GM_SUCCESS);

} /* gm_arch_map_io_space() */


void
gm_arch_unmap_io_space(gm_instance_state_t *is,
                       gm_u32_t            offset,
                       gm_u32_t            len,
                       void                **kaddr)
{
   GM_PARAMETER_MAY_BE_UNUSED (offset);
   GM_PARAMETER_MAY_BE_UNUSED (len);
   GM_PARAMETER_MAY_BE_UNUSED (kaddr);
  
   if (is->arch.piomap)
   {
      pciio_piomap_free(is->arch.piomap);

      GM_PRINT (GM_IRIX_DEBUG_PIO,
                ("gm_arch_unmap_io_space() freed the piomap for unit %d\n",
                 is->id));
   }
   else
      GM_NOTE (
           ("gm_arch_unmap_io_space() found no allocated piomap for unit %d\n",
             is->id));

} /* gm_arch_unmap_io_space() */


/**********************************************************************
 * DMA region manipulation
 **********************************************************************/

/* Allocate LEN bytes of DMA memory that is contiguous in kernel space
 * but possibly segmented in DMA space.

   ////// HUH?  Does the above Solaris comment make sense for IRIX?
   //////       If so, explain more fully.  If not, pull it or replace
   //////       it with something more accurate and informative.
 *
 * If register_page_func is non-null, call it with (arg, dma address, page num)
 * for each page.
 */

#define GM_FNAME gm_arch_dma_region_alloc

gm_status_t
GM_FNAME(gm_instance_state_t         *is,
         gm_arch_dma_region_t        *r,
         gm_size_t                   len,
         gm_u32_t                    flags,
         gm_register_page_function_t register_page_func,
         void                        *arg                )
{
   gm_status_t    err_code;
   int            kmem_alloc_flags = KM_CACHEALIGN | KM_NOSLEEP;
   pciio_dmamap_t dma_map;      /* pointer to dmamap */
   iopaddr_t      pg0_dma_addr;
   gm_u32_t       pg0_len;
   paddr_t        kmemory,	/* allocated kernel memory for DMA */
                  physaddr;	/* corresponding physical address  */
   gm_u32_t       flags_copy;
   unsigned int   num_pages;	/* Number of pages required for 'len'  */
   int            gm_bits_dp = 8 * sizeof(gm_dp_t);
#if GM_IRIX_DEBUG_DMAMAPS
   int            flag_found = 0;
#endif

/* Set the following to 1 to experiment with PHYSCONTIG allocation of
 * dma-able memory.
 */
#define GM_PHYSCONTIG_EXPERIMENT 0

   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
             (GM_FNAME_STR " called with is=%p,\n", is));
   _GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
              ("    r=%p, len=%d (0x%x), flags=0x%x, arg=%p,\n",
               r, len, len, flags, arg));
   _GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
              ("    register_page_func=%p\n", register_page_func));
#if GM_IRIX_DEBUG_DMAMAPS
   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS, ("Decoded flags:\n"));
   if (flags & GM_ARCH_DMA_READ)
   {
      _GM_PRINT(GM_IRIX_DEBUG_DMAMAPS,("... GM_ARCH_DMA_READ\n"));
      flag_found = 1;
   }
   if (flags & GM_ARCH_DMA_WRITE)
   {
      _GM_PRINT(GM_IRIX_DEBUG_DMAMAPS,("... GM_ARCH_DMA_WRITE\n"));
      flag_found = 1;
   }
   if (flags & GM_ARCH_DMA_CONSISTENT)
   {
      _GM_PRINT(GM_IRIX_DEBUG_DMAMAPS,("... GM_ARCH_DMA_CONSISTENT\n"));
      flag_found = 1;
   }
   if (flags & GM_ARCH_DMA_CONTIGUOUS)
   {
      _GM_PRINT(GM_IRIX_DEBUG_DMAMAPS,("... GM_ARCH_DMA_CONTIGUOUS\n"));
      flag_found = 1;
   }
   if (flags & GM_ARCH_DMA_STREAMING)
   {
      _GM_PRINT(GM_IRIX_DEBUG_DMAMAPS,("... GM_ARCH_DMA_STREAMING\n"));
      flag_found = 1;
   }
   if (!flag_found)
      _GM_PRINT(GM_IRIX_DEBUG_DMAMAPS,("... <none>\n"));
#endif

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.dma_alloc_cnt++;  /* tally _calls_, not _successes_  */
#endif

   bzero((void *)r, sizeof(*r)); /* clean up the structure */

   gm_assert(GM_PAGE_LEN);
   num_pages = (unsigned int)(GM_PAGE_ROUNDUP(u32, len) / GM_PAGE_LEN);
   gm_always_assert(num_pages >= 1);

   flags_copy = flags;

   flags_copy &= ~GM_ARCH_DMA_RDWR;      /* see below */
   flags_copy &= ~GM_ARCH_DMA_STREAMING; /* see below */

/*    We don't think we need to tell IRIX anything based on this flag: */
   flags_copy &= ~GM_ARCH_DMA_CONSISTENT;

#if GM_PHYSCONTIG_EXPERIMENT
   if ((flags & GM_ARCH_DMA_CONTIGUOUS) ||
       ((len > GM_PAGE_LEN) && (len <= 4*GM_PAGE_LEN)) )
#else
   if (flags & GM_ARCH_DMA_CONTIGUOUS)
#endif
   {
      GM_NOTE (("Allocating a KM_PHYSCONTIG DMA map, 0x%x bytes\n", len));
      kmem_alloc_flags |= KM_PHYSCONTIG;
      flags_copy &= ~GM_ARCH_DMA_CONTIGUOUS;
   }

   if (flags_copy != 0)
   {
      GM_WARN((GM_FNAME_STR ": got an unexpected flag\n"));
      err_code = GM_INTERNAL_ERROR;
      goto abort_with_nothing;
   }

 
/* Allocate the kernel memory to be mapped                                 */
   kmemory = (paddr_t)gm_irix_kmemalloc(len, kmem_alloc_flags); 

   if (!kmemory)
   {
      GM_WARN((GM_FNAME_STR ": kmemalloc failed\n"));
      err_code = GM_OUT_OF_MEMORY;
      goto abort_with_nothing;
   }

/* Allocate the map object                                                 */

   dma_map = gm_irix_dmamap_alloc(is->arch.conn,
                                  len,
				  ((flags & GM_ARCH_DMA_READ)     ||
                                   (flags & GM_ARCH_DMA_STREAMING)  ),
                                  GM_IRIX_INST_USE_A64DMA(is));
   if (!dma_map)
   {
      GM_WARN((GM_FNAME_STR ": could not allocate DMA map\n"));
      err_code = GM_OUT_OF_MEMORY;
      goto abort_with_memory;
   }

   pg0_len      = (gm_u32_t)((len < GM_PAGE_LEN) ? len : GM_PAGE_LEN);
   pg0_dma_addr = gm_irix_dmamap_addr(dma_map, kmemory, &physaddr,
#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
                                      NULL, /* ==> call pciio_dmamap_addr() */
#endif
                                      pg0_len,
                                      GM_IRIX_INST_USE_A64DMA(is));

   if (!pg0_dma_addr)
   {
      GM_WARN((GM_FNAME_STR
               ": could not map pg 0 DMA address, %d bytes at kmem 0x%x\n",
               pg0_len, kmemory                                           ));
      err_code = GM_OUT_OF_MEMORY;
      goto abort_with_map;
   }

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.dma_pg_alloc_cnt += num_pages;
#endif
#if GM_IRIX_DEBUG_DMAMAPS
   dma_pages_active += num_pages;
   GM_PRINT (1, ("There are now %d total DMA pages allocated\n",
	         dma_pages_active));
#endif

/* Fill in the gm_arch_dma_region  */

   r->dma_vhdl    = is->arch.conn;
   r->dmamap      = dma_map;
   r->region_len  = len;
   r->last_pg_len = (unsigned)(len - GM_PAGE_LEN * (num_pages - 1));
   r->alloc_flags = flags;
   r->kmem_addr   = kmemory;
   r->phys_addr   = physaddr;
   r->cur_page    = 0;
   r->pages       = num_pages;
   r->dma_addr_p0 = (gm_dp_t)pg0_dma_addr;
#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
   r->dma_template= GM_IRIX_MAKE_DMA_TEMPLATE(pg0_dma_addr);
#endif
   r->map_active  = 1;
   if (GM_IRIX_INST_USE_A64DMA(is))
      r->dma_asize   = 64;
   else
      r->dma_asize   = 32;
   if (r->dma_asize > gm_bits_dp)
      GM_PANIC (("DMA map addr size (%d bits) > gm_dp_t (%d bits)\n",
                 r->dma_asize, gm_bits_dp));
   gm_assert(sizeof(r->dma_addr_p0)*8 >= r->dma_asize);

#if GM_IRIX_DEBUG_DMAMAPS
   GM_PRINT (1,
           ("gm_arch_dma_region at 0x%x has been filled in as follows:\n", r));
   _GM_PRINT(1, ("    Vertex Handle = %d (0x%x)\n", r->dma_vhdl,r->dma_vhdl));
   _GM_PRINT(1, ("    DMA map       = 0x%x\n", r->dmamap     ));
   _GM_PRINT(1, ("    Region Length = %d (0x%x)\n",
                                               r->region_len,r->region_len));
   _GM_PRINT(1, ("    Last pg. len. = %d (0x%x)\n",
                                               r->last_pg_len,r->last_pg_len));
   _GM_PRINT(1, ("    Alloc. Flags  = 0x%x\n", r->alloc_flags));
   _GM_PRINT(1, ("    Kmem Address  = 0x%x\n", r->kmem_addr  ));
   _GM_PRINT(1, ("    Phys Address  = 0x%x\n", r->phys_addr  ));
   _GM_PRINT(1, ("    Curr. Page #  = %d\n",   r->cur_page   ));
   _GM_PRINT(1, ("    Num. of Pages = %d\n",   r->pages      ));
   _GM_PRINT(1, ("    Pg 0 DMA Addr = 0x%x\n", r->dma_addr_p0));
#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
   _GM_PRINT(1, ("    DMA Addr tplt = 0x%x\n", r->dma_template));
#endif
   _GM_PRINT(1, ("    Active flag   = %d\n",   r->map_active )); 
   _GM_PRINT(1, ("    DMA addr size = %d bits %s\n",
                 r->dma_asize, 
                 (r->dma_asize == gm_bits_dp) ?
                                      "(same as gm_dp_t)" : 
                                      "(smaller than gm_dp_t)"));
#endif

 
/* Call the page registration functions for each page in the DMA region. */

   if (register_page_func != NULL)
   {
      int page_num;


      for (page_num = 0; page_num < num_pages; page_num++)
      {
	 gm_dp_t page_dp;
#if GM_IRIX_DEBUG_DMAMAPS
	 gm_dp_t page_dp_alt;
#endif

	 if (page_num == 0)
            page_dp = r->dma_addr_p0;
         else
	 {
            page_dp = (gm_dp_t)(gm_irix_dmamap_addr(
                         r->dmamap,
		         r->kmem_addr + (page_num)*GM_PAGE_LEN,
		         0,
#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
                         &(r->dma_template),
#endif
			 (page_num < (num_pages - 1)) ?
		                  GM_PAGE_LEN : r->last_pg_len,
                         (r->dma_asize == 64)));
#if GM_IRIX_DEBUG_DMAMAPS
            page_dp_alt = (gm_dp_t)(r->dma_addr_p0 + (page_num)*GM_PAGE_LEN);
            if (page_dp == page_dp_alt)
               GM_PRINT (GM_PRINT_LEVEL >= 1,
                         ("2 ways of dma_addr calc for page %d concur: 0x%p\n",
                           page_num, page_dp));
            else
            {
               GM_PRINT (GM_PRINT_LEVEL >= 1,
                         ("2 ways of dma_addr calc for page %d differ:\n",
	                  page_num));
               _GM_PRINT (GM_PRINT_LEVEL >= 1,
                          ("    pciio yields 0x%p, arithmetic yields 0x%p;\n",
	                   page_dp, page_dp_alt));
               _GM_PRINT (GM_PRINT_LEVEL >= 1, ("    USING PCIIO\n"));
            }
#endif
	 }

         if (register_page_func(arg, page_dp, page_num) != GM_SUCCESS)
         {
            err_code = GM_OUT_OF_MEMORY; /* out of contiguous mem */
            goto abort_with_active_map;
         }
      }
   }

   GM_ANNOUNCE_FNAME_RETURN(5, GM_SUCCESS);

abort_with_active_map:
#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.dma_pg_free_cnt += num_pages;
#endif
#if GM_IRIX_DEBUG_DMAMAPS
   dma_pages_active -= num_pages;
   GM_PRINT (1, ("There are now %d total DMA pages allocated\n",
	         dma_pages_active));
#endif
   pciio_dmamap_done(dma_map);

abort_with_map:
   gm_irix_dmamap_free(dma_map);

abort_with_memory:
   gm_irix_kmemfree((void *)kmemory, len); 

abort_with_nothing:
   r->map_active  = 0;

   GM_ANNOUNCE_FNAME_RETURN(1, err_code);

} /* gm_arch_dma_region_alloc() */

#undef GM_FNAME

 
#define GM_FNAME gm_arch_dma_region_dma_addr

gm_dp_t
GM_FNAME(gm_arch_dma_region_t *r)
{
   GM_PRINT (GM_PRINT_LEVEL >= 5, (GM_FNAME_STR " entered with %p\n", r));

   return gm_irix_region_dma_addr(r, 0);
}

#undef GM_FNAME




#define GM_FNAME gm_arch_dma_region_dma_addr_advance

gm_dp_t
GM_FNAME(gm_arch_dma_region_t *r)
{
   GM_PRINT (GM_PRINT_LEVEL >= 5, (GM_FNAME_STR " entered with %p\n", r));

   return gm_irix_region_dma_addr(r, 1);
}

#undef GM_FNAME




/* Return the DMA address of the current page in the DMA region. */

GM_STATIC gm_dp_t
gm_irix_region_dma_addr(gm_arch_dma_region_t *r, int advance_flag)
{
   gm_dp_t ret;
#if GM_IRIX_DEBUG_DMAMAPS
   gm_dp_t ret2;
#endif

   if (r->cur_page >= r->pages)
   {
      if (r->cur_page == r->pages)
      {
         GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
                   ("gm_arch_dma_region_dma_addr%s() called for page %d of"
                      " only %d pages\n",
		      (advance_flag) ? "_advance" : "",
                      r->cur_page, r->pages                                ));
      }
      else /* r->cur_page > r->pages */
      {
         GM_WARN(("gm_arch_dma_region_dma_addr%s() called for page %d of"
                  " only %d pages\n",
		  (advance_flag) ? "_advance" : "",
                  r->cur_page, r->pages                                   ));
#if 1
         gm_irix_programmed_panic("I want to see the traceback\n");
#endif
      }
      return ((gm_dp_t)0);
   }

   ASSERT(GM_PAGE_ALIGNED(r->dma_addr_p0));
   if (r->cur_page == 0)
      ret = r->dma_addr_p0;
   else
      ret =
         (gm_dp_t)(gm_irix_dmamap_addr(r->dmamap,
		                       r->kmem_addr +
                                           (r->cur_page)*GM_PAGE_LEN,
		                       0,
#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
                                       &(r->dma_template),
#endif
				       (r->cur_page < (r->pages - 1)) ?
		                             GM_PAGE_LEN : r->last_pg_len,
		                       (r->dma_asize == 64)));
#if GM_IRIX_DEBUG_DMAMAPS
   ret2 = (gm_dp_t)(r->dma_addr_p0 + (r->cur_page)*GM_PAGE_LEN);
   if (ret == ret2)
	GM_PRINT (GM_PRINT_LEVEL >= 1,
                ("2 methods of dma_addr calc for page %d concur: 0x%p\n",
                  r->cur_page, ret));
   else
   {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
                ("2 methods of dma_addr calculation for page %d differ:\n",
	         r->cur_page));
      _GM_PRINT (GM_PRINT_LEVEL >= 1,
                 ("    method 1 yields 0x%p, method 2 yields 0x%p;\n",
	          ret, ret2));
      _GM_PRINT (GM_PRINT_LEVEL >= 1, ("    USING METHOD 1\n"));
   }
#endif
   GM_PRINT (GM_PRINT_LEVEL >= 7,
             ("Page: number=%d, dma_addr=0x%x\n", r->cur_page, ret));

   if (advance_flag)
      r->cur_page++;

   return ret;
}

 
/* Return the kernel virtual address for a DMA region */

#define GM_FNAME gm_arch_dma_region_kernel_addr

void *
GM_FNAME(gm_arch_dma_region_t *r)
{
#if GM_IRIX_DEBUG_DMAMAPS

   static unsigned int rep_cnt  = 0;
   static paddr_t      rep_addr = (paddr_t)0;

/* NOTE:  The following algorithm reports the number of repetitive calls for
 *        a given region *only* on the subsequent call which establishes the
 *        end of such a sequence.
 */

   if (rep_addr == r->kmem_addr)

      rep_cnt++;		/* silently tally repeated calls */

   else                         /* it's a new kmem address */
   {
      if (rep_cnt > 1)
         GM_PRINT (1,
                (GM_FNAME_STR ": %d repetitions of preceeding\n", rep_cnt-1));

      rep_cnt  = 1;
      rep_addr = r->kmem_addr;

      GM_PRINT (1, (GM_FNAME_STR " returning kernel memory address\n"));
      _GM_PRINT (1,
                 ("    0x%x for region descriptor at %p\n", r->kmem_addr, r));
   }
#endif

   return (void *)(r->kmem_addr);
}

#undef GM_FNAME

/************
 * Required for non cache-coherent I/O architectures.  IRIX 6.5 is supposed
 * to make this functionality unnecessary.  We put a GM_NOTE here to remind
 * ourselves that we should verify these suppositions absolutely.
 *
 * Curiously, GM makes only one call to this function, and does *NOT*
 * check its return status.
 *
 * NOTE TO MYSELF: ////// Add a field to gm_irix_mld_struct to tally and
 *                        report the number of calls to this function.
 ************/

int
gm_arch_dma_region_sync(gm_arch_dma_region_t *r, int command)
{
   static int printed = 0;

   if (GM_IRIX_DEBUG_DMAMAPS)
      GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
                ("gm_arch_dma_region_sync() called with %p, %d (0x%x)\n",
                 r, command, command));
   else if (!printed)
   {
      GM_NOTE (("gm_arch_dma_region_sync() called with %p, %d (0x%x)\n",
                r, command, command));
      _GM_NOTE (("- (printing only once)\n"));
      printed = 1;
   }
   
   return 0;
}




/* Return the STS bits for a DMA region. */

gm_s32_t
gm_arch_dma_region_status(gm_arch_dma_region_t *r)
{
   GM_PARAMETER_MAY_BE_UNUSED (r);
   return GM_STS_BITS; 
}

 
#define GM_FNAME gm_arch_dma_region_free

void
GM_FNAME(gm_arch_dma_region_t *r)
{
   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS, (GM_FNAME_STR " entered with %p\n", r));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.dma_free_cnt++;	/* tally _calls_, not _successes_ */
#endif

   if (r->dmamap)
   {
      if (r->map_active)
      {
#if GM_IRIX_DEBUG_MEMLEAKS
         gm_irix_mld_struct.dma_pg_free_cnt += r->pages;
#endif
#if GM_IRIX_DEBUG_DMAMAPS
         dma_pages_active -= r->pages;
         GM_PRINT (1, ("There are now %d total DMA pages allocated\n",
	               dma_pages_active));
#endif
         pciio_dmamap_done(r->dmamap);
      }
      gm_irix_dmamap_free(r->dmamap);
   }
   
   if (r->kmem_addr)
      gm_irix_kmemfree((void *)(r->kmem_addr), r->region_len);
 
   bzero((void *)r, sizeof(*r));

   GM_ANNOUNCE_FNAME_EXIT(5);
}

#undef GM_FNAME

/**********************************************************************
 * Kernel memory management
 **********************************************************************/

/************
 * Kernel Memory Allocation for GM's use
 ************/

/* gm_arch_kernel_malloc()   (malloc() analogue for the driver)
 *
 * Input parameters:
 *
 *    len    length of memory to allocate
 *    flags  [ignored; see note below]
 *
 * Function return:
 *
 *    if successful, a pointer to the buffer we allocated for the caller;
 *    if failure, the null pointer.
 *
 * Notes:
 *
 *    We allocate eight more bytes than the user requested.  The extra eight
 *    bytes is a header, hidden at the beginning of the allocated area; the
 *    pointer we return to the user locates the byte following the header.
 *    The header is used to store the area's length, for the eventual
 *    kmem_free() call (see gm_arch_kernel_free()).
 *
 *    We don't use zalloc, because we shouldn't preempt the caller's
 *    prerogative to decide whether or not to zero its own memory allocations.
 *
 *    The buffer and header are aligned on an eight-byte boundary.
 *    ////// Does IRIX guarantee alignment???  (Solaris does.) //////
 *
 *    Although the flags parameter is declared in gm_impl.h, and there
 *    is a flag value defined there as well, in fact, all extant calls
 *    to this function pass 0 as the actual flags parameter.  Further,
 *    the defined flag value is GM_KERNEL_MALLOC_PAGEABLE; but in IRIX,
 *    all kernel memory (xkphys and xkseg) is guaranteed non-pageable
 *    (assuming that I'm reading the SGI docs correctly).
 *
 *    That's why the code below ignores the 'flags' parameter.
 */

void *
gm_arch_kernel_malloc(unsigned long len, int flags)
{
   gm_u32_t *p;
   GM_PARAMETER_MAY_BE_UNUSED (flags);
   
#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.kernel_alloc_cnt++; /* tally _calls_, not _successes_  */
#endif

   p = (gm_u32_t *)gm_irix_kmemalloc(len+8, KM_NOSLEEP);

   if (!p)
      return p;   /* alloc failed; return NULL     */

   p[0] = (gm_u32_t)len+8;  /* Save total length in header   */
   return &p[2];            /* Caller doesn't see the header */
}


 
/* gm_arch_kernel_free()   (free() analogue for the driver)
 *
 * Input parameter:
 *
 *    ptr    pointer to caller's part of allocated area
 *
 * Notes:
 *
 *    We free the eight-byte header along with the caller's part of the
 *    buffer.
 *
 *    The IRIX kernel will object if the address and length passed to
 *    kmem_free() don't exactly match a previously kmem_alloc()'d area.
 */

void
gm_arch_kernel_free(void *ptr)
{
   gm_u32_t *p;


#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.kernel_free_cnt++;
#endif

   p = ptr;
   gm_irix_kmemfree(&p[-2], p[-2]);  /* using the length stored in header */
}

 
/* generic memory allocations */

/* These functions are based on an idea in the linux gm_arch.c.
 * By using these functions as the sole interface to the kmem_ etc.
 * system kernel functions, we (1) can instrument the program for debugging,
 * and (2) only have one instance of the system calls to get right.
 * For additional possibilities, see the linux gm_arch.c.
 *
 * The 'flags' parameter is intended for the flags defined in <sys/kmem.h>
 * as parameters to kmem_alloc() and kmem_zalloc().  Here they are:
 *
 *   KM_SLEEP           0
 *   KM_NOSLEEP         0x0001          * must match VM_NOSLEEP     *
 *   KM_PHYSCONTIG      0x8000          * physically contiguous     *
 *   KM_CACHEALIGN      0x0800          * guarantee that memory is  *
 *                                      *    cache aligned          *
 *   KM_NODESPECIFIC    0x1000000       * Allocate on specific node *
 */

#define GM_FNAME gm_irix_kmemalloc

void *
GM_FNAME(size_t memsize, int flags)
{
#if GM_IRIX_DEBUG_MEMLEAKS

   void * memaddr;


   gm_irix_mld_struct.kmalloc_cnt++;

   memaddr = kmem_alloc(memsize, flags);
   GM_PRINT (GM_IRIX_DEBUG_MEMORY,
             (GM_FNAME_STR ": 0x%x bytes at address 0x%x, flags 0x%x\n",
              memsize, memaddr, flags));
   return (memaddr);

#else

   GM_PRINT (GM_IRIX_DEBUG_MEMORY,
             (GM_FNAME_STR " allocating 0x%x bytes with flags 0x%x\n",
              memsize, flags));
   return kmem_alloc(memsize, flags);

#endif
}

#undef GM_FNAME

 
#define GM_FNAME gm_irix_kmemzalloc

void *
GM_FNAME(size_t memsize, int flags)
{
#if GM_IRIX_DEBUG_MEMLEAKS

   void * memaddr;

   gm_irix_mld_struct.kmalloc_cnt++;

   memaddr = kmem_zalloc(memsize, flags);
   GM_PRINT (GM_IRIX_DEBUG_MEMORY,
             (GM_FNAME_STR ": 0x%x bytes at address 0x%x, flags 0x%x\n",
              memsize, memaddr, flags));
   return (memaddr);

#else

   GM_PRINT (GM_IRIX_DEBUG_MEMORY,
             (GM_FNAME_STR " allocating 0x%x bytes with flags 0x%x\n",
              memsize, flags));
   return kmem_zalloc(memsize, flags);

#endif
}

#undef GM_FNAME




#define GM_FNAME gm_irix_kmemfree

void
GM_FNAME(void *mem, size_t memsize)
{

   GM_PRINT (GM_IRIX_DEBUG_MEMORY,
             (GM_FNAME_STR " freeing 0x%x bytes at 0x%x\n", memsize, mem));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.kmfree_cnt++;
#endif

   kmem_free(mem, memsize);
}

#undef GM_FNAME

/************
 * GM synchronization functions
 ************/

void
gm_arch_sync_init(gm_arch_sync_t      *ss,  /* our synch structure */
                  gm_instance_state_t *is)  /* our instance stuff  */
/*
 * Description: initializes a gm_arch_sync_t structure
 */
{
#if !GM_DEBUG
   GM_PARAMETER_MAY_BE_UNUSED (is);      /* unused if not GM_DEBUG */
#endif

   GM_PRINT (GM_IRIX_DEBUG_SYNC,
             ("gm_arch_sync_init entered with %p, %p\n", ss, is));
 
   gm_arch_sync_reset(ss);

   MUTEX_INIT(&ss->simple_mu, MUTEX_DEFAULT, "GM simple mutex"      );
   MUTEX_INIT(&ss->outer_mu,  MUTEX_DEFAULT, "GM outer synch mutex" );
   MUTEX_INIT(&ss->inner_mu,  MUTEX_DEFAULT, "GM inner synch mutex" );
   sv_init   (&ss->sv,        SV_DEFAULT,    "GM synch var"         );

#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(ss, "at exit from gm_arch_sync_init()");
#endif
}



void
gm_arch_sync_reset(gm_arch_sync_t *s)
/*
 * Description: resets the control variables in a gm_arch_sync_t
 *              structure (e.g., when closing a port-state)
 *
 * ////////     Do we need to use something like atomic_set() here???
 */
{
   GM_PRINT (GM_IRIX_DEBUG_SYNC,
             ("gm_arch_sync_reset entered with %p\n", s));
#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at entry to gm_arch_sync_reset()");
#endif

   s->wake_cnt         = 0;
   s->sim_mu_entry_cnt = 0;
   s->sim_mu_exit_cnt  = 0;
   strcpy(s->usage, "");

#if GM_IRIX_DEBUG_WAKE
   s->status = GM_ARCH_SYNC_STATUS_INITIAL;
#endif
}


#define GM_FNAME gm_arch_sync_destroy

void
GM_FNAME(gm_arch_sync_t *s)
/*
 * Description: de-initializes a gm_arch_sync_t structure; this is the
 *              inverse of gm_arch_sync_init()
 */
{
   GM_PRINT (GM_IRIX_DEBUG_SYNC, (GM_FNAME_STR " entered with %p\n", s));

#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at entry to gm_arch_sync_destroy()");
#endif

   sv_destroy(&s->sv);
   MUTEX_DESTROY(&s->inner_mu);
   MUTEX_DESTROY(&s->outer_mu);
   MUTEX_DESTROY(&s->simple_mu);
}

#undef GM_FNAME

 
void
gm_arch_mutex_enter(gm_arch_sync_t *s)
/*
 * Description: claims a mutex lock
 */
{
   GM_PRINT (GM_IRIX_DEBUG_MUTEX,
             ("gm_arch_mutex_enter entered with %p\n", s));
 
#if GM_IRIX_DEBUG_MUTEX
   gm_irix_debug_sync(s, "at entry to gm_arch_mutex_enter()");
#endif

   MUTEX_LOCK(&s->simple_mu, GM_IRIX_MUTEX_PRIO);

   s->sim_mu_entry_cnt++;  /* This assignment may not be bulletproof */
#if GM_IRIX_DEBUG_MUTEX
   gm_irix_debug_sync(s, "at exit from gm_arch_mutex_enter()");
#endif
}



void
gm_arch_mutex_exit(gm_arch_sync_t *s)
/*
 * Description: releases a mutex lock
 */
{
   GM_PRINT (GM_IRIX_DEBUG_MUTEX,
             ("gm_arch_mutex_exit entered with %p\n", s)); 

#if GM_IRIX_DEBUG_MUTEX
   gm_irix_debug_sync(s, "at entry to gm_arch_mutex_exit()");
   delay((long)drv_usectohz(800000)); /* Wait 0.8 secs for printouts to complete */
#endif

   s->sim_mu_exit_cnt++;  /* This assignment may not be bulletproof */
   MUTEX_UNLOCK(&s->simple_mu);
}

 
/* Sleep functions.  Return a reason code. */

gm_arch_sleep_status_t
gm_arch_signal_sleep(gm_arch_sync_t *s)
/*
 * Description: Sleep until awakened or signalled; return a reason code
 *
 * NOTE: The lock that we hold on entry to SV_WAIT_SIG() is released
 *       inside that function before it begins to wait.
 */
{
   gm_arch_sleep_status_t ret_code;
   int                    wait_ret;
#if GM_IRIX_DEBUG_WAKE || GM_DEBUG
   int                    ret_cnt;            /* for debug-trace only */
#endif


   GM_PRINT (GM_PRINT_LEVEL >= 5,
             ("gm_arch_signal_sleep() entered with %p\n", s));

#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at entry to gm_arch_signal_sleep()");
#endif

   /* ensure only one sleeper at a time */
   MUTEX_LOCK(&s->outer_mu, GM_IRIX_MUTEX_PRIO);

   /* protect the wake_cnt variable     */
   MUTEX_LOCK(&s->inner_mu, GM_IRIX_MUTEX_PRIO);

   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("gm_arch_signal_sleep() entered mutex with wake_cnt %d\n",
              s->wake_cnt));

   if (s->wake_cnt > 0)             /* the wake got here first           */
   {
      ret_code=GM_SLEEP_WOKE;
#if GM_IRIX_DEBUG_WAKE
      s->status = GM_ARCH_SYNC_STATUS_WAKED_B4_WAIT;
#endif
      goto wakened;
   }

   /* wake_cnt was <= 0 at entry to inner_mu, so we must sleep */
   GM_PRINT (GM_PRINT_LEVEL >= 7, ("Calling SV_WAIT_SIG(%p,%p,0)\n",
                                   &(s->sv), &(s->inner_mu)));

   wait_ret = SV_WAIT_SIG(&s->sv, &s->inner_mu, 0);

   /* Either sv has been signaled or we have been interrupted; in either
      case, IRIX has released inner_mu.                                  */
   if (wait_ret)
   {
      ret_code = GM_SLEEP_WOKE;
#if GM_IRIX_DEBUG_WAKE
      s->status = GM_ARCH_SYNC_STATUS_WAKED_BY_SIGNAL;
#endif
   }

   else
   {
      ret_code = GM_SLEEP_INTERRUPTED;
#if GM_IRIX_DEBUG_WAKE
      ret_cnt = s->wake_cnt;
      s->status = GM_ARCH_SYNC_STATUS_WAKED_BY_INTERRUPT;
#endif
      goto interrupted;
   }

   MUTEX_LOCK(&s->inner_mu, GM_IRIX_MUTEX_PRIO); /* protect wake_cnt again  */

wakened:
   s->wake_cnt--;                   /* Account for leaving wait state */
#if GM_IRIX_DEBUG_WAKE
   ret_cnt = s->wake_cnt;
#endif
   MUTEX_UNLOCK(&s->inner_mu);      /* We're done with wake_cnt       */

interrupted:
   MUTEX_UNLOCK(&s->outer_mu);      /* It's now safe for another sleeper */

   if (ret_code == GM_SLEEP_INTERRUPTED)
      GM_PRINT (GM_PRINT_LEVEL >= 1,
                ("gm_arch_signal_sleep returning GM_SLEEP_INTERRUPTED\n"));
   else
      GM_PRINT (GM_PRINT_LEVEL >= 5,
                ("gm_arch_signal_sleep returning GM_SLEEP_WOKE\n"));

   GM_PRINT (GM_IRIX_DEBUG_WAKE, ("    - wake_cnt = %d\n", ret_cnt));
         
#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at exit from gm_arch_signal_sleep()");
#endif

   return(ret_code);
}

 
gm_arch_sleep_status_t
gm_arch_timed_sleep(gm_arch_sync_t *s, int seconds)
/*
 * Description: Wait until a mutex becomes available or a specified
 *              interval has elapsed, whichever comes first; return a code
 *              to the caller to indicate which of the two occurred
 *
 * NOTE: Per email (2, 3, and 9 March 1999) from Todd Pisek at SGI, as
 *       further interpreted by Glenn and me, the function description
 *       for sv_timedwait() is :
 *
 *       void sv_timedwait(sv_t       *syncvar,
 *                         int        flags,
 *                         void       *lock,
 *                         int        cookie,
 *                         int        fast,
 *                         timespec_t *wait_time,
 *                         timespec_t *time_left);
 *       
 *       where the parameter descriptions are:
 *
 *         syncvar   sync variable on which to queue
 *         flags     flags to pass to queueing logic
 *         lock      lock to be released (could be lock_t or mutex_t)
 *         cookie    0 if lock is mutex_t; return val from LOCK_ENTER if lock_t
 *         fast      TRUE => "fast time", FALSE => "normal HZ"
 *         wait_time amount of time to wait
 *         time_left unexpired time remaining after signal received
 *
 * NOTE: The lock that we hold on entry to sv_timedwait() is released
 *       inside that function before it begins to wait.
 */
{
   timespec_t             time_to_wait,
                          time_remaining;
   gm_arch_sleep_status_t ret_code;
#if GM_IRIX_DEBUG_WAKE || GM_DEBUG
   int                    ret_cnt;            /* for debug-trace only */
#endif


   GM_PRINT (GM_PRINT_LEVEL >= 5,
             ("gm_arch_timed_sleep() entered with %p, %d\n", s, seconds));

#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at entry to gm_arch_timed_sleep()");
#endif

#if 1
/* maxstern HACK for when there is lots of LANAI debugging going on */
   time_to_wait.tv_sec    = seconds * 3;
#else
   time_to_wait.tv_sec    = seconds;
#endif
   time_to_wait.tv_nsec   = 0;
   time_remaining.tv_sec  = 0;
   time_remaining.tv_nsec = 0;

   /* ensure only one sleeper at a time */
   MUTEX_LOCK(&s->outer_mu, GM_IRIX_MUTEX_PRIO);

   /* protect the wake_cnt variable     */
   MUTEX_LOCK(&s->inner_mu, GM_IRIX_MUTEX_PRIO);

   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("gm_arch_timed_sleep() entered mutex with wake_cnt %d\n",
              s->wake_cnt));

   if (s->wake_cnt > 0)             /* the wake got here first           */
   {
      ret_code=GM_SLEEP_WOKE;
#if GM_IRIX_DEBUG_WAKE
      s->status = GM_ARCH_SYNC_STATUS_WAKED_B4_WAIT;
#endif
      goto wakened;
   }

   /* wake_cnt was <= 0 at entry to inner_mu, so we must sleep */
#if GM_IRIX_DEBUG_WAKE
   GM_PRINT (1, ("About to call sv_timedwait(%p,0,%p,0,0,%d,0)\n",
                 &(s->sv), &(s->inner_mu), time_to_wait.tv_sec));
#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "just before sv_timedwait()");
#else
   gm_irix_decode_sync_status(s->status);
#endif
#endif
   sv_timedwait(&(s->sv),
                0,                /* don't need any flags       */
                &(s->inner_mu),
                0,                /* 0 indicates mutex lock     */
                0,                /* don't think we want "fast" */
                &time_to_wait,
                &time_remaining);

   /* Either sv has been signaled or we have timed out; in either case,
      IRIX has released inner_mu.                                       */
   if ( ( time_remaining.tv_sec > 0                                  ) ||
        ((time_remaining.tv_sec == 0) && (time_remaining.tv_nsec > 0))    )
   {
      ret_code = GM_SLEEP_WOKE;
#if GM_IRIX_DEBUG_WAKE
      s->status = GM_ARCH_SYNC_STATUS_WAKED_BY_SIGNAL;
#endif
   }

   else
   {
      ret_code = GM_SLEEP_TIMED_OUT;
#if GM_IRIX_DEBUG_WAKE
      ret_cnt = s->wake_cnt;
      s->status = GM_ARCH_SYNC_STATUS_WAKED_BY_TIMEOUT;
#endif
      goto timed_out;
   }

   MUTEX_LOCK(&s->inner_mu, GM_IRIX_MUTEX_PRIO); /* protect wake_cnt again  */

wakened:
   s->wake_cnt--;                   /* Account for leaving wait state */
#if GM_IRIX_DEBUG_WAKE
   ret_cnt = s->wake_cnt;
#endif
   MUTEX_UNLOCK(&s->inner_mu);      /* We're done with wake_cnt       */

timed_out:
   MUTEX_UNLOCK(&s->outer_mu);      /* It's now safe for another sleeper */

   if (ret_code == GM_SLEEP_TIMED_OUT)
      {
         GM_PRINT (1,
                ("gm_arch_timed_sleep returning GM_SLEEP_TIMED_OUT time=%d\n",
			time_to_wait.tv_sec));
      }
   else
      {
	GM_PRINT (GM_PRINT_LEVEL >= 5,
                  ("gm_arch_timed_sleep returning GM_SLEEP_WOKE\n"));
      }

   _GM_PRINT (GM_IRIX_DEBUG_WAKE,
              ("    - wake_cnt = %d, time_remaining = %d.%06d\n",
               ret_cnt, time_remaining.tv_sec, time_remaining.tv_nsec));
         
#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at exit from gm_arch_timed_sleep()");
#endif

   return(ret_code);
}

 
void
gm_arch_wake(gm_arch_sync_t *s)
/*
 * Description: Wake the thread sleeping on the synchronization variable
 *
 * NOTE: That thread entered the sleep state via either gm_arch_signal_sleep()
 *       or gm_arch_timed_sleep().
 */
{
/* Turn this on to try using sv_broadcast() instead of sv_signal() */
#define GM_IRIX_TRY_SV_BROADCAST 1

   int awakened_count;


   GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_arch_wake() entered with %p\n", s));

#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at entry to gm_arch_wake()");
#endif

   MUTEX_LOCK(&s->inner_mu, GM_IRIX_MUTEX_PRIO);
   GM_PRINT (GM_PRINT_LEVEL >= 6,
             ("gm_arch_wake() got mutex; wake_cnt = %d++\n", s->wake_cnt));

   s->wake_cnt++;
#if GM_IRIX_TRY_SV_BROADCAST
#if GM_IRIX_DEBUG_WAKE
   GM_PRINT (1, ("About to call sv_broadcast\n"));
   gm_irix_decode_sync_status(s->status);
#endif
   awakened_count = sv_broadcast(&s->sv);
   if (awakened_count != 1)
      GM_WARN(("sv_broadcast() woke %d processes; 1 process was expected\n",
               awakened_count));
#else
#if GM_IRIX_DEBUG_WAKE
   GM_PRINT (1, ("About to call sv_signal\n"));
   gm_irix_decode_sync_status(s->status);
#endif
   awakened_count = sv_signal(&s->sv);
   if (awakened_count != 1)
      GM_WARN(("sv_signal() returned %d; 1 was expected\n", awakened_count));
#endif

   MUTEX_UNLOCK(&s->inner_mu);

   GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_arch_wake() returning\n"));

#if GM_IRIX_DEBUG_SYNC
   gm_irix_debug_sync(s, "at exit from gm_arch_wake()");
#endif
}




void
gm_arch_spin(gm_instance_state_t *is, gm_u32_t usecs)
/*
 * Description: Busywait for the indicated duration
 */
{
   GM_PRINT (GM_PRINT_LEVEL >= 9,
             ("gm_arch_spin entered with %p, %d\n", is, usecs));

#if GM_IRIX_COMPLICATED_SPIN  /* ///// What to do??? ///// */
/* This was copied from Solaris.  I don't know whether it's applicable
 * and/or needed in IRIX.  Keeping it here as a placeholder.
 */
   long timeout_time;


   drv_getparm(LBOLT, &timeout_time);
   timeout_time += (long)drv_usectohz(usecs);

   MUTEX_LOCK(&is->arch.spin_sync.inner_mu, GM_IRIX_MUTEX_PRIO);
   sv_timedwait_sig(&is->arch.spin_sync.sv, &is->arch.spin_sync.inner_mu,
                    timeout_time);
   MUTEX_UNLOCK(&is->arch.spin_sync.inner_mu);
#else
   GM_PARAMETER_MAY_BE_UNUSED (is);
   delay((long)drv_usectohz(usecs));

#endif
}

/**********************************************************************
 * Memory Registration support
 *
 * Memory registration is a technique of locking user memory pages
 * and "registering" their locations for increased efficiency.  Here's
 * an outline of how it works, with some intermediate functions omitted
 * for brevity (as of this writing, 2000/09/14):
 *
 * . The user process calls gm_dma_malloc(), which calls gm_register_memory();
 *   or the user process calls gm_register_memory() directly.
 *
 * . gm_register_memory() calls _gm_user_ioctl() which eventually results in
 *   a call to ioctl().
 *
 * . gm_irix:<pfx>ioctl() is entered and eventually calls gm_ioctl().
 *
 * . gm_ioctl() calls gm_arch_lock_user_buffer_page().
 *
 **********************************************************************/

#if GM_CAN_REGISTER_MEMORY

void
gm_irix_setup_memory_registration(void)
{
   char *usage_page_lock_hash = "page_lock_hash";


   gm_assert(gm_irix_page_lock_hash == (struct gm_hash *)NULL);
   gm_irix_page_lock_hash = gm_create_hash(gm_irix_page_lock_hash_compare,
				           gm_irix_page_lock_hash_hash,
				           sizeof(alenaddr_t),
				           sizeof(gm_irix_page_lock_hash_t),
				           16, 0);
   if (gm_irix_page_lock_hash == (struct gm_hash *)NULL)
   {
      GM_WARN(("Could not allocate page_lock hash table\n"));
      gm_always_assert(0);
   }
   GM_PRINT(GM_DEBUG_MEM_REGISTER,
            ("Allocated page_lock hash table at 0x%p\n",
             gm_irix_page_lock_hash));
   gm_irix_page_lock_hash_sync = (gm_arch_sync_t *)gm_create_mutex();
   if (!gm_irix_page_lock_hash_sync)
      GM_PANIC (("Couldn't allocate page_lock_hash sync\n"));

   if (strlen(usage_page_lock_hash) < GM_IRIX_SYNC_USAGE_LEN)
      strcpy(gm_irix_page_lock_hash_sync->usage, usage_page_lock_hash);
}



void
gm_irix_teardown_mem_registration(void)
{
   if (gm_irix_page_lock_hash)
   {
      gm_destroy_hash(gm_irix_page_lock_hash);
      gm_irix_page_lock_hash = NULL;
   }

   if (gm_irix_page_lock_hash_sync)
      gm_destroy_mutex((struct gm_mutex *)gm_irix_page_lock_hash_sync);
}



/*  ///////////////////////////////////////////////////////////////
 *  //                                                           //
 *  // Once we have achieved full implementation of this driver  //
 *  // on SMP machines, we should consider adding the suggested  //
 *  // dki_dcache_inval() and dki_dcache_wb() calls, for use     //
 *  // ONLY on uniprocessor machines.  We must take care to      //
 *  // avoid the overhead of kernel-mode switching if and when   //
 *  // we do this.  See the SGI Device Driver Programmer's       //
 *  // Guide for further information.                            //
 *  //                                                           //
 *  ///////////////////////////////////////////////////////////////
 */

 
#define GM_FNAME gm_arch_lock_user_buffer_page

gm_status_t
GM_FNAME(gm_instance_state_t *is,
         gm_up_t             uaddr,
         gm_dp_t             *dma_addr,
         gm_arch_page_lock_t *lock)
/*
 * Description: "locks down" a user buffer page and prepares it for DMA use.
 *
 *      Upon successful exit from this function, the page of memory at the
 *      specified UVA has been locked down, and DMA to this page, using the
 *      address returned at dma_addr, has been enabled.
 *
 * Input Parameters:  is       is the instance state
 *                    uaddr    is the user VMA of the page
 *                    lock     is the structure identifying the locked status
 *                             of this page
 *
 * Output Parameters: dma_addr is the DMA address that the LANai will use
 *                    lock     has been updated with pertinent info about this
 *                             locked page
 *
 * Assumptions: * uaddr is page-aligned.
 *              * The job of this function is to lock exactly one page.
 *
 * Notes: This implementation of memory registration support is based on
 *        numerous communications between me (Max) and SGI, culminating in
 *        emails between me and Tony Vollem et al., dated from September 22
 *        to October 12, 2000.  Following this, the algorithm was further
 *        refined to allow multiple pinning of the same user memory page.
 *
 *        Two methods for pinning user memory were discussed: fast_userdma()
 *        and fast_undma() on the one hand, and kern_mpin() and kern_munpin()
 *        on the other.  We first chose the kern_mpin() approach, because it
 *        is used in the successful SGI product "GSN".  However, we later
 *        were requested by Tony Vollem to replace it with the fast_userdma()
 *        method.  On October 12, Rob Warnock of SGI supplied this rationale:
 *
 *           Besides "kern_mpin()" *not* being a "documented" interface
 *           [whereas "fast_userdma()", though not in the current Irix Driver
 *           Writer's Guide on our web page, *should* be there], there's also
 *           a serious security problem with it, namely, "kern_mpin()" is
 *           *exactly* the same as a user "mpin()" call, except done by the
 *           kernel instead of the user.  This means that if a driver does a
 *           "kern_mpin()" for async DMA and then returns to user mode, then
 *           nothing prevents the user program from doing an "munpin()" call
 *           that unpins the pages while the DMA is still active! (*Yikes!!*)
 *           Likewise, if the process should exit for any reason, the usual
 *           automatic "munpin()" is done, too. (More *Yikes!!*)
 *
 *           On the other hand, pages pinned by "userdma()/fast_userdma()",
 *           *remain* pinned until a matching "undma()/fast_undma()" call,
 *           regardless.
 *
 *        fast_userdma() and fast_undma() are not documented anywhere in
 *        publicly-available SGI documentation -- not even man pages.  So we
 *        are depending on back-channel communications with SGI engineers for
 *        all information beyond the function declarations in <sys/buf.h>.
 *        OTOH, Rob says that I could use the equivalent userdma() and undma()
 *        functions, which _are_ documented, and which are simple wrappers
 *        around the fast_*dma() equivalents.  HOWEVER -- the documented
 *        functions do not implement the cookies, so this is questionable.
 */
{
   int            ret_val;
   gm_status_t    hash_status;
   pciio_dmamap_t dma_map;      /* pointer to dmamap */
   alenaddr_t     page_phys_addr = 0;
   size_t         page_phys_len;
   alenlist_t     phys_addr_list;
   int            fast_cookie;
   int            fast_flags = B_READ | B_WRITE | B_ASYNC | B_PHYS;
   iopaddr_t      page_dma_addr;
   ulong_t        ulong_pid;
   pid_t          user_pid;
   gm_irix_page_lock_hash_t *lock_hash,
                            new_lock_hash;
   gm_irix_page_info_t      *page_info=NULL;
   gm_irix_fastdma_info_t   *fastdma_info=NULL;
   
/* Next three fields copied out of lock hash to avoid mutex deadlocks */
   int            ref_count;
#if GM_DEBUG
/* if'd to avoid compiler diagnostics when not GM_DEBUG               */
   int            pin_count,
	          unpin_count;
#endif
   

   GM_PRINT (GM_DEBUG_MEM_REGISTER,
             (GM_FNAME_STR " entered with %p, %p,\n", is, uaddr));
   _GM_PRINT (GM_DEBUG_MEM_REGISTER, ("   %p, %p\n", dma_addr, lock));

   gm_assert(GM_PAGE_LEN);
   gm_assert(GM_PAGE_ALIGNED(uaddr));
   gm_assert(sizeof(caddr_t) == sizeof(uaddr));

/* Determine the user context PID */
   ret_val = drv_getparm(PPID, &ulong_pid);
   user_pid = (pid_t)ulong_pid;
   gm_always_assert(ret_val == 0);

/* Pin down the user VM page */
   ret_val = fast_userdma((void *)uaddr,
                          GM_PAGE_LEN,
                          fast_flags,
                          &fast_cookie);
   if (ret_val)
   {
      GM_WARN((GM_FNAME_STR ": fast_userdma() returned %d\n", ret_val));
      goto abort_with_nothing;
   }

/* Convert UVA to physical addr., in the form of an alenlist  */
   phys_addr_list = uvaddr_to_alenlist(NULL, (uvaddr_t)uaddr, GM_PAGE_LEN, 0);
   if (phys_addr_list == NULL)
   {
      GM_WARN((GM_FNAME_STR ": uvaddr_to_alenlist returned NULL\n"));
      goto abort_with_fast_dma;
   }

/* Extract the physical addr from the alenlist  */
   ret_val = alenlist_get(phys_addr_list, NULL, 0,
                          &page_phys_addr, &page_phys_len, 0);
   if (ret_val != ALENLIST_SUCCESS)
   {
      GM_WARN(("alenlist_get() returned %d for page at UVA 0x%x\n",
               ret_val, uaddr));
      goto abort_with_phys_addr_list;
   }
   if (!page_phys_addr)
   {
      GM_WARN(("alenlist_get() reported success, but returned NULL\n"));
      _GM_WARN(("   DMA address for page at UVA 0x%x\n", uaddr));
      goto abort_with_phys_addr_list;
   }
   gm_always_assert(page_phys_len == GM_PAGE_LEN);

/* Allocate a DMA map object  */
   dma_map = gm_irix_dmamap_alloc(is->arch.conn,
                                  GM_PAGE_LEN,
				  1, /* prefetch necessary ??? ///// */
                                  GM_IRIX_INST_USE_A64DMA(is));
   if (!dma_map)
   {
      GM_WARN((GM_FNAME_STR ": could not allocate DMA map\n"));
      goto abort_with_phys_addr_list;
   }

/* Extract the DMA addr from the map, given the physaddr  */
   GM_PRINT ((GM_IRIX_DEBUG_DMAMAPS || GM_DEBUG_MEM_REGISTER),
             ("Calling pciio_dmamap_addr(%p, %p, 0x%x)\n",
	      dma_map, page_phys_addr, GM_PAGE_LEN));
   page_dma_addr = pciio_dmamap_addr(dma_map, page_phys_addr, GM_PAGE_LEN);
   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
             ("pciio_dmamap_addr for %d bytes at physaddr 0x%x\n",
              GM_PAGE_LEN, page_phys_addr));
   _GM_PRINT (GM_IRIX_DEBUG_DMAMAPS, 
              ("   (UVA %p) returned 0x%x\n", uaddr, page_dma_addr));
   if (!page_dma_addr)
   {
      GM_WARN(("pciio_dmamap_addr() failed for page at UVA 0x%x\n", uaddr));
      goto abort_with_map;
   }

   if ((!GM_IRIX_INST_USE_A64DMA(is)) && (sizeof(page_dma_addr) > 4))
      /* we only grok 32 bits */
      gm_always_assert((page_dma_addr & 0x0ffffffff00000000) == 0);

   if(!lock->page_info) {

      page_info = gm_irix_kmemzalloc(sizeof(*page_info), KM_NOSLEEP);
      if (!page_info)
      {
         GM_WARN(("Unable to allocate %d bytes of KMem\n",
                  sizeof(*page_info)));
         goto abort_with_map;
       }

       page_info->is         = is;
       page_info->uva        = uaddr;
       page_info->upid       = user_pid;
       page_info->cookie     = fast_cookie;
       page_info->fast_flags = fast_flags;
       page_info->physaddr   = page_phys_addr;
       page_info->map        = dma_map;
       page_info->dma_addr   = page_dma_addr;
       page_info->next       = (gm_irix_page_info_t *)NULL;
       page_info->prev       = (gm_irix_page_info_t *)NULL;
       page_info->lock       = ( void *)lock;
       lock->page_info = page_info;

   }

/* allocate argument for call back function and set it */

   fastdma_info = gm_irix_kmemzalloc(sizeof(gm_irix_fastdma_info_t),
                                     KM_NOSLEEP);
   if (!fastdma_info)
   {
       GM_WARN(("Unable to allocate %d bytes of KMem\n",
                sizeof(gm_irix_fastdma_info_t)));
       goto abort_with_kmemory1;
   }

   fastdma_info->uva        = uaddr;
   fastdma_info->cookie     = fast_cookie;
   fastdma_info->fast_flags = fast_flags;

   ret_val = add_exit_callback(user_pid, 
                               0, 
                               gm_irix_fast_dma_callback,
                               (void *)fastdma_info);
   gm_always_assert(ret_val == 0);

/* Record this pin in our hash table */
   gm_mutex_enter((struct gm_mutex *)gm_irix_page_lock_hash_sync);
   
   lock_hash = gm_hash_find(gm_irix_page_lock_hash, &page_phys_addr);
   if (!lock_hash)
   {
      new_lock_hash.pin_count   = 0;
      new_lock_hash.unpin_count = 0;
      new_lock_hash.ref_count   = 0;
      hash_status = gm_hash_insert(gm_irix_page_lock_hash,
                                   &page_phys_addr,
                                   &new_lock_hash);
      if (hash_status != GM_SUCCESS)
      {
         gm_mutex_exit((struct gm_mutex *)gm_irix_page_lock_hash_sync);
	 GM_NOTE(("gm_hash_insert(%p, %p, %p failed\n",
                  gm_irix_page_lock_hash, &page_phys_addr, &new_lock_hash));
	 goto abort_with_kmemory2;
      }
      lock_hash = gm_hash_find(gm_irix_page_lock_hash, &page_phys_addr);
      gm_assert(lock_hash);
   }
   ref_count = lock_hash->ref_count;
   lock_hash->pin_count++;
   lock_hash->ref_count++;

#if GM_DEBUG
   /* Copy the hash fields to avoid deadlock with print mutex */
   pin_count   = lock_hash->pin_count;
   unpin_count = lock_hash->unpin_count;
#endif

   gm_mutex_exit((struct gm_mutex *)gm_irix_page_lock_hash_sync);

   GM_PRINT(GM_DEBUG_MEM_REGISTER,
            ("Locked phys page %p, lock = %p len = %d ref_count = %d\n",
             page_phys_addr, lock, GM_PAGE_LEN,ref_count));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.dma_pg_alloc_cnt++; /* IMPROVE THIS */
   gm_irix_mld_struct.user_lock_cnt++;
#endif
#if GM_IRIX_DEBUG_DMAMAPS
   dma_pages_active++;		          /* IMPROVE THIS */
   GM_PRINT (1, ("There are now %d total DMA pages allocated\n",
	         dma_pages_active));
#endif

/* Don't need the alenlist any more */
   alenlist_destroy(phys_addr_list);

/* Set the return parameter */
   *dma_addr = (gm_dp_t)page_dma_addr;

  /* Report success */
   GM_PRINT (GM_DEBUG_MEM_REGISTER,
             (GM_FNAME_STR " locked user page at %p\n", page_info->uva));
   _GM_PRINT (GM_DEBUG_MEM_REGISTER,
              ("    with dma address 0x%x\n", page_dma_addr));

   GM_ANNOUNCE_FNAME_RETURN(5, GM_SUCCESS);

abort_with_kmemory2:
   ret_val = add_exit_callback(user_pid, 
                               ADDEXIT_REMOVE,
                               gm_irix_fast_dma_callback,
                               (void *)fastdma_info);
   gm_irix_kmemfree(fastdma_info, sizeof(gm_irix_fastdma_info_t));
abort_with_kmemory1:
   if(page_info)
      gm_irix_kmemfree(page_info, sizeof(gm_irix_page_info_t));
abort_with_map:
   gm_irix_dmamap_free(dma_map);
abort_with_phys_addr_list:
   alenlist_destroy(phys_addr_list);
abort_with_fast_dma:
   fast_undma((void *)uaddr, GM_PAGE_LEN, fast_flags, &fast_cookie);
abort_with_nothing:
   return(GM_FAILURE);

} /* gm_arch_lock_user_buffer_page() */

#undef GM_FNAME

 

#define GM_FNAME gm_arch_unlock_user_buffer_page

void
GM_FNAME(gm_arch_page_lock_t *thelock)
/*
 * Description: releases a registered user buffer page and all of its
 *              associated rexources.
 *
 * Input Parameters:  thelock  is the structure identifying the locked status
 *                             of this page
 */
{
   GM_PRINT (GM_DEBUG_MEM_REGISTER,
             (GM_FNAME_STR " entered with %p\n", thelock));

   gm_irix_unlock_page(thelock->page_info);
}

#undef GM_FNAME




#define GM_FNAME gm_irix_fast_dma_callback

void
GM_FNAME(void *the_parm)
{
   gm_irix_fastdma_info_t *fastdma_info=(gm_irix_fastdma_info_t *)the_parm;

   GM_PRINT(GM_DEBUG_MEM_REGISTER,
            (GM_FNAME_STR " entered with %p\n", the_parm));

   fast_undma((caddr_t)(fastdma_info->uva),
              GM_PAGE_LEN,
              fastdma_info->fast_flags,
              &(fastdma_info->cookie));

   gm_irix_kmemfree(fastdma_info, sizeof(gm_irix_fastdma_info_t));

}

#undef GM_FNAME

 

#define GM_FNAME gm_irix_unlock_page

GM_STATIC void
GM_FNAME(gm_irix_page_info_t *page_info)
/*
 * Description: releases a registered user buffer page and all of its
 *              associated rexources.
 *
 * Input Parameters:  page_info  is the structure containing all information
 *                               required to release this page.
 */
{
   gm_irix_page_lock_hash_t *lock_hash;
/* Next three fields copied out of lock hash to avoid mutex deadlocks */
   int            ref_count,ret_val;
   int            pin_count,
	          unpin_count;
   pid_t          user_pid;
   ulong_t        ulong_pid;


   GM_PRINT (GM_DEBUG_MEM_REGISTER,
             (GM_FNAME_STR " entered with %p\n", page_info));

   gm_assert(page_info);

   gm_mutex_enter((struct gm_mutex *)gm_irix_page_lock_hash_sync);
   lock_hash = gm_hash_find(gm_irix_page_lock_hash, &(page_info->physaddr));
   if (!lock_hash)
   {
      gm_mutex_exit((struct gm_mutex *)gm_irix_page_lock_hash_sync);
      GM_WARN((GM_FNAME_STR " called for phys page %p, not found in hash\n",
               page_info->physaddr));
      _GM_WARN(("... NOT UNPINNING THIS PAGE\n"));
      return;
   }

   ref_count = lock_hash->ref_count;
   lock_hash->unpin_count++;
   if (lock_hash->ref_count > 0)
      lock_hash->ref_count--;

   /* Copy the hash fields so we can free the mutex and avoid deadlocks */
   pin_count   = lock_hash->pin_count;
   unpin_count = lock_hash->unpin_count;

   gm_mutex_exit((struct gm_mutex *)gm_irix_page_lock_hash_sync);

   if (!(ref_count))
   {
      gm_assert (pin_count == unpin_count);

      GM_WARN(("Redundant unlock of phys page %p ignored\n",
               page_info->physaddr));
      return;
   }

   gm_forget_dma_addr (page_info->is->dma_page_bitmap, page_info->dma_addr);
   pciio_dmamap_done(page_info->map);
   gm_irix_dmamap_free(page_info->map);
 
   GM_PRINT(GM_DEBUG_MEM_REGISTER,
            ("Unlocked phys page %p, pins = %d, unpins = %d\n",
	     page_info->physaddr, pin_count, unpin_count));
   _GM_PRINT(GM_DEBUG_MEM_REGISTER,
	     ("UVA %p, user pid %d, flags 0x%x, cookie 0x%x\n",
	      page_info->uva, page_info->upid,
              page_info->fast_flags, page_info->cookie));
   _GM_PRINT(GM_DEBUG_MEM_REGISTER,
             ("There are now %d total pages locked\n", pages_locked));

   gm_irix_kmemfree(page_info, sizeof(*page_info));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.user_unlock_cnt++;
   gm_irix_mld_struct.dma_pg_free_cnt++;
#endif
#if GM_IRIX_DEBUG_DMAMAPS
   dma_pages_active--;
   GM_PRINT (1, ("There are now %d total DMA pages allocated\n",
	         dma_pages_active));
#endif
}

#undef GM_FNAME



GM_STATIC long
gm_irix_page_lock_hash_compare(void *a, void *b)
{
   GM_PRINT(GM_DEBUG_MEM_REGISTER,
            ("gm_irix_page_lock_hash_compare entered with ->%p, ->%p\n",
	     *((alenaddr_t *)a),
	     *((alenaddr_t *)b) ));

   return (*((alenaddr_t *)a) != *((alenaddr_t *)b));
}




GM_STATIC unsigned long
gm_irix_page_lock_hash_hash(void *a)
{
   GM_PRINT(GM_DEBUG_MEM_REGISTER,
            ("gm_irix_page_lock_hash_hash entered with %p -> %p\n",
             a, *((alenaddr_t *)a) ));

   return gm_crc((char *)a, sizeof(alenaddr_t));
}

#endif /* GM_CAN_REGISTER_MEMORY */

/**********************************************************************
 * Internal functions in support of DMA
 **********************************************************************/

/* Allocate a DMA map */

GM_STATIC pciio_dmamap_t
gm_irix_dmamap_alloc(vertex_hdl_t the_vhdl,
                     size_t       the_size,
                     gm_u32_t     prefetch,
                     gm_u32_t     dma_64_bit)
{
   int dma_alloc_flags;

   if (prefetch)
      dma_alloc_flags = PCIIO_DMA_DATA | GM_IRIX_ENDIANNESS;
   else
      dma_alloc_flags = PCIIO_DMA_CMD  | GM_IRIX_ENDIANNESS;

   if (dma_64_bit)
   {
      GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
                ("Allocating a 64-bit capable DMA map for minor %d\n",
	         the_vhdl));
      dma_alloc_flags |= PCIIO_DMA_A64;
   }
   else
      GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
                ("Allocating a 32-bit-specific DMA map for minor %d\n",
	         the_vhdl));

   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
             ("Calling pciio_dmamap_alloc(%d, dev_desc, 0x%x, 0x%x)\n",
	      the_vhdl /* , device desc */, the_size, dma_alloc_flags));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.dmamap_alloc_cnt++; /* tally _tries_, not _successes_ */
#endif

   return (pciio_dmamap_alloc(the_vhdl,
                              device_desc_default_get(the_vhdl),
                              the_size,
                              dma_alloc_flags));
}



/* Perform a DMA mapping */
/* NOTE: The use of kvtophys() to prepare the address for the
 * pciio_dmamap_addr() call was suggested by Wendy Nunn of SGI in email on
 * October 6, 1999.  Before I added this call, DMA was not working on the
 * Octane, although it had worked fine without it on the O200.
 */

GM_STATIC iopaddr_t
gm_irix_dmamap_addr(pciio_dmamap_t the_dmamap,
                    paddr_t        the_addr,
                    paddr_t        *the_physaddr, /* NULL means don't care */
#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
                    iopaddr_t      *the_template,
#endif
                    size_t         the_size,
                    gm_u32_t       dma_64_bit)
{
   iopaddr_t return_addr;
   paddr_t   physical_addr;


   physical_addr = kvtophys((void *)the_addr);

   if (gm_need_kvtophys_analysis)
   {
      if (physical_addr == the_addr)
	 GM_NOTE (("On this machine, kvtophys() appears to be a no-op\n"));
      else
	 GM_NOTE (("On this machine, kvtophys() converted\n"));
	 _GM_NOTE (("          kernel address 0x%p to physical address 0x%p\n",
		    the_addr, physical_addr));
      gm_need_kvtophys_analysis = 0;
   }

#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
   if (the_template)
   {
      return_addr = (iopaddr_t)physical_addr | *the_template;
      GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
                ("Calculated dma address 0x%x for %d bytes\n",
                 return_addr, the_size));
      _GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
                 ("... at kmem addr 0x%x\n", the_addr));
      _GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
                 (" (physaddr 0x%x)\n", physical_addr));
      goto exit_with_success;
   }
#endif

   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
             ("Calling pciio_dmamap_addr(%p, %p, 0x%x)\n",
	      the_dmamap, physical_addr, the_size));
   return_addr = pciio_dmamap_addr (the_dmamap, physical_addr, the_size);
   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
             ("pciio_dmamap_addr for %d bytes at kmem addr 0x%x\n",
              the_size, the_addr));
   _GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
     ("   (physaddr 0x%x) returned 0x%x\n", physical_addr, return_addr));

#if GM_IRIX_MINIMIZE_PCIMAP_ADDR_CALLS
exit_with_success:
#endif

   if ((!dma_64_bit) && (sizeof(return_addr) > 4))
      /* we only grok 32 bits */
      gm_always_assert((return_addr & 0x0ffffffff00000000) == 0);

   if (the_physaddr)
      *the_physaddr = physical_addr;

   return return_addr;

} /* gm_irix_dmamap_addr() */



/* Free a DMA map */

GM_STATIC void
gm_irix_dmamap_free(pciio_dmamap_t the_dma_map)
{
   GM_PRINT (GM_IRIX_DEBUG_DMAMAPS,
             ("Calling pciio_dmamap_free(0x%x)\n", the_dma_map));

#if GM_IRIX_DEBUG_MEMLEAKS
   gm_irix_mld_struct.dmamap_free_cnt++;
#endif

   pciio_dmamap_free(the_dma_map);
}

/**********************************************************************
 * Port-state support: initialization, open, close, and finalization
 **********************************************************************/

/* This is called just after the port state is created (in gm_minor.c)
   to perform architecture-specific initialization. */

#define GM_FNAME gm_arch_port_state_init

gm_status_t
GM_FNAME(gm_port_state_t *ps)
{
   GM_PARAMETER_MAY_BE_UNUSED (ps);        /* If not GM_DEBUG */
   GM_PRINT (GM_DEBUG_PORT_STATE,
             (GM_FNAME_STR " entered with %p for gm_minor %d\n",
              ps, ps->minor));

   return GM_SUCCESS;
}

#undef GM_FNAME




/* This is called just after the port state is initialized (in
   gm_port_state_open()) to perform architecture-specific initialization. */

#define GM_FNAME gm_arch_port_state_open

gm_status_t
GM_FNAME(gm_port_state_t *ps)
{
   GM_PARAMETER_MAY_BE_UNUSED (ps);        /* If not GM_DEBUG */
   gm_assert(ps->instance);
   GM_PRINT (GM_DEBUG_PORT_STATE, 
             (GM_FNAME_STR " entered for unit %d, port %d, minor %d\n",
              ps->instance->id, ps->id, ps->minor));

   return GM_SUCCESS;
}

#undef GM_FNAME

 
/* This is called just before the port state is finalized (in
   gm_port_state_close()) to perform architecture-specific finalization. */

#define GM_FNAME gm_arch_port_state_close

void
GM_FNAME(gm_port_state_t *ps)
{
   GM_PARAMETER_MAY_BE_UNUSED (ps);        /* If not GM_DEBUG */
   gm_assert(ps->instance);
   GM_PRINT (GM_DEBUG_PORT_STATE, 
             (GM_FNAME_STR " entered for unit %d, port %d, minor %d\n",
              ps->instance->id, ps->id, ps->minor));

   return;
}

#undef GM_FNAME




/* This is called just before the port state is destroyed (in
   gm_minor.c) to perform architecture-specific finalization. */

#define GM_FNAME gm_arch_port_state_fini

void
GM_FNAME(gm_port_state_t *ps)
{
   GM_PARAMETER_MAY_BE_UNUSED (ps);        /* If not GM_DEBUG */
   GM_PRINT (GM_DEBUG_PORT_STATE, 
             (GM_FNAME_STR " entered for port %d, minor %d\n",
              ps->id, ps->minor));
}

#undef GM_FNAME

/**********************************************************************
 * gm_ioctl support
 **********************************************************************/

/* Copy user data to a kernel buffer. */

/* NOTE: the IRIX man pages for copyin() and copyout() state:
 *       "Drivers usually convert a return value of -1 into an EFAULT error."
 */
gm_status_t
gm_arch_copyin(gm_port_state_t *ps,     /* portstate not needed in IRIX */
               void            *what,   /* copy from this user address  */
               void            *where,  /* to this kernel address       */
	       gm_size_t       amount)  /* for this length              */
{
   int copyin_status;


   GM_PARAMETER_MAY_BE_UNUSED (ps);        /* If not GM_DEBUG */
   GM_PRINT (GM_PRINT_LEVEL >= 7,
             ("gm_arch_copyin() entered with %p, %p,\n", ps, what));
   _GM_PRINT (GM_PRINT_LEVEL >= 7,
              ("    %p, %ld (0x%lx)\n", where, amount, amount));

   switch(amount)
   {
#if 0
/* Use of fuword() and fubyte() is disabled because valid return of a value
 * of -1 is indistinguishable from function failure.
 */
   case 4:
      copyin_status = fuword(what);
      if (copyin_status == -1)
         break;
      else
      {
         *((gm_u32_t *)where) = (gm_u32_t)copyin_status;
         copyin_status = 0;
      }
      break;

   case 1:
      copyin_status = fubyte(what);
      if (copyin_status == -1)
         break;
      else
      {
         *((gm_u8_t *)where) = (gm_u8_t)copyin_status;
         copyin_status = 0;
      }
      break;
#endif      
   default:
      copyin_status = copyin((caddr_t)what, (caddr_t)where, (int)amount);
   }
      
   if (copyin_status)
      return GM_MEMORY_FAULT;	/* User address was invalid */
   else
      return GM_SUCCESS;
}




/* Copy kernel data to a user buffer */

gm_status_t
gm_arch_copyout(gm_port_state_t *ps,     /* portstate not needed in IRIX  */
                void            *what,   /* copy from this kernel address */
                void            *where,  /* to this user address          */
                gm_size_t       amount)  /* for this length               */
{
   int copyout_status;


   GM_PARAMETER_MAY_BE_UNUSED (ps);        /* If not GM_DEBUG */
   GM_PRINT (GM_PRINT_LEVEL >= 7,
             ("gm_arch_copyout() entered with %p, %p,\n", ps, what));
   _GM_PRINT (GM_PRINT_LEVEL >= 7,
              ("    %p, %ld (0x%lx)\n", where, amount, amount));

   switch(amount)
   {
   case 4:
      gm_assert(sizeof(int) == 4);
      copyout_status = suword(where, *((int *)what));
      break;

   case 1:
      copyout_status = subyte(where, *((char *)what));
      break;

   default:
      copyout_status = copyout((caddr_t)what, (caddr_t)where, (int)amount);
   }

   if (copyout_status)
      return GM_MEMORY_FAULT;	/* User address was invalid */
   else
      return GM_SUCCESS;
}

/**********************************************************************
 *  Miscellaneous required functions
 **********************************************************************/


gm_status_t
gm_arch_gethostname (char *ptr, int len)
/*
 * Description: Copy the host name into the area provided by the caller
 * 
 * Since this is the kernel, we can pick the nodename right out
 * of the kernel data area.  On the other hand, doing the strncpy() directly
 * from the kernel data area into the caller's data area seemed to cause a
 * problem in my tests; note that ptr actually is a PCI address, so the copy
 * is into the device memory.  Using the local variable solved the problem.
 */
{
   gm_size_t  our_len; 
   int  truncated = 0;
   char name_copy[GM_MAX_HOST_NAME_LEN];


   if (len <= GM_MAX_HOST_NAME_LEN)
      our_len = len;
   else
   {
      our_len = GM_MAX_HOST_NAME_LEN;
      GM_NOTE (("gm_arch_gethostname len %d greater than %d; using %d\n",
	          len, our_len, our_len));
   }

   gm_bzero(ptr, our_len);

   strncpy(name_copy, utsname.nodename, our_len);
   if(name_copy[our_len-1] != 0)
   {
      name_copy[our_len-1] = 0;
      truncated  = 1;
   }

   GM_PRINT (GM_PRINT_LEVEL >= 5,
             ("gm_arch_gethostname(%p, %d) returning \"%s\"\n",
              ptr, len, name_copy));
   if (truncated)
      GM_PRINT (GM_PRINT_LEVEL >= 5, (" - TRUNCATED\n"));

   strcpy(ptr, name_copy);

   return (GM_SUCCESS);
}




/* Return the value of GM_PAGE_LEN set during driver initialization. */

gm_status_t
gm_arch_get_page_len(unsigned long *page_len)
{
   GM_PRINT (GM_PRINT_LEVEL >= 5,
             ("gm_arch_get_page_len entered with %p\n", page_len));

   if (!GM_PAGE_LEN)
   {
      GM_WARN(("gm_irix internal error: GM_PAGE_LEN unknown in call\n"));
      _GM_WARN(("    to gm_arch_get_page_len()\n"));
      return GM_FAILURE;
   }
   *page_len = GM_PAGE_LEN;
   return GM_SUCCESS;
}

/************************************************************************
 * Diagnostic and debugging support
 ************************************************************************/


/* Convert GM specific error code to IRIX's closest equivalent.
 *
 * NOTE: IRIX provides an extended discussion of the meaning that's
 *       intended for each code defined in <errno.h>; for that discussion,
 *       see the man page "intro(2)".  Read it from the IRIX TechPubs
 *       website, use INFOSEARCH, or enter the command "man 2 intro".
 */

int
gm_irix_localize_status (gm_status_t status)
{
#define CASE(from,to) case from : return to
  switch (status)
    {
      CASE (GM_SUCCESS, 0);
      CASE (GM_INPUT_BUFFER_TOO_SMALL, EFAULT);
      CASE (GM_OUTPUT_BUFFER_TOO_SMALL, EFAULT);
      CASE (GM_TRY_AGAIN, EAGAIN);
      CASE (GM_BUSY, EBUSY);
      CASE (GM_MEMORY_FAULT, EFAULT);
      CASE (GM_INTERRUPTED, EINTR);
      CASE (GM_INVALID_PARAMETER, EINVAL);
      CASE (GM_OUT_OF_MEMORY, ENOMEM);
      CASE (GM_INVALID_COMMAND, EINVAL);
      CASE (GM_PERMISSION_DENIED, EPERM);
      CASE (GM_INTERNAL_ERROR, ENXIO);
      CASE (GM_UNATTACHED, EUNATCH);
      CASE (GM_UNSUPPORTED_DEVICE, ENXIO);
    default:
      return gm_irix_localize_status (GM_INTERNAL_ERROR);
    }

}

/* gm_irix_get_bridge_version(): determine XIO Bridge revision level
 *
 * Description:
 *
 *    The bridge version is found in the high-order four bits of the 
 *    "Bridge ID register."  That register, in turn, is found at offset
 *    0 from the configuration space of the 'controller' vertex of the
 *    hwgraph.
 *
 *    See also ./gm/tests/bridgeversion.c.
 */

char
gm_irix_get_bridge_version(gm_instance_state_t *is)
{
   __uint32_t bridge_id_reg;
   void       *bridge_cfg_space;

#error function gm_irix_get_bridge_version() is incomplete

/* need *CORRECT* code to set bridge_cfg_space */
   bridge_cfg_space =
      (void *)pciio_piotrans_addr(is->arch.conn,
				  0,          /* dev desc not needed (?)    */
				  PCIIO_SPACE_CFG,
				  0x100000,   /* a guess at the offset: 1MB */
				  0x100000,   /* a guess at the size: 1MB   */
				  GM_IRIX_ENDIANNESS);

   if (!bridge_cfg_space)
   {
      GM_WARN(("Could not determine bridge version for unit %d\n", is->id));
      return('?');
   }

   /* "Danger, Will Robinson, Danger!!" */
   bridge_id_reg = *(__uint32_t *)(((char *)bridge_cfg_space) + 4);

   return (char)((bridge_id_reg>>28) + 'A' - 1);
}

/************
 * Utility functions for generic error printing (invoked only via macros
 * GM_PRINT, GM_WARN, etc.; must have external linkage)
 * 
 * NOTE: SGI says that, because the IRIX kernel is threaded, it's ok
 *       to use mutexes, even in "bottom-half" functions (i.e., in
 *       interrupt context).
 *
 *       ON THE OTHER HAND, we clearly can't call gm_arch_mutex_enter() etc.
 *       to enter and exit the print mutex, because we would get into an
 *       infinite regress if any debug traces were enabled in that function.
 ************/

void
gm_irix_print(char *format, ...)
{
   va_list ap;


   MUTEX_LOCK(&gm_irix_print_mu, GM_IRIX_MUTEX_PRIO);

   gm_irix_cmn_err_start(CE_CONT, "");
   va_start(ap, format);
   gm_irix_cmn_err_finish(CE_CONT, format, ap);
   va_end(ap);

   MUTEX_UNLOCK(&gm_irix_print_mu);
}




void
gm_irix_warn(char *format, ...)
{
   va_list ap;


   MUTEX_LOCK(&gm_irix_print_mu, GM_IRIX_MUTEX_PRIO);

   gm_irix_cmn_err_start(CE_CONT, "");
   va_start(ap, format);
   gm_irix_cmn_err_finish(CE_CONT, format, ap);
   va_end(ap);

   MUTEX_UNLOCK(&gm_irix_print_mu);
}




void
gm_irix_note(char *format, ...)
{
   va_list ap;


   MUTEX_LOCK(&gm_irix_print_mu, GM_IRIX_MUTEX_PRIO);

   gm_irix_cmn_err_start(CE_CONT, "");
   va_start(ap, format);
   gm_irix_cmn_err_finish(CE_CONT, format, ap);
   va_end(ap);

   MUTEX_UNLOCK(&gm_irix_print_mu);
}

 
void
gm_irix_panic(char *format, ...)
{
   va_list ap;


   /* Locks can not be held across calls to cmn_err with CE_PANIC */
   /*   MUTEX_LOCK(&gm_irix_print_mu, GM_IRIX_MUTEX_PRIO); */
   gm_irix_cmn_err_start(CE_CONT, "");
   va_start(ap, format);
   gm_irix_cmn_err_finish(CE_CONT, format, ap);
   va_end(ap);
   cmn_err(CE_PANIC, "(gm panic)");
   /*   MUTEX_UNLOCK(&gm_irix_print_mu); */
   /*   drv_usecwait(100000); */
}



void gm_arch_abort (void)
{
#if 0	/* This is what Linux does.  Maybe we should imitate some of this? */
#if GM_DEBUG
	if (debug_is && debug_is->lanai.running)
		gm_disable_lanai(debug_is);
	if (debug_is && atomic_read(&debug_is->page_hash.sync.mu.count) == 0)
		gm_arch_mutex_exit(&debug_is->page_hash.sync);
	if (debug_is && atomic_read(&debug_is->pause_sync.mu.count) == 0)
		gm_arch_mutex_exit(&debug_is->pause_sync);
	if (gm_in_interrupt) {
#if GM_DEBUG_SETJMP
		printk(KERN_EMERG "Aie, GM-PANIC inside interrupt, let go out of this\n");
		__longjmp(intr_jmp_buf, 1);
#else  /* !GM_DEBUG_SETJMP */
		panic("GM-PANIC in interrupt handler:cannot recover");
#endif /* !GM_DEBUG_SETJMP */
	}
	if (gm_current->state & PF_EXITING) {
		/* we probably failed in the close procedure, so we will never execute the dec count in linux_close */
		MOD_DEC_USE_COUNT;
	}
	hack_sys_exit(12);
#else
	panic("gm_arch_abort() called");
#endif
#endif /* What Linux does */

   cmn_err(CE_PANIC, "(gm_arch_abort called)");
}



#if GM_IRIX_DEBUG_MEMLEAKS
/*
 * Debugging routine for memory, lock, etc. leak detection
 */
void
gm_irix_memleak_report(char *the_comment)
{
   GM_PARAMETER_MAY_BE_UNUSED (the_comment);        /* If not GM_DEBUG */
   GM_PRINT (1, ("Resource metering %s:\n", the_comment));
   _GM_PRINT (1, ("  kmallocs      = %d\n", gm_irix_mld_struct.kmalloc_cnt  ));
   _GM_PRINT (1, ("    (GM kmallocs = %d)\n",
                  gm_irix_mld_struct.kernel_alloc_cnt));
   _GM_PRINT (1, ("  kmfrees       = %d\n", gm_irix_mld_struct.kmfree_cnt   ));
   _GM_PRINT (1, ("    (GM kmfrees  = %d)\n",
                  gm_irix_mld_struct.kernel_free_cnt ));
   _GM_PRINT (1, ("  dma allocs    = %d\n", gm_irix_mld_struct.dma_alloc_cnt));
   _GM_PRINT (1, ("  dma frees     = %d\n", gm_irix_mld_struct.dma_free_cnt ));
   _GM_PRINT (1, ("  dmamap allocs = %d\n",
                  gm_irix_mld_struct.dmamap_alloc_cnt));
   _GM_PRINT (1, ("    (ifnet dmamap allocs = %d)\n",
                  gm_irix_mld_struct.ifn_dmamap_alloc_cnt));
   _GM_PRINT (1, ("  dmamap frees  = %d\n",
                  gm_irix_mld_struct.dmamap_free_cnt ));
   _GM_PRINT (1, ("    (ifnet dmamap frees  = %d)\n",
                  gm_irix_mld_struct.ifn_dmamap_free_cnt));
   _GM_PRINT (1, ("  dma pg allocs = %d\n",
                  gm_irix_mld_struct.dma_pg_alloc_cnt));
   _GM_PRINT (1, ("    (ifnet dma pg allocs = %d)\n",
                  gm_irix_mld_struct.ifn_dma_pg_alloc_cnt));
   _GM_PRINT (1, ("  dma pg frees  = %d\n",
                  gm_irix_mld_struct.dma_pg_free_cnt ));
   _GM_PRINT (1, ("    (ifnet dma pg frees  = %d)\n",
                  gm_irix_mld_struct.ifn_dma_pg_free_cnt ));
   _GM_PRINT (1, ("  user buffer pages locked   = %d\n",
                  gm_irix_mld_struct.user_lock_cnt   ));
   _GM_PRINT (1, ("  user buffer pages unlocked = %d\n",
                  gm_irix_mld_struct.user_unlock_cnt ));
   _GM_PRINT (1, ("  hwgraph vertexes added     = %d\n",
                  gm_irix_mld_struct.vertex_add_cnt  ));
   _GM_PRINT (1, ("  hwgraph edges added        = %d\n",
                  gm_irix_mld_struct.edge_add_cnt    ));
   _GM_PRINT (1, ("  hwgraph edges removed      = %d\n",
                  gm_irix_mld_struct.edge_remove_cnt ));
   _GM_PRINT (1, ("  gm_minor_alloc() successes = %d\n",
                  gm_irix_mld_struct.minor_alloc_cnt ));
   _GM_PRINT (1, ("  gm_minor_free() calls      = %d\n",
                  gm_irix_mld_struct.minor_free_cnt  ));
   _GM_PRINT (1, ("  gm_port_state_close() calls = %d\n",
                  gm_irix_mld_struct.pstate_close_cnt));

   GM_PRINT (1, ("Retained resources:\n"));
   _GM_PRINT (1, ("  kmem_alloc areas   = %d\n",
                  gm_irix_mld_struct.kmalloc_cnt -
                  gm_irix_mld_struct.kmfree_cnt   ));
   _GM_PRINT (1, ("  DMA maps           = %d\n",
                  gm_irix_mld_struct.dmamap_alloc_cnt -
                  gm_irix_mld_struct.dmamap_free_cnt   ));
   _GM_PRINT (1, ("  DMA pages          = %d\n",
                  gm_irix_mld_struct.dma_pg_alloc_cnt -
                  gm_irix_mld_struct.dma_pg_free_cnt   ));
   _GM_PRINT (1, ("  Locked bufr pages  = %d\n",
                  gm_irix_mld_struct.user_lock_cnt -
                  gm_irix_mld_struct.user_unlock_cnt   ));
}
#endif


/* Determine whether a memory area is entirely zero  */
/* Returns 1 if all bytes are zero, else returns 0 */
int
gm_irix_area_is_0(u_char *cptr, int len)
{
   int      all_zero = 1;
   int      i;


   for (i = 0; i < len; i++)
   {
      if (cptr[i])
      {
	 all_zero = 0;
	 break;
      }
   }
   return (all_zero);
}


/* Hex dump as a debugging aid                     */
/* Returns 0 if all bytes are zero, else returns 1 */
int
gm_irix_hexdump(u_char *cptr, int len)
{
   int      i,j;
   int      num_bytes;
   char     dump_line[256],
            part0[128],
            part1[128],
            part2[128],
            part3[128];
  char ht[768] =
  " 0\0 1\0 2\0 3\0 4\0 5\0 6\0 7\0 8\0 9\0 a\0 b\0 c\0 d\0 e\0 f\0"
/* Avoid octal encodings, which begin with \00 .. \07                */
  "10\0""11\0""12\0""13\0""14\0""15\0""16\0""17\0"
  "18\0""19\0""1a\0""1b\0""1c\0""1d\0""1e\0""1f\0"
  "20\0""21\0""22\0""23\0""24\0""25\0""26\0""27\0"
  "28\0""29\0""2a\0""2b\0""2c\0""2d\0""2e\0""2f\0"
  "30\0""31\0""32\0""33\0""34\0""35\0""36\0""37\0"
  "38\0""39\0""3a\0""3b\0""3c\0""3d\0""3e\0""3f\0"
  "40\0""41\0""42\0""43\0""44\0""45\0""46\0""47\0"
  "48\0""49\0""4a\0""4b\0""4c\0""4d\0""4e\0""4f\0"
  "50\0""51\0""52\0""53\0""54\0""55\0""56\0""57\0"
  "58\0""59\0""5a\0""5b\0""5c\0""5d\0""5e\0""5f\0"
  "60\0""61\0""62\0""63\0""64\0""65\0""66\0""67\0"
  "68\0""69\0""6a\0""6b\0""6c\0""6d\0""6e\0""6f\0"
  "70\0""71\0""72\0""73\0""74\0""75\0""76\0""77\0"
  "78\0""79\0""7a\0""7b\0""7c\0""7d\0""7e\0""7f\0"
  "80\081\082\083\084\085\086\087\088\089\08a\08b\08c\08d\08e\08f\0"
  "90\091\092\093\094\095\096\097\098\099\09a\09b\09c\09d\09e\09f\0"
  "a0\0a1\0a2\0a3\0a4\0a5\0a6\0a7\0a8\0a9\0aa\0ab\0ac\0ad\0ae\0af\0"
  "b0\0b1\0b2\0b3\0b4\0b5\0b6\0b7\0b8\0b9\0ba\0bb\0bc\0bd\0be\0bf\0"
  "c0\0c1\0c2\0c3\0c4\0c5\0c6\0c7\0c8\0c9\0ca\0cb\0cc\0cd\0ce\0cf\0"
  "d0\0d1\0d2\0d3\0d4\0d5\0d6\0d7\0d8\0d9\0da\0db\0dc\0dd\0de\0df\0"
  "e0\0e1\0e2\0e3\0e4\0e5\0e6\0e7\0e8\0e9\0ea\0eb\0ec\0ed\0ee\0ef\0"
  "f0\0f1\0f2\0f3\0f4\0f5\0f6\0f7\0f8\0f9\0fa\0fb\0fc\0fd\0fe\0ff";

  char ct[512] =
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* 00 - 0f */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* 10 - 1f */
  " \0!\0\"\0#\0$\0%\0&\0\'\0(\0)\0*\0+\0,\0-\0.\0/\0" /* 20 - 2f */
  "0\0""1\0""2\0""3\0""4\0""5\0""6\0""7\0"             /* 30 - 37 */
                         "8\09\0:\0;\0<\0=\0>\0\?\0"   /* 38 - 3f */
  "@\0A\0B\0C\0D\0E\0F\0G\0H\0I\0J\0K\0L\0M\0N\0O\0"   /* 40 - 4f */
  "P\0Q\0R\0S\0T\0U\0V\0W\0X\0Y\0Z\0[\0\\\0]\0^\0_\0"  /* 50 - 5f */
  "`\0a\0b\0c\0d\0e\0f\0g\0h\0i\0j\0k\0l\0m\0n\0o\0"   /* 60 - 6f */
  "p\0q\0r\0s\0t\0u\0v\0w\0x\0y\0z\0{\0|\0}\0~\0.\0"   /* 70 - 7f */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* 80 - 8f */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* 90 - 9f */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* a0 - af */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* b0 - bf */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* c0 - cf */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* d0 - df */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* e0 - ef */
  ".\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0.\0"   /* f0 - ff */ ;


   num_bytes = (len > 256) ? 256 : len;

   if (num_bytes < 1)
   {
      _GM_PRINT(1, ("gm: ** Length-param = %d, dumping 1 byte\n", num_bytes));
      num_bytes = 1;
   }

   if (gm_irix_area_is_0(cptr, num_bytes))
   {
      _GM_PRINT(1, ("gm: %d bytes at 0x%x are all zero\n", num_bytes, cptr));
      return(0);
   }

   _GM_PRINT(1, ("gm: Dumping %d bytes at 0x%x:\n", num_bytes, cptr));

   for (i = 0; i < num_bytes; i+=16)
   {
      sprintf(part0,
              "%s %s %s %s %s %s %s %s",
              &(ht[3*(cptr[i])]),   &(ht[3*(cptr[i+1])]),
              &(ht[3*(cptr[i+2])]), &(ht[3*(cptr[i+3])]),
              &(ht[3*(cptr[i+4])]), &(ht[3*(cptr[i+5])]),
              &(ht[3*(cptr[i+6])]), &(ht[3*(cptr[i+7])]));
      sprintf(part2,
              "%s%s%s%s%s%s%s%s",
              &(ct[2*(cptr[i])]),   &(ct[2*(cptr[i+1])]),
              &(ct[2*(cptr[i+2])]), &(ct[2*(cptr[i+3])]),
              &(ct[2*(cptr[i+4])]), &(ct[2*(cptr[i+5])]),
              &(ct[2*(cptr[i+6])]), &(ct[2*(cptr[i+7])]));

      if ((i+8) >= num_bytes)
      {
         strcpy(part1,"                       ");
         strcpy(part3,"");
         for (j = 7; (i+j) >= num_bytes; j--) /* blank excess locations */
         {
            part0[j*3]     = ' ';
            part0[j*3 + 1] = ' ';
            part2[j]       = ' ';
         }
      }
      else
      {
         sprintf(part1,
                 "%s %s %s %s %s %s %s %s",
                 &(ht[3*(cptr[i+8])]),  &(ht[3*(cptr[i+9])]),
                 &(ht[3*(cptr[i+10])]), &(ht[3*(cptr[i+11])]),
                 &(ht[3*(cptr[i+12])]), &(ht[3*(cptr[i+13])]),
                 &(ht[3*(cptr[i+14])]), &(ht[3*(cptr[i+15])]));
         sprintf(part3,
                 "%s%s%s%s%s%s%s%s",
                 &(ct[2*(cptr[i+8])]),  &(ct[2*(cptr[i+9])]),
                 &(ct[2*(cptr[i+10])]), &(ct[2*(cptr[i+11])]),
                 &(ct[2*(cptr[i+12])]), &(ct[2*(cptr[i+13])]),
                 &(ct[2*(cptr[i+14])]), &(ct[2*(cptr[i+15])]));
         if ((i+16) > num_bytes)
         {
            for (j = 7; (i+j+8) >= num_bytes; j--) /* blank excess locations */
            {
               part1[j*3]     = ' ';
               part1[j*3 + 1] = ' ';
               part3[j]       = ' ';
            }
         }
      }

      sprintf(dump_line, "    %s  %s  | %s %s\n", part0, part1, part2, part3);

      _GM_PRINT(1, ("%s", dump_line));
   }

   return(1);
}



/*
 * gm_irix_programmed_panic()
 *
 * Description:  Programmed PANIC, for use when we need to see state and
 *               context at a defined point, for debugging
 */
void
gm_irix_programmed_panic(char *the_comment)
{
   GM_PANIC (("Deliberate PANIC %s:\n", the_comment));
}

/* Functions to backup cmn_err() printouts by saving in memory */

#if GM_IRIX_USE_MONSTER
void
gm_irix_cmn_err_start(int the_severity, char * the_string)
/*
 * This outputs the first part of a line, to be followed by a call to
 * gm_irix_cmn_err_finish() to complete the line.  There are no variable
 * parameters.
 */
{
   bzero((void *)gm_monster_current, GM_MONSTER_LINESIZ);
   strncpy(gm_monster_current, the_string, GM_MONSTER_LINESIZ);
   GM_MONSTER_SLOT_TERMINATE

   cmn_err(the_severity, the_string); /* Now call the real cmn_err() */
}



void
gm_irix_cmn_err_finish(int the_severity, char * the_format, va_list the_arg)
/*
 * This completes the line begun by gm_irix_cmn_err_start().  It has
 * variable parameters.
 */
{
   size_t  size_so_far = strlen(gm_monster_current);
   size_t  size_left   = GM_MONSTER_LINESIZ - size_so_far;
   char    rest_of_msg[1024];


   vsprintf(rest_of_msg, the_format, the_arg);
/* Sigh.  For some reason, there's no strncat() in the IRIX kernel, though
 * strncpy() and strncmp() do exist there.
 */
   strncpy(&(gm_monster_current[size_so_far]), rest_of_msg, size_left);
   GM_MONSTER_SLOT_TERMINATE

   GM_MONSTER_NEXT_SLOT

   icmn_err(the_severity, the_format, the_arg); /* Call the real icmn_err() */
}

#endif


/*********************************************************************
 **                                                                 **
 **   END of gm_arch.c                                              **
 **                                                                 **
 *********************************************************************/
