/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

#include "gm_internal.h"
#include "gm_debug.h"

#define GM_DEBUG_UC 0

#define lanai_read_control(unit) (*(gm_u32_t*)is->lanai.control_regs)
#define lanai_write_control(unit,val) (*(gm_u32_t*)is->lanai.control_regs=(val))
#define fflush(arg)

/* A hack function to return an estimate of the time */

static int
cop_time (gm_instance_state_t * is)
{
  static int usecs;
  gm_arch_spin (is, 10);
  usecs += 2;
  return usecs >> 20;
}

/************
 * Trivially modified version of example code from UC doc.
 ************/

/* cop_ack: returns the value of the ack from the uC */

static int
cop_ack (gm_instance_state_t * is)
{
  if (lanai_read_control (unit) & 0x1000)
    return (1);
  else
    return (0);
}

/* cop_in_data: returns the value of the data bit from the uC */

static int
cop_in_data (gm_instance_state_t * is)
{
  if (lanai_read_control (unit) & 0x2000)
    return (1);
  else
    return (0);
}

/* cop_out_data: sets the data output bit to 1 or 0, depending on d */

static void
cop_out_data (gm_instance_state_t * is, int d)
{
  if (d)
    {
      lanai_write_control (unit, lanai_read_control (unit) | 0x0800);
    }
  else
    {
      lanai_write_control (unit,
			   lanai_read_control (unit) & ~(gm_u32_t) 0x0800);
    }
}

/* cop_req: returns the current value of the request to the uC */

static int
cop_req (gm_instance_state_t * is)
{
  if (lanai_read_control (unit) & 0x0400)
    return (1);
  else
    return (0);
}

/* cop_toggle: toggles the request to the uC */

static void
cop_toggle (gm_instance_state_t * is)
{
  if (cop_req (is))
    {
      lanai_write_control (unit,
			   lanai_read_control (unit) & ~(gm_u32_t) 0x0400);
    }
  else
    {
      lanai_write_control (unit, lanai_read_control (unit) | 0x0400);
    }
}

/* cop_wait_for_ack: waits for the cop to acknowledge.
                     if 3 seconds go by, it toggles the request,
                     waits 2 seconds, toggles the request again,
                     and resumes waiting for an ack.
                     If it does this 2 times and still doesn't get
                     an ack, the uC must be hosed, so
                     it fails and returns -1
*/

static int
cop_wait_for_ack (gm_instance_state_t * is)
{
  int start;
  int numtries;

  numtries = 0;
  start = cop_time (is);
  GM_PRINT (GM_DEBUG_UC, ("**"));
  while (cop_ack (is) != cop_req (is))
    {
      if (cop_time (is) >= start + 3)
	{
	  _GM_PRINT (GM_DEBUG_UC, ("*"));
	  fflush (stdout);
	  numtries++;
	  if (numtries >= 2)
	    return (-1);
	  cop_toggle (is);
	  start = cop_time (is);
	  while (cop_time (is) <= start + 2)
	    {;
	    }

	  _GM_PRINT (GM_DEBUG_UC, ("*"));
	  fflush (stdout);
	  cop_toggle (is);
	  start = cop_time (is);
	}
    }
  _GM_PRINT (GM_DEBUG_UC, ("\n"));
  return (0);
}

/* consecutive1s keeps track of how many 1 bits we've send in a row */

static int consecutive1s = 0;

/* cop_receive: sends a 0x00 byte to the uC, receives a byte from the uC. 
                returns the byte received, or -1 if failure */

static int
cop_receive (gm_instance_state_t * is)
{
  int i;
  int d;

  d = 0;
  cop_out_data (is, 0);
  consecutive1s = 0;
  for (i = 0; i < 8; i++)
    {
      if (cop_wait_for_ack (is))
	return (-1);
      d <<= 1;
      if (cop_in_data (is))
	d |= 1;
      cop_toggle (is);
      if (cop_wait_for_ack (is))
	return (-1);
    }
  return (d);
}

/* cop_send: sends a byte to the uC, adding a zero bit when required.
             returns 0 if successful, -1 if failure */

static int
cop_send (gm_instance_state_t * is, int d)
{
  int i;

  for (i = 0; i < 8; i++)
    {

      if (d & 0x80)
	{
	  consecutive1s++;
	  cop_out_data (is, 1);
	  cop_toggle (is);
	  if (cop_wait_for_ack (is))
	    return (-1);
	  if (consecutive1s >= 8)
	    {
	      consecutive1s = 0;
	      cop_out_data (is, 0);
	      cop_toggle (is);
	      if (cop_wait_for_ack (is))
		return (-1);
	    }
	}
      else
	{
	  cop_out_data (is, 0);
	  consecutive1s = 0;
	  cop_toggle (is);
	  if (cop_wait_for_ack (is))
	    return (-1);
	}
      d <<= 1;
    }
  return (0);
}

/* cop_wakeup: wakes the uC up. returns 0 if successful, -1 if failure */

static int
cop_wakeup (gm_instance_state_t * is)
{
  int i;

  cop_out_data (is, 1);
  for (i = 0; i < 10; i++)
    {
      cop_toggle (is);
      if (cop_wait_for_ack (is))
	return (-1);
    }
  consecutive1s = 0;
  cop_out_data (is, 0);
  cop_toggle (is);
  if (cop_wait_for_ack (is))
    return (-1);
  return (0);
}

/* cop_end: ends the present command. returns 0 if successful, -1 if failure */

static int
cop_end (gm_instance_state_t * is)
{
  cop_out_data (is, 1);
  cop_toggle (is);
  if (cop_wait_for_ack (is))
    return (-1);
  if (cop_ack == 0)
    {
      cop_toggle (is);
      if (cop_wait_for_ack (is))
	return (-1);
    }
  return (0);
}

#if 0
/* read_ucver: prints the version information out */

static int
read_ucver (gm_instance_state_t * is)
{
  int c, numzeroes;

  if (cop_wakeup (is) < 0)
    {
      GM_NOTE (("\nError - uC not responding\n"));
      return (-1);
    }
  cop_send (is, 0x11);
  cop_send (is, 0x00);
  c = cop_receive (is);
  GM_INFO (("microcontroller version code: %02x", c));
  numzeroes = 0;
  while (numzeroes < 2)
    {
      c = cop_receive (is);
      if (c < 0)
	break;
      if (c)
	{
	  _GM_INFO (("%c", c));
	  numzeroes = 0;
	}
      else
	{
	  _GM_INFO (("\n"));
	  numzeroes++;
	}
    }
  cop_end (is);
  return (0);
}
#endif

#if 0
/* do_sramwr: writes d into SRAM address a */

static int
do_sramwr (gm_instance_state_t * is, int a, int d)
{
  if (cop_wakeup (is) < 0)
    {
      GM_NOTE (("\nError - uC not responding\n"));
      return (-1);
    }
  cop_send (is, 0x20);
  cop_send (is, a);
  cop_send (is, d);
  cop_end (is);
  return (0);
}
#endif


/****************
 * libgm internal functions supplied by this file.  (They must have a
 * GM prefix.)
 ****************/

int
__gm_cop_send (gm_instance_state_t * is, int d)
{
  return cop_send (is, d);
}

int
__gm_cop_receive (gm_instance_state_t * is)
{
  return cop_receive (is);
}

int
__gm_cop_wakeup (gm_instance_state_t * is)
{
  return cop_wakeup (is);
}

int
__gm_cop_end (gm_instance_state_t * is)
{
  return cop_end (is);
}


/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
