/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This module initializes and finalizes ports */

#include "gm_call_trace.h"
#include "gm_debug_host_dma.h"
#include "gm_debug_port_state.h"
#include "gm_compiler.h"
#include "gm_impl.h"
#include "gm_enable_security.h"
#include "gm_pio.h"


static struct gm_mutex *local_mutex;
static int initialized;

gm_status_t __gm_init_port_state ()
{
  gm_mutex_enter (_gm_global_mutex);
  if (!initialized++)
    {
      local_mutex = gm_create_mutex ();
    }
  gm_mutex_exit (_gm_global_mutex);
  return GM_SUCCESS;
}

void
__gm_finalize_port_state ()
{
  gm_mutex_enter (local_mutex);
  initialized--;
  gm_mutex_exit (local_mutex);
  return;
}

/************
 * open
 ************/

static gm_status_t
gm_register_recv_queue_page (void *arg, gm_dp_t dma_addr, gm_u32_t page_num)
{
  gm_port_state_t *ps = (gm_port_state_t *) arg;
  gm_instance_state_t *is = ps->instance;
  unsigned int port_num = ps->id;
  unsigned int i;

  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("gm_register_recv_queue_page arg = %p, dma_addr = "GM_U64_TMPL","
	     " page_num = %x\n", arg, GM_U64_ARG (dma_addr), (int) page_num));

  /* Record the DMA address for the recv queue slots in this page */

  if (!GM_POWER_OF_TWO (sizeof (gm_recv_queue_slot_t)))
    {
      GM_NOTE (("Internal error: sizeof (gm_recv_queue_slot_t) is not 2^N\n"
		"sizeof (gm_recv_queue_slot_t) is 0x%x\n",
		sizeof (gm_recv_queue_slot_t)));
      GM_RETURN_STATUS (GM_FAILURE);
    }

  for (i = 0; i < GM_RECV_QUEUE_SLOTS_PER_PAGE; i++)
    {
      unsigned slot_num =
	(unsigned) (page_num * GM_RECV_QUEUE_SLOTS_PER_PAGE + i);

      if (slot_num >= GM_NUM_RECV_QUEUE_SLOTS)
	break;

      gm_write_lanai_global_dp
	(is, port[port_num].recv_queue_slot_dma_addr[slot_num],
	 (gm_dp_t) (dma_addr + i * sizeof (gm_recv_queue_slot_t)));
      GM_PRINT (GM_PRINT_LEVEL >= 10,
		("gm_register_recv_queue_page:  i=%d  dp = 0x%lx\n", i,
		 dma_addr + i * sizeof (gm_recv_queue_slot_t)));
    }

  GM_RETURN_STATUS (GM_SUCCESS);
}

gm_status_t gm_port_state_open (gm_port_state_t * ps, unsigned int port_num)
{
  gm_instance_state_t *is;
  gm_status_t status = GM_INTERNAL_ERROR;

  GM_CALLED ();

  gm_mutex_enter (local_mutex);

  GM_PRINT (GM_DEBUG_PORT_STATE,
	    ("gm_port_state_open (%p, %d) called.\n", ps, port_num));

  if (port_num > GM_NUM_PORTS)
    {
      status = GM_INVALID_PARAMETER;
      goto abort_with_mutex;
    }

  gm_assert (ps);
  is = ps->instance;
  gm_assert (is);

  /* Verify that the port is not already in use. */

  GM_PRINT (GM_DEBUG_PORT_STATE,
            ("[%d]: gm_port_state_open: instance = %p\n", is->id, is));

  if (gm_hash_find (is->port_hash, &port_num) != 0)
    {
      if (!GM_ENABLE_SECURITY && !is->lanai.running && (port_num == 0))
	{
	  GM_NOTE (("Allowing user to open a port (%d) that is in use for"
		    " debug.\n", port_num));
	}
      else
	{
	  GM_PRINT(GM_PRINT_LEVEL>=1,
		   ("User tried to claim a port number (%d) that is in use.\n",
		    port_num));
	  status = GM_BUSY;
	  goto abort_with_mutex;
	}
    }

  /* Verify the LANai is running. */

  if (!is->lanai.running)
    {
      if (!GM_ENABLE_SECURITY && (port_num == 0))
	{
	  GM_NOTE (("LANai is not running. Allowing port=0 open for"
		    " debugging\n"));
	}
      else
	{
	  GM_NOTE (("Could not open port because LANai is not running.\n"));
	  status = GM_INTERNAL_ERROR;
	  goto abort_with_mutex;
	}
    }

  /* Check permissions */

#if !GM_ENABLE_SECURITY
  ps->privileged = 1;
#endif

  GM_PRINT (GM_DEBUG_PORT_STATE, ("Checked permissions.\n"));
  if (!ps->privileged && (port_num == GM_DAEMON_PORT_ID
			  || port_num == GM_ETHERNET_PORT_ID
			  || port_num == GM_MAPPER_PORT_ID))
    {
      status = GM_PERMISSION_DENIED;
      goto abort_with_mutex;
    }

  /* Verify any previous port open has finished shutting down. */

  if (gm_read_lanai_global_u32 (is, port[port_num].active_subport_cnt) != 0)
    {
      if (!GM_ENABLE_SECURITY && (port_num == 0))
	{
	  GM_NOTE (("Still shutting down. Allowing port=0 open for"
		    " debugging\n"));
	}
      else
	{
	  GM_NOTE (("Could not open port because it is still shutting"
		    " down active_count=%d.\n",
		    gm_read_lanai_global_u32 (is,
                                         port[port_num].active_subport_cnt)));
	  status = GM_STILL_SHUTTING_DOWN;
	  goto abort_with_mutex;
	}
    }

  /* HACK: Clean up the port state in the lanai, since the process of
     closing down the port can corrupt the reset state. */

  GM_PRINT (GM_DEBUG_PORT_STATE,
	    ("[%d]: HACK: Clean up the port state in the lanai\n", is->id));
  if (is->lanai.running)
    gm_pause_lanai (is);
  GM_STBAR ();
  gm_write_lanai_global_s32 (is, port_to_close, port_num);
  GM_STBAR ();
  if (is->lanai.running)
    gm_unpause_lanai (is);

  GM_PRINT (GM_DEBUG_PORT_STATE,
	    ("[%d]: Creating port %d.\n", is->id, port_num));

  if (is->lanai.running)
    gm_pause_lanai (is);

  gm_write_lanai_global_u32 (is, port[port_num].privileged, ps->privileged);

  /* prep the host side of the port */

  ps->opened = 1;
  ps->id = port_num;
  ps->compatibility_mode = 0;
  /* ps->privileged; set by arch */
  ps->min_dma_mem = 0;
  ps->max_dma_mem = 0x01000000;	/* BAD: We don't even check these */
  /* GM_NOTE (("DMA memory limits are disabled.\n")); */
  gm_init_mapping_spec (ps);
  gm_arch_sync_init (&ps->sleep_sync, is);
  /* ps->copy_block: 0 */
  /* ps->recv_queue_region: initialized by this function call */

  GM_PRINT (GM_DEBUG_HOST_DMA, ("Allocating receive queue.\n"));
  status = gm_arch_dma_region_alloc (ps->instance,
				     &ps->recv_queue_region,
				     ps->mappings.recv_queue.len,
#ifdef WIN32
				     /* This region must be MMAP'd,
				        and NT can only MMAP
				        contiguous regions. */
				     GM_ARCH_DMA_CONTIGUOUS |
#endif
				     GM_ARCH_DMA_RDWR |
				     GM_ARCH_DMA_CONSISTENT,
				     gm_register_recv_queue_page, ps);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (
	       ("Could not create recv queue of length %d.\n",
		ps->mappings.recv_queue.len));

      goto abort_with_sleep_sync;
    }
  gm_bzero (gm_arch_dma_region_kernel_addr (&ps->recv_queue_region),
	    ps->mappings.recv_queue.len);

  /* Insert port in instance's port hash table to enable port_num->ps
     translation */

  status = gm_hash_insert (is->port_hash, &port_num, ps);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not insert port state in instance's port hash.\n"));
      goto abort_with_dma_region;
    }

  /* Peform architecture-specific port opening actions */

  status = gm_arch_port_state_open (ps);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not perform arch-specific port state open.\n"));
      goto abort_with_hash_insertion;
    }

  gm_write_lanai_global_u32 (is, port[ps->id].privileged, ps->privileged);
  gm_write_lanai_global_s32 (is, port_to_open, ps->id);

  GM_STBAR ();
  if (is->lanai.running)
    gm_unpause_lanai (is);
  gm_mutex_exit (local_mutex);
  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_hash_insertion:
  gm_hash_remove (is->port_hash, &port_num);
abort_with_dma_region:
  gm_arch_dma_region_free (&ps->recv_queue_region);
  /* Arrange for LANai to clear the registered DMA addrs for the recv queue. */
  gm_write_lanai_global_s32 (is, port_to_close, port_num);
abort_with_sleep_sync:
  gm_arch_sync_destroy (&ps->sleep_sync);
  ps->opened = 0;
  if (is->lanai.running)
    gm_unpause_lanai (is);
abort_with_mutex:
  gm_mutex_exit (local_mutex);
  GM_RETURN_STATUS (status);
}

/************
 * close
 ************/

#define GM_DEBUG_ADDR_TABLES 0

void
gm_port_state_close (gm_port_state_t * ps)
{
  gm_instance_state_t *is;
  unsigned int port_num;

  GM_CALLED ();

  if (!ps->opened)
    {
      GM_RETURN_NOTHING ();
    }

  is = ps->instance;
  gm_assert (is);
  port_num = ps->id;
  is->opener_pids[port_num] = 0;

  GM_PRINT (GM_DEBUG_PORT_STATE,
            ("[%d]: Closing port state %p, port_num %d.\n",
             is->id, ps, port_num));

  /* print debugging info */

  {
    unsigned int active;
    unsigned int i;

    active = gm_read_lanai_global_u32 (is, port[port_num].active_subport_cnt);
    if (active)
      GM_NOTE (("There are %d active subports for port %d at close.\n",
		active, port_num));

    if (GM_DEBUG_ADDR_TABLES)
      {
	GM_PRINT (1, ("gm_port_state_close(): addr tables:\n"));
	gm_pause_lanai (is);
	for (i = 0; i <= is->max_node_id; i++)
	  {
	    if (
		(is->ethernet.addr_tab.volatile_ethernet_addr[i].
		 as_unswapped_u64) && !gm_strlen (is->name_table.entry[i]))
	      continue;
	    GM_PRINT
	      (1,
	       ("  %d %02x:%02x:%02x:%02x:%02x:%02x %s\n",
		i,
		is->ethernet.addr_tab.volatile_ethernet_addr[i].as_bytes[0],
		is->ethernet.addr_tab.volatile_ethernet_addr[i].as_bytes[1],
		is->ethernet.addr_tab.volatile_ethernet_addr[i].as_bytes[2],
		is->ethernet.addr_tab.volatile_ethernet_addr[i].as_bytes[3],
		is->ethernet.addr_tab.volatile_ethernet_addr[i].as_bytes[4],
		is->ethernet.addr_tab.volatile_ethernet_addr[i].as_bytes[5],
		is->name_table.entry[i]));
	  }
	gm_unpause_lanai (is);
      }
  }

  /* Release resources allocated since open. */

  gm_disable_port_DMAs (ps);	/* closes the port */
#if !GM_CAN_REGISTER_MEMORY
  GM_PRINT (GM_PRINT_LEVEL >= 3, ("about to free copy block.\n"));
  gm_free_copy_block (ps);
#endif

  /* release resources from open. */

  gm_arch_port_state_close (ps);
  gm_hash_remove (ps->instance->port_hash, &ps->id);
  GM_PRINT (GM_DEBUG_HOST_DMA, ("Freeing receive queue.\n"));
  gm_arch_dma_region_free (&ps->recv_queue_region);
  /* gm_kton_s32 (is, &is->lanai.globals->port_to_close, port_num); */
  /* ^^^^ This was done by gm_disable_port_DMAs() above */
  gm_arch_sync_destroy (&ps->sleep_sync);
  ps->opened = 0;
  GM_RETURN_NOTHING ();
}

/****************************************************************
 * Thread safe port state acquisition
 ****************************************************************/

/* Return the port state given the instance state and the port number.
   WARNING: This is the unprotected version called from the interrupt
   handler, since mutex's cannot be acquire in the interrupt handler
   of most OS's.  This is safe in the interrupt handler because the
   port closure code ensures that no interrupts will be generated for
   the port before the port is destroyed. */

gm_port_state_t *
__gm_port_state_for_id (gm_instance_state_t * is, unsigned id)
{
  return (gm_port_state_t *) gm_hash_find (is->port_hash, &id);
}

/* Acquire the locked port state identified by IS and PORT.  If you
   get a port state with this call, you must release the port state
   with gm_release_port_state() when you are done with it. */


gm_port_state_t *
gm_acquire_port_state_for_id (gm_instance_state_t * is, unsigned port)
{
  gm_port_state_t *ps;

  gm_mutex_enter (local_mutex);
  ps = __gm_port_state_for_id (is, port);
  if (!ps)
    goto abort_with_mutex;
  gm_arch_mutex_enter (&ps->sync);
  gm_mutex_exit (local_mutex);
  return ps;

abort_with_mutex:
  gm_mutex_exit (local_mutex);
  return 0;
}

/* Acquire the locked port state identified by INSTANCE and PORT.  If
   you get a port state with this call, you must release the port
   state with gm_release_port_state() when you are done with it. */

gm_port_state_t *
gm_acquire_port_state_for_ids (unsigned int instance, unsigned int port)
{
  gm_instance_state_t *is;
  gm_port_state_t *ps;

  gm_mutex_enter (local_mutex);
  is = gm_instance_for_id (instance);
  if (!is)
    goto abort_with_mu;
  ps = __gm_port_state_for_id (is, port);
  if (!ps)
    goto abort_with_mu;
  gm_arch_mutex_enter (&ps->sync);
  gm_mutex_exit (local_mutex);
  return ps;

abort_with_mu:
  gm_mutex_exit (local_mutex);
  return 0;
}

/* Release the port state acquired with one of the gm_acquire_port_state*
   functions above. */

void
gm_release_port_state (gm_port_state_t * ps)
{
  gm_arch_mutex_exit (&ps->sync);
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
