/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file implements PIO read and write macros for the driver.  For
   most systems, these macros simply do nothing more than endian conversion.
   For systems that don't support PIO reads and writes, these macros
   perform word reads or word read-modify-writes to simulate partword
   IO, pausing the LANai during any read-modify-write to ensure the
   operation is performed atomically with respect to the MCP and any
   other driver threads accessing the same board. */

#ifndef _gm_pio_h_
#define _gm_pio_h_

#include "gm.h"
#include "gm_compiler.h"
#include "gm_config.h"
#include "gm_impl.h"
#include "gm_types.h"

/****************************************************************
 * Overrides
 ****************************************************************/

#if 0
#undef GM_NO_PARTWORD_PIO_READ
#undef GM_NO_PARTWORD_PIO_WRITE
#define GM_NO_PARTWORD_PIO_READ 1
#define GM_NO_PARTWORD_PIO_WRITE 1
#endif

/****************************************************************
 * Utility functions internal to this file.
 ****************************************************************/

static gm_inline void
__gm_pio_pause_lanai (struct gm_instance_state *is)
{
  gm_assert (is->pause_rqst == 0);
  gm_pause_lanai (is);
}

/****************************************************************
 * PIO without endian conversion (but with endian checking)
 ****************************************************************/

/* All driver PIO is performed either by these macros, or by
   gm_copy_to_io_space(). */

/****************
 * reads
 ****************/

struct gm_instance_state;

static gm_inline gm_u64_n_t
__gm_pio_read_u64 (struct gm_instance_state *is, volatile gm_u64_n_t * ptr)
{
  GM_PARAMETER_MAY_BE_UNUSED (is);
  return *(gm_u64_n_t *) ptr;
}

static gm_inline gm_u32_n_t
__gm_pio_read_u32 (struct gm_instance_state *is, volatile gm_u32_n_t * ptr)
{
  GM_PARAMETER_MAY_BE_UNUSED (is);
  return *(gm_u32_n_t *) ptr;
}

static gm_inline gm_u16_n_t
__gm_pio_read_u16 (struct gm_instance_state *is, volatile gm_u16_n_t * ptr)
{
  gm_u16_n_t ret;
  GM_PARAMETER_MAY_BE_UNUSED (is);

  if (GM_NO_PARTWORD_PIO_READ)
    {
      volatile gm_u32_n_t w, *wptr;
      long index;

      index = ((long) ptr >> 1) & 1;
      wptr = (gm_u32_n_t *) & ptr[-index];
      w = *wptr;
      ret = ((gm_u16_n_t *) & w)[index];
    }
  else
    {
      ret = *ptr;
    }
  return ret;
}

static gm_inline gm_u8_n_t
__gm_pio_read_u8 (struct gm_instance_state *is, volatile gm_u8_n_t * ptr)
{
  gm_u8_n_t ret;
  GM_PARAMETER_MAY_BE_UNUSED (is);

  if (GM_NO_PARTWORD_PIO_READ)
    {
      volatile gm_u32_n_t w, *wptr;
      long index;

      index = (long) ptr & 3;
      wptr = (gm_u32_n_t *) & ptr[-index];
      w = *wptr;
      ret = ((gm_u8_n_t *) & w)[index];
    }
  else
    {
      ret = *ptr;
    }
  return ret;
}

/****************
 * writes
 ****************/

static gm_inline void
__gm_pio_write_u64 (struct gm_instance_state *is, volatile gm_u64_n_t * ptr,
		    gm_u64_n_t val)
{
  GM_PARAMETER_MAY_BE_UNUSED (is);
  *ptr = val;
}

static gm_inline void
__gm_pio_write_u32 (struct gm_instance_state *is, volatile gm_u32_n_t * ptr,
		    gm_u32_n_t val)
{
  GM_PARAMETER_MAY_BE_UNUSED (is);
  *ptr = val;
}

static gm_inline void
__gm_pio_write_u16 (struct gm_instance_state *is, volatile gm_u16_n_t * ptr,
		    gm_u16_n_t val)
{
  if (GM_NO_PARTWORD_PIO_WRITE)
    {
      volatile gm_u32_n_t w;
      volatile gm_u32_n_t *wptr;
      long index;

      __gm_pio_pause_lanai (is);

      index = (long) ptr & 2;

      wptr = (gm_u32_n_t *) ((char *) ptr - index);
      w = *wptr;
      ((gm_u16_n_t *) & w)[index >> 1] = val;
      *wptr = w;
      gm_unpause_lanai (is);
    }
  else
    {
      *ptr = val;
    }
}

/* write a byte using read-modify-write */

static gm_inline void
___gm_pio_write_u8 (volatile gm_u8_n_t * ptr, gm_u8_n_t val)
{
  volatile gm_u32_n_t w;
  volatile gm_u32_n_t *wptr;
  long index;

  index = (long) ptr & 3;

  wptr = (gm_u32_n_t *) ((char *) ptr - index);
  w = *wptr;
  ((gm_u8_n_t *) & w)[index] = val;
  *wptr = w;
}

static gm_inline void
__gm_pio_write_u8 (struct gm_instance_state *is, volatile gm_u8_n_t * ptr,
		   gm_u8_n_t val)
{
  if (GM_NO_PARTWORD_PIO_WRITE)
    {
      __gm_pio_pause_lanai (is);
      ___gm_pio_write_u8 (ptr, val);
      gm_unpause_lanai (is);
    }
  else
    {
      *ptr = val;
    }
}

/****************************************************************
 * PIO with endian conversion
 ****************************************************************/

/* Peform PIO with endian conversion. */

/****************
 * reads
 ****************/

static gm_inline gm_u64_t
__gm_ntok_u64 (struct gm_instance_state *is, volatile gm_u64_n_t * ptr)
{
  return gm_ntoh_u64 (__gm_pio_read_u64 (is, ptr));
}

static gm_inline gm_u32_t
__gm_ntok_u32 (struct gm_instance_state *is, volatile gm_u32_n_t * ptr)
{
  return gm_ntoh_u32 (__gm_pio_read_u32 (is, ptr));
}

static gm_inline gm_s32_t
__gm_ntok_s32 (struct gm_instance_state *is, volatile gm_s32_n_t * ptr)
{
  return (gm_s32_t) gm_ntoh_u32 (__gm_pio_read_u32 (is, (gm_u32_n_t *) ptr));
}

static gm_inline gm_lp_t
__gm_ntok_lp (struct gm_instance_state *is, volatile gm_lp_n_t * ptr)
{
  gm_lp_n_t l;

  GM_N (l) = GM_N (__gm_pio_read_u32 (is, (gm_u32_n_t *) ptr));
  return gm_ntoh_lp (l);
}

static gm_inline gm_up_t
__gm_ntok_up (struct gm_instance_state *is, volatile gm_up_n_t * ptr)
{
  if (sizeof (gm_up_t) == 8)
    return (gm_up_t) gm_ntoh_u64 (__gm_pio_read_u64 (is, (gm_u64_n_t *) ptr));
  else if (sizeof (gm_up_t) == 4)
    return (gm_up_t) gm_ntoh_u32 (__gm_pio_read_u32 (is, (gm_u32_n_t *) ptr));
  else
    {
      gm_abort ();
      return 0;
    }
}

static gm_inline gm_dp_t
__gm_ntok_dp (struct gm_instance_state *is, volatile gm_dp_n_t * ptr)
{
  if (sizeof (gm_dp_t) == 8)
    return (gm_dp_t) gm_ntoh_u64 (__gm_pio_read_u64 (is, (gm_u64_n_t *) ptr));
  else if (sizeof (gm_dp_t) == 4)
    return (gm_dp_t) gm_ntoh_u32 (__gm_pio_read_u32 (is, (gm_u32_n_t *) ptr));
  else
    {
      gm_abort ();
      return 0;
    }
}

static gm_inline gm_u16_t
__gm_ntok_u16 (struct gm_instance_state *is, volatile gm_u16_n_t * ptr)
{
  return gm_ntoh_u16 (__gm_pio_read_u16 (is, ptr));
}

static gm_inline gm_s16_t
__gm_ntok_s16 (struct gm_instance_state *is, volatile gm_s16_n_t * ptr)
{
  return (gm_s16_t) gm_ntoh_u16 (__gm_pio_read_u16 (is, (gm_u16_n_t *) ptr));
}

static gm_inline gm_u8_t
__gm_ntok_u8 (struct gm_instance_state *is, volatile gm_u8_n_t * ptr)
{
  return gm_ntoh_u8 (__gm_pio_read_u8 (is, ptr));
}

static gm_inline gm_s8_t
__gm_ntok_s8 (struct gm_instance_state *is, volatile gm_s8_n_t * ptr)
{
  return (gm_s8_t) gm_ntoh_u8 (__gm_pio_read_u8 (is, (gm_u8_n_t *) ptr));
}

/****************
 * writes
 ****************/

static gm_inline void
__gm_kton_u64 (struct gm_instance_state *is, volatile gm_u64_n_t * ptr,
	       gm_u64_t val)
{
  __gm_pio_write_u64 (is, ptr, gm_hton_u64 (val));
}

static gm_inline void
__gm_kton_u32 (struct gm_instance_state *is, volatile gm_u32_n_t * ptr,
	       gm_u32_t val)
{
  __gm_pio_write_u32 (is, ptr, gm_hton_u32 (val));
}

static gm_inline void
__gm_kton_s32 (struct gm_instance_state *is, volatile gm_s32_n_t * ptr,
	       gm_u32_t val)
{
  __gm_pio_write_u32 (is, (gm_u32_n_t *) ptr, gm_hton_u32 (val));
}

static gm_inline void
__gm_kton_lp (struct gm_instance_state *is, volatile gm_lp_n_t * ptr,
	      gm_lp_t val)
{
  __gm_pio_write_u32 (is, (gm_u32_n_t *) ptr, gm_hton_u32 (val));
}

static gm_inline void
__gm_kton_dp (struct gm_instance_state *is, volatile gm_dp_n_t * ptr,
	      gm_dp_t val)
{
  if (sizeof (gm_dp_t) == 8)
    __gm_pio_write_u64 (is, (gm_u64_n_t *) ptr, gm_hton_u64 ((gm_u64_t) val));
  else if (sizeof (gm_dp_t) == 4)
    __gm_pio_write_u32 (is, (gm_u32_n_t *) ptr, gm_hton_u32 ((gm_u32_t) val));
  else
    gm_abort ();
}

static gm_inline void
__gm_kton_up (struct gm_instance_state *is, volatile gm_up_n_t * ptr,
	      gm_up_t val)
{
  if (sizeof (gm_up_t) == 8)
    __gm_pio_write_u64 (is, (gm_u64_n_t *) ptr, gm_hton_u64 ((gm_u64_t) val));
  else if (sizeof (gm_up_t) == 4)
    __gm_pio_write_u32 (is, (gm_u32_n_t *) ptr, gm_hton_u32 ((gm_u32_t) val));
  else
    gm_abort ();
}

static gm_inline void
__gm_kton_u16 (struct gm_instance_state *is, volatile gm_u16_n_t * ptr,
	       gm_u16_t val)
{
  __gm_pio_write_u16 (is, ptr, gm_hton_u16 (val));
}

static gm_inline void
__gm_kton_s16 (struct gm_instance_state *is, volatile gm_s16_n_t * ptr,
	       gm_s16_t val)
{
  __gm_pio_write_u16 (is, (gm_u16_n_t *) ptr, gm_hton_u16 ((gm_u16_t) val));
}

static gm_inline void
__gm_kton_u8 (struct gm_instance_state *is, volatile gm_u8_n_t * ptr,
	      gm_u8_t val)
{
  __gm_pio_write_u8 (is, ptr, gm_hton_u8 (val));
}

static gm_inline void
__gm_kton_s8 (struct gm_instance_state *is, volatile gm_s8_n_t * ptr,
	      gm_s8_t val)
{
  __gm_pio_write_u8 (is, (gm_u8_n_t *) ptr, gm_hton_u8 ((gm_u8_t) val));
}

/****************************************************************
 * Symbolic PIO
 ****************************************************************/

/* These functions provide the only interface for accessing the lanai
   globals, to ensure that it is done correctly. */

/********************************
 * lanai globals
 ********************************/

/****************
 * reading
 ****************/

#define gm_ntok(is, fld, sfx)						\
__gm_ntok_##sfx (is, &(((gm_lanai_globals_t *)				\
			is->lanai.globals)->fld))

#define gm_read_lanai_global_u64(is, fld) gm_ntok (is, fld, u64)
#define gm_read_lanai_global_u32(is, fld) gm_ntok (is, fld, u32)
#define gm_read_lanai_global_s32(is, fld) gm_ntok (is, fld, s32)
#define gm_read_lanai_global_lp(is, fld) gm_ntok (is, fld, lp)
#define gm_read_lanai_global_up(is, fld) gm_ntok (is, fld, up)
#define gm_read_lanai_global_dp(is, fld) gm_ntok (is, fld, dp)
#define gm_read_lanai_global_u16(is, fld) gm_ntok (is, fld, u16)
#define gm_read_lanai_global_s16(is, fld) gm_ntok (is, fld, s16)
#define gm_read_lanai_global_u8(is, fld) gm_ntok (is, fld, u8)

/****************
 * writing
 ****************/

#define gm_kton(is, fld, val, sfx)					\
__gm_kton_##sfx (is,							\
		 &(((gm_lanai_globals_t *) is->lanai.globals)->fld),	\
		 val)

#define gm_write_lanai_global_u64(is, fld, val) gm_kton (is, fld, val, u64)
#define gm_write_lanai_global_u32(is, fld, val) gm_kton (is, fld, val, u32)
#define gm_write_lanai_global_s32(is, fld, val) gm_kton (is, fld, val, s32)
#define gm_write_lanai_global_lp( is, fld, val) gm_kton (is, fld, val,  lp)
#define gm_write_lanai_global_up( is, fld, val) gm_kton (is, fld, val,  up)
#define gm_write_lanai_global_dp( is, fld, val) gm_kton (is, fld, val,  dp)
#define gm_write_lanai_global_u16(is, fld, val) gm_kton (is, fld, val, u16)
#define gm_write_lanai_global_s16(is, fld, val) gm_kton (is, fld, val, s16)
#define gm_write_lanai_global_u8( is, fld, val) gm_kton (is, fld, val,  u8)
#define gm_write_lanai_global_s8( is, fld, val) gm_kton (is, fld, val,  s8)

/********************************
 * special registers
 ********************************/

/****************
 * reading
 ****************/

#define gm_rd_spec(is, fld, sfx)					\
__gm_ntok_##sfx (is, &(((gm_lanai_special_registers_t *)		\
			is->lanai.special_regs)				\
		       ->fld))

#define gm_read_lanai_special_reg_u64(is, fld) gm_rd_spec (is, fld, u64)
#define gm_read_lanai_special_reg_u32(is, fld) gm_rd_spec (is, fld, u32)
#define gm_read_lanai_special_reg_s32(is, fld) gm_rd_spec (is, fld, s32)
#define gm_read_lanai_special_reg_lp(is, fld) gm_rd_spec (is, fld, lp)
#define gm_read_lanai_special_reg_up(is, fld) gm_rd_spec (is, fld, up)
#define gm_read_lanai_special_reg_dp(is, fld) gm_rd_spec (is, fld, dp)
#define gm_read_lanai_special_reg_u16(is, fld) gm_rd_spec (is, fld, u16)
#define gm_read_lanai_special_reg_u8(is, fld) gm_rd_spec (is, fld, u8)
#define gm_read_lanai_special_reg_s8(is, fld) gm_rd_spec (is, fld, s8)

/****************
 * writing
 ****************/

#define gm_wr_spec(is, fld, val, sfx)					\
__gm_kton_##sfx (is,							\
		 &(((gm_lanai_special_registers_t *)			\
		    is->lanai.special_regs)				\
		   ->fld),						\
		 val)

#define gm_write_lanai_special_reg_u64(is,fld,val) gm_wr_spec (is,fld,val,u64)
#define gm_write_lanai_special_reg_u32(is,fld,val) gm_wr_spec (is,fld,val,u32)
#define gm_write_lanai_special_reg_s32(is,fld,val) gm_wr_spec (is,fld,val,s32)
#define gm_write_lanai_special_reg_lp( is,fld,val) gm_wr_spec (is,fld,val, lp)
#define gm_write_lanai_special_reg_up( is,fld,val) gm_wr_spec (is,fld,val, up)
#define gm_write_lanai_special_reg_dp( is,fld,val) gm_wr_spec (is,fld,val, dp)
#define gm_write_lanai_special_reg_u16(is,fld,val) gm_wr_spec (is,fld,val,u16)
#define gm_write_lanai_special_reg_u8( is,fld,val) gm_wr_spec (is,fld,val, u8)

#define gm_bcopy_to_lanai_globals(is, from, to, len)			\
__gm_kton_bcopy (is,							\
		 from,							\
		 (void *) &((gm_lanai_globals_t *)			\
			    is->lanai.globals)->to,			\
		 len)

#define gm_bcopy_from_lanai_globals(is, from, to, len)			\
__gm_kton_bcopy (is,							\
	         (void *) &((gm_lanai_globals_t *)			\
	         	    is->lanai.globals)->from,			\
	         to,							\
	         len)

#define gm_read_lanai_global_byte_array(is, from, to)			\
__gm_ntok_bcopy (is,							\
		 (void *) &((gm_lanai_globals_t *)			\
			    is->lanai.globals)->from,			\
		 to,							\
		 sizeof (((gm_lanai_globals_t *)			\
			  is->lanai.globals)->from))

#define gm_write_lanai_global_byte_array(is, from, to)			\
gm_bcopy_to_lanai_globals (is,						\
			   from,					\
			   to,						\
			   sizeof (((gm_lanai_globals_t *)		\
				    is->lanai.globals)->to))

#define gm_strncpy_to_lanai_globals(is, to, from)			\
__gm_kton_strncpy (is,							\
		   (gm_u8_n_t *) &((gm_lanai_globals_t *)		\
				   is->lanai.globals)->to,		\
		   from,						\
		   sizeof (((gm_lanai_globals_t *)			\
			    is->lanai.globals)->to))

#define gm_clear_lanai_global(is, ptr)					\
__gm_kton_bzero (is,							\
		 (void *) &((gm_lanai_globals_t *)			\
			    is->lanai.globals)->ptr,			\
		 sizeof (((gm_lanai_globals_t *)			\
			  is->lanai.globals)->ptr))

/****************************************************************
 * prototypes
 ****************************************************************/

void __gm_kton_bcopy (struct gm_instance_state *is,
		      void *_from, volatile gm_u8_n_t * _to,
		      gm_size_t len);
void __gm_ntok_bcopy (struct gm_instance_state *is, volatile gm_u8_n_t * from,
		      void *to, gm_size_t len);
void __gm_nton_bcopy (struct gm_instance_state *is, volatile gm_u8_n_t * from,
		      volatile gm_u8_n_t * to, gm_size_t len);
gm_u8_n_t *__gm_kton_strncpy (struct gm_instance_state *is,
			      volatile gm_u8_n_t * to, char *from,
			      gm_size_t len);
char *__gm_ntok_strncpy (struct gm_instance_state *is, char *_to,
			 volatile gm_u8_n_t * from, gm_size_t len);
void __gm_kton_bzero (struct gm_instance_state *, volatile gm_u8_n_t * ptr,
		      gm_size_t len);

#endif /* _gm_pio_h_ */

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
