/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

#ifndef _gm_impl_h_
#define _gm_impl_h_

/**********************************************************************
 * This file includes internal definitions for the portable GM driver
 * code in (gm.c).
 **********************************************************************/

#if !GM_KERNEL
#error GM_KERNEL not defined
#endif

/* Import the architecture-specific data structures that we need to
   build our generic data structures. */

#include "gm_config.h"
#include "gm_arch_types.h"
#include "gm_internal.h"
#include "gm_io.h"

#include "gm_lanai.h"

#include "gm_debug_instance_state.h"

/************
 * Inferred definitions
 ************/

/* GM_SUPPORT_PCI */

#if GM_SUPPORT_PCI_REV_1
#  define GM_SUPPORT_PCI 1
#elif GM_SUPPORT_PCI_REV_2
#  define GM_SUPPORT_PCI 1
#elif GM_SUPPORT_PCI_REV_3
#  define GM_SUPPORT_PCI 1
#else
#  define GM_SUPPORT_PCI 0
#endif

/************
 * Typedefs
 ************/

/* Values returned by gm_signal_sleep and gm_timed_sleep */
   
typedef int gm_arch_sleep_status_t;

/* max number of pages = 2^(GM_NUM_PAGE_TABLE_SEGMENTS-1)*2^(PAGE_BITS-4) */
#if GM_SIZEOF_UP_T == 4
#define GM_NUM_PAGE_TABLE_SEGMENTS 20
#elif GM_SIZEOF_UP_T == 8
#define GM_NUM_PAGE_TABLE_SEGMENTS 40
/* with 8k pages -> 2^(39+9+13=61) bytes manageable: enough for a very long time :-) */
#else
#error Unsupported GM_SIZEOF_UP_T
#endif

typedef struct gm_page_hash_table
  {
    gm_s32_t max_index;
    gm_u32_t segment_cnt;
    gm_u32_t filled_entries;
    struct 
    {
      gm_u32_t entry_cnt;
      gm_pte_t *entry;
      void **user_entry;
      gm_arch_dma_region_t dma_region;
    } segment[GM_NUM_PAGE_TABLE_SEGMENTS];
    gm_arch_dma_region_t bogus_sdma_region;
    gm_arch_dma_region_t bogus_rdma_region;
    gm_arch_sync_t sync;
    gm_u32_t total_refs;
    struct gm_hash *page_lock_hash;
  }
gm_page_hash_table_t;

/* The driver state for an instance (a device) */

/* Flags for gm_instance_state.flags                                      */
#define GM_INSTANCE_64BIT_DMA_ADDR_OK 1    /* OK to use 64-bit DMA addrs. */

typedef struct gm_instance_state
  {
    unsigned int id;		/* unit number.         */
    unsigned int flags;		/* Flags, defined above */

  /*************
   * LANai access fields
   *************/

    /* Record of lanai mappings */

    struct
      {
	gm_myrinet_eeprom_t eeprom;
	/* This pointer is opaque to ensure that the special registers
	   are referenced only through the pio.h interface. */
	void *special_regs;
	volatile gm_u32_n_t *control_regs;
	volatile gm_u8_n_t *sram;
	/* The globals are not a structure to insure that they are
	   referenced only throught the pio.h interface. */
	volatile void *globals;	/* Set iff MCP running. */
	gm_u32_t running;
      }
    lanai;

    /* Pointer to LANai interface card, mapped into kernel memory. */

    void *board_base;
    gm_u32_t board_span;

    /* Lanai synchronization */

    volatile gm_s32_t pause_rqst; /* sized to match LANai fields */
    volatile gm_s32_t pause_ack; /* sized to match LANai fields. */
    gm_arch_sync_t pause_sync;
  
    gm_page_hash_table_t page_hash; /* page hash table */

    /* Information specific to they type of adapter */

    /* The following structure holds bus specific information */
    union
    {
#if GM_SUPPORT_PCI
      struct {
	gm_pci_config_t config;
      } pci;
#endif
      gm_u32_t _hack_to_ensure_nonempty;
    } ifc;

    /* Function table */

    void (*_set_EIMR)          (struct gm_instance_state *, gm_u32_t);
    void (*set_ISR)            (struct gm_instance_state *, gm_u32_t);
    gm_u32_t (*get_EIMR)       (struct gm_instance_state *);
    gm_u32_t (*get_ISR)        (struct gm_instance_state *);

    gm_u32_t pausible;		/* cached value of lanai EIMR */
    
    struct
    {
      struct
      {
	gm_arch_dma_region_t dma_region;
	gm_unique_id_64_t *volatile_ethernet_addr;
	struct gm_hash *cache_hash;
      } addr_tab;
      struct
      {
	gm_ethernet_recv_token_t *start;
	gm_ethernet_recv_token_t *slot;
	gm_ethernet_recv_token_t *limit;
      } recv_token_queue;
    } ethernet;

    struct 
    {
      gm_arch_dma_region_t dma_region;
      char (*entry)[GM_MAX_HOST_NAME_LEN];
    } name_table;
    
    unsigned max_node_id;	/* Type must match gm_port_t.max_node_id */

    struct gm_hash *port_hash;

    gm_arch_instance_info_t arch; /* Architecture-specific field. */


    /* A bitmap to sanity check all DMA access from the LANAI, before
       LANAI is initialized the bitmap variables point to the
       bitmap_copy, after bitmap_copy is not used anymore, variable
       point to LANAI memory. */

#if GM_DMA_PAGE_BITMAP_COPY_NEEDED
    GM_BITMAP_DECL (dma_page_bitmap_copy, GM_MAX_HOST_DMA_PAGES);
#endif
    GM_BITMAP_DECL ((*dma_page_bitmap), GM_MAX_HOST_DMA_PAGES);

    /* record or minor device numbers for clone devices */
    gm_arch_minor_t clone_minor;
    gm_arch_minor_t privileged_clone_minor;

    void (*ethernet_recv_intr_callback) (void *, unsigned int, gm_u16_t csum);
    void *ethernet_recv_intr_context;

    void (*ethernet_sent_intr_callback) (void *);
    void *ethernet_sent_intr_context;

    unsigned long cache_line_size;
    gm_pid_t opener_pids[GM_NUM_PORTS];
    
    /* in gm_uc.c consecutive1s keeps track of how many 1 bits we've send in a row */
    int consecutive1s; 

    gm_u32_t bus_width;
    gm_u32_t bus_rate;
  }
gm_instance_state_t;

/* The driver state for a user port. */

typedef struct gm_port_state
  {
    /************ fields for half-opened ports in gm_minor.c */

    gm_instance_state_t *instance;
    gm_arch_sync_t sync;
    unsigned int privileged;
    gm_arch_minor_t minor;
    
    /************ fields initialized in gm_port_state_open() */

    char opened;
    char is_a_kernel_port;
    gm_u32_t id;
    gm_s32_t compatibility_mode;
    gm_u32_t min_dma_mem;
    gm_u32_t max_dma_mem;
    gm_mapping_specs_t mappings;

    /* Fields for sleeping */
    gm_arch_sync_t sleep_sync;

    struct
      {
	gm_size_t len;	/* total of all allocated segments */
	unsigned int segment_cnt;
	struct gm_segment_desc
	  {
	    gm_size_t len;
	    char *data;
	    gm_arch_dma_region_t dma_region;
	  }
	segment[32];
      }
    copy_block;
    gm_arch_dma_region_t recv_queue_region;

    gm_arch_port_info_t arch;	/* Architecture-specific field. */

    /* records of the input and output buffers passed to gm_ioctl */
    void *in_buff;
    void *out_buff;

    unsigned long mem_reg_len;
    gm_pid_t opener_pid;
    gm_u8_t  opener_string[GM_OPENER_STRING_MAX_LEN];
  }
gm_port_state_t;

/************************************************************************
 * Inline functions
 ************************************************************************/

#define GM_MAX(a, b) ((a)>(b) ? (a) : (b))
#define GM_MIN(a, b) ((a)<(b) ? (a) : (b))

#if 0
static gm_inline gm_s32_t
gm_max (gm_s32_t a, gm_s32_t b)
{
  return a>b ? a : b;
}

static gm_inline gm_s32_t
gm_min (gm_s32_t a, gm_s32_t b)
{
  return a<b ? a : b;
}
#endif

/************************************************************************
 * Prototypes
 ************************************************************************/

void *_gm_alloc_pages (unsigned long alloc_len);
void _gm_free_pages (void *addr, unsigned long alloc_len);

/* performs gm_copy or calls gm_arch_copyin() or gm_arch_copyout() as
   appropriate. */

gm_status_t gm_copyout (gm_port_state_t *ps,
			void *from, void *to, gm_size_t len);
gm_status_t gm_copyin (gm_port_state_t *ps,
		       void *from, void *to, gm_size_t len);

gm_s32_t gm_interrupting (gm_instance_state_t *is);
void gm_handle_claimed_interrupt (gm_instance_state_t *is);
gm_s32_t gm_intr (gm_instance_state_t * is);

void gm_init_mapping_spec (gm_port_state_t *ps);

void gm_disable_lanai (gm_instance_state_t * is);

void gm_free_page_hash_table (gm_instance_state_t * is);

void gm_disable_port_DMAs (gm_port_state_t * ps);

void gm_free_copy_block (gm_port_state_t * ps);

gm_status_t gm_ioctl (gm_port_state_t *ps, unsigned int cmd,
		      void *in_buffer, unsigned int in_buff_len,
		      void *out_buff, unsigned int out_buff_len,
		      unsigned *copied_out);

gm_status_t gm_minor_ioctl (gm_arch_minor_t minor, unsigned int gm_ioctl_cmd,
			    void *in_buffer, unsigned int in_buff_len,
			    void *out_buff, unsigned int out_buff_len,
			    unsigned *copied_out);

gm_status_t gm_expand_copy_block (gm_port_state_t * ps,
				  gm_u32_t length);

gm_status_t gm_add_segment_to_page_table (gm_port_state_t * ps,
					  gm_up_t user_addr,
					  gm_u32_t offset,
					  gm_u32_t len);

gm_status_t gm_prepare_to_mmap (gm_port_state_t * ps,
				gm_offset_t off,
				gm_size_t len,
				unsigned int requested_permissions);

gm_status_t gm_finish_mmap (gm_port_state_t * ps, gm_offset_t off,
			    gm_size_t len, gm_up_t addr);

int gm_mapping_in_io_space (gm_port_state_t *ps, gm_offset_t off);
gm_status_t gm_mmap (gm_port_state_t * ps, gm_offset_t off, void **);

gm_status_t gm_lanai_reset_on (gm_instance_state_t *is);
gm_status_t gm_lanai_reset_off (gm_instance_state_t *is);
gm_status_t gm_board_reset_on (gm_instance_state_t *is);
gm_status_t gm_board_reset_off (gm_instance_state_t *is);
gm_status_t gm_ebus_reset_on (gm_instance_state_t *is);
gm_status_t gm_ebus_reset_off (gm_instance_state_t *is);
gm_status_t gm_enable_interrupts (gm_instance_state_t *is);
gm_status_t gm_disable_interrupts (gm_instance_state_t *is);

gm_status_t gm_copy_eeprom (gm_instance_state_t *is);
gm_status_t gm_instance_init (gm_instance_state_t *is, gm_u32_t unit,
			      enum gm_bus_type type);
void gm_instance_finalize (gm_instance_state_t *is);
gm_instance_state_t *gm_instance_for_id (unsigned int id);

gm_status_t gm_alloc_page_hash_table (gm_instance_state_t * is);
    
gm_status_t kernel_enable_raw_receives (gm_port_state_t * ps);
    
gm_status_t kernel_set_acceptable_sizes (gm_port_state_t * ps,
					 enum gm_priority priority,
					 gm_u32_t mask);
gm_status_t gm_read_correct_sequence_bit (gm_instance_state_t *is,
                                          int                 *correct);
#if GM_DEBUG_INSTANCE_STATE
extern void gm_dump_instance (gm_instance_state_t * is);
#endif


void gm_pause_lanai (gm_instance_state_t * is);
void gm_unpause_lanai (gm_instance_state_t * is);
   
gm_status_t __gm_init_port_state (void); /* used by gm_init() only */
void __gm_finalize_port_state (void); /* used by gm_init() only */
gm_status_t gm_port_state_open (gm_port_state_t *ps, unsigned int id);
void gm_port_state_close (gm_port_state_t *ps);

void gm_set_EIMR (gm_instance_state_t * is, gm_u32_t val);

/* defined in libgm/gm_open.c */
gm_status_t gm_init_opened_port_state (gm_port_state_t *ps);

/* minor number management functions */

gm_status_t gm_minor_alloc (gm_instance_state_t *is, gm_arch_minor_t *minor);
void gm_minor_free (gm_arch_minor_t minor);
gm_status_t gm_minor_reserve (gm_instance_state_t *is, gm_arch_minor_t minor);
gm_instance_state_t *gm_minor_get_instance_state (gm_arch_minor_t minor);
gm_port_state_t *gm_minor_get_port_state (gm_arch_minor_t minor);
gm_status_t gm_minor_set_port_num (gm_arch_minor_t minor, unsigned int id);
void gm_minor_set_instance_state (gm_arch_minor_t minor,
				  gm_instance_state_t *is);
void gm_minor_mutex_enter (gm_arch_minor_t minor);
void gm_minor_mutex_exit (gm_arch_minor_t minor);
#if 0
gm_arch_ioctl_context_t gm_minor_get_arch_ioctl_context (gm_arch_minor_t
							 minor);
void gm_minor_set_arch_ioctl_context (gm_arch_minor_t minor,
				      gm_arch_ioctl_context_t context);
#endif

/* port state management functions */

gm_port_state_t *__gm_port_state_for_id (gm_instance_state_t *is,
					 unsigned int port);
gm_port_state_t *gm_acquire_port_state_for_id (gm_instance_state_t *is,
					       unsigned int port);
gm_port_state_t *gm_acquire_port_state_for_ids (unsigned int instance,
						unsigned int port);
void gm_release_port_state (gm_port_state_t *ps);

/* microcontroller control operations */

int __gm_cop_send (gm_instance_state_t *is, int d);
int __gm_cop_receive (gm_instance_state_t *is);
int __gm_cop_wakeup (gm_instance_state_t *is);
int __gm_cop_end (gm_instance_state_t *is);

gm_status_t gm_kernel_sleep (gm_port_state_t * ps);

/************************************************************************
 * Architecture-specific functions.
 ************************************************************************/

/************
 * The following must be implemented in gm_arch.c to support the generic
 * driver code in gm.c and gm_instance.c.
 ************/

/* gm_arch_kernel_malloc flags */

#define GM_KERNEL_MALLOC_PAGEABLE 1

void *gm_arch_kernel_malloc (unsigned long len, int flags);
void gm_arch_kernel_free (void *ptr);

/* sync */

void gm_arch_sync_init (gm_arch_sync_t *s, gm_instance_state_t *is);
void gm_arch_mutex_enter (gm_arch_sync_t *s);
void gm_arch_mutex_exit (gm_arch_sync_t *s);
void gm_arch_sync_destroy (gm_arch_sync_t *s);
void gm_arch_sync_reset (gm_arch_sync_t *s);
void gm_arch_wake (gm_arch_sync_t * s);
gm_arch_sleep_status_t gm_arch_signal_sleep (gm_arch_sync_t *s);
gm_arch_sleep_status_t gm_arch_timed_sleep (gm_arch_sync_t *s, int seconds);

/* DMA regions */     

/************
 * This declares gm_register_page_function_t as the type of the function
 * pointer passed to gm_arch_dma_region_alloc.  The use of this type is
 * backward compatible.  The function definition for gm_arch_dma_region_alloc()
 * in some arch's has been changed to use it.  When convenient, it would be a
 * good thing to update the remaining ones.
 ************/
typedef gm_status_t (*gm_register_page_function_t) (
                                            void *,    /* arbitrary arg    */
			 	            gm_dp_t,   /* host dma address */
				            gm_u32_t); /* page number      */


gm_status_t gm_arch_dma_region_alloc (gm_instance_state_t * is,
				      gm_arch_dma_region_t * r,
				      gm_size_t len, gm_u32_t flags,
				      gm_register_page_function_t reg,
				      void *user_arg);
void gm_arch_dma_region_free (gm_arch_dma_region_t * r);
gm_dp_t gm_arch_dma_region_dma_addr (gm_arch_dma_region_t * r);
gm_dp_t gm_arch_dma_region_dma_addr_advance (gm_arch_dma_region_t * r);
void * gm_arch_dma_region_kernel_addr (gm_arch_dma_region_t * r);
gm_status_t gm_arch_dma_region_sync (gm_arch_dma_region_t * r, int command);

/* memory mapping (LANai memory into kernel) */

gm_status_t gm_arch_map_io_space (gm_instance_state_t *is, gm_u32_t offset,
				  gm_u32_t len, void **kaddr);
void gm_arch_unmap_io_space (gm_instance_state_t *is, gm_u32_t offset,
			gm_u32_t len, void **kaddr);

/* memory mapping (kernel memory into user space) */

gm_status_t gm_arch_mmap_contiguous_segment (gm_port_state_t *ps,
					     void * kaddr,
					     unsigned long blockSize,
					     gm_up_t *vaddr);
void gm_arch_munmap_contiguous_segments (gm_port_state_t *ps);

/* misc */

gm_status_t gm_arch_gethostname (char *ptr, int len);
void gm_arch_spin (gm_instance_state_t *is, gm_u32_t usecs);
gm_status_t gm_arch_get_page_len (unsigned long *page_len);

/* port state initialization/finalization */

gm_status_t gm_arch_port_state_init (gm_port_state_t *ps);
gm_status_t gm_arch_port_state_open (gm_port_state_t *ps);
void gm_arch_port_state_close (gm_port_state_t *ps);
void gm_arch_port_state_fini (gm_port_state_t *ps);

/* sbus */

gm_s32_t gm_arch_dma_region_status (gm_arch_dma_region_t * r);
gm_status_t gm_arch_sbus_copy_eeprom (gm_instance_state_t *is);

/* pci */

gm_status_t gm_arch_read_pci_config_32 (gm_instance_state_t *is,
					gm_offset_t offset,
					gm_u32_t *value);
gm_status_t gm_arch_write_pci_config_32 (gm_instance_state_t *is,
					 gm_offset_t offset,
					 gm_u32_t value);
gm_status_t gm_arch_read_pci_config_16 (gm_instance_state_t *is,
					gm_offset_t offset,
					gm_u16_t *value);
gm_status_t gm_arch_write_pci_config_16 (gm_instance_state_t *is,
					 gm_offset_t offset,
					 gm_u16_t value);
gm_status_t gm_arch_read_pci_config_8 (gm_instance_state_t *is,
				       gm_offset_t offset,
				       gm_u8_t *value);
gm_status_t gm_arch_write_pci_config_8 (gm_instance_state_t *is,
					gm_offset_t offset,
					gm_u8_t value);


#if GM_CAN_REGISTER_MEMORY
gm_status_t gm_arch_lock_user_buffer_page (gm_instance_state_t *is,
					   gm_up_t uaddr,
					   gm_dp_t *dma_addr,
					   gm_arch_page_lock_t *lock);
void gm_arch_unlock_user_buffer_page (gm_arch_page_lock_t *lock);
gm_u32_t gm_dereference_mapping (gm_instance_state_t *is,
				 gm_up_t user_vma,
				 gm_u32_t port);

gm_u32_t gm_refs_to_mapping_in_page_table (gm_instance_state_t *is,
					   gm_up_t user_vma,
					   gm_u32_t port);

gm_status_t
gm_add_mapping_to_page_table (gm_instance_state_t *is,
			      gm_up_t user_vma,
			      unsigned int port,
			      gm_dp_t dma_addr);


#endif /* GM_CAN_REGISTER_MEMORY */

#if GM_USE_DEFAULT_INTERRUPT_HANDLER
gm_port_state_t *gm_arch_port_for_id (gm_instance_state_t * is, unsigned port);
#endif

#if !GM_COHERENT
gm_status_t gm_arch_recv_queue_update (struct gm_port *p, int slot_num);
#endif

void *gm_arch_page_malloc (unsigned long len);
void gm_arch_page_free (void *addr);

/* architecture-specific functions to copy data into and out of the kernel */

gm_status_t gm_arch_copyout (gm_port_state_t *ps,
			     void *from, void *to, gm_size_t len);
gm_status_t gm_arch_copyin (gm_port_state_t *ps,
			    void *from, void *to, gm_size_t len);

#if !GM_ENABLE_VM
gm_dp_t gm_arch_no_VM_dma_bits_to_clear (gm_instance_state_t *is);
gm_dp_t gm_arch_no_VM_dma_bits_to_set (gm_instance_state_t *is);
#endif /* !GM_ENABLE_VM */

/* architecture functions for an atomic counter */
#if GM_HAS_ATOMIC_T
#if 0
/* Does not appear to be used. --Glenn */
gm_u32_t gm_arch_atomic_preinc(gm_atomic_t *i);
#endif
void gm_arch_atomic_set(gm_atomic_t *i, gm_u32_t read);
gm_u32_t gm_arch_atomic_read(gm_atomic_t *i);
#endif

/* directcopy support */

#if GM_ENABLE_DIRECTCOPY
gm_status_t gm_arch_directcopy_get(void * source_addr, 
				   void * target_addr, 
				   ulong length,
				   uint pid_source);
#endif

/* This should never get called in production drivers.  For debugging
   only. */

void gm_arch_abort (void);

/****************
 * Optional architecture-specific hooks:
 ****************/

#ifndef GM_ARCH_RECV_QUEUE_UPDATE
#define GM_ARCH_RECV_QUEUE_UPDATE(p)
#endif

/****************************************************************
 * Do not put new prototypes here.
 ****************************************************************/

#endif /* _gm_impl_h_ */
