/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

#if !GM_KERNEL
#error GM_KERNEL is not defined for kernel compilations
#endif

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_debug_lanai_dma.h"
#include "gm_debug_host_dma.h"
#include "gm_debug_mem_register.h"
#include "gm_debug_page_table.h"
#include "gm_debug_port_state.h"
#include "gm_enable_galvantech_workaround.h"
#include "gm_enable_paranoid_interrupts.h"
#include "gm_enable_security.h"
#include "gm_impl.h"
#include "gm_internal.h"
#include "gm_page_hash.h"
#include "gm_pio.h"
#include "gm_pte.h"
#include "gm_trace.h"

/************
 * GCC compatibility functions.
 ************/

#if GM_NEED_MEMCPY
/* Needed to allow gcc structure copy to work. */
void *
memcpy (void *to, const void *from, size_t bytes)
{
  bcopy ((void *) from, to, bytes);
  return to;
}
#endif

#if GM_NEED_MEMSET
/* Needed for gcc struct constructors. */
void *
memset (void *s, int c, size_t n)
{
  while (n--)
    ((char *) s)[n] = c;
  return s;
}
#endif

/************
 * easy-to-use copyin/copyout.
 ************/

gm_status_t
gm_copyout (gm_port_state_t * ps, void *from, void *to, gm_size_t len)
{
  gm_instance_state_t *is;
  gm_status_t ret;
  void *buf = 0;

  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 11,
	    ("gm_copyOUT(%p, from=%p, to=%p, len=%d)\n", ps, from, to, len));

  /* Avoid partword I/O, if needed. */

  if (GM_NO_PARTWORD_PIO_READ)
    {
      /* If copying data from the LANai SRAM, use special PIO call to copy
         the data. */

      is = ps->instance;
      gm_assert (is);
      if ((char *) from >= (char *) is->lanai.sram
	  && (char *) from < ((char *) is->lanai.sram
			      + is->lanai.eeprom.lanai_sram_size))
	{
	  buf = gm_malloc (len);
	  if (!buf)
	    {
	      GM_RETURN_STATUS (GM_OUT_OF_MEMORY);
	    }
	  /*printf("Doing special copyout for no_pio_read\n"); */
	  __gm_ntok_bcopy (is, from, buf, len);
	  from = buf;
	}
    }

  if (to == ps->out_buff)
    {
      ret = gm_arch_copyout (ps, from, to, len);
    }
  else
    {
      gm_bcopy (from, to, len);
      ret = GM_SUCCESS;
    }

  /* Free any buffer used to avoid partword PIO. */

  if (buf)
    {
      gm_free (buf);
    }

  GM_RETURN_STATUS (ret);
}

gm_status_t
gm_copyin (gm_port_state_t * ps, void *from, void *to, gm_size_t len)
{
  gm_assert ((char *) to < (char *) ps->instance->lanai.sram
	     || (char *) to >= ((char *) ps->instance->lanai.sram
				+
				ps->instance->lanai.eeprom.lanai_sram_size));

  GM_PRINT (GM_PRINT_LEVEL >= 11,
	    ("gm_copyIN(%p, from=%p, to=%p, len=%d)\n", ps, from, to, len));
  if (from == ps->in_buff)
    return gm_arch_copyin (ps, from, to, len);
  gm_bcopy (from, to, len);
  return GM_SUCCESS;
}

/***********************************************************************
 * Utility functions
 ***********************************************************************/

/* Note: May be called by interrupt handler. */

void
gm_disable_lanai (gm_instance_state_t * is)
{
  GM_CALLED ();

  gm_assert (is);
  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Disabling the LANai\n"));

  if (is->lanai.running)
    {
      gm_assert (is->lanai.special_regs);

      gm_disable_interrupts (is);

      /* Reset the LANai, unless an assertion failed (so the LANai will
         flash SOS */

      if (is->lanai.globals == 0
	  || (gm_read_lanai_global_lp (is, interrupt.failed_assertion.file)
	      == 0)
	  || (gm_read_lanai_global_u32 (is, interrupt.failed_assertion.line)
	      == 0))
	{
	  gm_lanai_reset_on (is);
	}

      gm_set_EIMR (is, 0);
      is->lanai.running = 0;
      GM_STBAR ();
    }

  GM_RETURN_NOTHING ();
}

/*----------------------------------------------------------------------
 - Pausing the LANai MCP
 ----------------------------------------------------------------------*/

#define GM_DEBUG_PAUSE 0

static void
gm_pause_lanai__report_error (gm_instance_state_t * is)
{
  gm_lp_t lanai_current_handler;
  gm_lp_t lanai_last_dispatch;

  lanai_last_dispatch = gm_read_lanai_global_lp (is, last_dispatch);
  if (lanai_last_dispatch)
    {
      char last_dispatch[GM_MAX_FIRMWARE_STRING_LEN + 1];

      __gm_ntok_strncpy (is, last_dispatch,
			 is->lanai.sram + lanai_last_dispatch,
			 GM_MAX_FIRMWARE_STRING_LEN);
      last_dispatch[GM_MAX_FIRMWARE_STRING_LEN] = 0;
      GM_INFO (("Last dispatch was at %s\n", last_dispatch));
    }
  lanai_current_handler = gm_read_lanai_global_lp (is, current_handler);
  if (lanai_current_handler)
    {
      char current_handler[GM_MAX_FIRMWARE_STRING_LEN + 1];
      char current_handler_extra[GM_MAX_FIRMWARE_STRING_LEN + 1];

      __gm_ntok_strncpy (is, current_handler,
			 is->lanai.sram + lanai_current_handler,
			 GM_MAX_FIRMWARE_STRING_LEN);
      current_handler[GM_MAX_FIRMWARE_STRING_LEN] = 0;
      __gm_ntok_strncpy
	(is, current_handler_extra,
	 is->lanai.sram + gm_read_lanai_global_lp (is, current_handler_extra),
	 GM_MAX_FIRMWARE_STRING_LEN);
      current_handler_extra[GM_MAX_FIRMWARE_STRING_LEN] = 0;

      GM_INFO (("current handler is %s (%s)\n",
		current_handler, current_handler_extra));
    }
}

/* Obtain exclusive access to LANai memory by requesting that the
   LANai control program pause and acknowlege that it is paused via an
   interrupt.  4-phase acks are used. */
void
gm_pause_lanai (gm_instance_state_t * is)
{
  GM_CALLED ();

  /* Only one thread may pause the LANai at a time. */

  GM_PRINT (GM_DEBUG_PAUSE,
	    ("Entering pause mutex in sync %p.\n", &is->pause_sync));

  gm_arch_mutex_enter (&is->pause_sync);
  GM_PRINT (GM_DEBUG_PAUSE, ("Entered pause mutex.\n"));

  /* If LANai not running, no need to pause. */

  if (!is->pausible  || !is->lanai.running)
    {
      GM_PRINT (1||GM_DEBUG_PAUSE, ("LANai not running.  Using shortcut.\n"));
      is->pause_rqst = 1;
      is->pause_ack = 1;
      return;
    }

  /* Verify not violating protocol. */

  gm_assert (is->lanai.running);
  gm_assert (is->lanai.globals);
  gm_assert (is->pause_rqst == 0);
  gm_assert (is->pause_ack == 0);
  gm_assert (gm_read_lanai_global_u32 (is, magic) == 0xcafebabe);
  GM_PRINT (GM_DEBUG_PAUSE,
	    ("lanai pause_rqst=0x%x\n",
	     (int) gm_read_lanai_global_u32 (is, pause_rqst)));
  gm_assert (gm_read_lanai_global_u32 (is, pause_rqst) == 0);
  /* Tell lanai to pause, caching pause value. */

  GM_PRINT (GM_DEBUG_PAUSE, ("Telling LANai to pause.\n"));
  is->pause_rqst = 1;
  GM_STBAR ();
  gm_write_lanai_global_u32 (is, pause_rqst, 1);
  GM_STBAR ();
  /* readback to force write */
  gm_read_lanai_global_u32 (is, pause_rqst);
  gm_assert (is->pause_rqst != is->pause_ack);

  /* Wait for pause ack to be asserted. */

  GM_PRINT (GM_DEBUG_PAUSE, ("Waiting for LANai to pause.\n"));
  if (gm_arch_timed_sleep (&is->pause_sync, 10) == GM_SLEEP_TIMED_OUT)
    {
      
      GM_NOTE (("LANai[%d] interface not responding (no pause interrupt?)\n",
		is->id));
      /* This display and the similar one at gm_unpause_lanai() should be
       * made into a common subroutine.
       */
      _GM_NOTE (("    isr=0x%x eimr=0x%x\n",
		 is->get_ISR (is), is->get_EIMR (is)));
      _GM_NOTE (("    sram[0] = 0x%x 0x%x 0x%x 0x%x \n",
		 *(unsigned int *) is->lanai.sram,
		 *(unsigned int *)&is->lanai.sram[4],
		 *(unsigned int *)&is->lanai.sram[8],
		 *(unsigned int *)&is->lanai.sram[12]));
      _GM_NOTE (("    DMA len=%d lar=0x%x ear=0x%x%08x\n",
		 gm_read_lanai_global_u32 (is, dma_descriptor.len),
		 gm_read_lanai_global_lp (is, dma_descriptor.lanai_addr),
		 gm_read_lanai_global_u32 (is, dma_descriptor.ebus_addr_high),
		 gm_read_lanai_global_u32 (is, dma_descriptor.ebus_addr_low)));

      GM_NOTE(("     at the moment, pause_rqst = %d\n",
		gm_read_lanai_global_u32 (is, pause_rqst)));
      gm_pause_lanai__report_error (is);
      goto abort;		/* Timed out. */
    }
  gm_assert (is->pause_ack != 0);
  GM_PRINT (GM_DEBUG_PAUSE, ("LANai paused.\n"));

  GM_RETURN_NOTHING ();

abort:
  gm_disable_lanai (is);	/* Please talk to Glenn before disabling */
  GM_RETURN_NOTHING ();
}

/* Release exclusive access to LANai memory. */

void
gm_unpause_lanai (gm_instance_state_t * is)
{
  GM_CALLED ();
  GM_PRINT (GM_DEBUG_PAUSE, ("Unpausing the LANai.\n"));

  /* Make sure all writes have been flushed. */

  GM_STBAR ();

  if (!is->pausible || !is->lanai.running)
    {
      GM_PRINT (1 || GM_DEBUG_PAUSE, ("LANai not running.  Using shortcut.\n"));
      is->pause_rqst = 0;
      is->pause_ack = 0;
      goto done;
    }

  /* Verify not violating protocol. */

  gm_assert (is->lanai.running);
  gm_assert (is->lanai.globals);
  gm_assert (gm_read_lanai_global_u32 (is, magic) == 0xcafebabe);
  gm_assert (is->pause_rqst != 0);
  gm_assert (gm_read_lanai_global_u32 (is, pause_rqst) != 0);
  GM_PRINT (GM_DEBUG_PAUSE,
	    ("lanai pause_rqst=0x%x\n",
	     (int) gm_read_lanai_global_u32 (is, pause_rqst)));
  gm_assert (is->pause_ack != 0);

  /* Signal LANai to resume. */

  gm_assert (is->pause_rqst == is->pause_ack);
  is->pause_rqst = 0;
  GM_STBAR ();
  gm_write_lanai_global_u32 (is, pause_rqst, 0);
  GM_STBAR ();
  /* read back to force write */
  gm_read_lanai_global_u32 (is, pause_rqst);

  /* Wait for LANai to resume. */

  GM_PRINT (GM_DEBUG_PAUSE, ("Waiting for LANai to resume.\n"));

  if (gm_arch_timed_sleep (&is->pause_sync, 10) == GM_SLEEP_TIMED_OUT)
    {
      GM_NOTE (("LANai[%d] interface not responding (no unpause interrupt?)\n",
		is->id));
      /* This display and the similar one at gm_pause_lanai() should be
       * made into a common subroutine.
       */
      _GM_NOTE (("    isr=0x%x eimr=0x%x\n",
		 is->get_ISR (is), is->get_EIMR (is)));
      _GM_NOTE (("    sram[0] = 0x%x 0x%x 0x%x 0x%x \n",
		 *(unsigned int *) is->lanai.sram,
		 *(unsigned int *)&is->lanai.sram[4],
		 *(unsigned int *)&is->lanai.sram[8],
		 *(unsigned int *)&is->lanai.sram[12]));
      _GM_NOTE (("    DMA len=%d lar=0x%x ear=0x%x%08x\n",
		 gm_read_lanai_global_u32 (is, dma_descriptor.len),
		 gm_read_lanai_global_lp  (is, dma_descriptor.lanai_addr),
		 gm_read_lanai_global_u32 (is, dma_descriptor.ebus_addr_high),
		 gm_read_lanai_global_u32 (is, dma_descriptor.ebus_addr_low)));
      _GM_NOTE (("    DMA specials L4 EAR=0x%x LAR=0x%x DMA_CTR=0x%x\n",
		 gm_read_lanai_special_reg_u32 (is, l4.gmEAR),
		 gm_read_lanai_special_reg_u32 (is, l4.LAR),
		 gm_read_lanai_special_reg_u32 (is, l4.DMA_CTR)));
      GM_NOTE(("     at the moment, pause_rqst = %d\n",
		gm_read_lanai_global_u32 (is, pause_rqst)));
      gm_pause_lanai__report_error (is);
      goto abort;		/* Timed out. */
    }

  gm_assert (is->pause_ack == 0);

  GM_PRINT (GM_DEBUG_PAUSE, ("LANai resumed.\n"));

done:
  GM_PRINT (GM_DEBUG_PAUSE, ("Exiting pause mutex %p.\n", &is->pause_sync));
  gm_arch_mutex_exit (&is->pause_sync);
  GM_PRINT (GM_DEBUG_PAUSE, ("Exited pause mutex %p.\n", &is->pause_sync));
  GM_RETURN_NOTHING ();

abort:
  gm_disable_lanai (is);	/* Please talk to Glenn before disabling */
  GM_PRINT (GM_DEBUG_PAUSE, ("Exiting pause mutex.\n"));
  gm_arch_mutex_exit (&is->pause_sync);
  GM_RETURN_NOTHING ();
}

/* See gm_unique_id_to_node_id.c and gm_node_id_to_unique_id.c
   for use of this table. */

/*----------------------------------------------------------------------
 - Mapping spec initialization
 ----------------------------------------------------------------------*/

/* Initialize the "gm_mapping_specs_t mappings" field of a
   gm_port_state_t structure.  This structure describes the what
   offsets should be mapped to map each device resource.  All entries
   in the mapping spec are specified, even if the user will not be
   allowed to map the region due to lack of access privileges.

   Called from PASSIVE_LEVEL, but may be called from any IRQL. */

void
gm_init_mapping_spec (gm_port_state_t * ps)
{
  gm_instance_state_t *is;
  gm_mapping_specs_t *m;
  int port_id;

  GM_CALLED ();

  gm_assert (GM_PAGE_LEN);

  is = ps->instance;
  m = &ps->mappings;
  port_id = ps->id;

  /*** The Device is mapped as follows

    +------------------+ <- offset 0
    | Control regs     | (1 page in device)
    +------------------+
    | Special regs     | (1 page in device)
    +------------------+
    | LANai SRAM       | (many pages in device)
    +------------------+
    | Recv queue       | (several pages host DMAable memory)
    +------------------+
    | copy block       | (many pages host DMAable memory)
    +------------------+
    */

  m->control_regs.offset = 0;
  m->control_regs.len = (gm_u32_t) GM_PAGE_LEN;
  m->control_regs.permissions = (ps->privileged ? GM_MAP_RDWR : 0);

  m->special_regs.offset = (m->control_regs.offset
			    + (gm_offset_t) m->control_regs.len);
  m->special_regs.len = (gm_u32_t) GM_PAGE_LEN;
  m->special_regs.permissions = (ps->privileged ? GM_MAP_RDWR : 0);

  m->sram.offset = (m->special_regs.offset
		    + (gm_offset_t) m->special_regs.len);
  m->sram.len = is->lanai.eeprom.lanai_sram_size;
  m->sram.permissions = (ps->privileged ? GM_MAP_RDWR : 0);
  gm_assert (m->sram.len);

  m->recv_queue.offset = (m->sram.offset + (gm_offset_t) m->sram.len);
  m->recv_queue.len =
#if GM_BC_DRIVER
    ps->compatibility_mode ? 0 :
#endif
    GM_RECV_QUEUE_ALLOC_LEN;
  m->recv_queue.permissions = GM_MAP_RDWR;

  m->copy_block.offset = (m->recv_queue.offset
			  + (gm_offset_t) m->recv_queue.len);
  m->copy_block.len =
#if GM_CAN_REGISTER_MEMORY
    0
#else
    ps->max_dma_mem
#endif
    ;
  m->copy_block.permissions = GM_MAP_RDWR;

/*** Pointers to the pieces of LANai SRAM region of the device */

  /*** The SRAM is divided as follows.

    +------------------+ <- address 0
    | control program  |
    ...
    |                  |
    +------------------+
    | Send queue array | (1 page per port)
    +------------------+
    | Hash piece table | (several pages)
    +------------------+ <- top of SRAM
    */

  m->hash_piece_ptrs.len = (gm_u32_t) GM_PAGE_HASH_PIECE_REF_TABLE_LEN;
  m->hash_piece_ptrs.offset = (m->sram.offset + (gm_offset_t) m->sram.len
			       - (gm_offset_t) m->hash_piece_ptrs.len);
  m->hash_piece_ptrs.permissions = ps->privileged ? GM_MAP_RDWR : 0;

  /* Send queue pages are below the hash piece table page.  Report the
     offsets for the one page associated with this port. */

  m->send_queue.len = (gm_u32_t) GM_PAGE_LEN;
  m->send_queue.offset = (gm_u32_t) (m->hash_piece_ptrs.offset
				     - GM_PAGE_LEN * (GM_NUM_PORTS -
						      port_id));

  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("****** send_queue.offset for port = %d   = 0x%x\n", port_id,
	     m->send_queue.offset));
  _GM_PRINT (GM_PRINT_LEVEL >= 5,
	     ("****** calculation = 0x%x - (%d * (%d - %d))\n",
	      m->hash_piece_ptrs.offset, GM_PAGE_LEN, GM_NUM_PORTS, port_id));

  m->send_queue.permissions = GM_MAP_RDWR;

  /* The RTC page is the first page is SRAM, and may be mapped read-only */

  m->RTC.offset = m->sram.offset;
  m->RTC.len = (gm_u32_t) GM_PAGE_LEN;
  m->RTC.permissions = GM_MAP_READ;

  GM_RETURN_NOTHING ();
}

/*----------------------------------------------------------------------
 - Hash table
 ----------------------------------------------------------------------*/

/* Register a page-sized piece of the page hash table.  This function
   is used as a gm_dma_region_alloc callback, with arg being a pointer
   to the instance state. */

static gm_status_t
gm_register_page_hash_piece (void *arg, gm_dp_t dma_addr, gm_u32_t page_num)
{
  gm_dp_n_t *slot;
  gm_instance_state_t *is = arg;
  gm_u32_t offset;

  GM_PRINT (GM_DEBUG_PAGE_HASH,
	    ("gm_register_page_hash_piece (%p, 0x%x, %d) called \n",
	     arg, dma_addr, page_num));

  gm_assert (page_num < GM_MAX_NUM_HOST_HASH_TABLE_PIECES);
  gm_assert (is);
  gm_assert (is->lanai.sram);

  /* Compute pointer to LANai-resident array of pointers to hash table pieces
     at end of LANai SRAM. */

  slot = (gm_dp_n_t *)
    ((char *) is->lanai.sram + (is->lanai.eeprom.lanai_sram_size
				- GM_PAGE_HASH_PIECE_REF_TABLE_LEN));

  /* determine offset in table for records so we don't overwrite
     earlier records.  (We convert the number of PTEs to the number of
     pages.) */

  offset = (gm_u32_t) ((is->page_hash.max_index + 1)
		       * sizeof (gm_pte_t) / GM_PAGE_LEN);

  /* Record page dma address. */

  GM_PRINT (GM_PRINT_LEVEL >= 6,
	    ("Registering a page hash piece #0x%x at %p.\n", page_num,
	     &slot[page_num + offset]));
  slot[page_num + offset] = gm_hton_dp (dma_addr);
  GM_PRINT (GM_PRINT_LEVEL >= 6, ("Registered a page hash piece\n"));
  return GM_SUCCESS;
}

/* Fill the unfilled slots in LANai page table piece table with
   copies of the legitimate filled entries at the bottom of the table.

   Pointers to the page hash table pieces are stored in an array in
   the LANai SRAM, but this array may be not full.  Rather then tell
   the LANai the actual fullness of the table, fill the table
   cyclically with references to the pieces.  This way, the LANai
   control program need not know the actual number of pieces, but only
   the constant size of its worst-case piece table. This works because
   the size of the LANai table is guaranteed to be a multiple of the
   number of pieces, because each is a power of two.

   For example, if the page hash table is two pages long, and
   GM_PAGE_HASH_PIECE_REF_TABLE_LEN is 8, the DMA pointers in the
   GM_PAGE_HASH_PIECE table in the LANai would be initialized as
   follows:
   
     +---+---+---+---+---+---+---+---+
     |   |   |   |   |   |   |   |   | Page hash piece table in LANai SRAM.
     +---+---+---+---+---+---+---+---+
       |   |   |   |   |   |   |   |
   .---'---|---'---|---'---|---'   |
   |       |       |       |       | DMA pointers
   |       |-------'-------'-------'
   V       |
   +-----+ V
   |     | +-----+  Page hash table in two locked pages of host DMAable 
   +-----+ |     |  memory. Note that the pages need not be contiguous 
	   +-----+  in host DMA space.

   Note how the first two references in the page hash piece table are
   mirrored in the remaining slots in this table.  This allows the
   LANai to ignore the fact that the host hash table is dynamically
   sized and assume that there are GM_PAGE_HASH_PIECE_REF_TABLE_LEN
   pages in the table. */

static void
gm_mirror_page_hash_pieces (gm_instance_state_t * is, int pte_cnt)
{
  unsigned int legit_slots;
  gm_dp_t *slot;

  legit_slots = (unsigned int) (pte_cnt * sizeof (gm_pte_t) / GM_PAGE_LEN);
  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("There are 0x%x pages of non-redundant pieces.\n", legit_slots));

  /* Compute the location of the page hash piece table in the LANai. */

  gm_assert (is->lanai.sram);
  gm_assert (is->lanai.eeprom.lanai_sram_size);
  slot = (gm_dp_t *) ((char *) is->lanai.sram
		      + (is->lanai.eeprom.lanai_sram_size
			 - GM_PAGE_HASH_PIECE_REF_TABLE_LEN));

  /* Mirror the legitimate slots in the table. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("mirroring page hash pieces.\n"));
  gm_assert (legit_slots > 0);
  gm_assert (legit_slots < GM_PAGE_HASH_PIECE_REF_TABLE_LEN);
  gm_assert (GM_POWER_OF_TWO (legit_slots));
  gm_assert (GM_POWER_OF_TWO (GM_PAGE_HASH_PIECE_REF_TABLE_LEN));
  do
    {
      /* Double the number of legitimate entries. */
      __gm_nton_bcopy (is,
		       (gm_u8_n_t *) & slot[0],
		       (gm_u8_n_t *) & slot[legit_slots],
		       ((gm_u8_n_t *) & slot[legit_slots]
			- (gm_u8_n_t *) & slot[0]));
      legit_slots *= 2;
    }
  while (legit_slots < GM_MAX_NUM_HOST_HASH_TABLE_PIECES);
  gm_assert (legit_slots == GM_MAX_NUM_HOST_HASH_TABLE_PIECES);

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Done mirroring page hash piece table.\n"));
}

/* Utility function for following functions. */

static void
_gm_page_hash_segment_bin_for_bin (gm_page_hash_table_t * hash, int *segment,
				   unsigned int *bin)
{
  int seg, nib;

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("_gm_page_hash_segment_bin_for_bin (%p, %p, %p )\n", hash,
	     segment, bin));

  gm_assert (hash->max_index > 0);
  /* Since *bin is unsigned, if it has been decremented from 0 it will be
     greater than max_index.                                              */
  gm_assert (*bin <= (gm_u32_t) hash->max_index);

  /* Scan backwards (largest segment first) for constant, rather than
     log(table_size), average time. */

  nib = hash->max_index - *bin;	/* bins from end */
  seg = hash->segment_cnt - 1;
  while ((unsigned int) nib >= hash->segment[seg].entry_cnt)
    {
      nib -= hash->segment[seg].entry_cnt;
      seg--;
      gm_assert (seg >= 0);
    }
  gm_assert (nib >= 0);
  gm_assert (nib < (int) hash->segment[seg].entry_cnt);
  *bin = hash->segment[seg].entry_cnt - 1 - nib;
  gm_assert (*bin < hash->segment[seg].entry_cnt);
  *segment = seg;
}

/* Return the gm_pte_t structure corresponding to page
   hash table bin number BIN. */

static gm_pte_t *
gm_pte_for_bin (gm_page_hash_table_t * hash, unsigned int bin)
{
  unsigned int segment;

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("gm_pte_for_bin(%p, 0x%x)\n", hash, bin));

  gm_assert ((int) bin <= hash->max_index);
  _gm_page_hash_segment_bin_for_bin (hash, (int *) &segment, &bin);

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("gm_pte_for_bin: segment = 0x%x  bin = 0x%x\n", segment, bin));

  return &hash->segment[segment].entry[bin];
}

/* Return the bin corresponding to USER_VMA for port PORT.  If
   there is an existing entry for the mapping, return the bin
   containing it.  If there is no existing entry, return the empty
   entry where the info should be stored. */

static gm_u32_t
gm_bin_for_mapping (gm_page_hash_table_t * hash, gm_up_t page_port)
{
  gm_pte_t *e;
  unsigned int bin;

  GM_PRINT (GM_PRINT_LEVEL >= 6, ("gm_bin_for_mapping (%p, %lx) called\n",
				  hash, (unsigned long) page_port));

  gm_assert (GM_POWER_OF_TWO (sizeof (gm_pte_t)));
  gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));
  gm_assert (hash->max_index > 0);

  gm_assert (page_port);

  /* Scan for the mapping bin. */

  bin = GM_HASH_PAGE_PORT (page_port) & hash->max_index;
  GM_PRINT (GM_PRINT_LEVEL >= 6, ("Hashed to bin 0x%x.\n", bin));

  e = gm_pte_for_bin (hash, bin);
  while (gm_pte_get_user_page (e))
    {
      if (gm_ntoh_up (e->page_port) == page_port)
	{
	  return bin;
	}

      /* Try the next bin */

      gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));
      bin = (bin + 1) & hash->max_index;
      e = gm_pte_for_bin (hash, bin);
    }
  return bin;
}

#if GM_PRINT_LEVEL >= 5
void
gm_print_page_hash_table (gm_page_hash_table_t * hash)
{
  int bin;
  gm_pte_t *e;

  GM_PRINT (GM_PRINT_LEVEL >= 5,
	    ("\n\ngm_print_page_hash_table: hash->max_index = 0x%x.\n",
	     hash->max_index));
  for (bin = 0; bin <= hash->max_index; bin++)
    {
      gm_up_t page;

      e = gm_pte_for_bin (hash, bin);
      page = gm_pte_get_user_page (e);
      if (!page)
	continue;
      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("bin = 0x%x, virt_page = 0x%x, port = 0x%x;\n",
		 bin, page, gm_pte_get_port_id (e)));
    }
}
#endif

/* Double the size of the page hash table (or initialize it if the
   size is zero).  Note that this does not invalidate the cached PTEs
   in the LANai. */

static gm_status_t
gm_grow_page_hash_table (gm_instance_state_t * is)
{
  gm_u32_t N, seg;
  gm_page_hash_table_t *hash;

  GM_PRINT ((GM_DEBUG_PAGE_TABLE || GM_DEBUG_PAGE_HASH), 
            ("Growing page hash table.\n"));

  hash = &is->page_hash;

  /* Determine new page hash table segment number. */

  seg = hash->segment_cnt;

  /* Verify we can handle another page table segment. */

  if (hash->max_index == 0x3fffffff || seg == GM_NUM_PAGE_TABLE_SEGMENTS)
    {
      GM_NOTE (("Hash table reached addressing limit.\n"));
      goto abort_with_nothing;
    }

  /* compute number of entries to allocate. */

  if (hash->max_index == -1)
    {
      /* This is the first allocation, so create 1 page worth of entries. */
      N = (gm_u32_t) (GM_PAGE_LEN / sizeof (gm_pte_t));
    }
  else
    {
      /* Double the number of entries. */
      N = hash->max_index + 1;
    }

#if 0
#warning feldy
  if (!(GM_POWER_OF_TWO (N)))
    {
      GM_PRINT (GM_PRINT_LEVEL >= 0,
		("Number of entries (%d) is not a power of 2\n", N));
      _GM_PRINT (GM_PRINT_LEVEL >= 0,
		 ("GM_PAGE_LEN = %d  sizeof(gm_pte_t) = %d\n", GM_PAGE_LEN,
		  sizeof (gm_pte_t)));
      goto abort_with_nothing;
    }
#endif

  gm_assert (GM_POWER_OF_TWO (N));
  gm_assert (GM_PAGE_ALIGNED (N * sizeof (gm_pte_t)));

  GM_PRINT (GM_DEBUG_HOST_DMA,
	    ("Adding 0x%x entries to page hash table.\n", N));

  if (gm_arch_dma_region_alloc (is,
				&hash->segment[seg].dma_region,
				N * sizeof (gm_pte_t),
				GM_ARCH_DMA_READ | GM_ARCH_DMA_WRITE
				| GM_ARCH_DMA_CONSISTENT,
#if GM_ENABLE_VM
				gm_register_page_hash_piece,
#else
				NULL,
#endif
				is) != GM_SUCCESS)
    {
      GM_NOTE (("Failed to allocate segment of page hash table.\n"));
      goto abort_with_nothing;
    }
  hash->segment[seg].entry
    = (gm_pte_t *) (gm_arch_dma_region_kernel_addr
		    (&hash->segment[seg].dma_region));
  gm_bzero (hash->segment[seg].entry, N * sizeof (gm_pte_t));

  /* Commit to using this segment now that it has been fully allocated. */

  hash->max_index += N;
  gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));
  hash->segment_cnt = seg + 1;
  hash->segment[seg].entry_cnt = N;

  /* Until now, we have not modified the page table from the point of
     view of the LANai, but we are now about to, so pause the LANai
     during the changes. */

  if (is->lanai.running)
    gm_pause_lanai (is);

  /* Remove and reinsert each entry in the page hash table to restore
     hash table consistency.  This is done in reverse order to
     minimize the need to rehash clusters of PTEs. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Rehashing all PTEs.\n"));
  {
    gm_s32_t bin;

    for (bin = N - 1; bin >= 0; bin--)
      {
	gm_pte_t *e, *new_e;
	int new_bin;

	e = gm_pte_for_bin (hash, bin);
	gm_assert (e);
	if (!gm_pte_get_user_page (e))
	  continue;

	/* find new location for entry if needed. */
	new_bin = gm_bin_for_mapping (hash, gm_pte_get_page_port (e));
	if (new_bin == bin)
	  continue;

	GM_PRINT (GM_PRINT_LEVEL >= 5,
		  ("*** Relocating a page hash mapping from bin "
		   "0x%x to bin 0x%x\n", bin, new_bin));
	_GM_PRINT (GM_PRINT_LEVEL >= 5,
		   ("virtual_addr = %p, port = 0x%x\n",
		    (unsigned int) gm_pte_get_user_page (e),
		    gm_pte_get_port_id (e)));

	/* copy entry to new location */
	GM_PRINT (GM_PRINT_LEVEL >= 5,
		  ("Calling gm_pte_for_bin(?,%d)\n", new_bin));
	new_e = gm_pte_for_bin (hash, new_bin);
	*new_e = *e;
#if GM_CAN_REGISTER_MEMORY
	gm_hash_rekey (hash->page_lock_hash, e, new_e);
#endif
	gm_pte_clear (e);

	/* Arrange to reinsert all entries after the moved entry.
	   This while loop will not overrun the end of the page table
	   because either the table is empty, or there is guaranteed
	   to be an empty slot between the original BIN and the end of
	   the table because the table is less than half full and
	   entries in the upper half of the table never need
	   relocating, and BIN starts in the first half of the
	   table. */
	GM_PRINT (GM_PRINT_LEVEL >= 5, ("Scanning \n"));
	do
	  {
	    bin++;
	    gm_assert (bin <= hash->max_index);
	  }
	while (gm_pte_get_user_page (gm_pte_for_bin (hash, bin)));
      }
  }

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Mirroring PTE piece table.\n"));
  /* Update the page hash piece table in the LANai. */
  gm_mirror_page_hash_pieces (is, hash->max_index + 1);

#if GM_PRINT_LEVEL >= 5
  gm_print_page_hash_table (hash);
#endif

  if (is->lanai.running)
    gm_unpause_lanai (is);

  return GM_SUCCESS;

abort_with_nothing:
  return GM_FAILURE;
}

/****************************************************************
 * Page table
 ****************************************************************/

/* Allocate the page hash table.

   The page hash table keeps track of the mappings of virtual
   addresses for each port to dma addresses.  It is big, so we store
   it in host memory and the MCP caches entries in LANai SRAM.

   There is one page hash table per device.  We would prefer to have
   just one shared table, but there appears to be no clean way to do
   this in Solaris.  It might be possible, but doesn't seem worth the
   effort.

   The hash table potentially needs enough storage to store an entry
   for every page of physical memory.  It is not necessary to allocate
   more capacity than this, because it is impossible to allocate all
   physical pages for DMA, but we dynamically allocate enough capacity
   to keep the hash table at least 50% empty to minimize hash
   collisions.

   The hash table uses a linear-scan-on-miss strategy.  That is, if
   the location for an entry is occupied, the entry is stored in the
   next empty bin.

   NOTE: No need to mutex protect the table as it is being created. */

gm_status_t gm_alloc_page_hash_table (gm_instance_state_t * is)
{
  gm_status_t status;
  gm_page_hash_table_t *hash;

  GM_CALLED ();

  hash = &is->page_hash;

  GM_PRINT (GM_DEBUG_PAGE_TABLE, ("**************************\n"));
  GM_PRINT (GM_DEBUG_PAGE_TABLE, ("allocating page hash table\n"));
  GM_PRINT (GM_DEBUG_PAGE_TABLE, ("**************************\n"));

  /* Initialize hash table */

  gm_bzero (hash, sizeof (*hash));

  gm_arch_sync_init (&hash->sync, is);
  hash->max_index = -1;
  gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));

  /* Allocate pages to be used for send and receive DMA any time the user
     specifies a send or receive using nonDMAable memory.  Such receives are
     discarded by DMAing them into the bogus rdma page, and such sends send
     all zeros from the bogus send page. The pointers are passed to the LANai
     eventually, where the MCP uses them for any DMA to or from a non-DMA
     virtual memory address page. */

  /* Alloc page for bogus receives. */
  /* mark page for dma reads and writes for DMA testing */

  GM_PRINT ((GM_DEBUG_PAGE_TABLE | GM_DEBUG_HOST_DMA),
            ("Allocating bogus receive page.\n"));
  status 
    = gm_arch_dma_region_alloc (is, &hash->bogus_rdma_region,
				GM_PAGE_LEN,
				GM_ARCH_DMA_READ | GM_ARCH_DMA_WRITE | 
				GM_ARCH_DMA_CONSISTENT, 
				NULL, NULL);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not allocate dma page for bogus receives.\n"));
      goto abort_with_nothing;
    }

  /* Alloc page for bogus sends. */
  /* mark page for dma reads and writes for DMA testing */

  GM_PRINT ((GM_DEBUG_PAGE_TABLE | GM_DEBUG_HOST_DMA),
            ("Allocating bogus send page.\n"));
  status
    = gm_arch_dma_region_alloc (is, &hash->bogus_sdma_region,
				GM_PAGE_LEN,
				GM_ARCH_DMA_READ | GM_ARCH_DMA_WRITE |
				GM_ARCH_DMA_CONSISTENT,
				NULL, NULL);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not allocate dma page for bogus sends.\n"));
      goto abort_with_bogus_rdma_page;
    }

  /* Initialize page for bogus sends to be filled with 0xAAAAAAAAs.
     We used to use 0xdeadbeef, but that was endian dependent. */

  {
    gm_u32_t *p, *p_limit;

    gm_assert (GM_PAGE_LEN % 4 == 0);
    p = gm_arch_dma_region_kernel_addr (&hash->bogus_sdma_region);
    p_limit = (gm_u32_t *) ((char *) p + GM_PAGE_LEN);

    for (; p < p_limit; p++)
      *p = 0xAAAAAAAA;
  }

#if GM_ENABLE_VM
  /* Create hash table for memory page locks.  This is done even if we
     don't need the locks, since it doesn't waste much memory and is
     simpler. */

  hash->page_lock_hash = gm_create_hash (gm_hash_compare_ptrs,
					 gm_hash_hash_ptr,
					 0, sizeof (gm_arch_page_lock_t),
					 0, 0);
  if (!hash->page_lock_hash)
    {
      GM_NOTE (("Could not create page lock hash table.\n"));
      goto abort_with_bogus_sdma_page;
    }

  /* Allocate the minimal page hash table, since the LANai needs to be
     able to assume that it exists. */

  status = gm_grow_page_hash_table (is);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not allocate the page hash table.\n"));
      goto abort_with_page_hash_lock;
    }
#endif /* GM_ENABLE_VM */


  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_page_hash_lock:
  gm_destroy_hash (hash->page_lock_hash);
abort_with_bogus_sdma_page:
  gm_arch_dma_region_free (&is->page_hash.bogus_sdma_region);
abort_with_bogus_rdma_page:
  gm_arch_dma_region_free (&is->page_hash.bogus_rdma_region);
abort_with_nothing:
  gm_bzero (&is->page_hash, sizeof (is->page_hash));
  GM_RETURN_STATUS (GM_FAILURE);
}

void
gm_free_page_hash_table (gm_instance_state_t * is)
{
  unsigned int seg;

  gm_assert (is);
  gm_assert (!is->lanai.running);
#if 0
  if (is->lanai.running)
    gm_disable_lanai (is);
#endif

  seg = is->page_hash.segment_cnt;
  GM_PRINT ((GM_DEBUG_PAGE_TABLE | GM_DEBUG_HOST_DMA),
	    ("Freeing page hash table regions (there are %d).\n", seg));
  gm_assert (seg < 64);
  while (seg--)
    {
      GM_PRINT (GM_DEBUG_PAGE_TABLE,
		("Freeing page hash table segment %d.\n", seg));
      gm_arch_dma_region_free (&is->page_hash.segment[seg].dma_region);
    }
  is->page_hash.segment_cnt = seg;

  GM_PRINT (GM_DEBUG_PAGE_TABLE, ("Destroying page lock hash\n"));
  gm_destroy_hash (is->page_hash.page_lock_hash);

  GM_PRINT ((GM_DEBUG_PAGE_TABLE | GM_DEBUG_HOST_DMA),
            ("Freeing bogus SDMA region.\n"));
  gm_arch_dma_region_free (&is->page_hash.bogus_sdma_region);

  GM_PRINT ((GM_DEBUG_PAGE_TABLE | GM_DEBUG_HOST_DMA),
            ("Freeing bogus RDMA region.\n"));
  gm_arch_dma_region_free (&is->page_hash.bogus_rdma_region);

  gm_bzero (&is->page_hash, sizeof (is->page_hash));
}

/* Increments the reference count of a mapping already in the page
   table.

   Return values:
   GM_SUCCESS on success
   GM_INVALID_PARAMETER if their is no extant mapping to reference
   GM_FAILURE otherwise

   Returns GM_SUCCESS unless the reference is not in the table
   or the mapping has been referenced too many times. */

static gm_status_t
gm_reference_mapping (gm_instance_state_t * is,
		      gm_up_t user_vma, gm_u32_t port)
{
  unsigned int bin;
  gm_pte_t *e;
  gm_page_hash_table_t *hash;

  gm_assert (GM_PAGE_ALIGNED (user_vma));
  gm_assert (port <= GM_NUM_PORTS);

  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("gm_reference_mapping(%p, %lx, %d)\n",
	     is, (unsigned long) user_vma, port));

  hash = &is->page_hash;

  if (GM_ENABLE_GALVANTECH_WORKAROUND)
    {
      gm_assert (sizeof (gm_pte_t) == 32);
    }
  gm_assert (GM_POWER_OF_TWO (sizeof (gm_pte_t)));
  gm_assert (user_vma);
  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("hash->max_index = 0x%x\n", hash->max_index));
  gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));

  GM_PRINT (GM_PRINT_LEVEL >= 10, ("finding refs for mapping (%p,%x)->XXX\n",
				   user_vma, port));

  bin = gm_bin_for_mapping (hash, user_vma + port);

  GM_PRINT (GM_PRINT_LEVEL >= 10,
	    ("gm_reference_mapping: bin = 0x%x\n", bin));

  e = gm_pte_for_bin (hash, bin);

  /* verify entry is in the table */

  if (!gm_pte_get_user_page (e))
    {
      return GM_INVALID_PARAMETER;
    }

  /* increment reference count, checking for overflow */

  if (gm_pte_incr_ref_cnt (e) != GM_SUCCESS)
    {
      GM_PRINT (GM_DEBUG_PAGE_TABLE,
		("*** gm_reference_mapping: overflow?? \n"));
      return GM_FAILURE;
    }

  /* increment reference count */

  hash->total_refs++;

#if GM_ENABLE_GALVANTECH_WORKAROUND
  /* update entry checksum */

  e->checksum = gm_hton_u32 (gm_galvantech_PTE_checksum (e));
#endif

  return GM_SUCCESS;
}

/* Add the translation (USER_ADDR,PORT)->DMA_ADDR to the page
   translation hash table.

   GM_RETURNS: the number of earlier references to this mapping.

   WARNING: This function assumes it has exclusive access to the page
   hash table. Consequently, the ps->instance->page_hash.sync mutex must
   have been allocated by a caller. */

gm_status_t
gm_add_mapping_to_page_table (gm_instance_state_t * is,
			      gm_up_t user_vma,
			      unsigned int port, gm_dp_t dma_addr)
{
  gm_u32_t bin;
  gm_pte_t *e;
  gm_page_hash_table_t *hash;
  gm_status_t status;
#if GM_CAN_REGISTER_MEMORY
  gm_arch_page_lock_t _lock, *lock;
#endif

  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("gm_add_mapping_to_page_table(%p, %lx, %d, 0x%x)\n",
	     is, (unsigned long) user_vma, port, dma_addr));

  hash = &is->page_hash;

  if (GM_ENABLE_GALVANTECH_WORKAROUND)
    {
      gm_assert (sizeof (gm_pte_t) == 32);
    }
  gm_assert (GM_POWER_OF_TWO (sizeof (gm_pte_t)));
  gm_assert (user_vma);
  gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));
  gm_assert (hash->max_index > 0);
  gm_assert (GM_PAGE_ALIGNED (user_vma));
  gm_assert (GM_PAGE_ALIGNED (dma_addr));

#if 0 && (!GM_OS_SOLARIS7 && GM_CPU_sparc64)
#if __GNUC__
#warning Slowly verifying ability to read/write VMAs.
#endif

  /* verify that we can store all the bits of the page number */

  {
    gm_pte_t test_entry;

    gm_pte_set_user_page (&test_entry, user_vma);
    if (gm_pte_get_user_page (&test_entry) != user_vma)
      {
	GM_NOTE (("PTE virtual page storage test failed\n"));
	status = GM_INTERNAL_ERROR;
	goto abort_with_nothing;
      }
  }
#endif

  /* If the mapping is already in the table, reference it and return. */

  status = gm_reference_mapping (is, user_vma, port);

  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("gm_add_mapping_to_page_table: "
	     "gm_reference_mapping returns %d\n", status));

  if (status == GM_SUCCESS)
    {
      gm_assert (gm_noticed_dma_addr (is->dma_page_bitmap, dma_addr)
		 == GM_SUCCESS);
      GM_PRINT (GM_DEBUG_PAGE_TABLE, ("%p already in the table\n",
				      (char *) user_vma));
      return GM_SUCCESS;
    }

  if (status == GM_FAILURE)
    return GM_FAILURE;

  gm_assert (status == GM_INVALID_PARAMETER);

  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("gm_add_mapping_to_page_table: invalid_parameter?\n"));

  /* grow the hash table if adding a new entry would make the table
     overfull. */

  if (hash->filled_entries >= (gm_u32_t) (hash->max_index / 2))
    {
      if (gm_grow_page_hash_table (is) != GM_SUCCESS)
	goto abort_with_nothing;
      gm_assert (hash->filled_entries < (gm_u32_t) (hash->max_index >> 1));
    }

  /* Compute the bin in which to record the mapping */

  bin = gm_bin_for_mapping (hash, (gm_up_t) user_vma + port);
  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("Using bin 0x%x (0 earlier references).\n", bin));

  /* Record entry in PTE at position BIN */

  e = gm_pte_for_bin (hash, bin);
  gm_assert (gm_pte_get_user_page (e) == 0);
  gm_pte_set_user_page (e, (gm_up_t) user_vma);

  gm_pte_set_port_id (e, port);
  /* e->dma_page_num set below */
  gm_pte_set_ref_cnt (e, 1);

#if GM_CAN_REGISTER_MEMORY

  /* Allocate storage for the page lock */

  gm_bzero (&_lock, sizeof (_lock));
  gm_assert (!gm_hash_find (hash->page_lock_hash, e));
  if (gm_hash_insert (hash->page_lock_hash, e, &_lock) != GM_SUCCESS)
    {
      GM_NOTE (("Could not allocate storage for page lock.\n"));
      goto abort_with_hash_entry;
    }
  lock = (gm_arch_page_lock_t *) gm_hash_find (hash->page_lock_hash, e);
  gm_assert (lock);

  /* Lock the page and get its dma address */

  if (gm_arch_lock_user_buffer_page (is, user_vma, &dma_addr, lock)
      != GM_SUCCESS)
    {
      GM_NOTE (("GM: Could not lock user buffer.\n"));
      GM_PRINT (GM_PRINT_LEVEL >= 0, ("Failed to add mapping (0x%lx,0x%x)\n",
				      (unsigned long) user_vma, port));

      goto abort_with_storage;
    }
  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("adding mapping (%lx,%x)->%p\n",
	     (unsigned long) user_vma, port, dma_addr));
#endif

  /* Record the DMA address of the mapping */

  gm_pte_set_dma_page (e, dma_addr);
#if GM_ENABLE_GALVANTECH_WORKAROUND
  e->checksum = gm_hton_u32 (gm_galvantech_PTE_checksum (e));
#endif

  gm_notice_dma_addr (is->dma_page_bitmap, dma_addr);

  hash->filled_entries++;
  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("There are now %d filled hash bins. (gm_add_mapping)\n",
	     hash->filled_entries));
  hash->total_refs++;
  return GM_SUCCESS;

#if GM_CAN_REGISTER_MEMORY
abort_with_storage:
  gm_hash_remove (hash->page_lock_hash, e);
abort_with_hash_entry:
  gm_pte_set_ref_cnt (e, 0);
  gm_pte_set_port_id (e, 0);
  gm_pte_set_user_page (e, 0);
#if GM_ENABLE_GALVANTECH_WORKAROUND
  e->checksum = gm_hton_u32 (gm_galvantech_PTE_checksum (e));
#endif
#endif
abort_with_nothing:
  return GM_FAILURE;
}

/* GM_RETURNS: the number of references to a mapping in the page table,
   or 0 if the mapping is not in the table or if the DMA address of
   the mapping does not match the DMA address specified. */

gm_u32_t
gm_refs_to_mapping_in_page_table (gm_instance_state_t * is,
				  gm_up_t user_vma, gm_u32_t port)
{
  unsigned bin;
  gm_pte_t *e;
  gm_page_hash_table_t *hash;

  hash = &is->page_hash;

  gm_assert (GM_PAGE_ALIGNED (user_vma));
  if (GM_ENABLE_GALVANTECH_WORKAROUND)
    {
      gm_assert (sizeof (gm_pte_t) == 32);
    }
  gm_assert (GM_POWER_OF_TWO (sizeof (gm_pte_t)));
  gm_assert (user_vma);
  gm_assert (hash->max_index > 0);
  gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));

  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("finding refs for mapping (%lx,%x)->XXX\n",
	     (unsigned long) user_vma, port));

  bin = gm_bin_for_mapping (hash, user_vma + port);
  e = gm_pte_for_bin (hash, bin);
  gm_assert (gm_pte_get_user_page (e) != 0 || gm_pte_get_ref_cnt (e) == 0);
  return gm_pte_get_ref_cnt (e);
}

/* Remove a reference to the translation (USER_ADDR,PORT)->DMA_ADDR
   from the page translation hash table.  Returns the number of
   remaining references to this mapping.

   WARNING: It is an error to call this function to remove a non-existant
   mapping from the page table.

   WARNING: This function assumes it has exclusive access to the page
   hash table. Consequently, the ps->instance->page_hash.sync mutex must
   have been allocated by a caller. */

gm_u32_t
gm_dereference_mapping (gm_instance_state_t * is,
			gm_up_t user_vma, gm_u32_t port)
{
  unsigned bin;
  gm_pte_t *e;
  gm_page_hash_table_t *hash;

  GM_CALLED ();

  hash = &is->page_hash;

  if (GM_ENABLE_GALVANTECH_WORKAROUND)
    {
      gm_assert (sizeof (gm_pte_t) == 32);
    }
  gm_assert (GM_POWER_OF_TWO (sizeof (gm_pte_t)));
  gm_assert (user_vma);
  gm_assert (GM_PAGE_ALIGNED (user_vma));
  gm_assert (hash->max_index > 0);
  gm_assert (GM_POWER_OF_TWO (hash->max_index + 1));

  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("removing mapping (%lx,%x)->??\n",
	     (unsigned long) user_vma, port));

  /* Find the mapping. */

  bin = gm_bin_for_mapping (hash, (gm_up_t) user_vma + port);
  e = gm_pte_for_bin (hash, bin);
  gm_assert (gm_pte_get_user_page (e));

  /* Dereference the bin */

  {
    gm_u32_t remaining_refs;

    remaining_refs = gm_pte_decr_ref_cnt (e);
    hash->total_refs--;
    GM_PRINT (GM_DEBUG_PAGE_TABLE,
	      ("%d refs remaining.\n", (int) hash->total_refs));
#if GM_ENABLE_GALVANTECH_WORKAROUND
    e->checksum = gm_hton_u32 (gm_galvantech_PTE_checksum (e));
#endif
    if (remaining_refs != 0)
      GM_RETURN (remaining_refs);
  }

  /****
   * The mapping is no longer referenced
   ****/

#if GM_CAN_REGISTER_MEMORY

  /* Unlock the page and free the lock */

  {
    gm_arch_page_lock_t *lock;
    lock = (gm_arch_page_lock_t *) gm_hash_find (hash->page_lock_hash, e);
    gm_assert (lock);
#if GM_OS_LINUX
    /*
      GM_PRINT (GM_DEBUG_PAGE_TABLE,
      ("removing mapping (%lx,%x)->%p=%p\n",
      (unsigned long) gm_pte_get_user_page (e),
      port,
      gm_pte_get_dma_page (e),
      (void *) (lock->pagenum * GM_PAGE_LEN)));
    */
#endif
    gm_arch_unlock_user_buffer_page (lock);
    gm_hash_remove (hash->page_lock_hash, e);
  }

#endif

  /* remove the entry from the table */

  gm_pte_clear (e);
  hash->filled_entries--;
  GM_PRINT (GM_DEBUG_PAGE_TABLE,
	    ("There are now %d filled hash bins. "
	     "(gm_dereference_mapping)\n", hash->filled_entries));

  /* Remove and reinsert each entry following the removed entry. */

  bin = (bin + 1) & hash->max_index;
  e = gm_pte_for_bin (hash, bin);
  while (gm_pte_get_user_page (e))
    {
      gm_u32_t new_bin;

      new_bin = gm_bin_for_mapping (hash, gm_pte_get_page_port (e));
      if (new_bin != bin)
	{
	  gm_pte_t *new_e;

	  new_e = gm_pte_for_bin (hash, new_bin);

	  /* Move entry to new location. */
	  *new_e = *e;
	  gm_pte_clear (e);
#if GM_CAN_REGISTER_MEMORY
	  /* update the lock hash table key, since it has changed */
	  gm_hash_rekey (hash->page_lock_hash, e, new_e);
#endif
	}

      bin = (bin + 1) & hash->max_index;
      e = gm_pte_for_bin (hash, bin);
    }

  GM_RETURN (0);
}

/* Remove all page table entries for port PORT from the page table for
   the interface described by IS.  This prevents the LANai from
   performing DMAs into any of the memory associated with that
   port. */

void
gm_disable_port_DMAs (gm_port_state_t * ps)
{
  unsigned int i, port;
  gm_instance_state_t *is;

  GM_CALLED ();

  GM_PRINT (GM_DEBUG_PORT_STATE, ("Disabling port DMAs.\n"));

  port = ps->id;
  is = ps->instance;
  gm_assert (is);

  gm_arch_mutex_enter (&is->page_hash.sync);
  if (!GM_ENABLE_SECURITY && !is->lanai.running)
    {
      GM_NOTE (("disable_port_DMAs called when LANai is not running\n"));
    }
  else
    {
      gm_pause_lanai (is);
    }

  /* Remove the page table entries if the page hash table exists. */

  GM_PRINT (GM_PRINT_LEVEL >= 4, ("Removing page hash table entries.\n"));
  if (is->page_hash.max_index != -1)	/* if table exists */
    {
      /* scan for PTEs for this port. */

      for (i = 0; i <= (unsigned int) is->page_hash.max_index; i++)
	{
	  gm_u32_t port_num;
	  gm_up_t user_vma;
	  gm_pte_t *e;

	  e = gm_pte_for_bin (&is->page_hash, i);
	  user_vma = gm_pte_get_user_page (e);
	  if (!user_vma)
	    continue;

	  port_num = gm_pte_get_port_id (e);
	  if (port_num != port)
	    continue;

	  /* Remove the entry from the page hash table. */

	  gm_pte_set_ref_cnt (e, 1);
	  gm_dereference_mapping (is, user_vma, port);
	  /* Arrange to recheck same slot. */
	  i--;
	}
    }

  if (is->lanai.running)
    {
      /* Deschedule any pending host wakeup. */

      GM_PRINT (GM_PRINT_LEVEL >= 4, ("Descheduling host wakeup.\n"));
      /* gm_kton_8 (is, &is->lanai.globals->port[port].wake_host, 0);
         close, below, will do this, and we can't do partword PIO
         while the LANai is paused. */
      gm_arch_sync_reset (&ps->sleep_sync);

      /* Tell LANai to close PORT,  */

      GM_PRINT (GM_PRINT_LEVEL >= 4, ("Telling LANai to close port.\n"));
      gm_write_lanai_global_s32 (is, port_to_close, port);
    }

  /* Make changes visible to LANai. */

  GM_PRINT ((GM_DEBUG_HOST_DMA),
            ("Synchronizing dma regions.\n"));
  {
    unsigned int j;
    for (j = 0; j < is->page_hash.segment_cnt; j++)
      {
	gm_arch_dma_region_sync (&is->page_hash.segment[j].dma_region,
				 GM_ARCH_SYNC_FOR_DEVICE);
      }
  }

  /* Allow the LANai to reset the port */

  if (!GM_ENABLE_SECURITY && !is->lanai.running)
    {
    }
  else
    {
      gm_unpause_lanai (is);
      gm_pause_lanai (is);

      gm_unpause_lanai (ps->instance);
    }

  gm_arch_mutex_exit (&is->page_hash.sync);

  GM_RETURN_NOTHING ();
}

#if !GM_CAN_REGISTER_MEMORY
/* Attempt to grow the copy block for device to LENGTH */

gm_status_t gm_expand_copy_block (gm_port_state_t * ps, gm_u32_t length)
{
  gm_instance_state_t *is;
  struct gm_segment_desc *seg;
  gm_status_t status;

  GM_CALLED ();
  GM_PRINT (GM_DEBUG_HOST_DMA, ("*** Expanding the copy block ***\n"));

  is = ps->instance;

  /* Disallow oversized allocations. */

  if (length > ps->max_dma_mem)
    GM_RETURN_STATUS (GM_PERMISSION_DENIED);

  /* Expand the copy block by adding segments until the copy block is large
     enough. */

  while (ps->copy_block.len < length)
    {
      gm_size_t alloc_len;

      /* Should never need more than 16 segments. */

      gm_assert (ps->copy_block.segment_cnt < 32);

      seg = &ps->copy_block.segment[ps->copy_block.segment_cnt];

      /* Alloc segments of length GM_PAGE_LEN, GM_PAGE_LEN, GM_PAGE_LEN*2,
         GM_PAGE_LEN*4, &c */

#if GM_BC_DRIVER
      if (ps->compatibility_mode)
	alloc_len = ps->max_dma_mem;
      else
	alloc_len = (unsigned) (ps->copy_block.len ?
				ps->copy_block.len : GM_PAGE_LEN);
#else
      alloc_len = (unsigned) (ps->copy_block.len ?
			      ps->copy_block.len : GM_PAGE_LEN);
#endif

      gm_assert (GM_POWER_OF_TWO (alloc_len));

      GM_PRINT (GM_PRINT_LEVEL >= 3,
		("[%d]: Allocating a segment of 0x%x bytes for port %d.\n",
		 is->id, alloc_len, ps->id));

      status = gm_arch_dma_region_alloc (is, &seg->dma_region, alloc_len,
					 GM_ARCH_DMA_RDWR
					 | GM_ARCH_DMA_CONSISTENT,
					 NULL, NULL);
      if (status != GM_SUCCESS)
	{
	  GM_NOTE (("[%d]: Allocation of dma segment failed.\n", is->id));
	  goto abort_with_nothing;
	}
      seg->data = (char *) gm_arch_dma_region_kernel_addr (&seg->dma_region);
      GM_PRINT (GM_DEBUG_HOST_DMA,
		("New copy block segment at KVA %p.\n", seg->data));

#if GM_BC_DRIVER

#error not platform-independent

      /* Require compatibility-mode DMA memory to be contiguous. */

      if (ps->compatibility_mode && seg->dma_region.cookie_cnt != 1)
	{
	  status = GM_OUT_OF_MEMORY;
	  GM_NOTE (("Could not allocate contiguous memory for "
		    "compatibility-mode port.\n"));
	  goto abort_with_dma_region;
	}
#endif

      seg->len = alloc_len;

      ps->copy_block.segment_cnt++;
      ps->copy_block.len += alloc_len;
      gm_assert (GM_POWER_OF_TWO (ps->copy_block.len));
    }
  GM_RETURN_STATUS (GM_SUCCESS);

  /* Here we abort only the failed segment allocation, although we may have
     allocated other segments in this call. */

#if GM_BC_DRIVER
abort_with_dma_region:
  gm_arch_dma_region_free (&seg->dma_region);
#endif
abort_with_nothing:
  GM_RETURN_STATUS (status);
}
#endif

#if !GM_CAN_REGISTER_MEMORY
/* Free all the copy block for a device. Called when the device is closed. */

void
gm_free_copy_block (gm_port_state_t * ps)
{
  unsigned cnt;

  GM_CALLED ();

  gm_assert (ps);

  /* BAD: Should sync with LANai and inform it that the mappings have
     changed. (this is currently done by the caller) */

  GM_PRINT (GM_PRINT_LEVEL >= 4,
	    ("Freeing the copy block with %d segments.\n",
	     ps->copy_block.segment_cnt));
  cnt = ps->copy_block.segment_cnt;
  while (cnt--)
    {
      GM_PRINT (GM_DEBUG_HOST_DMA, ("Freeing a copy block segment.\n"));
      gm_arch_dma_region_free (&ps->copy_block.segment[cnt].dma_region);
    }
  ps->copy_block.segment_cnt = 0;

  ps->copy_block.len = 0;
  GM_RETURN_NOTHING ();
}
#endif

#ifndef WIN32
gm_status_t gm_kernel_sleep (gm_port_state_t * ps)
{
  unsigned port_id;

  GM_PRINT (GM_PRINT_LEVEL >= 7, ("gm_arch_kernel_sleep() called.\n"));

  gm_assert (ps);

  if (!ps->opened)
    {
#ifdef __GNUC__
      GM_NOTE (("User tried API function "
		__FUNCTION__ " " "on port before setting port number.\n"));
#else
      GM_NOTE (("User tried API function on port before setting port "
		"number.\n"));
#endif
      return GM_PERMISSION_DENIED;
    }

  port_id = ps->id;

  /* GM daemon MAY NOT use this ioctl */

  if (port_id == GM_DAEMON_PORT_ID)
    return (GM_PERMISSION_DENIED);

  /* Sleep until a message or signal is received. */

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Port %d sleeping.\n", port_id));
  switch (gm_arch_signal_sleep (&ps->sleep_sync))
    {
    case GM_SLEEP_INTERRUPTED:

      /* Got a signal */

      GM_PRINT (GM_PRINT_LEVEL >= 5, ("Port %d woke by signal.\n", port_id));
      GM_RETURN_STATUS (GM_INTERRUPTED);
    case GM_SLEEP_TIMED_OUT:

      /* Timeout */

      GM_PRINT (GM_PRINT_LEVEL >= 5, ("Port %d woke by timeout.\n", port_id));
      break;

      /* success */

    default:
      GM_PRINT (GM_PRINT_LEVEL >= 5, ("Port %d woke.\n", port_id));
      break;
    }
  GM_RETURN_STATUS (GM_SUCCESS);
}
#endif

/****************************************************************
 * Interrupt handing
 ****************************************************************/

#define GM_INTR_PRINT (0)

/* Determine if IS generated an interrupt.

   Returns GM_ARCH_INTR_CLAIMED if the board is generating an interrupt.
   Otherwise, returns GM_ARCH_INTR_UNCLAIMED. */

gm_s32_t gm_interrupting (gm_instance_state_t * is)
{
  unsigned eisr, eimr;

  eisr = is->get_ISR (is);
  eimr = is->get_EIMR (is);

/* feldy */
  if (eimr != GM_HOST_SIG_BIT) 
     GM_PRINT(1,("gm_interrupting: eimr looks bad = 0x%08x\n",eimr));

#ifdef DELETED_FLEDY
  gm_assert_p (eimr == GM_HOST_SIG_BIT);
#endif /* DELETED_FELDY */

  if (!(eisr & eimr))
    {
/* feldy */
      GM_PRINT (GM_INTR_PRINT,
                ("gm_intr: unclaimed interrupt isr=0x%x eimr=0x%x\n",
                 eisr,eimr));
      return GM_ARCH_INTR_UNCLAIMED;
    }
  return GM_ARCH_INTR_CLAIMED;
}

#if GM_USE_DEFAULT_INTERRUPT_HANDLER
#if !defined GM_ARCH_INTR_CLAIMED || !defined GM_ARCH_INTR_UNCLAIMED
#error GM_ARCH_INTR_CLAIMED and GM_ARCH_INTR_UNCLAIMED must be defined
#endif

/* New-style interrupt handler.

   Like the old-style interrupt handler gm_intr(), but assumes the
   caller has already called gm_interrupting() to verify that IS is
   interrupting. */

void
gm_handle_claimed_interrupt (gm_instance_state_t * is)
{
  enum gm_interrupt_type type, type_retry;

  gm_assert (is);
  gm_assert (is->lanai.running);

  GM_STBAR ();
  gm_set_EIMR (is, 0);		/* Solaris bug workaround */
  GM_STBAR ();

  gm_assert (is->lanai.special_regs);
  gm_assert (is->lanai.running);

  GM_PRINT (GM_INTR_PRINT,
	    ("*** got an interrupt to handler %d  type = %d\n",
	     is->id, gm_read_lanai_global_u32 (is, interrupt.type)));

  type =
    (enum gm_interrupt_type) gm_read_lanai_global_u32 (is, interrupt.type);
  GM_PRINT (GM_INTR_PRINT, ("gm_intr: interrupt type = 0x%x \n", type));
retry:
  switch (type)
    {
    case GM_PAUSE_INTERRUPT:
      GM_PRINT (GM_INTR_PRINT, ("got a pause interrupt for is=%p\n", is));
      gm_assert (is->pause_rqst);
      is->pause_ack = 1;
      gm_arch_wake (&is->pause_sync);
      break;

    case GM_UNPAUSE_INTERRUPT:
      GM_PRINT (GM_INTR_PRINT, ("got an unpause interrupt for is=%p\n", is));
      gm_assert (!is->pause_rqst);
      is->pause_ack = 0;
      gm_arch_wake (&is->pause_sync);
      break;

    case GM_CLEAR_TABLES_INTERRUPT:
      GM_PRINT (GM_INTR_PRINT,
		("got clear_tables interrupt for is=%p\n", is));

      {
	gm_size_t e_addr_table_size, e_name_table_size;

	e_addr_table_size =
	  (is->max_node_id + 1) * sizeof (gm_unique_id_64_t);
	gm_bzero (is->ethernet.addr_tab.volatile_ethernet_addr,
		  e_addr_table_size);

	e_name_table_size = (is->max_node_id + 1) * GM_MAX_HOST_NAME_LEN;
	gm_bzero (is->name_table.entry, e_name_table_size);
      }

      break;



      /* Determine which port caused the interrupt, and wake it. */
    case GM_WAKE_INTERRUPT:
      {
	unsigned int port;
	gm_port_state_t *ps;

	port = gm_read_lanai_global_u32 (is, interrupt.wake.port);
	GM_PRINT (GM_INTR_PRINT, ("Waking port %d if needed.\n", port));
	gm_assert (port < GM_NUM_PORTS);

	ps = __gm_port_state_for_id (is, port);
	if (!ps)
	  GM_PANIC (("Could not wake port %d, which is not open.\n", port));
	if (port == GM_MAPPER_PORT_ID)
	  GM_PRINT (GM_INTR_PRINT, ("Waking mapper port.\n"));
	gm_arch_wake (&ps->sleep_sync);
      }
      break;

    case GM_ETHERNET_RECV_INTERRUPT:
      GM_PRINT (GM_INTR_PRINT,
		("Performing ethernet recv callback. (0x%x, %d, %x)\n",
		 is->ethernet_recv_intr_context,
		 (int) (gm_read_lanai_global_u32
			(is, interrupt.ethernet_recv.len)),
		 (int) (gm_read_lanai_global_u16
			(is, interrupt.ethernet_recv.checksum))));
      if (!is->ethernet_recv_intr_callback)
	{
	  GM_WARN (("No ethernet recv callback for ethernet interrupt.\n"));
	  break;
	}
      is->ethernet_recv_intr_callback
	(is->ethernet_recv_intr_context,
	 (unsigned int) gm_read_lanai_global_u32 (is,
						  interrupt.ethernet_recv.
						  len),
	 gm_read_lanai_global_u16 (is, interrupt.ethernet_recv.checksum));
      break;

    case GM_ETHERNET_SENT_INTERRUPT:
      GM_PRINT (GM_INTR_PRINT, ("Performing ethernet sent callback.\n"));
      if (!is->ethernet_sent_intr_callback)
	{
	  GM_WARN (("No ethernet sent callback for ethernet interrupt.\n"));
	  break;
	}
      is->ethernet_sent_intr_callback (is->ethernet_sent_intr_context);
      break;

      /* Print a debugging message from the LANai. */

    case GM_PRINT_INTERRUPT:
      {
	gm_lp_t print_string;
	char *s;

	print_string = gm_read_lanai_global_lp (is, interrupt.print.string);
	gm_assert (print_string);
	s = (char *) is->lanai.sram + print_string;
	/* NOTE: don't ever turn this off here.  Turn it off in the MCP. */
#if !GM_OS_VXWORKS
	GM_INFO (("@@@@@@@@@@@@@ lanai sez: %s\n", s));
#endif /* GM_OS_VXWORKS */
      }
      break;

      /* Print the details of a failed assertion in the LANai. */

    case GM_FAILED_ASSERTION_INTERRUPT:
      if (gm_read_lanai_global_lp (is, interrupt.failed_assertion.text))
	{
	  gm_lp_t gm_file;
	  gm_lp_t gm_text;
	  gm_u32_t gm_line;

	  gm_file
	    = gm_read_lanai_global_lp (is, interrupt.failed_assertion.file);
	  gm_line
	    = gm_read_lanai_global_u32 (is, interrupt.failed_assertion.line);
	  gm_text
	    = gm_read_lanai_global_lp (is, interrupt.failed_assertion.text);

	  GM_NOTE (("*********** Failed assertion in LANai MCP (%s:%d %s)\n",
		    (char *) is->lanai.sram + gm_file, gm_line,
		    ((char *) is->lanai.sram + gm_text)));
	}
      else
	{
	  GM_NOTE (("Failed assertion in LANai MCP (%s:%d)\n",
		    ((char *) is->lanai.sram
		     + gm_read_lanai_global_lp (is,
						interrupt.failed_assertion.
						file)),
		    gm_read_lanai_global_u32 (is,
					      interrupt.failed_assertion.
					      line)));
	}

      GM_PRINT
	(GM_TRACE_LANAI_DMA,
	 ("L7 DMA len=%d lar=0x%x ear=0x%x%08x\n",
	  gm_read_lanai_global_u32 (is, dma_descriptor.len),
	  gm_read_lanai_global_lp (is, dma_descriptor.lanai_addr),
	  gm_read_lanai_global_u32 (is,
				    dma_descriptor.ebus_addr_high),
	  gm_read_lanai_global_u32 (is, dma_descriptor.ebus_addr_low)));
      GM_PRINT
	(GM_TRACE_LANAI_DMA,
	 ("L4 DMA specials EAR=0x%x LAR=0x%x DMA_CTR=0x%x\n",
	  gm_read_lanai_special_reg_u32 (is, l4.gmEAR),
	  gm_read_lanai_special_reg_u32 (is, l4.LAR),
	  gm_read_lanai_special_reg_u32 (is, l4.DMA_CTR)));

      /* gm_disable_lanai (is); */
      break;

    default:
      type_retry =
	(enum gm_interrupt_type) gm_read_lanai_global_u32 (is,
							   interrupt.type);
      /* some pyxis chipsets are buggy, reading across PCI with PIO
         may return -1, be kind and give the machine a second chance */
      if (type == (gm_u32_t) - 1 && type_retry != (gm_u32_t) - 1)
	{
	  type = type_retry;
	  goto retry;
	}
      GM_WARN (("gm_intr: unit=%d got an unrecognized interrupt of type"
		" 0x%x,retry=0x%x\n", is->id, type, type_retry));
      gm_disable_lanai (is);
      break;

    case GM_WRITE_INTERRUPT:

      /* Perform writes using "GM_PRINT (GM_PRINT_LEVEL >= 0, ("%s",
         buf))".  We don't use "GM_PRINT (GM_PRINT_LEVEL >= 0, (buf))"
         to prevent printf-like interpretation of the data in buf.  It
         is safe to null-terminate the buffer in LANai sram because
         the MCP blocks until this interrupt is handled. */

      /* HACK: Here we use __gm_pio_read_u8() and __gm_pio_write_u8() directly,
         since we cannot use gm_kton_u8() and gm_ntok_u8() in the interrupt
         handler, because we must do a partword PIO and cannot make the
         target 32 bits, and because it happens to be safe because
         the write() implementation in the LANai blocks until the interrupt is
         completed. */

      {
	gm_u32_t len;
	gm_lp_t from_lp;
	gm_u8_n_t save;
	gm_u8_n_t *from;

	from_lp = gm_read_lanai_global_lp (is, interrupt.write.buffer);
	from = (gm_u8_n_t *) is->lanai.sram + from_lp;
	len = gm_read_lanai_global_u32 (is, interrupt.write.length);
	save = __gm_pio_read_u8 (is, &from[len]);
	/* HACK (see above) */
	___gm_pio_write_u8 (&from[len], gm_hton_u8 (0));
	GM_INFO (("LANAI[%d,0x%x,%p,%d]: GM_WRITE_INTERRUPT\n    %s\n",
                   is->id, from_lp, from, len, from));
	___gm_pio_write_u8 (&from[len], save);	/* HACK (see above) */
	break;
      }

    case GM_NO_INTERRUPT:
#if !GM_ENABLE_PARANOID_INTERRUPTS
      GM_NOTE (("spurious interrupt detected\n"));
      gm_disable_lanai (is);
#endif
      break;
    }

  GM_PRINT (GM_INTR_PRINT, ("gm_intr: done with handling \n"));
  /* remove the interrupt from the interrupt queue. */

  GM_STBAR ();
  gm_write_lanai_global_u32 (is, interrupt.type, GM_NO_INTERRUPT);

  /* Claim the interrupt. */

  GM_STBAR ();
  is->set_ISR (is, GM_HOST_SIG_BIT);
  GM_STBAR ();
  gm_set_EIMR (is, GM_HOST_SIG_BIT);	/* Solaris bug workaround */
  GM_STBAR ();

  /* read back the ISR to force the write to land in the LANai */
  /* before returning from the interrupt routine */
  is->get_ISR (is);

  GM_PRINT (GM_INTR_PRINT, ("gm_intr: return from interrupt \n"));
}

/* Interrupt handler.

   This function is invoked when the LANai generates an interrupt.
   The function determines the cause of the interrupt (pause, resume,
   or port wake), and acts accordingly.  Wake events cause the
   corresponding sleeping port to be awakened.

   The MCP and the interrupt handler interract such that each
   interrupt corresponds to exactly one interrupt event.  This is done
   by having the LANai block until any earlier posted interrupt is
   acknowleged (by clearing the GM_HOST_SIG_BIT of ISR) before posting
   the next interrupt (by setting GM_HOST_SIG_BIT). The host clears
   GM_HOST_SIG_BIT after processing each interrupt. */

int gm_in_intr;

gm_s32_t gm_intr (gm_instance_state_t * is)
{
  gm_s32_t ret;

  gm_in_intr = 1;
  ret = gm_interrupting (is);
  if (ret == GM_ARCH_INTR_CLAIMED)
    {
      gm_handle_claimed_interrupt (is);
    }
  gm_in_intr = 0;
  return ret;
}
#endif

/*
 * gm_ioctl() perform most IOCTLs in a platform-independent
 * fashion.  System that do not know the input and output buffer
 * lengths (most unix drivers) should pass INT_MAX for IN_LEN and
 * OUT_LEN, effectively disabling the length checks, and should
 * implement gm_copyin() and gm_copyout() to return an
 * error on buffer overflow.
 */

#if GM_CPU_sparc64 && GM_OS_LINUX
struct gm_off_len_uvma32
{
  gm_u32_t offset;
  gm_u32_t len;
  gm_up_t uvma;
};
#endif

#define GM_BREAK_WITH_STATUS(s) do{status=(s);goto done;}while(0)

/* This GM_REQUIRE_OPEN_PORT macro should not be used for any ioctl
   that covers a kernel version of a GM API function, because the
   check should be inside the API function in this case, to ensure
   that the check is peformed when the kernel calls the function. */

#define GM_REQUIRE_OPEN_PORT() do {					\
  if (!ps->opened)							\
    {									\
      GM_NOTE (("User attempted port IOCTL on uninitialized port.\n"));	\
      GM_RETURN_STATUS (GM_PERMISSION_DENIED);				\
    }									\
} while (0)

#define GM_DEBUG_IOCTL 0
gm_status_t
gm_ioctl (gm_port_state_t * ps, unsigned int cmd,
	  void *in_buff, unsigned int in_buff_len,
	  void *out_buff, unsigned int out_buff_len, unsigned int *copied_out)
{
  gm_status_t status;
  gm_port_t *gm_fake_port;	/* HACK */
  gm_u32_t output_bytes = 0;
  gm_u32_t register_by_struct = 0;
  gm_u32_t can_copy_status = 1;
  
  GM_PRINT (GM_DEBUG_IOCTL, ("gm_ioctl called with cmd 0x%x %s.\n",
			     cmd, _gm_ioctl_cmd_name (cmd)));

  /* record the input and output buffers for gm_copyin() and gm_copyout() */

  ps->in_buff = in_buff;
  ps->out_buff = out_buff;

  /* Compute a pointer to a fake port such that the "kernel_port_state" field
     of the port is valid. */

  gm_fake_port
    = (gm_port_t *) ((char *) &ps
		     - GM_OFFSETOF (struct gm_port, kernel_port_state));
  
  switch (cmd)
    {
    case GM_SET_REGISTER_MEMORY_LENGTH:
      GM_PRINT (GM_DEBUG_IOCTL, ("gm_ioctl: GM_SET_REGISTER_MEMORY_LENGTH\n"));
      GM_REQUIRE_OPEN_PORT ();
      GM_PRINT (GM_PRINT_LEVEL >= 4,
		("Setting memory (de)registration length.\n"));
      if (in_buff_len < sizeof (gm_u32_t))
	{
	  GM_PRINT (GM_PRINT_LEVEL >= 4,
		    ("gm_ioctl: GM_SET_REGISTER_MEMORY_LENGTH %d < %d \n",
		     in_buff_len, sizeof (gm_u32_t)));
	  GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
	}
      {
	gm_u32_t mem_reg_len;
	status = gm_arch_copyin (ps, in_buff, &mem_reg_len,
				 sizeof (mem_reg_len));
	ps->mem_reg_len = mem_reg_len;
      }
      if (status != GM_SUCCESS)
	{
	  GM_NOTE (("could not read memory registration length.\n"));
	  break;
	}
      else
	{
	  GM_PRINT
	    (GM_PRINT_LEVEL >= 4,
	     ("gm_ioctl: GM_SET_REGISTER_MEMORY_LENGTH OK so far = %d\n",
	      ps->mem_reg_len));
	}
      break;

    case GM_REGISTER_MEMORY_BY_STRUCT:
      register_by_struct = 1;
    case GM_REGISTER_MEMORY:
      if (GM_CAN_REGISTER_MEMORY)
	{
	  gm_up_t user_vma;
	  gm_page_hash_table_t *hash;
	  gm_u32_t offset;

	  GM_REQUIRE_OPEN_PORT ();
	  GM_PRINT (GM_DEBUG_MEM_REGISTER,
		    ("GM_REGISTER_MEMORY%s: in_buff = %p\n",
		     register_by_struct ? "_BY_STRUCT" : "", in_buff));

	  user_vma = (gm_up_t) in_buff;
	  hash = &ps->instance->page_hash;

	  if (cmd == GM_REGISTER_MEMORY)
	    {
	      /* Can't touch the user's memory space */
	      can_copy_status = 0;
	    }

	  /* verify user and kernel pointers are the same size. */
	  gm_assert (sizeof (void *) == sizeof (gm_up_t));

	  if (ps->id == GM_DAEMON_PORT_ID)
	    {
	      status = GM_PERMISSION_DENIED;
	      break;
	    }

	  if (register_by_struct)
	    {
	      struct gm_off_len_uvma s;
	      if (in_buff_len < sizeof (struct gm_off_len_uvma))
		{
		  GM_PRINT
		    (GM_DEBUG_IOCTL,
		     ("gm_ioctl: GM_REGISTER_MEMORY_BY_STRUCT %d < %d \n",
		      in_buff_len, sizeof (struct gm_off_len_uvma)));
		  GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
		}
	      status
		= gm_arch_copyin (ps, in_buff, &s,
				  sizeof (struct gm_off_len_uvma));
	      if (status != GM_SUCCESS)
		{
		  GM_NOTE (("could not read memory registration struct.\n"));
		  break;
		}
	      else
		{
		  GM_PRINT
		    (GM_PRINT_LEVEL >= 4,
		     ("Copied register_memory_struct information\n"));
		}
	      user_vma = s.uvma;
	      ps->mem_reg_len = GM_STATIC_CAST (unsigned long, s.len);
	    }

	  GM_PRINT (GM_DEBUG_MEM_REGISTER,
		    ("registering 0x%x bytes at user addr 0x%lx\n",
		     ps->mem_reg_len, (unsigned long) user_vma));

	  /* verify there is something to (de)register */

	  if ((!register_by_struct) && (in_buff_len < ps->mem_reg_len))
	    {
	      GM_NOTE (("User memory registration buffer too small.\n"));
	      status = GM_INPUT_BUFFER_TOO_SMALL;
	      break;
	    }

	  /* Verify buffer edges are page aligned. */

	  if (!ps->mem_reg_len
	      || !GM_PAGE_ALIGNED (ps->mem_reg_len)
	      || !GM_PAGE_ALIGNED (user_vma))
	    {
	      GM_NOTE (("User asked to register buffer of non-page size"
			" or alignment.\n"));
	      GM_PRINT (GM_PRINT_LEVEL >= 0,
			("mem_reg_len=%d len_aligned=%d vma_aligned=%d\n",
			 ps->mem_reg_len, GM_PAGE_ALIGNED (ps->mem_reg_len),
			 GM_PAGE_ALIGNED (user_vma)));
	      status = GM_INVALID_PARAMETER;
	      break;
	    }

	  gm_arch_mutex_enter (&hash->sync);

	  /* Register the memory one page at a time.  Extreme care must
	     be taken to wrap all accesses to the user-supplied buffer
	     in exception handlers in case the user has passed a bad
	     buffer. */

	  offset = 0;
	  while (offset < ps->mem_reg_len)
	    {
	      /* Register the page, storing the "mdl" using the user pointer
	         field. */

	      gm_assert (ps);
	      gm_assert (ps->instance);
	      GM_PRINT (GM_DEBUG_MEM_REGISTER,
			("Adding mapping for unit %d - call "
                         "gm_add_mapping_to_page_table (\n"
                         "    %p, %p, %d, 0)\n", ps->instance->id,
                         ps->instance, user_vma + offset, ps->id));
	      status =
		gm_add_mapping_to_page_table (ps->instance, user_vma + offset,
					      ps->id, 0);
	      if (status != GM_SUCCESS)
		{
		  status = GM_INVALID_PARAMETER;
		  GM_NOTE (("gm_add_mapping_to_page_table failed"
			    " len=%d (0x%x) bytes?\n",
			    ps->mem_reg_len, ps->mem_reg_len));
		  goto gm_register_loop__abort;
		}

	      /* Prepare for next iteration */

	      offset += GM_PAGE_LEN;
	    }
	  gm_arch_mutex_exit (&hash->sync);
	  status = GM_SUCCESS;
	  break;

	gm_register_loop__abort:
	  /* loic: we need to pause, the Lanai may have loaded some
	     entries into its cache */
	  gm_pause_lanai (ps->instance);
	  while (offset > 0)
	    {
	      offset -= GM_PAGE_LEN;
	      gm_dereference_mapping (ps->instance, user_vma + offset,
				      ps->id);
	    }
	  gm_unpause_lanai (ps->instance);
	  gm_arch_mutex_exit (&hash->sync);
	  break;
	}
      else
	{
	  status = GM_INVALID_COMMAND;
	  break;
	}

      /* Unregister the indicated pages. */

    case GM_DEREGISTER_MEMORY_BY_STRUCT:
      register_by_struct = 1;
    case GM_DEREGISTER_MEMORY:
      GM_REQUIRE_OPEN_PORT ();
      GM_PRINT (GM_DEBUG_MEM_REGISTER,
		("GM_DEREGISTER_MEMORY%s\n",
		 register_by_struct ? "_BY_STRUCT" : ""));
      if (ps->id == GM_DAEMON_PORT_ID)
	{
	  status = GM_PERMISSION_DENIED;
	  break;
	}
      {
	gm_u32_t offset;
	gm_page_hash_table_t *hash;
	gm_up_t user_vma, save_user_vma;

	if (register_by_struct)
	  {
	    struct gm_off_len_uvma s;
	    if (in_buff_len < sizeof (struct gm_off_len_uvma))
	      {
		GM_PRINT (GM_PRINT_LEVEL >= 4,
			  ("gm_ioctl: GM_DEREGISTER_MEMORY_BY_STRUCT "
			   "%d < %d \n", in_buff_len,
			   sizeof (struct gm_off_len_uvma)));
		GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
	      }
	    status =
	      gm_arch_copyin (ps, in_buff, &s,
			      sizeof (struct gm_off_len_uvma));
	    if (status != GM_SUCCESS)
	      {
		GM_NOTE (("could not read memory deregistration struct.\n"));
		break;
	      }
	    else
	      {
		GM_PRINT (GM_PRINT_LEVEL >= 4,
			  ("Copied deregister_memory_struct information\n"));
	      }
	    user_vma = save_user_vma = s.uvma;
	    ps->mem_reg_len = GM_STATIC_CAST (unsigned  long, s.len);
	  }

	else
	  {
	    if (in_buff_len < ps->mem_reg_len)
	      {
		GM_NOTE (("Memory registration buffer too small.\n"));
		status = GM_INPUT_BUFFER_TOO_SMALL;
		goto gm_deregister_abort_with_nothing;
	      }

	    GM_PRINT (GM_PRINT_LEVEL >= 4,
		      ("Type3InputBuffer = %p.\n", in_buff));
	    if (!in_buff)
	      {
		GM_NOTE (("User did not specify input buffer.\n"));
		status = GM_INVALID_PARAMETER;
		goto gm_deregister_abort_with_nothing;
	      }

	    user_vma = save_user_vma = (gm_up_t) in_buff;
	  }

	hash = &ps->instance->page_hash;

	/* Verify that each of the pages is in the page table.  If
	   not, the user tried to deregister an illegitimate
	   region, and we fail. */

	gm_arch_mutex_enter (&hash->sync);

	GM_PRINT (GM_DEBUG_MEM_REGISTER,
		  ("deregistering 0x%x bytes at user addr 0x%lx\n",
		   ps->mem_reg_len, (unsigned long) user_vma));

	offset = 0;
	while (offset < ps->mem_reg_len)
	  {
	    if (!gm_refs_to_mapping_in_page_table (ps->instance,
						   user_vma + offset, ps->id))
	      {
		GM_NOTE (("User tried to deregister unregistered page.\n"));
		status = GM_INVALID_PARAMETER;
		goto gm_deregister_abort_with_mutex;
	      }
	    offset += GM_PAGE_LEN;
	  }

	/* Dereference each page of the buffer, and unlock the page if
	   it is no longer referenced. */

	GM_PRINT (GM_DEBUG_MEM_REGISTER,
		  ("Deregistering user buffer pages.\n"));
	gm_pause_lanai (ps->instance);
	offset = 0;
	user_vma = save_user_vma;
	while (offset < ps->mem_reg_len)
	  {
	    gm_dereference_mapping (ps->instance, user_vma + offset, ps->id);

	    /* iterate */
	    offset += GM_PAGE_LEN;
	  }
	gm_unpause_lanai (ps->instance);
	gm_arch_mutex_exit (&hash->sync);
	status = GM_SUCCESS;
	break;

      gm_deregister_abort_with_mutex:
	gm_arch_mutex_exit (&hash->sync);
      gm_deregister_abort_with_nothing:
	break;
      }
#undef break

      /************
       * Get the page_hash_cache_size  */

    case GM_GET_PAGE_HASH_CACHE_SIZE:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_PAGE_HASH_CACHE_SIZE\n"));
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = _gm_get_page_hash_cache_size(gm_fake_port, in_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (unsigned int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);


      /************
       * Get the max node id for this device. */

    case GM_GET_MAX_NODE_ID:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_MAX_NODE_ID\n"));
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = gm_max_node_id (gm_fake_port, in_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (unsigned int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Get the max node id inuse for this device. */

    case GM_GET_MAX_NODE_ID_INUSE:
      GM_PRINT (GM_PRINT_LEVEL >= 3,
		("gm_ioctl:: GM_GET_MAX_NODE_ID_INUSE\n"));
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = gm_max_node_id_inuse (gm_fake_port, in_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (unsigned int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Set the process_id of the opener of this port */

    case GM_SET_OPENER_PID:
      {
	int pid;
	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_ioctl:: GM_SET_OPENER_PID\n"));
	if (in_buff_len < sizeof (gm_u32_t))
	  GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
	status = gm_copyin (ps, in_buff, &pid, sizeof (gm_u32_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_set_opener_pid (gm_fake_port, pid);
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	GM_BREAK_WITH_STATUS (GM_SUCCESS);
      }

      /************
       * Get the process_ids of the openers of each port */

    case GM_GET_OPENER_PIDS:
      GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_ioctl:: GM_GET_OPENER_PIDS\n"));
      if (out_buff_len < (GM_NUM_PORTS * sizeof (gm_u32_t)))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = _gm_get_opener_pids (gm_fake_port, out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = (GM_NUM_PORTS * sizeof (gm_u32_t));
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Get the host id for this device. */

    case GM_GET_NODE_ID:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_NODE_ID\n"));
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = gm_get_node_id (gm_fake_port, (unsigned int *) out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (unsigned int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Get the host name. */

    case GM_GET_HOST_NAME:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_HOST_NAME\n"));
      if (in_buff_len < GM_MAX_HOST_NAME_LEN)
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      status = gm_get_host_name (gm_fake_port, in_buff);
      output_bytes = GM_MAX_HOST_NAME_LEN;
      GM_BREAK_WITH_STATUS (status);

      /************
       * Get a string from the LANai. */

    case GM_GET_FIRMWARE_STRING:
      if (!ps->privileged)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (in_buff_len < GM_MAX_FIRMWARE_STRING_LEN)
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      {
	gm_lp_t lanai_ptr;

	status = gm_copyin (ps, in_buff, &lanai_ptr, sizeof (lanai_ptr));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_get_firmware_string (gm_fake_port, lanai_ptr, out_buff, &output_bytes);
	output_bytes += 1;
	GM_BREAK_WITH_STATUS (status);
      }

      /************
       * convert a node id to a unique id */

    case GM_NODE_ID_TO_UNIQUE_ID:
      /* Might be called once per node, so has higher print level */
      GM_PRINT (GM_PRINT_LEVEL >= 11,
		("gm_ioctl:: GM_NODE_ID_TO_UNIQUE_ID\n"));
      if (out_buff_len < 6)
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      if (in_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      {
	unsigned int node_id;
	status = gm_copyin (ps, in_buff, &node_id, sizeof (node_id));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	if (node_id > ps->instance->max_node_id)
	  GM_BREAK_WITH_STATUS (GM_INVALID_PARAMETER);
	status = gm_node_id_to_unique_id (gm_fake_port, node_id,
					  (char *) out_buff);

	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
      }
      output_bytes = 6;
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * convert a unique id to a node id */

    case GM_UNIQUE_ID_TO_NODE_ID:
      /* Might be called once per node, so has higher print level */
      GM_PRINT (GM_PRINT_LEVEL >= 11,
		("gm_ioctl:: GM_UNIQUE_ID_TO_NODE_ID\n"));
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      if (in_buff_len < 6)
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      status = gm_unique_id_to_node_id (gm_fake_port,
					(char *) in_buff,
					(unsigned int *) out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = 6;
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Return the page size, as computed by the driver. */

    case GM_GET_PAGE_LEN:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_PAGE_LEN\n"));
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      if (!GM_PAGE_LEN)
	GM_BREAK_WITH_STATUS (GM_INTERNAL_ERROR);
      status = gm_copyout (ps, &GM_PAGE_LEN, out_buff, sizeof (GM_PAGE_LEN));
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (gm_u32_t);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Return the gm_mapping_specs structure for the port.  This
       * structure specifies the locations of the various
       * memory-mapped regions that the user may map. */

    case GM_GET_MAPPING_SPECS:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_MAPPING_SPECS\n"));
      if (out_buff_len < sizeof (gm_mapping_specs_t))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = _gm_get_mapping_specs (gm_fake_port,
				      (gm_mapping_specs_t *) out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (gm_mapping_specs_t);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Get board "ethernet" ID. */

    case GM_GET_UNIQUE_BOARD_ID:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_UNIQUE_BOARD_ID\n"));
      if (in_buff_len < 6)
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      status = gm_unique_id (gm_fake_port, (char *) out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = 6;
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Get "ethernet" ID of the mapper node. */

    case GM_GET_MAPPER_UNIQUE_ID:
      GM_PRINT (GM_PRINT_LEVEL >= 3,
		("gm_ioctl:: GM_GET_MAPPER_UNIQUE_ID\n"));
      if (in_buff_len < 6)
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      status = gm_get_mapper_unique_id (gm_fake_port, (char *) out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = 6;
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Set the mask of sizes that are not acceptable to the receiving port */

    case GM_SET_ACCEPTABLE_SIZES_LOW:
      GM_PRINT (GM_PRINT_LEVEL >= 3,
		("gm_ioctl:: GM_SET_ACCEPTABLE_SIZES_LOW\n"));
      if (in_buff_len < sizeof (gm_u32_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	gm_u32_t sizes;
	status = gm_copyin (ps, in_buff, &sizes, sizeof (sizes));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status =
	  gm_set_acceptable_sizes (gm_fake_port, GM_LOW_PRIORITY, sizes);
	GM_BREAK_WITH_STATUS (status);
      }

    case GM_SET_ACCEPTABLE_SIZES_HIGH:
      GM_PRINT (GM_PRINT_LEVEL >= 3,
		("gm_ioctl:: GM_SET_ACCEPTABLE_SIZES_HIGH\n"));
      if (in_buff_len < sizeof (gm_u32_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	gm_u32_t sizes;
	status = gm_copyin (ps, in_buff, &sizes, sizeof (sizes));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status =
	  gm_set_acceptable_sizes (gm_fake_port, GM_HIGH_PRIORITY, sizes);
	GM_BREAK_WITH_STATUS (status);
      }

      /************
       * Convert a host name to a node ID */

    case GM_HOST_NAME_TO_NODE_ID:
      GM_PRINT (GM_PRINT_LEVEL >= 3,
		("gm_ioctl:: GM_HOST_NAME_TO_NODE_ID\n"));
      if (in_buff_len < GM_MAX_HOST_NAME_LEN)
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      {
	unsigned int node_id;

	node_id = gm_host_name_to_node_id (gm_fake_port, (char *) in_buff);
	status = gm_copyout (ps, &node_id, out_buff, sizeof (node_id));
      }
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (unsigned int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Convert a node ID to a host name */

    case GM_NODE_ID_TO_HOST_NAME:
      GM_PRINT (GM_PRINT_LEVEL >= 11,
		("gm_ioctl:: GM_NODE_ID_TO_HOST_NAME\n"));
      if (in_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (out_buff_len < GM_MAX_HOST_NAME_LEN)
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      {
	unsigned int node_id;
	char *host_name;

	status = gm_copyin (ps, in_buff, &node_id, sizeof (node_id));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	if (node_id > ps->instance->max_node_id)
	  GM_BREAK_WITH_STATUS (GM_INVALID_PARAMETER);
	host_name = gm_node_id_to_host_name (gm_fake_port, node_id);
	if (!host_name)
	  GM_BREAK_WITH_STATUS (GM_INVALID_PARAMETER);
	status = gm_copyout (ps, host_name, out_buff, GM_MAX_HOST_NAME_LEN);
      }
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = GM_MAX_HOST_NAME_LEN;
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /*******************************
      ********************************
      **** Trivial ioctls
      ********************************
      ********************************/

      /************
       * Copy the LANai eeprom to a user buffer */

    case GM_GET_EEPROM:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_EEPROM\n"));
      if (out_buff_len < sizeof (gm_myrinet_eeprom_t))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = _gm_get_eeprom (gm_fake_port, out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (gm_myrinet_eeprom_t);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Copy the LANai globals to a user buffer */

    case GM_GET_GLOBALS:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_GLOBALS\n"));
      if (out_buff_len < sizeof (struct gm_lanai_globals))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = _gm_get_globals (gm_fake_port, out_buff,
				(sizeof (struct gm_lanai_globals)
				 + ((ps->instance->max_node_id+1) * 
					sizeof (struct gm_connection))));
      /* (ps->instance->max_node_id+1) was 1024 */

      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = out_buff_len;
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

    case GM_GET_GLOBALS_BY_REQUEST:
      GM_PRINT (GM_PRINT_LEVEL >= 4,
		("gm_ioctl:: GM_GET_GLOBALS_BY_REQUEST\n"));
      {
	gm_u32_t space[2];
	gm_globals_request_t *request = (gm_globals_request_t *) & space[0];

	if (in_buff_len < 2 * sizeof (gm_u32_t))
	  {
	    GM_PRINT (GM_PRINT_LEVEL >= 0,
		      ("gm_ioctl: GM_GET_GLOBALS_BY_REQUEST %d < %d \n",
		       in_buff_len, 2 * sizeof (gm_u32_t)));
	    GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
	  }
	status = gm_arch_copyin (ps, in_buff, request, 2 * sizeof (gm_u32_t));
	if (status != GM_SUCCESS)
	  {
	    GM_NOTE (("could not read globals request struct.\n"));
	    break;
	  }
	else
	  {
	    GM_PRINT (GM_PRINT_LEVEL >= 4,
		      ("Copied globals request struct. off=%d len=%d\n",
		       request->offset, request->len));
	  }

	status = _gm_get_globals_by_request (gm_fake_port, out_buff,
					     request->offset, request->len);
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);

	output_bytes = request->len;
      }
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

    case GM_GET_GLOBALS_OFFSET:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_GLOBALS_OFFSET\n"));
      if (out_buff_len < sizeof (gm_u32_t))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      /* note the 1024 is a HACK */
      {
	gm_u32_t offset;

	offset = (gm_u32_t) ((char *) ps->instance->lanai.globals
			     - (char *) ps->instance->lanai.sram);
	status = gm_arch_copyout (ps, &offset, out_buff, sizeof (offset));
      }
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (gm_u32_t);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /*******************************
      ********************************
      **** Mapper-specific IOCTLS ****
      ********************************
      ********************************/

      /************
       * Allow the caller to perform raw receives */

    case GM_ENABLE_RAW_RECEIVES:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_ENABLE_RAW_RECEIVES\n"));
      if (ps->id != GM_MAPPER_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (_gm_enable_raw_receives (gm_fake_port) != GM_SUCCESS)
	status = GM_PERMISSION_DENIED;
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Set a route. */

    case GM_SET_ROUTE:
      /* Might be called once per node, so has higher print level */
      GM_PRINT (GM_PRINT_LEVEL >= 11, ("gm_ioctl:: GM_SET_ROUTE\n"));
      if (ps->id != GM_MAPPER_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (in_buff_len < sizeof (gm_route_info_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	gm_route_info_t ri;
	status = gm_copyin (ps, in_buff, &ri, sizeof (gm_route_info_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_set_route (gm_fake_port,
				ri.target_node_id, ri.length,
				(char *) &(ri.route[0]));
	GM_BREAK_WITH_STATUS (status);
      }

	/***********
	 * Clear all routes. */
    case GM_CLEAR_ALL_ROUTES:
      GM_PRINT (GM_PRINT_LEVEL >= 7, ("gm_ioctl:: GM_CLEAR_ALL_ROUTES\n"));
      if (ps->id != GM_MAPPER_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	status = _gm_clear_all_routes (gm_fake_port);
	GM_BREAK_WITH_STATUS (status);
      }

      /************
       * Get a route. */

    case GM_GET_ROUTE:
      GM_PRINT (GM_PRINT_LEVEL >= 11, ("gm_ioctl:: GM_GET_ROUTE\n"));
      if (in_buff_len < sizeof (gm_route_info_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (out_buff_len < sizeof (gm_route_info_t))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      {
	gm_route_info_t ri;
	unsigned int route_length;

	status = gm_copyin (ps, in_buff, &ri, sizeof (gm_route_info_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	route_length = ri.length;
	status = _gm_get_route (gm_fake_port,
				ri.target_node_id, (char *) ri.route,
				&route_length);
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	ri.length = (gm_u8_t) route_length;
	status = gm_copyout (ps, &ri, out_buff, sizeof (gm_route_info_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	output_bytes = sizeof (gm_route_info_t);
	GM_BREAK_WITH_STATUS (GM_SUCCESS);
      }

      /************
       * Set the node ID.  For mapper only! */

    case GM_SET_NODE_ID:
      GM_PRINT (GM_DEBUG_IOCTL, ("case GM_SET_NODE_ID\n"));
      gm_assert (ps);
      if (!ps->privileged)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (in_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	unsigned int node_id;
	status = gm_copyin (ps, in_buff, &node_id, sizeof (unsigned int));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_set_node_id (gm_fake_port, node_id);
	GM_BREAK_WITH_STATUS (status);
      }

      /************
       * Set the mapper level.  For mapper only! */

    case GM_SET_MAPPER_LEVEL:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_SET_MAPPER_LEVEL\n"));
      if (ps->id != GM_MAPPER_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (in_buff_len < sizeof (gm_s32_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	gm_s32_t level;
	status = gm_copyin (ps, in_buff, &level, sizeof (gm_s32_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_set_mapper_level (gm_fake_port, level);
	GM_BREAK_WITH_STATUS (status);
      }



      /************
       * Set the node type*/

    case GM_SET_NODE_TYPE:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_SET_NODE_TYPE\n"));
      if (ps->id != GM_MAPPER_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (in_buff_len < sizeof (gm_s32_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	gm_s32_t node_type;
	status = gm_copyin (ps, in_buff, &node_type, sizeof (gm_s32_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_set_node_type (gm_fake_port, node_type);
	GM_BREAK_WITH_STATUS (status);
      }

      /************
       * Get the node type*/

    case GM_GET_NODE_TYPE:

      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_NODE_TYPE\n"));
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = gm_get_node_type (gm_fake_port, (int *) out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

      /************
       * Set enable_nack_down flag for this port */

    case GM_SET_ENABLE_NACK_DOWN:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_SET_ENABLE_NACK_DOWN\n"));
      if (in_buff_len < sizeof (gm_u32_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      {
	gm_u32_t flag;
	status = gm_copyin (ps, in_buff, &flag, sizeof (gm_u32_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = gm_set_enable_nack_down(gm_fake_port, flag);
	GM_BREAK_WITH_STATUS (status);
      }

      /*cop support */
    case GM_COP_WAKEUP:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_COP_WAKEUP\n"));
      if (GM_ENABLE_SECURITY
	  && (!ps->privileged || ps->id != GM_MAPPER_PORT_ID))
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      status = _gm_cop_wakeup (gm_fake_port);
      GM_BREAK_WITH_STATUS (status);

    case GM_COP_END:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_COP_END\n"));
      if (GM_ENABLE_SECURITY
	  && (!ps->privileged || ps->id != GM_MAPPER_PORT_ID))
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      status = _gm_cop_end (gm_fake_port);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

    case GM_COP_SEND:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_COP_SEND\n"));
      if (GM_ENABLE_SECURITY
	  && (!ps->privileged || ps->id != GM_MAPPER_PORT_ID))
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (in_buff_len < sizeof (gm_s32_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      {
	gm_s32_t d;
	status = gm_copyin (ps, in_buff, &d, sizeof (gm_s32_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_cop_send (gm_fake_port, d);
	GM_BREAK_WITH_STATUS (status);
      }

    case GM_COP_RECEIVE:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_COP_SEND\n"));
      if (GM_ENABLE_SECURITY
	  && (!ps->privileged || ps->id != GM_MAPPER_PORT_ID))
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (out_buff_len < sizeof (unsigned int))
	GM_BREAK_WITH_STATUS (GM_OUTPUT_BUFFER_TOO_SMALL);
      status = _gm_cop_receive (gm_fake_port, (int *) out_buff);
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

#if 0
      /************
       * Set the number of physical pages.  For mapper only! */

    case GM_SET_PHYSICAL_PAGES:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_SET_PHYSICAL_PAGES\n"));
      if (ps->id != GM_MAPPER_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (in_buff_len < sizeof (gm_u32_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      status = gm_copyin (ps, in_buff, &gm_physical_pages,
			  sizeof (gm_physical_pages));
      GM_BREAK_WITH_STATUS (status);
#endif

      /************
       * Set the host name.  For mapper only! */

    case GM_SET_HOST_NAME:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_SET_HOST_NAME\n"));
      if (ps->id != GM_MAPPER_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	struct
	{
	  unsigned int node_id;
	  char host_name[GM_MAX_HOST_NAME_LEN];
	}
	in;

	GM_PRINT (0, ("*** input host name between %p and %p.\n",
		      &in, &in + 1));

	if (in_buff_len < sizeof (in))
	  GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
	status = gm_copyin (ps, in_buff, &in, sizeof (in));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_set_host_name (gm_fake_port, in.node_id, in.host_name);
	GM_BREAK_WITH_STATUS (status);
      }

    case GM_SET_UNIQUE_ID:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_SET_UNIQUE_ID\n"));
      if (ps->id == GM_DAEMON_PORT_ID)
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
      {
	struct
	{
	  unsigned int node_id;
	  char unique[6];
	}
	in;

	if (in_buff_len < sizeof (in))
	  GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
	status = gm_copyin (ps, in_buff, &in, sizeof (in));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	status = _gm_set_unique_id (gm_fake_port,
				    in.node_id, (char *) &in.unique[0]);
	GM_BREAK_WITH_STATUS (status);
      }

#ifndef WIN32
    case GM_SLEEP:
      GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_ioctl:: GM_SLEEP\n"));
      status = gm_kernel_sleep (ps);
      GM_BREAK_WITH_STATUS (status);
#endif

    case GM_SET_PORT_NUM:
      if (ps->opened)
	{
	  GM_NOTE (("User attempted to re-set port number.\n"));
	  GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);
	}
      GM_PRINT (GM_DEBUG_PORT_STATE, ("handling GM_SET_PORT_NUM.\n"));

      {
	gm_u32_t port_num;

	status = gm_copyin (ps, in_buff, &port_num, sizeof (port_num));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	GM_PRINT (GM_PRINT_LEVEL >= 5,
		  ("Calling gm_port_state_open (%p, %d)\n", ps, port_num));
	status = gm_port_state_open (ps, port_num);
      }

      if (status != GM_SUCCESS)
	{
	  gm_assert (!ps->opened);
	  GM_PRINT(GM_PRINT_LEVEL>=1, ("Could not open port state.\n"));
	  break;
	}
      gm_assert (ps->opened);
      GM_PRINT (GM_DEBUG_PORT_STATE, ("opened the port state.\n"));
      break;

    case GM_GET_KERNEL_BUILD_ID_LEN:
      GM_PRINT (GM_PRINT_LEVEL >= 3,
		("gm_ioctl:: GM_GET_KERNEL_BUILD_ID_LEN\n"));
      {
	int kernel_build_id_len;

	kernel_build_id_len = (int) (gm_strlen (_gm_build_id) + 1);
	status =
	  gm_copyout (ps, &kernel_build_id_len, out_buff, sizeof (int));
      }
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = sizeof (int);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

    case GM_GET_KERNEL_BUILD_ID:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl:: GM_GET_KERNEL_BUILD_ID\n"));
      status = gm_copyout (ps, (void *) _gm_build_id, out_buff,
			   (unsigned int) (gm_strlen (_gm_build_id) + 1));
      if (status != GM_SUCCESS)
	GM_BREAK_WITH_STATUS (status);
      output_bytes = (gm_u32_t) (gm_strlen (_gm_build_id) + 1);
      GM_BREAK_WITH_STATUS (GM_SUCCESS);

    case GM_FINISH_MMAP:
      GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_ioctl: GM_FINISH_MMAP\n"));
      {
	struct gm_off_len_uvma s;
#if GM_CPU_sparc64 && GM_OS_LINUX
	struct gm_off_len_uvma32 s_32;
	status = gm_copyin (ps, in_buff, &s_32, sizeof (s_32));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	s.offset = (unsigned long) s_32.offset;
	s.len = (unsigned long) s_32.len;
	s.uvma = s_32.uvma;
#else
	status = gm_copyin (ps, in_buff, &s, sizeof (s));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
#endif
	status = _gm_finish_mmap (gm_fake_port, s.offset, s.len, s.uvma);
	GM_BREAK_WITH_STATUS (status);
      }


#if GM_ENABLE_TRACE
      /************
       * Get the kernel traces */

    case GM_GET_KTRACE:
      GM_PRINT (GM_PRINT_LEVEL >= 11, ("gm_ioctl:: GM_GET_KTRACE\n"));
      if (out_buff_len < sizeof (gm_file_trace_t) * GM_HOST_NUMTRACE)
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      {
	status = gm_get_ktrace (gm_fake_port, out_buff);
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	output_bytes = sizeof (gm_file_trace_t) * GM_HOST_NUMTRACE;
	GM_BREAK_WITH_STATUS (GM_SUCCESS);
      }
#endif

    case GM_WRITE_LANAI_REGISTER:
      if (GM_ENABLE_SECURITY
	  && (!ps->privileged || ps->id != GM_MAPPER_PORT_ID))
	GM_BREAK_WITH_STATUS (GM_PERMISSION_DENIED);

      if (in_buff_len < sizeof (gm_lanai_register_access_t))
	GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
      {
	gm_lanai_register_access_t a;
	status = gm_copyin (ps, in_buff, &a, sizeof (gm_lanai_register_access_t));
	if (status != GM_SUCCESS)
	  GM_BREAK_WITH_STATUS (status);
	/*
	if ((((char*)gm_fake_port->kernel_port_state->instance->lanai.special_regs) + a.offset) !=
	    (char*)&((gm_lanai_special_registers_t*)
	    gm_fake_port->kernel_port_state->instance->lanai.special_regs)->l9.MYRINET)
	  gm_linux_vprintk ("problem\n");
	else
	  gm_linux_vprintk ("not a problem\n");
	*/
	/*if (gm_fake_port->kernel_port_state->instance->lanai.eeprom.lanai_cpu_version == 0x900)
	  gm_write_lanai_special_reg_u32 (gm_fake_port->kernel_port_state->instance, l9.MYRINET, a.value);
	*/
	__gm_kton_u32 (gm_fake_port->kernel_port_state->instance,
	(gm_u32_n_t*)(((char*)gm_fake_port->kernel_port_state->instance->lanai.special_regs) + a.offset), a.value);
	
	GM_BREAK_WITH_STATUS (GM_SUCCESS);
      }


      
#if GM_ENABLE_DIRECTCOPY
      /************
       * Provide a memory copy between two separated user-spaces */

    case GM_DIRECTCOPY_GET:
      GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_ioctl:: GM_DIRECTCOPY_GET\n"));
#if GM_OS_LINUX
      {
	gm_directcopy_t args;
	gm_port_state_t *source_ps;
	gm_u16_t pid_source;

	if (in_buff_len < sizeof (gm_directcopy_t))
	  {
	    GM_PRINT (GM_PRINT_LEVEL >= 4,
		      ("gm_ioctl: GM_DIRECTCOPY_GET %d < %d \n", in_buff_len,
		       sizeof (gm_directcopy_t)));
	    GM_BREAK_WITH_STATUS (GM_INPUT_BUFFER_TOO_SMALL);
	  }
	status =
	  gm_arch_copyin (ps, in_buff, &args, sizeof (gm_directcopy_t));
	if (status != GM_SUCCESS)
	  {
	    GM_NOTE (("could not read GET directcopy parameters struct.\n"));
	    break;
	  }
	else
	  {
	    GM_PRINT (GM_PRINT_LEVEL >= 4,
		      ("Copied GET directcopy parameters information\n"));
	  }

	/* PS is the port state of the port doing the copy. */
	if (ps->id == args.source_port_id
	    && ps->instance->id == args.source_instance_id)
	  {
	    /* User requested copy from its own memory space.  
	       Abort to prevent attempting to acquire the lock for
	       this port state, which we already hold. */
	    goto abort_with_nothing;
	  }
	source_ps = gm_acquire_port_state_for_ids (args.source_instance_id,
						   args.source_port_id);
	if (!source_ps)
	  goto abort_with_nothing;
	if (!source_ps->opened)
	  goto abort_with_lock;
	/* we need the pid of the other process to acces its virtual
	   memory space via the kernel */
	pid_source = source_ps->opener_pid;

	/* we ask the kernel to copy the data by a GET (copy by the
	   receiver process) */
	status = gm_arch_directcopy_get ((void *) args.source_addr,
					 (void *) args.target_addr,
					 args.length, pid_source);
	/* we unlock the port state */
	gm_release_port_state (source_ps);
	break;

	/* something goes wrong ? */
      abort_with_lock:
	gm_release_port_state (source_ps);
      abort_with_nothing:
	status = GM_FAILURE;
      }
#else
      status = GM_FAILURE;
#endif
#endif

    default:
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("gm_ioctl: *** unhandled ioctl 0x%x ***\n", cmd));
      GM_BREAK_WITH_STATUS (GM_INVALID_COMMAND);
    }
done:

  GM_PRINT (GM_DEBUG_IOCTL, ("Done with switch\n"));

  /* If there was an error, inform the user via the copy buffer, ensuring
     that the OS does not get a chance to mangle the error code. */
  if (can_copy_status
      && status != GM_SUCCESS && out_buff && out_buff_len >= sizeof (status))
    {
      GM_PRINT (GM_DEBUG_IOCTL,
		("ioctl (0x%x) failed, copying status = %d (0x%x)\n", cmd,
		 status, status));
      output_bytes = sizeof (status);
      gm_copyout (ps, (void *) &status, out_buff, output_bytes);
    }
  else if (!can_copy_status)
    {
      GM_PRINT (GM_DEBUG_IOCTL,
		("gm_ioctl NOT copying status = %d\n", status));
    }

  GM_PRINT (GM_DEBUG_IOCTL, ("copied_out = %p\n", copied_out));

  if (copied_out)
    *copied_out = output_bytes;

#if 0
  /* This NT HACK is no longer used.  Instead, it is the caller's
     responsibility to localize GM error codes if needed. */
  if (status == GM_FAILURE)
    status = GM_OUT_OF_MEMORY;
#endif

  if (status != GM_SUCCESS)
    {
      GM_PRINT (GM_DEBUG_IOCTL,
		("ioctl %s returning with error status 0x%x.\n",
		 _gm_ioctl_cmd_name (cmd), status));
    }

  return status;
}

#undef GM_REQUIRE_OPEN_PORT
#undef GM_BREAK_WITH_STATUS

/************************************************************************
 * Unix mmap functions.
 ************************************************************************/

static void
gm_print_port_state (struct gm_port_state *ps)
{
  GM_PARAMETER_MAY_BE_UNUSED (ps);
  
  GM_PRINT (1, ("id = %d\n", (int) ps->id));
  _GM_PRINT (1, ("instance = %p\n", ps->instance));
  _GM_PRINT (1, ("compatibility_mode = %d\n", (int) ps->compatibility_mode));
  _GM_PRINT (1, ("privileged = %d\n", (int) ps->privileged));
  _GM_PRINT (1, ("min_dma_mem = 0x%x\n", (int) ps->min_dma_mem));
  _GM_PRINT (1, ("max_dma_mem = 0x%x\n", (int) ps->max_dma_mem));
  _GM_PRINT (1, ("mapping_specs = {...}\n"));
  _GM_PRINT (1, ("sleep_sync = {...}\n"));
  _GM_PRINT (1, ("copy_block = {...}\n"));
  _GM_PRINT (1, ("recv_queue_region = {...}\n"));
  _GM_PRINT (1, ("arch = {...}\n"));
}

/* The Unix Mmap functions here are split into 3 phases for compatibility
   across Unix platforms:

   (1) Verify that the MMAP will succeed (gm_prepare_to_mmap()).
   (2) Map the individual pages (gm_mmap()).
   (3) Update data internal data structures depending on the
   successful mapping.b (gm_finish_mmap()).

   Phase 2 is platform-dependent, so we just provide a function to return
   a pointer to the kernel page to be mapped. */

gm_status_t
gm_prepare_to_mmap (gm_port_state_t * ps, gm_offset_t off, gm_size_t len,
		    unsigned int permissions)
{
  gm_instance_state_t *is;

  GM_CALLED ();

  is = ps->instance;

#if 1
  if (!is)
    gm_print_port_state (ps);
#endif

  gm_assert (is);

  /* Verify that the user has set the port number before trying to mmap. */

  if (!ps->opened)
    {
      GM_NOTE (("User failed to set port number before mmap().\n"));
      GM_RETURN_STATUS (GM_INVALID_PARAMETER);
    }

  GM_PRINT (GM_PRINT_LEVEL >= 2,
	    ("gm_prepare_to_mmap called for instance 0x%x, port 0x%x.\n",
	     is->id, ps->id));
  _GM_PRINT (GM_PRINT_LEVEL >= 2, ("  offset=0x%lx len=0x%lx\n", off, len));

  /* Verify alignments */

  if (!GM_PAGE_ALIGNED (len) || !GM_PAGE_ALIGNED (off))
    {
      GM_NOTE (("alignment problem in gm_prepare_to_mmap\n"));
      _GM_NOTE (("len = 0x%lx, off = 0x%lx\n", len, off));
      GM_RETURN_STATUS (GM_INVALID_PARAMETER);
    }


#if GM_BC_DRIVER
  if (ps->compatibility_mode)
    {
      status = gm_expand_copy_block (ps, ps->max_dma_mem);
      GM_RETURN_STATUS (status);
    }
#endif

  /* Allow individual pages to be mapped */

#define GM_ARCH_MAPS_PAGE_BY_PAGE 1

#if GM_ARCH_MAPS_PAGE_BY_PAGE
#define LE <=
#else
#define LE ==
#endif

#define LEGIT_MAPPING(type)						\
  (ps->mappings.type.offset LE off					\
   && (off+len) LE (ps->mappings.type.offset + ps->mappings.type.len) 	\
   && (ps->mappings.type.permissions & permissions) == permissions)

  if (LEGIT_MAPPING (control_regs)
      || LEGIT_MAPPING (special_regs)
      || LEGIT_MAPPING (sram)
      || LEGIT_MAPPING (recv_queue)
      || LEGIT_MAPPING (send_queue) || LEGIT_MAPPING (RTC))
    {
      GM_PRINT (GM_PRINT_LEVEL >= 6, ("mapping is legit\n"));
      GM_RETURN_STATUS (GM_SUCCESS);
    }

#undef LEGIT_MAPPING
#undef LE

#if !GM_CAN_REGISTER_MEMORY
  /* Allow mappings within the copy block, but only if the copy block
     can be expanded to satisfy the request.

     The Copy block is needed only on machines that cannot perform
     memory registration. */

  if (ps->mappings.copy_block.offset <= off
      && off + len <= (ps->mappings.copy_block.offset
		       + ps->mappings.copy_block.len))
    {
      /* If needed, expand the copy block to satisfy the request. */

      if (off + len > ps->mappings.copy_block.offset + ps->copy_block.len)
	{
	  gm_status_t status;
	  status
	    = gm_expand_copy_block (ps,
				    (gm_u32_t) (off + len -
						ps->mappings.copy_block.
						offset));
	  if (status != GM_SUCCESS)
	    {
	      GM_PRINT (GM_PRINT_LEVEL >= 2,
			("copy_block %d %d %d", off, len,
			 ps->mappings.copy_block.offset));
	      GM_RETURN_STATUS (status);
	    }
	}

      /* Verify that the mapping falls completely within a copy block
         segment */

      {
	struct gm_segment_desc *d;

	/* determine which segment holds the start of the mapping */

	off -= ps->mappings.copy_block.offset;
	d = &ps->copy_block.segment[0];
	while (off >= d->len)
	  {
	    off -= d->len;
	    d++;
	  }

	/* make sure the mapping does not run past the end of the segment */

	if (off + len > d->len)
	  {
	    GM_PRINT (GM_PRINT_LEVEL >= 2,
		      ("seg running copy_block %d %d %d", off, len, d->len));
	    GM_RETURN_STATUS (GM_INVALID_PARAMETER);
	  }
      }

      GM_RETURN_STATUS (GM_SUCCESS);
    }
#endif

  /* Did not recognize the requested mapping, so fail. */

  GM_RETURN_STATUS (GM_INVALID_PARAMETER);
}

gm_status_t
gm_finish_mmap (gm_port_state_t * ps, gm_offset_t off, gm_size_t len,
		gm_up_t uaddr)
{
  gm_instance_state_t *is;
  GM_PARAMETER_MAY_BE_UNUSED (len);
  
  GM_CALLED ();

  GM_PRINT ((GM_PRINT_LEVEL >= 5),
	    ("gm_finish_mmap (0x%p, 0x%lx, 0x%lx, 0x%p) called\n", ps, off,
	     len, (void*) uaddr));
  is = ps->instance;
  gm_assert (is);

#if !GM_CAN_REGISTER_MEMORY
  /* If needed, add the mapped pages to hash table */

  {
    unsigned long start, limit;

    start = GM_MAX (off, ps->mappings.copy_block.offset);
    limit = GM_MIN (off + len,
		    ps->mappings.copy_block.offset + ps->copy_block.len);
    if (limit > start)
      {
	gm_status_t status;

	gm_assert (GM_PAGE_ALIGNED (start));
	gm_assert (GM_PAGE_ALIGNED (limit));
	status = gm_add_segment_to_page_table
	  (ps, uaddr,
	   (gm_u32_t) (start - ps->mappings.copy_block.offset),
	   (gm_u32_t) (limit - start));
	if (status != GM_SUCCESS)
	  GM_RETURN_STATUS (status);
      }
  }

#endif

  /* If the recv queue was mapped, record the host address of each
     slot of the recv queue in the LANai for the MCP, which sometimes
     needs to pass host addresses back to the host in the recv
     queue. */

  if (off == ps->mappings.recv_queue.offset)
    {
      int i;

      gm_assert (is->lanai.globals != 0);

      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("Receive queue mapped to user address %p.\n",
		 (void*) uaddr));

      for (i = 0; i < GM_NUM_RECV_QUEUE_SLOTS; i++)
	{
	  gm_write_lanai_global_up
	    (is, port[ps->id].recv_queue_slot_host_addr[i],
	     (gm_up_t) uaddr + i * sizeof (gm_recv_queue_slot_t));
	}

      GM_STBAR ();

      GM_PRINT (GM_PRINT_LEVEL >= 5, ("*************************\n"));
      _GM_PRINT (GM_PRINT_LEVEL >= 5, ("recv queue DMA addr = %p.\n",
				       gm_arch_dma_region_dma_addr
				       (&ps->recv_queue_region)));
      _GM_PRINT (GM_PRINT_LEVEL >= 5,
		 ("recv queue VM addr = %p.\n", (void*) uaddr));
      _GM_PRINT (GM_PRINT_LEVEL >= 5, ("*************************\n"));
    }

  GM_RETURN_STATUS (GM_SUCCESS);
}

#if !GM_CAN_REGISTER_MEMORY
/* Add the LEN/GM_PAGE_LEN pages at OFFSET in the copy_block for the
   PORT into the page hash table for the device. */
gm_status_t
gm_add_segment_to_page_table (gm_port_state_t * ps, gm_up_t user_addr,
			      gm_u32_t offset, gm_u32_t len)
{
  int i;
  gm_instance_state_t *is;

  GM_CALLED ();

  GM_PRINT (GM_PRINT_LEVEL >= 2,
	    ("gm_add_segment_to_page_table(uaddr=%p off=0x%x len=0x%x) called\n",
	     user_addr, offset, len));

  gm_assert (GM_PAGE_ALIGNED (user_addr));
  gm_assert (GM_PAGE_ALIGNED (offset));
  gm_assert (GM_PAGE_ALIGNED (len));

  is = ps->instance;

  gm_assert (ps);
  gm_assert (is);

  gm_arch_mutex_enter (&is->page_hash.sync);

  /* No need to pause the LANai when ADDING entries to the page table. */

#if GM_BC_DRIVER
  /* For backwards compatibility, reproduce the mlanai driver HACK of placing
     some info in the first page of the copy block.  We also harmlessly place
     it in the first page of every segment. */

  if (ps->compatibility_mode && offset == 0)
    {
      ((int *) ps->copy_block.segment[0].data)[0]
	= (int) (gm_arch_dma_region_dma_addr
		 (&ps->copy_block.segment[0].dma_region));

      ((int *) ps->copy_block.segment[0].data)[1] = ps->max_dma_mem;

      ((int *) ps->copy_block.segment[0].data)[2]
	= ps->copy_block.segment[0].dma_region.sts;
    }
#endif

  for (i = 0; len && i < 32; i++)
    {
      while (len && offset < ps->copy_block.segment[i].len)
	{
	  gm_dp_t dma_addr;

	  GM_PRINT (GM_PRINT_LEVEL >= 5, ("len = 0x%lx.\n", len));

	  dma_addr
	    =
	    gm_arch_dma_region_dma_addr_advance (&ps->copy_block.
						 segment[i].dma_region);

          GM_PRINT ((GM_DEBUG_PAGE_HASH | GM_DEBUG_HOST_DMA),
                    ("gm_add_segment_to_page_table():\n"
                     "    i=%d, len=0x%x, offset=0x%p, dma_addr=0x%p\n",
		     i, len, offset, dma_addr));

          if (!dma_addr)
	    {
	      GM_WARN (("DMA addr is 0; not adding mapping\n"));
	      break;
	    }

	  GM_PRINT (GM_PRINT_LEVEL >= 5,
		    ("calling gm_add_mapping_to_page_table\n"));
	  if (gm_add_mapping_to_page_table (is, user_addr, ps->id, dma_addr)
	      != GM_SUCCESS)
	    {
	      GM_NOTE (("Failed to add a mapping to the page table offset"
			" = 0x%x\n", offset));
	      while (offset)
		{
		  /* No need to pause the LANai when dereferencing mappings.
		     --Glenn */
		  GM_PRINT (GM_PRINT_LEVEL >= 5,
			    ("dereferncing mapping for user addr %p\n",
			     user_addr));
		  gm_dereference_mapping (is, user_addr, ps->id);
		  user_addr -= GM_PAGE_LEN;
		  offset -= GM_PAGE_LEN;
		}
	      goto abort_with_mutex;
	    }

	  user_addr += GM_PAGE_LEN;
	  offset += GM_PAGE_LEN;
	  len -= GM_PAGE_LEN;
	}
      offset -= ps->copy_block.segment[i].len;
    }

  GM_PRINT (GM_PRINT_LEVEL >= 6, ("Freeing mutex.\n"));
  gm_arch_mutex_exit (&is->page_hash.sync);

  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_mutex:
  gm_arch_mutex_exit (&is->page_hash.sync);
  GM_RETURN_STATUS (GM_FAILURE);
}
#endif

/* Return a pointer to the resource in kernel memory that is mapped to
   offset OFF in the device.  The resource can be the LANai control
   registers, special registers, SRAM, or copy block.  The offsets
   and sizes of these various regions can be obtained using the
   GM_GET_MAPPING_SPECS ioctl, which returns a description of the
   mappings in a "gm_mapping_specs" structure.

   The pages of the copy block must be obtained in order and without
   skipping any offsets.

   gm_prepare_to_mmap ensures that all offsets passed to gm_mmap are
   legitimate, so there is no need to range check offsets here.
   However, checks must be performed here for "promiscuous" mappings
   of the device registers an SRAM areas other than the "send queue".

   Returns pointer to resource, or (void *)0 on failure 
   (was returning (-1) before, changed for consistency with _gm_mmap
*/

gm_status_t gm_mmap (gm_port_state_t * ps, gm_offset_t orig_off, void **kptr)
{
  gm_instance_state_t *is;
  gm_offset_t off = orig_off;

  GM_CALLED ();

  *kptr = 0;
  if (!ps->opened)
    {
      GM_NOTE (("User tried to map memory before setting port ID.\n"));
      GM_RETURN_STATUS (GM_FAILURE);
    }

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("Mapping offset 0x%lx\n", orig_off));

  if (!GM_PAGE_ALIGNED (off))
    {
      GM_NOTE (("Could not map non-page-aligned offset 0x%lx.\n", off));
      GM_RETURN_STATUS (GM_FAILURE);
    }

  is = ps->instance;
  gm_assert (is);

#if GM_BC_DRIVER

  /* Special case for /dev/mlanai mappings */

  if (ps->compatibility_mode)
    {
      if (off < GM_BC_PAGE_LEN)
	{
	  *kptr = is->lanai.control_regs + off;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
      off -= GM_BC_PAGE_LEN;

      if (off < GM_BC_PAGE_LEN)
	{
	  *kptr = is->lanai.special_regs + off;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
      off -= GM_BC_PAGE_LEN;

      if (off < ps->mappings.sram.len)
	{
	  *kptr = is->lanai.sram + off;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
      off -= ps->mappings.sram.len;
      if (off < ps->copy_block.segment[0].len)
	{
	  *kptr = ps->copy_block.segment[0].data + off;
	  GM_RETURN_STATUS (GM_SUCCESS);
	}
      GM_WARN (("failed compatibility mode mapping at offset 0x%lx", off));

      GM_RETURN_STATUS (GM_FAILURE);
    }
#endif

#define MAPPING_OF_TYPE(type, off, permissions)				\
  (ps->mappings.type.offset <= off					\
   && off < (ps->mappings.type.offset					\
	     + (gm_offset_t) ps->mappings.type.len))

/*** Is this a control register page? */

  if (MAPPING_OF_TYPE (control_regs, off, permissions))
    {
      off -= ps->mappings.control_regs.offset;
      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("mapping control reg page (off = 0x%lx)\n", off));
      *kptr = (char *) is->lanai.control_regs + off;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

/*** Is this a special register page? */

  if (MAPPING_OF_TYPE (special_regs, off, permissions))
    {
      off -= ps->mappings.special_regs.offset;
      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("mapping special reg page (off = 0x%lx)\n", off));
      *kptr = (char *) is->lanai.special_regs + off;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

/*** Is this an SRAM page? */

  gm_assert (ps->mappings.sram.len == is->lanai.eeprom.lanai_sram_size);
  if (MAPPING_OF_TYPE (sram, off, permissions))
    {
      off -= ps->mappings.sram.offset;
      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("mapping sram page (off = 0x%lx)\n", off));
      *kptr = (char *) is->lanai.sram + off;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

  if (MAPPING_OF_TYPE (send_queue, off, permissions))
    {
      GM_PANIC (("loic: oops: this code looks wrong\n"));
      off -= ps->mappings.send_queue.offset;
      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("mapping send_queue page (off = 0x%lx)\n", off));
      *kptr = (char *) is->lanai.sram + off;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

  if (MAPPING_OF_TYPE (RTC, off, permissions))
    {
      GM_PRINT (glenn, ("User mapped the RTC\n"));
      off -= ps->mappings.RTC.offset;
      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("mapping RTC page (off = 0x%lx)\n", off));
      *kptr = (char *) is->lanai.sram + off;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

/*** Is this a recv queue page? */

  if (MAPPING_OF_TYPE (recv_queue, off, permissions))
    {
      off -= ps->mappings.recv_queue.offset;
      gm_assert (gm_arch_dma_region_kernel_addr (&ps->recv_queue_region));
      GM_PRINT (GM_DEBUG_HOST_DMA,
		("mapping recv queue page (off = 0x%lx)\n", off));
      *kptr =
	(char *) gm_arch_dma_region_kernel_addr (&ps->recv_queue_region) +
	off;
      GM_RETURN_STATUS (GM_SUCCESS);
    }

#if !GM_CAN_REGISTER_MEMORY
/*** Is this a copy block page? */

  if (MAPPING_OF_TYPE (copy_block, off, permissions))
    {
      int i;

      off -= ps->mappings.copy_block.offset;

      gm_assert (off < ps->max_dma_mem);
      gm_assert (off < ps->copy_block.len);

      GM_PRINT (GM_PRINT_LEVEL >= 5,
		("mapping copy block page (off = 0x%lx)\n", off));

      /* Compute I and OFF such that the page at offset OFF in segment I of
         the copy block is the page to be mapped. */

      for (i = 0;
	   off >= ps->copy_block.segment[i].len;
	   off -= ps->copy_block.segment[i++].len)
	{
	  gm_assert (i < 32);
	}
      *kptr = ps->copy_block.segment[i].data + off;
      GM_RETURN_STATUS (GM_SUCCESS);
    }
#endif

/*** The offset is invalid. */
  GM_NOTE (("User attempted invalid mapping at offset 0x%lx\n", off));
  GM_RETURN_STATUS (GM_FAILURE);
}



int
gm_mapping_in_io_space (gm_port_state_t * ps, gm_offset_t off)
{
  return (MAPPING_OF_TYPE (control_regs, off, 0)
	  || MAPPING_OF_TYPE (special_regs, off, 0)
	  || MAPPING_OF_TYPE (sram, off, 0));
}

#undef MAPPING_OF_TYPE



/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
