/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997, 1998, 1999 by Myricom, Inc.                 *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/*
    gb.c
    source file for gm ifnet driver freebsd
    finucane@myri.com

    this code depends on the following #defines being defined or not:

    INSIST
    GM_EMULATE_BYTE_DMAS
*/


#ifndef GM_ENABLE_FREEBSD_CURRENT
#define GM_ENABLE_FREEBSD_CURRENT 0
#endif

#define INSIST 1

#if (GM_DEBUG > 0 ) || defined (INSIST)
#define insist(e) do { if (!(e)) {printf ("assertion failed in file %s, line %d (%s)\n",__FILE__,__LINE__, #e); goto exception;}} while (0)
#else
#define insist(e) do {if (0) goto exception;} while (0);
#endif

#include "gb.h"
#include "gm_debug.h"

void gb_printAlignment (struct mbuf*m)
{
  struct mbuf*t;
  static int lastLength = 0;
  /*
    if (lastLength == m->m_pkthdr.len)
    return;
  */
  lastLength = m->m_pkthdr.len;
  
  printf ("(%d)", m->m_pkthdr.len);
  
  for (t = m; t; t = t->m_next)
  {
    if (t->m_len == 0)
      continue;

    printf ("  %x:%d ", t, t->m_len);
  }

  printf ("\n");
  exception: return;
}
  
void gb_printMbuf (struct mbuf*m)
{
  int i, np, length;
  struct mbuf*t;

  printf ("mbuf length is %d\n", m->m_pkthdr.len);
  
  np = 0;
  
  for (t = m; t; t = t->m_next)
  {
    if (t->m_len == 0)
      continue;

    printf ("piece %d\n", np++);

    for (i = 0; i < t->m_len; i++)
      printf ("%02x%c",0xff & ((char*)(t->m_data))[i], i % 8 == 7 ? '\n' : ' ');
    printf ("\n");
  }
  exception: return;
}


int gb_if_done (struct ifnet*ifp)
{
  return(0);
}

static gb_Device*gb_devices [GB_MAX_DEVICES];

int gb_isEmpty (gb_Queue*q)
{
  insist (q);
  insist (q->size >= 0);
  insist (q->head && q->size || !q->head && !q->size);
  return !q->head;
  exception: return 0;
}

int gb_put (gb_Queue*q, gb_Item*p)
{
  insist (q && p);
  insist (q->size >= 0);
  
  p->next = 0;
  if (!q->tail)
  {
    insist (!q->head && q->size == 0);
    q->tail = q->head = p;
  }
  else
  {
    insist (q->head && q->size);
    q->tail->next = p;
    q->tail = p;
  }
  q->size++;
  return 1;
  
  exception: return 0;
}

gb_Item*gb_get (gb_Queue*q)
{
  gb_Item*p;
  insist (q);
  insist (q->size >= 0);
  
  if (!q->head)
  {
    insist (!q->tail && q->size == 0);
    return 0;
  }
  p = q->head;
  q->head = q->head->next;
  if (!q->head)
  {
    insist (p == q->tail && q->size == 1);
    q->tail = 0;
  }
  q->size--;
  return p;
  exception: return 0;
}


gb_Item*gb_newItem (gb_Device*d)
{
  gb_Item*p = 0;
  insist (d);
  
  p = (gb_Item*) malloc (sizeof (gb_Item), M_DEVBUF, M_NOWAIT);
  insist (p);
  bzero (p, sizeof (gb_Item));
  
  p->device = d;
  p->refs = 1;
  
  return p;
  exception: return 0;
}

void gb_freeItem (gb_Item*p)
{
  insist (p);
  
  free (p, M_DEVBUF);
  exception:;
}

gb_Item*gb_newBufferItem (gb_Device*d, int size)
{
  gb_Item*p;
  int i, j;
  gb_Item**kludge;
  
  insist (d);
  
  p = gb_newItem (d);
  insist (p);
   
  p->osize = size;
  p->ma = (char*) contigmalloc (size + sizeof (char*), M_DEVBUF, M_NOWAIT,
				0x100000, 0xffffffff, PAGE_SIZE, 0);
  insist (p->ma);
  p->va = GB_ROUND_POINTER (p->ma + sizeof (char*));

  /*for gb_reclaimReceiveMbufItem, we store pointer to item right behind buffer pointer*/
  kludge = (gb_Item**) (((char*)p->va) - sizeof (char*));

  insist ((char*)kludge >= p->ma);
  
  *kludge = p;
  
  insist (p->va);
  insist (p == GB_BUFFER_TO_ITEM (p->va));
  
  p->pa = GB_VTOP (p->va);
  insist (GB_ALIGNED (p->pa));
    
  insist (p->pa);
  return p;
  exception: return 0;
}

int gb_dmaUnloadPieces (gb_Item*p)
{
  int i;
  insist (p && p->device);
  insist (p->numPieces >= 0 && p->numPieces <= GM_MAX_ETHERNET_GATHER_CNT);
  p->numPieces = 0;
  return 1;
  exception: return 0;
}

int gb_dmaLoadPieces (gb_Item*p, struct mbuf*m, gm_ethernet_segment_descriptor_t*pieces)
{
  struct mbuf*t;
  
  insist (p && p->device);
  insist (p->numPieces == 0);
  insist (m);

  p->m = m;
  p->numPieces = 0;
  
  for (t = m; t; t = t->m_next)
  {
    if (t->m_len == 0)
      continue;
    
    insist (p->numPieces >= 0 && p->numPieces < GM_MAX_ETHERNET_GATHER_CNT);
    
    pieces [p->numPieces].ptr = gm_hton_dp ((int) GB_VTOP (t->m_data));
    pieces [p->numPieces].len = gm_hton_u32 (t->m_len);
    
    p->numPieces++;
  }
  
  return p->numPieces;
  exception: return 0;
}

void gb_freeBufferItem (gb_Item*p)
{
  int i;

  insist (p);

  insist (p->ma);
  kmem_free (kernel_map, (vm_offset_t) p->ma, PAGE_SIZE);
  gb_freeItem (p);
  exception:;
}

int gb_feedGm (gb_Device*d)
{
  gb_Item*p;
  gm_ethernet_segment_descriptor_t piece;
  
  insist (d && d->opened);
  
  while (d->pendingReceives.size < GM_NUM_ETHERNET_RECV_TOKENS && (p = gb_get (&d->freeReceives)))
  {
    p = gb_newMbufItem (d, p);
    insist (p);

    piece.ptr = gm_hton_dp ((int) p->pa);
    piece.len = gm_hton_u32 (p->osize);

    gb_put (&d->pendingReceives, p);

    insist(d->port);

    _gm_provide_ethernet_scatter_list (d->port, 1, &piece);

  }

  return 1;
  exception: return 0;
}

void gb_reclaimReceiveMbufItem (caddr_t address, u_int length)
{
  gb_Item*p = GB_BUFFER_TO_ITEM (address);
  int s;
  
  if (!p)
    return;

  insist (address);
  insist (p && p->device);
  insist (p->m);

  insist (p->refs >= 0);
  
  if (--p->refs)
    return;

  if (p->refs)
    return;
  
  s = splimp ();
  
  gb_put (&p->device->freeReceives, p);
 
  if (p->device->opened)
    gb_feedGm (p->device);

  splx (s);

  exception:;
}

gb_Item*gb_newMbufItem (gb_Device*d, gb_Item*p)
{
  insist (d && p);
  
  MGETHDR (p->m, M_DONTWAIT, MT_DATA);
  insist (p->m);

  p->refs = 1;
  
  p->m->m_next = 0;
  p->m->m_nextpkt = 0;
  p->m->m_flags |= M_EXT;
  p->m->m_ext.ext_buf = p->m->m_data = p->va;
  p->m->m_ext.ext_size = p->osize;
  p->m->m_ext.ext_free = gb_reclaimReceiveMbufItem;
  p->m->m_ext.ext_ref = gb_ref;

  return p;

  exception: return 0;
}

void gb_freeMbuf (gb_Item*p)
{
  struct mbuf*m;
  insist (p);
  insist (p && p->m);
  
  p->m->m_flags = 0;

  m_free (p->m);

  exception:;
}

void gb_freeMbufItem (gb_Item*p)
{
  insist (p && p->m);
  gb_freeMbuf (p);
  gb_freeItem (p);
  exception:;
}

gb_Device*gb_newDevice ()
{
  gb_Device*d = 0;
  int i;

  d = (gb_Device*) malloc (sizeof (gb_Device), M_DEVBUF, M_NOWAIT);
  bzero (d, sizeof (gb_Device));
  
  insist (d);
 
  d->port = NULL; 

  insist (gb_isEmpty (&d->pendingSends));
  insist (gb_isEmpty (&d->pendingReceives));
  
  strncpy (d->name, GB_NAME, GB_NAME_LEN);  
  
  for (i = 0; i < GB_NUM_SEND_BUFFERS; i++)
    gb_put (&d->freeSends, gb_newBufferItem (d, GB_BUFFER_SIZE));
  
  for (i = 0; i < GB_NUM_RECEIVE_BUFFERS; i++)
    gb_put (&d->freeReceives, gb_newBufferItem (d, GB_BUFFER_SIZE));
  
  insist (d->freeReceives.size == GB_NUM_RECEIVE_BUFFERS);
  insist (d->freeSends.size == GB_NUM_SEND_BUFFERS);
  
  return d;
  exception: return 0;
}

void gb_if_init (void*p)
{
  return;
}

void gb_if_start (struct ifnet*ifp)
{
  struct mbuf*m,*t;
  int s;
  int r;
  gb_Device*d = (gb_Device*) ifp;
  
  insist (ifp);

  s = splimp ();
  
  if (!(m = ifp->if_snd.ifq_head)) 
  {
    splx (s);
    return;
  }
  
  IF_DEQUEUE (&ifp->if_snd, m);

  do
  {
    t = m->m_nextpkt;
    r *= gb_send (d, m);
    m = t;
  } while (m);
  
  splx (s);
  
  exception: return; 
}

void gb_if_watchdog (struct ifnet*ifp)
{
  return;
}

int gb_if_reset (int unit, int dc)
{
  insist (unit >= 0 && unit < GB_MAX_DEVICES);
  insist (gb_devices [unit]);
  
  gb_devices [unit]->arpcom.ac_if.if_flags |= IFF_RUNNING;
  
  return 0;
  exception: return 0;
}

int gb_if_ioctl (struct ifnet*ifp, u_long command, caddr_t data)
{
  int s;
  gb_Device*d = (gb_Device*) ifp;
  struct ifdevea*ifd = (struct ifdevea*) data;
  struct ctrreq*ctr = (struct ctrreq*) data;
  struct ifreq*ifr = (struct ifreq*) data;
  u_short ifmtu;
  int status = 0;
  int do_start = 0;
  
  insist (d);
  
  s = splimp ();
  
  switch (command)
  {
    case SIOCSIFADDR:
    case SIOCGIFADDR:
      status = ether_ioctl(ifp, command, data);
      break;
    case SIOCSIFMTU:
      if (ifr->ifr_mtu > GM_IP_MTU)
	status = EINVAL;
      else
	ifp->if_mtu = ifr->ifr_mtu;
      break;
    case SIOCSIFFLAGS:
      if (ifp->if_flags & IFF_UP)
      {
	if (!d->opened)
	  gb_open (d);
	do_start = 1;
      }
      else
      {
	if (d->opened)
	  gb_close (d);
      }
      status = 0;
      break;
    case SIOCADDMULTI:
    case SIOCDELMULTI:
      status = 0;
      break;
    case SIOCGIFMEDIA:
    case SIOCSIFMEDIA:
      /*      status = ifmedia_ioctl(ifp, ifr, &d->ifmedia, command);*/
      status = 0;
      
      break;
    default:
      status = 0;
  }
 
  splx (s);
  /* we do this here after unlocking to prevent a double-lock in gb_if_start */
  if (do_start) 
  {
    gb_if_start (ifp);
  }
  return status;
  exception:return 0;
}

int gb_detach (int unit)
{
  int s, ifp_valid;
  gb_Device*d;
  struct ifnet*ifp, *g_ifp;
  
  insist (unit >= 0 && unit < GB_MAX_DEVICES);
  d = gb_devices [unit];
  insist (d);
  
  s = splnet();
  g_ifp = gbtoifp (d);
  
  TAILQ_FOREACH(ifp, &ifnet, if_link)
    {
      if (ifp == g_ifp)
      {
	ifp_valid = 1;
	break;
      }
    }
  if (ifp_valid)
  {
    struct ifaddr *ia = 0;
    struct in_ifaddr *in_ia = 0;
    
    TAILQ_FOREACH (ia, &ifp->if_addrhead, ifa_link)
      {
	if (ia->ifa_addr->sa_family != AF_INET)
	  continue;
	in_ia = (struct in_ifaddr *)ia;
	in_ifscrub (g_ifp, in_ia);
	TAILQ_REMOVE (&in_ifaddrhead, in_ia, ia_link);
      }
    if_detach (g_ifp);
  }
  splx(s);
  if (d->opened)
    gb_close (d);

  gb_freeDevice (d);
  gb_devices [unit] = 0;
  
  return 1;
  exception: return 0;
}

int gb_attach (int unit)
{
  int i, r;
  struct ifnet*ifp;
  struct sockaddr_in*sin;
  gb_Device*d;
  
  d  = gb_newDevice ();
  insist (d);
  
  d->unit = unit;
  
#if GM_EMULATE_BYTE_DMAS
  d->forceAlignment = 0;
#else
  d->forceAlignment = 1;
#endif

  r = gb_open (d);
  insist (r);
  
  ifp = &d->arpcom.ac_if;
  
  ifp->if_softc = d;
  ifp->if_unit = d->unit;
  ifp->if_name = d->name;
  ifp->if_mtu = GM_IP_MTU;
  ifp->if_flags =  IFF_BROADCAST | IFF_MULTICAST;
  ifp->if_ioctl = gb_if_ioctl;  
  ifp->if_output = ether_output;
  ifp->if_start = gb_if_start;
  ifp->if_watchdog = gb_if_watchdog;
  ifp->if_init = gb_if_init;
  ifp->if_baudrate = 1000000000;
  ifp->if_snd.ifq_maxlen = 512;
  ifp->if_timer = 0;

  bcopy (d->address, d->arpcom.ac_enaddr, 6);

#if GM_ENABLE_FREEBSD_CURRENT
  ether_ifattach (ifp, ETHER_BPF_UNSUPPORTED);
#else
  if_attach(ifp);
  ether_ifattach (ifp);
#endif

  r = gb_close (d);
  insist (r);
  
  /*ether_ifattach reset mtu*/
  ifp->if_mtu = GM_IP_MTU;

  gb_devices [unit] = d;
  
  ifp->if_flags &= ~IFF_OACTIVE;
  ifp->if_flags |= IFF_RUNNING;

  return 1;
  exception: return 0;
}

int gb_close (gb_Device*d)
{  
  gb_Item*p;
  
  insist (d);
  insist (d->opened);
  insist(d->port);

  gm_ethernet_set_sent_intr_callback (d->port, 0, 0);  
  gm_ethernet_set_recv_intr_callback (d->port, 0, 0);
  
  gm_close (d->port);
  d->port = NULL;
  d->opened = 0;
  
  while (p = gb_get (&d->pendingReceives))
    gb_put (&d->freeReceives, p);
  
  while (p = gb_get (&d->pendingSends))
  {
    if (p->m)
    {
      m_freem (p->m);
      p->m = 0; 
    }
    gb_put (&d->freeSends, p);
  }
  
  ((struct ifnet*) d)->if_flags &= ~IFF_UP;
  
  return 1;
  exception: return 0;
}

int gb_open (gb_Device*d)
{
  int r;
  
  insist (d);
  insist (!d->opened);
  /*insist (d->freeSends.size == GB_NUM_SEND_BUFFERS);*/
  insist (d->freeReceives.size == GB_NUM_RECEIVE_BUFFERS);


  r = gm_open (&d->port, d->unit, GM_ETHERNET_PORT_ID, d->name, GM_API_VERSION_1_0);
  insist (r == GM_SUCCESS);
  insist(d->port);
  
  d->opened = 1;
  
  r = gm_get_unique_board_id (d->port, d->address);
  insist (r == GM_SUCCESS);
  
  gm_ethernet_set_sent_intr_callback (d->port, (void*) gb_sendDone, d);
  gm_ethernet_set_recv_intr_callback (d->port, (void*) gb_receiveDone, d);
  
  gb_feedGm (d);

  ((struct ifnet*) d)->if_flags |= IFF_UP;

  return 1;
  exception: return 0;
}

void gb_freeDevice (gb_Device*d)
{
  gb_Item*p;
  
  insist (gb_isEmpty (&d->pendingSends));
  insist (gb_isEmpty (&d->pendingReceives));

  while (p = gb_get (&d->freeSends))
    gb_freeBufferItem (p);
  
  while (p = gb_get (&d->freeReceives))
    gb_freeBufferItem (p);

  insist (gb_isEmpty (&d->freeSends));
  insist (gb_isEmpty (&d->freeReceives));

  free (d, M_DEVBUF);  
  exception:;
}

int gb_send (gb_Device*d, struct mbuf*m)
{
  struct mbuf *t,*nextPacket;
  gb_Item*p = 0 ;
  char*offset;
  unsigned char*address;
  int numPieces;
  int i, aligned, broadcast;
  gm_ethernet_segment_descriptor_t pieces [GM_MAX_ETHERNET_GATHER_CNT];
  
  insist (d && m);

  /* catch any sends that happen after port close */ 
  if (!d->opened) 
  {
    GM_NOTE (("gb_send: dropping packet, GM port not open\n"));
    d->arpcom.ac_if.if_snd.ifq_drops++;
    m_freem (m);
    return(0);
  }

  if (d->pendingSends.size >= GM_NUM_SEND_TOKENS || !(p = gb_get (&d->freeSends)))
  {
    d->arpcom.ac_if.if_snd.ifq_drops++;
    m_freem (m);
    return 0;
  }
  insist (p);

  /*  gb_printAlignment (m);*/

  numPieces = p->size = 0;
  aligned = 1;
  
  for (t = m; t; t = t->m_next)
  {
    if (t->m_len == 0)
      continue;
    
    insist (t->m_data);
    
    numPieces++;
    p->size += t->m_len;
    
    aligned = aligned && GB_ALIGNED ((unsigned long)t->m_data) && GB_ALIGNED (t->m_len);
  }

  if (p->size > (GM_IP_MTU + GB_BUFFER_SIZE))
  {
    d->arpcom.ac_if.if_oerrors++;
    d->arpcom.ac_if.if_snd.ifq_drops++;
    m_freem (m);
    gb_put (&d->freeSends, p);
    return 0;
  } 

  address = (unsigned char*) m->m_data;  
  broadcast = address [0] & 0x1;

  if (numPieces > GM_MAX_ETHERNET_GATHER_CNT || !aligned && d->forceAlignment)
  {
    p->m = 0;
    
    offset = p->va;
    for (t = m; t; t = t->m_next)
    {
      if (t->m_len == 0)
	continue;      
     
      bcopy (t->m_data, offset, t->m_len);
      offset += t->m_len;
    }
    insist (offset ==  p->va + p->size);

    insist (GB_ALIGNED (p->pa));
    
    pieces [0].ptr = gm_hton_dp ((int) p->pa);
    pieces [0].len = gm_hton_u32 (GB_ROUND_INT (p->size));
    
    numPieces = 1;
    m_freem (m);
  }
  else numPieces = gb_dmaLoadPieces (p, m, pieces);

  insist (numPieces <= GM_MAX_ETHERNET_GATHER_CNT);
 
  gb_put (&d->pendingSends, p);
  insist (d->pendingSends.size <= GM_NUM_SEND_TOKENS);
  
  insist(d->port);

  if (broadcast)
    gm_ethernet_broadcast (d->port, numPieces, pieces, 1);
  else
    gm_ethernet_send (d->port, numPieces, pieces, address, 1);

  return 1;
  exception: return 0;
}

void gb_sendDone (gb_Device*d)
{
  gb_Item*p;
  struct mbuf*m;
  int s;
    
  insist (d);

  /*interrupt already locked by GM, so we don't need splimps and splixes*/
  
  p = gb_get (&d->pendingSends);
  insist (p);
  insist ((p->size > 0) && (p->size <= GB_BUFFER_SIZE));
  
  d->arpcom.ac_if.if_opackets++;
  d->arpcom.ac_if.if_obytes += p->size;
  
  if (p->m)
  {
    gb_dmaUnloadPieces (p);
    m_freem (p->m);
    p->m = 0; 
  }
  
  gb_put (&d->freeSends, p);
  
  exception: return;
}

void gb_receiveDone (gb_Device*d, unsigned length)
{
  gb_Item*p;
  struct ether_header *eh;
  
  insist (d);
 
  insist ((length > 0) && (length <= GB_BUFFER_SIZE));

  /*interrupt already locked by GM, so we don't need splimps and splixes*/

  p = gb_get (&d->pendingReceives);
  insist (p);

  if (d->pendingReceives.size < GB_MIN_RECEIVE_BUFFERS)
  {
    gb_freeMbuf (p);
    gb_put (&d->freeReceives, p);
    gb_feedGm (d);
    d->arpcom.ac_if.if_snd.ifq_drops++;
  }
  else
  {
    insist (p->m->m_ext.ext_buf == p->va);
    insist (p->m);
    insist (p->va == p->m->m_data);
    insist (p->m->m_ext.ext_buf == p->va);
    insist (p->m->m_ext.ext_size == GB_BUFFER_SIZE);
    insist (p->m->m_ext.ext_free == gb_reclaimReceiveMbufItem);
    insist (p->m->m_next == 0);

    eh = (struct ether_header*) (p->va + 2);

    p->m->m_data = p->va + sizeof (struct ether_header) + 2;
    p->m->m_pkthdr.len = p->m->m_len = length - sizeof (struct ether_header) - 2;

    p->m->m_pkthdr.rcvif = &d->arpcom.ac_if;    
    d->arpcom.ac_if.if_ipackets++;
    d->arpcom.ac_if.if_ibytes += length;    
    
    ether_input (&d->arpcom.ac_if, eh, p->m);
  }  
  exception:;
}


void gb_ref (caddr_t buffer, u_int size)
{
  gb_Item*p = GB_BUFFER_TO_ITEM (buffer);
  insist (p);
  insist (p->refs >= 0);
  
  p->refs++;  
  exception:;
}
