/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1999 by Myricom, Inc.                                   *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/*
 * This file includes the OS-specific driver code for FreeBSD
 *
 * Author: Andrew Gallatin <gallatin@freebsd.org>, <gallatin@cs.duke.edu>
 *
 * This driver was written to be a loadable module using the FreeBSD
 * 4.0 "newbus" api.  As such, it will require backporting to work under 
 * older versions of FreeBSD which do not support loadable PCI drivers
 *
 * It has been tested primarily on FreeBSD/alpha
 * (and minimally on FreeBSD/i386)
 */

/*
 * FreeBSD specific device driver defines 
 */


#include "gm_arch.h"

#include <sys/malloc.h>
#include <sys/proc.h>
#include <machine/stdarg.h>
#include <machine/limits.h>
#include <vm/vm_pageout.h>		/* for vm_page_max_wired */
#include <sys/lock.h>
#include <vm/vm_map.h>			/* for vm_map type */
#include <sys/proc.h>			/* for process struct */
#include "gm_call_trace.h"
#include "gm_debug.h"


extern int gb_attach(int unit);
extern int gb_detach(int unit);

/*
 * The GM includes
 */

#include "gm_page_hash.h"
#include "gm_lanai.h"
#include "gm_instance.h"
#include "gm_malloc_debug.h"
#include "gm_debug_lanai_dma.h"
#include "gm_debug_malloc.h"


#define UNIMPL printf("unimplemented function called at %s:%d\n", __FILE__,__LINE__)



/***********************************************************************
 * Globals
 ***********************************************************************/

/*
 * GM devices
 */

/* un-staticized for debugging.. */
/*static */ gm_instance_state_t *dev_head = 0;
/*static */ gm_instance_state_t *gm_instances[GM_ARCH_MAX_INSTANCE];
/*static */ int gm_num_instance = 0;
/*static */ int gm_max_user_locked_pages;
/*static */ int gm_skip_init;


#if GM_DEBUG
int gm_print_level = GM_PRINT_LEVEL;
static gm_instance_state_t *debug_is;
#endif



#define MAX_NMYRI 4
#define MAX_MINOR_SIZE 64

int num_gm = 0;

#define GET_MINOR(minor_id) (minor_id - (MAX_NMYRI*2 - num_gm*2))
#define EXTERN_MINOR(minor_id) (minor_id + (MAX_NMYRI*2 - num_gm*2))
#define INVALID_MINOR(minor_id) ((minor_id +(MAX_NMYRI*2 - num_gm*2)) >= MAX_MINOR_SIZE)


/*
 * FreeBSD Device Driver routines.
 * This section is coded to the FreeBSD 4.x "newbus" API and will
 * *not* work on earlier version of FreeBSD.  FreeBSD 4.x was chosen
 * because it allows loadable PCI device drivers.
 */

static int gm_probe(device_t dev);
static int gm_attach(device_t dev);
static int gm_detach(device_t dev);
static int gm_shutdown(device_t dev);
static void gm_freebsd_intr(void *arg);

static device_method_t gm_methods[] =
{
		/* Device interface */
	DEVMETHOD(device_probe, gm_probe),
	DEVMETHOD(device_attach, gm_attach),
	DEVMETHOD(device_detach, gm_detach),
	DEVMETHOD(device_shutdown, gm_shutdown),

	{0, 0}
};

static driver_t gm_driver =
{
	"gm",
	gm_methods,
	sizeof(gm_instance_state_t),
};

static devclass_t gm_devclass;

DRIVER_MODULE(gm, pci, gm_driver, gm_devclass, 0, 0);

static d_open_t gm_freebsd_open;
static d_close_t gm_freebsd_close;
static d_ioctl_t gm_freebsd_ioctl;
static d_mmap_t gm_freebsd_mmap;

#define CDEV_MAJOR 201
static struct cdevsw gm_cdevsw =
{
		/* open */ gm_freebsd_open,
		/* close */ gm_freebsd_close,
		/* read */ noread,
		/* write */ nowrite,
		/* ioctl */ gm_freebsd_ioctl,
		/* poll */ nopoll,
		/* mmap */ gm_freebsd_mmap,
		/* strategy */ nostrategy,
		/* name */ "gm",
		/* maj */ CDEV_MAJOR,
		/* dump */ nodump,
		/* psize */ nopsize,
		/* flags */ 0,
		/* bmaj */ -1
};


static void
gm_stop_dma(device_t dev)
{
	/*
	 * disable bus mastering.
	 */
	u_long val;
	val = pci_read_config(dev, PCIR_COMMAND, 2);
	val &= ~(PCIM_CMD_MEMEN | PCIM_CMD_BUSMASTEREN);
	pci_write_config(dev, PCIR_COMMAND, val, 2);
}


/*
 * the boot-time or load-time probe routine.
 */
/*
 * Return identification string if this is device is ours.
 */
static int
gm_probe(device_t dev)
{
	static int once = 0;
	extern struct cdevsw *cdevsw[];

	if (!once++)
		if (!cdevsw[CDEV_MAJOR])
			cdevsw_add(&gm_cdevsw);

	if (((pci_get_vendor(dev) == GM_PCI_VENDOR_MYRICOM) &&
			(pci_get_device(dev) == GM_PCI_DEVICE_MYRINET)) ||
	    ((pci_get_vendor(dev) == GM_PCI_VENDOR_MYRICOM2) &&
			(pci_get_device(dev) == GM_PCI_DEVICE_MYRINET))) {
		device_set_desc(dev, "Myrinet PCI interface");
		return 0;
	}

	return ENXIO;
}

gm_status_t gm_arch_page_len(unsigned long *result);

static int
gm_attach(device_t dev)
{
	int error = ENXIO;
	gm_instance_state_t *is = device_get_softc(dev);
	struct ifnet *ifp;
	int s;
	u_long val;
	int rid;
	s = splimp();

	is->arch.dev = dev;

	/*
	 * disable DMA
	 */

	gm_stop_dma(dev);

	/*
	 * Map control/status registers.
	 */

#define GM_PCI_SHMEM 0x10
	rid = GM_PCI_SHMEM;			/* shared memory base address */
	is->arch.mem = bus_alloc_resource(dev, SYS_RES_MEMORY, &rid,
									  0, ~0, 1, RF_ACTIVE);
	if (!is->arch.mem) {
		device_printf(dev, "could not map memory\n");
		error = ENXIO;
		goto fail_with_nothing;
	}

	/* 
	   *  map the device into the kernel virtual address space.  FreeBSD
	   *  would rather we used bus_space access, but the MI portions of GM
	   *  don't support that. 
	 */
	is->arch.csr = rman_get_virtual(is->arch.mem);

#ifdef __alpha__
	is->arch.csr = (caddr_t) pci_cvt_to_dense((vm_offset_t) is->arch.csr);
	is->arch.csr = (caddr_t) ALPHA_PHYS_TO_K0SEG((vm_offset_t) is->arch.csr);
#endif
	printf("GM: board mapped at KVA: %p\n", is->arch.csr);


	/*
	 * Allocate our interrupt.
	 */
	rid = 0;
	is->arch.irq = bus_alloc_resource(dev, SYS_RES_IRQ, &rid, 0, ~0, 1,
									  RF_SHAREABLE | RF_ACTIVE);
	if (is->arch.irq == NULL) {
		device_printf(dev, "could not map interrupt\n");
		goto fail_with_mem;
	}

	error = bus_setup_intr(dev, is->arch.irq, INTR_TYPE_NET,
						   gm_freebsd_intr, is, &is->arch.ih);
	if (error) {
		device_printf(dev, "could not setup irq\n");
		goto fail_with_intr;
	}



	/*
	 *  Some GM setup
	 */

	(void) gm_arch_page_len(&GM_PAGE_LEN);	/* can't fail, really.. */


	/*  
	 *  this is an arbitrary limit to prevent the user from
	 *  bringing the system to a complete stop 
	 */
 
	gm_max_user_locked_pages = cnt.v_free_target * 2;


	if (gm_instance_init(is, gm_num_instance, GM_MYRINET_BUS_PCI) != GM_SUCCESS) {
		GM_NOTE (("gm_instance_init failed\n"));
		goto fail_with_irq;
	}
	/*
	 * enable interrupts
	 */
	if (gm_enable_interrupts(is) != GM_SUCCESS) {
		GM_NOTE (("Cannot enable interrupts.\n"));
		goto fail_with_is;
	}
	gm_num_instance++;
	num_gm++;
	gm_instances[device_get_unit(dev)] = is;
	/*
	 * Enable bus mastering.
	 */
	val = pci_read_config(dev, PCIR_COMMAND, 2);
	val |= (PCIM_CMD_MEMEN | PCIM_CMD_BUSMASTEREN);
	pci_write_config(dev, PCIR_COMMAND, val, 2);

	/*
	   *  success!!
	 */


  /************
   * Create minor devices.  /dev/gm%d is the clone device for normal users,
   * and /dev/gmp%d is the clone device for privileged users.
   ************/

	/* Determine minor node number to use for /dev/gm? . */

	GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating minor number.\n"));
	if (gm_minor_alloc(is, &is->clone_minor) != GM_SUCCESS) {
		GM_NOTE (("Failed to allocate minor number.\n"));
		goto fail_with_is;
	}
	GM_PRINT (GM_PRINT_LEVEL >= 1, ("using minor number %ld.\n", (long) is->clone_minor));
	gm_minor_get_port_state(is->clone_minor)->instance = is;
	gm_minor_get_port_state(is->clone_minor)->privileged = 0;

	/* Determine minor node number to use for /dev/gmp? . */

	GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating minor number.\n"));
	if (gm_minor_alloc(is, &is->privileged_clone_minor) != GM_SUCCESS) {
		GM_NOTE (("Failed to allocate minor number.\n"));
		goto fail_with_minor;
	}
	GM_PRINT (GM_PRINT_LEVEL >= 1, ("using minor number %ld.\n",
				 (long) is->privileged_clone_minor));
	gm_minor_get_port_state(is->privileged_clone_minor)->instance = is;
	gm_minor_get_port_state(is->privileged_clone_minor)->privileged = 1;


	device_printf(dev, "GM interface attached\n");
	splx(s);

	GM_INFO (("Attaching the IP interface\n"));

	if (!gb_attach(device_get_unit(dev))) {
		GM_NOTE (("IP interface attach FAILED\n"));
	}
	else {
		GM_INFO (("IP interface attach ok\n"));
	}

	return 0;


fail_with_minor:
	gm_minor_free(is->clone_minor);
  fail_with_is:
	gm_instance_finalize(is);
#ifdef  notyet
	gm_destroy_hash(is->arch.page_lock_hash);
#endif

  fail_with_irq:
	bus_release_resource(dev, SYS_RES_IRQ, 0, is->arch.irq);
  fail_with_intr:
	bus_teardown_intr(dev, is->arch.irq, is->arch.ih);
  fail_with_mem:
	bus_release_resource(dev, SYS_RES_MEMORY, GM_PCI_SHMEM, is->arch.mem);
  fail_with_nothing:
	gm_stop_dma(dev);
	splx(s);
	return error;
}


static int
gm_detach(device_t dev)
{
	gm_instance_state_t *is = device_get_softc(dev);
	int s;
	extern struct cdevsw *cdevsw[];
	int dev_index;

	s = splimp();

    dev_index = device_get_unit(dev);

	GM_INFO (("unloading GM driver dev_index = %d\n",dev_index));

	/* IP stuff */
	gb_detach(dev_index);

	/*
	 * Deallocate resources.
	 */
	bus_teardown_intr(dev, is->arch.irq, is->arch.ih);
	bus_release_resource(dev, SYS_RES_IRQ, 0, is->arch.irq);
	bus_release_resource(dev, SYS_RES_MEMORY, GM_PCI_SHMEM, is->arch.mem);
	cdevsw[CDEV_MAJOR] = 0;


	if (is->privileged_clone_minor) {
    	gm_minor_free (is->privileged_clone_minor);
		is->privileged_clone_minor = NULL;
	}

    if (is->clone_minor) {
		gm_minor_free (is->clone_minor);
		is->clone_minor = NULL;
	}

	GM_WARN (("Driver UNLOAD is buggy and is disabled - reboot your machine.\n"));
#ifdef DELETE
	gm_disable_interrupts(is);
	gm_instance_finalize(is);
	num_gm--;

	/*
	 * Stop DMA 
	 */
	gm_stop_dma(dev);
#endif DELETE

	splx(s);

	if ((dev_index >=0) && (dev_index < GM_ARCH_MAX_INSTANCE)) {
		gm_instances[dev_index] = NULL;
	}

	GM_INFO (("unloaded GM driver\n"));

	return 0;
}

/*
 * Device shutdown routine. Called at system shutdown after sync. The
 * main purpose of this routine is to shut off receiver DMA so that
 * kernel memory doesn't get clobbered during warmboot.
 */
static int
gm_shutdown(device_t dev)
{
	/*
	 * Make sure that DMA is disabled prior to reboot. Not doing
	 * do could allow DMA to corrupt kernel memory during the
	 * reboot before the driver initializes.
	 */
	gm_stop_dma(dev);
	return 0;
}


/*
 * Misc required functions
 */

/* 
 *  This is called just after the port state is created (in gm_minor.c)
 *  to perform architecture-specific initialization. 
 */

gm_status_t
gm_arch_port_state_init(gm_port_state_t * ps)
{
	return GM_SUCCESS;
}

/* 
 *  This is called just before the port state is destroyed (in
 *  gm_minor.c) to perform architecture-specific finalization. 
 */

void
gm_arch_port_state_fini(gm_port_state_t * ps)
{
	return;
}

/* 
 * This is called at the end of gm_port_state_open() to perform architecture-
 * specific initialization. 
 */

gm_status_t
gm_arch_port_state_open(gm_port_state_t * ps)
{
	return GM_SUCCESS;
}

/* 
 * This is called at the start of gm_port_state_close to perform
 *  architecture-specific finalization. 
 */

void
gm_arch_port_state_close(gm_port_state_t * ps)
{
	return;
}


/*
 * Utility functions for gm_ioctl( ) 
 */

gm_status_t
gm_arch_copyin(gm_port_state_t * ps, /* not used */
			   void *what,		/* copy from this user address  */
			   void *where,		/* to this kernel address       */
			   gm_size_t amount) /* for this length              */
{				
	int errno;

	GM_PRINT (GM_PRINT_LEVEL >= 9, ("copyin: from %p to %p len %ld\n", what, where, amount));
	errno = copyin(what, where, amount);
	return errno ? GM_FAILURE : GM_SUCCESS;
}

gm_status_t
gm_arch_copyout(gm_port_state_t * ps, /* not used */
		void *what,	/* copy from this kernel address */
		void *where,	/* to this user address          */
		gm_size_t amount) /* for this length               */
{					
	int errno;

	GM_PRINT (GM_PRINT_LEVEL >= 9, ("copyout: from %p to %p len %ld\n", what, where, amount));
/*
	if (amount > (unsigned int) (1 << 12)) {
		GM_PRINT (GM_PRINT_LEVEL >= 0, ("copyout - reduce the length from %d to %d\n",amount,(1<<12)));
		amount = (1 << 12);
	}
*/
	errno = copyout(what, where, amount);
	return errno ? GM_FAILURE : GM_SUCCESS;
}

/*
 *  Synchronization functions
 */

void
gm_arch_sync_init(gm_arch_sync_t * s, gm_instance_state_t * is)
{
	s->held = s->waiters = 0;
}

void
gm_arch_sync_reset(gm_arch_sync_t * s)
{
	s->held = s->waiters = 0;
}

void
gm_arch_sync_destroy(gm_arch_sync_t * s)
{
}


static void
lock_acquire(gm_arch_sync_t * lock)
{
	int s = splimp();

#if !defined(MAX_PERF)
	if (curproc && lock->held == curproc->p_pid)
		panic("locking against gm_self");
#endif

	while (lock->held) {
		lock->waiters++;
		tsleep(lock, PWAIT, "gm_arch_sync_t", 0);
		lock->waiters--;
	}
	lock->held = (curproc ? curproc->p_pid : -1);
	splx(s);
}

static void
lock_release(gm_arch_sync_t * lock)
{
	int s = splimp();
#if !defined(MAX_PERF)
	if (lock->held == 0)
		panic("releasing a free lock");
	if (curproc && lock->held != curproc->p_pid)
		panic("I wasn't the lockholder");
#endif
	if (lock->waiters)
		wakeup(lock);
	lock->held = 0;
	splx(s);
}

void
gm_arch_mutex_enter(gm_arch_sync_t * s)
{
	lock_acquire(s);
}

void
gm_arch_mutex_exit(gm_arch_sync_t * s)
{
	lock_release(s);
}

void
gm_arch_wake(gm_arch_sync_t * s)
{
	int ipl;
	GM_PRINT (GM_PRINT_LEVEL >= 9, ("gm_arch_wake() on chan. 0x%lx.\n", (vm_offset_t) s));
	ipl = splimp();
	wakeup(s);
	s->wake_cnt++;
	splx(ipl);
}


gm_arch_sleep_status_t
gm_arch_timed_sleep(gm_arch_sync_t * s, int seconds)
{
	int ipl;
	ipl = splimp();
/*  GM_NOTE (("gm_arch_timed_sleep() on chan. 0x%lx\n", (vm_offset_t)s)); */

	while (s->wake_cnt <= 0) {
		if (tsleep(s, PZERO, "gm_arch_timed_sleep", hz * seconds)) {
/*          GM_NOTE (("gm_arch_timed_sleep returning GM_SLEEP_TIMED_OUT\n")); */
			splx(ipl);
			return GM_SLEEP_TIMED_OUT;
		}
	}
	s->wake_cnt--;
	splx(ipl);
/*  GM_NOTE (("gm_arch_timed_sleep() returning.\n")); */
	return GM_SLEEP_WOKE;
}

gm_arch_sleep_status_t
gm_arch_signal_sleep(gm_arch_sync_t * s)
{
	int ipl;
	ipl = splimp();
/*  GM_NOTE (("gm_arch_signal_sleep() on chan. 0x%lx\n", (vm_offset_t)s)); */

	while (s->wake_cnt <= 0) {
		if (tsleep(s, PZERO | PCATCH, "gm_arch_signal_sleep", 0)) {
/*          GM_NOTE (("gm_arch_signal_sleep returning GM_SLEEP_INTERRUPTED\n")); */
			splx(ipl);
			return GM_SLEEP_INTERRUPTED;
		}
	}
	s->wake_cnt--;
	splx(ipl);
/*  GM_NOTE (("gm_arch_signal_sleep() returning.\n")); */
	return GM_SLEEP_WOKE;
}


void
gm_arch_spin(gm_instance_state_t * is, gm_u32_t usecs)
{
	DELAY(usecs);
}


/*
 * DMA region functions 
 */


static gm_dp_t
region_dma_addr(gm_arch_dma_region_t * r, void *addr)
{

#ifdef __alpha__
	return (gm_dp_t) alpha_XXX_dmamap((vm_offset_t) addr);
#endif

#ifdef __i386__
	return (gm_dp_t) vtophys(addr);
#endif

}


gm_status_t gm_arch_dma_region_alloc (gm_instance_state_t * is,
				      gm_arch_dma_region_t * r,
				      gm_size_t len, gm_u32_t flags,
				      gm_register_page_function_t reg,
				      void *user_arg);

gm_status_t
gm_arch_dma_region_alloc(gm_instance_state_t * is,
						 gm_arch_dma_region_t * r,
						 gm_size_t len, gm_u32_t flags,
						 gm_register_page_function_t register_page_func,
						 void *user_arg)
{
	void *addr;
	int pages, page_num;
	GM_PRINT (GM_PRINT_LEVEL >= 1, ("in gm_arch_dma_region_alloc; len = %d,%s\n", len,
				 ((flags & GM_ARCH_DMA_CONTIGUOUS)
				  ? "contiguous"
				  : "non-contiguous")));

	pages = round_page(len) / GM_PAGE_LEN;
	gm_always_assert(pages >= 1);

	/* 
	   * grab the memory and remember how we allocated it 
	 */
	r->is = is;
	r->flags = flags;
	if (len < GM_PAGE_LEN)
		GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_arch_dma_region_alloc: length(%d) < GM_PAGE_LEN\n", len));

	r->len = round_page(len);
	if (flags & GM_ARCH_DMA_CONTIGUOUS) {
		/*
		 * use contigmalloc to get contig memory in DMA space
		 */
		r->type = GM_ARCH_CONTIGMALLOC_REGION;
		r->alloc_addr =
			(void *) vm_page_alloc_contig(r->len, 0x100000,
										  0xffffffff, PAGE_SIZE);
		r->addr = r->alloc_addr;
	}
	else {
		/*
		 * use normal malloc
		 */
		r->type = GM_ARCH_MALLOC_REGION;
		r->alloc_addr = malloc(r->len + GM_PAGE_LEN, M_DEVBUF, M_NOWAIT);
		r->addr = (void *) round_page((vm_offset_t) r->alloc_addr);
	}

	r->addr2 = r->addr;
	if (register_page_func) {
		page_num = 0;
		addr = r->addr;
		while (page_num < pages) {
			gm_dp_t bus_addr = region_dma_addr(r, addr);
			register_page_func(user_arg, bus_addr, page_num);
			page_num++;
			addr += PAGE_SIZE;
		}
	}

	return (GM_SUCCESS);
}

void
gm_arch_dma_region_free(gm_arch_dma_region_t * r)
{
	switch (r->type) {
	 case GM_ARCH_MALLOC_REGION:
		 free(r->alloc_addr, M_DEVBUF);
		 break;
	 case GM_ARCH_CONTIGMALLOC_REGION:
		 kmem_free(kernel_map, (vm_offset_t) r->alloc_addr, r->len);
		 break;
	 default:
		 gm_always_assert(0);
		 break;
	}
}

void *
gm_arch_dma_region_kernel_addr(gm_arch_dma_region_t * r)
{
	return r->addr;
}

gm_s32_t
gm_arch_dma_region_status(gm_arch_dma_region_t * r)
{
	return 0xf;					/* FreeBSD doesn't run on SBUS equipped hosts */
}

gm_dp_t
gm_arch_dma_region_dma_addr(gm_arch_dma_region_t * r)
{
	gm_dp_t ret;
	ret = region_dma_addr(r, r->addr);
	GM_PRINT (GM_PRINT_LEVEL >= 8, ("gm_arch_dma_region_dma_addr:0x%p->0x%x\n", r->addr, ret));
	return ret;
}

gm_dp_t
gm_arch_dma_region_dma_addr_advance(gm_arch_dma_region_t * r)
{
	void *previous = r->addr2;
	r->addr2 += PAGE_SIZE;
	return region_dma_addr(r, previous);
}

gm_status_t
gm_arch_dma_region_sync(gm_arch_dma_region_t * r, int command)
{
	/* this is a no-op on x86 and alpha */
	return GM_SUCCESS;
}

void *
gm_arch_kernel_malloc(unsigned long len, int flags)
{
	return malloc((unsigned long) len, M_DEVBUF, M_NOWAIT);
}

void
gm_arch_kernel_free(void *ptr)
{
	free(ptr, M_DEVBUF);
}

gm_status_t
gm_arch_map_io_space(gm_instance_state_t * is, gm_u32_t offset,
					 gm_u32_t len, void **kaddr)
{
	GM_PRINT (GM_PRINT_LEVEL >= 3, ("gm_arch_map_io_space: offset=0x%x\n", offset));
	*kaddr = (void *) ((vm_offset_t) is->arch.csr + (vm_offset_t) offset);
	return GM_SUCCESS;
}

void
gm_arch_unmap_io_space(gm_instance_state_t * is, gm_u32_t offset,
					   gm_u32_t len, void **kaddr)
{
	/* whatever .. */
}

gm_status_t
gm_arch_mmap_contiguous_segment(gm_port_state_t * ps, void *kaddr,
								unsigned long blockSize, gm_up_t *vaddr)
{
	/* implement me */
	UNIMPL;
	return GM_FAILURE;
}

void
gm_arch_munmap_contiguous_segments(gm_port_state_t * ps)
{
	/* implement me */
	UNIMPL;
}

/*
 * Miscellaneous functions
 */

gm_status_t
gm_arch_page_len(unsigned long *result)
{
	*result = cnt.v_page_size;
	return GM_SUCCESS;
}

gm_status_t
gm_arch_physical_pages(unsigned long *result)
{
	*result = cnt.v_page_count;
	return GM_SUCCESS;
}

gm_status_t
gm_arch_gethostname(char *ptr, int len)
{
	bzero(ptr, len);
	strncpy(ptr, hostname, len - 1);
    ptr[len-1]=0;
	return GM_SUCCESS;
}


/*
 * PCI config space functions
 */

gm_status_t
gm_arch_read_pci_config_32(gm_instance_state_t * is,
			   gm_offset_t offset,
			   gm_u32_t * value)
{
	*value = pci_read_config(is->arch.dev, offset, 4);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_read_pci_config_16(gm_instance_state_t * is,
			   gm_offset_t offset,
			   gm_u16_t * value)
{
	*value = pci_read_config(is->arch.dev, offset, 2);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_read_pci_config_8(gm_instance_state_t * is,
			  gm_offset_t offset,
			  gm_u8_t * value)
{
	*value = pci_read_config(is->arch.dev, offset, 1);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_32(gm_instance_state_t * is,
			    gm_offset_t offset,
			    gm_u32_t value)
{
	pci_write_config(is->arch.dev, offset, value, 4);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_16(gm_instance_state_t * is,
			    gm_offset_t offset,
			    gm_u16_t value)
{
	pci_write_config(is->arch.dev, offset, value, 2);
	return GM_SUCCESS;
}

gm_status_t
gm_arch_write_pci_config_8(gm_instance_state_t * is,
			   gm_offset_t offset,
			   gm_u8_t value)
{
	pci_write_config(is->arch.dev, offset, value, 1);
	return GM_SUCCESS;
}



/* 
 *  open, close, mmap, etc
 */

int
gm_freebsd_open(dev_t dev, int oflags, int devtype, struct proc *p)
{
	gm_status_t status;
	gm_port_state_t *ps;
	gm_instance_state_t *is;
	int unit;
	u_int minor_id;

	GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_freebsd_open: open called gm dev 0x%x minor=%d\n", dev, minor(dev)));

	minor_id = minor(dev);
	if (minor_id >= (MAX_NMYRI * 2)) {
		GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_freebsd_open: open for 'dynamic' device\n"));
	}
	else {
		if (minor_id < 0) {
			GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_freebsd_open: open for bad minor_id=%d\n",minor_id));
			return ENODEV;
		}

		unit = minor_id / 2;
		is = gm_instances[unit];

		if (!is) {
			GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_freebsd_open: open for bad instance=%d\n",unit));
			return ENODEV;
		}
	}

	/* success, so commit */

	GM_PRINT (GM_PRINT_LEVEL >= 4, ("gm_freebsd_open: User opened dev 0x%x with minor num %d.\n",
				 dev, minor(dev)));
	GM_RETURN_INT (0);
}




int
gm_freebsd_close(dev_t dev, int fflag, int devtype, struct proc *p)
{
	gm_port_state_t *ps;
	gm_instance_state_t *is;
	int unit;

	GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_freebsd_close called  dev = 0x%x  minor = %d\n",
				 dev, minor(dev)));

	if (minor(dev) >= (MAX_NMYRI * 2)) {

		GM_PRINT (GM_PRINT_LEVEL >= 1,("closing 'dynamic' minor = %d\n",GET_MINOR(minor(dev)) ));

		ps = gm_minor_get_port_state(GET_MINOR(minor(dev)));
		if (!ps) {
			GM_NOTE ((__GM_FUNCTION__
					  ": could not get port_state for minor = %d\n",
					  GET_MINOR(minor(dev))));
			return ENODEV;
		}

		if (ps->opened) {
			gm_port_state_close(ps);
		}
		gm_minor_free(GET_MINOR(minor(dev)));
	}
	else {
		unit = minor(dev) / 2;

		is = gm_instances[unit];

		if (!is) {
			GM_WARN (("gm_freebsd_close: instance[%d] not found\n", unit));
			return ENODEV;
		}
	}

	GM_RETURN_INT (0);
}

static int
gm_freebsd_localize_status(gm_status_t status)
{
#define CASE(from,to) case from : return to
	switch (status) {
		 CASE(GM_SUCCESS, 0);
		 CASE(GM_INPUT_BUFFER_TOO_SMALL, EFAULT);
		 CASE(GM_OUTPUT_BUFFER_TOO_SMALL, EFAULT);
		 CASE(GM_TRY_AGAIN, EAGAIN);
		 CASE(GM_BUSY, EBUSY);
		 CASE(GM_MEMORY_FAULT, EFAULT);
		 CASE(GM_INTERRUPTED, EINTR);
		 CASE(GM_INVALID_PARAMETER, EINVAL);
		 CASE(GM_OUT_OF_MEMORY, ENOMEM);
		 CASE(GM_INVALID_COMMAND, EINVAL);
		 CASE(GM_PERMISSION_DENIED, EPERM);
	 default:
		 CASE(GM_INTERNAL_ERROR, EPFNOSUPPORT);
		 CASE(GM_UNATTACHED, -1);	/* ??? */
		 CASE(GM_UNSUPPORTED_DEVICE, ENXIO);
	}
}

/*
 * dumb-down the BSD ioctl method (where copyins are done by the
 * generic specfs ioctl hander) & have the GM ioctl function call
 * copyin/copyout as if we were a sysv style system 
 */


int
gm_freebsd_ioctl(dev_t dev, u_long cmd, caddr_t data,
				 int fflag, struct proc *p)
{
	gm_port_state_t *ps;
	gm_status_t status = GM_SUCCESS;
	vm_offset_t user_addr;
	int error;
	unsigned copied_out;


	GM_PRINT (GM_PRINT_LEVEL >= 5, ("ioctl: gm dev=0x%x minor=%d ptr = %p %s\n",
				 dev, minor(dev), data, _gm_ioctl_cmd_name(cmd)));

#warning need to return an error code if fail

	if (cmd == GM_GET_DEV) {
		gm_instance_state_t *is;
		gm_port_state_t *clone_ps;
		gm_status_t status = 0;
		u_int minor_id = minor(dev);

		GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_freebsd_ioctl: command is GM_GET_DEV\n"));

		if (minor_id >= (MAX_NMYRI * 2)) {
			GM_WARN (("gm_freebsd_ioctl() WRONG %d\n", minor_id));
			return ENODEV;
		}

		is = gm_instances[(u_int) minor_id / 2];

		if (!is) {
			return ENODEV;
		}

		clone_ps = gm_minor_get_port_state(minor(dev));

		if (!clone_ps) {
			GM_NOTE (("gm_freebsd_ioctl: gm_minor_get_port_state(%d) failed\n",
					  minor(dev)));
			return ENODEV;
		}

		GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_freebsd_ioctl: Got clone_ps 0x%p %d.\n", clone_ps,minor(dev)));

		/* Determine an unused minor number for the opened device. */

		if (gm_minor_alloc(clone_ps->instance, &minor_id) != GM_SUCCESS) {
			GM_NOTE (("gm_freebsd_ioctl: Could not create new minor number.\n"));
			error = EINVAL;
			goto abort_with_nothing;
		}

		if (INVALID_MINOR(minor_id)) {
			gm_minor_free(minor_id);
			goto abort_with_nothing;
		}

		GM_PRINT (GM_PRINT_LEVEL >= 1, ("gm_freebsd_ioctl: Allocated a new minor number %d.\n", minor_id));
		ps = gm_minor_get_port_state(minor_id);
		if (!ps) {
			GM_NOTE ((__GM_FUNCTION__
					  ": gm_minor_get_port_state(%d) for new device failed\n",
					  minor_id));
			return ENODEV;
		}

		GM_PRINT (GM_PRINT_LEVEL >= 1,("got new port state = %p for minor=%d\n",ps,minor_id));

		/* Record the instance associated with this minor number. */

		ps->instance = clone_ps->instance;
		gm_assert(ps->instance);

		/* Record the privileges from the clone device */

		ps->privileged = clone_ps->privileged;

		GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_freebsd_ioctl: Copyout minor=%d to arg = 0x%lx\n",
					 EXTERN_MINOR(minor_id), data));
		{
			int extern_minor = EXTERN_MINOR(minor_id);
			user_addr = *(vm_offset_t *) data;
			gm_arch_copyout(clone_ps, &extern_minor, (void *)user_addr, sizeof(int));
			GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_freebsd_ioctl: new extern minor=%d\n", extern_minor));
		}
		return 0;
	}
	else {
		u_int minor_id = GET_MINOR(minor(dev));
		GM_PRINT (GM_PRINT_LEVEL >= 5,("ioctl using new minor = %d\n",minor_id));

		ps = gm_minor_get_port_state(minor_id);
		gm_assert(ps);
		if (!ps) {
			return ENXIO;
		}
		gm_arch_mutex_enter(&ps->sync);
		user_addr = *(vm_offset_t *) data;
		GM_PRINT (GM_PRINT_LEVEL >= 5, ("ioctl: minor=%d  ps=0x%p user_addr=0x%lx\n", 
			minor_id,ps,user_addr));
		status = gm_ioctl(ps, cmd, (void *) user_addr, INT_MAX,
						  (void *) user_addr, INT_MAX, 0);
		gm_arch_mutex_exit(&ps->sync);

		GM_PRINT (GM_PRINT_LEVEL >= 5, ("ioctl returns %d\n",
					 gm_freebsd_localize_status(status)));
		return gm_freebsd_localize_status(status);
	}

  abort_with_nothing:
	return (error);
}



/* Map interface memory into the process.  This can be used to map the
   LANai control register, special registers, SRAM, or copy block into
   the user space.  The offsets and sizes of these various regions can
   be obtained using the GM_GET_MAPPING_SPECS ioctl, which returns
   a description of the mappings in a "gm_mapping_specs"
   structure.

   The pages of the copy block must be mapped in order and without
   skipping any offsets.

   gm_segmap ensures that all offsets passed to gm_mmap are
   legitimate, so there is no need to range check offsets here.
   However, checks must be performed here for "promiscuous" mappings
   of the device registers an SRAM areas other than the "send queue"
   and "activation" pages. */

int
gm_freebsd_mmap(dev_t dev, vm_offset_t off, int prot)
{
	void *result;
	gm_port_state_t *ps;
	int minor_id;
	gm_instance_state_t *is;
	u_long phys;
	long pfn = 0;
	off_t memory_offset = 0;
	gm_status_t status;

	minor_id = minor(dev);

	GM_PRINT (GM_PRINT_LEVEL >= 6, ("gm_freebsd_mmap(0x%x, 0x%x, 0x%x) called   minor = %d\n",
			 minor_id, dev, off, prot));

	ps = gm_minor_get_port_state(GET_MINOR(minor_id));
	if (!ps)  {
		GM_NOTE (("gm_freebsd_mmap: can't get port_state for minor = %d\n",
				  GET_MINOR(minor_id)));
		goto abort_with_nothing;
	}

	gm_arch_mutex_enter(&ps->sync);

	/* Verify the mapping is legal and ensure all resources are
	   available, allocating them if needed. */

	status = gm_prepare_to_mmap(ps, off, GM_PAGE_LEN,
		(prot & PROT_WRITE ? GM_MAP_WRITE : 0
		| prot & PROT_READ ? GM_MAP_READ : 0));
	if (status != GM_SUCCESS) {
		GM_NOTE (("Error preparing to mmap.\n"));
		goto abort_with_mutex;
	}

	GM_PRINT (GM_PRINT_LEVEL >= 5, ("Mapping offset 0x%x for minor device %d\n", off,
				 GET_MINOR(minor(dev))));

	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_freebsd_mmap: Calling gm_mmap().\n"));
	status = gm_mmap(ps, off, &result);
	if (status != GM_SUCCESS) {
		GM_NOTE (("gm_freebsd_mmap: gm_mmap returns -1\n"));
		goto abort_with_nothing;
	}
	gm_assert(result);

	if (!gm_mapping_in_io_space(ps, off)) {
		phys = vtophys(result);
	}
	else {
#ifdef __alpha__
		phys = (vm_offset_t) result;
#endif
#ifdef __i386__
		phys = vtophys(result);
#endif
	}
	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_freebsd_mmap: phys = 0x%lx\n", phys));

#if defined (__alpha__)
	pfn = alpha_btop(phys);
#endif
#if defined (__i386__)
	pfn = i386_btop(phys);
#endif

	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_freebsd_mmap: kseg 0x%lx frame 0x%lx\n", result, pfn));
  abort_with_mutex:
	gm_arch_mutex_exit(&ps->sync);
	GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_freebsd_mmap: returning 0x%lx.\n", pfn));
	GM_RETURN_INT ((int) pfn);

  abort_with_nothing:
	GM_RETURN_INT (0);
}

void
gm_freebsd_intr(void *arg)
{
	gm_instance_state_t *is;
	is = (gm_instance_state_t *) arg;
	if (gm_intr(is) == GM_ARCH_INTR_CLAIMED)
		GM_PRINT (GM_PRINT_LEVEL >= 9, ("the interrupt was claimed\n"));
	else
		GM_PRINT (GM_PRINT_LEVEL >= 9, ("the interrupt was not claimed\n"));
	return;
}

gm_status_t
gm_arch_lock_user_buffer_page(gm_instance_state_t * is, gm_up_t uaddr,
							  gm_dp_t * dma_addr, gm_arch_page_lock_t * lock)
{
	int error = 0;
	struct proc *p = curproc;
	vm_map_t map;
	vm_offset_t addr;
	GM_PRINT (GM_PRINT_LEVEL >= 1, ("enter locking memory\n"));
	if (!p) {
		GM_NOTE (("gm locking memory: curproc was null\n"));
		return GM_OUT_OF_MEMORY;
	}
	map = &p->p_vmspace->vm_map;
	/* must be page aligned */
	addr = trunc_page((vm_offset_t) uaddr);
	if (1 + cnt.v_wire_count > vm_page_max_wired) {
		GM_NOTE (("gm locking memory: too many wired pages (max = %d pages)\n",
				  vm_page_max_wired));
		return GM_OUT_OF_MEMORY;
	}
	error = vm_map_user_pageable(map, addr, addr + PAGE_SIZE, FALSE);
	if (error) {
		GM_NOTE (("gm locking memory: user attempted to lock invalid page\n"));
		return GM_FAILURE;
	}
	lock->proc = p;
	lock->addr = addr;
#ifdef __alpha__
	*dma_addr = alpha_XXX_dmamap_or | pmap_extract(vm_map_pmap(map), addr);
#endif
#ifdef __i386__
	*dma_addr = pmap_extract(vm_map_pmap(map), addr);
#endif
	GM_PRINT (GM_PRINT_LEVEL >= 1, ("locking memory: success\n"));
	return GM_SUCCESS;
}


void
gm_arch_unlock_user_buffer_page(gm_arch_page_lock_t * lock)
{
	vm_map_t map;
	vm_offset_t addr;

	map = &lock->proc->p_vmspace->vm_map;
	addr = lock->addr;
	vm_map_user_pageable(map, addr, addr + PAGE_SIZE, TRUE);
}

void gm_arch_abort (void)
{
	printf ("don't panic, but gm_arch_abort () was called.");
}



/*
  This file uses feldy indentation:

  Local Variables:
  tab-width:4
  c-file-style:"bsd"
  End:
*/
