#!/usr/bin/wish
# file : rc.tcl
#
# this file loads and save the ~/.bwatchrc.tcl resource file
# which contains all user configurable options
#
# we first check if ~/.bWatchrc.tcl exists.  If so than we load i.e.
# source that script.  If ~/.bWatch.tcl does NOT exist than we create 
# one with default options


#--------------------------------------------------------------------
# Procedure saverc
#
#
#
#--------------------------------------------------------------------

proc saverc {} {
    
    global debug display colour freeSwapLimit freeMemLimit cell blank loadLimit
    global listOfHosts bWatchVersion homeDir
    
    # we first open a the file

    set errorCode [catch {set f [open "$homeDir/.bWatchrc.tcl" w+]} \
		       errorMessage]

    if {$errorCode} {
      	.console.messageList insert end \
	    "saverc :: error opening $homeDir/.bWatchrc.tcl."
      	.console.messageList insert end \
	    "saverc :: $errorMessage"
    } else {
	set errorCode [catch {
	    
	    
	    # Now we write our defaults into ~/.bWatchrc.tcl
	    puts $f "set bWatchrcVersion $bWatchVersion"
	    puts $f "set listOfHosts {$listOfHosts}"
	    puts $f ""
	    puts $f "# set the columns which should be displayed"
	    puts $f "#"
	    puts $f "# 1 - information will be displayed for every host"
	    puts $f "# 0 - column will not be shown"
	    puts $f ""
	    
	    
	    puts $f "# number of users on the system"
	    puts $f "set display(numUsers) $display(numUsers)"
	    puts $f ""
	    
	    puts $f "# system time"
	    puts $f "set display(time) $display(time)"
	    puts $f ""
	    
	    puts $f "# 1 minute average load"
	    puts $f "#set display(load1) $display(load1)"
	    puts $f ""
    
	    puts $f "# 5 minutes average load"
	    puts $f "set display(load5) $display(load5)" 
	    puts $f ""
	    
	    puts $f "# 15 minutes average load"
	    puts $f "set display(load15) $display(load15)"
	    puts $f ""
	    
	    puts $f "# number of processes"
	    puts $f "set display(numProcesses) $display(numProcesses)"
	    puts $f ""
	    
	    puts $f "# total RAM"
	    puts $f "set display(totalMemory) $display(totalMemory)"
	    puts $f ""
	    
	    puts $f "# available free RAM"
	    puts $f "set display(freeMemory) $display(freeMemory)"
	    puts $f ""
	    
	    puts $f "# shared RAM"
	    puts $f "set display(sharedMemory) $display(sharedMemory)"
	    puts $f ""
	    
	    puts $f "# buffered memory" 
	    puts $f "set display(buffers) $display(buffers)"
	    puts $f ""
	    
	    puts $f "# cached memory"
	    puts $f "set display(cache) $display(cache)"
	    puts $f ""
	    
	    puts $f "# total swap space"
	    puts $f "set display(totalSwap) $display(totalSwap)"
	    puts $f ""
	    
	    puts $f "# available swap space"
	    puts $f "set display(freeSwap) $display(freeSwap)"    
	    puts $f ""
	    
	    puts $f "# foreground and background coulours"
	    puts $f "# of the heading of the table"
	    puts $f ""
	    
	    puts $f "set colour(headingBG) $colour(headingBG)"
	    puts $f "set colour(headingFG) $colour(headingFG)"
	    puts $f ""
	    
	    puts $f "# foreground and background colours"
	    puts $f "# of the first column showing host names"
	    puts $f ""
	    
	    puts $f "set colour(hostNameBG) $colour(hostNameBG)"
	    puts $f "set colour(hostNameFG) $colour(hostNameFG)"
	    puts $f ""
	    puts $f ""
	    
	    # the system load, memory etc is divided into 5 operating states represented
	    # by different colours :
	    #
	    # idle           - idle state, almost no memory usage (not implemented yet)
	    # neutral        - normal load conditions, CPU crunching away doing normal work
	    # firstWarning   - quite high load, low memory.  Not a major problem
	    # secondWarning  - overloaded, running out of memory.  Should investigate
	    # error          - load very high or running out of swap.  Can cause problems eg
	    #                  processes might start dying 
	    
	    puts $f "set colour(neutralFG) $colour(neutralFG)"
	    puts $f "set colour(firstWarningFG) $colour(firstWarningFG)"
	    puts $f "set colour(secondWarningFG) $colour(secondWarningFG)"
	    puts $f "set colour(errorFG) $colour(errorFG)"
	    puts $f "set colour(errorBG) $colour(errorBG)"
	    puts $f ""
	    puts $f ""
	    
	    # The five different operating states give us information
	    # about the system and how it is performing.
	    # Please modify the numbers to suit your system.
	    # numbers below represent my choice for topcat 
	    
	    # load limits
	    
	    # idle is up to neutral e.g 0.0 - 0.5 (not yet used)
	    #set loadsLimit(neutral)                   0.5
	    
	    puts $f "set loadLimit(firstWarning) $loadLimit(firstWarning)"
	    puts $f "set loadLimit(secondWarning) $loadLimit(secondWarning)"
	    puts $f "set loadLimit(error) $loadLimit(error)"
	    puts $f ""
	    puts $f ""
	    
	    # memory limits in KB
	    #set freeMemLimit(neutral)
	    puts $f "set freeMemLimit(firstWarning) $freeMemLimit(firstWarning)"
	    puts $f "set freeMemLimit(secondWarning) $freeMemLimit(secondWarning)"
	    puts $f "set freeMemLimit(error) $freeMemLimit(error)"
	    puts $f ""
	    puts $f ""
	    
	    # swap limits in KB
	    
	    # give me first warning when 
	    # swap less then 20 MB
	    puts $f "set freeSwapLimit(firstWarning) $freeSwapLimit(firstWarning)"
	    
	    # give me socond warning when
	    # swap less then 10 MB
	    puts $f "set freeSwapLimit(secondWarning) $freeSwapLimit(secondWarning)"
	    
	    # give me third warning when 
	    # swap less then 5 MB
	    puts $f "set freeSwapLimit(error) $freeSwapLimit(error)"
	    puts $f ""
	    puts $f ""
	    
	    # initial display 
	    
	    # what should be displayed when
	    # information cannot be obtained
	    puts $f "set blank $blank"
	    puts $f ""
	    
	    # default width of all the cells
	    puts $f "set cell(width) $cell(width)"
	} errorMessage]
	
	if ${errorCode} {
	    .console.messageList insert end \
		"saverc :: $errorMessage"
	}
	close $f
    }
}


#--------------------------------------------------------------------
# Procedure loadrc
#
#        try loading file ~/.bWatchrc.tcl  If this file does NOT 
#        exist then call procedure saverc which will create this
#        file and save default options  
#
#--------------------------------------------------------------------

proc loadrc {} {
    global debug display colour freeSwapLimit freeMemLimit cell blank loadLimit
    global listOfHosts homeDir

    # we first check if the file exists
    set errorCode [catch {source "$homeDir/.bWatchrc.tcl"} errorMessage]
    if {$errorCode} {
	.console.messageList insert end \
	    "loadrc : $errorMessage"
	.console.messageList insert end \
	    "loadrc : Creating $homeDir/.bWatchrc.tcl ... "
	# save default
	saverc
    } else {
	.console.messageList insert end \
	    "$homeDir/.bWatchrc.tcl loaded."
    }
}











    










#----------------------------------------------------------------------------
# procedure showConsole
#
#      procedure creates a toplevel .console which is the console for bWatch
#      and contains all error messages returned from commands sent to hosts
#
#----------------------------------------------------------------------------

proc showConsole {} {
    toplevel .console
    wm title .console "bWatch Console"

    # main frame which will hold the list box and the scroll bar

    frame .console.mainFrame
    
    pack .console.mainFrame \
	-fill both \
	-expand 1

    # the listbox of error messages

    listbox .console.messageList \
	-yscrollcommand ".console.vScrollBar set" \
	-xscrollcommand ".console.hScrollBar set" \
	-width 50 \
	-height 5 

    pack .console.messageList \
	-in .console.mainFrame \
	-side left \
	-fill both \
	-expand 1

    # Vertical scroll bar

    scrollbar .console.vScrollBar \
	-orient vertical \
	-command ".console.messageList yview"
    
    pack .console.vScrollBar \
	-in .console.mainFrame \
	-side right \
	-fill y 

    # Horisontal scroll bar

    scrollbar .console.hScrollBar \
	-orient horizontal \
	-command ".console.messageList xview"
    
    pack .console.hScrollBar \
	-fill x

    # clear button

    button .console.clearButton \
	-text "Clear" \
	-command {.console.messageList delete 0 1000}

    pack .console.clearButton \
	-fill x 

    return
}


# file main.tcl
# This file contains the main procedure refresh
# which is the main loop of the software.
#----------------------------------------------------------------------------
# procedure refresh
# 
#      this procedure goes through the listOfHosts list and for each
#      of the hosts listed collects the relevent information
#----------------------------------------------------------------------------

proc refresh {} {

    global upTime numProcesses listOfHosts load1 load5 load15 loads freeMemory
    global time numUsers colour display error
    
    # for each of the hosts in the listOfHosts
    # update its informations
    
    foreach host $listOfHosts {
	
	# reset the error flag
	unSetError $host
	."$host"HostName configure \
	    -fg $colour(hostNameBG) \
	    -bg $colour(hostNameFG)
	update

	# check if the host is alive
	# if not hen set error flag 

	set pingResult [pingHost $host]
	
	if {$pingResult} {
	    setError $host
	} else { 
	    
	    # we only do the following if host is alive !
	    # if not then give up
	    # get uptime
	    set upTimeResult [getUpTime $host]
	}
	update
	
	# check for errors

	if {$upTimeResult == "error"} {
	    setError $host
	}
	update

	# if there was an error then set error
	# status for this host and return
	
	if {$error($host)} {
	    setError $host
	} else {
	    displayUpTime $host $upTimeResult
	}
	update
	
	# get number of processes if number of processes column
	# is enabled and there was no error in previous operation
	if {$display(numProcesses) && ($error($host) == 0)} {
	    set numProcessesResult [getNumberOfProcesses $host]
	    if {$numProcessesResult == "error"} {
		setError $host
	    } else {
		displayNumberOfProcesses $host $numProcessesResult
	    }
	    update
	}
	
	# if no previous errors then
	# collect the memory information from /proc/meminfo
	# and store it locally

	if {$error($host)} {
	} else {
	    set memInfoResult [getMemInfo $host]
	    if {$memInfoResult == "error"} {
		setError $host
	    }
	}
	
	update
	
	# get total memory if the total memory column is enabled
	# and no errors were detected in previous operations on this host

	if {$display(totalMemory) && ($error($host) == 0)} {
	    set totalMemResult [getTotalMemory $host]
	    if {$totalMemResult == "error"} {
		setError $host
	    } else {
		displayTotalMemory $host $totalMemResult
	    }
	    update
	}
	
	# get free memory 
	if {$display(freeMemory) && ($error($host) == 0)} {
	    set freeMemResult [getFreeMemory $host]
	    if {$freeMemResult == "error"} {
		setError $host
	    } else {
		displayFreeMemory $host $freeMemResult
	    }
	    update
	}


	# get shared memory if there were no previous errors
	# and this display is enabled
	if {$display(sharedMemory) && ($error($host) == 0)} {
	    set sharedMemResult [getSharedMemory $host]
	    if {$sharedMemResult == "error"} {
		setError $host
	    } else {
		displaySharedMemory $host $sharedMemResult
	    }
	    update
	}


	# get buffers
	if {$display(buffers) && ($error($host) == 0)} {
	    set buffersResult [getBuffers $host]
	    if {$buffersResult == "error"} {
		setError $host
	    } else {
		displayBuffers $host $buffersResult
	    }
	    update
	}


	# get cache 
	if {$display(cache) && ($error($host) == 0)} {
	    set cacheResult [getCache $host]
	    if {$cacheResult == "error"} {
		setError $host
	    } else {
		displayCache $host $cacheResult
	    }
	    update
	}


	# get Total Swap
	if {$display(totalSwap) && ($error($host) == 0)} {
	    set totSwapResult [getTotalSwap $host]
	    if {$totSwapResult == "error"} {
		setError $host
	    } else {
		displayTotalSwap $host $totSwapResult
	    }
	    update
	}


	# get Free Swap
	if {$display(freeSwap) && ($error($host) == 0)} {
	    set freeSwapResult [getFreeSwap $host]
	    if {$freeSwapResult == "error"} {
		setError $host
	    } else {
		displayFreeSwap $host $freeSwapResult
	    }
	    update
	}

	# if we had any errors than we set the colour of the host name label
	# to error color.  If there were no errors then we reset the coulour
	# to the original.
	
	if {$error($host)} {
	    ."$host"HostName configure \
		-fg $colour(errorBG) \
		-bg $colour(errorFG)
	} else {
	    	."$host"HostName configure \
	    -fg $colour(hostNameFG) \
	    -bg $colour(hostNameBG)
	}
    }
    
    # update the screen after each host's info
    # has been collected
    update
}

#!/usr/bin/wish
#
# file init.tcl
#
# there are no more user configurable parts in this file
# please customise $HOME/.bWatchrc.tcl

# everything starts here ...

# version number
set bWatchVersion 1.0.2

# debugging level
set debug 0

#------------------------------------------------------------------------
# nothing beyond here should have to be changed
#------------------------------------------------------------------------

# set the homeDir variable from the enviroment variable $HOME
# but catch all errors !
set errorCode [catch {set homeDir    $env(HOME)} errorMessage]
if {$errorCode} {
    puts "init.tcl : Can't find your home directory.  Giving up."
    exit
}
set bWatchDir  "$homeDir"

# window manager options
# let us catch this because the DISPLAY variable might not
# be set or we might be trying to run bWatch in a text console

set errorCode [catch {wm title . "bWatch $bWatchVersion"} errorMessage]
if {$errorCode} {
    puts "init.tcl : I can't set the window title.  Giving up."
    exit
}

# other window manager options ...
wm iconname . "bWatch $bWatchVersion"


# remove any old tmp files ( this was for version 1.0.0 )
# catch {exec /bin/rm -f /tmp/bWatch*}

# show the console window
# all error messgages will be logged there

showConsole

# we now set the default values.  If $HOME/.bWatchrc.tcl exists
# it will be loaded later and these values will be over written

# get the name of the machine on which bWatch is running
set errorCode [catch {set hostName [exec hostname]} errorMessage]

if {$debug && $errorCode} {
    puts "init.tcl : $errorMessage"
} else { 
    regexp {[a-zA-Z0-9]+} $hostName localHost 
    set listOfHosts "$localHost"
}

# set the columns which should be displayed
#
# 1 - information will be displayed for every host
# 0 - column will not be shown

# number of users on the system
set display(numUsers)         1 

# system time
set display(time)             1

# 1 minute average load
set display(load1)            1

# 5 minutes average load
set display(load5)            1

# 15 minutes average load
set display(load15)           1

# number of processes
set display(numProcesses)     1

# total RAM 
set display(totalMemory)      1

# available free RAM
set display(freeMemory)       1

# shared RAM
set display(sharedMemory)     1

# buffered memory
set display(buffers)          1

# cached memory
set display(cache)            1

# total swap space
set display(totalSwap)        1

# available swap space
set display(freeSwap)         1


# set commands
#
# these are the commands used to obtain information about the hosts

set command(ping)               "/bin/ping"
set command(uptime)             "/usr/bin/uptime "
set command(rsh)                "/usr/bin/rsh"
set command(date)               "/bin/date" 
set command(numberOfProcesses)  "/bin/ps ax | wc"
set command(cat)                "/bin/cat"
set command(memInfo)            "cat /proc/meminfo"


# set colours

# foreground and background coulours
# of the heading of the table

set colour(headingBG)         #066
set colour(headingFG)         #FFF

# foreground and background colours
# of the first column showing host names

set colour(hostNameBG)        #006
set colour(hostNameFG)        #FFF

# the system load, memory etc is divided into 5 operating states represented
# by different colours :
#
# idle           - idle state, almost no memory usage (not implemented yet)
# neutral        - normal load conditions, CPU crunching away doing normal work
# firstWarning   - quite high load, low memory.  Not a major problem
# secondWarning  - overloaded, running out of memory.  Should investigate
# error          - load very high or running out of swap.  Can cause problems eg
#                  processes might start dying 
  
set colour(neutralFG)         black
set colour(neutralBG)         white
set colour(firstWarningFG)    orange
set colour(secondWarningFG)   #A00
set colour(errorFG)           #F00
set colour(errorBG)           #FFF


# The five different operating states give us information
# about the system and how it is performing.
# Please modify the numbers to suit your system.
# numbers below represent my choice for topcat 

# load limits

# idle is up to neutral e.g 0.0 - 0.5 (not yet used)
#set loadsLimit(neutral)                   0.5
set loadLimit(firstWarning)                1.5
set loadLimit(secondWarning)               2.5 
set loadLimit(error)                       5.0

# memory limits in KB
#set freeMemLimit(neutral)
set freeMemLimit(firstWarning)            1024
set freeMemLimit(secondWarning)            512
set freeMemLimit(error)                      1

# swap limits in KB

# give me first warning when 
# swap less then 20 MB
set freeSwapLimit(firstWarning)          20480

# give me socond warning when
# swap less then 10 MB
set freeSwapLimit(secondWarning)         10240

# give me third warning when 
# swap less then 5 MB
set freeSwapLimit(error)                  5120     



# initial display 

# what should be displayed when
# information cannot be obtained
set blank                            "-------"

# default width of all the cells
set cell(width)                              8


# We now load the resource file if there is one
# if the we succeed then our default values just set will
# be overwritten

loadrc


#----------------------------------------------------------------------------
# procedure getHostInfo
# 
# creates a toplevel window and displays more detailed
# information about the host
#
# NOT USED YET
#----------------------------------------------------------------------------

proc getHostInfo {host} {

    toplevel ."$host"
    wm title ."$host" "Information about $host"
    
    set allProcesses [getAllProcesses $host]
    
    text ."$host".test \
	-width 120 \
	-height 40
    
    pack ."$host".test \
	
    
    ."$host".test insert end $allProcesses
    

    # build button
    
    frame ."$host".buttonFrame
    
    pack ."$host".buttonFrame \
	-expand 1 \
	-fill x \
	-in ."$host" 
    
    button ."$host".refreshButton \
	-command {} \
	-text "?????" 
    
    pack ."$host".refreshButton \
	-in ."$host".buttonFrame \
	-side left \
	-fill x
    
    
    button ."$host".exitButton \
	-text {Kill Me} \
	-command {}
    
    
    pack ."$host".exitButton \
	-in ."$host".buttonFrame \
	-side left \
	-fill x
    
    bind ."$host".exitButton <ButtonPress-1> {
	set blah [regexp {\.\"[0-9a-zA-Z]+\"} %W myname]
	destroy $myname
    } 
}


#----------------------------------------------------------------------------
# procedure setError
#
#        this procedure sets the error for a particular $host by changing
#        all its columns to $blank with coulour $colour(errorFG)
#
# arguments
#        host - the host for which the error should be set
#
#----------------------------------------------------------------------------

proc setError {host} {
    global colour blank numUsers time load1 load5 load15 numProcesses \
	totalmemory freeMemory sharedMemory buffers cache totalSwap FreeSwap \
	error

    # set the error flag to true
    set error($host) 1

    ."$host"HostName configure \
	-fg $colour(errorBG) \
	-bg $colour(errorFG)
    

    catch {."$host"numUsers configure \
	       -fg $colour(errorFG)}

    catch {."$host"time configure \
	       -fg $colour(errorFG)}

    catch {."$host"load1 configure \
	       -fg $colour(errorFG)}

    catch {."$host"load5 configure \
	       -fg $colour(errorFG)}

    catch {."$host"load15 configure \
	       -fg $colour(errorFG)}

    catch {."$host"numProcesses configure \
	       -fg $colour(errorFG)}

    catch {."$host"totalMemory configure \
	       -fg $colour(errorFG)}

    catch {."$host"freeMemory configure \
	       -fg $colour(errorFG)}

    catch {."$host"sharedMemory configure \
	       -fg $colour(errorFG)}

    catch {."$host"buffers configure \
	       -fg $colour(errorFG)}

    catch {."$host"cache configure \
	       -fg $colour(errorFG)}

    catch {."$host"totalSwap configure \
	       -fg $colour(errorFG)}

    catch {."$host"freeSwap configure \
	       -fg $colour(errorFG)}
}

#----------------------------------------------------------------------------
# procedure unSetError
#
#        this procedure unsets the error for a particular $host by changing
#        all its columns 
#
# arguments
#        host - the host for which the error should be unset
#
#----------------------------------------------------------------------------

proc unSetError {host} {
    global colour blank numUsers time load1 load5 load15 numProcesses \
	totalmemory freeMemory sharedMemory buffers cache totalSwap FreeSwap \
	error

    # set the error flag to true
    set error($host) 0

    ."$host"HostName configure \
	-fg $colour(neutralFG) \
	-bg $colour(neutralBG)
    

    catch {."$host"numUsers configure \
	       -fg $colour(neutralFG)}

    catch {."$host"time configure \
	       -fg $colour(neutralFG)}

    catch {."$host"load1 configure \
	       -fg $colour(neutralFG)}

    catch {."$host"load5 configure \
	       -fg $colour(neutralFG)}

    catch {."$host"load15 configure \
	       -fg $colour(neutralFG)}

    catch {."$host"numProcesses configure \
	       -fg $colour(neutralFG)}

    catch {."$host"totalMemory configure \
	       -fg $colour(neutralFG)}

    catch {."$host"freeMemory configure \
	       -fg $colour(neutralFG)}

    catch {."$host"sharedMemory configure \
	       -fg $colour(neutralFG)}

    catch {."$host"buffers configure \
	       -fg $colour(neutralFG)}

    catch {."$host"cache configure \
	       -fg $colour(neutralFG)}

    catch {."$host"totalSwap configure \
	       -fg $colour(neutralFG)}

    catch {."$host"freeSwap configure \
	       -fg $colour(neutralFG)}
}

#----------------------------------------------------------------------------
# procedure getDate
# 
#        procedure executes the command(date) and returns its result
#        no error checking is done
#
#----------------------------------------------------------------------------

proc getDate {} {

    global command
    set date [exec $command(date)]
    return $date
}


#----------------------------------------------------------------------------
# procedure pingHost
#
#        procedure sends one ping to $host using the $command(ping)
#        if an error occurs pingHost will write this error to the console
#        
# Arguments :
#        $host - the name of the host to ping
#
# Results :
#        procedure returns 0 if the host replied to the ping or
#        returns the error code which was returned by command(ping) 
# 
#----------------------------------------------------------------------------

proc pingHost {host} {
    
    global command
    set errorCode \
	[catch {set pingResult \
		    [exec $command(ping) -c 1 $host]} errorMessage]
    
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: pingHost($host) : $errorMessage"
	return $errorCode
    } else {
	return 0
    }
}


#----------------------------------------------------------------------------
# procedure getUpTime
# 
#        Procedure sends the $command(uptime) to the $host and stores the
#        result in local variable upTime
# 
# Arguments :
#        $host - the name of the host to which $command(uptime) should be
#                send to
#
# Results :
#        Procedure returns either the string which wqas returned by 
#        $command(uptime) or the string "error" if any errors occured
#
#----------------------------------------------------------------------------

proc getUpTime {host} {

    global command

    set errorCode \
	[catch {set upTime [exec $command(rsh) -n \
				$host $command(uptime)]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getUpTime($host) : $errorMessage"
	return "error"
    } else {
	return $upTime
    }
}



#----------------------------------------------------------------------------
# procedure getMemInfo
# 
# returns the ammount of free memory left on the host
# $host is passed as an argument to this procedure
#----------------------------------------------------------------------------

proc getMemInfo {host} {

    global command bWatchDir

    set errorCode \
	[catch {exec $command(rsh) -n \
		    $host $command(memInfo) > $bWatchDir/bWatchMemInfo } errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getMemInfo($host) : $errorMessage"
	return "error"
    } else {
	return
    }
}

#----------------------------------------------------------------------------
# procedure getFreeMemory
# 
# returns the ammount of free memory left on the host
# $host is passed as an argument to this procedure
#----------------------------------------------------------------------------

proc getFreeMemory {host} {

    global command bWatchDir

    set errorCode \
	[catch {set freeMemory \
		    [exec cat $bWatchDir/bWatchMemInfo | grep MemFree]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getFreeMemory($host) : $errorMessage"
	return "error"
    } else {
	return $freeMemory
    }
}


#----------------------------------------------------------------------------
# procedure getTotalMemory
# 
# returns the ammount of free memory left on the host
# $host is passed as an argument to this procedure
#----------------------------------------------------------------------------

proc getTotalMemory {host} {

    global command bWatchDir

    set errorCode \
	[catch {set totalMemory \
		    [exec cat $bWatchDir/bWatchMemInfo | grep MemTotal]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getTotalMemory($host) : $errorMessage"
	return "error"
    } else {
	return $totalMemory
    }
}


#----------------------------------------------------------------------------
# procedure getSharedMemory
# 
#----------------------------------------------------------------------------

proc getSharedMemory {host} {

    global command bWatchDir

    set errorCode \
	[catch {set sharedMemory \
		    [exec cat $bWatchDir/bWatchMemInfo | grep MemShared]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getSharedMemory($host) : $errorMessage"
	return "error"
    } else {
	return $sharedMemory
    }
}


#----------------------------------------------------------------------------
# procedure getBuffers
#----------------------------------------------------------------------------

proc getBuffers {host} {

    global command bWatchDir

    set errorCode \
	[catch {set buffers \
		    [exec cat $bWatchDir/bWatchMemInfo | grep Buffers]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getBuffers($host) : $errorMessage"
	return "error"
    } else {
	return $buffers
    }
}


#----------------------------------------------------------------------------
# procedure getCache
#----------------------------------------------------------------------------

proc getCache {host} {

    global command bWatchDir

    set errorCode \
	[catch {set cache \
		    [exec cat $bWatchDir/bWatchMemInfo | grep Cached]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getCache($host) : $errorMessage"
	return "error"
    } else {
	return $cache
    }
}


#----------------------------------------------------------------------------
# procedure getTotalSwap
#----------------------------------------------------------------------------

proc getTotalSwap {host} {

    global command bWatchDir

    set errorCode \
	[catch {set totalSwap \
		    [exec cat $bWatchDir/bWatchMemInfo | grep SwapTotal]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getTotalSwap($host) : $errorMessage"
	return "error"
    } else {
	return $totalSwap
    }
}


#----------------------------------------------------------------------------
# procedure getFreeSwap
#----------------------------------------------------------------------------

proc getFreeSwap {host} {

    global command bWatchDir

    set errorCode \
	[catch {set freeSwap \
		    [exec cat $bWatchDir/bWatchMemInfo | grep SwapFree]} errorMessage] 
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getFreeSwap($host) : $errorMessage"
	return "error"
    } else {
	return $freeSwap
    }
}


#----------------------------------------------------------------------------
# procedure getNumberOfProcesses 
#
# returns number of current processes on $host
# $host is this procedures argument
#----------------------------------------------------------------------------

proc getNumberOfProcesses {host} {

    global command

    set errorCode \
	[catch {set numProcesses [exec $command(rsh) -n \
				      $host $command(numberOfProcesses)]} errorMessage]
    # check if the command returned an error
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: getNumberOfProcesses($host) : $errorMessage"
	return "error"
    } else {
	return $numProcesses
    }
}

#----------------------------------------------------------------------------
# procedure displayUpTime
#
#     this procedure accepts as an argument the result
#     of getUpTime procedure call.
#     It splits this one long string into 3 load values, system time
#     and number of users on the system checking for errors 
#     and displays it on the screen.
#
# args 
#     host   - name of this host
#     uptime - uptime result for this host.  if there was an error
#              in obtaining this result then the value will be "error" 
#----------------------------------------------------------------------------

proc displayUpTime {host upTimeResult} {

    global load1 load5 load15 time numUsers colour loadLimit blank display debug
   
    regexp {([0-9]+\.[0-9]+), ([0-9]+\.[0-9]+), ([0-9]+\.[0-9]+)} \
	"$upTimeResult" loadInfo 
    
    if {$debug} {
	puts "displayUpTime($host) : upTimeResult =  $upTimeResult"
	puts "displayUpTime($host) : loadInfo =  $loadInfo"
    }
    
    # check if time should be displayed
    if {$display(time)} {
	regexp {([0-9]+:[0-9]+)} \
	    "$upTimeResult" time($host)
	."$host"time configure \
	    -fg $colour(neutralFG)
    }

    # check if number of users should be displayed
    if {$display(numUsers)} { 
	regexp {([0-9]+ user.)} \
	    "$upTimeResult" numUsers($host)
	
	."$host"numUsers configure \
	    -fg $colour(neutralFG)
    }
    
    # now we split this string into a list of three number
    set loadsList [split $loadInfo ,]
    set load1($host)  [lindex $loadsList 0]
    set load5($host)  [lindex $loadsList 1]
    set load15($host) [lindex $loadsList 2]
    
        
    if {$display(load1)} {
	# determine the colour of 1 minute load
	if {$load1($host) > $loadLimit(firstWarning)} {
	    if {$load1($host) > $loadLimit(secondWarning)} {
		if {$load1($host) > $loadLimit(error)} {
		    ."$host"load1 configure \
			-fg $colour(errorBG) \
			-bg $colour(errorFG)
		} else {
		    ."$host"load1 configure \
			-fg $colour(secondWarningFG)
		}
	    } else {
		."$host"load1 configure \
		    -fg $colour(firstWarningFG)
	    }
	} else {
	    ."$host"load1 configure \
		-fg $colour(neutralFG)
	}
    }
    

    if {$display(load5)} {
	# determine the colour of 1 minute load
	if {$load5($host) > $loadLimit(firstWarning)} {
	    if {$load5($host) > $loadLimit(secondWarning)} {
		if {$load5($host) > $loadLimit(error)} {
		    ."$host"load5 configure \
			-fg $colour(errorBG) \
			-bg $colour(errorFG)
		} else {
		    ."$host"load5 configure \
			-fg $colour(secondWarningFG)
		}
	    } else {
		."$host"load5 configure \
		    -fg $colour(firstWarningFG)
	    }
	} else {
	    ."$host"load5 configure \
		-fg $colour(neutralFG)
	}
    }
    

    if {$display(load15)} {
	# determine the colour of 1 minute load
	if {$load15($host) > $loadLimit(firstWarning)} {
	    if {$load15($host) > $loadLimit(secondWarning)} {
		if {$load15($host) > $loadLimit(error)} {
		    ."$host"load15 configure \
			-fg $colour(errorBG) \
			-bg $colour(errorFG)
		} else {
		    ."$host"load15 configure \
			-fg $colour(secondWarningFG)
		}
	    } else {
		."$host"load15 configure \
		    -fg $colour(firstWarningFG)
	    }
	} else {
	    ."$host"load15 configure \
		-fg $colour(neutralFG)
	}
    }
    

}


#----------------------------------------------------------------------------
# procedure displayNumberOfProcesses
#
#----------------------------------------------------------------------------

proc displayNumberOfProcesses {host numProcs} {

    global colour numProcesses

    regexp {[0-9]+} $numProcs numProcesses($host) 
}


#----------------------------------------------------------------------------
# procedure displayTotalMemory
#
#----------------------------------------------------------------------------

proc displayTotalMemory {host totalMem} {

    global colour totalMemory

    regexp {[0-9]+} "$totalMem" totalMemory($host)
    set totalMemory($host) "[expr $totalMemory($host) / 1024] Mb"
}


#----------------------------------------------------------------------------
# procedure displayFreeMemory
#
#----------------------------------------------------------------------------

proc displayFreeMemory {host freeMem} {

    global colour freeMemory freeMemLimit

    regexp {[0-9]+} "$freeMem" freeMemory($host)
#    set freeMemory($host) "[expr $freeMemory($host) / 1024] Mb"

    # colour
    
    if {$freeMemory($host) < $freeMemLimit(firstWarning)} {
	if {$freeMemory($host) < $freeMemLimit(secondWarning)} {
	    if {$freeMemory($host) < $freeMemLimit(error)} {
		."$host"freeMemory configure \
		    -fg $colour(errorBG) \
		    -bg $colour(errorFG)
	    } else {
		."$host"freeMemory configure \
		    -fg $colour(secondWarningFG)
	    }
	} else {
	    ."$host"freeMemory configure \
		-fg $colour(firstWarningFG)
	}
    } else {
	."$host"freeMemory configure \
	    -fg $colour(neutralFG)
    }    
	    
    set freeMemory($host) "$freeMemory($host) Kb"
}


#----------------------------------------------------------------------------
# procedure displaySharedMemory
#
#----------------------------------------------------------------------------

proc displaySharedMemory {host shared} {

    global sharedMemory

    regexp {[0-9]+} "$shared" sharedMemory($host)
    set sharedMemory($host) "[expr $sharedMemory($host) / 1024] Mb"
}


#----------------------------------------------------------------------------
# procedure displayBuffers
#
#----------------------------------------------------------------------------

proc displayBuffers {host buff} {

    global buffers

    regexp {[0-9]+} "$buff" buffers($host)
    set buffers($host) "[expr $buffers($host) / 1024] Mb"
}


#----------------------------------------------------------------------------
# procedure displayCache
#
#----------------------------------------------------------------------------

proc displayCache {host c} {

    global cache

    regexp {[0-9]+} "$c" cache($host)
    set cache($host) "[expr $cache($host) / 1024] Mb"
}


#----------------------------------------------------------------------------
# procedure displayTotalSwap
#
#----------------------------------------------------------------------------

proc displayTotalSwap {host tSwap} {

    global totalSwap

    regexp {[0-9]+} "$tSwap" totalSwap($host)
    set totalSwap($host) "[expr $totalSwap($host) / 1024] Mb"
}


#----------------------------------------------------------------------------
# procedure display freeSwap
#
#----------------------------------------------------------------------------

proc displayFreeSwap {host fSwap} {

    global freeSwap freeSwapLimit colour

    regexp {[0-9]+} "$fSwap" freeSwap($host)
    if {$freeSwap($host) < $freeSwapLimit(firstWarning)} {
	if {$freeSwap($host) < $freeSwapLimit(secondWarning)} {
	    ."$host"freeSwap configure \
		-fg $colour(secondWarningFG)
	} else {
	    ."$host"freeSwap configure \
		-fg $colour(firstWarningFG)
	}
    } else {
	."$host"freeSwap configure \
	    -fg $colour(neutralFG)
    }    
    set freeSwap($host) "[expr $freeSwap($host) / 1024] Mb"
}

#----------------------------------------------------------------------------
# Now build the screen
#
# main screen layout
# build the frammes where all the widgets will go into

# toolbar frame is reserved for future use.  It will
# hold a toolbar with tool buttons

frame .toolbarFrame

pack .toolbarFrame \
	-side top

# main holds all the labels showing memory, load etc
# for all the hosts

frame .mainFrame 

pack .mainFrame \
    -side top \
    -fill both \
    -expand 1

# button frame is the frame at the bottom of the screen
# which has two buttons at the moment : [refresh] and [exit]

frame .buttonFrame 

pack .buttonFrame \
	-side bottom \
	-fill x

# frame for the top heading for labels explaining
# what each column represents

frame .headingFrame 

pack .headingFrame \
    -in .mainFrame \
    -side top \
    -fill both \
    -expand 1

# build the headings

# name of the host heading 

label .hostnameHeading \
    -text "Host\nName" \
    -bg $colour(headingBG) \
    -fg $colour(headingFG) \
    -relief raised \
    -width $cell(width)

pack .hostnameHeading \
    -in .headingFrame \
    -side left \
    -fill both \
    -expand 1

# number of users heading

if {$display(numUsers)} {
    label .numUsersHeading \
	-text "Num\nUsers" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .numUsersHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# system time heading

if {$display(time)} {
    label .timeHeading \
	-text "Time" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .timeHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# 1 minute load heading

if {$display(load1)} {
    label .load1Heading \
	-text "1 min\nLoad" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .load1Heading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the 5 minute load column 

if {$display(load5)} {
    label .load5Heading \
	-text "5 min\nLoad" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)

    pack .load5Heading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for rhe 15 minute load column

if {$display(load15)} {
    label .load15Heading \
	-text "15 min\nLoad" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .load15Heading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the number of processes column

if {$display(numProcesses)} {
    label .numOfProcessesHeading \
	-text "Num\nprocs." \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .numOfProcessesHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the Total Memory column

if {$display(totalMemory)} {
    label .totalMemoryHeading \
	-text "Total\nMem" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)

    pack .totalMemoryHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}


# build the heading for the FREE MEMORY column

if  {$display(freeMemory)} {
    label .freeMemoryHeading \
	-text "Free\nMem" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .freeMemoryHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the SHARED MEMORY column

if  {$display(sharedMemory)} {
    label .sharedMemoryHeading \
	-text "Shared\nMem" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .sharedMemoryHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the BUFFERS column

if  {$display(buffers)} {
    label .buffersHeading \
	-text "Buffers" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .buffersHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the CACHE column

if  {$display(cache)} {
    label .cacheHeading \
	-text "Cache" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .cacheHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the TOTAL SWAP column

if  {$display(totalSwap)} {
    label .totalSwapHeading \
	-text "Total\nSwap" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .totalSwapHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}

# build the heading for the FREE SWAP column

if  {$display(freeSwap)} {
    label .freeSwapHeading \
	-text "Free\nSwap" \
	-bg $colour(headingBG) \
	-fg $colour(headingFG) \
	-relief raised \
	-width $cell(width)
    
    pack .freeSwapHeading \
	-in .headingFrame \
	-side left \
	-fill both \
	-expand 1
}


# build buttons

button .refreshButton \
	-command {refresh} \
	-text "Refresh" 
	
pack .refreshButton \
	-in .buttonFrame \
	-side left \
	-expand 1 \
	-fill x


button .exitButton \
	-command {exit} \
	-text "Exit"

pack .exitButton \
	-in .buttonFrame \
	-side left \
	-expand 1 \
	-fill x


#----------------------------------------------------------------------------
# for each of the hosts in listOfHosts
# create a label in the table
#----------------------------------------------------------------------------

foreach host $listOfHosts {

    # initilize the colours and display

    set time($host)                $blank 
    set numUsers($host)            $blank
    set load1($host)               $blank
    set load5($host)               $blank
    set load15($host)              $blank
    set numProcesses($host)        $blank
    set totalMemory($host)         $blank
    set freeMemory($host)          $blank
    set sharedMemory($host)        $blank
    set buffers($host)             $blank
    set cache($host)               $blank
    set totalSwap($host)           $blank
    set freeSwap($host)            $blank 
    set timeColour($host)          $colour(neutralFG)
    set numUsersColour($host)      $colour(neutralFG)
    set load1Colour($host)         $colour(neutralFG)
    set load5Colour($host)         $colour(neutralFG)
    set load15Colour($host)        $colour(neutralFG)
    set numProcessesColour($host)  $colour(neutralFG)
    set totalMemoryColour($host)   $colour(neutralFG)
    set freeMemoryColour($host)    $colour(neutralFG)
    set sharedMemoryColour($host)  $colour(neutralFG)
    set buffersColour($host)       $colour(neutralFG)
    set cacheColour($host)         $colour(neutralFG)
    set totalSwapColour($host)     $colour(neutralFG)
    set freeSwapColour($host)      $colour(neutralFG)

    # build horisontal frame which will contain all the
    # labels with information for this host

    frame ."$host"Frame

    pack ."$host"Frame \
	    -in .mainFrame \
	    -side top \
	    -fill both \
	    -expand 1

    # host name
    # we always require this !

    label ."$host"HostName \
	-text $host \
	-bg $colour(hostNameBG) \
	-fg $colour(hostNameFG) \
	-width $cell(width) \
	-relief raised
    
    pack ."$host"HostName \
	-side left \
	-in ."$host"Frame \
	-fill both \
	-expand 1

    # number of user
    
    if {$display(numUsers)} {
	label ."$host"numUsers \
	    -textvariable numUsers($host) \
	    -fg $numUsersColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"numUsers  \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # system time
    
    if {$display(time)} {
	label ."$host"time \
	    -textvariable time($host) \
	    -fg $timeColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken

	pack ."$host"time  \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # 1 minute load

    if {$display(load1)} {
	label ."$host"load1 \
	    -textvariable load1($host) \
	    -fg $load1Colour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"load1 \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # 5 minute load
	
    if {$display(load5)} {
	label ."$host"load5 \
	    -textvariable load5($host) \
	    -fg $load5Colour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"load5 \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # 15minute load

    if {$display(load15)} {
	label ."$host"load15 \
	    -textvariable load15($host) \
	    -fg $load15Colour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"load15 \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # number of processes

    if {$display(numProcesses)} {
        label ."$host"numProcesses \
	    -textvariable numProcesses($host) \
	    -fg $numProcessesColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken

	pack ."$host"numProcesses \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # label to show the ammount of free memory on each host    
    
    if {$display(totalMemory)} {
	label ."$host"totalMemory \
	    -textvariable totalMemory($host) \
	    -fg $totalMemoryColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"totalMemory \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # label to show the ammount of free memory on each host    
    
    if {$display(freeMemory)} {
	label ."$host"freeMemory \
	    -textvariable freeMemory($host) \
	    -fg $freeMemoryColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"freeMemory \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }
    # label to show the ammount of shared memory on each host    
    
    if {$display(sharedMemory)} {
	label ."$host"sharedMemory \
	    -textvariable sharedMemory($host) \
	    -fg $sharedMemoryColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"sharedMemory \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # label to show the ammount of buffers on each host    
    
    if {$display(buffers)} {
	label ."$host"buffers \
	    -textvariable buffers($host) \
	    -fg $buffersColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"buffers \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # label to show the ammount of cache on each host    
    
    if {$display(cache)} {
	label ."$host"cache \
	    -textvariable cache($host) \
	    -fg $cacheColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"cache \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }

    # label to show the ammount of Total Swap on each host    
    
    if {$display(totalSwap)} {
	label ."$host"totalSwap \
	    -textvariable totalSwap($host) \
	    -fg $totalSwapColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"totalSwap \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }
    # label to show the ammount of free memory on each host    
    
    if {$display(freeSwap)} {
	label ."$host"freeSwap \
	    -textvariable freeSwap($host) \
	    -fg $freeSwapColour($host) \
	    -width $cell(width) \
	    -justify left \
	    -relief sunken
	
	pack ."$host"freeSwap \
	    -side left \
	    -in ."$host"Frame \
	    -fill both \
	    -expand 1
    }
    update
}

refresh



