/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This fairly simple programs is used to automatically generate
   conservative Makefile dependency lists for the source files
   specified on stdin.  Unlike other similar programs, this program is
   designed to generate a single dependency file including dependencies
   for all architectures.

   The program works as follows: The user supplies on STDIN a list of
   all source files within the project, including all header files.
   The program scans these files for include lines, and then outputs a
   dependency list, indicating that each C source file depends on all
   header files with the same basename.  This conservative strategy
   works quite well for the GM source tree.

   This program has some GM-specific features.  It treats
   the mcp/, drivers/, and libgm/ directories specially. */


#include "gm_config.h"
#include "gm.h"

#ifdef STDC_HEADERS
#include <stdlib.h>		/* malloc */
#include <string.h>
#include <stdio.h>
#endif

#if defined HAVE_UNISTD_H
#include <unistd.h>		/* mmap, lseek */
#endif

#ifdef HAVE_MMAP
#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>		/* mmap */
#endif
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>		/* lseek */
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef WIN32
#include <io.h>
#define open _open
#define read _read
#define lseek _lseek
#define close _close
#define ssize_t int
#define getpagesize() 4096
#endif

/****************************************************************
 * Structures
 ****************************************************************/

struct file_ref
{
  struct file_ref *next;
  struct file *file;
};

struct basename_ref
{
  struct basename_ref *next;
  char *basename;
};

struct file
{
  struct file *next;
  char *fullname;
  struct basename_ref *basename_refs;
};

/****************************************************************
 * globals
 ****************************************************************/

struct gm_hash *basename_to_file_refs;

/****************************************************************
 * mmap() emulation
 ****************************************************************/

#ifndef HAVE_MMAP

#ifndef MAP_PRIVATE
#define MAP_PRIVATE 1
#endif

#ifndef PROT_READ
#define PROT_READ 1
#endif

static void *
my_mmap (void *addr, size_t len, int prot, int flags, int fd, off_t off)
{
  off_t pos;
  char *buf;

  /* save current position */

  pos = lseek (fd, 0, SEEK_CUR);
  if (pos == -1)
    {
      perror ("seek failed");
      goto abort_with_nothing;
    }

  /* seek to start of the region to mmap. */

  {
    off_t result;
    result = lseek (fd, off, SEEK_SET);
    if (result == -1)
      {
	perror ("seek failed");
	goto abort_with_nothing;
      }
  }

  /* allocate a buffer for the data */

  buf = (void *) malloc (len + 1);
  if (!buf)
    {
      perror ("could not malloc for my_mmap()");
      goto abort_with_pos;
    }

  /* read the requested data into the buffer and null terminate */

  {
    ssize_t result;

    result = read (fd, buf, len);
    if (result == -1)
      {
	perror ("could not read in file.\n");
	goto abort_with_buf;
      }
    buf[len] = 0;
  }

  /* reset the file position. */

  pos = lseek (fd, pos, SEEK_SET);
  if (pos == -1)
    {
      perror ("could not reset pos in my_mmap()");
      goto abort_with_buf;
    }

  return buf;

abort_with_buf:
  free (buf);
abort_with_pos:
  lseek (fd, pos, SEEK_SET);
abort_with_nothing:
  return (void *) -1;
}


int
my_munmap (void *ptr, size_t len)
{
  free (ptr);
  return 0;
}

#else /* HAVE_MMAP */

static void *
my_mmap (void *addr, size_t len, int prot, int flags, int fd, off_t off)
{
  return mmap (addr, len, prot, flags, fd, off);
}

static int
my_munmap (void *ptr, size_t len)
{
  return munmap (ptr, len);
}

#endif /* HAVE_MMAP */

/****************************************************************
 * Utility functions
 ****************************************************************/

static char *
gm_strdup (char *s)
{
  char *b;

  b = (char *) malloc (strlen (s) + 1);
  if (!b)
    return 0;
  return strcpy (b, s);
}

static char *
gm_basename (char *fullname)
{
  char *b;

  for (b = fullname + strlen (fullname) - 1; b > fullname; b--)
    {
      if (*b == '/')
	return b + 1;
    }
  return fullname;
}

static char *
suffix (char *fullname)
{
  char *b;

  for (b = fullname + strlen (fullname) - 1; b > fullname; b--)
    {
      if (*b == '.')
	return b;
    }
  return fullname + strlen (fullname);	/* no suffix */
}

static char *
_index (const char *s, int c)
{
  while (*s)
    {
      if (*s == c)
	return (char *) s;
      s++;
    }
  return 0;
}

static char *
_rindex (const char *s, int c)
{
  const char *ret = 0;

  while (*s)
    {
      if (*s == c)
	ret = s;
      s++;
    }
  return (char *) ret;
}

/* This HACK assumes _fputs is used for only one file. */

static void
_fputs (const char *s, FILE * f)
{
  enum
  { LINE_LEN = 79 };
  static char buf[LINE_LEN + 1];
  static char *buf_pos = buf;
  char c;

  while (*s)
    {
      c = *s++;
      *buf_pos++ = c;
      if (c == '\n')
	{
	  *buf_pos = 0;
	  fputs (buf, f);
	  buf_pos = buf;
	}
      else if (buf_pos == &buf[LINE_LEN])
	{
	  char *r;

	  *buf_pos = 0;
	  r = _rindex (buf, ' ');
	  if (r)
	    {
	      *r = 0;
	      fputs (buf, f);
	      fputs ("\\\n", f);
	      *r = ' ';
	      buf_pos = buf;
	      do
		*buf_pos++ = *r++;
	      while (*r);
	    }
	  else
	    {
	      fputs (buf, f);
	      fputs ("\\\n", f);
	      buf_pos = buf;
	    }
	}
    }
}

/****************************************************************
 * Header output functions
 ****************************************************************/

static gm_status_t
output_headers1 (struct file *f, struct gm_hash *exclude)
{
  struct basename_ref *br;
  gm_status_t status;

  if (gm_hash_find (exclude, f->fullname))
    return GM_SUCCESS;

  _fputs (" ", stdout);
  _fputs (f->fullname, stdout);

  status = gm_hash_insert (exclude, f->fullname, f);
  if (status != GM_SUCCESS)
    {
      fprintf (stderr, "Could not insert file in hash table.\n");
      return status;
    }

  for (br = f->basename_refs; br; br = br->next)
    {
      struct file_ref *fr;

      for (fr = gm_hash_find (basename_to_file_refs, br->basename);
	   fr; fr = fr->next)
	{
	  status = output_headers1 (fr->file, exclude);
	  if (status != GM_SUCCESS)
	    return status;
	}
    }

  return GM_SUCCESS;
}

static gm_status_t
output_headers (struct file *f)
{
  struct gm_hash *exclude_hash;
  gm_status_t status;

  /* allocate resources, recursively output headers, and free resources */

  exclude_hash
    = gm_create_hash (gm_hash_compare_strings, gm_hash_hash_string,
		      0, 0, 0, 0);
  if (!exclude_hash)
    {
      fprintf (stderr, "Could not create hash table.\n");
      return GM_FAILURE;
    }
  status = output_headers1 (f, exclude_hash);
  if (status != GM_SUCCESS)
    goto abort_with_hash;
  gm_destroy_hash (exclude_hash);

  return GM_SUCCESS;

 abort_with_hash:
  gm_destroy_hash (exclude_hash);
  return status;
}

/****************************************************************
 * Scanning
 ****************************************************************/

static gm_status_t
notice_file_basename (struct file *f)
{
  char *b;
  struct file_ref *fr, *first;
  gm_status_t status;

  b = gm_basename (f->fullname);

  /* allocate storage for the file reference */

  fr = (struct file_ref *) calloc (1, sizeof (struct file_ref));
  if (!fr)
    goto abort_with_nothing;
  fr->file = f;

  /* add the file reference to the list of file references for the
     basename */

  first = gm_hash_find (basename_to_file_refs, b);
  if (first)
    {
      fr->next = first->next;
      first->next = fr;
    }
  else
    {
      status = gm_hash_insert (basename_to_file_refs, b, fr);
      if (status != GM_SUCCESS)
	goto abort_with_fr;
    }
  return GM_SUCCESS;

abort_with_fr:
  free (fr);
abort_with_nothing:
  return GM_FAILURE;
}

/* record the fact that file F includes a file named BASENAME. */

static gm_status_t
record_basename_ref (struct file *f, char *basename)
{
  struct basename_ref *fr;

  fr = (struct basename_ref *) malloc (sizeof (struct basename_ref));
  if (!fr)
    goto abort_with_nothing;
  fr->basename = gm_strdup (basename);
  if (!fr->basename)
    goto abort_with_fr;
  fr->next = f->basename_refs;
  f->basename_refs = fr;
  return GM_SUCCESS;

abort_with_fr:
  free (fr);
abort_with_nothing:
  return GM_FAILURE;
}

/* Build a structure describing the file and all of the local includes
   that it references. */

static struct file *
scan (char *fullname)
{
  int fd;
  void *mm;
  char *cp, c, *tmp, *mark;
  struct file *f;
  long length;
  int err;

  /****************
   * Create the file record
   ****************/

  /* alloc */

  f = (struct file *) calloc (1, sizeof (struct file));
  if (!f)
    {
      perror ("could not malloc space for file");
      goto abort_with_nothing;
    }

  /* record the full relative path to the file */

  while (strncmp (fullname, "./", 2) == 0)
    fullname += 2;
  f->fullname = gm_strdup (fullname);
  if (!f->fullname)
    goto abort_with_f;

  /****************
   * record that basename maps to the file
   ****************/

  if (notice_file_basename (f) != GM_SUCCESS)
    goto abort_with_fullname;

  /****************
   * Scan the file for local include file references.
   ****************/

  fd = open (fullname, O_RDONLY);
  if (fd == -1)
    {
      perror ("could not open file");
      goto abort_with_fullname;
    }

  /* determine the length of the file. */

  length = lseek (fd, 0, SEEK_END);
  if (length == -1)
    {
      perror ("could not determine file length");
      goto abort_with_open;
    }

  /* round length up to a page boundary. */

  length = (length + getpagesize () - 1) & ~(getpagesize () - 1);
  if (!length)
    goto munmap_done;

  /* map the file. */

  mm = my_mmap (0, length, PROT_READ, MAP_PRIVATE, fd, 0);
  if (mm == 0 || mm == (void *) -1)
    {
      perror ("could not map file");
      goto abort_with_open;
    }

#define NEXT(cnt) do							\
  {									\
    cp += cnt;								\
    if ((char *) cp - (char *) mm >= length)				\
      goto scan_done;							\
    c = *cp;								\
    if (!c)								\
      goto scan_done;							\
  } while (0)

  cp = mm;
  c = *cp;
  while (1)
    {
      /* In this loop outside the NEXT() macro, cp points to the
         current character, and c is a copy of that character. */

      /* match "#"' */

      if (c != '#')
	goto next_line;
      NEXT (1);

      /* eat whitespace */

      while (c == ' ' || c == '\t')
	NEXT (1);

      /* match "include" */

      if (strncmp (cp, "include", 7) != 0)
	goto next_line;
      NEXT (7);

      /* match whitespace */

      if (c != ' ' && c != '\t')
	goto next_line;

      /* eat extra whitespace */

      while (c == ' ' || c == '\t')
	NEXT (1);

      /* match "\"" */

      if (c != '"' && c != '<')
	goto next_line;
      NEXT (1);

      /* eat and record filename.  (eat first to catch EOF) */

      mark = cp;
      while (c != '"' && c != '>')
	NEXT (1);
      tmp = (char *) malloc (cp - mark + 1);
      if (!tmp)
	goto abort_with_my_mmap;
      gm_bcopy (mark, tmp, cp - mark);
      tmp[cp - mark] = 0;
      record_basename_ref (f, gm_basename (tmp));
      free (tmp);

      /* find start of next line. */

    next_line:
      while (c != '\n')
	NEXT (1);
      NEXT (1);
    }
scan_done:

  err = my_munmap (mm, length);
  if (err)
    {
      perror ("could not munmap");
      goto abort_with_open;
    }
munmap_done:
  err = close (fd);
  if (err)
    {
      perror ("could not close");
      goto abort_with_fullname;
    }
  return f;

abort_with_my_mmap:
  my_munmap (mm, length);
abort_with_open:
  close (fd);
abort_with_fullname:
  free (f->fullname);
abort_with_f:
  free (f);
abort_with_nothing:
  return 0;
}

/****************************************************************
 * Extra files not included in the source release
 ****************************************************************/

static struct file *
notice_generated_file (char *fullname)
{
  struct file *f;

  f = (struct file *) calloc (1, sizeof (struct file));
  if (!f)
    {
      perror ("could not malloc space for file");
      goto abort_with_nothing;
    }

  /* record the full relative_path to the file */

  while (strncmp (fullname, "./", 2) == 0)
    fullname += 2;
  f->fullname = gm_strdup (fullname);
  if (!f->fullname)
    {
      perror ("could not strdup");
      goto abort_with_f;
    }

  /* notice the basename */

  if (notice_file_basename (f) != GM_SUCCESS)
    {
      perror ("could not notice file basename");
      goto abort_with_fullname;
    }

  return f;

abort_with_fullname:
  free (f->fullname);
abort_with_f:
  free (f);
abort_with_nothing:
  return 0;
}

static gm_status_t
notice_generated_files (void)
{
  char *files[] = { "include/gm_auto_config.h", 0 }, **f = files;

  do
    {
      if (!notice_generated_file (*f))
	return GM_FAILURE;
    }
  while (*++f);

  return GM_SUCCESS;
}

/****************************************************************
 * main program
 ****************************************************************/

int
main (int argc, char *argv[])
{
  struct file *file, *scanned_files = 0;
  char fullname[256];
  gm_status_t status;

  status = gm_init ();
  if (status != GM_SUCCESS)
    return 1;

  /* initialize globals */

  basename_to_file_refs = gm_create_hash (gm_hash_compare_strings,
					  gm_hash_hash_string, 0, 0, 0, 0);
  if (!basename_to_file_refs)
    {
      fprintf (stderr,
	       "Could not create basename_to_file_refs hash table.\n");
      return 1;
    }

  /* scan each file and record its includes */

  while (fgets (fullname, 256, stdin))
    {
      int len;

      /* ignore empty lines */

      if (fullname[0] == 0 || fullname[0] == '\n')
	continue;

      /* strip trailing newline, if any */

      len = strlen (fullname);
      if (fullname[len - 1] == '\n')
	fullname[len - 1] = 0;

      /* scan the file for include files */

      file = scan (fullname);
      if (!file)
	return 1;
      file->next = scanned_files;
      scanned_files = file;
    }

  /* also notice the files that may need to be generated. */

  if (notice_generated_files () != GM_SUCCESS)
    return 1;

  /* print the dependency list for each .c* file */

  for (file = scanned_files; file; file = file->next)
    {
      char *f, *s, save;

      f = file->fullname;
      s = suffix (f);

      /* output only .c* file dependencies */

      if (strncmp (s, ".c", 2) != 0)
	continue;

      /* output targets and trivial .c* dependency. */

      save = *s;
      *s = 0;

      /* support kernel targets in libgm/ and drivers/ directories
         only. */

      if (!strncmp (f, "mcp/", 4)
	  || !strncmp (f, "libgm/", 6))
	{
	  _fputs (f, stdout), _fputs (".32b_l4_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l5_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l6_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l7_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l8_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l9_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l4_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l4_4k_compact_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l5_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l6_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l7_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l8_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l9_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l4_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l4_8k_compact_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l5_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l6_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l7_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l8_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l9_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l4_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l5_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l6_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l7_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l8_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".32b_l9_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l4_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l5_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l6_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l7_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l8_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l9_0k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l4_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l4_4k_compact_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l5_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l6_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l7_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l8_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l9_4k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l4_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l4_8k_compact_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l5_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l6_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l7_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l8_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l9_8k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l4_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l5_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l6_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l7_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l8_16k_o ", stdout);
	  _fputs (f, stdout), _fputs (".64b_l9_16k_o ", stdout);
	  
	  _fputs (f, stdout), _fputs ("_k.o ", stdout);
	  _fputs (f, stdout), _fputs ("_k.obj ", stdout);
	  
	  _fputs (f, stdout), _fputs (".o ", stdout);
	  _fputs (f, stdout), _fputs (".obj :", stdout);
	}
      else if (!strncmp (f, "drivers/", 6))
	{
	  _fputs (f, stdout), _fputs ("_k.o ", stdout);
	  _fputs (f, stdout), _fputs ("_k.obj :", stdout);
	}
      else
	{
	  _fputs (f, stdout), _fputs (".o ", stdout);
	  _fputs (f, stdout), _fputs (".obj :", stdout);
	}

      *s = save;
      
      /* create a timestamp to represent the dependencies by proxy.
	 This dramatically reduces the memory of Gnu make when it
	 reads the output. */

      _fputs (" ", stdout);
      _fputs (f, stdout);
      _fputs (".ts\n", stdout);
      _fputs (f, stdout);
      _fputs (".ts :", stdout);

      /* output header dependencies */

      if (output_headers (file) != GM_SUCCESS)
	{
	  fprintf (stderr, "failed to output headers.\n");
	  exit (1);
	}

      /* output a line to touch the timestamp. */
      
      _fputs ("\n\t@ echo This file is a timestamp. > ", stdout);
      _fputs (f, stdout);
      _fputs (".ts\n", stdout);
    }

  /* finalize globals */

  gm_destroy_hash (basename_to_file_refs);

  gm_finalize ();

  return 0;
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
