#ifdef __STDC__
#if GM_OS_VXWORKS
#include "vxWorks.h"
#endif  /* GM_OS_VXWOKRS */
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "stats.h"
#include <float.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

/* This implementation suffers from roundoff error, but is very
   simple and correct assuming negligible roundoff error. */

struct stats
{
  unsigned int count;
  struct stat *stats;
};

struct stat
{
  struct stat *next;
  double value;
};

static struct stat *free_stats;

static
struct stat *
_stat_alloc (void)
{
  struct stat *p;
  
  p = free_stats;
  if (p)
    {
      free_stats = p->next;
    }
  else
    {
      do
	p = (struct stat *) malloc (sizeof (struct stat));
      while (!p);
    }
  return p;
}

static
void
_stat_free (struct stat *p)
{
  p->next = free_stats;
  free_stats = p;
}

/****************
 * clear (reset) a stats structure for reuse.
 ****************/

void
stats_clear (struct stats *s)
{
  struct stat *p;
  
  p = s->stats;
  while (p)
    {
      s->stats = p->next;
      _stat_free (p);
      p = s->stats;
    }

  s->count = 0;
}

struct stats *
stats_alloc()
{
  struct stats *s;

  s = (struct stats *) calloc (1, sizeof (struct stats));
  if (s)
    stats_clear (s);
  return s;
}

void
stats_free (struct stats *s)
{
  struct stat *next, *p;
  
  stats_clear (s);
  
  p = free_stats;
  free_stats = 0;
  while (p)
    {
      next = p->next;
      free (p);
      p = next;
    }
  free (s);
}

void
stats_append (struct stats *s, double value)
{
  struct stat *p;

  p = _stat_alloc ();
  p->value = value;
  p->next = s->stats;
  s->stats = p;
  s->count++;
}

/****************
 * mean
 ****************/

/* compute the mean, minimizing roundoff error */

static struct stat *_mean_pos;

static double
_mean (unsigned int count)
{
  if (count == 1)
    {
      double ret;

      ret = _mean_pos->value;
      _mean_pos = _mean_pos->next;
      return ret;
    }
  else
    {
      return _mean (count/2) + _mean (count - count/2);
    }
}

double
stats_get_mean (struct stats *s)
{
  if (0 == s->count)
    return 0.0;
  _mean_pos = s->stats;
  return _mean (s->count) / (double) s->count;
}

double
stats_get_max (struct stats *s)
{
  struct stat *pos;
  double max;

  pos = s->stats;
  if (!pos)
    return 0.0;
  max = pos->value;
  while ((pos = pos->next) != 0)
    {
      if (pos->value > max)
	max = pos->value;
    }
  return max;
}

double
stats_get_min (struct stats *s)
{
  struct stat *pos;
  double min;

  pos = s->stats;
  if (!pos)
    return 0.0;
  min = pos->value;
  while ((pos = pos->next) != 0)
    {
      if (pos->value < min)
	min = pos->value;
    }
  return min;
}

/****************
 * printing
 ****************/

void
stats_output_points (FILE *f, const char *mytemplate, struct stats *s)
{
  struct stat *pos;

  for (pos = s->stats; pos; pos = pos->next)
    {
      fprintf (f, mytemplate, pos->value);
    }
}

/****************
 * stddev
 ****************/

static struct stat *_sum_diff_squared_pos;

static double
_sum_diff_squared (unsigned int count, double mean)
{
  if (count == 1)
    {
      double ret;

      ret = _sum_diff_squared_pos->value - mean;
      ret *= ret;
      _sum_diff_squared_pos = _sum_diff_squared_pos->next;
      return ret;
    }
  else
    {
      return (_sum_diff_squared (count/2, mean)
	      + _sum_diff_squared (count - count/2, mean));
    }
}

static struct stat *_sum_diff_pos;

static double
_sum_diff (unsigned int count, double mean)
{
  if (count == 1)
    {
      double ret;

      ret = _sum_diff_pos->value - mean;
      _sum_diff_pos = _sum_diff_pos->next;
      return ret;
    }
  else
    {
      return _sum_diff (count/2, mean) + _sum_diff (count - count/2, mean);
    }
}

double
stats_get_standard_deviation (struct stats *s)
{
  double N, mean, variance, sum_diff;

  if (s->count == 1)
    return 0.0;
  N = (double) s->count;
  if (N < 1.0)
    return HUGE_VAL;
  mean = stats_get_mean (s);
  
  _sum_diff_pos = s->stats;
  sum_diff = _sum_diff (s->count, mean);
  _sum_diff_squared_pos = s->stats;
  variance = 1 / (N - 1) * (_sum_diff_squared (s->count, mean)
			    - (sum_diff * sum_diff / N));
  if (variance < 0.0)
    return 0.0;
  return sqrt (variance);
}

unsigned int
stats_get_count (struct stats *s)
{
  return s->count;
}

void
stats_scale (struct stats *s, double scale)
{
  struct stat *pos;

  for (pos = s->stats; pos; pos = pos->next)
    {
      pos->value *= scale;
    }
}

#if GM_ENABLE_STATS_SOCKET
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#endif /* GM_ENABLE_STATS_SOCKET */

int socket_open (char*hostname, int port)
{
#if !GM_ENABLE_STATS_SOCKET
  return 0;
#else
  int fd;
  struct hostent *he;
  struct sockaddr_in sin;
  
  memset (&sin, 0, sizeof (sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons (port);

  if (he = gethostbyname (hostname))
    memcpy ((char*)&sin.sin_addr, he->h_addr, he->h_length);
  else if ((sin.sin_addr.s_addr = inet_addr (hostname)) == -1)
    {
      printf ("can't get IP address of %s\n", hostname);
      return 0;
    }
  
  if ((fd = socket (PF_INET, SOCK_STREAM, 0)) < 0)
    {
      perror ("socket");
      return 0;
    }
  
  if (connect (fd, (struct sockaddr*) &sin, sizeof (sin)) < 0)
    {
      perror ("connect");
      return 0;
    }	  
  return fd;
#endif /* GM_ENABLE_STATS_SOCKET */
}

int socket_printf (int fd, char*fmt, ...)
{
#if !GM_ENABLE_STATS_SOCKET
  return 0;
#else  /* GM_ENABLE_STATS_SOCKET */
  char buffer[1000];
  va_list args;
  int ret;
  
  *buffer = 0;
  
  va_start (args, fmt);
  ret = vsprintf (buffer, fmt, args);
  va_end (args);

  write (fd, buffer, strlen (buffer) + 1);

  return ret;
#endif /* GM_ENABLE_STATS_SOCKET */
}

void socket_stats_output_points (int fd, const char *mytemplate,
				 struct stats *s)
{
#if GM_ENABLE_STATS_SOCKET
  struct stat *pos;

  for (pos = s->stats; pos; pos = pos->next)
    {
      socket_printf (fd, mytemplate, pos->value);
    }
#endif /* GM_ENABLE_STATS_SOCKET */
}

void socket_close (int fd)
{
#if GM_ENABLE_STATS_SOCKET
  close (fs);
#endif /* GM_ENABLE_STATS_SOCKET */
}

