/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#include <math.h>

#include "gm.h"
#include "gm_types.h"

#include "logp_main.h"

#ifdef VXWORKS
#include <time.h>
#include <sys/times.h>
#else
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif

#if 0
#define RTT_out /* creates a file and outputs all RTT values for each length */
#define TRACK
#define SUM_FILE /* file with rtt values for each size */ 
#endif

/* test parameters */

#define TEST_PRIORITY	GM_LOW_PRIORITY

#ifdef WIN32
#define MIN_TEST_SIZE	16
#define MAX_TEST_SIZE	16
#define MIN_TEST_LEN (16)
/* #define MAX_TEST_LEN(size) (gm_max_length_for_size (size)) */
#define MAX_TEST_LEN(size) 16384
#define LEN_INC 64
/* #define LEN_INC 64 */
#else

#ifndef MAX
#define MAX(a,b) (a<b) ? b:a;
#endif

#ifndef MIN
#define MIN(a,b) (a<b) ? a:b;
#endif

#define MIN_TEST_SIZE	12
#define MAX_TEST_SIZE	12

#define MIN_TEST_LEN 16

#define LEN_INC 8
#define MAX_TEST_LEN(size) (gm_max_length_for_size (size))

/*
   #define MIN_TEST_SIZE        5
   #define MAX_TEST_SIZE        14
   #define MIN_TEST_LEN(size) 16  
   #define LEN_INC 8
   #define MAX_TEST_LEN(size) (gm_max_length_for_size (size))
 */
#endif /* defined WIN32 */

extern int my_delay( int d );
extern void all_time_1us(double *base1, double *slope1);


int quiet = 0;
gm_u32_t total_sent = 0, total_recvd = 0;
int rand_seed = 42;

enum test_message_types
{
  test_bogus_type = 14,
  test_ping_type,
  test_forward_type
};

typedef struct test_message
{
  gm_u32_n_t type;

  gm_u8_n_t size;
  gm_u8_n_t sender;
  gm_u16_n_t len;

  char payload[1];

}
test_message_t;

static int
payload_len (int len)
{
  return len - sizeof (gm_u32_t) - sizeof (gm_u32_t);
}

/* Define the type of message we will
   be sending and receiving. */
typedef struct chainable_test_message
{
  /* Use the 8 bytes at the head of the
     message for user data that will not
     be sent over the network. */
  struct chainable_test_message *next;
  int size;
#if GM_SIZEOF_VOID_P == 8
  int pad;
#endif
  test_message_t message;

}
chainable_test_message_t;


#define HEADER_SIZE ((char*)&((struct chainable_test_message*)(0))->message-(char*)0)

chainable_test_message_t *free_send_messages[32];

static chainable_test_message_t *
ctp_parent (test_message_t * p)
{
  return (chainable_test_message_t *) ((char *) p - HEADER_SIZE);
}

/*
 * add a buffer to the (user managed) list of free send
 * buffers
 */
static void
free_send_message (test_message_t * p)
{
  chainable_test_message_t *ctp = ctp_parent (p);
  gm_u8_t p__size;
  
  /* printf ("ctp->size = %d, p->size = %d\n", ctp->size, p->size); */

  p__size = gm_ntoh_u8 (p->size);
  gm_assert (ctp->size == p__size);

  ctp->next = free_send_messages[p__size];
  free_send_messages[p__size] = ctp;
}

/*
 * quiet is: 0 for verbose, 1 for quiet, and 2 for silent (except for errors)
 */
#define TEST_PRINT(level, args)						\
do { if (level>=quiet) printf args; } while (0)


/*
 * better (but slow) buffer fill and verify routines; these limit
 * throughput to a few MB/s
 */

int rand_seed;

static unsigned
my_rand (void)
{
  if ((int) rand_seed < 0)
    return rand_seed = rand_seed << 1 ^ 0x04C11DB7;
  return rand_seed = rand_seed << 1;
}

static void
my_srand (int seed)
{
  rand_seed = seed ? seed : -1;	/* seed must not be 0 */
}

/* Generate a random number in the */
/* range [0,a). */
static unsigned
my_rand_mod (unsigned a)
{
  unsigned ret;
  unsigned mask;

  mask = (1 << gm_log2_roundup (a + 1)) - 1;
  do
    ret = my_rand () & mask;
  while (ret > a);		/* < 1 retry needed, on average */
  return ret - 1;
}

static void
fill_buffer (char *buf, int len, char seed)
{
  int i;

  my_srand (seed);

  buf[0] = seed;

  for (i = 1; i < len; i++)
    {
      buf[i] = my_rand_mod (256);
    }
}

static int
verify_buffer (char *buf, int len)
{
  int i;
  char val;
  char seed = buf[0];
  my_srand (seed);

  for (i = 1; i < len; i++)
    {
      val = my_rand_mod (256);
      if (buf[i] != val)
	{
	  TEST_PRINT (2, ("data error; at pos %d of %d expected 0x%x "
			  "and got 0x%x\n"
			  "            memory location %p\n",
			  i, len, val&0xff, buf[i]&0xff, &buf[i]));
	  return 1;
	}
    }

  return 0;
}


static int got_alarm = 0;
static gm_alarm_t my_alarm;
static
void
set_got_alarm (void *ignored)
{
  got_alarm = 1;
}

unsigned int this_node_id;
struct gm_port *port;
int board = 0;

static
void
test_init (int port_id, enum gm_priority test_priority, int min_test_size,
	   int max_test_size)
{
  int i, size;
  gm_status_t status;

  /*
   * initialize GM
   */

  status = gm_open (&port, board, port_id, "all-size test",
		    GM_API_VERSION_1_0);
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not open GM port", status);
      gm_exit (status);
    }
  TEST_PRINT (0, ("GM is initialized.\n"));

  gm_initialize_alarm (&my_alarm);

  status = gm_set_acceptable_sizes (port, test_priority,
                                    ((1 << (max_test_size + 1)) - 1)
                                    ^ ((1 << min_test_size) - 1));
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not set GM acceptable sizes", status);
      gm_exit (status);
    }
      
  /* Set the remote node ID to this node ID */
  
  status = gm_get_node_id (port, &this_node_id);
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not determine node id", status);
      gm_exit (status);
    }
  
  /* let GM manage the send tokens */
  gm_free_send_tokens (port, GM_LOW_PRIORITY, gm_num_send_tokens(port));
  /*  gm_free_send_tokens (port, GM_HIGH_PRIORITY, gm_num_send_tokens(port) / 2);*/

  /* setup recv tokens */
  for (size = min_test_size; size <= max_test_size; size++)
    {

      /* allocate as many as we want - GM is not keeping track
         of them */
#ifdef WIN32
      int send_buffers = 3;
#else
      /*      int send_buffers = 9;*/
      int send_buffers = 60;
#endif

      /* total over all sizes must be <= GM_NUM_RECV_TOKENS */
#ifdef WIN32
      int recv_tokens_of_size = 3;
#else
      /*      int recv_tokens_of_size = 9;*/
      int recv_tokens_of_size = 40;
#endif

      /* allocate receive buffers */
      for (i = 0; i < recv_tokens_of_size; i++)
	{
	  chainable_test_message_t *ctp;

	  ctp = (chainable_test_message_t *)
	    gm_dma_calloc (port, 1, gm_max_length_for_size (size) + HEADER_SIZE);

	  if (!ctp)
	    {
	      fprintf (stderr, "Could not allocate dmaable memory.\n");
	      gm_exit (GM_OUT_OF_MEMORY);
	    }

	  ctp->message.size = gm_hton_u8 (size);

#if GM_DEBUG_BUFFERS
	  gm_register_buf ((void *) &ctp->message, size);
#endif

	  /* Make sure message is aligned. */
	  gm_always_assert (GM_DMA_ALIGNED (ctp));

	  gm_provide_receive_buffer (port, &ctp->message, size, test_priority);
	}

      /* allocate send buffers */
      for (i = 0; i < send_buffers; i++)
	{
	  chainable_test_message_t *ctp;

	  ctp = ((chainable_test_message_t *)
		 gm_dma_malloc (port, gm_max_length_for_size (size) + HEADER_SIZE));
	  if (!ctp)
	    {
	      fprintf (stderr, "Could not allocate DMAable memory "
		       "for a send buffer.\n");
	      gm_exit (GM_OUT_OF_MEMORY);
	    }
	  gm_always_assert (GM_DMA_ALIGNED (ctp));

#if GM_DEBUG_BUFFERS
	  gm_register_buf ((void *) &ctp->message, size);
#endif

	  fill_buffer ((char *)ctp, gm_max_length_for_size (size)+ HEADER_SIZE, 0);

	  ctp->next = 0;
	  ctp->size = size;
	  ctp->message.size = gm_hton_u8 (size);

	  free_send_message (&ctp->message);
	}
    }
}

static
void
die (gm_status_t exit_value)
{
  TEST_PRINT (1, ("Total sent = %d.  Total received = %d.\n",
		  total_sent, total_recvd));

  gm_exit (exit_value);
}

static void
handle_sigint (int i)
{
  TEST_PRINT (1, ("Caught SIGINT.\n"));
  die (GM_INTERRUPTED);
}

#define MAX_TRACK 50000
static char racks[MAX_TRACK];
char *tracks;

typedef struct send_t{
  int sends;
  double utime;
  double ut_per_send;
}send_t;

#define BURSTS 27

struct send_t send_time[BURSTS]; 
int send_time_count = 0;

double Os_time = 0;
double gap = 0;
int   gap_count = 0;
double Or_time = 0;
double transition = 0;
double latency = 0;
double avg_RTT = 0;

#define BIG_COUNT 120
#define MAX_DELTA 64
#define BIG_MAX_DELTA 128
#define TRANSITION_DELTA 32 /* delta less than this goes to gap  for big M */

double max_delta = MAX_DELTA;
unsigned int res_file = 0;
unsigned int dat_file = 0;

static
void
print_results(FILE* ftrack,int delta)
{
  int i;

  /* start at i = 1 to skip the first "warm-up" run */
  for (i = 1; i < send_time_count; i++)
  {
     send_time[i].ut_per_send = 
      (send_time[i].utime)/send_time[i].sends;
      /* (send_time[i].utime/1E6)/send_time[i].sends*1E6; */

#if 0
     printf("%d sends, in %.3f usec => %.3f usec/send, with delay %d usec\n", 
	   send_time[i].sends ,
	   (send_time[i].utime),
	   send_time[i].ut_per_send, delta);
#endif

     if (dat_file){
       fprintf(ftrack, "%d, %.3f\n",
	       send_time[i].sends,
	       send_time[i].ut_per_send);
     }
  }

  if (delta == 0){ /* calculate Os */
    i = 1;
    while (send_time[i].sends < GM_NUM_SEND_TOKENS){
      Os_time += send_time[i].ut_per_send;
      i++;
    }
    Os_time = Os_time/i;
  }
  else if (delta == max_delta){
    Or_time = send_time[send_time_count-1].ut_per_send - delta - Os_time;
  }

  for (i = 1; i < send_time_count; i++){
    if ((send_time[i].sends > BIG_COUNT) && (delta < TRANSITION_DELTA)){
      gap += send_time[i].ut_per_send;
      gap_count ++; 
    }
  }
  send_time_count = 0;
}

FILE *allRTT = NULL;
FILE *allLat = NULL;
FILE *allgap = NULL;
FILE *lenRTT = NULL;
FILE *lenLat = NULL;
FILE *lengap = NULL;
FILE *lenBW = NULL;

static
void
print_results0(int len)
{
  int i;

  /* start at i = 1 to skip the first "warm-up" run */
  for (i = 1; i < send_time_count; i++)
  {
     send_time[i].ut_per_send = 
      (send_time[i].utime)/send_time[i].sends;
      /* (send_time[i].utime/1E6)/send_time[i].sends*1E6;*/
     fprintf(lenBW,"%d %f \n",len,
	     ((double)len/(send_time[i].ut_per_send)));

  }
  send_time_count = 0;
}

static
void
print_tracks(FILE* trackfile)
{
#ifdef TRACK
  char *ptr;
  char *ptr_end;

  ptr = &racks[0];   
  ptr_end = &racks[MAX_TRACK];

  while ((ptr <= ptr_end) && (ptr <= tracks)){
    ptr+= fprintf(trackfile,"%s\n", ptr);
  }
  tracks = &racks[0]; /* re-init */
#endif
}

static
void 
calc_parameters(int size, int len, FILE* outf)
{
  if (!avg_RTT)
    avg_RTT = 70;

  latency = (avg_RTT/2) - Os_time - Or_time;

  if (res_file){
     fprintf(outf,"\nLogP PARAMETERS\n");
     fprintf(outf," Os is %f usec\n", (Os_time));
     fprintf(outf," Or is %f usec\n", (Or_time));
     if (gap_count >0){
        fprintf(outf," gap is %f usec\n",(gap/(gap_count)));
	fprintf(outf," message length is %d bytes\n", len);
	fprintf(outf," expected peak BW is %f MBytes/sec\n", 
		((double)len/((gap/(gap_count)))));
     }
     fprintf(outf," RTT is %f\n", avg_RTT);
     fprintf(outf," Latency is %f usec\n", (latency));
     fprintf(outf," curve transition should occur at %f messages\n", (avg_RTT/Os_time));
  }

#ifdef ALL_SIZE
  fprintf(allRTT,"%d %f\n", size, avg_RTT);
  fprintf(allLat,"%d %f \n", size,(latency));
  fprintf(allgap,"%d %f \n", size,(gap/(gap_count*1E6))*1E6);
#endif

  fprintf(lenRTT,"%d %f\n", len, avg_RTT);
  fprintf(lenLat,"%d %f \n", len,(latency));
  fprintf(lengap,"%d %f \n", len,(gap/(gap_count*1E6))*1E6);
  fprintf(lenBW,"%d %f \n",len,((double)len/((gap/(gap_count*1E6))*1E6)));

  gap = 0;
  gap_count = 0;
}

static
void
usage (void)
{
  TEST_PRINT (2, ("\nusage: logp_test [-b|-u|-o] [-s] [-r] [-p port]\n"));
  TEST_PRINT (2, ("               [-t] [-f] [-c count] [-v]\n"));
  TEST_PRINT (2, ("               [-B board] [-l length_increment]\n"));
  TEST_PRINT (2, ("               [-h remote_host] [-q [-q]]\n"));
  TEST_PRINT (2, ("               [-e estimated_RTT] \n\n"));

  TEST_PRINT (2, ("-b       RTT test\n"));
  TEST_PRINT (2, ("-u       send overhead (latency) test\n"));
  TEST_PRINT (2, ("-o       RTT and overlapping sends and receives test (default)\n"));
  TEST_PRINT (2, ("-s       send\n"));
  TEST_PRINT (2, ("-r       receive\n"));
  TEST_PRINT (2, ("-h host  machine with which to exchange messages\n"));
  TEST_PRINT (2, ("-B #     board to use in this machine\n"));
  TEST_PRINT (2, ("-p #     specifies port number (default = 2)\n"));
  TEST_PRINT (2, ("-l #     specifies starting length increment (default = 1)\n"));
  TEST_PRINT (2, ("-v       verify contents and order of all messages "
		  "(slow)\n"));
  TEST_PRINT (2, ("-c #     specifies number of messages for RTT test "
		  "(default = 100)\n"));
  TEST_PRINT (2, ("-e #     estimated RTT\n"));
  TEST_PRINT (2, ("-q       quiet (-q -q = very quiet)\n"));
  TEST_PRINT (2, ("-t       creates LogP output data files - many per len\n"));
  TEST_PRINT (2, ("-f       creates LogP output result files - 1 per len\n"));
  TEST_PRINT (2, ("--min-size=#   minimum size to test\n"));
  TEST_PRINT (2, ("--max-size=#   maximum size to test\n"));
  TEST_PRINT (2, ("--size=# min and max size to test - also sets length\n"));
  TEST_PRINT (2, ("--min-len=#    minimum length to test\n"));
  TEST_PRINT (2, ("\n"));
  TEST_PRINT (2, ("By default, packets are sent to the local host.\n"));
  TEST_PRINT (2, ("GM sizes must be from 1 to 31.\n"));
  TEST_PRINT (2, ("\n"));
  die (GM_INVALID_PARAMETER);
}

double seconds, seconds_adj;
double RTT_cum = 0;
unsigned int check = 0;
unsigned int count_per_len = 1;
unsigned int message_count = 100;
unsigned int i_recv = 0, i_send = 0;
unsigned int bid = 2;
unsigned int len_inc = 1; 
unsigned int port_id = 2;
unsigned int other_port_id = 2;
unsigned int min_len = 0;
unsigned int min_size = 0;
unsigned int max_size = 0;
unsigned int the_size = 0;
unsigned int est_RTT = 0;
char *remote_host_name =0;
unsigned int remote_node_id = 0;

static
void
parse_args (int argc, char *argv[])
{
  int i;

  for (i = 1; i < argc; i++)
    {
      if (strcmp (argv[i], "-q") == 0)
	{
	  if (++quiet > 2)
	    quiet = 2;
	  continue;
	}
      if (strcmp (argv[i], "-b") == 0)
	{
	  bid = 1;
	  continue;
	}
      if (strcmp (argv[i], "-u") == 0)
	{
	  bid = 0;
	  continue;
	}
      if (strcmp (argv[i], "-o") == 0)
	{
	  bid = 2;
	  continue;
	}
      if (strcmp (argv[i], "-s") == 0)
	{
	  i_send = 1;
	  continue;
	}
      if (strcmp (argv[i], "-r") == 0)
	{
	  i_recv = 1;
	  continue;
	}
      if (strcmp (argv[i], "-v") == 0)
	{
	  check = 1;
	  continue;
	}
      if (strcmp (argv[i], "-t") == 0)
	{
	  dat_file = 1;
	  continue;
	}
      if (strcmp (argv[i], "-f") == 0)
	{
	  res_file = 1;
	  continue;
	}
      if (strcmp (argv[i], "-h") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      TEST_PRINT (2, ("Host name expected after '-h'.\n"));
	      usage ();
	    }
	  remote_host_name = argv[++i];
	  continue;
	}
      if (strcmp (argv[i], "-p") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      TEST_PRINT (2, ("Port number expected after '-p'.\n"));
	      usage ();
	    }
	  port_id = atoi (argv[++i]);
	  if (port_id < 2 || port_id > gm_num_ports (port))
	    {
	      TEST_PRINT (2, ("bad port number: %d\n", port_id));
	      usage ();
	    }
	  continue;
	}
      if (strcmp (argv[i], "-B") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      TEST_PRINT (2, ("Board number expected after '-B'.\n"));
	      usage ();
	    }
	  board = atoi (argv[++i]);
	  if (board < 0)
	    {
	      TEST_PRINT (2, ("bad board number: %d\n", board));
	      usage ();
	    }
	  continue;
	}
      if (strcmp (argv[i], "-o") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      TEST_PRINT (2, ("Port number expected after '-o'.\n"));
	      usage ();
	    }
	  other_port_id = atoi (argv[++i]);
	  if (other_port_id < 2 || port_id > gm_num_ports (port))
	    {
	      TEST_PRINT (2, ("bad port number: %d\n", port_id));
	      usage ();
	    }
	  continue;
	}
     if (strcmp (argv[i], "-l") == 0)
       {
          if (i + 1 >= argc)
            {
              TEST_PRINT (2, ("length_increment expected after '-l'.\n"));
              usage ();
            }
          len_inc = atoi (argv[++i]);
          if ((len_inc <= 0) || (len_inc % 8))
            {
              TEST_PRINT (2, ("bad length_increment: %d\n", len_inc));
              usage ();
            }
          continue;
       }
      if (strcmp (argv[i], "-e") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      TEST_PRINT (2, ("Time expected after '-e'.\n"));
	      usage ();
	    }
	  est_RTT = atoi (argv[++i]);
	  if (est_RTT < 1)
	    {
	      TEST_PRINT (2, ("bad time: %d\n", est_RTT));
	      usage ();
	    }
	  continue;
	}
      if (strcmp (argv[i], "-c") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      TEST_PRINT (2, ("Count expected after '-t'.\n"));
	      usage ();
	    }
	  message_count = atoi (argv[++i]);
	  if (message_count < 1)
	    {
	      TEST_PRINT (2, ("bad count: %d\n", message_count));
	      usage ();
	    }
	  continue;
	}

#define check_opt(prefix, var, test)                                    \
      do                                                                \
        {                                                               \
          if (!strncmp (argv[i], prefix, strlen (prefix)))              \
            {                                                           \
              var = atoi (argv[i]+strlen (prefix));                     \
              if (!(test))                                              \
                {                                                       \
                  TEST_PRINT (2, ("bad value for option \"%s\"\n", argv[i]));                   usage ();                                             \
                }                                                       \
              goto success;                                             \
            }                                                           \
        }                                                               \
      while (0) 

			
      check_opt ("--min-size=", min_size, (min_size
					       >= gm_min_message_size (port)));
      check_opt ("--max-size=", max_size, max_size < 32);
      check_opt ("--min-len=", min_len, min_len >= sizeof(test_message_t) && !(min_len % 8));
      check_opt ("--size=", the_size, the_size < 32);


      TEST_PRINT (2, ("unknown arg: '%s'\n", argv[i]));
      usage ();
    success:
				/* Semicolon to keep Microsoft
                                   compiler happy. */
      ;
    }

  if (min_size > max_size) {
    TEST_PRINT(2, ("min_size(%d) > max_size(%d) - you need to adjust one or the other\n",min_size,max_size));
    usage ();
  }


  /*
   * let the user know what the parameters are
   */
  TEST_PRINT (1, ("Using port %d.\n", port_id));

  if (i_send)
    TEST_PRINT (1, ("I'm the sender.\n"));

  if (i_recv)
    TEST_PRINT (1, ("I'm the receiver.\n"));

  if (!i_send && !i_recv)
    {
      TEST_PRINT (1, ("Send or receive not specified; assuming loopback.\n"));
      i_send = i_recv = 1;
    }

  if (bid == 2)
    TEST_PRINT (1, ("Overlapping send-receive test.\n"));
  else if (bid == 1)
    TEST_PRINT (1, ("Bidirectional (latency) test.\n"));
  else
    TEST_PRINT (1, ("Unidirectional (bandwidth) test.\n"));

  for (i = 0; i < 32; i++)
    {
      free_send_messages[i] = 0;
    }
  if (check)
    TEST_PRINT (1, ("Verififying contents of all messages.\n"));
  else
    TEST_PRINT (1, ("Not touching contents of messages.\n"));

}

int good_dir = 0;

static
int
open_file (void) 
{
  char fn[255];
  FILE *f;
  int success;  

  /* test to see that we can open a file in the directory */
  sprintf (fn, "logp_res/test");
  f = fopen (fn, "w");
  if (!f)
     success = 0;
  else {
    success = 1;
    fclose(f);
  }

  return success;
}

static
FILE *
open_allRTT (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/RTT.bysize");
  else
    sprintf (fn, "RTT.bysize");

  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_allLat (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/Lat.bysize");
  else
    sprintf (fn, "Lat.bysize");
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_allgap (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/gap.bysize");
  else
    sprintf (fn, "gap.bysize");
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_lenRTT (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/RTT.bylen");
  else
    sprintf (fn, "RTT.bylen");
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_lenLat (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/Lat.bylen");
  else
    sprintf (fn, "Lat.bylen");
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_lengap (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/gap.bylen");
  else
    sprintf (fn, "gap.bylen");
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_lenBW (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/BW.bylen");
  else
    sprintf (fn, "BW.bylen");
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_lenBW0 (void)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/BW0.bylen");
  else
    sprintf (fn, "BW0.bylen");
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_sum_file (int size)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/size%d.sum.dat", size);
  else
    sprintf (fn, "size%d.sum.dat", size);
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_track_file (int size, int len, int delay)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/l%d.d%d.dat",len, delay);
  else
    sprintf (fn, "l%d.d%d.dat",len, delay);
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

static
FILE *
open_result_file (int size,int len)
{
  char fn[255];
  FILE *f;
  
  if (good_dir)
    sprintf (fn, "logp_res/s%d.l%d.res", size,len);
  else
    sprintf (fn, "s%d.l%d.res", size,len);
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

typedef struct out_buffer
{
  int len;
  double RTT;
} 
out_buffer;

#define TEST_SIZE 1000  /* must be as big at the count for RTT */
struct out_buffer out[TEST_SIZE];
int test_count = 0;

static
FILE *
open_RTT_file (int size, int len)
{
  char fn[255];
  FILE *f;
  
  test_count = 0; /* opening new file - clear the count */
  if (good_dir)
    sprintf (fn, "logp_res/s%d.l%d.RTT",size,len);
  else
    sprintf (fn, "s%d.l%d.RTT",size,len);
  f = fopen (fn, "w");
  if (!f)
    TEST_PRINT (2, ("Could not open file \"%s\"\n", fn));
  return f;
}

/* TIMING ROUTINES */

#ifdef VXWORKS

#include <tickLib.h>
int
gettimeofday(struct timeval *tv, struct timezone *tz)
{
    TICK mytick;

    mytick = vxAbsTicks;

    tv->tv_sec = mytick.lower / CLOCKS_PER_SEC;
    tv->tv_usec = 0;

    return (0);
}
#endif


static struct timeval begin;

long start_usec = 0;

double 
get_useconds (void)
{
  double          mytime;

  gettimeofday(&begin, (struct timezone *) NULL);
  if (!start_usec) {
	start_usec = begin.tv_sec;
  }
  mytime = (((double) (begin.tv_sec-start_usec))*(double) 1000000.00) + 
			((double) begin.tv_usec);

  return mytime;
}

static
void
time_gettime(int n, double *cost)
{
  int i;
  double start, mid, end;

  start = get_useconds();

  for (i = 0; i < n; i++){
    mid = (get_useconds()) + i;
  }
  GM_PARAMETER_MAY_BE_UNUSED (mid);
  end = (get_useconds() - start);

  TEST_PRINT (0,("%d calls to get_useconds in %.0f usecs, or %.4f usec/call\n", 
		(n+2), (end), (end/((n+2))) ));

  *cost = end/(n+2);

  TEST_PRINT (0,("get_useconds cost is %.4f usec/call\n",*cost));
}

typedef struct sent_buff{
  int send;
  gm_up_t *pp;
}sent_buff;

int sent_test_cnt = 0;

struct sent_buff sent_test[1000];


static int
logp_test (int argc, char **argv)
{
  gm_status_t status = GM_SUCCESS;
  unsigned int pending_send_count;
  unsigned int send_len, send_size, send_cnt_per_len;
  unsigned int recv_len, recv_size, recv_cnt_per_len;
  unsigned int recv_cnt = 0, send_cnt = 0, sent_cnt = 0;
  gm_recv_event_t *event;
  double t0 = 0.0;
  int cnt;
  double tn_start = 0;
  char out_seq = 1, in_seq = 1;
  double base, slope;
  double gettime_cost;
  int gettime_count = 0;
  int d;
  int tmp;
  double delta = 0.0;
  double RTT;
  int loop;
  int msg_size;
  int start, end;
  int len_start;
  int num_bursts = BURSTS ;
  int bursts[BURSTS] = 
  {1,2,4,6,8,10,12,14,16,18,20,22,24,26,28,30,32,36,40,44,48,64,80,96,112,128,144};
  /* the first entry is to warm up the program, and we discard the data */

#ifdef RTT_out
  FILE *data = NULL;
#endif
#ifdef SUM_FILE
  FILE *sum = NULL;
#endif
  FILE *track = NULL;
  FILE *bigtrack = NULL;
  const char *pathname = "logp_res";
  
  signal (SIGINT, handle_sigint);
  
  parse_args (argc, argv);

  printf("Timing gettime\n");
  fflush(stdout);
  time_gettime(1E7, &gettime_cost);
  
  if (bid) {
    printf("Doing all_time_1us\n");
    fflush(stdout);
    all_time_1us(&base,&slope);
  }

  printf("\nOK, done with timing calibration\n");
  fflush(stdout);
  if (i_send){
    if (bid == 2){

#ifdef VXWORKS
      good_dir = 0;
#else
      mkdir(pathname,0777);
      good_dir = open_file();
#endif

#ifdef ALL_SIZE
      allRTT = open_allRTT();
      allLat = open_allLat();
      allgap = open_allgap();
#endif      

      lenRTT = open_lenRTT();
      lenLat = open_lenLat();
      lengap = open_lengap();
      lenBW = open_lenBW();
    }
    else if (bid==0){
      lenBW = open_lenBW0();
      fprintf (lenBW, "# length and BW for %d sends\n", message_count);
    }
  }

  /* if a size is chosen */
  if (the_size) {
    start = the_size;
    end = the_size;
  }
  /* else if min and max is selected */
  else if (min_size && max_size){
    start = min_size;
    end = max_size;
  }
  /* else do all legal sizes - or just size 14 */
  else{
    start = 14; /* 4 */
    end = 14; 
  }

  /***** SIZE *****/
  for (msg_size = start; msg_size <= end; msg_size++){
    min_size = msg_size;
    max_size = msg_size;

    test_init (port_id, TEST_PRIORITY, min_size, max_size);

    if (remote_host_name)
      {
	remote_node_id = gm_host_name_to_node_id (port, remote_host_name);
	if (remote_node_id == GM_NO_SUCH_NODE_ID)
	  {
	    TEST_PRINT (2, ("Error: host '%s' not found.\n",
			    remote_host_name));
	    gm_exit (GM_FAILURE);
	  }
	if (i_send){
	  TEST_PRINT (1, ("Sending to remote GM node %d.\n", remote_node_id));
	}
	if (i_recv){
	  TEST_PRINT (1, ("Receiving from remote GM node %d.\n", remote_node_id));
	}
	if (i_send && i_recv)
	  {
	    TEST_PRINT (2, ("Error: both loopback mode and remote host "
			    "specified.\n"));
	    gm_exit (GM_INVALID_PARAMETER);
	  }
      }
    else
      {
	if (!(i_send && i_recv))
	  {
	    TEST_PRINT (2, ("Error: no remote host specified and not in "
			    "loopback mode.\n"));
	    gm_exit (GM_FAILURE);
	  }
	TEST_PRINT (1, ("Performing loopback test.\n"));
      
	remote_node_id = this_node_id;
      }

#ifdef SUM_FILE
    if ((i_send) && (bid)){
      sum = open_sum_file(min_size);
      fprintf (sum, "Time in usec for %d sends and receives with size %d:\n", message_count, min_size);
      fprintf (sum, "min len, total time, time/msg, delay, avg RTT\n");
    }
#endif

    len_start = 8;

    for (min_len = len_start; min_len <= gm_max_length_for_size(msg_size); min_len+= len_inc){

      /* this is so that the len_inc will increase with min_len */
      if (min_len >= 256){ 
	len_inc = 16;
        if (min_len >= 512){
	  len_inc *=2;
	  if (min_len >= 1024){
	    len_inc *=2;
	    if (min_len >= 2048){
	      len_inc *=2;
	      if (min_len >= 4096){
		len_inc *=2;
		if (min_len >= 8192){
		  len_inc *=2;
		  if (min_len >= 16384){
		    len_inc *=2;
		  }
		}
	      }
	    }
	  }
	}
      }

      TEST_PRINT(2,("len = %d, size = %d, \n", min_len,msg_size));
      TEST_PRINT(2,("len_inc = %d \n",len_inc));
 
      if (bid == 2){ /* going to calculate RTT first (bid=1) then do bid=2 */
	loop = 1;
	bid = 0;     /* it is incremented below */
      }
      else
	loop = 0;

      /***** LOOP *****/
      do {            /* while (loop) */
	if (loop){
	  bid ++;
	  if (bid == 2){ 
	    loop = 0;
	    num_bursts = BURSTS;
	    bursts[0] = 1;
	    bursts[1] = 2;

	    if (min_len <= 512)
	      max_delta = MAX_DELTA;
	    else if (min_len <= 1024)
	      max_delta = BIG_MAX_DELTA;
	    else if (min_len <= 2048)
	      max_delta = BIG_MAX_DELTA;
	    else if (min_len <= 4096)
	      max_delta = BIG_MAX_DELTA*2;
	    else if (min_len <= 8192)
	      max_delta = BIG_MAX_DELTA*4;
	    else 
	      max_delta = BIG_MAX_DELTA*8;
	  
	  }
	}


	if (res_file && (bid == 2) && i_send)
	  bigtrack = open_result_file(min_size,min_len);

	if ((bid == 1) || (bid==0)){
	  max_delta = 0;
	  num_bursts = 2;
	  bursts[0] = 2; /* to prime the system - we throw these away */
	  bursts[1] = message_count;
	}

#ifdef RTT_out
	if (i_send && (bid == 1))
	  {
	    data = open_RTT_file (min_size, min_len);
	    if (!data)
	      {
		status = GM_FAILURE;
		goto abort_with_open_port;
	      }
	  }
#endif

	/***** DELTA LOOP *****/
	for (delta = 0; delta <= max_delta; delta *= 2) {

	  if (i_send && !(bid == 1))
	    {
	      if (dat_file && (bid == 2)){
		track = open_track_file(min_size,min_len,delta);
	      }
	      tracks = &racks[0];
	    }

	  if (delta){
	     d = (int)((delta - base)/slope); 
	     d = MAX(0,d);
	  }
	  else 
	    d = 0;

	  /***** BURSTS LOOP *****/
	  for (cnt = 0 ; cnt < num_bursts ; cnt++){
	    int fast = 0;
	    count_per_len = bursts[cnt]; 

#ifdef TRACK
	    if (i_send && !(bid==1)){
	      tracks+= sprintf (tracks, "\n Sending %d messages \n",count_per_len);
	    }
#endif

	    recv_cnt_per_len = send_cnt_per_len = 0;
	    recv_size = send_size = min_size;
	    recv_len = send_len = min_len;

	    if (i_send)
	      {
		if (bid)
		  pending_send_count = 1;
		else
		  pending_send_count = 10000;

		goto try_send;
	      }
	    else
	      {
		pending_send_count = 0;
	      }
      
	    while (1)
	      {
		event = gm_receive (port);

		switch (gm_ntoh_u8 (event->recv.type))
		  {
		  case GM_SENT_EVENT:
		    {
		      gm_up_n_t *pp;
		
		      /*
		       * recycle send buffers
		       */
		      pp = gm_ntohp (event->sent.message_list);

		      do
			{
			  free_send_message (gm_ntohp(*pp));
			  gm_free_send_token (port, TEST_PRIORITY);

			  sent_cnt ++;
			}
		      while (gm_ntohp (*++pp));

		      TEST_PRINT (0, ("gm_sent_event\n"));
#ifdef TRACK
		      if (i_send && !(bid==1)){
			tracks+= sprintf (tracks, "gm_sent_event\n");
		      }
#endif

		      if ((bid == 0) && (send_cnt == 0) 
			  && i_send && (sent_cnt == count_per_len)){
			sent_cnt = 0;
			goto done;
		  
		      }
		      else
			goto try_send;
		    }
	      
		  case GM_FAST_RECV_EVENT:
		  case GM_FAST_HIGH_RECV_EVENT:
		  case GM_FAST_PEER_RECV_EVENT:
		  case GM_FAST_HIGH_PEER_RECV_EVENT:
		    fast = 1;
		  case GM_RECV_EVENT:
		  case GM_HIGH_RECV_EVENT:
		  case GM_PEER_RECV_EVENT:
		  case GM_HIGH_PEER_RECV_EVENT:
		    {
		      test_message_t *p;
		      int size = gm_ntoh_u8 (event->recv.size);
		      int len = gm_ntoh_u32 (event->recv.length);
		      int seq;
#ifdef TRACK
		      if (!(recv_cnt)){
			if (i_send && (bid==2)){
			  tracks+= sprintf (tracks, "First receive at %d-th send\n", send_cnt);
			}
		      }
#endif
		      if (i_recv)
			TEST_PRINT (0,("gm_recv_event \n"));		

		      total_recvd++;
		      if ((tn_start) && (bid == 1) && (i_send)){ 
			 /* only do this if tn_start has been set */
			 /* assumes that gettime is not called between
			   setting tn_start and here */

			 RTT = (get_useconds() -
			       tn_start) - gettime_cost; 
			 gettime_count ++;

			 TEST_PRINT (0,("gm_recv_event, RTT = %f usec\n", RTT));
			 tn_start = 0;

			 if (!est_RTT){
			    out[test_count].len = send_len;
			    out[test_count].RTT = RTT;
			    test_count++;
			
			    RTT_cum += RTT;
			 }
			 else{
			    /* only do RTT's in the right range */
			    if (((1.3 * est_RTT) > RTT) && (RTT > (0.7 * est_RTT))){
			       out[test_count].len = send_len;
			       out[test_count].RTT = RTT;
			       test_count++;
			
			       RTT_cum += RTT;
			    }
			 }
		      }
#ifdef TRACK

		      if (i_send && !(bid==1)){
			tracks+= sprintf (tracks, "gm_receive_event\n");
		      }
#endif
		
		      if (fast)
			{
			  p = (test_message_t *) gm_ntohp (event->recv.message);
			}
		      else
			{
			  p = (test_message_t *) gm_ntohp (event->recv.buffer);
			}
		
		      if (gm_ntoh_u32(p->type) != test_ping_type)
			{
			  TEST_PRINT (2, ("oops: type = 0x%x when it should be 0x%x\n",
					  (unsigned int) gm_ntoh_u32(p->type), test_ping_type));
			  /*abort ();*/
			  gm_exit(GM_INTERNAL_ERROR);
			}
		
		      /*
		       * verify message contents
		       */
		      if (check)
			{
			  seq = p->payload[0];
			  if (in_seq != seq)
			    {
			      int i;
			
			      TEST_PRINT (2, ("sequence error; size = %d, len = %d, "
					      "p = %p, expected 0x%x and got 0x%x\n",
					      size, len, p, in_seq, seq));
			      for (i = 0; i < len; i++)
				TEST_PRINT (2, ("%02x ",
						((int) ((char *) p)[i]) & 0xff));
			      TEST_PRINT (2, ("\n"));
			    }
			  else
			    {
			      if (verify_buffer (p->payload, payload_len (len)) != 0)
				{
				  TEST_PRINT (2, ("data error; size = %d, len = %d, "
						  "p = %p\n",
						  size, len, p));
				}
			    }
			  if (++in_seq == 127)
			    in_seq = 0;
			}

		      /*
		       * recycle receive buffer
		       */
		      gm_provide_receive_buffer (port,
						 gm_ntohp (event->recv.buffer),
						 (unsigned int)
						 gm_ntoh_u8 (event->recv.size),
						 TEST_PRIORITY);
		


		      recv_cnt++;
		
		      if (++recv_cnt_per_len >= count_per_len)
			{
			  recv_cnt_per_len = 0;
			  recv_len += LEN_INC;
			  /*if (recv_len > MAX_TEST_LEN (recv_size))*/
			  {
			    recv_size++;
			    if (recv_size > max_size)
			      {
				TEST_PRINT (0, ("Done receiving.\n"));
				if (bid == 2){
				  if (i_send){
				    recv_cnt = 0;
				    goto done;
				  }
				}
				else{
#ifdef TRACK0
				  if (i_send)
				    tracks+= sprintf (tracks, "goto done from recv \n");
#endif
				  goto done;
				}
			      }
			    else
			      {
				recv_len = min_len;
			      }
			  }
			}

		      if (bid)
			{
			  if (!i_send)
			    send_len = len;
			  pending_send_count++;
			  /* gm_assert (pending_send_count == 1);*/
			  goto try_send;
			}
		    }
	      
		  case GM_NO_RECV_EVENT:
		  try_send:


		  /* Enqueue any pending send */
		  while (pending_send_count &&
			 free_send_messages[send_size] &&
			 gm_alloc_send_token (port, TEST_PRIORITY))
		    {
		
		      test_message_t *pkt = &free_send_messages[send_size]->message;
		
		      /*
		       * fill message with known values
		       */
		      if (check)
			{
			  fill_buffer (pkt->payload, payload_len (send_len), out_seq);
		    
			  if (++out_seq == 127)
			    out_seq = 0;
			}
		
		      pkt->size = gm_hton_u8 (send_size);
		      pkt->sender = gm_hton_u8 (this_node_id);
		      pkt->len = gm_hton_u16 ((gm_u16_t) send_len);
		      pkt->type = gm_hton_u32 (test_ping_type);
		
		      if (!send_cnt){
			/* if this is the first send, start timer */

			t0 = get_useconds();
			gettime_count ++;
#ifdef RTT_out
			if ((bid==1) && i_send){
			  fprintf (data, "RTT for %d messages of length %d\n", count_per_len, min_len);
			}
#endif
		      }

		      if (bid==1) {
			tn_start = get_useconds();
			gettime_count ++;
		      }

		      gm_send (port, pkt, send_size,
			       send_len, TEST_PRIORITY,
			       remote_node_id, other_port_id);

		      if (i_recv)
			TEST_PRINT (0, ("gm send called\n"));

		      tmp = 0;
		      tmp = my_delay(d);
		      GM_PARAMETER_MAY_BE_UNUSED (tmp);
		      
		      free_send_messages[send_size]
			= free_send_messages[send_size]->next;

		      if ((bid==1) || ((bid==2) && i_recv))
			pending_send_count--;
		
		      total_sent++;
		      send_cnt++;
#ifdef TRACK
		      if ((!(bid == 1)) && (i_send)){
			tracks+= sprintf (tracks, "gm_send\n");
		      }
#endif

#ifdef TRACK0		
		      if ((!(bid == 1)) && (i_send)){

			seconds = (get_useconds() - t0);
			gettime_count ++;

			seconds_adj = seconds - gettime_count*gettime_cost;

			tracks+= sprintf (tracks, "gm_send:  %f usec, adjusted %f us, gettime_count = %d \n",
				seconds, seconds_adj, gettime_count);
		      }
#endif

		      if (++send_cnt_per_len >= count_per_len)
			{
			  seconds = (get_useconds() - t0);
			  seconds = seconds - gettime_count*gettime_cost;

			  TEST_PRINT (0, ("sent %d messages of size %d length %d in %f usec\n",
					 send_cnt, send_size, send_len, seconds));

			  if (i_send){

			    if (bid==1){
			      TEST_PRINT (0, ("cumulative RTT is %f usec.\n",(RTT_cum)));
			      /* avg RTT is based on *receive* count, 
				 others measure time for a certain number 
				 of *sends* */

			      if (test_count){
				avg_RTT = (RTT_cum/(test_count));
#ifdef SUM_FILE
				fprintf (sum,"%8d %10.2f %10.2f %3.2f %8.2f\n", 
				min_len, (seconds), (seconds/(send_cnt)),delta,avg_RTT);
#endif
			      }
			      RTT_cum = 0;
#ifdef RTT_out
			      {
				int i;
				for (i = 0; i< test_count; i++){
				  fprintf(data, "%f\n", out[i].RTT);
				}
			      }
			      fprintf(data, " used %d samples\n",test_count);
#endif
			      test_count = 0;
			      recv_cnt = 0;
			    }
			    else {
			      send_time[send_time_count].sends = send_cnt;
			      send_time[send_time_count].utime = seconds;
			      send_time_count ++;
			    }
			  }

			  send_cnt = 0;
			  gettime_count = 0;
		    
			  send_cnt_per_len = 0;
			  send_len += LEN_INC;
			  /*if (send_len > MAX_TEST_LEN (send_size))*/
			  {
			    send_size++;
			    if (send_size > max_size)
			      {
				TEST_PRINT (0, ("Done sending.\n"));
				if (bid == 2){
				  if (i_recv){
				    send_cnt = 0;
				    goto done;
				  }
				}
				else if (bid==0){
				  if ((sent_cnt <count_per_len) && (i_send)){
				/* do nothing-will goto done from sent_event */
#ifdef TRACK
				    tracks+= sprintf (tracks, "don't go to done from send \n");
#endif
				  }
				}
				else if (!i_recv)
				  goto done;
			      }
			    else
			      {
				send_len = min_len;
			      }
			  }
			}
		    }

		  break;

		  case GM_ALARM_EVENT:
		    gm_unknown (port, event);
		    if (!got_alarm)
		      break;
		    got_alarm = 0;
		    TEST_PRINT (0, ("gm_alarm_event\n"));
	      
		    /* REMOVED STUFF HERE SINCE NOT USING ALARM */
		    /* .........................................*/

		    gm_set_alarm (port, &my_alarm, (gm_u32_t) (seconds * 1e6),
				  set_got_alarm, 0);
		    break;
	      
		    /*
		     * should never happen
		     */
		  default:
		    gm_unknown (port, event);
		  }
	      }

	  done: {};

	  }  /* end of count loop */
	  if ((i_send) && (bid == 2)){
	    print_results(track,delta);
	    if (dat_file){
	      fclose(track);
	    }
	  }
	  if (!delta) delta = 1; /* so we can (delta *= 2) */
	}  /* end of delta (delay) loop */

	TEST_PRINT (0, ("Test completed.\n"));

	print_tracks(bigtrack);
	if (i_send){
	  if (bid == 2){
	    calc_parameters(msg_size,min_len,bigtrack);
	    if (res_file)
	      fclose(bigtrack);
	  }
	  else if (bid == 1){
	    TEST_PRINT(0,("Average RTT is %f\n", avg_RTT));
	  }
	  else if (bid == 0){
	    print_results0(min_len);
	  }
	}

      } while (loop); /* do rtt and then logp calcs */

    } /* min_len loop */
#ifndef WIN32
    /* leave time to finish ongoing communications */
    my_delay(20000000);
#endif

#ifdef RTT_out
  abort_with_open_port:
#endif
    gm_close (port);
    TEST_PRINT (1, ("GM deinitialized.\n"));

  }/* msg_size loop */

  die (status);

  return 0;
}

#define GM_MAIN logp_test
#include "gm_main.h"
