/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include <stdio.h>

enum test_message_types {
  test_bogus_type = 0,
  test_ping_type,
};

#include "gm.h"

#define REPORT_PROGRESS	0
#define TEST_PRIORITY	GM_LOW_PRIORITY
#define MIN_TEST_SIZE	32
#define MAX_TEST_SIZE	32
#define TEST_TIME	4
#define TEST_MESSAGE_CNT	1

typedef struct test_message
{
  gm_u32_t route;
  gm_u32_t type;
  gm_u8_t size;
  char     payload[1];
} test_message_t;

				/* Define the type of message we will
                                   be sending and receiving. */
typedef struct chainable_test_message
{
				/* Use the 8 bytes at the head of the
                                   message for user data that will not
                                   be sent over the network. */
  struct chainable_test_message *next;
  gm_u8_t  reserved[4];
  test_message_t message;

} chainable_test_message_t;

chainable_test_message_t *free_send_messages[33];

static /* inline  */
void
free_send_message (test_message_t *p)
{
  chainable_test_message_t *ctp = (chainable_test_message_t *)((char *)p-8);

  ctp->next = free_send_messages[p->size];
  free_send_messages[p->size] = ctp;
}

gm_port_t port;

void
test_init ()
{
  int i, size, j;
  gm_unique_id_t unique_id;
  char *unique = unique_id.as_bytes;
  gm_status_t status;

  for (i=0; i<33; i++) free_send_messages[i] = 0;
  
				/* Initialize GM */
  status = gm_mapper_open (&port, 0);
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not open mapper port", status);
      gm_exit (status);
    }

  gm_free_send_tokens (&port, GM_LOW_PRIORITY, GM_NUM_SEND_TOKENS/2);
  gm_free_send_tokens (&port, GM_HIGH_PRIORITY, GM_NUM_SEND_TOKENS/2);
  
  /* Setup recv tokens */
  for (size = MIN_TEST_SIZE; size <= MAX_TEST_SIZE; size++)
    {
      int tokens_of_size;

      /*
      if (size<8)
	tokens_of_size = 512/(1<<size);
      else
	tokens_of_size = 2;
	*/

      tokens_of_size = 5;

      for (i = 0; i < tokens_of_size; i++)
	{
	  chainable_test_message_t *ctp;
	  
	  ctp = (chainable_test_message_t *)
	    gm_dma_calloc (&port, 1, GM_MTU + GM_MAX_NETWORK_DIAMETER + 8);
	  gm_always_assert (ctp);
      
	  ctp->message.type = test_ping_type;
	  ctp->message.size = size;
	  
				/* Make sure message is aligned. */
	  gm_always_assert (GM_DMA_ALIGNED (ctp));

#if GM_DEBUG_BUFFERS
	  gm_register_buf (&ctp->message, -1);
#endif
	  _gm_provide_raw_receive_buffer (&port, &ctp->message);

				/* Alloc send messages. */
	  ctp
	    = ((chainable_test_message_t *)
	       gm_dma_calloc (&port, 1, GM_MTU + GM_MAX_NETWORK_DIAMETER + 8));
	  gm_always_assert (ctp);
				/* Fill the message with 0xaa */
	  for (j = 0; j < GM_MTU + GM_MAX_NETWORK_DIAMETER + 8; j++)
	    ((char *)ctp)[j] = 'w';

	  ctp->next = 0;
	  ctp->message.size = size;
	  ctp->message.type = test_ping_type;

#if GM_DEBUG_BUFFERS
	  gm_register_buf (&ctp->message, -1);
#endif

	  free_send_message (&ctp->message);
	}
    }

  status = gm_get_unique_board_id (&port, &unique_id);
  if (status != GM_SUCCESS)
    {
      fprintf (stderr, "Could not determine board ID.\n");
      gm_exit (status);
    }
  
				/* Print the ethernet ID of the board. */
  printf ("Board has ethernet id ");
  for (i=0; i<5; i++)
    {
      printf ("%02X", (unsigned int)unique[i] & 0xff);
      putchar (':');
    }
  printf ("%02X", (unsigned int)unique[i] & 0xff);
  putchar ('\n');
}

int
gm_raw ()
{
  unsigned int pending_send_count = 1, size = MIN_TEST_SIZE;
  unsigned int recv_cnt = 0;
  unsigned int send_cnt = 0;
  unsigned int sent_cnt = 0;
  gm_recv_event_t *event;

  int i;

  printf ("Warning: this test will only work with a loopback cable.\n");
  
  test_init ();
  
  pending_send_count = 1;

  gm_set_alarm (&port, TEST_TIME * 1000 * 1000);

  while (1)
    {
      /* Enqueue any pending send */
      while (free_send_messages[size]
	     && gm_alloc_send_token (&port, TEST_PRIORITY))
	{
	  gm_always_assert (free_send_messages[size]->message.type
		     == test_ping_type);
	  _gm_raw_send (&port, &free_send_messages[size]->message, 64);
	  free_send_messages[size] = free_send_messages[size]->next;
	  send_cnt++;
	  
	}

      event = gm_blocking_receive (&port);

      switch (gm_recv_event_type (event))
	{
	case GM_SENT_EVENT:
	  //printf ("got a sent event\n");
	  {
	    void **pp;
	    
	    pp = (void **) ntohl ((long) event->sent.message_list);
	    do
	      {
		*pp = gm_ntohp (*pp);
		free_send_message (*pp);
		gm_free_send_token (&port, TEST_PRIORITY);
		sent_cnt++;
	      }
	    while (*++pp);
	  }
	  break;
	    
	case GM_RAW_RECV_EVENT:
	  /* Recycle the receive buffer. */
	  gm_assert (!((void *)port.recv_queue_limit
			 < (void *)event->recv.buffer
			 && (void *)event->recv.buffer
			 < (void *)port.recv_queue_start));
	  
	  {
	    test_message_t *pkt = gm_ntohp (event->recv.buffer);

	    if (pkt->type != test_ping_type) {
	      printf ("oops; type = 0x%lx\n", pkt->type);
	    }
	    
	    gm_always_assert (pkt->type == test_ping_type);
	    _gm_provide_raw_receive_buffer (&port,
					   (void *)ntohl ((long) event->recv.buffer));
	    recv_cnt++;
	    
	    //printf ("Message received\n");
	  }
	  
	  break;
	  
	case GM_ALARM_EVENT:
	  printf ("Received %d of %d.  ", recv_cnt, send_cnt);
	  printf ("Sent %d of %d.\n", sent_cnt, send_cnt);
	  gm_set_alarm (&port, TEST_TIME * 1000 * 1000);
	  break;

	case GM_HIGH_RECV_EVENT:
	case GM_FAST_HIGH_RECV_EVENT:
	case GM_HIGH_PEER_RECV_EVENT:
	case GM_FAST_HIGH_PEER_RECV_EVENT:
	case GM_RECV_EVENT:
	case GM_FAST_RECV_EVENT:
	case GM_PEER_RECV_EVENT:
	case GM_FAST_PEER_RECV_EVENT:
	  abort ();
	  
	case GM_NO_RECV_EVENT:
	  break;
	  
	default:
	  gm_unknown (&port, event);
	  break;

	}
    }
  return 0;
}

#define GM_MAIN gm_raw
#include "gm_main.h"
