/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#include "gm.h"

#if GM_OS_VXWORKS
#include <time.h>
#elif !defined WIN32
#include <sys/time.h>
#endif

				/* sleep() */
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#define DEBUG_NWAY 0

extern int quiet;

/*
 * quiet is: 0 for verbose, 1 for quiet, and 2 for silent (except for errors)
 */
#define TEST_PRINT(level, args)  do { \
  if (level>=quiet) printf args; \
} while (0)

/****************************************************************
 * typedefs
 ****************************************************************/

enum test_message_types
{
  test_bogus_type = 0,
  test_ping_type,
  test_forward_type
};

typedef struct test_message
{
  gm_u32_n_t seq_no;
  gm_u32_n_t type;

  gm_u32_n_t size;
  gm_u32_n_t sender;

  gm_u8_n_t payload[1];

}
test_message_t;

/* test parameters */

#define MAX_NODES 64

#define GM_NWAY_DELAY 2

float PROB_FORWARD = 0.5;
float PROB_SLEEP = (float) 0.000005;

#define MIN_TEST_SIZE	5
#define MAX_TEST_SIZE	14

#define MIN_TEST_LEN 8
#define MAX_TEST_LEN(size) (gm_max_length_for_size (size))

#define ROUNDUP(n,m)	(((gm_u32_t)(n)+(gm_u32_t)(m)-1)&~((gm_u32_t)(m)-1))
#define DMA_ROUNDUP(p) 	ROUNDUP (p, GM_DMA_GRANULARITY)

gm_s64_t total_sent = 0, total_recvd = 0;
gm_s64_t bytes_sent = 0, bytes_recvd = 0;
gm_s64_t old_bytes_sent = 0, old_bytes_recvd = 0;
gm_s64_t sleeps = 0;

gm_s64_t total_senta[MAX_NODES];
gm_s64_t total_recvda[MAX_NODES + 1];	/* last location is for error */
#ifdef PRINT_LOTS
gm_s64_t bytes_senta[MAX_NODES];
gm_s64_t bytes_recvda[MAX_NODES + 1];
gm_s64_t old_bytes_senta[MAX_NODES];
gm_s64_t old_bytes_recvda[MAX_NODES];
#endif

struct gm_port *port_g = NULL;
int num_nodes = -1;
char *host_name[MAX_NODES];
int node_id[MAX_NODES];

int cycle = 0;
unsigned int target_index = 0;
int quiet = 0;
int rand_seed;
int verify = 0;
unsigned port_id_g = 2;
unsigned board_num = 0;
int fast = 0;
unsigned int this_node_id;
int slp = 0;
int in_seq[MAX_NODES], out_seq[MAX_NODES];

static gm_inline int
payload_len (int len)
{
  return len - GM_OFFSETOF (test_message_t, payload);
}

/****************************************************************
 * DMAable Buffer allocation
 ****************************************************************/

/* The type of message we will
   be sending and receiving. */

typedef struct chainable_test_message
{
  /* Use the 8 bytes at the head of the
     message for user data that will not
     be sent over the network. */
  struct chainable_test_message *next;
  gm_u32_t size;
  test_message_t message;

}
chainable_test_message_t;

chainable_test_message_t *free_send_messages[32];

static gm_inline chainable_test_message_t *
ctm_parent (test_message_t * p)
{
  return ((chainable_test_message_t *)
	  ((char *) p - GM_OFFSETOF (chainable_test_message_t, message)));

}

static void
verify_send_buffer_size (void *buffer, unsigned int size)
{
  gm_assert (ctm_parent (buffer)->size == size);
}

/*
 * add a buffer to the (user managed) list of free send
 * buffers
 */
static gm_inline void
free_send_message (test_message_t * p)
{
  chainable_test_message_t *ctm = ctm_parent (p);
  gm_u32_t p__size;

  p__size = gm_ntoh_u32 (p->size);
  gm_assert (ctm->size == p__size);

  ctm->next = free_send_messages[p__size];
  free_send_messages[p__size] = ctm;
}

static gm_inline test_message_t *
alloc_send_message (unsigned int size)
{
  chainable_test_message_t *ctm;
  ctm = free_send_messages[size];
  if (!ctm)
    return 0;
  free_send_messages[size] = ctm->next;
  return &ctm->message;
}

static gm_status_t
init_send_buffer_allocator (struct gm_port *port)
{
  unsigned int send_buffers;
  unsigned int ui, size;

  for (size = MIN_TEST_SIZE; size <= MAX_TEST_SIZE; size++)
    {
      send_buffers = 3;		/* FIXME */

      /* allocate send buffers */
      for (ui = 0; ui < send_buffers; ui++)
	{
	  chainable_test_message_t *ctm;

	  ctm = ((chainable_test_message_t *)
		 gm_dma_calloc (port, 1,
				GM_OFFSETOF (chainable_test_message_t,
					     message) +
				gm_max_length_for_size (size)));
	  if (!ctm)
	    return GM_OUT_OF_MEMORY;

	  gm_always_assert (GM_DMA_ALIGNED (ctm));

	  ctm->next = 0;
	  ctm->size = size;
	  ctm->message.size = gm_hton_u32 (size);
	  ctm->message.type = gm_hton_u32 (-1);

#if GM_DEBUG_BUFFERS
	  gm_register_buf (&ctm->message, size);
#endif

	  free_send_message (&ctm->message);
	}
    }
  return GM_SUCCESS;
}

/****************************************************************
 * 
 ****************************************************************/

#if 0
#error not updated
/*
 * simple buffer fill and verify routines
 */
static gm_inline void
fill_buffer (char *buf, int len, char seed)
{
  int i;

  buf[0] = seed;

  for (i = 1; i < len; i++)
    {
      buf[i] = seed;
    }
}

static gm_inline int
verify_buffer (char *buf, int len)
{
  int i;
  char val;
  char seed = buf[0];

  for (i = 1; i < len; i++)
    {
      val = seed;
      if (buf[i] != val)
	{
	  TEST_PRINT (2,
		      ("data error; at pos %d expected 0x%x and got 0x%x\n",
		       i, val, buf[i]));
	  return 1;
	}
    }

  return 0;
}

#else

/*
 * better (but slow) buffer fill and verify routines; these limit
 * throughput to a few MB/s
 */

static gm_inline int
verify_buffer (char *buf, int len)
{
  int i;
  char val;
  char seed = buf[0];
  gm_srand (seed);

  for (i = 1; i < len; i++)
    {
      val = gm_rand_mod (256);
      if (buf[i] != val)
	{
	  TEST_PRINT (2,
		      ("data error; at pos %d expected 0x%x and got 0x%x\n",
		       i, val, buf[i]));
	  return 1;
	}
    }

  return 0;
}

static gm_inline void
fill_buffer (char *buf, int len, char seed)
{
  int i;

  buf[0] = seed;
  gm_srand (seed);

  for (i = 1; i < len; i++)
    {
      buf[i] = gm_rand_mod (256);
    }
}

#endif

/****************************************************************
 * error handling
 ****************************************************************/

static void
err_exit (const char *str, gm_status_t status)
{
  gm_perror (str, status);
  if (port_g)
    {
      gm_close (port_g);
      port_g = NULL;
    }
  gm_exit (status);
}

/****************************************************************
 * sending
 ****************************************************************/

/* structure for recording information about sends for debugging
   purposes. */

struct nway_send_record
{
  struct gm_port *port;
  void *message;
  unsigned int size;
  unsigned long len;
  unsigned int priority;
  unsigned int target_node_id;
};

static struct gm_lookaside *send_record_lookaside;
static int outstanding_send_cnt[GM_NUM_PRIORITIES];

/* Initialize debugging support.  This function is called only if
   DEBUG_NWAY is nonzero */

static void
nway_debug_init (struct gm_port *port)
{
  if (!send_record_lookaside)
    {
      send_record_lookaside
	= gm_create_lookaside (sizeof (struct nway_send_record),
			       gm_num_send_tokens (port));
      if (!send_record_lookaside)
	err_exit ("could not create send record lookaside list",
		  GM_OUT_OF_MEMORY);
    }
}

/****************
 * Send token management
 ****************/

static int _lst, _hst;

static gm_inline void
free_low_send_tokens (int cnt)
{
  _lst += cnt;
}
static gm_inline void
free_high_send_tokens (int cnt)
{
  _hst += cnt;
}
static gm_inline void
free_low_send_token (void)
{
  _lst++;
}
static gm_inline void
free_high_send_token (void)
{
  _hst++;
}
static gm_inline int
alloc_low_send_token (void)
{
  return _lst ? _lst-- : 0;
}
static gm_inline int
alloc_high_send_token (void)
{
  return _hst ? _hst-- : 0;
}

/****************
 * Send Completion Handlers
 ****************/

static void check_sent (struct gm_port *port, void *message,
			enum gm_priority priority);


static void
dropped_or_resumed (struct gm_port *port, void *context, gm_status_t status)
{
  enum gm_priority priority;
  GM_PARAMETER_MAY_BE_UNUSED (port);
  
  gm_assert (status == GM_SUCCESS);

  priority = (enum gm_priority) ((long) context);
  gm_assert (priority == GM_LOW_PRIORITY || priority == GM_HIGH_PRIORITY);

  if (priority == GM_LOW_PRIORITY)
    free_low_send_token ();
  else
    free_high_send_token ();
}

/* either drop or resume sends on a connection that has experienced an error */

static void
drop_or_resume (struct gm_port *port, struct nway_send_record *r)
{
  if ((gm_rand () & 0xf) < 8)
    {
      TEST_PRINT (1, ("dropping %s priority sends\n",
		      r->priority == GM_LOW_PRIORITY ? "low" : "high"));
      gm_drop_sends (port, r->priority, r->target_node_id, port_id_g,
		     dropped_or_resumed, (void *) ((long) r->priority));
    }
  else
    {
      TEST_PRINT (1, ("resuming %s priority sends\n",
		      r->priority == GM_LOW_PRIORITY ? "low" : "high"));
      gm_resume_sending (port, r->priority, r->target_node_id, port_id_g,
			 dropped_or_resumed, (void *) ((long) r->priority));
    }
}

/* Handler called when simple sends complete. */

static void
sent (struct gm_port *port, void *context, gm_status_t status)
{
  struct nway_send_record *r;
  test_message_t *m;

  r = (struct nway_send_record *) context;

  if (status != GM_SUCCESS)
    {
      if (status != GM_SEND_DROPPED)
	{
	  gm_perror ("send completed with error", status);
	  TEST_PRINT (0,
		      ("(size=%u, len=%lu, %s priority, target_node_id=%u)\n",
		       r->size, r->len, r->priority ? "high" : "low",
		       r->target_node_id));
	  drop_or_resume (port, r);
	}
    }

  m = r->message;
  check_sent (port, m, GM_LOW_PRIORITY);
  gm_always_assert (gm_ntoh_u32 (m->type) == test_ping_type);
  free_send_message (m);
  free_low_send_token ();
  gm_lookaside_free (r);
}

/* Handler called when forwarded sends complete. */

static void
forwarded (struct gm_port *port, void *context, gm_status_t status)
{
  struct nway_send_record *r;
  test_message_t *m;

  r = (struct nway_send_record *) context;

  if (status != GM_SUCCESS)
    {
      if (status != GM_SEND_DROPPED)
	{
	  gm_perror ("send completed with error", status);
	  drop_or_resume (port, r);
	}
    }

  m = r->message;
  check_sent (port, m, GM_HIGH_PRIORITY);
  gm_always_assert (gm_ntoh_u32 (m->type) == test_forward_type);
  gm_provide_receive_buffer (port, m, r->size, GM_LOW_PRIORITY);
  free_high_send_token ();
  gm_lookaside_free (r);
}

/****************
 * send functions
 ****************/

/* cover functions for gm_send_to_peer() to add debugging hooks. */

static void
send_to_peer (struct gm_port *port, void *message, unsigned int size,
	      unsigned long len, unsigned int priority,
	      unsigned int target_node_id)
{
  struct nway_send_record *r;

  nway_debug_init (port);

  gm_always_assert (target_node_id);

  /* record the details about the send */

  r = ((struct nway_send_record *)
       gm_lookaside_alloc (send_record_lookaside));
  if (!r)
    err_exit ("could not record send details", GM_OUT_OF_MEMORY);
  r->port = port;
  r->message = message;
  r->size = size;
  r->len = len;
  r->priority = priority;
  r->target_node_id = target_node_id;

  /* count the number of sends */

  if (DEBUG_NWAY)
    {
      ++outstanding_send_cnt[priority];
      gm_printf ("outstanding sends before send: %d(low) %d(high)\n",
		 outstanding_send_cnt[GM_LOW_PRIORITY],
		 outstanding_send_cnt[GM_HIGH_PRIORITY]);
    }
  gm_send_to_peer_with_callback (port, message, size, len, priority,
				 target_node_id,
				 priority ==
				 GM_HIGH_PRIORITY ? forwarded : sent,
				 (void *) r);
}

/* debug function to verify that a buffer was actually used for an
   earlier send and remove the record for that send.  Also, this
   updates the outstanding send counters. */

static void
check_sent (struct gm_port *port, void *message, enum gm_priority priority)
{
  GM_PARAMETER_MAY_BE_UNUSED (message);
  
  if (DEBUG_NWAY)
    {
      nway_debug_init (port);

      /* count the number of sends */

      --outstanding_send_cnt[priority];
      gm_printf ("outstanding sends after sent: %d(low) %d(high)\n",
		 outstanding_send_cnt[GM_LOW_PRIORITY],
		 outstanding_send_cnt[GM_HIGH_PRIORITY]);
    }
}

static void
try_send (void)
{
  unsigned int send_len, send_size, target;

  while (1)
    {
      test_message_t *message;

      if (fast)
	{
	  send_size = MAX_TEST_SIZE;
	}
      else
	{
	  send_size = (((unsigned int) gm_rand ()
			% (MAX_TEST_SIZE - MIN_TEST_SIZE + 1))
		       + MIN_TEST_SIZE);
	}

      /* send_size = 12; */

      send_len
	=
	(DMA_ROUNDUP (gm_rand () % (gm_max_length_for_size (send_size) - 16))
	 + 16);
      /* send_len = gm_max_length_for_size (send_size) - 16; */

      gm_always_assert (send_size >= MIN_TEST_SIZE
			&& send_size <= MAX_TEST_SIZE);

      if (!alloc_low_send_token ())
	break;
      message = alloc_send_message (send_size);
      if (!message)
        {
           free_low_send_token();
           break;
        }

      if ((gm_rand () & 0xffff) / 16384.0 < PROB_SLEEP)
	slp = 1;

      if (cycle)
	{
      	  target = (unsigned int) (target_index++) % (unsigned int) num_nodes;
	}
      else 
	{
      	  target = (unsigned int) gm_rand () % (unsigned int) num_nodes;
	}

      gm_assert (send_len >= 8
		 && (send_len <= gm_max_length_for_size (send_size)));

      /*
       * fill message with known values
       */
      if (verify)
	{
	  fill_buffer ((char *) message->payload, payload_len (send_len),
		       (char) out_seq[target]);
	  message->seq_no = gm_hton_u32 (out_seq[target]);
	  out_seq[target]++;
	}

      message->type = gm_hton_u32 (test_ping_type);

      verify_send_buffer_size (message, send_size);
      gm_always_assert (target < (unsigned int) num_nodes);
      gm_assert (send_len <= gm_max_length_for_size (send_size));

      message->sender = gm_hton_u32 ((gm_u32_t) this_node_id);

      gm_assert (gm_ntoh_u32 (message->type) == test_ping_type);
      send_to_peer (port_g, message, send_size,
		    send_len, GM_LOW_PRIORITY, node_id[target]);

      total_sent++;
      total_senta[target]++;
      bytes_sent += send_len;
#ifdef PRINT_LOTS
      bytes_senta[target] += send_len;
#endif
    }
}

/****************************************************************
 * test proper
 ****************************************************************/

#define TIME_CALIBRATION 1e6

static void
die (void)
{
  if (port_g)
    {
      gm_close (port_g);
      port_g = NULL;
    }
  gm_exit (GM_FAILURE);
}

static int
get_local_id (int gm_id)
{
  int j;
  /* if no match is found, the count will go into a safe place */
  int val = MAX_NODES;

  for (j = 0; j < num_nodes; j++)
    {
      if (node_id[j] == gm_id)
	{
	  val = j;
	  break;
	}
    }
  return (val);
}

static int got_alarm;
static gm_alarm_t my_alarm;

static void
set_got_alarm (void *ignored)
{
  GM_PARAMETER_MAY_BE_UNUSED (ignored);
  
  got_alarm = 1;
}

static void
test_init (struct gm_port *port, int port_id,
	   int min_test_size, int max_test_size)
{
  int i, size;
  gm_status_t status;
  GM_PARAMETER_MAY_BE_UNUSED (port_id);
  
  /*
   * initialize GM
   */
  gm_assert (port);
  TEST_PRINT (1, ("GM is initialized.\n"));

  gm_initialize_alarm (&my_alarm);

  status = gm_set_acceptable_sizes (port, GM_LOW_PRIORITY,
				    ((1 << (max_test_size + 1)) - 1)
				    ^ ((1 << min_test_size) - 1));
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not set GM acceptable sizes", status);
      if (port)
	{
	  gm_close (port);
	  port = NULL;
	}

      gm_exit (status);
    }

  /* note the initial number of send tokens */

  free_low_send_tokens (gm_num_send_tokens (port) / 2);
  free_high_send_tokens (gm_num_send_tokens (port) / 2);

  /* allocate receive buffers */

  for (size = min_test_size; size <= max_test_size; size++)
    {
      /* total over all sizes must be <= GM_NUM_RECV_TOKENS */
      int recv_tokens_of_size = gm_num_receive_tokens (port) / 32;

      recv_tokens_of_size = 3;

      /* allocate low priority receive buffers */
      for (i = 0; i < recv_tokens_of_size; i++)
	{
	  void *message;

	  message = gm_dma_calloc (port, 1, gm_max_length_for_size (size));
	  if (!message)
	    err_exit ("could not allocate receive buffer\n",
		      GM_OUT_OF_MEMORY);

	  /* Make sure message is aligned. */
	  gm_always_assert (message);

#if GM_DEBUG_BUFFERS
	  gm_register_buf (message, size);
#endif

	  memset (message, 0, gm_max_length_for_size (size));
	  gm_provide_receive_buffer (port, message, size, GM_LOW_PRIORITY);
	}

      /* allocate high priority receive buffers */
      for (i = 0; i < recv_tokens_of_size; i++)
	{
	  void *message;

	  message = gm_dma_calloc (port, 1, gm_max_length_for_size (size));
	  if (!message)
	    err_exit ("could not allocate receive buffer\n",
		      GM_OUT_OF_MEMORY);

	  /* Make sure message is aligned. */
	  gm_always_assert (GM_DMA_ALIGNED (message));

#if GM_DEBUG_BUFFERS
	  gm_register_buf (message, size);
#endif

	  memset (message, 0, gm_max_length_for_size (size));
	  gm_provide_receive_buffer (port, message, size, GM_HIGH_PRIORITY);
	}

    }

  /* allocate send buffers */

  status = init_send_buffer_allocator (port);
  if (status != GM_SUCCESS)
    err_exit ("could not initialize send buffer allocator", status);

  status = gm_get_node_id (port, &this_node_id);
  if (status != GM_SUCCESS)
    {
      fprintf (stderr, "could not get node id\n");
      if (port)
	{
	  gm_close (port);
	  port = NULL;
	}
      gm_exit (status);
    }
}

static gm_inline void
handle_sigint (int i)
{
  GM_PARAMETER_MAY_BE_UNUSED (i);
  
  TEST_PRINT (1, ("Caught SIGINT.\n"));
  die ();
}

static void
usage (void)
{
  TEST_PRINT (2, ("\nusage: gm_nway [-p port] [-B board] [-f] [-v] [-q [-q]] "
		  "host1 host2 ...\n\n"));

  TEST_PRINT (2, ("-f     send 'fast' for nway blast\n"));
  TEST_PRINT (2, ("-B #   specifies board number (default = 0)\n"));
  TEST_PRINT (2, ("-p #   specifies port number (default = 2)\n"));
  TEST_PRINT (2, ("-v     verify contents of messages (slow)\n"));
  TEST_PRINT (2, ("-q     quiet (-q -q = very quiet)\n"));
  TEST_PRINT (2, ("--nodomain don't print hostname domain (off)\n"));
  TEST_PRINT (2, ("--cycle choose target in a cycle (off)\n"));
  TEST_PRINT (2, ("-help  print this help message\n"));

  TEST_PRINT (2, ("\n"));
  if (port_g)
    {
      gm_close (port_g);
      port_g = NULL;
    }
  gm_exit (GM_FAILURE);
}

static void
parse_args (int argc, char *argv[])
{
  int i;
  gm_status_t status;
  int nodomain = 0;

  signal (SIGINT, handle_sigint);

  for (argc--, argv++; argc; argc--, argv++)
    {
      if (strcmp (*argv, "-q") == 0)
	{
	  if (++quiet > 2)
	    quiet = 2;
	  continue;
	}
      else if (strcmp (*argv, "--help") == 0)
	{
	  usage ();
	}
      else if (strcmp (*argv, "--nodomain") == 0)
	{
	  nodomain++;
	  printf("Will not print domain of hostnames.\n");
	}
      else if (strcmp (*argv, "--cycle") == 0)
	{
	  cycle++;
	  printf("Will cycle through send targets.\n");
	}
      else if (strcmp (*argv, "-v") == 0)
	{
	  if (++verify > 2)
	    verify = 2;
	  continue;
	}
      else if (strcmp (*argv, "-f") == 0)
	{
	  fast = 1;
	  continue;
	}
      else if (strcmp (*argv, "-p") == 0)
	{
	  argc--, argv++;
	  if (!argc)
	    {
	      TEST_PRINT (2, ("Port number expected after '-p'.\n"));
	      usage ();
	    }
	  if (sscanf (*argv, "%i", &port_id_g) != 1
	      || port_id_g < 1 || port_id_g > gm_num_ports (port_g))
	    {
	      TEST_PRINT (2, ("bad port number: %d\n", port_id_g));
	      usage ();
	    }
	  continue;
	}
      else if (strcmp (*argv, "-B") == 0)
	{
	  argc--, argv++;
	  if (!argc)
	    {
	      TEST_PRINT (2, ("Board number expected after '-B'.\n"));
	      usage ();
	    }
	  if (sscanf (*argv, "%i", &board_num) != 1)
	    {
	      TEST_PRINT (2, ("bad Board number: %s\n", *argv));
	      usage ();
	    }
	  continue;
	}
      else
	break;
    }

  /* Open the user-specified port */

  port_g = NULL;
  status = gm_open (&port_g, board_num, port_id_g, "n-way test",
		    GM_API_VERSION_1_1);
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not open GM port", status);
      if (port_g)
	{
	  gm_close (port_g);
	  port_g = NULL;
	}
      gm_exit (status);
    }

  /* Convert the remaining host names to GM IDs. */

  if (!argc)
    {
      TEST_PRINT (2, ("No hosts specified."));
      usage ();
    }
  else
    {
      /* Convert host names to GM IDs */

      num_nodes = 0;
      for (; *argv; argv++)
	{
	  unsigned int id;

	  id = gm_host_name_to_node_id (port_g, *argv);
	  if (id == GM_NO_SUCH_NODE_ID)
	    {
	      TEST_PRINT
		(2, ("Could not translate node name \"%s\" to a GM ID.\n",
		     *argv));
	      usage ();
	    }
#if 0
	  /* Ignore this host in the list of hosts. */
	  if (id == port->this_node_id)
	    continue;
#endif
	  if (num_nodes >= MAX_NODES)
	    {
	      TEST_PRINT (2, ("too many nodes (%d max)\n", MAX_NODES));
	    }
	  host_name[num_nodes] = *argv;
	  if (nodomain)
	    {
	      /* if nodomain, then null terminate at the 1st '.' */
              char *ptr = 0;
	      ptr = strchr(host_name[num_nodes],'.');
	      if (ptr)
		ptr[0] = 0;
	    }
	  node_id[num_nodes] = id;
	  num_nodes++;
	}
    }

  if (fast)
    {
      /* don't forward, don't sleep */
      PROB_FORWARD = 0.0;
      PROB_SLEEP = 0.0;
    }


  /*
   * let the user know what the parameters are
   */
  TEST_PRINT (2, ("Interacting with hosts:\n"));
  for (i = 0; i < num_nodes; i++)
    TEST_PRINT (2, ("\t\"%s\" (GM ID %d)\n", host_name[i], node_id[i]));

  TEST_PRINT (1, ("Using port %d.\n", port_id_g));
  if (fast)
    TEST_PRINT (1, ("'fast' mode - sending as fast as possible\n"));

  switch (verify)
    {

    case 0:
      TEST_PRINT (1, ("Not touching contents of messages.\n"));
      break;

    case 2:
    case 1:
      TEST_PRINT (1, ("Verififying contents of messages.\n"));
      break;

    default:
      gm_always_assert (0);
    }

  if (num_nodes < 1)
    {
      TEST_PRINT (0, ("No nodes specified.\n"));
      usage ();
    }

}

static int
gm_nway (int argc, char *argv[])
{
  int in_fw_seq[MAX_NODES], out_fw_seq[MAX_NODES];
  int errs[MAX_NODES];
  double gm_stime = 0;
  double oldstime = 0;
  gm_recv_event_t *event;
  int i;
  int k;

  GM_VAR_MAY_BE_UNUSED (gm_stime);
  GM_VAR_MAY_BE_UNUSED (oldstime);

  cycle = 0;

  parse_args (argc, argv);
  test_init (port_g, port_id_g, MIN_TEST_SIZE, MAX_TEST_SIZE);

  for (k = 0; k < MAX_NODES; k++)
    {
      total_senta[k] = 0;
      total_recvda[k] = 0;
#ifdef PRINT_LOTS
      bytes_senta[k] = 0;
      bytes_recvda[k] = 0;
      old_bytes_senta[k] = 0;
      old_bytes_recvda[k] = 0;
#endif
    }

#ifndef WIN32
  srand48 (time (0));
#endif

  for (i = 0; i < MAX_NODES; i++)
    {
      in_seq[i] = out_seq[i] = 0;
      in_fw_seq[i] = out_fw_seq[i] = 1000000;
      errs[i] = 0;
    }

  gm_set_alarm (port_g, &my_alarm,
		(unsigned int) (GM_NWAY_DELAY * TIME_CALIBRATION),
		set_got_alarm, 0);

  while (1)
    {
      try_send ();

      event = gm_receive (port_g);

      if (!fast)
	{
	  if (slp)
	    {
	      printf ("sleeping\n");
	      gm_sleep (1);
	      sleeps++;
	      printf ("awake\n");
	      slp = 0;
	    }
	}

      switch (GM_RECV_EVENT_TYPE (event))
	{
	case GM_PEER_RECV_EVENT:
	case GM_FAST_PEER_RECV_EVENT:
	  {
	    test_message_t *buffer;
	    test_message_t *p;
	    unsigned int size = gm_ntoh_u8 (event->recv.size);
	    unsigned int len = gm_ntoh_u32 (event->recv.length);
	    int sender = gm_ntoh_u16 (event->recv.sender_node_id);
	    int seq;
	    int ptype;

	    total_recvd++;
	    bytes_recvd += len;
	    total_recvda[get_local_id (sender)]++;
#ifdef PRINT_LOTS
	    bytes_recvda[get_local_id (sender)] += len;
#endif
	    buffer = (test_message_t *) gm_ntohp (event->recv.buffer);

	    if (GM_RECV_EVENT_TYPE (event) == GM_FAST_PEER_RECV_EVENT)
	      {
		p = (test_message_t *) gm_ntohp (event->recv.message);
	      }
	    else
	      {
		p = buffer;
	      }

	    ptype = gm_ntoh_u32 (p->type);
	    gm_always_assert (ptype == test_ping_type);

	    /*
	     * verify message contents
	     */
	    if (verify)
	      {
		seq = gm_ntoh_u32 (p->seq_no);

		if (1 && (in_seq[get_local_id (sender)] != seq))
		  {

		    TEST_PRINT (2,
				("low  seqerr; size %d, len %d, sender %d, "
				 "buf %p; exp 0x%x, got 0x%x\n", size, len,
				 sender, p, in_seq[get_local_id (sender)],
				 seq));

		    /*
		       for (i = 0; i < len; i++)
		       TEST_PRINT
		       (2, ("%02x ", ((int) ((char *) p)[i]) & 0xff));
		       TEST_PRINT (2, ("\n"));
		     */
		    /* print_all_buffers_size (6); */
		    /* die(); */
		    errs[get_local_id (sender)]++;
		    if (errs[get_local_id (sender)] > 30)
		      die ();
		  }
		else
		  {
		    if (verify_buffer ((char *) p->payload,
				       payload_len (len)) != 0)
		      {
			TEST_PRINT (2, ("data error; size = %d, len = %d, "
					"p = %p\n", size, len, p));
			errs[get_local_id (sender)]++;
			if (errs[get_local_id (sender)] > 10)
			  die ();
			/* print_all_buffers_size (6); */
			/* die(); */
		      }
		  }
		in_seq[get_local_id (sender)]++;
	      }

	    if ((gm_rand () & 0xffff) / 16384.0 > PROB_FORWARD)
	      {
		/*
		 * recycle receive buffer
		 */
		gm_provide_receive_buffer (port_g,
					   buffer,
					   gm_ntoh_u8 (event->recv.size),
					   GM_LOW_PRIORITY);
	      }
	    else
	      {
		/*
		 * forward the message to a random target
		 */
		int target;

		if (!alloc_high_send_token ())
		  {
		    /* memset (buffer, 0,
		       gm_max_length_for_size (event->recv.size)); */

		    gm_provide_receive_buffer (port_g, buffer,
					       gm_ntoh_u8 (event->recv.size),
					       GM_LOW_PRIORITY);
		    break;
		  }

		target = (unsigned int) gm_rand () % (unsigned int) num_nodes;

		buffer->type = gm_hton_u32 (test_forward_type);

		/*
		 * fill message with known values
		 */
		if (verify)
		  {
		    fill_buffer ((char *) buffer->payload,
				 payload_len (len),
				 (char) out_fw_seq[target]);
		    buffer->seq_no = gm_hton_u32 (out_fw_seq[target]);
		    out_fw_seq[target]++;
		  }

		gm_always_assert (target >= 0 && target < num_nodes);
		gm_assert (len <= gm_max_length_for_size (size));

		buffer->sender = gm_hton_u32 ((gm_u32_t) this_node_id);

		gm_assert (gm_ntoh_u32 (buffer->type) == test_forward_type);
		send_to_peer (port_g, buffer, size, len,
			      GM_HIGH_PRIORITY, node_id[target]);

		total_sent++;
		total_senta[target]++;
		bytes_sent += len;
#ifdef PRINT_LOTS
		bytes_senta[target] += len;
#endif
	      }
	  }
	  break;

	case GM_HIGH_PEER_RECV_EVENT:
	case GM_FAST_HIGH_PEER_RECV_EVENT:

	  /* Handle the reception of forwarded packets */

	  {
	    test_message_t *p;
	    int size = gm_ntoh_u8 (event->recv.size);
	    int len = gm_ntoh_u32 (event->recv.length);
	    int sender = gm_ntoh_u16 (event->recv.sender_node_id);
	    int seq;
	    int ptype;

	    total_recvd++;
	    bytes_recvd += len;
	    total_recvda[get_local_id (sender)]++;
#ifdef PRINT_LOTS
	    bytes_recvda[get_local_id (sender)] += len;
#endif
	    if (GM_RECV_EVENT_TYPE (event) == GM_FAST_HIGH_PEER_RECV_EVENT)
	      {
		p = (test_message_t *) gm_ntohp (event->recv.message);
	      }
	    else
	      {
		p = (test_message_t *) gm_ntohp (event->recv.buffer);
	      }

	    ptype = gm_ntoh_u32 (p->type);
	    gm_always_assert (ptype == test_forward_type);

	    /*
	     * verify message contents
	     */
	    if (verify)
	      {
		seq = gm_ntoh_u32 (p->seq_no);

		if (1 && (in_fw_seq[get_local_id (sender)] != seq))
		  {

		    TEST_PRINT (2, ("high seqerr; size %d, len %d, "
				    "sender %d, buf %p;"
				    " exp 0x%x, got 0x%x\n",
				    size, len, sender, p,
				    in_fw_seq[get_local_id (sender)], seq));

		    /*
		       for (i = 0; i < len; i++)
		       TEST_PRINT (2, ("%02x ",
		       ((int) ((char *) p)[i]) & 0xff));
		       TEST_PRINT (2, ("\n"));
		     */
		    errs[get_local_id (sender)]++;
		    if (errs[get_local_id (sender)] > 30)
		      die ();

		  }
		else
		  {
		    if (verify_buffer ((char *) p->payload, payload_len (len))
			!= 0)
		      {
			TEST_PRINT (2, ("data error; size = %d, len = %d, "
					"p = %p\n", size, len, p));
			errs[get_local_id (sender)]++;
			if (errs[get_local_id (sender)] > 30)
			  die ();
		      }
		  }
		in_fw_seq[get_local_id (sender)]++;
	      }

	    /*
	     * recycle receive buffer
	     */
	    gm_provide_receive_buffer (port_g,
				       gm_ntohp (event->recv.buffer),
				       size, GM_HIGH_PRIORITY);
	  }
	  break;

	case GM_NO_RECV_EVENT:
	  break;

	case GM_ALARM_EVENT:
	  gm_unknown (port_g, event);
	  if (!got_alarm)
	    break;
	  got_alarm = 0;

	  for (i = 1; i <= num_nodes; i++)
	    {
	      if (errs[i] != 0)
		{
		  TEST_PRINT (1, ("%d: in = %d  out = %d  in fwd = %d  "
				  "out fwd = %d  errs = %d %s\n",
				  i, in_seq[i], out_seq[i], in_fw_seq[i],
				  out_fw_seq[i], errs[i],
				  (i == (int) this_node_id) ? "(*)" : ""));
		}
	    }
	  TEST_PRINT (1, ("Msgs sent = %.0f  Msgs received = %.0f.\n",
			  (float) total_sent, (float) total_recvd));
	  TEST_PRINT (1, ("MBytes sent = %.0f.  MBytes received = %.0f.\n",
			  (float) (bytes_sent / (1024 * 1024)),
			  (float) (bytes_recvd / (1024 * 1024))));
	  TEST_PRINT (1,
		      ("MB/s sent = %3.3f.  MB/s received = %3.3f.\n",
		       ((float) ((int) bytes_sent - (int) old_bytes_sent) /
			(float) (GM_NWAY_DELAY * 1024 * 1024)),
		       ((float) ((int) bytes_recvd - (int) old_bytes_recvd) /
			(float) (GM_NWAY_DELAY * 1024 * 1024))));

	  /* PRINT Messages sent and received */
	  TEST_PRINT (1, ("Msgs sent, to hosts:\n"));

	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, (" %12s", host_name[k]));
	    }
	  TEST_PRINT (1, ("\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, ("%13.0f", (float) total_senta[k]));
	    }
	  TEST_PRINT (1, ("\n\n"));

	  TEST_PRINT (1, ("Msgs received, from hosts:\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, (" %12s", host_name[k]));
	    }
	  TEST_PRINT (1, ("\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, ("%13.0f", (float) total_recvda[k]));
	    }
	  TEST_PRINT (1, ("\n\n"));

#ifdef PRINT_LOTS
	  /* PRINT MBytes sent to and received from each host */
	  TEST_PRINT (1, ("MBytes sent, to hosts:\n"));

	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, (" %12s", host_name[k]));
	    }
	  TEST_PRINT (1, ("\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1,
			  ("%13.0f",
			   (float) (bytes_senta[k] / (1024 * 1024))));
	    }
	  TEST_PRINT (1, ("\n\n"));

	  TEST_PRINT (1, ("Msgs received, from hosts:\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, (" %12s", host_name[k]));
	    }
	  TEST_PRINT (1, ("\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1,
			  ("%13.0f",
			   (float) (bytes_recvda[k] / (1024 * 1024))));
	    }
	  TEST_PRINT (1, ("\n\n"));

	  /* PRINT MB/s sent to and received from each host */
	  TEST_PRINT (1, ("MB/s sent, to hosts:\n"));

	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, (" %12s", host_name[k]));
	    }
	  TEST_PRINT (1, ("\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1,
			  ("%13.3f",
			   ((float)
			    ((int) bytes_senta[k] -
			     (int) old_bytes_senta[k]) / (float) (DELAY *
								  1024 *
								  1024))));
	      old_bytes_senta[k] = bytes_senta[k];
	    }
	  TEST_PRINT (1, ("\n\n"));

	  TEST_PRINT (1, ("MB/s received, from hosts:\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1, (" %12s", host_name[k]));
	    }
	  TEST_PRINT (1, ("\n"));
	  for (k = 0; k < num_nodes; k++)
	    {
	      TEST_PRINT (1,
			  ("%13.3f",
			   ((float)
			    ((int) bytes_recvda[k] -
			     (int) old_bytes_recvda[k]) / (float) (DELAY *
								   1024 *
								   1024))));
	      old_bytes_recvda[k] = bytes_recvda[k];
	    }
	  TEST_PRINT (1, ("\n\n"));

	  gm_stime = ((gm_s64_t) gm_ticks (port_g)) / 2e6;	/* 2 million ticks per second */
	  TEST_PRINT (1, ("Slept %.0f time(s).  Last printed status %3.2f "
			  "seconds ago.\n\n",
			  (float) sleeps, gm_stime - oldstime));
	  oldstime = gm_stime;
#endif

	  old_bytes_sent = bytes_sent;
	  old_bytes_recvd = bytes_recvd;

	  gm_set_alarm (port_g, &my_alarm,
			(unsigned int) (GM_NWAY_DELAY * TIME_CALIBRATION),
			set_got_alarm, 0);
	  break;

	  /*
	   * should never happen
	   */
	default:
	  gm_unknown (port_g, event);
	  break;
	}
    }

  TEST_PRINT (1, ("Test completed.\n"));

  gm_close (port_g);
  port_g = NULL;

  TEST_PRINT (1, ("GM deinitialized.\n"));

  die ();

  return 0;
}

/****************************************************************
 * Entry point
 ****************************************************************/

#define GM_MAIN gm_nway
#include "gm_main.h"

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
