/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1999 by Myricom, Inc.                                   *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

/*************************************************************************/
/********* WARNING - this is unsupported code ****************************/
/*************************************************************************/

#include "gm.h"
#include <stdio.h>

/****************************************************************
 * local variables
 ****************************************************************/

static unsigned int target_node_id = GM_NO_SUCH_NODE_ID;
static unsigned long usecs = 300000;
static int max_len = 4096;
static int max_size = 12;
static int min_len = 8;
static int min_size = 12;
static int port_num = 2;
static int stride = 128;
static int done;
static struct gm_alarm alarm;

static struct gm_port *port;
static void *message;
static unsigned long recv_cnt;

/****************************************************************
 * funtions
 ****************************************************************/

/* alarm callback function to set the DONE global */

static void
set_done (void *arg)
{
  done = 1;
}

/* Until DONE is set, receive packets and reply to them, performing a
   ping-pong test. */

void
ping_pong_until_done ()
{
  union gm_recv_event *event;

  /* ping-pong any received message, catching sent events to avoid
     an error message. */

  while (!done)
    {
      event = gm_receive (port);
      switch (GM_RECV_EVENT_TYPE (event))
	{
	case GM_SENT_EVENT:
	  break;
	case GM_PEER_RECV_EVENT:
	case GM_FAST_PEER_RECV_EVENT:
	  gm_send_to_peer (port,
			   message,
			   gm_ntohc (event->recv.size),
			   gm_ntohl (event->recv.length),
			   GM_LOW_PRIORITY,
			   gm_ntohs (event->recv.sender_node_id));
	  gm_provide_receive_buffer (port,
				     gm_ntohp (event->recv.buffer),
				     gm_ntohc (event->recv.size),
				     GM_LOW_PRIORITY);
	  recv_cnt++;
	  break;
	default:
	  gm_unknown (port, event);
	}
    }

  /* wait up to .1 seconds to receive a packet and then return without
     replying to it. */

  done = 0;
  gm_set_alarm (port, &alarm, 100000, set_done, &done);
  while (!done)
    {
      event = gm_receive (port);
      switch (GM_RECV_EVENT_TYPE (event))
	{
	case GM_SENT_EVENT:
	  break;
	case GM_PEER_RECV_EVENT:
	case GM_FAST_PEER_RECV_EVENT:
	  gm_provide_receive_buffer (port,
				     gm_ntohp (event->recv.buffer),
				     gm_ntohc (event->recv.size),
				     GM_LOW_PRIORITY);
	  return;
	default:
	  gm_unknown (port, event);
	}
    }
  gm_cancel_alarm (&alarm);
}

static int
gm_lat (int argc, char **argv)
{
  gm_status_t status;
  int size, i, len;
  char target_host_name[33] = "";
  char *program_name;
  int num_recv_buffers[] = {
    0, 0, 0, 32, 16, 8, 4, 2,
    2, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2
  };

  /****************
   * parse arguments
   ****************/

  program_name = *argv;
  argv++;
  while (*argv)
    {
      if (sscanf (*argv, "--min-size=%i", &min_size))
	argv++;
      else if (sscanf (*argv, "--max-size=%i", &max_size))
	argv++;
      else if (sscanf (*argv, "--min-len=%i", &min_len))
	argv++;
      else if (sscanf (*argv, "--max-len=%i", &max_len))
	argv++;
      else if (sscanf (*argv, "--port-num=%i", &port_num))
	argv++;
      else if (sscanf (*argv, "--stride=%i", &stride))
	argv++;
      else if (sscanf (*argv, "--target=%32s", target_host_name))
	argv++;
      else if (sscanf (*argv, "--usecs=%li", &usecs))
	argv++;
      else
	{
	  fprintf (stderr, "Unrecognized argument: \"%s\"\n", *argv);
	  fprintf (stderr,
		   "Syntax: %s [options]\n"
		   "Valid options include:\n"
		   "  --min-size=%%i\n"
		   "  --max-size=%%i\n"
		   "  --min-len=%%i\n"
		   "  --max-len=%%i\n"
		   "  --port-num=%%i\n"
		   "  --stride=%%i\n"
		   "  --target=%%32s\n"
		   "  --usecs=%%li\n"
		   "Omitting the \"--target=\" option causes"
		   " a reply server to be started.\n", program_name);
	  return 1;
	}
    }

  /****************
   * initialize
   ****************/

  status = gm_init ();
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not initialize GM", status);
      goto abort_with_nothing;
    }
  status = gm_open (&port, 0, port_num, "GM latency test.",
		    GM_API_VERSION_1_0);
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not open GM port", status);
      goto abort_with_init;
    }
  message = gm_dma_malloc (port, gm_max_length_for_size (max_size));
  if (!message)
    {
      gm_perror ("could not allocate send buffer", GM_OUT_OF_MEMORY);
      goto abort_with_port;
    }
  for (size = min_size; size <= max_size; size++)
    for (i = 0; i < num_recv_buffers[size]; i++)
      {
	void *buffer = gm_dma_malloc (port, gm_max_length_for_size (size));
	if (!buffer)
	  {
	    gm_perror ("could not allocate recv buffer.", GM_OUT_OF_MEMORY);
	    goto abort_with_message;
	  }
	gm_provide_receive_buffer (port, buffer, size, GM_LOW_PRIORITY);
      }
  target_node_id = gm_host_name_to_node_id (port, target_host_name);
  gm_initialize_alarm (&alarm);

  /****************
   * manage the test or act as a reply server
   ****************/

  if (target_node_id != GM_NO_SUCH_NODE_ID)
    {
      /* this program sends the first packet and reports all statistics */

      for (size = min_size; size <= max_size; size++)
	{
	  for (len = min_len;
	       len <= (int) gm_max_length_for_size (size); len += stride)
	    {
	      done = 0;
	      gm_set_alarm (port, &alarm, usecs, set_done, 0);
	      gm_send_to_peer (port, message, size, len, GM_LOW_PRIORITY,
			       target_node_id);
	      ping_pong_until_done ();
	      if (recv_cnt)
		gm_printf ("%d", usecs / recv_cnt);
	    }
	}
      gm_printf ("\n");
    }
  else
    {
      /* This program merely replies to each packet received. */

      printf ("Latency test server is now running.\n");
      ping_pong_until_done ();	/* will never return, since we did not
				   set an alarm. */
    }

  /****************
   * error handling
   ****************/

abort_with_message:
  gm_dma_free (port, message);
abort_with_port:
  gm_close (port);
  /* should free recv buffers here, but we didn't keep track of them. */
abort_with_init:
  gm_finalize ();
abort_with_nothing:
  return status;
}

#define GM_MAIN gm_lat
#include "gm_main.h"

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
