/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: finucane@myri.com */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "gm.h"

#  define insist(e) gm_always_assert (e)

/************
 * Dual type -- used as key for test
 ************/

struct dual 
{
  void *a;
  void *b;
};

static long compare_duals (void *a, void *b)
{
  return gm_memcmp (a, b, sizeof (struct dual));
}

static unsigned long hash_dual (void *a)
{
  return gm_crc (a, sizeof (struct dual));
}

/************
 * Triple type -- used as data for test.
 ************/

struct triple
{
  void *a;
  void *b;
  void *c;
};

/************************************************************************/

static long my_random (int max)
{
  int n = rand ();
  double d = (double) n / (double) RAND_MAX;
  int a = (int) (d * max);

  if (a >= max)
    a = max;

  if (a < 0)
    a = 0;
  
  return a;
}

void*pointers [1000];
unsigned long alloc_count;
unsigned long lookaside_count;

static void *look_test (struct gm_lookaside*t, int n)
{
  int i;
  void *p;
  
  for (i = 0; i < n; i++)
  {
    pointers[i] = gm_lookaside_alloc (t);
    insist (pointers[i]);
  }
  p = gm_lookaside_alloc (t);
  alloc_count++;
  
  for (i = 0; i < n; i++)
    gm_lookaside_free (pointers[i]);
  return p;
}


static void look_tests (void)
{
  
  struct gm_lookaside *a;
  struct gm_lookaside *b;
  int n, m, i;

  alloc_count = 0;
  lookaside_count = 0;
    
  gm_printf ("a\n");
  for (n = 1; n < 100; n*=2)
  {
    for (m = 1; m < 1000; m++)
    {
      a = gm_create_lookaside (m, n);
      b = gm_create_lookaside (n, m);
      lookaside_count++;
      
      insist (a && b);
      
      look_test (a, n);
      look_test (b, m);

      gm_destroy_lookaside (a);
      gm_destroy_lookaside (b);
      
    }
  }

  gm_printf ("b\n");
  for (i= 0; i < 10*1000; i++)
  {
    n = 1 + my_random (100);
    m = 1 + my_random (1000);

    fflush (stdout);

    a = gm_create_lookaside (m, n);
    insist (a);
    lookaside_count++;
    
    look_test (a, n*2);
    gm_destroy_lookaside (a);
  }

  gm_printf ("c\n");
  for (m = 1; m < 1000*1000; m*=2)
  {
    a = gm_create_lookaside (m, 2);
    insist (a);
    lookaside_count++;
    
    look_test (a, 2);
    gm_destroy_lookaside (a);
  }
  
  gm_printf ("d\n");
  for (n = 1; m < 1000*1000; n*=2)
  {
    a = gm_create_lookaside (2, n);
    lookaside_count++;

    look_test (a, n);
    gm_destroy_lookaside (a);
  }

  gm_printf ("3\n");
  gm_printf ("gm_create_lookaside:%ld\ngm_lookaside_alloc:%ld\n",
	  lookaside_count, alloc_count);
  

}

static void create_hashes (struct gm_hash**a,
			   struct gm_hash**b,
			   struct gm_hash**c,
			   struct gm_hash**d,
			   struct gm_hash**e)
{
  
  *a = gm_create_hash (gm_hash_compare_ints,
		      gm_hash_hash_int,
		      sizeof (int),
		      sizeof (int),
		      100,
		      0);
  
  insist (*a);
  
  *b = gm_create_hash (gm_hash_compare_longs,
		      gm_hash_hash_long,
		      sizeof (long),
		      sizeof (long),
		      100,
		      0);
  insist (*b);
  
  *c = gm_create_hash (gm_hash_compare_ptrs,
		      gm_hash_hash_ptr,
		      0,
		      0,
		      100,
		      0);
  insist (*c);
  
  *d = gm_create_hash (gm_hash_compare_strings,
		      gm_hash_hash_string,
		      0,
		      0,
		      100,
		      0);
  insist (*d);
  
  *e = gm_create_hash (compare_duals,
		      hash_dual,
		      sizeof (struct dual),
		      sizeof (struct triple),
		      0,
		      0);
  insist (*d);
}

static void destroy_hashes (struct gm_hash*a,
			    struct gm_hash*b,
			    struct gm_hash*c,
			    struct gm_hash*d,
			    struct gm_hash*e)
{
  gm_destroy_hash (a);
  gm_destroy_hash (b);
  gm_destroy_hash (c);
  gm_destroy_hash (d);
  gm_destroy_hash (e);
}

static char *str_dup (char*s)
{
  return strcpy (gm_malloc (strlen (s) + 1), s);
}

char str [300];
  
static void hash_tests (char*fn)
{
  FILE*fp;
  char*s;
  gm_status_t status;
  int line_count, i;
  char**pa;
  char*p;
  struct gm_hash*a;
  struct gm_hash*b;
  struct gm_hash*c;
  struct gm_hash*d;
  struct gm_hash*e;
  long seed;
  
  if (!fn)
    {
      fprintf (stderr, "No input file specified.\n");
      gm_exit (GM_INVALID_PARAMETER);
    }

  create_hashes (&a, &b, &c, &d, &e);
  destroy_hashes (a, b, c, d, e);
  create_hashes (&a, &b, &c, &d, &e);

  fp = fopen (fn, "r");
  if (!fp)
    {
      fprintf (stderr, "Could not open file \"%s.\"", fn);
      gm_exit (GM_FAILURE);
    }

  line_count = 0;
  
  gm_printf ("Counting lines.\n");
  while ((s = fgets (str, 256, fp)) != 0)
    line_count++;

  pa = malloc (sizeof (char*) * line_count);
  insist (pa);
  
  rewind (fp);
  i = 0;
  
  gm_printf ("Inserting %d lines into hash tables.\n", line_count);
  while ((s = (char*)fgets (str, 256, fp)) != 0)
  {
    char *found;

    pa[i] = str_dup (s);
    insist (pa[i]);

    status = gm_hash_insert (d, pa[i], pa[i]);
    insist (status == GM_SUCCESS);

    found = gm_hash_find (d, pa[i]);
    insist (found);
    insist (!strcmp (found, pa[i]));

    status = gm_hash_insert (c, pa[i], pa[i]);
    insist (status == GM_SUCCESS);
    insist (gm_hash_find (c, pa[i]) == pa[i]);

    i++;
  }

  gm_printf ("Verifying hash tables.\n");
  for (i = 0; i < line_count; i++)
  {
    insist (pa[i]);

    p = gm_hash_find (d, pa[i]);
    insist (p);
    insist (!strcmp (p, pa[i]));

    insist (gm_hash_find (c, pa[i]) == pa[i]);
  }

  gm_printf ("Removing lines from hash tables.\n");
  insist (i == line_count);
  for (i = 0; i < line_count; i++)
  {
    p = gm_hash_find (d, pa[i]);
    insist (p);
    insist (!strcmp (p, pa[i]));
    p = gm_hash_remove (d, p);
    insist (!strcmp (p, pa[i]));


    p = gm_hash_find (c, pa[i]);
    insist (p == pa[i]);
    p = gm_hash_remove (c, p);
    insist (p == pa[i]);
  }

  gm_printf ("Destroying hash tables c and d.\n");
  gm_destroy_hash (d);
  gm_destroy_hash (c);

  gm_printf ("Filling hash tables a and b\n");
  for (i = 0; i < 1000*1000; i++)
  {
    int the_int = i, the_int_pp = i+1, *_int;
    long the_long = i, the_long_pp = i+1, *_long;
    
    status = gm_hash_insert (a, &the_int, &the_int_pp);
    insist (status == GM_SUCCESS);
    _int = (int *) gm_hash_find (a, &the_int);
    insist (_int);
    insist (*_int == the_int_pp);

    status = gm_hash_insert (b, &the_long, &the_long_pp);
    insist (status == GM_SUCCESS);
    _long = (long *) gm_hash_find (b, &the_long);
    insist (_long);
    insist (*_long = the_long_pp);
  }
  
  gm_printf ("Removing entries from hash tables a and b.\n");
  for (i = 0; i < 1000*1000; i++)
  {
    int the_int = i, *_int;
    long the_long = i, *_long;
    
    _int = (int *) gm_hash_find (a, &the_int);
    insist (_int);
    insist (the_int + 1	== *_int);
    _int = (int *) gm_hash_remove (a, &the_int);
    insist (_int);
    insist (the_int + 1	== *_int);
  
    _long = (long *) gm_hash_find (b, &the_long);
    insist (_long);
    insist (the_long + 1 == *_long);
    _long = (long *) gm_hash_remove (b, &the_long);
    insist (_long);
    insist (the_long + 1 == *_long);
  }

  gm_printf ("Scanning for erroneous leftovers in hash tables a and b.\n");
  for (i = 0; i < 1000 * 1000; i++)
  {
    int the_int = i;
    
    insist (!gm_hash_find (a, &the_int));
    insist (!gm_hash_remove (a, &the_int));

    insist (!gm_hash_find (b, &the_int));
    insist (!gm_hash_remove (b, &the_int));
  }

  gm_printf ("Testing large keys and larger data, with copies.\n");
  seed = (long) time (0) & 0xffff;
  gm_printf ("seed = %ld\n", seed);
  srand (seed);
  
  
  gm_printf ("Inserting.\n");
  for (i = 0; i < 1000 * 1000; i++)
  {
    struct dual dual;
    struct triple triple, *_triple;
    
    dual.a = (void *) my_random (RAND_MAX);
    dual.b = (void *) my_random (RAND_MAX);
    
    triple.a = (void *) ((long) dual.a + (long) dual.b);
    triple.b = (void *) ((long) dual.a - (long) dual.b);
    triple.c = (void *) ((long) dual.a ^ (long) dual.b);
    
    status = gm_hash_insert (e, &dual, &triple);
    insist (status == GM_SUCCESS);

    _triple = gm_hash_find (e, &dual);
    insist (_triple);
    insist (_triple != &triple);
    insist (!memcmp (&triple, _triple, sizeof (triple)));
  } 

  gm_printf ("Removing.\n");
  srand (seed);
  for (i = 0; i < 1000*1000; i++)
  {
    struct dual dual;
    struct triple triple, *_triple;
    
    dual.a = (void *) my_random (RAND_MAX);
    dual.b = (void *) my_random (RAND_MAX);
    
    triple.a = (void *) ((long) dual.a + (long) dual.b);
    triple.b = (void *) ((long) dual.a - (long) dual.b);
    triple.c = (void *) ((long) dual.a ^ (long) dual.b);
    
    _triple = gm_hash_find (e, &dual);
    insist (_triple);
    insist (!memcmp (&triple, _triple, sizeof (triple)));
    insist (_triple != &triple);
    
    _triple = gm_hash_remove (e, &dual);
    insist (_triple);
    insist (!memcmp (&triple, _triple, sizeof (triple)));
    insist (_triple != &triple);
  }

  gm_printf ("Checking for leftovers.\n");
  srand (seed);
  for (i = 0; i < 1000*1000; i++)
  {
    struct dual dual;
    struct triple *_triple;
    
    dual.a = (void *) my_random (RAND_MAX);
    dual.b = (void *) my_random (RAND_MAX);
    
    _triple = gm_hash_find (e, &dual);
    insist (!_triple);
    
    _triple = gm_hash_remove (e, &dual);
    insist (!_triple);
  }

  gm_destroy_hash (a);
  gm_destroy_hash (b);
  gm_destroy_hash (e);
}

static int
gm_hash (int argc, char*argv[])
{
  int fails = 0;

  if (argc != 2)
    {
      gm_printf ("Syntax: %s <textfile>\n", argv[0]);
      gm_exit (GM_INVALID_PARAMETER);
    }
  
  gm_init ();
  
  look_tests ();
  hash_tests (argv[1]);

{
 struct gm_hash * h;
 int data = 1234;
 int *pdata;
 void * key = (void*)0x112233;

 h = gm_create_hash (gm_hash_compare_ptrs,gm_hash_hash_ptr,
                                 0 /*key_len */, 4 /* data_len */, 0, 0);
 

 
 gm_hash_insert(h,key,&data);
 pdata = gm_hash_find(h,key);
 if (*pdata != data) {
   fails++;
   printf("\n******* Loic/Alice test fails *******\n");
 }
}

  gm_finalize ();

  if (!fails) 
     gm_printf ("test passed\n");
  return 0;
}

#define GM_MAIN gm_hash
#include "gm_main.h"
