/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1999 by Myricom, Inc.                                   *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* Description (should be put in a --help display):
 *
 *   gm_debug always displays its estimate of DMA rates.  Then, if
 *   GM_ENABLE_ERROR_COUNTERS was set at build time, it will display some
 *   LANai counters; the display will be longer if GM_ENABLE_DEBUG_COUNTERS
 *   was set.  Following this, any additional output as specified by the
 *   flags listed below will be generated.
 *
 * Flags:
 *
 *   -B n    Use board (unit) n     [default: -B 0]
 *   -C n    Dump all connections from 1 through n
 *   -c n    Dump specific connection n
 *   -i      Print miscellaneous internal debugging info
 *   -L      Suppress LANai rates   [default: print them]
 *   -l      Print a report of LANai dispatches from the dispatch log
 *           (requires GM_LOG_DISPATCHES)
 *   -p n    Use port n             [default: -p 0]
 *   -R      Repeat all the output in a loop
 *   -s n    Sleep for n seconds between each repetition
 *                                  [default: don't sleep]
 *   -t      Display the time of day before each repetition
 *
 *   --abort Instead of normal termination, call abort(), which should
 *           produce a 'core' file
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

/* This program uses undocumented software interfaces that are not
   intended for use in code written outside Myricom.  Hence the
   following undocumented includes. */

#include "gm_internal.h"
#include "gm_debug_counters.h"
#include "gm_enable_error_counters.h"
#include "gm_enable_debug_counters.h"
#include "gm_enable_packet_counters.h"
#include "gm_error_counters.h"

/* Following macros needed by GM_ERROR_COUNTERS and GM_DEBUG_COUNTERS */
#define GM_ERROR_CNT(name,desc) _P_NAMED ("", name, name ## _error_cnt);
#define GM_DEBUG_CNT(name,desc) _P_NAMED ("", name, name ## _debug_cnt);

#define _P_NAMED(prefix, name, fld) do {								\
	printf (prefix "  %-40s %12d\n",									\
            #name, gm_ntoh_u32 (globals->fld));							\
  } while (0)

#define P(prefix, fld) _P_NAMED (prefix, fld, fld)

/* this 1024 was hard-coded in drivers/gm.c */
/* don't really want to get all the data from a card with 2Meg+ */
#define MY_MAX_NODE	(3*1024)

unsigned int space[(sizeof(struct gm_lanai_globals) +
		       ((MY_MAX_NODE+1) * sizeof(struct gm_connection))) / sizeof(int)];

struct gm_lanai_globals *globals = (struct gm_lanai_globals *) space;
gm_u64_t current_rtc;

gm_port_t *port;

static void
print_debug_info(void)
{
  printf ("GM version is %s,\n build ID is \"%s.\"\n",
          _gm_version, _gm_build_id);
#ifdef GM_DEBUG
  printf ("GM_DEBUG = %d\n", GM_DEBUG);
#else
  printf ("GM_DEBUG is undefined\n");
#endif
  printf ("Print level is %d\n", GM_PRINT_LEVEL);
  printf ("Sized types:\n"
          "    gm_s64_t  %d\n"
          "    gm_up_t   %d\n"
          "    gm_dp_t   %d\n"
          "    void *    %d\n"
          "    long      %d\n",
          (int)sizeof(gm_s64_t), (int)sizeof(gm_up_t), (int)sizeof(gm_dp_t),
          (int)sizeof(void *),   (int)sizeof(long));
  printf ("Some defines:\n"
          "    GM_SIZEOF_VOID_P     %d\n"
          "    GM_SIZEOF_UP_T       %d\n"
          "    GM_SIZEOF_DP_T       %d\n"
          "    GM_NUM_PORTS         %d\n"
          "    GM_DMA_GRANULARITY   %d\n"
          "    GM_RDMA_GRANULARITY  %d\n"
          "    GM_IP_MTU            %d\n",
          GM_SIZEOF_VOID_P, GM_SIZEOF_UP_T,     GM_SIZEOF_DP_T,
          GM_NUM_PORTS,     GM_DMA_GRANULARITY, GM_RDMA_GRANULARITY,
          GM_IP_MTU);
  printf ("Some structure sizes:\n"
          "    gm_port_protected_lanai_side_t  %d\n"
          "    gm_pte_t                        %d\n"
          "    gm_page_hash_cache_t            %d\n",
          (int)sizeof(gm_port_protected_lanai_side_t),
          (int)sizeof(gm_pte_t),
          (int)sizeof(gm_page_hash_cache_t));
  printf ("    gm_cached_pte_t                 %d\n"
          "    gm_recv_queue_slot_t            %d\n"
          "    struct gm_lanai_globals         %d (0x%x)\n"
          "    struct gm_connection            %d\n",
          (int)sizeof(gm_cached_pte_t),
          (int)sizeof(gm_recv_queue_slot_t),
          (int)sizeof(struct gm_lanai_globals),
          (int)sizeof(struct gm_lanai_globals),
          (int)sizeof(struct gm_connection));

#ifdef GM_SUPPORT_L4
  printf ("GM_SUPPORT_L4 = %d\n", GM_SUPPORT_L4);
#else
  printf ("GM_SUPPORT_L4 is undefined\n");
#endif

#ifdef GM_SUPPORT_L7
  printf ("GM_SUPPORT_L7 = %d\n", GM_SUPPORT_L7);
#else
  printf ("GM_SUPPORT_L7 is undefined\n");
#endif

#ifdef GM_SUPPORT_L8
  printf ("GM_SUPPORT_L8 = %d\n", GM_SUPPORT_L8);
#else
  printf ("GM_SUPPORT_L8 is undefined\n");
#endif

#ifdef GM_SUPPORT_L9
  printf ("GM_SUPPORT_L9 = %d\n", GM_SUPPORT_L9);
#else
  printf ("GM_SUPPORT_L9 is undefined\n");
#endif

#ifdef GM_SUPPORT_PCI_REV_1
  printf ("GM_SUPPORT_PCI_REV_1 = %d\n", GM_SUPPORT_PCI_REV_1);
#else
  printf ("GM_SUPPORT_PCI_REV_1 is undefined\n");
#endif

#ifdef GM_SUPPORT_PCI_REV_2
  printf ("GM_SUPPORT_PCI_REV_2 = %d\n", GM_SUPPORT_PCI_REV_2);
#else
  printf ("GM_SUPPORT_PCI_REV_2 is undefined\n");
#endif

#ifdef GM_SUPPORT_PCI_REV_3
  printf ("GM_SUPPORT_PCI_REV_3 = %d\n", GM_SUPPORT_PCI_REV_3);
#else
  printf ("GM_SUPPORT_PCI_REV_3 is undefined\n");
#endif

#ifdef GM_SUPPORT_SBUS
  printf ("GM_SUPPORT_SBUS = %d\n", GM_SUPPORT_SBUS);
#else
  printf ("GM_SUPPORT_SBUS is undefined\n");
#endif

  printf ("page size is %d\n", (int)(GM_PAGE_LEN));

  printf ("Some derived values:\n"
          "    GM_RECV_QUEUE_SLOTS_PER_PAGE %d\n"
          "    GM_NUM_RECV_QUEUE_SLOTS      %d\n",
		  (int)(GM_RECV_QUEUE_SLOTS_PER_PAGE),
          (GM_NUM_RECV_QUEUE_SLOTS));
}


static void
print_dma_rates(void)
{
  unsigned long rate1=0,
                rate2=0;
  int           i;

  printf ("\n");
  for (i = 0; i < 4; i++) {
	rate1 = rate2 = 0;
	
	/* shift >>1 is divide by 2 since RTC ticks at 2MHz */
	if (gm_ntoh_u32(globals->e2l_time[i])>>1) {
		rate1 = (GM_PAGE_LEN)/(gm_ntoh_u32(globals->e2l_time[i])>>1);
	}
	if (gm_ntoh_u32(globals->l2e_time[i])>>1) {
		rate2 = (GM_PAGE_LEN)/(gm_ntoh_u32(globals->l2e_time[i])>>1);
	}

	printf ("DMA rate for %lu Byte transfers (%dbit / %dMHz bus)\n",
			GM_PAGE_LEN,
			gm_ntoh_u32(globals->bus_width)?gm_ntoh_u32(globals->bus_width):32,
			gm_ntoh_u32(globals->bus_rate)? gm_ntoh_u32(globals->bus_rate):33);
    switch (i) {
	case 0:
		printf ("1st: 8 pages from bogus sdma pg, 8 to bogus rdma\n");
		break;
	case 1:
		printf ("2nd: sdma a page, rdma a page 8 times\n");
		break;
	case 2:
		printf ("3rd: sdma and rdma to/from alternating pages (coarse grain)\n");
		break;
	case 3:
		printf ("4th: sdma and rdma to/from alternating pages (fine grain)\n");
		break;
    default:
		printf ("CASE %d is BOGUS ******\n", i);
	}

	printf ("\tbus_read  (send) = %lu MBytes/s\n"
		    "\tbus_write (recv) = %lu MBytes/s\n\n",
			rate1, rate2);
  } /* for i */
}



static void
print_log(void)
{
#if GM_LOG_DISPATCHES
	int index;
	int start_slot;
	gm_u32_t prev_time = 0;

	for (index=0; index<GM_LOG_LEN; index++) {
		if (!gm_ntoh_lp(globals->log[index])) {
			index++;
			break;
		}
	}
	if (index>=GM_LOG_LEN) {
		index = 0;
	}
	start_slot = index;
	printf ("record_log (count) = %d\n",
			gm_ntoh_s32 (globals->record_log));
	printf ("log_end = 0x%x, log_start = 0x%x\n",
			gm_ntoh_lp (globals->log_end),
			gm_ntoh_lp(globals->log_slot));

/* this calculation led to problems on some wedged LANai boards */
/*
	start_slot = (GM_LOG_LEN
				  - ((gm_ntoh_lp (globals->log_end)
					  - gm_ntoh_lp (globals->log_slot))
					 / sizeof (globals->log[0])));
	index = start_slot;
*/

	printf("start_slot = 0x%x\n", start_slot);
	printf("Timestamp      Event\n");
	do {
		gm_status_t status;
		gm_u32_t len=0;

		if (gm_ntoh_lp (globals->log[index]) != 0) {
			char str[GM_MAX_FIRMWARE_STRING_LEN+1];
				status = (_gm_get_firmware_string
					  (port, gm_ntoh_lp (globals->log[index]), str, &len));
			if (status == GM_SUCCESS) {
				str[GM_MAX_FIRMWARE_STRING_LEN] = 0;
				if (!prev_time) 
				   prev_time = gm_ntoh_u32(globals->logtime[index]);
				printf ("0x%08x  %s\n",
					gm_ntoh_u32(globals->logtime[index])-prev_time,str);
				prev_time = gm_ntoh_u32(globals->logtime[index]);
			}
			else {
				gm_perror ("firmware string error", status);
			}

		}
		if (++index >= GM_LOG_LEN) {
			index = 0;
		}
	} while (index != start_slot);

	printf("dispatch counts:\n");

	{
		int j;
		for (j=0; j<256; j++) {
			if (gm_ntoh_u32 (globals->dispatch_cnt[j][0])
				|| gm_ntoh_u32 (globals->dispatch_cnt[j][1]))
			{
				printf("%2d:  %9d\t%9d\n",j,
					   gm_ntoh_u32 (globals->dispatch_cnt[j][0]),
					   gm_ntoh_u32 (globals->dispatch_cnt[j][1]));
			}
		}
	}
#endif
}



static void
print_lanai_cts()
{
    printf("First two words of LANai globals: 0x%x, 0x%x\n",
           gm_ntoh_u32 (((gm_u32_n_t *) globals)[0]),
           gm_ntoh_u32 (((gm_u32_n_t *) globals)[1]) );

	printf("Some counters from LANai\n"
		   "\n");
	if (GM_ENABLE_PACKET_COUNTERS) {
		printf("netsend_cnt      %d\n", gm_ntoh_u32 (globals->netsend_cnt));
		printf("netrecv_cnt      %d\n", gm_ntoh_u32 (globals->netrecv_cnt));
	}

	if (GM_ENABLE_ERROR_COUNTERS) {
		printf("error counters\n");
		GM_ERROR_COUNTERS
		printf("\n");
		printf("       packet_sexno %04x %04x\n", 
			   gm_ntoh_s16 (globals->packet_sexno.parts.sesno),
			   gm_ntoh_s16 (globals->packet_sexno.parts.seqno));
		printf("       ack_sexno %04x %04x\n", 
			   gm_ntoh_s16 (globals->ack_sexno.parts.sesno),
			   gm_ntoh_s16 (globals->ack_sexno.parts.seqno));
		printf("\n"
			   "\n");
	}

	if (GM_ENABLE_DEBUG_COUNTERS) {
		printf("debug counters\n");
		GM_DEBUG_COUNTERS
	}

	printf("port counters\n");
	P("", port[2].active_subport_cnt);
}



static void
dump_connection(struct gm_connection *c, int number)
{
	if (c) {
		printf("globals->connection[%d]\n",number);
		printf("\tcrcs = 0x%x\n",gm_ntoh_u8(c->probable_crc_error_cnt));
		printf("\tbadcrc      = 0x%x\n",gm_ntoh_u32(c->bad_crc_cnt));
		printf("\tack_pending = 0x%x\n",gm_ntoh_u8(c->ack_pending));
		printf("\thas_route   = 0x%x\n",gm_ntoh_u8(c->has_route));
		printf("\tsend_sexno  = %04x %04x\n",
		       gm_ntoh_s16(c->send_sexno.parts.sesno),
		       gm_ntoh_s16(c->send_sexno.parts.seqno));
		printf("\tack.sexno   = %04x %04x\n",
			   gm_ntoh_s16(c->ack_packet.sexno.parts.sesno),
			   gm_ntoh_s16(c->ack_packet.sexno.parts.seqno));
		printf("\tclose_sexno = %04x %04x\n",
			   gm_ntoh_s16(c->close_sexno.parts.sesno),
			   gm_ntoh_s16(c->close_sexno.parts.seqno));
		printf("\topen_time   = 0x%lx\n",(long) gm_ntoh_s64(c->open_time));
		printf("\tclose_time  = 0x%lx\n",(long) gm_ntoh_s64(c->close_time));
		printf("\tcurrentRTC  = 0x%lx\n",(long) current_rtc);
		printf("\tack.type subtype = %04x %04x\n",
			   gm_ntoh_u16(c->ack_packet.type),
			   gm_ntoh_u16(c->ack_packet.subtype));
		printf("\tack.target  = 0x%x   ",
			   gm_ntoh_u16(c->ack_packet.target_node_id));
		printf("\tack.sender  = 0x%x\n",
			   gm_ntoh_u16(c->ack_packet.sender_node_id));
	}
}

extern int atoi (const char *);

static int
gm_debug (int argc, char **argv)
{
	int board_num = 0,
        port_num  = 0;
	int i;
	int globals_len;
	int do_abort           = 0,
	    do_print_dma_rates = 1,
        do_print_info      = 0,
        do_print_log       = 0,
        do_print_lanai_cts = 1,
        do_repeat          = 0,
	    do_timestamp       = 0;
	int sleep_secs     = 0;
	int max_connection = 0;
	int one_connection = 0;
	gm_status_t status;
    time_t uct;
	char   *time_string;

/*  Parse run-time flags */

	for (i = 1; i < argc; i++) {
		if (strcmp(argv[i], "-B") == 0) {
			if (i + 1 >= argc) {
				printf("Board number expected after '-B'.\n");
				gm_exit (GM_INVALID_PARAMETER);
			}
			board_num = atoi(argv[++i]);
			if (board_num < 0) {
				printf("bad board number: %d\n", board_num);
				gm_exit(GM_INVALID_PARAMETER);
			}
			continue;
		}
		if (strcmp(argv[i], "-C") == 0) {
			if (i + 1 >= argc) {
				printf("Max connection expected after '-C'.\n");
				gm_exit(GM_INVALID_PARAMETER);
			}
			max_connection = atoi(argv[++i]);
			if (max_connection < 0) {
				printf("bad 'max connection': %d\n", max_connection);
				gm_exit(GM_INVALID_PARAMETER);
			}
			continue;
		}
		if (strcmp(argv[i], "-c") == 0) {
			if (i + 1 >= argc) {
				printf("Connection number expected after '-c'.\n");
				gm_exit(GM_INVALID_PARAMETER);
			}
			one_connection = atoi(argv[++i]);
			if (one_connection <= 0) {
				printf("bad 'one connection': %d\n", one_connection);
				gm_exit(GM_INVALID_PARAMETER);
			}
			continue;
		}
		if (strcmp(argv[i], "-i") == 0) {
			do_print_info = 1;
			continue;
		}
		if (strcmp(argv[i], "-L") == 0) {
            do_print_lanai_cts = 0;
			continue;
		}
		if (strcmp(argv[i], "-l") == 0) {
            if (GM_LOG_DISPATCHES)
			    do_print_log = 1;
            else
			    printf("flag %s ignored because LANai is not logging dispatches\n",
				       argv[i]);
			continue;
		}
		if (strcmp(argv[i], "-p") == 0) {
			if (i + 1 >= argc) {
				printf("Port number expected after '-p'.\n");
				gm_exit(GM_INVALID_PARAMETER);
			}
			port_num = atoi(argv[++i]);
			if (port_num < 0) {
				printf("bad port number: %d\n", port_num);
				gm_exit(GM_INVALID_PARAMETER);
			}
			continue;
		}
		if (strcmp(argv[i], "-R") == 0) {
			/* repeat */
			do_repeat = 1;
			continue;
		}
		if (strcmp(argv[i], "-s") == 0) {
			if (i + 1 >= argc) {
				printf("Sleep secs expected after '-s'.\n");
				gm_exit(GM_INVALID_PARAMETER);
			}
			sleep_secs = atoi(argv[++i]);
			if (sleep_secs < 0) {
				printf("bad sleep seconds: %d\n", sleep_secs);
				gm_exit(GM_INVALID_PARAMETER);
			}
			continue;
		}
		if (strcmp(argv[i], "-t") == 0) {
			/* repeat */
			do_timestamp = 1;
			continue;
		}

		if (strcmp(argv[i], "--abort") == 0) {
			do_abort = 1;
			continue;
		}

		else {
			printf ("unrecognized argument \"%s\".\n"
					"Valid arguments are\n"
					"\t-B <board_number>\n"
					"\t-c <connection_number>\n"
					"\t-C <max_connection_number>\n"
					"\t-i (causes debugging info to be printed)\n"
					"\t-L (suppresses printing of the LANai counters)\n"
					"\t-l (causes log to be printed)\n"
					"\t-p <port_number>\n"
					"\t-R (causes output to repeat in a loop)\n"
					"\t-s <seconds_to_sleep>\n"
					"\t-t (causes timestamp to be printed)\n"
					"\t--abort (causes gm_debug to abort, dumping core)\n"
					, argv[i]);
			gm_exit (GM_INVALID_PARAMETER);
		}
	}

    printf("Opening board %d, port %d\n", board_num, port_num);
	fflush(stdout);				/* just in case we hang in the open */

	status = _gm_open_common (&port, board_num, port_num, "gm_debug",
				  GM_API_VERSION_1_0);
	if (status != GM_SUCCESS) {
		printf    ("gm_debug: _gm_open_common board %d port %d failed:\n",
                   board_num, port_num);
		gm_perror ("          Error status", status);
		gm_exit(status);
	}

    /* This needs to wait until after open, otherwise some globals like
     * GM_PAGE_LEN won't be set.
     */
	if (do_print_info)
		print_debug_info();

	globals_len = sizeof(struct gm_lanai_globals) +
	    (MY_MAX_NODE * sizeof(struct gm_connection));

	if (do_repeat) {
		printf("\nWill run in a loop\n");
	}

	do {
		if (do_timestamp) {
			uct = time(NULL);
			time_string = asctime(localtime(&uct));
			printf("\n---> %s", time_string);
		}

	  	current_rtc = gm_ticks(port);
		status = _gm_get_globals(port, (gm_u8_t *) globals, globals_len);
		if (status != GM_SUCCESS) {
			printf("gm_debug: _gm_get_globals failed\n");
			gm_exit(status);
		}

		/* now we have a copy of the lanai_globals */
		/* use GDB? */

		if (do_print_dma_rates)
		    print_dma_rates();

		if (do_print_log)
			print_log();

        if (do_print_lanai_cts)
			print_lanai_cts();

		if (max_connection) {
			int j;
			for (j=1; j<=max_connection; j++) {
				dump_connection(&globals->connection[j],j);
			}
		}
		if (one_connection) {
				dump_connection(&globals->connection[one_connection],
								one_connection);
		}

/* ********************************************************************* */
/* =========> *//* THIS IS A GOOD LINE FOR A BREAKPOINT */
/* ********************************************************************* */

		if (sleep_secs) {
		  printf("Going to sleep for %d seconds\n", sleep_secs);
		  gm_sleep (sleep_secs);
		}

	} while (do_repeat);

	if (do_abort) {
		printf("do_abort is set - will now 'abort' this program"
               " to create a core file\n\n");
		fflush(stdout);
		abort();
	}

	gm_close(port);

	return(0);
}

#define GM_MAIN gm_debug
#include "gm_main.h"

/*
  This comment holds special emacs settings for this file.

  Local Variables:
  mode:c
  c-file-style:"bsd"
  tab-width:4
  End:
*/
