/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997, 1998, 1999 by Myricom, Inc.                 *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

#ifdef __STDC__
#if GM_OS_VXWORKS
#include "vxWorks.h"
#endif  /* GM_OS_VXWOKRS */
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#ifdef HAVE_UNISTD_H
#include "unistd.h"
#endif

#include "gm.h"
#include "gm_trace.h"
#include "stats.h"
#include "math.h"

#define GM_DEBUG_ALLSIZE 0
#define GM_DEBUG_ALLSIZE_ALARM 0
#define GM_LONG_PACKET_ERROR_PRINT 1

#define PRINTF(a) do {if (GM_DEBUG_ALLSIZE) printf a;} while (0)

/* test parameters */

#ifndef WIN32

#define LEN_INC 8
#define MAX_TEST_LEN(size) (gm_max_length_for_size (size))
#define MAX_TEST_SIZE 12
#define MIN_TEST_LEN 1
#define MIN_TEST_SIZE 12
#define RECV_TOKENS_OF_SIZE 20
#define SEND_BUFFERS 4

#else /* WIN32 */

#define MIN_TEST_SIZE 12
#define MAX_TEST_SIZE 12
#define MIN_TEST_LEN 1
#define MAX_TEST_LEN(size) (gm_max_length_for_size (size))
#define LEN_INC	8
#define SEND_BUFFERS 3
#define RECV_TOKENS_OF_SIZE 6

#endif

/* Possible useful test for setting the number of receive tokens to 0
   to test behaviour when the receiver does not provide receive tokens
   fast enough. */

#if 0
#undef RECV_TOKENS_OF_SIZE
#define RECV_TOKENS_OF_SIZE 1
#endif

/****************************************************************
 * Typedefs
 ****************************************************************/

enum {
  NO,
  YES,
  MAYBE
};

enum { GM_ARBITRARY_RECV_BUFFER_TAG = 42 };

/****************
 * messages
 ****************/

enum message_type
{
  TYPE_BLANK = 0,
  
  TYPE_LEN_MASK = 0xf,
  TYPE_REPLY = 0x10,
  TYPE_VERIFY = 0x20,
  
  TYPE_SEND_MODE_MASK = 0xc0,
  TYPE_SEND_MODE_NONPEER = 0x00,
  TYPE_SEND_MODE_PEER = 0x40,
  TYPE_SEND_MODE_DATAGRAM = 0x80,
  TYPE_SEND_MODE_DATAGRAM_OR_RELIABLE = 0xc0
};

#define MESSAGE_LEN(m) ((m)->type & TYPE_LEN_MASK			\
			? (m)->type & TYPE_LEN_MASK			\
			: gm_ntoh_u32 ((m)->large.length))

typedef union message
{
  gm_u8_t type;
  struct small_message {
    gm_u8_n_t type;
    gm_u8_n_t data[1];
  } small;
  struct large_message {
    gm_u8_n_t type;
    gm_u8_n_t pad[3];
    gm_u32_n_t length;
    gm_u8_n_t data[1];
  } large;
} message_t;

typedef struct message_ref
{
  struct message_ref *next;
  struct message_ref *prev_created;
  char *original_ptr;
  message_t *message;
  int size;
  char is_free;
} message_ref_t;

/****************
 * other
 ****************/

struct test;

typedef void send_func_t (struct test *t,
			  struct message_ref *ref,
			  unsigned int size,
			  unsigned long len,
			  enum message_type basic_type,
			  unsigned int target_node_id,
			  unsigned int target_port_id);
static send_func_t _send_abort;
static send_func_t _send_datagram;
static send_func_t _send_datagram_or_reliable;
static send_func_t _send_to_nonpeer;
static send_func_t _send_to_peer;

typedef struct enqueued_send
{
  struct enqueued_send *next;
  struct test *test;
  message_ref_t *message_ref;
  unsigned int size;
  unsigned long len;
  enum message_type basic_type;
  unsigned int target_node_id;
  unsigned int target_port_id;
  send_func_t *send;
  void (*fill) (struct test *t, gm_u8_n_t *buf, unsigned int len, char seed);
} enqueued_send_t;

typedef void recv_func_t (struct test *t, struct gm_port *p,
			  gm_recv_event_t *e, send_func_t send);
static recv_func_t recv_only;
static recv_func_t recv_and_verify;
static recv_func_t recv_and_reply;
static recv_func_t recv_and_reply_and_verify;
static recv_func_t recv_slave;

typedef struct test
{
  char *points;
  char *remote_host_name;
  char *title;
  char *ylabel;
  enum gm_priority priority;
  int datagrams;
  int do_copy;
  int fast;
  int hysteresis;
  int quiet;
  int rand_seed;
  int randomize;
  int reverse;
  int stats_socket;
  struct stats *stats;
  unsigned int count_per_len;
  unsigned int density;
  unsigned int enable_nack_down_flag;
  unsigned int enable_nack_down_set;
  unsigned int exit_on_error;
  unsigned int length_inc;
  unsigned int max_len;
  unsigned int max_size;
  unsigned int min_len;
  unsigned int min_sends_per_point;
  unsigned int min_size;
  unsigned int offset;
  unsigned int outstanding_send_cnt;
  unsigned int ping_pong;	/* else spray */
  unsigned int port_id;
  unsigned int report_bytes_per_sec; /* else report Mbyte/sec*/
  unsigned int report_latency;	/* else report bandwidth */
  unsigned int send_cnt;
  unsigned int target_node_id;
  unsigned int target_port_id;
  unsigned int timer;
  unsigned int verify;
  unsigned long min_usecs_per_point;
  
  message_ref_t *free_send_message_refs[32];
  /* keep track of the memory allocated so we can free it later*/
  message_ref_t *last_created_message_ref;
  
  FILE *data_file;
  struct 
  {
    FILE *file;
    char *plot_string;
    char *basename;
  } gnuplot;
  
  enqueued_send_t *first_enqueued_send;
  enqueued_send_t *last_enqueued_send;

  struct gm_lookaside *send_record_lookaside;
  struct gm_lookaside *enqueued_send_lookaside;

  unsigned int this_node_id;
  struct gm_port *port;
  int close_called;
  int board;
  struct length_record
  {
    struct length_record *next;
    unsigned long length;
  } *length_stack;
  enum message_type basic_type;
  
  /****
   * handles
   ****/

  send_func_t *_send;
  void (*_sent) (struct gm_port *p, void *c);
  void (*fill_buffer) (struct test *t, gm_u8_n_t *buf, unsigned int len,
		       char seed);
  void (*build_length_stack) (struct test *t, unsigned int);
  recv_func_t *recv;
  void (*start_sending) (struct test *, unsigned int s, unsigned long l);
  void (*test_do) (struct test *t, unsigned int size);
  int (*verify_buffer) (struct test *t, gm_u8_n_t *buf, unsigned int len);
  void (*record_point) (struct test *t, unsigned long len, unsigned int cnt,
			gm_u64_t time);
  void (*report) (struct test *t, unsigned long len);
  gm_recv_event_t *(*gm_receive) (struct gm_port *port);
  
} test_t;

typedef struct send_record
{
  struct send_record *next;
  test_t *test;
  message_ref_t *message_ref;
  unsigned int size;
  unsigned long len;
} send_record_t;

/****************************************************************
 * utility functios
 ****************************************************************/

#if !GM_OS_VXWORKS
/* Microsoft compiler can't convert u64 to double */
static double
u64_to_double (gm_u64_t u64)
{
#ifdef WIN32
  return 2.0 * (double) ((gm_s64_t) (u64 >> 1)) + (int) (u64 & 0x01);
#else
  return (double) u64;
#endif
}
#endif

/****************************************************************
 * Auto-cleanup support functions
 ****************************************************************/

/* like gm_on_exit(), only it cleans up and exits on failure */

static void
on_exit_must_succeed (gm_on_exit_callback_t callback, void *arg)
{
  gm_status_t status;

  status = gm_on_exit (callback, arg);
  if (status != GM_SUCCESS)
    {
      (*callback) (status, arg);
      gm_exit (status);
    }
}

/* gm_on_exit() callback to close a file */

static void
fclose_callback (gm_status_t status, FILE *file)
{
  fclose (file);
}

/* gm_on_exit() callback to close a stats socket */

static void
close_stats_socket_callback (gm_status_t status, int *fd_ptr)
{
  if (GM_ENABLE_STATS_SOCKET)
    {
      socket_close (*fd_ptr);
    }
}

/****************************************************************
 * Message References
 *
 * Message refs are created at init time and kept in a sort of
 * lookaside list, from which they can be alloc's and to which they
 * can be freed.
 ****************************************************************/

static message_ref_t *
create_message_ref (test_t *t, unsigned int size)
{
  message_t *message;
  message_ref_t *ref;
  struct gm_port *port = t->port;
  
  ref = gm_calloc (1, sizeof (message_ref_t));
  if (!ref)
    {
      fprintf (stderr, "memory allocation failed\n");
      goto abort_with_nothing;
    }
  ref->original_ptr = gm_dma_calloc (port, 1, gm_max_length_for_size (size)+GM_PAGE_LEN);
  ref->message 
    = message
    = (message_t *)((((unsigned long)ref->original_ptr + (GM_PAGE_LEN-1))&
		~(GM_PAGE_LEN-1)) + t->offset);
  if (!ref->original_ptr)
    {
      fprintf (stderr, "out of DMAable memory\n");
      goto abort_with_ref;
    }

  ref->size = size;
  gm_always_assert (GM_DMA_ALIGNED (message));

  /* record the ref so we can deallocate it later */
  ref->prev_created = t->last_created_message_ref;
  t->last_created_message_ref = ref;
  
  return ref;

 abort_with_ref:
  gm_free (ref);
 abort_with_nothing:
  return 0;
}

static void
destroy_all_message_refs (test_t *t)
{
  message_ref_t *ref;
  
  while (t->last_created_message_ref)
    {
      /* splice the reference out of the created list */
      
      ref = t->last_created_message_ref;
      t->last_created_message_ref = ref->prev_created;

      /* destory the referenced message and the reference */
      
      gm_dma_free (t->port, ref->original_ptr);
      gm_free (ref);
    }
}

static message_ref_t *
alloc_message_ref (test_t *t, unsigned int size)
{
  message_ref_t *ret;

  gm_assert (size < 32);
  ret = t->free_send_message_refs[size];
  if (ret)
    {
      t->free_send_message_refs[size] = ret->next;
      PRINTF (("allocated message ref %p of size %d\n",
	       ret, ret->size));
      ret->next = 0;
      if (GM_DEBUG_ALLSIZE)
	{
	  gm_assert (ret->is_free);
	  ret->is_free = 0;
	}
    }
  return ret;
}

static void
free_message_ref (test_t *t, message_ref_t *ref)
{
  unsigned int size;
  message_ref_t **mrp;

  /* After a message is sent, add it to the free send buffer reference
     list */

  gm_assert (ref);
  gm_assert (t);
  size = ref->size;
  if (GM_DEBUG_ALLSIZE)
    {
      gm_always_assert (ref);
      gm_always_assert (!ref->is_free);
      ref->is_free = 1;
    }
  mrp = &t->free_send_message_refs[size];
  ref->next = *mrp;
  *mrp = ref;
  PRINTF (("freed message ref %p of size %u\n", ref, ref->size));
}

/****************************************************************
 * Packet counts.
 ****************************************************************/

/* HACK: These are globals so handle_sigint can access them. */

static unsigned long total_queued;
static unsigned long total_sent;
static unsigned long total_recvd;

/* extra "status" parameter allows this to be used as a gm_on_exit()
   callback */

static void
print_packet_counts (gm_status_t status, void *ignored)
{
  GM_PARAMETER_MAY_BE_UNUSED (ignored);
  
  fprintf (stderr,
	   "\nTotal queued   = %lu.\n"
	     "Total sent     = %lu.\n"
	     "Total received = %lu.\n",
	   total_queued, total_sent, total_recvd);
}

/****************************************************************
 * Interrupt handling.
 ****************************************************************/

static void
handle_sigint (int i)
{
  GM_PARAMETER_MAY_BE_UNUSED (i);
  fprintf (stderr, "*** program interrupted by user ***\n");
  gm_exit (GM_INTERRUPTED);
}

/****************************************************************
 * Command line processing
 ****************************************************************/

static void
#ifdef va_dcl
usage (t, fmt, va_alist)
     test_t *t;
     const char *fmt;
     va_dcl
#else
usage (test_t *t, const char *fmt, ...)
#endif
{
  va_list ap;

#define TP(a) fprintf (stderr, a)
  TP ("\nusage: gm_allsize [options]\n");
  TP ("  option         default description\n");
  TP ("  =============    ===== =======================================\n");
  TP ("  --bandwidth      [off] report bandwidth\n");
  TP ("  --bps            [off] report bandwidth in bits/sec\n");
  TP ("  --blocking       [off] use blocking receives\n");
  TP ("  --copy           [off] copy on send and recv side \n");
  TP ("  --dots           [off] use dots instead of points in gnuplots\n");
  TP ("  --datagrams      [off] use datagram sends\n");
  TP ("  --density=#      [3]   density of samples for geometric plot\n");
  TP ("  --enable-nack-down=# [0/off] enable/disable the sending of nack-down messages\n");
  TP ("  --fast           [off] perform faster packet verification, if any\n");
  TP ("  --geometric      [off] geometric length progression\n");
  TP ("  --hysteresis     [off] perform up-down hysteresis check\n");
  TP ("  --latency        [on]  report latency\n");
  TP ("  --linear         [on]  linear length progression\n");
  TP ("  --max-plot       [off] plot maximum value\n");
  TP ("  --max-len=#      [12]  maximum length to test\n");
  TP ("  --max-size=#     [~0]  maximum size to test\n");
  TP ("  --min-plot       [off] plot minimum value\n");
  TP ("  --min-usecs-per-point=# [5000] minimum test time in usecs\n");
  TP ("  --min-sends-per-point=# [100] minimum # of sends per point\n");
  TP ("  --min-len=#      [1]   minimum length to test\n");
  TP ("  --min-size=#     [12]  minimum size to test\n");
  TP ("  --nonblocking    [on]  use nonblocking receives\n");
  TP ("  --offset=#       [0]   offset of send/recv buffer\n");
  TP ("  --randomize      [off] randomize the length progression\n");
  TP ("  --random-lengths [off] random length progression\n");
  TP ("  --randomize      [off] shuffle the test lengths\n");
  TP ("  --reverse        [off] perform test in reverse order\n");
  TP ("  --scatter-plot   [on]  plot all measured values\n");
  TP ("  --silent         [off] run test without reporting data points\n");
  TP ("  --size=#         [12]  minimum and maximum size to test\n");
  TP ("  --slave          [off] run in slave mode\n");
  TP ("  --stddev-plot    [on]  plot mean and standard deviation\n");
  TP (GM_ENABLE_STATS_SOCKET ? 
      "  --socket-output=host:port\n"
      "                   [none] output to remote host instead of stdout\n"
      : "" );
  TP ("  -B #             [0]   board to use in this machine\n");
  TP ("  -b               [on]  bidirectional (ping pong) test (default)\n");
  TP ("  -bw              [off] alias for --bandwidth\n");
  TP ("  -c #             [1]   specifies number of messages per length\n");
  TP ("  -d #             [2]   specifies port number of remote slave\n");
  TP ("  -e               [off] exit when an error is detected\n");
  TP ("  -g               [off] alias for \"--geometric\"\n");
  TP ("  -h host          []    machine running in slave mode\n");
  TP ("                         (for point to point testing)\n");
  TP ("  -l #             [1]   length increment to test\n");
  TP ("  -lat             [on]  alias for --latency\n");
  TP ("  -o path[.gpl]    []    specifies output file.  If the specified\n");
  TP ("                         file name ends in \".gpl\" a gnuplot will\n");
  TP ("                         be generated\n");
  TP ("  -p #             [2]   specifies local port number\n");
  TP ("  -size #          [12]  minimum and maximum size to test\n");
  TP ("  -u               [off] unidirectional (spray) test\n");
  TP ("  -v               [off] verify contents of all messages (slow)\n");
  TP ("  \n");
  TP ("  By default, packets are sent to the local host.\n");
  TP ("  GM sizes must be from 1 to 31.\n");
  TP ("  \n");

#ifdef __STDC__
  va_start (ap, fmt);
#else
  va_start (ap);
#endif
  vfprintf (stderr, fmt, ap);
  va_end (ap);

  gm_exit (GM_FAILURE);
#undef TP
}

/****************************************************************
 * Packet content verification
 ****************************************************************/

static
char
random_seed (void)
{
  char ret;
  do
    {
      ret = gm_rand() & 0xff;
    }
  while (ret == 0);
  return ret;
}

/* don't fill or verify */

static void
fill_buffer_not (test_t *t, gm_u8_n_t *buf, unsigned int len, char seed)
{
  if (t->do_copy)
    {
      char *copybuf;
      copybuf = malloc(len);
      if (copybuf) 
	{
	  gm_bcopy(buf,copybuf,len);
	  free(copybuf);
	}
    }

  return;
}

static int
verify_buffer_not (test_t *t, gm_u8_n_t *buf, unsigned int len)
{
  if (t->do_copy)
    {
      char *copybuf;
      copybuf = malloc(len);
      if (copybuf) 
        {
          gm_bcopy(buf,copybuf,len);
          free(copybuf);
        }
    }

  return 1;
}

/* simple buffer fill and verify routines */

static void
fill_buffer_fast (test_t *t, gm_u8_n_t *buf, unsigned int len, char seed)
{
  unsigned int i;

  buf[0] = gm_hton_u8 (seed);

  for (i = 1; i < len; i++)
    buf[i] = gm_hton_u8 ((gm_u8_t) (i & 0xFF));

  if (t->do_copy)
    {
      char *copybuf;
      copybuf = malloc(len);
      if (copybuf) 
        {
          gm_bcopy(buf,copybuf,len);
          free(copybuf);
        }
    }

}

static int
verify_buffer_fast (test_t *t, gm_u8_n_t *buf, unsigned int len)
{
  unsigned int i;
  unsigned char val;

  for (i = 1; i < len; i++)
    {
      val = i&0xFF; /*seed;*/
      if ((gm_ntoh_u8 (buf[i]) & 0xff) != (val & 0xff))
	{
	  printf ("data error; at pos %d expected 0x%x and got "
		  "0x%x\n"
		  "            memory location %p\n",
		  i, val, gm_ntoh_u8 (buf[i]), &buf[i]);

	  if ( !GM_LONG_PACKET_ERROR_PRINT )
	    {
	      if ((i+8) < len)
		{
		  printf ("            next 8 bytes"
			  " %02x %02x %02x %02x %02x %02x %02x %02x \n",
			  gm_ntoh_u8 (buf[i+1]), gm_ntoh_u8 (buf[i+2]),
			  gm_ntoh_u8 (buf[i+3]), gm_ntoh_u8 (buf[i+4]),
			  gm_ntoh_u8 (buf[i+5]), gm_ntoh_u8 (buf[i+6]),
			  gm_ntoh_u8 (buf[i+7]), gm_ntoh_u8 (buf[i+8]));
		}
	      return 1;
	    }
	  else /* GM_LONG_PACKET_ERROR_PRINT */
	    {
	      unsigned int j;
	      
	      /* if we got an error, print out the entire buffer */
	      printf ("Received:\n");
	      for (j = 1; j < len; j++)
		{
		  printf ("%02x ", gm_ntoh_u8 (buf[j]));
		  if( (j % 16) == 0 ) printf("\n");
		}
	      printf ("\nExpected:\n");
	      for (j = 1; j < len; j++)
		{
		  printf("%02x ", (unsigned char) (j & 0xff));
		  if( (j % 16) == 0 ) printf("\n");
		}
	      printf ("\n");
	      return 1;	
	    }
	  break;
	}
    }

  if (t->do_copy)
    {
      char *copybuf;
      copybuf = malloc(len);
      if (copybuf) 
        {
          gm_bcopy(buf,copybuf,len);
          free(copybuf);
        }
    }
  
  return 0;
}

static void
fill_buffer_thorough (test_t *t, gm_u8_n_t *buf, unsigned int len, char seed)
{
  unsigned int i;

  gm_srand (seed);

  buf[0] = gm_hton_u8 (seed);
  
  for (i = 1; i < len; i++)
    {
      buf[i] = gm_hton_u8 ((gm_u8_t) (gm_rand_mod (255) + 1));
    }
  
  if (t->do_copy)
    {
      char *copybuf;
      copybuf = malloc(len);
      if (copybuf) 
        {
          gm_bcopy(buf,copybuf,len);
          free(copybuf);
        }
    }
}

static int
verify_buffer_thorough (test_t *t, gm_u8_n_t *buf, unsigned int len)
{
  unsigned int i = 0, rv=0;
  char val;
  char *pattern;
  char seed = gm_ntoh_u8 (buf[0]);

  gm_srand (seed);

  pattern = gm_malloc (len);
  if (!pattern)
    {
      printf ("verify_buffer_thorough:\n"
	      "Can't gm_malloc a len = %d buffer for the pattern\n", len);
      return 1;
    }

  for (i = 1; i < len; i++)
    pattern[i] = gm_rand_mod (255) + 1;

  for (i = 1; i < len; i++)
    {
      val = pattern[i];
      if ((gm_ntoh_u8 (buf[i]) & 0xff) != (val & 0xff))
	{
	  rv = 1;
	  printf ("data error; msg %lu at pos %d of %d expected 0x%x"
		  " and got 0x%x\n"
		  "            memory location %p\n",
		  total_recvd, i, len, val&0xff, gm_ntoh_u8 (buf[i]) & 0xff,
		  &buf[i]);

	  if (!GM_LONG_PACKET_ERROR_PRINT)
	    {
	      if ((i+8) < len)
		{
		  printf ("            next 8 bytes"
			  " %02x %02x %02x %02x %02x %02x %02x %02x \n",
			  gm_ntoh_u8 (buf[i+1]), gm_ntoh_u8 (buf[i+2]),
			  gm_ntoh_u8 (buf[i+3]), gm_ntoh_u8 (buf[i+4]),
			  gm_ntoh_u8 (buf[i+5]), gm_ntoh_u8 (buf[i+6]),
			  gm_ntoh_u8 (buf[i+7]), gm_ntoh_u8 (buf[i+8]));
		  printf ("        expected 8 bytes"
			  " %02x %02x %02x %02x %02x %02x %02x %02x \n",
			  (gm_u8_t) pattern[i+1], (gm_u8_t) pattern[i+2],
			  (gm_u8_t) pattern[i+3], (gm_u8_t) pattern[i+4],
			  (gm_u8_t) pattern[i+5], (gm_u8_t) pattern[i+6],
			  (gm_u8_t) pattern[i+7], (gm_u8_t) pattern[i+8]);
		}
	    }
	  else /* GM_LONG_PACKET_ERROR_PRINT */
	    {
	      unsigned int j;
	      
	      /* if we got an error, print out the entire buffer, flagging the
	         discrepencies. */
	      printf ("Received:\n");
	      for (j = 0; j < len; j++)
		{
		  printf (gm_ntoh_u8 (buf[j]) != (unsigned char) pattern[j]
			  ? "%02x'"
			  : "%02x ",
			  gm_ntoh_u8 (buf[j]));
		  if( ((j+1) % 16) == 0 ) printf("\n");
		}
	      printf ("\nExpected:\n");
	      for (j = 0; j < len; j++)
		{
		  printf (gm_ntoh_u8 (buf[j]) != (unsigned char) pattern[j]
			  ? "%02x'"
			  : "%02x ",
			  (unsigned char) pattern[j]);
		  if( ((j+1) % 16) == 0 ) printf("\n");
		}
	      printf ("\n");
	    }
	  break;
	}
    }

  if (t->do_copy)
    {
      char *copybuf;
      copybuf = malloc(len);
      if (copybuf) 
        {
          gm_bcopy(buf,copybuf,len);
          free(copybuf);
        }
    }

  if (pattern)
    gm_free (pattern);
  return rv;
}

static void
fill_message (test_t *t, message_t *m, unsigned long len,
	      enum message_type basic_type)
{
  switch (len)
    {
    case 0:
      fprintf (stderr, "cannot fill zero-length packets");
      if (t->exit_on_error)
	{
	  gm_exit (GM_FAILURE);
	}
      break;

    case 1:
      m->small.type = gm_hton_u8 ((gm_u8_t) (basic_type + 1));
      break;

    case 2:
    case 3:
    case 4:
    case 5:
    case 6:
    case 7:
      m->small.type
	= gm_hton_u8 ((gm_u8_t) (basic_type + (unsigned char) len));
      t->fill_buffer (t, m->small.data,
		      len - GM_OFFSETOF (message_t, small.data),
		      random_seed ());
      break;

    case 8:
      m->large.type = gm_hton_u8 ((gm_u8_t) (basic_type + 8));
      t->fill_buffer (t, m->large.pad, sizeof (m->large.pad),
		      (char)gm_rand ());
      m->large.length = gm_hton_u32 (8);
      break;

    default:
      m->large.type = gm_hton_u8 ((gm_u8_t) basic_type);
      t->fill_buffer (t, m->large.pad, sizeof (m->large.pad),
		      (char)gm_rand ());
      m->large.length = gm_hton_u32 (len);
      t->fill_buffer (t, m->large.data, len, (char)gm_rand ());
      break;
    }
}

/* BAD: should check for legit types here and make sure the length passed
   in here is form the event, not the packet. */

static int
verify_message (test_t *t, message_t *m, unsigned long len)
{
  int error = 0;

  if (MESSAGE_LEN (m) != len)
    {
      fprintf
	(stderr,
	 "Packet message length 0x%x does not match length reported"
	 " by GM 0x%lx\n",
	 MESSAGE_LEN (m), len);
      return 1;
    }
  
  switch (MESSAGE_LEN (m))
    {
    case 0:
    case 1:
    case 2:
      break;

    case 3:
    case 4:
    case 5:
    case 6:
    case 7:
      error = t->verify_buffer (t, m->small.data,
				len - GM_OFFSETOF (message_t, small.data));
      break;

    case 8:
      error = t->verify_buffer (t, m->large.pad, sizeof (m->large.pad));
      break;

    default:
      error = (t->verify_buffer (t, m->large.pad, sizeof(m->large.pad))
	       || t->verify_buffer (t, m->large.data,
				    len - GM_OFFSETOF (message_t,
						       large.data)));
      break;
    }
  if (error && t->exit_on_error)
    {
      gm_exit (GM_FAILURE);
    }
  return error;
}

/****************************************************************
 * Gnuplot support
 ****************************************************************/

#define GNUPLOT_EXT ".gpl"

static
int
#ifdef va_dcl
gnuplot_printf (t, fmt, va_alist)
     test_t *t;
     const char *fmt;
     va_dcl
#else
gnuplot_printf (test_t *t, const char *fmt, ...)
#endif
{
  int ret;
  va_list ap;

  if (t->gnuplot.file == 0)
    return 0;
  
#ifdef __STDC__
  va_start (ap, fmt);
#else
  va_start (ap);
#endif

  ret = vfprintf (t->gnuplot.file, fmt, ap);
  va_end (ap);
  return ret;
}

static
FILE *
gnuplot_open (test_t *t, const char *filename)
{
  size_t basename_len;
  
  /* open the file for output */
  
  t->gnuplot.file = fopen (filename, "w");
  if (!t->gnuplot.file)
    {
      fprintf (stderr, "Could not open gnuplot file \"%s\""
	       " for writing", filename);
      perror (" ");
      gm_exit (GM_FAILURE);
    }

  /* build the basename for the filename */
  
  basename_len = strlen (filename) - strlen (GNUPLOT_EXT);
  t->gnuplot.basename = gm_malloc (basename_len + 1);
  gm_always_assert (t->gnuplot.basename);
  strncpy (t->gnuplot.basename, filename, basename_len);
  t->gnuplot.basename[basename_len] = 0;

  return t->gnuplot.file;
}

static
void
gnuplot_start_size (test_t *t, unsigned int size, int reversed)
{
  char *filename, *new_plot_string;
  const char *_reversed = "_reversed";
  
  if (!t->gnuplot.file)
    return;
  
  gm_always_assert (size <= 99);
  filename = gm_malloc (strlen (t->gnuplot.basename) + 1 + 2
			+ strlen (_reversed));
  sprintf (filename, "%s_%u%s", t->gnuplot.basename, size,
	   reversed ? _reversed : "");
  t->data_file = fopen (filename, "w");
  if (!t->data_file)
    {
      fprintf (stderr, "could not open data output file \"%s\"",
	       filename);
      perror (" ");
      gm_exit (GM_FAILURE);
    }
  
  if (t->gnuplot.plot_string)
    {
      new_plot_string = (char *) gm_malloc (strlen (t->gnuplot.plot_string)
					    + 3 + strlen (filename) + 2);
      gm_always_assert (new_plot_string);
      sprintf (new_plot_string, "%s, \"%s\"", t->gnuplot.plot_string,
	       filename);
      gm_free (t->gnuplot.plot_string);
    }
  else
    {
      const char *plot = "plot \"";
      
      new_plot_string = (char *) gm_malloc (strlen (plot)
					    + strlen (filename)
					    + 2);
      sprintf (new_plot_string, "%s%s\"", plot, filename);
    }
  t->gnuplot.plot_string = new_plot_string;
  
  gm_free (filename);
  fprintf (t->data_file, "# output for gnuplot\n");
}

static void report_scatter (test_t *t, unsigned long len);
static void report_stddev (test_t *t, unsigned long len);
static void record_point_bandwidth (test_t *t, unsigned long len,
				    unsigned int cnt, gm_u64_t time);

static
void
gnuplot_close (gm_status_t status, test_t *t)
{
  GM_PARAMETER_MAY_BE_UNUSED (status);
  
  if (!t->gnuplot.file)
    return;

  if (t->record_point == record_point_bandwidth)
    {
      fprintf (t->gnuplot.file, "set nologscale y\n");
    }
  fprintf (t->gnuplot.file, "set terminal x11\n");
  /* fprintf (t->gnuplot.file, "set output STDOUT\n"); */
  fprintf (t->gnuplot.file, "set grid\n");
  fprintf (t->gnuplot.file, "set title \"%s %s\"\n", t->title, t->ylabel);
  fprintf (t->gnuplot.file, "set xlabel \"Message Length (bytes)\"\n");
  fprintf (t->gnuplot.file, "set ylabel \"%s\"\n", t->ylabel);
  fprintf (t->gnuplot.file, "set key below\n");
  if (t->report == report_stddev)
    fprintf (t->gnuplot.file, "set data style yerrorbars\n");
  else
    fprintf (t->gnuplot.file, "set data style %s\n", t->points);

  /* output command to plot all files */

  if (t->gnuplot.plot_string)
    {
      fputs (t->gnuplot.plot_string, t->gnuplot.file);
      gm_free (t->gnuplot.plot_string);
      t->gnuplot.plot_string = 0;
    }
  
  gnuplot_printf (t, "\npause -1 \"Hit return to continue\"\n");

  /* close the gnuplot file */

  fflush (t->gnuplot.file);
  fclose (t->gnuplot.file);
  t->gnuplot.file = 0;
}


/****************************************************************
 * data output
 ****************************************************************/

#if 0
/* function to print both to stdout and to any open output file. */
/* This function and its use excited some ppc-linux bugs. 3/20/3000 feldy */
static
int
#ifdef va_dcl
printf_both (t, fmt, va_alist)
     test_t *t;
     const char *fmt;
     va_dcl
#else
printf_both (test_t *t, const char *fmt, ...)
#endif
{
  int ret;
  va_list ap;

#ifdef __STDC__
  va_start (ap, fmt);
#else
  va_start (ap);
#endif

  if (t->data_file)
    {
      vfprintf (t->data_file, fmt, ap);
      fflush (t->out);
    }
  ret = vfprintf (stdout, fmt, ap);
  fflush (stdout);
  gm_sleep (0);
  va_end (ap);
  return ret;
}
#endif

/****************
 * record_point_...
 ****************/

/* handlers for recording the output of a single timed test */

static
void
record_point_latency (test_t *t, unsigned long len, unsigned int cnt,
		      gm_u64_t time)
{
  /*GM_PARAMETER_MAY_BE_UNUSED (len);*/
  
#if GM_OS_VXWORKS
  /* vxworks can't do most floating point or 64 bit math
     so we do a little magic to give vxworks meaningful numbers */
  const unsigned scale_factor = 100;
  unsigned long ltime;
  
  ltime = (unsigned long) time;
#endif /* GM_OS_VXWORKS */
  
  if (t->ping_pong && (t->target_node_id != t->this_node_id))
    {
      static int noted;
      
      if (!noted)
	{
	  printf ("# (reporting half round trip time)\n");
	  noted = 1;
	}
#if GM_OS_VXWORKS
      stats_append (t->stats,
		    (double)(scale_factor * ltime / cnt / 2) / scale_factor);
#else /*!GM_OS_VXWORKS*/
      stats_append (t->stats, u64_to_double (time) / (double) (cnt) / 2.0);
#endif /*!GM_OS_VXWORKS*/
      
    }
  else
    {
#if GM_OS_VXWORKS
      stats_append (t->stats,
		    (double)(scale_factor * ltime / cnt ) / scale_factor
		    );
#else /*!GM_OS_VXWORKS*/
      stats_append (t->stats, u64_to_double (time) / (double) (cnt));
#endif /*!GM_OS_VXWORKS*/
    }
}

static
void
record_point_bandwidth (test_t *t, unsigned long len, unsigned int cnt,
			gm_u64_t time)
{
#if GM_OS_VXWORKS
  /* vxworks can't do most floating point or 64 bit math
     so we do a little magic to give vxworks meaningful numbers */
  unsigned long ltime;	
  const unsigned long scale_factor = 10;

  ltime = (unsigned long) time;
  
  stats_append (t->stats,
		(double)( scale_factor * cnt * len / ltime ) / scale_factor );
#else /* !GM_OS_VXWORKS */  
  if (time == 0)
    time = 1;

  if (t->report_bytes_per_sec)
    {
      stats_append (t->stats,
		    ((double)1e6 *  (double) (cnt) * (double) (len) / 
		     u64_to_double (time)));
    }
  else 
    {
      stats_append (t->stats,
		    ((double) (cnt) * (double) (len) / u64_to_double (time)));
    }
#endif
}
 
/****************
 * report_...
 ****************/


/* handlers for reporting a summary of earlier record data points */

static void
report_min (test_t *t, unsigned long len)
{
  if (t->data_file)
    {
      fprintf(t->data_file, "%lu %f\n", len, stats_get_min (t->stats));
    }
  fprintf(stdout, "%lu %f\n", len, stats_get_min (t->stats));
  if (t->stats_socket)
    {
      socket_printf (t->stats_socket, "%lu %f\n", len,
		     stats_get_min (t->stats));
    }
}

static void
report_max (test_t *t, unsigned long len)
{
  if (t->data_file)
    {
      fprintf(t->data_file, "%lu %f\n", len, stats_get_max (t->stats));
    }
  fprintf(stdout, "%lu %f\n", len, stats_get_max (t->stats));
  if (t->stats_socket)
    {
      socket_printf (t->stats_socket, "%lu %f\n", len,
		     stats_get_max (t->stats));
    }
}

static void
report_stddev (test_t *t, unsigned long len)
{
  if (t->data_file)
    {
      fprintf(t->data_file, "%lu %f %f\n", len, stats_get_mean (t->stats),
	      stats_get_standard_deviation (t->stats));
    }
  fprintf(stdout, "%lu %f %f\n", len, stats_get_mean (t->stats),
	  stats_get_standard_deviation (t->stats));
  if (t->stats_socket)
    {
      socket_printf (t->stats_socket, "%lu %f %f\n", len,
		     stats_get_mean (t->stats),
		     stats_get_standard_deviation (t->stats));
    }
}

static void
report_scatter (test_t *t, unsigned long len)
{
  char fmt[20];
  sprintf (fmt, "%lu %%.2f\n", len);
  stats_output_points (stdout, fmt, t->stats);

  if (t->stats_socket)
    {
      socket_stats_output_points (t->stats_socket, fmt, t->stats);
    }
  
  if (t->data_file)
    stats_output_points (t->data_file, fmt, t->stats);
}

static void
report_nothing (test_t *t, unsigned long len)
{
  GM_PARAMETER_MAY_BE_UNUSED (t);
  GM_PARAMETER_MAY_BE_UNUSED (len);
  
  return;
}

/****************
 * label_output
 ****************/

/* function to output a comment describing the data that follows
   in the output file */

static
void
label_output (test_t *t, unsigned int size, int reversed)
{
  GM_PARAMETER_MAY_BE_UNUSED (reversed);
  
  if (t->report == report_nothing)
    return;

  if (t->this_node_id == t->target_node_id)
    {
      if (t->ping_pong)
	{
	  if (t->record_point == record_point_bandwidth)
	    {
	      t->title = "Reliable Loopback Ping-Pong";
	      if (t->report_bytes_per_sec)
		t->ylabel = "Bandwidth (Byte/s)";
	      else
		t->ylabel = "Bandwidth (MByte/s)";
	    }
	  else
	    {
	      t->title = "Reliable Loopback";
	      t->ylabel = "Latency (usec)";
	    }
	}
      else
	{
	  if (t->record_point == record_point_bandwidth)
	    {
	      t->title = "Reliable Loopback";
	      if (t->report_bytes_per_sec)
		t->ylabel = "Bandwidth (Byte/s)";
	      else
		t->ylabel = "Bandwidth (MByte/s)";
	    }
	  else
	    {
	      t->title = "Reliable Loopback";
	      t->ylabel = "Gap (usec)";
	    }
	}
    }
  else
    {
      if (t->ping_pong)
	{
	  if (t->record_point == record_point_bandwidth)
	    {
	      t->title = "Reliable Point-to-Point Ping-Pong";
	      if (t->report_bytes_per_sec)
		t->ylabel = "Bandwidth (Byte/s)";
	      else
		t->ylabel = "Bandwidth (MByte/s)";
	    }
	  else
	    {
	      t->title = "Reliable Point-To-Point Half-Round-Trip";
	      t->ylabel = "Latency (usec)";
	    }
	}
      else
	{
	  if (t->record_point == record_point_bandwidth)
	    {
	      t->title = "Reliable Point-To-Point";
	      if (t->report_bytes_per_sec)
		t->ylabel = "Bandwidth (Byte/s)";
	      else
		t->ylabel = "Bandwidth (MByte/s)";
	    }
	  else
	    {
	      t->title = "Reliable Point-To-Point";
	      t->ylabel = "Gap (usec)";
	    }
	}
    }
  if (t->data_file)
    fprintf(t->data_file, "# %s %s\n", t->title, t->ylabel);
  fprintf(stdout, "# %s %s\n", t->title, t->ylabel);
  if (t->data_file)
    fprintf(t->data_file, "# size %d\n", size);
  fprintf(stdout, "# size %d\n", size);
}

/****************************************************************
 * recv handlers
 ****************************************************************/

static void
recv_only (test_t *t, struct gm_port *port, gm_recv_event_t *e,
	   send_func_t send)
{
  void *buffer;
  unsigned int size;
  GM_PARAMETER_MAY_BE_UNUSED (t);
  GM_PARAMETER_MAY_BE_UNUSED (send);
  
  gm_assert (gm_ntoh_u8 (e->recv.tag) == GM_ARBITRARY_RECV_BUFFER_TAG);
  
  buffer = gm_ntohp (e->recv.buffer);
  size = gm_ntoh_u8 (e->recv.size);
  gm_provide_receive_buffer_with_tag (port, buffer, size, GM_LOW_PRIORITY,
				      GM_ARBITRARY_RECV_BUFFER_TAG);
  total_recvd++;
}

static void
recv_and_verify (test_t *t, struct gm_port *port, gm_recv_event_t *e,
		 send_func_t send)
{
  void *message;
  int rv = 0;

  GM_PARAMETER_MAY_BE_UNUSED (send);
  
  message = gm_ntohp (e->recv.buffer);
  rv = verify_message (t, message, gm_ntoh_u32 (e->recv.length));
  if (rv && t->exit_on_error)
    gm_exit (1);

  recv_only (t, port, e, _send_abort);
}

static void
send (test_t *t, unsigned int size,
      unsigned long len, enum message_type basic_type,
      unsigned int target_node_id,
      unsigned int target_port_id,
      send_func_t send_func);

static void
recv_and_reply (test_t *t, struct gm_port *port, gm_recv_event_t *e,
		send_func_t send_func)
{
  void (*fill_buffer_save) (struct test *, gm_u8_n_t *, unsigned int , char );

  /* disable buffer filling for the reply, since the default
     "t->fill_buffer" does fill the buffer. */
  
  fill_buffer_save = t->fill_buffer;
  t->fill_buffer = fill_buffer_not;
  send (t, gm_ntoh_u8 (e->recv.size), gm_ntoh_u32 (e->recv.length),
	t->basic_type, gm_ntoh_u16 (e->recv.sender_node_id),
	gm_ntoh_u8 (e->recv.sender_port_id),
	send_func);
  t->fill_buffer = fill_buffer_save;
  
  recv_only (t, port, e, _send_abort);
}

static void
recv_and_reply_and_verify (test_t *t, struct gm_port *port, gm_recv_event_t *e,
			   send_func_t send_func)
{
  message_ref_t *ref;
  unsigned int len;
  
  len = gm_ntoh_u32 (e->recv.length);
  
  recv_and_verify (t, port, e, _send_abort);
  
  send (t, gm_ntoh_u8 (e->recv.size), len,
	t->basic_type, gm_ntoh_u16 (e->recv.sender_node_id),
	gm_ntoh_u8 (e->recv.sender_port_id), send_func);
}

static void
recv_slave (test_t *t, struct gm_port *port, gm_recv_event_t *e,
	    send_func_t ignored_send_func)
{
  message_t *message;
  enum message_type type;
  GM_PARAMETER_MAY_BE_UNUSED (ignored_send_func);
  
  PRINTF (("slave received following event\n"
	   "  message = %p\n"
	   "  buffer = %p\n"
	   "  length = %u\n"
	   "  sender_node_id = %u\n"
	   "  tag %u\n"
	   "  size %u\n"
	   "  sender_port_id = %u\n",
	   gm_ntohp (e->recv.message),
	   gm_ntohp (e->recv.buffer),
	   gm_ntoh_u32 (e->recv.length),
	   (unsigned int) gm_ntoh_u16 (e->recv.sender_node_id),
	   (unsigned int) gm_ntoh_u8 (e->recv.tag),
	   (unsigned int) gm_ntoh_u8 (e->recv.size),
	   (unsigned int) gm_ntoh_u8 (e->recv.sender_port_id)));
  
  message = (message_t *) gm_ntohp (e->recv.buffer);
  type
    = (enum message_type) (message->type
			   & (TYPE_SEND_MODE_MASK | TYPE_REPLY | TYPE_VERIFY));
  PRINTF (("type is 0x%x\n", (int) type));
  
  switch (type)
    {
    case TYPE_SEND_MODE_NONPEER:
    case TYPE_SEND_MODE_PEER:
    case TYPE_SEND_MODE_DATAGRAM:
    case TYPE_SEND_MODE_DATAGRAM_OR_RELIABLE:
      recv_only (t, port, e, _send_abort);
      break;

      /* reply */
      
    case TYPE_REPLY + TYPE_SEND_MODE_NONPEER:
      recv_and_reply (t, port, e, _send_to_nonpeer);
      break;
    case TYPE_REPLY + TYPE_SEND_MODE_PEER:
      recv_and_reply (t, port, e, _send_to_peer);
      break;
    case TYPE_REPLY + TYPE_SEND_MODE_DATAGRAM:
      recv_and_reply (t, port, e, _send_datagram);
      break;
    case TYPE_REPLY + TYPE_SEND_MODE_DATAGRAM_OR_RELIABLE:
      recv_and_reply (t, port, e, _send_datagram_or_reliable);
      break;

      /* verify */

    case TYPE_VERIFY + TYPE_SEND_MODE_NONPEER:
    case TYPE_VERIFY + TYPE_SEND_MODE_PEER:
    case TYPE_VERIFY + TYPE_SEND_MODE_DATAGRAM:
    case TYPE_VERIFY + TYPE_SEND_MODE_DATAGRAM_OR_RELIABLE:
      recv_and_verify (t, port, e, _send_abort);
      break;

      /* reply and verify */

    case TYPE_REPLY + TYPE_VERIFY + TYPE_SEND_MODE_NONPEER:
      recv_and_reply_and_verify (t, port, e, _send_to_nonpeer);
      break;
      
    case TYPE_REPLY + TYPE_VERIFY + TYPE_SEND_MODE_PEER:
      recv_and_reply_and_verify (t, port, e, _send_to_peer);
      break;
      
    case TYPE_REPLY + TYPE_VERIFY + TYPE_SEND_MODE_DATAGRAM:
      recv_and_reply_and_verify (t, port, e, _send_datagram);
      break;

    case TYPE_REPLY + TYPE_VERIFY + TYPE_SEND_MODE_DATAGRAM_OR_RELIABLE:
      recv_and_reply_and_verify (t, port, e, _send_datagram_or_reliable);
      break;

    default:
      { 
	int i;
	unsigned char *cptr = (unsigned char *)message;
	printf ("receive packet of unrecognized type %u (0x%x)\n",
		(unsigned int) type, (unsigned int) type);
        for (i=0; i<16; i++) 
	  {
	    printf("%02x ",*cptr++);
	  }
	printf("\n");
      }
      break;
    }
}

/****************************************************************
 * send completion handlers
 ****************************************************************/

void free_send_token (test_t *t);

/* handle send completion by either using the send token for an
   enqueued send or recycling it. */

static void
_sent_ping_pong (struct gm_port *port, void *context)
{
  test_t *t;
  message_ref_t *ref;
  send_record_t *rec;
  GM_PARAMETER_MAY_BE_UNUSED (port);

  /* free the resources for this send */
  
  rec = (send_record_t *) context;
  t = rec->test;
  ref = rec->message_ref;
  gm_assert (ref);
  gm_lookaside_free (rec);
  free_message_ref (t, ref);
}

/* For each completed send in a unidirectional test, reenqueue the send */

static void
_sent_spray (struct gm_port *port, void *context)
{
  send_record_t *rec;
  test_t *t;
  message_ref_t *ref;
  unsigned int size, target_node_id, target_port_id;
  unsigned long len;
  enum message_type basic_type;
  GM_PARAMETER_MAY_BE_UNUSED (port);
  
  rec = (send_record_t *) context;
  t = rec->test;

  /* free the message ref so the send below will probably have the
     resources it needs */
  
  ref = rec->message_ref;
  gm_assert (ref);
  free_message_ref (t, ref);

  /* send another message */
  
  size = rec->size;
  len = rec->len;
  basic_type = t->basic_type;
  target_node_id = t->target_node_id;
  target_port_id = t->target_port_id;
  PRINTF (("reusing message ref %p of size %d\n", ref, ref->size));
  send (t, size, len, basic_type, target_node_id, target_port_id,
	t->_send);

  /* Free the send record */
  
  gm_lookaside_free (rec);
}

/* send completion handler */

static void
sent (struct gm_port *port, void *context, gm_status_t status)
{
  send_record_t *rec;
  test_t *t;

  rec = (send_record_t *) context;
  t = rec->test;
  if (status != GM_SUCCESS)
    {
      gm_perror ("gm_allsize: send failed", status);
      if (t->exit_on_error)
	{
	  gm_exit (status);
	}
      fprintf(stderr,"gm_allsize: not exiting after a send error\n");
      fflush(stderr);
    }
  free_send_token (t);
  t->_sent (port, context);
  total_sent++;
  t->outstanding_send_cnt--;
}

/****************************************************************
 * send functions
 ****************************************************************/

static void
_send_datagram (struct test *t, struct message_ref *ref, unsigned int size,
		unsigned long len,
		enum message_type basic_type,
		unsigned target_node_id,
		unsigned target_port_id)
{
  struct gm_port *port;
  message_t *m;
  unsigned int priority;
  send_record_t *rec;
  GM_PARAMETER_MAY_BE_UNUSED (basic_type);
  
  port = t->port;
  m = ref->message;
  priority = t->priority;
  rec = (send_record_t *) gm_lookaside_alloc (t->send_record_lookaside);
  gm_assert (rec);
  if (len <= 4096)
    {
      gm_datagram_send (port, m, size, len, priority, target_node_id,
			target_port_id, sent, rec);
    }
  else
    {
      gm_send_with_callback (port, m, size, len, priority, target_node_id,
			     target_port_id, sent, rec);
    }
  rec->test = t;
  rec->message_ref = ref;
  rec->size = size;
  rec->len = len;
}

static void
_send_datagram_or_reliable (struct test *t,
			    struct message_ref *ref,
			    unsigned int size,
			    unsigned long len,
			    enum message_type basic_type,
			    unsigned target_node_id,
			    unsigned target_port_id)
{
  if (gm_rand_mod (2))
    _send_to_nonpeer (t, ref, size, len, basic_type, target_node_id,
		      target_port_id);
  else
    _send_datagram (t, ref, size, len, basic_type, target_node_id,
		    target_port_id);
}

static void
_send_to_nonpeer (test_t *t, message_ref_t *ref, unsigned int size,
		  unsigned long len,
		  enum message_type basic_type,
		  unsigned target_node_id,
		  unsigned target_port_id)
{
  struct gm_port *port;
  message_t *m;
  unsigned int priority;
  send_record_t *rec;
  GM_PARAMETER_MAY_BE_UNUSED (basic_type);
  
  port = t->port;
  m = ref->message;
  priority = t->priority;
  rec = (send_record_t *) gm_lookaside_alloc (t->send_record_lookaside);
  gm_assert (rec);
  gm_send_with_callback (port, m, size, len, priority, target_node_id,
			 target_port_id, sent, rec);
  rec->test = t;
  rec->message_ref = ref;
  rec->size = size;
  rec->len = len;
}

static void
_send_to_peer (test_t *t, message_ref_t *ref, unsigned int size,
	       unsigned long len,
	       enum message_type basic_type,
	       unsigned int target_node_id,
	       unsigned int target_port_id)
{
  struct gm_port *port;
  message_t *m;
  unsigned int priority;
  send_record_t *rec;
  GM_PARAMETER_MAY_BE_UNUSED (basic_type);
  GM_PARAMETER_MAY_BE_UNUSED (target_port_id);
  
  port = t->port;
  m = ref->message;
  priority = t->priority;
  rec = (send_record_t *) gm_lookaside_alloc (t->send_record_lookaside);
  gm_assert (rec);
  gm_send_to_peer_with_callback (port, m, size, len, priority, target_node_id,
				 sent, rec);
  rec->test = t;
  rec->message_ref = ref;
  rec->size = size;
  rec->len = len;
}

static void
_send_abort (test_t *t, message_ref_t *ref, unsigned int size,
	     unsigned long len,
	     enum message_type message_type,
	     unsigned int target_node_id,
	     unsigned int target_port_id)
{
  GM_PARAMETER_MAY_BE_UNUSED (t);
  GM_PARAMETER_MAY_BE_UNUSED (ref);
  GM_PARAMETER_MAY_BE_UNUSED (size);
  GM_PARAMETER_MAY_BE_UNUSED (len);
  GM_PARAMETER_MAY_BE_UNUSED (message_type);
  GM_PARAMETER_MAY_BE_UNUSED (target_node_id);
  GM_PARAMETER_MAY_BE_UNUSED (target_port_id);
  
  /* This function should never be called */
  GM_ABORT ();
}

/* send a message if sufficient resources are available */

static
void
make_send_progress (test_t *t)
{
  message_ref_t *ref;
  enqueued_send_t *first;

  /* send as many packets as possible */

 again:
  first = t->first_enqueued_send;
  if (!first)
    {
      goto abort_with_nothing;
    }
      
  ref = alloc_message_ref (t, first->size);
  if (!ref)
    {
      goto abort_with_nothing;
    }

  if (!gm_alloc_send_token (t->port, t->priority))
    {
      goto abort_with_ref;
    }
      
  (*first->fill) (t, (gm_u8_n_t *) ref->message, first->len,
		  first->basic_type);
  (*first->send) (t, ref, first->size, first->len,
		  first->basic_type, first->target_node_id,
		  first->target_port_id);
  t->first_enqueued_send = first->next;
  gm_lookaside_free (first);
  goto again;
  return;
  
 abort_with_ref:
  free_message_ref (t, ref);
 abort_with_nothing:
  return;
}

/* send function */

static void
send (test_t *t, unsigned int size, unsigned long len,
      enum message_type basic_type, unsigned int target_node_id,
      unsigned int target_port_id, send_func_t send_func)
{
  enqueued_send_t *first;
  message_ref_t *ref;

  PRINTF (("send (%p, %u, %lu, %u, %u, %u)\n",
	   t, size, len, (int) basic_type, target_node_id,
	   target_port_id));

  t->outstanding_send_cnt++;
  t->send_cnt++;

  /* Attempt to send the packet now, else abort. */

  first = t->first_enqueued_send;
  if (first)
    {
      /* earlier sends are pending */
      goto abort_with_nothing;
    }

  ref = alloc_message_ref (t, size);
  if (!ref)
    {
      /* Need to wait for a send to complete to free a send buffer */
      goto abort_with_nothing;
    }

  if (!gm_alloc_send_token (t->port, t->priority))
    {
      /* Need to wait for a send to complete to free a send token */
      goto abort_with_ref;
    }

  fill_message (t, ref->message, len, basic_type);
  (*send_func) (t, ref, size, len, basic_type, target_node_id,
		target_port_id);
  total_queued++;
  return;

  /* If the attempt, above, to send the packet now aborted, then free
     any resources allocated for the send and enqueue the packet to be
     sent later. */
  
 abort_with_ref:
  free_message_ref (t, ref);
 abort_with_nothing:
  {
    enqueued_send_t *es;

    es = gm_lookaside_alloc (t->enqueued_send_lookaside);
    gm_assert (es);
    if (first)
      t->last_enqueued_send->next = es;
    else
      t->first_enqueued_send = es;
    t->last_enqueued_send = es;
    es->next = 0;
      
    es->test = t;
    es->size = size;
    es->len = len;
    es->basic_type = basic_type;
    es->target_node_id = target_node_id;
    es->target_port_id = target_port_id;
    es->send = send_func;
    es->fill = t->fill_buffer;
  }
  total_queued++;
  make_send_progress (t);
  return;
}

void
free_send_token (test_t *t)
{
  gm_free_send_token (t->port, t->priority);
  make_send_progress (t);
}

/****************************************************************
 * send start functions
 ****************************************************************/

static void
start_sending_spray (test_t *t, unsigned int size, unsigned long len)
{
  message_ref_t *ref;

  while ((ref = alloc_message_ref (t, size)) != 0)
    {
      free_message_ref (t, ref);
      send (t, size, len, t->basic_type, t->target_node_id,
	    t->target_port_id, t->_send);
    }
}

static void
start_sending_ping_pong (test_t *t, unsigned int size, unsigned long send_len)
{
  message_ref_t *ref;

  send (t, size, send_len, t->basic_type, t->target_node_id,
	t->target_port_id, t->_send);
}

/****************************************************************
 * length test
 ****************************************************************/

static void
test_size_len (test_t *t, unsigned int size, unsigned long send_len)
{
  gm_u64_t time = 0;
  int caught;
  struct gm_port *port = t->port;
  unsigned long total_send_cnt;
  void (*saved_sent) (struct gm_port *, void *);
  recv_func_t *saved_recv;

  /* The goal here is to send for at least 50*100 microseconds (100
     times the resolution of gm_ticks()) and to send at least 100
     packets by default. */

  for (total_send_cnt = t->min_sends_per_point;
       time < t->min_usecs_per_point;
       total_send_cnt *= 2)
    {
      gm_u64_t start_ticks;

      t->send_cnt = 0;
      gm_always_assert (t->outstanding_send_cnt == 0);
      start_ticks = gm_ticks (port);

      /* start sending */

      PRINTF (("starting to send\n"));
      t->start_sending (t, size, send_len);
      PRINTF (("done starting to send\n"));

      /* send the appropriate number of packets */

      while (t->send_cnt < total_send_cnt)
	{
	  gm_recv_event_t *e;

	  e = t->gm_receive (port);
	  switch (gm_ntoh_u8 (e->recv.type))
	    {
	    case GM_HIGH_RECV_EVENT:
	    case GM_RECV_EVENT:
	      t->recv (t, port, e, t->_send);
	      break;

	    default:
	      gm_unknown (t->port, e);
	      break;
	    }
	}

      /* wait for all sends to complete and catch any ping-pong
	 packet. */

      saved_sent = t->_sent;
      t->_sent = _sent_ping_pong;
      saved_recv = t->recv;
      t->recv = recv_only;
      caught = t->ping_pong ? 0 : 1;
      while (t->outstanding_send_cnt || !caught)
	{
	  gm_recv_event_t *e;

	  e = t->gm_receive (port);
	  switch (gm_ntoh_u8 (e->recv.type))
	    {
	    case GM_HIGH_RECV_EVENT:
	    case GM_RECV_EVENT:
	      recv_only (t, port, e, 0);
	      caught = 1;
	      break;
	    case GM_NO_RECV_EVENT:
	      break;
	    default:
	      gm_unknown (port, e);
	      break;
	    }
	}

      t->_sent = saved_sent;
      t->recv = saved_recv;

      /* compute time */

      time = (gm_ticks (port) - start_ticks) / 2;
    }

  t->record_point (t, send_len, t->send_cnt, time);
}

/****************************************************************
 *
 ****************************************************************/

static void
do_test_size_len (test_t *t, unsigned int size, unsigned long len)
{
  unsigned int cnt;

  stats_clear (t->stats);
  for (cnt = t->count_per_len; cnt; cnt--)
    {
      test_size_len (t, size, len);
    }
  t->report (t, len);
}

/****************************************************************
 * build length stack
 ****************************************************************/

/****************
 * length stack
 ****************/

static void
push_len (struct length_record **rp, unsigned long l)
{
  struct length_record *r;

  gm_always_assert (l != 0);
  do
    r = (struct length_record *) gm_malloc (sizeof (*r));
  while (!r);
  r->next = *rp;
  r->length = l;
  *rp = r;
}

static unsigned long
pop_len (struct length_record **rp)
{
  struct length_record *r;
  unsigned long l;

  r = *rp;
  if (!r)
    return 0;
  *rp = r->next;
  l = r->length;
  gm_free (r);
  return l;
}

static void
reverse_stack (struct length_record **rp)
{
  struct length_record *from, *to, *tmp;

  from = *rp;
  to = 0;
  while (from)
    {
      tmp = from->next;
      from->next = to;
      to = from;
      from = tmp;
    }
  *rp = to;
}

/****************
 * builders
 ****************/

static void
build_linear_lengths (test_t *t, unsigned int size)
{
  unsigned long l;
  unsigned long min, max;

  min = t->min_len > 1 ? t->min_len : 1;
  max = (gm_max_length_for_size (size) < t->max_len
	 ? gm_max_length_for_size (size)
	 : t->max_len);

  for (l = min; l <= max; l += t->length_inc)
    {
      push_len (&t->length_stack, l);
    }
  reverse_stack (&t->length_stack);

  gnuplot_printf (t, "set nologscale\n");
}

static void
build_geometric_lengths (test_t *t, unsigned int size)
{
  unsigned long l, incr;
  unsigned long min, max;

  min = t->min_len > 1 ? t->min_len : 1;
  max = (gm_max_length_for_size (size) < t->max_len
	 ? gm_max_length_for_size (size)
	 : t->max_len);

  l = min;
  while (l <= max)
    {
      push_len (&t->length_stack, l);

      /* compute next test length */

      incr = l >> t->density;
      if (incr != 0)
	l += incr;
      else
	l += 1;
    }
  reverse_stack (&t->length_stack);

  gnuplot_printf (t, "set logscale\n");
}

static void
build_random_lengths (test_t *t, unsigned int size)
{
  unsigned long l;
  unsigned long iterations;
  unsigned long min, max;

  min = t->min_len > 1 ? t->min_len : 1;
  max = (gm_max_length_for_size (size) < t->max_len
	 ? gm_max_length_for_size (size)
	 : t->max_len);

  for (iterations = (1 << size) / t->length_inc;
       iterations;
       iterations--)
    {
      l = gm_rand_mod (max - min + 1) + min;
      push_len (&t->length_stack, l);
    }
  gnuplot_printf (t, "set nologscale\n");
}

/****************
 * lengths_... modifications
 ****************/

static void
lengths_reverse (test_t *t)
{
  reverse_stack (&t->length_stack);
}

/* append reversed-order test */

static void
lengths_add_hysteresis (test_t *t)
{
  struct length_record *lr;

  reverse_stack (&t->length_stack);
  for (lr = t->length_stack; lr; lr = lr->next)
    {
      push_len (&t->length_stack, lr->length);
    }
}

/* randomize the length stack using a O(N) randomization algorith. */

static void
lengths_randomize (test_t *t)
{
  struct gm_hash *h;
  struct length_record *unrandomized_lengths, *randomized_lengths = 0;

  unrandomized_lengths = t->length_stack;
  h = gm_create_hash (gm_hash_compare_longs, gm_hash_hash_long,
		      sizeof (unsigned long), sizeof (unsigned long), 0, 0);
  while (1)
    {
      unsigned long gm_index, limit, c, len;

      /* put unrandomized lengths into a hash table */
      
      c = 0;
      while ((len = pop_len (&unrandomized_lengths)) != 0)
	{
	  gm_hash_insert (h, &c, &len);
	  c++;
	}
      if (c == 0)
	break;

      /* randomly extract half the entries */
      
      limit = c;
      while (c > limit/2)
	{
	  do
	    {
	      gm_index = gm_rand_mod (limit+1);
	    }
	  while (!gm_hash_find (h, &gm_index));
	  push_len (&randomized_lengths,
		    *(unsigned long *) gm_hash_find (h, &gm_index));
	  gm_hash_remove (h, &gm_index);
	  c--;
	}

      /* remove the unrandomized entries so we can start again with
         half as many lengths to randomize*/
      
      for (gm_index = 0; gm_index <= limit; gm_index++)
	{
	  if (gm_hash_find (h, &gm_index))
	    {
	      push_len (&unrandomized_lengths,
			*(unsigned long *) gm_hash_find (h, &gm_index));
	      gm_hash_remove (h, &gm_index);
	    }
	}
    }
  gm_destroy_hash (h);
  t->length_stack = randomized_lengths;
}

/****************************************************************
 * test_do
 ****************************************************************/

static void
test_do_normal (test_t *t, unsigned int size)
{
  unsigned long len;
  
  /* build list of test lengths */
  
  t->build_length_stack (t, size);

  /* massage the list of test lengths */

  if (t->reverse)
    lengths_reverse (t);
  if (t->randomize)
    lengths_randomize (t);
  if (t->hysteresis)
    lengths_add_hysteresis (t);

  /* perform the test */
  
  while ((len = pop_len (&t->length_stack)) != 0)
    do_test_size_len (t, size, len);
}

/* Act as a slave, like the netperf server */

static void
test_do_slave (test_t *t, unsigned int size)
{
  struct gm_port *port;
  GM_PARAMETER_MAY_BE_UNUSED (size);
  
  port = t->port;
  t->_sent = _sent_ping_pong;
  t->_send = _send_abort;
  while (1)
    {
      gm_recv_event_t *e;

      e = t->gm_receive (port);
      switch (gm_ntoh_u8 (e->recv.type))
	{
	case GM_HIGH_RECV_EVENT:
	case GM_RECV_EVENT:
	  recv_slave (t, port, e, _send_abort);
	  break;

	default:
	  gm_unknown (t->port, e);
	  /* fall through */
	case GM_NO_RECV_EVENT:
	  break;
	}
    }
}

/****************************************************************
 *  Create a new test state
 ****************************************************************/

static test_t *
new_test_state (void)
{
  test_t *t;
  int i;

  if (gm_init () != GM_SUCCESS)
    {
      goto abort_with_nothing;
    }
  t = (test_t *) gm_calloc (1, sizeof (*t));
  if (!t)
    {
      goto abort_with_init;
    }
  
  /* initialize globals here for VxWorks compat. */

  t->build_length_stack = build_linear_lengths;
  t->close_called = 0;
  t->count_per_len = 1;
  t->data_file = NULL;
  t->density = 3;
  t->enqueued_send_lookaside
    = gm_create_lookaside (sizeof (enqueued_send_t), 1);
  if (!t->enqueued_send_lookaside)
    {
      goto abort_with_tee;
    }
  t->exit_on_error = 0;
  for (i=0; i<32; i++)
    t->free_send_message_refs[i] = 0;
  t->gm_receive = gm_receive;
  t->last_created_message_ref = 0;
  t->length_inc = 1;
  t->max_len = ~0L;
  t->max_size = MAX_TEST_SIZE;
  t->min_len = 0;
  t->min_sends_per_point = 100;
  t->min_size = MIN_TEST_SIZE;
  t->min_usecs_per_point = 5000;
  t->offset = 0;
  t->ping_pong = 1;
  t->points = "points";
  t->port_id = 2;
  t->priority = GM_LOW_PRIORITY;
  t->quiet = 0;
  t->rand_seed = 42;
  t->record_point = record_point_latency;
  t->remote_host_name = 0;
  t->report = report_scatter;
  t->send_record_lookaside = gm_create_lookaside (sizeof (send_record_t), 1);
  if (!t->send_record_lookaside)
    {
      goto abort_with_enqueued_send_lookaside;
    }
  t->stats = stats_alloc ();
  if (!t->stats)
    {
      goto abort_with_send_record_lookaside;
    }
  t->stats_socket = 0;
  t->target_node_id = 0;
  t->target_port_id = 2;
  t->test_do = test_do_normal;
  t->timer = 0;
  t->verify = 0;
  
  return t;

 abort_with_send_record_lookaside:
  gm_destroy_lookaside (t->send_record_lookaside);
 abort_with_enqueued_send_lookaside:
  gm_destroy_lookaside (t->enqueued_send_lookaside);
 abort_with_tee:
  gm_free (t);
 abort_with_init:
  gm_finalize ();
 abort_with_nothing:
  return 0;
}

/* extra "status" parameter for gm_on_exit() compatibility. */

static void
free_test_state (gm_status_t status, test_t *t)
{
  GM_PARAMETER_MAY_BE_UNUSED (status);
  
  stats_free (t->stats);
  gm_destroy_lookaside (t->send_record_lookaside);
  gm_destroy_lookaside (t->enqueued_send_lookaside);
  gm_free (t);
  gm_finalize ();
}

/****************************************************************
 * argument parsing
 ****************************************************************/

char output_host [2500];

static void
parse_args (test_t *t, int argc, char *argv[])
{
  int i;
  int output_port;

  
  for (i = 1; i < argc; i++)
    {
#define F(name,action) if (strcmp (argv[i], name) == 0) { action; continue; }
      F ("--bandwidth", t->record_point = record_point_bandwidth);
      F ("--bps", t->report_bytes_per_sec = 1);
      F ("--blocking", t->gm_receive = gm_blocking_receive);
      F ("--copy", t->do_copy = 1);
      F ("--datagrams=maybe", t->datagrams = MAYBE);
      F ("--datagrams", t->datagrams = YES);
      F ("--dots", t->points = "dots");
      F ("--nonblocking", t->gm_receive = gm_receive);
      F ("--fast", t->fast = 1);
      F ("--geometric", t->build_length_stack = build_geometric_lengths);
      F ("--hysteresis", t->hysteresis = 1);
      F ("--latency", t->record_point = record_point_latency);
      F ("--linear", t->build_length_stack = build_linear_lengths);
      F ("--max-plot", t->report = report_max);
      F ("--min-plot", t->report = report_min);
      F ("--random-lengths", t->build_length_stack = build_random_lengths);
      F ("--randomize", t->randomize = 1;);
      F ("--reverse", t->reverse = 1;);
      F ("--scatter-plot", t->report = report_scatter);
      F ("--enable-nack-down", t->report = report_nothing);
      F ("--silent", t->report = report_nothing);
      F ("--slave", t->test_do = test_do_slave);
      F ("--stddev-plot", t->report = report_stddev);
      F ("-b", t->ping_pong = 1);
      F ("-bw", t->record_point = record_point_bandwidth);
      F ("-e", t->exit_on_error = 1);
      F ("-f", t->fast = 1);
      F ("-g", t->build_length_stack = build_geometric_lengths);
      F ("-lat", t->record_point = record_point_latency);
      F ("-q", t->quiet = t->quiet < 2 ? t->quiet + 1 : 2);
      F ("-u", t->ping_pong = 0);
      F ("-v", t->verify = 1);

      if (GM_ENABLE_STATS_SOCKET)
	{
	  if (sscanf (argv[i], "--socket-output=%[^:]:%d", output_host,
		      &output_port)
	      == 2)
	    {
	      if (!(t->stats_socket = socket_open (output_host, output_port)))
		gm_exit (GM_FAILURE);
	      on_exit_must_succeed (((gm_on_exit_callback_t)
				     close_stats_socket_callback),
				    &t->stats_socket);
	      continue;
	    }
	}
      
      if (sscanf (argv[i], "--enable-nack-down=%u", &t->enable_nack_down_flag) == 1)
	{
	  t->enable_nack_down_set = 1;
	  continue;
	}

      if (sscanf (argv[i], "--density=%u", &t->density) == 1) continue;
      if (sscanf (argv[i], "--min-usecs-per-point=%li",
		  &t->min_usecs_per_point) == 1) continue;
      if (sscanf (argv[i], "--min-sends-per-point=%i",
		  &t->min_sends_per_point) == 1) continue;

      if (sscanf (argv[i], "--offset=%i", &t->offset) == 1) 
	{
	  if (t->offset & 0x7)
	    {
		usage(t, "offset must be a multiple of 8\n");
	    }
	  if (t->offset >= GM_PAGE_LEN)
	    {
		usage(t, "offset must be less than GM_PAGE_LEN\n");
	    }
	  continue;
	}

      if (strcmp (argv[i], "-h") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      usage (t, "Host name expected after '-h'.\n");
	    }
	  t->remote_host_name = argv[++i];
	  continue;
	}
      if (strcmp (argv[i], "-o") == 0)
	{
	  const char *filename;

	  if (i + 1 >= argc)
	    {
	      usage (t, "file name expected after '-o'.\n");
	    }
	  filename = argv[i+1];
	  if (strcmp ("-", filename) == 0)
	    {
	      t->data_file = stdout;
	    }
	  else if (strlen (filename) > strlen (GNUPLOT_EXT)
		   && !strcmp (GNUPLOT_EXT , (filename + strlen (filename)
					      - strlen (GNUPLOT_EXT))))
	    {
	      t->gnuplot.file = gnuplot_open (t, filename);
	      if (t->gnuplot.file == 0)
		gm_exit (GM_FAILURE);
	      on_exit_must_succeed ((gm_on_exit_callback_t) gnuplot_close,
				    t);
	    }
	  else
	    {
	      t->data_file = fopen (filename, "w");
	      if (t->data_file == NULL)
		{
		  fprintf (stderr, "Could not open output file \"%s\".\n",
			   filename);
		  perror ("could not open file");
		  gm_exit (GM_FAILURE);
		}
	      on_exit_must_succeed ((gm_on_exit_callback_t) fclose_callback,
				    t->data_file);
	    }
	  i++;
	  continue;
	}
      if (strcmp (argv[i], "-c") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      usage (t, "Count expected after '-p'.\n");
	    }
	  t->count_per_len = atoi (argv[++i]);
	  if (t->count_per_len < 1)
	    {
	      usage (t, "Bad count: %d\n", t->count_per_len);
	    }
	  continue;
	}
      if (strcmp (argv[i], "-p") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      usage (t, "Port number expected after '-p'.\n");
	    }
	  t->port_id = atoi (argv[++i]);
	  if (t->port_id < 1 || t->port_id > gm_num_ports (t->port))
	    {
	      usage (t, "bad port number: %d\n", t->port_id);
	    }
	  continue;
	}
      if (strcmp (argv[i], "-d") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      usage (t, "Destination port number expected after '-d'.\n");
	    }
	  t->target_port_id = atoi (argv[++i]);
	  if (t->target_port_id < 1
	      || t->target_port_id > gm_num_ports (t->port))
	    {
	      usage (t, "bad destination port number: %d\n",
		     t->target_port_id);
	    }
	  continue;
	}
      if (strcmp (argv[i], "-B") == 0)
	{
	  if (i + 1 >= argc)
	    {
	      usage (t, "Board number expected after '-B'.\n");
	    }
	  t->board = atoi (argv[++i]);
	  if (t->board < 0)
	    {
	      usage (t, "bad board number: %d\n", t->board);
	    }
	  continue;
	}
      if (strcmp (argv[i], "-l") == 0)
        {
          if (i + 1 >= argc)
            {
              usage (t, "length_increment expected after '-l'.\n");
            }
          t->length_inc = atoi (argv[++i]);
          if ((t->length_inc <= 0))
            {
              usage (t, "bad length_increment: %d\n", t->length_inc);
            }
          continue;
        }
      if (strcmp (argv[i], "-s") == 0)
        {
          if (i + 1 >= argc)
            {
              usage (t, "size expected after '-s'.\n");
            }
          t->min_size = t->max_size = atoi (argv[++i]);
          continue;
        }

#define check_opt(prefix, var, test)					\
	  if (sscanf (argv[i], "--" prefix "=%i", &var) == 1)		\
	    {								\
	      if (!(test))						\
		{							\
		  usage (t, "bad value for option \"%s\"\n", argv[i]);	\
		}							\
	      continue;							\
	    }

      check_opt ("min-size", t->min_size,
		 t->min_size >= gm_min_message_size (t->port));
      check_opt ("max-size", t->max_size, t->max_size < 32);
      check_opt ("min-len", t->min_len, 1);
      check_opt ("max-len", t->max_len, 1);

#undef check_opt

      if (sscanf (argv[i], "--size=%i", &t->min_size) == 1)
	{
	  t->max_size = t->min_size;
	  if (!(t->min_size >= gm_min_message_size (t->port)
		&& t->max_size < 32))
	    {
	      usage (t, "bad value for option \"%s\"\n", argv[i]);
	    }
	  continue;
	}

      usage (t, "unknown arg: '%s'\n", argv[i]);
    }

  if (t->min_size > t->max_size)
    {
      usage (t, "min_size(%d) > max_size(%d) - you need to adjust one"
	     " or the other\n", t->min_size, t->max_size);
    }

  /* let the user know what the parameters are */

  printf ("# Opening board %d  port %d. Sending to port %d.\n",
	  t->board, t->port_id, t->target_port_id);

  if (t->verify)
    {
      printf ("# Verififying contents of all messages.\n");
      if (t->exit_on_error)
	{
	  printf ("# Will exit immediately if we see an error.\n");
	}
      else
	{
	  printf ("# Will NOT exit immediately if we see an error.\n");
	}
    }
  else
    {
      printf ("# Not verifying contents of messages.\n");
    }
  if (t->do_copy)
    {
      printf ("# Copying messages before send and after receive.\n");
    }

    
}

/****************************************************************
 * Support for GM_DEBUG_ALLSIZE_ALARM
 *
 * Optional alarm to periodically report progress.
 ****************************************************************/

static
gm_alarm_t _my_alarm;

static
void
my_alarm (void *context)
{
  test_t *t;

  t = (test_t *) context;
  gm_set_alarm (t->port, &_my_alarm, (gm_u64_t) 1000000L, my_alarm, context);
  print_packet_counts (GM_SUCCESS, 0);
  fprintf (stderr, "# outstanding_send_cnt = %u\n", t->outstanding_send_cnt);
}

/****************************************************************
 * Test Initialization
 ****************************************************************/

static gm_status_t
test_init (test_t *t)
{
  int i, size;
  gm_status_t status;

  /* initialize GM */

  t->port = NULL;
  status = gm_open (&t->port, t->board, t->port_id, "gm_allsize",
		    GM_API_VERSION_1_2);

  if (status != GM_SUCCESS)
    {
      gm_perror ("could not open port", status);
      gm_exit (status);
    }

  if (t->enable_nack_down_set)
    {
      fprintf(stderr, "Setting enable_nack_down to %d\n",t->enable_nack_down_flag);
      gm_set_enable_nack_down(t->port,t->enable_nack_down_flag);
    }

  /* optionally set an alarm to periodically report packet counts */

  if (GM_DEBUG_ALLSIZE_ALARM)
    {
      gm_initialize_alarm (&_my_alarm);
      gm_set_alarm (t->port, &_my_alarm, (gm_u64_t) 1000000UL, my_alarm,
		    (void *) t);
    }
  
  status = gm_set_acceptable_sizes (t->port, t->priority,
				    ((1 << (t->max_size + 1)) - 1)
				    ^ ((1 << t->min_size) - 1));
  if (status != GM_SUCCESS)
    {
      gm_perror ("could not set GM acceptable sizes", status);
      goto abort_with_open_port;
    }

  /* Set the remote node ID to this node ID */

  status = gm_get_node_id (t->port, &t->this_node_id);
  if (status != GM_SUCCESS)
    {
      printf ("Error: could not determine node id.\n");
      goto abort_with_open_port;
    }

  /* let GM manage the send tokens */

  gm_free_send_tokens (t->port, GM_LOW_PRIORITY,
		       gm_num_send_tokens (t->port) / 2);
  gm_free_send_tokens (t->port, GM_HIGH_PRIORITY,
		       gm_num_send_tokens (t->port) / 2);

  /* setup recv tokens */

  for (size = t->min_size;
       (unsigned int)size <= (unsigned int)t->max_size;
       size++)
    {
      /* allocate as many as we want - GM is not keeping track of them */

      int send_buffers = SEND_BUFFERS;

      /* total over all sizes must be <= GM_NUM_RECV_TOKENS */

      int recv_tokens_of_size = RECV_TOKENS_OF_SIZE;

      /* allocate receive buffers */

      for (i = 0; i < recv_tokens_of_size; i++)
	{
	  message_ref_t *ref;

	  ref = create_message_ref (t, size);
	  if (!ref)
	    {
	      status = GM_OUT_OF_MEMORY;
	      goto abort_with_message_refs;
	    }
	  gm_provide_receive_buffer_with_tag
	    (t->port, ref->message, size, t->priority,
	     GM_ARBITRARY_RECV_BUFFER_TAG);
	}

      /* allocate send buffers.  Alloc twice as many to allow for recv
         handlers allocating another ref before freeing the one used,
         and also allocate extra to allow for send completion
         handlers that have not yet been called. */

      for (i = 0; i < send_buffers; i++)
	{
	  message_ref_t *ref;

	  ref = create_message_ref (t, size);
	  if (!ref)
	    {
	      status = GM_OUT_OF_MEMORY;
	      goto abort_with_message_refs;
	    }
	  free_message_ref (t, ref);
	}
    }

  t->stats = stats_alloc ();
  if (!t->stats)
    {
      status = GM_OUT_OF_MEMORY;
      gm_perror ("Could not initialize statistics package.", status);
      goto abort_with_message_refs;
    }

  if (t->min_len < MIN_TEST_LEN)
    t->min_len = MIN_TEST_LEN;

  /* compute the remote node ID from the remote host name, if any, else
     assume loopback. */

  if (t->remote_host_name)
    {
      t->target_node_id = gm_host_name_to_node_id (t->port,
						   t->remote_host_name);
      if (t->target_node_id == GM_NO_SUCH_NODE_ID)
	{
	  usage (t, "Error: host '%s' not found.\n", t->remote_host_name);
	}
      printf ("remote GM node id %s (%d).\n",
	      t->remote_host_name, t->target_node_id);
    }
  else
    {
      gm_assert (t->target_node_id == 0);
      t->target_node_id = t->this_node_id;
    }

  /* compute basic type of sends */

  t->basic_type = TYPE_BLANK;
  if (t->ping_pong)
    t->basic_type |= TYPE_REPLY;
  if (t->verify)
    t->basic_type |= TYPE_VERIFY;
  if (t->datagrams == YES)
    t->basic_type |= TYPE_SEND_MODE_DATAGRAM;
  if (t->datagrams == MAYBE)
    t->basic_type |= TYPE_SEND_MODE_DATAGRAM_OR_RELIABLE;
  else if (t->target_port_id == t->port_id)
    t->basic_type |= TYPE_SEND_MODE_PEER;
  else
    t->basic_type |= TYPE_SEND_MODE_NONPEER;

  /****************
   * Setup the handlers.
   ****************/

  t->_send = (t->datagrams == MAYBE ? _send_datagram_or_reliable
	      : t->datagrams == YES ? _send_datagram
	      : t->target_port_id == t->port_id ? _send_to_peer
	      : _send_to_nonpeer);
  t->_sent = t->ping_pong ? _sent_ping_pong : _sent_spray;
  /* HACK: fill_buffer will be called by a slave only if it should fill */
  t->fill_buffer = (t->verify || t->test_do == test_do_slave
		    ? (t->fast ? fill_buffer_fast : fill_buffer_thorough)
		    : fill_buffer_not);
  /* t->build_length_stack set in parse_args() */
  t->recv = (t->ping_pong
	     ? (t->verify ? recv_and_reply_and_verify : recv_and_reply)
	     : (t->verify ? recv_and_verify : recv_only));
  t->start_sending = (t->ping_pong
		      ? start_sending_ping_pong
		      : start_sending_spray);
  /* t->test_do set in parse_args() */
  t->verify_buffer = (t->verify
		      ? (t->fast ? verify_buffer_fast : verify_buffer_thorough)
		      : verify_buffer_not);
  /* t->report set in parse_args() */

  return GM_SUCCESS;

 abort_with_message_refs:
  destroy_all_message_refs (t);
 abort_with_open_port:
  if (!t->close_called)
    {
      t->close_called = 1;
      gm_close (t->port);
    }
 abort_with_nothing:
  return status;
}

static void
test_finalize (gm_status_t status, test_t *t)
{
  GM_PARAMETER_MAY_BE_UNUSED (status);
  
  destroy_all_message_refs(t);
  if (!t->close_called)
    {
      t->close_called = 1;
      gm_close (t->port);
    }
}

static void
dump_args (test_t *t, int argc, char *argv[])
{
  int i;

  if (t->data_file)
    {
      fprintf(t->data_file, "#");
      for (i=0; i<argc; i++)
	fprintf(t->data_file, " %s", argv[i]);
      fprintf(t->data_file, "\n");
    }

  fprintf(stdout, "#");
  for (i=0; i<argc; i++)
    fprintf(stdout, " %s", argv[i]);
  fprintf(stdout, "\n");
}

/****************************************************************
 * main entry point
 ****************************************************************/

static int
gm_allsize (int argc, char *argv[])
{
  test_t *t;
  unsigned int size;
  gm_status_t status;
  
  /* clear globals */
  total_queued = total_recvd = total_sent = 0;

  /* arrange to report results on exit. */

  on_exit_must_succeed (print_packet_counts, 0);
  
  /* allocate test state */
  
  t = new_test_state ();	/* sets defaults */
  if (!t)
    {
      gm_exit (GM_OUT_OF_MEMORY);
    }
  on_exit_must_succeed ((gm_on_exit_callback_t) free_test_state, t);

  /* parse the command line arguments.  This will exit on an error, but
     that is OK, since we use gm_on_exit() to clean up. */
  
  parse_args (t, argc, argv);	/* override defaults */


  /* initialize the test using the information gathered in parse_args */
  
  status = test_init (t);		/* get ready to test */
  if (status != GM_SUCCESS)
    {
      gm_exit (status);
    }
  on_exit_must_succeed ((gm_on_exit_callback_t) test_finalize, t);

  /* catch control-C signals and clean up as best we can */

  signal (SIGINT, handle_sigint);

  /* perform the tests */

  for (size = t->min_size; size <= t->max_size; size++)
    {
      gnuplot_start_size (t, size, 0);
      label_output (t, size, 0);
      dump_args (t, argc, argv);
      t->test_do (t, size);	/* do test */
    }

  /* done */

  gm_exit (GM_SUCCESS);

  /* never get here */
  
  return 0;			/* retval to prevent compiler warning */
}

/****************************************************************
 * Entry point
 ****************************************************************/

#define GM_MAIN gm_allsize
#include "gm_main.h"

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
