/*
  conflicts.c
  examine routes
  finucane@myri.com (David Finucane)
*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "mt_MapFile.h"
#include "mt_RouteTable.h"
#include "mt_Follower.h"
#include "mt_Switch.h"

#define die(s){mt_Component::printFormat s; exit (1);}
#define insist(e) do { if (!(e)) {mt_Component::printFormat ("Assertion failed in file %s, line %d",__FILE__,__LINE__); goto exception;}} while (0)
#define insistp(e, s) do { if (!(e)) {mt_Component::printFormat s; goto exception;}} while (0)


static const char*DELIMITERS = "(), \n\r\t";
static const char*COMMENT_CHARS = "#;";

static void printLine (char*s)
{
  printf ("%s\n", s);
}

class co_Counts : public mt_Component, mt_Follower
{
  private:
  int numSwitches;
  int* counts;
  
  public:
  
  co_Counts (int numSwitches);
  ~co_Counts ();
  int dump (mt_Graph*g, FILE*fp);
  int step (mt_Node*n, int in);
  int scan (mt_RouteTable*routes, mt_Node*from, mt_Node*to);
};

co_Counts::co_Counts (int numSwitches)
{
  counts = new int [(this->numSwitches = numSwitches) * mt_Switch::NUM_PORTS];
  insistp (counts, ("co_Counts::co_Counts: alloc failed"));
  memset (counts, 0, numSwitches * mt_Switch::NUM_PORTS * sizeof (int));
  
  exception:;
}

co_Counts::~co_Counts ()
{
  if (counts) delete counts;
}

int co_Counts::step (mt_Node*n, int in)
{
  insist (this && n);
  insist (counts);
  
  if (n->isSwitch ())
  {
    int p = n->getTypeIndex ();
    insist (p >= 0 && p < numSwitches);
    insist (in >= 0 && in < mt_Switch::NUM_PORTS);
  
    counts [p * mt_Switch::NUM_PORTS + in] ++;
  }
  return 1;
  exception: return 0;
}

int co_Counts::dump (mt_Graph*g, FILE*fp)
{
  insist (this);
  insist (g && fp);
  
  for (int i = 0; i < numSwitches; i++)
    for (int j = 0; j < mt_Switch::NUM_PORTS; j++)
      fprintf (fp, "%s:%d %d\n", g->getSwitch (i)->getName (), j, counts [i * mt_Switch::NUM_PORTS + j]);
  
  return 1;
  exception: return 0;
}

int co_Counts::scan (mt_RouteTable*routes, mt_Node*from, mt_Node*to)
{
  mt_Route r;
  {
    
    insist (this);
    insist (from && to);

    mt_Node*other;
    int in;
     
    if (!routes->getRoute (from->getTypeIndex (), to->getTypeIndex (), 0, &r))
      return 0;

    return from->follow (&r, &other, &in, this) && other == to;
  }
  exception: return 0;
}

  
int main (int argc, char*argv[])
{  
  mt_Component::initialize (printLine);
    
  mt_RouteTable routes;
  mt_Graph graph;
  
  if (argc != 4)
    die (("usage: %s <mapfile> <routefile> <host pair file>\n", argv[0]));
 
  char*mapFile = argv [1];
  char*routeFile = argv [2];
  char*pairFile = argv [3];
  
  mt_MapFile mf (mapFile, mt_File::_READ);
  
  if (!mf.read (&graph))
    die (("main: couldn't parse map file %s", mapFile));
  
  if (!routes.fromFile (routeFile, &graph))
    die (("main: couldn't read %s", routeFile));
  
  co_Counts*counts;
  counts  = new co_Counts (graph.getNumSwitches ());
  insist (counts);
  
  mt_Tokenizer*tokens;
  tokens = new mt_Tokenizer (pairFile, DELIMITERS, mt_File::MAX_LINE, COMMENT_CHARS);
  insist (tokens);

  while (tokens->getNext ())
  {
    mt_Node*n1,*n2;

    if (!(n1 = graph.getNode (tokens->getWord ())))
      die (("host %s is not in map file %s", tokens->getWord (), mapFile));
    
    if (!tokens->getNext ())
      die (("odd number of hosts in %s", pairFile));

    if (!(n2 = graph.getNode (tokens->getWord ())))
      die (("host %s is not in map file %s", tokens->getWord (), mapFile));
    
    if (!counts->scan (&routes, n1, n2))
      return 0;
  }

  counts->dump (&graph, stdout);

  delete counts;
  delete tokens;
  
  exception: return 0;
}
