/*
  sw_TestPort.c
  to access swtich test port
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "sw_TestPort.h"
#include "mt_htonl.h"
#include "mx_defines.h"

#include "insist.h"

#include "hardware.h"
#include "test.h"

extern "C" int write ( int, const void *, size_t );
extern "C" int read ( int, void *, size_t );
extern "C" int close ( int );
extern "C" void usleep (int);

int sw_TestPort::usleep (int usecs)
{
  insist (this);
  ::usleep (usecs);
  return 1;
  exception: return 0;
}

sw_TestPort::sw_TestPort (char*switchname)
{
  insist (this);
  fd = 0;
  *this->switchname = 0;
  addressLength = 0;
  
  insist (switchname && strlen (switchname) < SWITCHNAME_LENGTH);
  strncpy (this->switchname, switchname, SWITCHNAME_LENGTH);
  
  exception:;
}


int sw_TestPort::open (int port)
{
  insist (this);
  insist (*switchname);  

  struct hostent *he;
  struct sockaddr_in sin;

  memset (&sin, 0, sizeof (sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons (port);

  if (!addressLength)
  {
    if ((he = gethostbyname (switchname)))
    {
      memcpy ((char*)&sin.sin_addr, he->h_addr, he->h_length);
      addressLength = he->h_length;
    }
    else
    {
      if ((sin.sin_addr.s_addr = inet_addr (switchname)) == (unsigned) -1)
      {
	printFormat ("can't get IP address of %s", switchname);
	return 0;
      }
      addressLength = sizeof (sin.sin_addr.s_addr);
    }
    memcpy (addressBuffer, (char*) &sin.sin_addr, addressLength);
  }
  else
  {
    memcpy ((char*)&sin.sin_addr, addressBuffer, addressLength);
  }
  
  if ((fd = socket (PF_INET, SOCK_STREAM, 0)) < 0)
  {
    printFormat ("socket failed");
    return 0;
  }
  
  if (connect (fd, (struct sockaddr*) &sin, sizeof (sin)) < 0)
  {
    printFormat ("connect failed");
    return 0;
  }

  if (::write (fd, MX_TEST_STRING, strlen (MX_TEST_STRING)) != (int) strlen (MX_TEST_STRING))
  {
    printFormat ("couldn't send key string.\n");
    ::close (fd);
    fd = 0;
    return 0;
  }
  return 1;
  exception: return 0;
}

int sw_TestPort::close ()
{
  insist (this);
  
  if (fd)
    ::close (fd);
  fd = 0;

  return 1;
  exception: return 0;
}

int sw_TestPort::send (char*m, int size)
{
  insist (this && fd);
  insist (m && size > 0 && size <= MX_TEST_MTU);
  insist (sizeof (int) == 4);
  
  int n;
  n = mt_htonl (size);
  return write (fd, &n, sizeof (int)) == sizeof (int) && write (fd, m, size) == size;  

  exception: return 0;
}

int sw_TestPort::receive (char*m, int size)
{
  int n; 
  insist (sizeof (int) == 4);
  
  if (read (fd, (char*)&n, sizeof (int)) != sizeof (int))
  {
    printFormat ("read of size field failed");
    return 0;
  }
  
  if (mt_htonl(n) != size)
  {
    printFormat ("unexpected size %d, expected %d", mt_htonl (n), size);
    return 0;
  }
  
  if (read (fd, m, size) != size)
  {
    printFormat ("failed to read %d bytes", size);
    return 0;
  }
  return 1;

  exception: return 0;
}

int sw_TestPort::read (int fd, char*buffer, int length)
{
  insist (this);
  insist (fd && buffer && length > 0);
  
  int total, n;
  
  for (total = 0; total < length; total += n)
  {
    if ((n = ::read (fd, buffer + total, length - total)) < 1)
      return 0;
  }
  return total;

  exception: return 0;
}

int sw_TestPort::send (char*request, int requestLength, char*response, int responseLength)
{
  insist (this);
  insist (request && response);

  if (!open (MX_TEST_PORT))
  {
    printFormat ("open failed");
    return 0;
  }
  
  if (!send (request, requestLength))
  {
    printFormat ("send failed");
    return 0;
  }
  
  if (!receive (response, responseLength))
  {
    printFormat ("receive failed");
    return 0;
  }
  
  if (!close ())
  {
    printFormat ("close failed");
    return 0;
  }
  return 1;
  exception: return 0;
}


int sw_TestPort::getPortStates (int xbar, sw_TestPortState*states)
{
  insist (this);
  insist (states);  

  int n;
  port_state_message_t psm; 
  psm.type = htonl (TEST_GET_PORT_STATES);
  psm.xbar = mt_htonl (xbar);
  psm.port = mt_htonl (0);

  if (!open (MX_TEST_PORT))
  {
    printFormat ("open failed");
    return 0;
  }

  if (!send ((char*)&psm, sizeof (port_state_message_t)))
  {
    printFormat ("send failed");  
    return 0;
  }
  
  if (read (fd, (char*)&n, sizeof (int)) != sizeof (int))
  {
    printFormat ("read of port states size failed");  
    return 0;
  }

  n = mt_htonl (n) / sizeof (port_state_message_t);  
      
  for (int i = 0; i < n; i++)
  {
    if (read (fd, (char*)&psm, sizeof (port_state_message_t)) != sizeof (port_state_message_t))
    {
      printFormat ("read failed");  
      return 0;
    }

    states [i].xbar = mt_htonl (psm.xbar);
    states [i].port = mt_htonl (psm.port);
    states [i].uptime = mt_htonl (psm.uptime);
    states [i].good_low = mt_htonl (psm.good_low);
    states [i].good_high = mt_htonl (psm.good_high);
    states [i].bad_low = mt_htonl (psm.bad_low);
    states [i].bad_high = mt_htonl (psm.bad_high);
    states [i].timeouts = mt_htonl (psm.timeouts);
    states [i].illegal_symbols = mt_htonl (psm.illegal_symbols);
    states [i].missed_beats = mt_htonl (psm.missed_beats);
    states [i].bad_routes = mt_htonl (psm.bad_routes);
    states [i].dead_routes = mt_htonl (psm.dead_routes);
    states [i].port_state = mt_htonl (psm.port_state);
    states [i].control = mt_htonl (psm.control);
  }

  if (!close ())
  {
    printFormat ("close failed");
    return 0;
  }

  return n;  
  exception: return 0;  
}


int sw_TestPort::getChipId (int xbar, char*low, char*high)
{
insist (this);
insist (low && high);
  
chip_id_message_t r;  
test_message_t m;
m.type = mt_htonl (TEST_READ_CHIP);
m.xbar = mt_htonl (xbar);

if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
  return 0;

*high = r.data.high;
*low = r.data.low;
return 1;
exception: return 0;
}

int sw_TestPort::getGlobalControl (int xbar, char*low)
{
  insist (this);
  insist (low);
  
  global_control_message_t r;
  test_message_t m;
  m.type = mt_htonl (TEST_READ_GLOBAL_CONTROL);
  m.xbar = mt_htonl (xbar);
  
  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;

  *low = r.data.control;
  return 1;
  exception: return 0;
}

int sw_TestPort::setGlobalControl (int xbar, char low)
{
  insist (this);
  
  global_control_message_t m;
  test_message_t r;
  
  m.type = mt_htonl (TEST_WRITE_GLOBAL_CONTROL);
  m.xbar = mt_htonl (xbar);
  m.data.control = low;

  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));
  exception: return 0;
}

int sw_TestPort::getOutputImpedence (int xbar, char*low, char*high)
{
  insist (this);
  insist (low && high);
  
  impedence2_message_t r;
  test_message_t m;
  m.type = mt_htonl (TEST2_READ_IMPEDENCE);
  m.xbar = mt_htonl (xbar);

  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;
  
  *low = r.data.low;
  *high = r.data.high;
  
  return 1;
  exception: return 0;
}

int sw_TestPort::setOutputImpedence (int xbar, char low, char high)
{
  insist (this);
  
  impedence2_message_t m;
  test_message_t r;
  m.type = mt_htonl (TEST2_WRITE_IMPEDENCE);
  m.xbar = mt_htonl (xbar);
  m.data.low = low;
  m.data.high = high;
  
  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));
  exception: return 0;
}


int sw_TestPort::getOutputImpedence (int xbar, char*low1, char*high1, char*low2, char*high2)
{
  insist (this);
  insist (low1 && high1 && low2 && high2);

  impedence_message_t r;
  test_message_t m;
  m.type = mt_htonl (TEST_READ_IMPEDENCE);
  m.xbar = mt_htonl (xbar);

  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;
  
  *low1 = r.data.low1;
  *low2 = r.data.low2;
  *high1 = r.data.high1;
  *high2 = r.data.high2;
  
  return 1;
  exception: return 0;
}

int sw_TestPort::setOutputImpedence (int xbar, char low1, char high1, char low2, char high2)
{
  insist (this);

  test_message_t r;
  impedence_message_t m;

  m.type = mt_htonl (TEST_WRITE_IMPEDENCE);
  m.xbar = mt_htonl (xbar);
  m.data.low1 = low1;
  m.data.low2 = low2;
  m.data.high1 = high1;
  m.data.high2 = high2;
  
  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));

  exception: return 0;
}

int sw_TestPort::getTestImpedence (int xbar, char*low, char*high)
{
  insist (this);
  insist (low && high);
  
  impedence_test_message_t r;
  test_message_t m;
  m.type = mt_htonl (TEST_READ_IMPEDENCE_TEST);
  m.xbar = mt_htonl (xbar);

  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;
  
  *low = r.data.low;
  *high = r.data.high;
  
  return 1;

  exception: return 0;
}

int sw_TestPort::setTestImpedence (int xbar, char low, char high)
{
  insist (this);

  test_message_t r;
  impedence_test_message_t m;

  m.type = mt_htonl (TEST_WRITE_IMPEDENCE_TEST);
  m.xbar = mt_htonl (xbar);
  m.data.low = low;
  m.data.high = high;
  
  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));

  exception: return 0;
}

int sw_TestPort::simpleCommand (int xbar, int type)
{
  insist (this);

  test_message_t m;
  m.type = mt_htonl (type);
  m.xbar = mt_htonl (xbar);
  return send ((char*) &m, sizeof (m), (char*) &m, sizeof (m));

  exception: return 0;
}

int sw_TestPort::enableUpdate (int xbar)
{
  insist (this);
  return simpleCommand (xbar, TEST_ENABLE_XBAR_UPDATE);
  exception: return 0;
}

int sw_TestPort::disableUpdate (int xbar)
{
  insist (this);
  return simpleCommand (xbar, TEST_DISABLE_XBAR_UPDATE);
  exception: return 0;
}

int sw_TestPort::disableImpedence (int xbar)
{
  insist (this);
  return simpleCommand (xbar, TEST_DISABLE_XBAR_IMP_UPDATE);
  exception: return 0;
}

int sw_TestPort::enableImpedence (int xbar)
{
  insist (this);
  return simpleCommand (xbar, TEST_ENABLE_XBAR_IMP_UPDATE);
  exception: return 0;
}

int sw_TestPort::setRelays (int xbar, int data)
{
  insist (this);
  
  test_message_t r;
  set_relays_message_t m;
  m.type = mt_htonl (TEST_SET_RELAYS);
  m.xbar = mt_htonl (xbar);
  m.data = mt_htonl (data);
  
  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));

  exception: return 0;
}

int sw_TestPort::resetXbar (int xbar)
{
  insist (this);
  return simpleCommand (xbar, TEST_RESET_XBAR);
  exception: return 0;
}

int sw_TestPort::getA2D (int xbar, int channel)
{  
  insist (this);
  
  a2d_message_t m;
  m.type = mt_htonl (TEST_READ_A2D);
  m.xbar = mt_htonl (xbar);
  m.data = mt_htonl (channel);

  if (!send ((char*) &m, sizeof (m), (char*) &m, sizeof (m)))
    return 0;

  return mt_htonl (m.data);
  exception: return 0;
}

int sw_TestPort::getVoltages (int xbar, int*low, int*high)
{
  insist (this);
  insist (low && high);
  
  *low = getA2D (xbar, OAL_CHANNEL);
  *high = getA2D (xbar, OAH_CHANNEL);

  return 1;
  exception: return 0;
}

int sw_TestPort::getVoltages (int xbar, int*low, int*high, int lowLimit, int highLimit)
{
  insist (this);
  insist (low && high);
  
  if (!getVoltages (xbar, low, high))
    return 0;
  
  if (*low > lowLimit) *low = lowLimit;
  if (*high < highLimit) *high = highLimit;
  
  return 1;
  exception: return 0;
}

int sw_TestPort::getLinkStatus (int xbar, char receive [16], char transmit [16])
{
  insist (this);
  insist (receive && transmit);  

  link_status_message_t r;
  test_message_t m;
  m.type = mt_htonl (TEST_READ_LINK_STATUS);
  m.xbar = mt_htonl (xbar);
  
  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;
  
  memcpy (receive, r.data.receive, 16);
  memcpy (transmit, r.data.transmit, 16);
  
  return 1;
  exception: return 0;
}

int sw_TestPort::setIcs (int value)
{
  insist (this);
  
  test_message_t r;
  write_ics_message_t m;
  m.type = mt_htonl (TEST_WRITE_ICS);
  m.val = mt_htonl (value);

  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));
  exception: return 0; 
  
}
int sw_TestPort::setBClock (int xbar, int value)
{
  insist (this);
  
  test2_change_bclk_t m;
  test_message_t r;

  m.type = mt_htonl (TEST2_CHANGE_BCLK);
  m.val = mt_htonl (value);

  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));
  exception: return 0;
}

int sw_TestPort::accessMicrocontroller (int port, int*receiveLength, char*buffer, int sendLength, ...)
{
  va_list ap;
  va_start (ap, sendLength);

  insist (this);
  insist (sendLength >= 0 && sendLength <= 96);

  access_uc_message_t m;
  
  m.type = mt_htonl (TEST_ACCESS_UC);
  m.port = mt_htonl (port);
  m.rxlen = receiveLength ? mt_htonl (*receiveLength) : 0;
  m.txlen = mt_htonl (sendLength);
  

  for (int i = 0; i < sendLength; i++)
  {
    int blah =  va_arg (ap, int);
    m.data [i] = (char) blah;
   }
  
  if (!send ((char*) &m, sizeof (m), (char*) &m, sizeof (m)))
  {
    va_end (ap);
    return 0;
  }
  
  if (receiveLength)
  {
    *receiveLength = mt_htonl (m.rxlen);
    if (*receiveLength > 0 && *receiveLength <= 96)
      memcpy (buffer, m.data, *receiveLength);
  }
  
  va_end (ap);
  
  return 1;
  exception: return 0;
}


int sw_TestPort::setChannel160 ()
{
  int dc = 0;
  char data [96];
  
  insist (this);

  return
    setIcs (0x606c01) &&
    accessMicrocontroller (0, &dc, data, 4, 0x21, 0x60, 0x6c, 0x01) &&
    accessMicrocontroller (0, &dc, data, 4, 0x21, 0x1e, 0x00, 0x00) &&
    accessMicrocontroller (0, &dc, data, 3, 0x2f, 0x1a, 0x05);
  
  exception: return 0;
}

int sw_TestPort::setChannel200 ()
{
  int dc = 0;
  char data [96];
  
  insist (this);

  return
    setIcs (0x60BC02) &&
    accessMicrocontroller (0, &dc, data, 3, 0x2f, 0x1a, 0x15) &&
    accessMicrocontroller (0, &dc, data, 4, 0x21, 0x60, 0x6c, 0x01) &&
    accessMicrocontroller (0, &dc, data, 4, 0x21, 0x00, 0xbc, 0x02);
  
  exception: return 0;
}

int sw_TestPort::getPacketCounters (int xbar, int good [NUM_PORTS], int bad [NUM_PORTS])
{
  insist (this);
  insist (good && bad);
  
  test_message_t m;
  m.type = mt_htonl (TEST_READ_PACKET_COUNTERS);
  m.xbar = mt_htonl (xbar);
  packet_counters_message_t r;

  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;

  for (int i = 0; i < NUM_PORTS; i++)
  {
    unsigned char high, low;
    
    high = r.data.counters [i].goodHigh;
    low =  r.data.counters [i].goodLow;    
    good [i] = high * 256 +  low;

    high = r.data.counters [i].crcHigh;
    low =  r.data.counters [i].crcLow;
    bad [i] = high  * 256 + low;
    
  }
  return 1;
  exception: return 0;
}

int sw_TestPort::setControl (int xbar, char control [NUM_PORTS])
{
  insist (this);
  insist (control);

  test_message_t r;
  control_message_t m;
  m.type = mt_htonl (TEST_WRITE_CONTROL);
  m.xbar = mt_htonl (xbar);
  
  for (int i = 0; i < NUM_PORTS; i++)
    m.data.control [i] = control [i];

  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));  
  
  exception: return 0;
}

int sw_TestPort::getControl (int xbar, char control [NUM_PORTS])
{
  insist (this);
  insist (control);

  test_message_t m;
  control_message_t r;
  m.type = mt_htonl (TEST_READ_CONTROL);
  m.xbar = mt_htonl (xbar);

  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;

  for (int i = 0; i < NUM_PORTS; i++)
    control [i] = r.data.control [i];
  
  return 1;
  exception: return 0;
}

int sw_TestPort::setHa (int on)
{
  insist (this);

  test_message_t m;
  m.type = on ? mt_htonl (TEST_TURN_HA_ON) : mt_htonl (TEST_TURN_HA_OFF);
  m.xbar = mt_htonl (0);
  
  return send ((char*) &m, sizeof (m), (char*) &m, sizeof (m));  

  exception: return 0;
}

int sw_TestPort::setAuto (int on)
{
  insist (this);

  insist (this);

  test_message_t m;
  m.type = on ? mt_htonl (TEST_TURN_AUTO_ON) : mt_htonl (TEST_TURN_AUTO_OFF);
  m.xbar = mt_htonl (0);
  
  return send ((char*) &m, sizeof (m), (char*) &m, sizeof (m));  


  exception: return 0;
}
  

double sw_TestPort::getVoltage (int xbar, int channel)
{
  insist (this);
  
  return (double) getA2D (xbar, channel) * VSCALE10;  

  exception: return 0;
}

int sw_TestPort::setSpeed (int xbar, char c1, char c2, char c3, char c4)
{
  insist (this);

  test_message_t r;
  speed_message_t m;
  
  m.type = mt_htonl (TEST_WRITE_SPEED);
  m.xbar = mt_htonl (xbar);
  m.data.bytes [0] = c1;
  m.data.bytes [1] = c2;
  m.data.bytes [2] = c3;
  m.data.bytes [3] = c4;
  
  return send ((char*) &m, sizeof (m), (char*) &r, sizeof (r));
  
  exception: return 0;
}

int sw_TestPort::getSpeed (int xbar, char speed [4])
{
  insist (this);
  insist (speed);
  
  test_message_t m;
  speed_message_t r;
  
  m.type = mt_htonl (TEST_READ_SPEED);
  m.xbar = mt_htonl (xbar);
  
  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;
  
  memcpy (speed, r.data.bytes, 4);
  return 1;
  
  exception: return 0;
}

int sw_TestPort::setSpeed (int xbar, int port, int fast)
{
  insist (this);
  insist (port >= 0 && port < NUM_PORTS);

  char speed [4];
  
  for (int i = 0; i < 4; i++)
    speed [i] = fast ? 0x0 : 0xff;

  char c, nc;
  c = fast ? 0x1 : 0x0;
  
  c = c << (port & 0x7);
  nc = ~( 1 << (port & 0x7));
  
  speed [0 + ((port >> 2) & 2)] &= nc;
  speed [0 + ((port >> 2) & 2)] |= c;

  speed [1 + ((port >> 2) & 2)] &= nc;
  speed [1 + ((port >> 2) & 2)] |= c;

  //  printf ("speed register is %x %x %x %x\n", speed [0], speed [1], speed [2], speed [3]);
  
  return setSpeed (xbar, speed [0], speed [1], speed [2], speed [3]);

  exception: return 0;
}




int sw_TestPort::getReceiveLinkStatusField (char receive [12 * 16 / 8], int port, int field)
{
  int e;
  insistf (receive && port >= 0 && port < NUM_PORTS && field >= 0 && field < 12);

  e = extractReceive (port, receive);
  return (e >> field) & 0x1;
  
  exception: return -1;
}


int sw_TestPort::getTransmitLinkStatusField (char transmit [4 * 16 / 8], int port, int field)
{
  int e;
  insistf (transmit && port >= 0 && port < NUM_PORTS && field >= 0 && field < 4);
  

  e = extractTransmit (port, transmit);

  return (e >> field) & 0x1;
  
  exception: return -1;
}


int sw_TestPort::getLinkStatus2 (int xbar, char receive [12 * 16 / 8], char transmit [4 * 16 / 8])
{
  insist (this);
  insist (receive && transmit);  

  link_status_message_t r;
  test_message_t m;
  m.type = mt_htonl (TEST_READ_LINK_STATUS);
  m.xbar = mt_htonl (xbar);
  
  if (!send ((char*) &m, sizeof (m), (char*) &r, sizeof (r)))
    return 0;

  memcpy (receive, r.data.receive, 12 * 16 / 8);
  memcpy (transmit, r.data.transmit + 8, 4 * 16 / 8);
  
  /*
    for (int i = 0; i < 12 * 16 / 8; i++)
    printFormat ("receive [%d] is %x", i, (unsigned char) receive [i]);

    for (int i = 0; i < 4 * 16 / 8; i++)
    printFormat ("transmit [%d] is %x", i, (unsigned char) transmit [i]);
  */
  return 1;
  exception: return 0;
}

int sw_TestPort::extractReceive(int port, char receive[12*16/8])
{
  int j,r,t;

  j = (12*port)/8;
  if (port&1) {
    r = receive[j];
    r<<=8;
    r &= 0xf00;
    t = receive[j+1];
    t&=0xff;
    r +=t;
  } else {
    r = receive[j];
    r <<=4;
    r &= 0xff0;
    t = receive[j+1];
    t >>=4;
    t &= 0x0f;
    r +=t;
  }
  return r;
}

int sw_TestPort::extractTransmit(int port, char transmit[4*16/8]) 
{
  int j, t;
  j = (4*port)/8;
  if (port&1) {
    t = transmit[j] & 0x0f;
  } else {
    t = (transmit[j]>>4)&0x0f;
  }
  return t;
}
