/*
  pi_Ping.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "insist.h"
#include "pi_Ping.h"
#include "mt_MapFile.h"
#include "mt_Mapper.h"
#include "mt_htonl.h"

pi_Ping::pi_Ping (mt_Node*node, mt_Network*network, mt_Calculator*calculator, pi_Options*options) : mt_Job (node, network)
{
  this->options = *options;
  this->calculator = calculator;
  currentSwitch = 0;
  counters = 0;
  phases = 0;
  numCounters = 0;
  numSent = numReceived = 0;
  this-> node = 0;
  numNodesSeen = 0;
  numBadPhases = 0;
  myself = 0;
}

pi_Ping::pi_Ping (mt_Node*node, mt_Network*network, mt_Graph*graph, pi_Options*options) : mt_Job (node, network)
{
  
  insist (node && node->isHost ());
  this->options = *options;

  this->calculator = 0;

  this->graph.clone (graph);
  myself = this->graph.getHost (node->getTypeIndex ());
  insist (myself);
  
  currentSwitch = 0;
  counters = 0;
  phases = 0;
  numCounters = 0;
  numSent = numReceived = 0;
  this->node = 0;
  numNodesSeen = 0;
  numBadPhases = 0;
  exception:;
}


pi_Ping::~pi_Ping ()
{
  if (phases) delete phases;
  if (counters) delete [] counters;
}

int pi_Ping::send (mt_Route*route, char*p, int length)
{
  mt_Node*fn;
  int fin;

  insist (this);
  insist (p);
  insist (length > 0);

  insist (options.testHosts || (myself->follow (route, &fn, &fin) && fn == myself));

  if (options.verbose)
    printFormat ("sending length %d to route %s\n", length, route->toString ());
  
  if (length % 4) length += 4 - length % 4;
  if (length > mtu)
    length = mtu;
  
  insist (network);

  while (!network->isAborted () && !network->send (this, route, p, length))
  {
    char*pp;
    int ll;
    
    if (!wait (mt_Network::SEND_DONE, 0, &pp, &ll))
      return 0;
  }
 
  numSent++;
  return 1;
  exception: return 0;
}

int pi_Ping::wait (int timeout)
{
  insist (this);
  insist (network);
  
  char*p;
  int length;
  
  network->setTimer (this, timeout);
  return wait (mt_Network::TIMEOUT, 0, &p, &length);

  exception: return 0;
}

int pi_Ping::wait (int event, int type, char**p, int*length)
{
  insist (this);
  insist (network);
  insist (p && length);

  int e;

  while ((e = network->wait (p, length)))
  {    
    if (e == mt_Network::RECEIVE)
    {
      insist (*p);
      insist (*length);
      
      mt_Message*m;
      m = (mt_Message*)*p;
      
      int mtype;
      int stype;
      int mphase;
      
      mtype = mt_htons (m->type);
      stype = mt_htons (m->subtype);
      mphase = mt_htonl (m->phase);

      if (mtype != mt_Message::GM_TYPE || stype < 0 || stype >= mt_Message::NUM_TYPES)
      {
	printFormat ("bad type %d", mtype);
	network->freePacket (*p);
	continue;
      }
      
      if (e == event && type == stype)
	return 1;
            
      mt_Node*node;
      int r, phase, from, to;

      if (!(r = phases->peek (&node, &phase, &from, &to)) || mphase != phase)
      {
	if (options.verbose)
	{
	  if (r)
	    printFormat ("bad phase %d, expected %d", mphase, phase);
	  else
	    printFormat ("bad phase %d, none expected", mphase);
	}
	
	addBadPhase (node, from, to);
	network->freePacket (*p);
	numBadPhases++;
	continue;
      }
      
      r = phases->get (&node, &phase, &from, &to);
      insist (r);

      if (handlers [stype])
	(this->*handlers [stype]) (node, from, to, *p, *length);

      network->freePacket (*p);
    }
    else if (e == event)
      return 1;
    else if (e == mt_Network::TIMEOUT)
      return 0;
  }

  if (network->isAborted ())
  {
    printFormat ("network aborted. is it a deadlock?");
    return 0;
  }
  
  insist (0);
  exception: return 0;
}

void pi_Ping::handleReply (mt_Node*n, int from, int to, char*p, int length)
{
  insist (this);
  insist (p);
  
  insist (n);
  insist (from >= 0 && from < mt_Switch::NUM_PORTS);
  insist (to >= 0 && to < mt_Switch::NUM_PORTS);
  
  if ((unsigned) length < sizeof (mt_ReplyMessage))
  {
    printFormat ("bad length");
    return;
  }
  
  mt_ReplyMessage*m;
  m = (mt_ReplyMessage*) p;
  m->swap ();

  insist (m->type == mt_Message::GM_TYPE && m->subtype == mt_Message::REPLY);
  
  {
    mt_Address a (m->address);
    
    if (options.verbose)
    {
      printFormat ("reply from %s: board_id=%s. gm_id=%d. map_v=%d. host_t=%d level=%d", 
		   m->getHostname (), a.toString(), m->gmId, m->mapVersion, m->hostType, m->level);
    }
    
    insist (counters);
    insist (n);
    
    insist (strstr (n->getName (), m->getHostname ()) ||
	    strstr (m->getHostname (), n->getName ()));
    insist (n->getAddress ()->isZero () || !a.compare (n->getAddress ()));
    insist (n->getTypeIndex () >= 0 && n->getTypeIndex () < numCounters);
    counters [n->getTypeIndex ()].addReceived (from, to);
    numReceived++;
  }
  
  exception: return;
}

void pi_Ping::printPacket (char*p, int length)
{
  int n = 0;
  char*buffer = (char*) malloc (mt_Network::MTU * 2);

  insist (buffer);
  insist (length <= mt_Network::MTU);
  
  for (int i = 0; i < length && n < mt_Network::MTU - 8; i++)
    n += sprintf (buffer + n, "%02x%c", (unsigned char) (0xff & p[i]), i % 8 == 7 ? '\n' : ' ');
  printFormat ("%s", buffer);
  
  free (buffer);
  
  exception:;
}

void  pi_Ping::handleProbe (mt_Node*node, int from, int to, char*p, int length)
{
  insist (this);
  insist (p);

  insist (node);
  insist (from >= 0 && from < mt_Switch::NUM_PORTS);
  insist (to >= 0 && to < mt_Switch::NUM_PORTS);

  if ((unsigned) length < sizeof (mt_ProbeMessage))
  {
    printFormat ("bad length");
    return;
  }
  
  mt_ProbeMessage*m;
  m = (mt_ProbeMessage*) p;
  m->swap ();

  insist (m->type == mt_Message::GM_TYPE && m->subtype == mt_Message::PROBE);
  
  if (options.verbose)
  {
    printFormat ("got a switch probe message from %s of length %d", node->getName (), length);
  }

  if (options.fillFast && options.minSize == 0 && options.maxSize == 0)
  {
    insist (length >= options.size);

    for (int i = sizeof (mt_ProbeMessage) + 1; i < options.size - 1; i++)
    {
      if (((p [i] + 1) & 0xff) != (p [i + 1] & 0xff))
      {
	printFormat ("mistake in packet contents at position %i", i);
	printPacket (p, length);
	break;
      }
    }
  }
  
  insist (counters);
  insist (node);
  insist (node->getTypeIndex () >= 0 && node->getTypeIndex () < numCounters);
  counters [node->getTypeIndex ()].addReceived (from, to);
  numReceived++;
  
  exception: return;
}


int pi_Ping::initialize ()
{
  char hostname [mt_Network::HOSTNAME_LENGTH + 1];
  int hostType;
  phase = 0;
  
  network = getNetwork ();
  insist (network);
  
  if (!network->open (this, &address, hostname, &hostType, options.unit))
    return 0;
  
  if (!(mtu = network->getReceiveMtu ()))
    return 0;

  if (!(phases = new pi_Queue (network->getNumReceiveBuffers ())))
    return 0;
  
  if (options.routeLength != -1)
  {
    route.fromBytes (options.hops, options.routeLength);
    if (options.verbose)
      printFormat ("trying %s", route.toString ());
  }
  if (myself)
  {
    insist (myself == graph.getNode (myself->getName ()));
  }
  else
  {
    if (!(*options.mapFile && *options.myself))
    {
      printFormat ("pi_Ping::initialize: specify either map-file and -myself, or -route");
      return 0;
    }

    insist (*options.mapFile && *options.myself);
    mt_MapFile mf (options.mapFile, mt_File::_READ);

    if (!mf.read (&graph))
    {
      printFormat ("couldn't read %s", options.mapFile);
      return 0;
    }
  
    if (!(myself = graph.getNode (options.myself)))
    {
      printFormat ("node \"%s\" is not in %s", options.myself, options.mapFile);
      return 0;
    }
    
    if (options.testHosts)
    {
      insist (calculator);
      if (!calculator->initialize (&graph, graph.getHost (0)))
      {
	printFormat ("couldn't calculate routes");
	return 0;
      }
      numCounters = graph.getNumHosts ();
      insist (numCounters > 0);
      counters = new pi_Counter [numCounters];
      insistp (counters, ("new pi_Counters failed"));
      for (int i = 0; i < numCounters; i++)
	counters [i].setNode (graph.getHost (i));
    }
    currentHost = -1;
    if (*options.node)
    {  
      if (!(node = graph.getNode (options.node)))
      {
	printFormat ("node \"%s\" is not in %s", options.node, options.mapFile);
	return 0;
      }
      if (!node->isHost () && options.testHosts || !node->isSwitch () && options.testSwitches)
      {
	printFormat ("node \"%s\" type mismatch", options.node);
	return 0;
      }
    }

    if (!deadlock.clone (&graph))
      return 0;

    if (options.testSwitches || options.testXbars)
    {
      minPort = 0;

      currentSwitch = myself->getNode (0);
      
      insistp (currentSwitch->isSwitch (), ("%s is not a switch", currentSwitch->getName ()));
      insistp (currentSwitch && currentSwitch->isSwitch (), ("mapper node not connected to switch"));

      maxPort = currentSwitch->getMaxNodes () - 1;
      if (options.port >= 0)
	minPort = maxPort = currentPort = options.port;
      currentPort = minPort;

      graph.clearMarks ();
      mt_Route r;
      myself->setRoute (0, r);
      
      currentSwitch->setIn (myself->getOpposite (0));

      numCounters = graph.getNumSwitches ();
      insist (numCounters > 0);
      counters = new pi_Counter [numCounters];
      insistp (counters, ("new pi_Counters failed"));
      
      for (int i = 0; i < numCounters; i++)
	counters [i].setNode (graph.getSwitch (i));
    }
  }
  if (options.fillFast)
    fillFast (buffer, mt_Network::MTU);
  else if (options.fillSlow)
    fillSlow (buffer, mt_Network::MTU);
  
  return 1;
  exception: return 0;
}

int pi_Ping::startLoopVariables ()
{
  insist (this);
  
  currentHost = -1;
  numNodesSeen = 0;

  if (options.testSwitches)
  { 
    minPort = 0;
    currentSwitch = myself->getNode (0);
    maxPort = currentSwitch->getMaxNodes () - 1;
    if (options.port >= 0)
      minPort = maxPort = currentPort = options.port;
    currentPort = minPort;
    graph.clearMarks ();
    currentSwitch->setMark (mt_Node::UP);
    switches.put (currentSwitch);
    currentSwitch = 0;
  }
  
  return 1;
  exception: return 0;
}

int pi_Ping::start ()
{
  insist (this);
  
  if (!initialize ())
    return 0;
  do
  {
    startLoopVariables ();
    ping ();

    if (options.forever)
      print ();
  }
  while (options.forever);

  if (options.testHosts)
    calculator->cleanup ();

  return 1;
  
  exception: return 0;
}

int pi_Ping::ping ()
{
      
  mt_Route r;
  char*p;
  int minSize, maxSize, incrementSize, count, length;
  count = -1;
  
  if (options.minSize && (options.testSwitches || options.testXbars))
  {
    minSize = options.minSize;
    maxSize = options.maxSize;
    
    incrementSize = options.incrementSize;
  }
  else
  {
    minSize = maxSize = 0;
    incrementSize = 1;
  }
  
  if (options.size)
    minSize = maxSize = options.size;
  
  if (minSize > mtu)
    minSize = mtu;
  if (maxSize > mtu)
    maxSize = mtu;
  
  count = options.count;
  
  insist (minSize <= maxSize);
  insist (incrementSize > 0);

  setHandler (mt_Message::PROBE, mt_fp (pi_Ping::handleProbe));
  setHandler (mt_Message::REPLY, mt_fp (pi_Ping::handleReply));

  if (options.testSwitches || options.testHosts)
  {
    while ((p = getNextMessage (&length, &r)))
    {
      if (!options.fearless && options.testSwitches && deadlock.deadlockable (&r, myself->getTypeIndex (), myself->getTypeIndex ()))
	continue;
      
      for (int size = minSize; size <= maxSize; size += incrementSize)
      {
	phases->clear ();
      
	for (int i = 0; i < count; i++)
	{
	  if (phases->full () && !wait (options.timeout))
	    return 0;
	
	  if (phases->full ())
	    phases->clear ();

	  phases->put (expectedNode, phase, expectedFrom, expectedTo);
	  counters [expectedNode->getTypeIndex ()].addSent (expectedFrom, expectedTo);
	  setPhase (p, phase++);

	  send (&r, p, size > length ? size : length);
	}
	if (!wait (options.timeout))
	  return 0;
      }
    }
  }
  
  if (options.testXbars)
  {
    computeSwitchRoutes ();

    for (int i = 0; i < graph.getNumSwitches (); i++)
    {
      if (node && i)
	break;
      
      currentSwitch = graph.getSwitch (i);
      insist (currentSwitch && currentSwitch->isSwitch ());

      if (node)
	currentSwitch = node;
      
      numNodesSeen++;
      
      int numNodes = currentSwitch->getMaxNodes ();
      
      for (int j = 0; j < numNodes; j++)
      {
	mt_Node*jn = currentSwitch->getNode (j);
	if (!jn || !jn->isSwitch ())
	  continue;

	if (options.inPort != -1 && options.inPort != j)
	  continue;
	
	for (int k = 0; k < numNodes; k++)
	{
	  mt_Node*kn = currentSwitch->getNode (k);
	  if (!kn || !kn->isSwitch ())
	    continue;

	  //if (!options.fearless && k == j) continue;
	
	  if (options.outPort != -1 && options.outPort != k)
	    continue;
	  
	  mt_ProbeMessage m (0, phase);
	  m.swap ();
	  length = sizeof (mt_ProbeMessage);
	  
	  mt_Route a (jn->getRoute (0), currentSwitch->getOpposite (j) - jn->getIn ());

	  mt_Node*fn;
	  int fin;
	  insist (myself->follow (&a,  &fn, &fin) && fn == currentSwitch);

	  mt_Route b (&a,  k - j);

	  insist (myself->follow (&b,  &fn, &fin) && fn == kn);

	  mt_Route c;
	  
	  if (kn != currentSwitch)
	  {
	    mt_Route t;
	    mt_Route d (&t, 0);
	    c = d;
	  }
	  mt_Route e (*currentSwitch->getRoute (0));
	  e.invert ();
	  mt_Route f (&b, &c);
	  mt_Route g (&f, currentSwitch->getIn () - k, &e);

	  if (!options.fearless && deadlock.deadlockable (&g, myself->getTypeIndex (), myself->getTypeIndex ()))
	    continue;

	  for (int size = minSize; size <= maxSize; size += incrementSize)
	  {
	    for (int cc = 0; cc < count; cc++)
	    {
	      if (phases->full () && !wait (options.timeout))
		return 0;
	
	      if (phases->full ())
		phases->clear ();
	    
	      expectedNode = currentSwitch;
	      expectedFrom = j;
	      expectedTo = k;
	      
	      phases->put (expectedNode, phase, expectedFrom, expectedTo);
	      counters [expectedNode->getTypeIndex ()].addSent (expectedFrom, expectedTo);
	      setPhase ((char*) &m, phase++);
	      memcpy (buffer, (char*)&m, sizeof (mt_ProbeMessage));
	      send (&g, buffer, size > length ? size : length);
	    }
	  }
	  if (options.outPort != -1)
	    break;
	}
	if (options.inPort != -1)
	  break;
      }
    }
  }

  insist (numSent >= numReceived);
  int timeout;
  timeout = options.timeout * (numSent - numReceived);
  
  if (!wait (timeout < 1000 * 1000 ? timeout : 1000 * 1000))
    return 0;
  
  clearHandler (mt_Message::PROBE);
  clearHandler (mt_Message::REPLY);
  
  return 1;
  exception: return 0;
}


int pi_Ping::print (int interrupted)
{
  insist (this);

  insist (*options.node || interrupted || options.testXbars ||
	  options.testHosts && numNodesSeen == graph.getNumHosts () - 1 ||
	  options.testSwitches && numNodesSeen == graph.getNumSwitches ());

  for (int i = 0; i < numCounters; i++)
  {
    if (!node || node == counters [i].getNode ())
      counters [i].print (myself, options.printXbars);
  }
  
  printFormat ("\ntested %d node%s", numNodesSeen, numNodesSeen != 1 ? "s" : "");

  if (numBadPhases + numReceived == numSent)
    printFormat ("received all %d messages", numSent);
  else
    printFormat ("received %d/%d messages", numReceived + numBadPhases, numSent);

  if (numBadPhases)
    printFormat ("there %s %d late message%s", numBadPhases != 1 ? "were" : "was", numBadPhases,
		 numBadPhases != 1 ? "s" : "");
  
  int missing;
  missing = numSent - numReceived - numBadPhases;
  
  if (missing)
    printFormat ("there %s %d missing message%s", missing != 1 ? "were" : "was",
		 missing, missing != 1 ? "s" : "");
  
  exception: return 0;
}

int pi_Ping::setPhase (char*p, int phase)
{
  mt_Message*m = (mt_Message*) p;
  
  insist (this);
  insist (p);
  
  m->phase = mt_htonl (phase);
  
  exception: return 0;  
}

int pi_Ping::computeSwitchRoutes ()
{
  int numSwitches = 0;
  insist (this);
  insist (myself);

  mt_Switch*s;
  s = (mt_Switch*) myself->getNode (0);
  insist (s);
  
  graph.clearMarks ();
  
  s->setMark (mt_Node::UP);
  s->setIn (myself->getOpposite (0));
  
  switches.empty ();
  switches.put (s);

  while ((s = (mt_Switch*) switches.get ()))
  {
    numSwitches++;
    
    int maxNodes = s->getMaxNodes ();
    
    for (int i = 0; i < maxNodes; i++)
    {
      mt_Node*n = s->getNode (i);
	  
      if (n && n->isSwitch () && !n->isMarked ())
      {
	mt_Node*fn;
	int fin;

	insist (n->isSwitch ());
	    
	mt_Route r (s->getRoute (0), i - s->getIn ());
	    
	n->setIn (s->getOpposite (i));
	n->setRoute (0, r);
	n->setMark (mt_Node::UP);
	insist (myself->follow (n->getRoute (0),  &fn, &fin) && fn == n);
	
	switches.put (n);
      }
    }
  }

  insist (numSwitches == graph.getNumSwitches ());
  return numSwitches;
  exception: return 0;
}

char*pi_Ping::getNextMessage (int*length, mt_Route*r)
{
  mt_Route inverse;
  mt_Node*fn;
  int fin;
    
  insist (this);
  insist (length && r);
  insist (network);

  if (options.testHosts)
  {
    mt_Node*n;
    n = 0;
    
    insist (myself);

    if (node)
    {
      if (numNodesSeen)
	return 0;
      
      numNodesSeen++;
      n = node;
      currentHost = graph.getNumHosts ();
    }
    else if (currentHost < graph.getNumHosts ())
    {
      if (++currentHost == myself->getTypeIndex ())
	currentHost++;
      if (currentHost < graph.getNumHosts ())
	n = graph.getHost (currentHost);
      else return 0;
      numNodesSeen++;
    }
    else return 0;   

    insist (n->isHost ());
    calculator->getRoute (myself->getTypeIndex (), n->getTypeIndex (), 0, r);
    
    expectedTo = 0;
    expectedNode = n;
    
    inverse = *r;
    inverse.invert ();
     
    mt_ScoutMessage m (0, phase, &inverse, &address, 0, 0);
    m.swap ();
    memcpy (buffer, (char*)&m, sizeof (mt_ScoutMessage));
    *length = sizeof (mt_ScoutMessage);

    if (options.verbose)
      printFormat ("testing host \"%s\"", n->getName ());
    
    return buffer;
  }
  else if (options.testSwitches)
  {
    for (;;)
    {
      if (currentPort > maxPort || !currentSwitch)
      {
	if (!(currentSwitch = (mt_Node*) switches.get ()))
	  return 0;
	
	expectedFrom = currentSwitch->getIn ();
	
	if (!node || node == currentSwitch)
	  numNodesSeen++;
	
	int maxNodes = currentSwitch->getMaxNodes ();
	
	for (int i = 0; i < maxNodes; i++)
	{
	  mt_Node*n = currentSwitch->getNode (i);
	  
	  if (n && n->isSwitch () && !n->isMarked ())
	  {
	    insist (n->isSwitch ());
	    
	    mt_Route r (currentSwitch->getRoute (0), i - currentSwitch->getIn ());
	    
	    n->setIn (currentSwitch->getOpposite (i));
	    n->setMark (mt_Node::UP);
	    n->setRoute (0, r);
	    insist (myself->follow (n->getRoute (0),  &fn, &fin) && fn == n);
	    switches.put (n);
	  }
	}
	currentPort = 0;
	maxPort = currentSwitch->getMaxNodes () - 1;
      }
      mt_Node*n = currentSwitch->getNode (currentPort);

      if ((!node || currentSwitch == node) && (n && n->isSwitch () || n == myself) && (options.port < 0 || currentPort == options.port))
	break;
      currentPort++;
    }
    
    if (!currentSwitch)
      return 0;
    
    expectedNode = currentSwitch;
    expectedTo = currentPort;
    
    insist (currentSwitch);
    insist (currentPort >= 0 && currentPort < currentSwitch->getMaxNodes ());
    insist (currentSwitch->getNode (currentPort));
    insist (currentSwitch->getNode (currentPort)->isSwitch () || currentSwitch->getNode (currentPort) == myself);

    mt_ProbeMessage m (0, phase);
    m.swap ();
    mt_Route a;

    myself->follow (currentSwitch->getRoute (0),  &fn, &fin);
    insist (fn == currentSwitch);
    insist (fin == currentSwitch->getIn ());

    int relativePort = currentPort - fin;
    
    if (relativePort)
    {
      mt_Route t (currentSwitch->getRoute (0), relativePort, 0, -relativePort);
      mt_Route tt (currentSwitch->getRoute (0), relativePort, -relativePort);
      a = currentSwitch->getNode (currentPort) == currentSwitch ? tt : t;
    }
    else
    {
      mt_Route t (currentSwitch->getRoute (0), 0);
      a = t;
    }
    mt_Route b (*currentSwitch->getRoute (0));    
    
    b.invert ();
    mt_Route t (&a, &b);
    
    if (currentPort == 0 && currentSwitch == myself->getNode (0))
    {
      mt_Route a;
      mt_Route b (&a, 0);
      t = b;
    }
    *r = t;
    *length = sizeof (mt_ProbeMessage);
    memcpy (buffer, (char*)&m, sizeof (mt_ProbeMessage));
    
    insist (myself->follow (currentSwitch->getRoute (0),  &fn, &fin) && fn == currentSwitch);
    insist (myself->follow (r, &fn, &fin) && fn == myself);
    
    if (options.verbose)
      printFormat ("testing switch \"%s\", port %d", currentSwitch->getName (), currentPort);
    
    currentPort++;
    
    return buffer;
  }
  exception: return 0;
}


void pi_Ping::dump (FILE*)
{
}
int pi_Ping::willWait ()
{
  return 1;
}

void pi_Ping::receive (int, char*, int)
{
  insist (0);  
  exception:return;
}

int pi_Ping::addBadPhase (mt_Node*n, int from, int to)
{
  insist (this);
  
  insist (n);
  insist (from >= 0 && from < mt_Switch::NUM_PORTS);
  insist (to >= 0 && to < mt_Switch::NUM_PORTS);
  
  counters [n->getTypeIndex ()].addReceived (from, to);
  exception: return 0;
}

int pi_Ping::setHandler (int type, pi_MessageHandler handler)
{
  insist (this);  
  insist (type >= 0 && type < mt_Message::NUM_TYPES);
  handlers [type] = handler;
  
  return 1;
  exception: return 0;
}

int pi_Ping::clearHandler (int type)
{
  insist (this);
  insist (type >= 0 && type < mt_Message::NUM_TYPES);
  handlers [type] = 0;

  return 1;
  exception: return 0;  
}

  
pi_Queue::pi_Queue (int size)
{
  insist (size > 0);
  this->size = size;
  phases = new pi_Phase [size];
  insistp (phases, ("alloc failed\n"));
  exception:;
}

pi_Queue::~pi_Queue ()
{
  if (phases) delete [] phases;
}

int pi_Queue::full ()
{
  return (tail + 1 == head || (tail == size - 1 && head == 0));
}

int pi_Queue::empty ()
{
  return tail == head;
}

int pi_Queue::clear ()
{
  insist (this);
  tail = head = 0;
  return 1;
  exception: return 0;
}

int pi_Queue::put (mt_Node*node, int phase, int from, int to)
{
  insist (this);
  insist (node);
  
  if (full ())
    return 0;

  phases [tail].node = node;
  phases [tail].phase = phase;
  phases [tail].from = from;
  phases [tail].to = to;
  
  if (++tail == size)
    tail = 0;
  return 1;
  exception: return 0;
}

int pi_Queue::peek ()
{
  insist (this);
  return !empty ();
  exception: return 0;
}

int pi_Queue::peek (mt_Node**node, int*phase, int*from, int*to)
{
  insist (this);
  insist (node && phase && from && to);
  
  if (empty ())
    return 0;

  *node = phases [head].node;
  *phase = phases [head].phase;
  *from = phases [head].from;
  *to = phases [head].to;

  return 1;
  exception: return 0;
}

int pi_Queue::get (mt_Node**node, int*phase, int*from, int*to)
{
  insist (this);
  insist (node && phase && from && to);

  if (empty ())
    return 0;  

  *node = phases [head].node;
  *phase = phases [head].phase;
  *from = phases [head].from;
  *to = phases [head].to;

  if (++head == size)
    head = 0;
  return 1;
  exception: return 0;
}

pi_Counter::pi_Counter ()
{
  node = 0;
  for (int i = 0; i < mt_Switch::NUM_PORTS; i++)
    for (int j = 0; j < mt_Switch::NUM_PORTS; j++)
      sent [i][j] = received [i][j] = 0;
}

int pi_Counter::print (mt_Node*myself, int xbars)
{
  char buffer [mt_File::MAX_LINE];
  int totalSent = 0;
  int totalReceived = 0;

  insist (this && myself);
  
  for (int i = 0; i < mt_Switch::NUM_PORTS; i++)
  {  
    for (int j = 0; j < mt_Switch::NUM_PORTS; j++)
    {
      totalSent+=sent [i] [j];
      totalReceived += received [i] [j];
    }
  }

  if (totalSent && (totalReceived != totalSent))
  {
    if (node->isHost ())
    {
      printFormat ("\nhost %s:", node->getName ());
      printFormat ("the route from %s to %s is %s", myself->getName (), node->getName(), node->getRoute (0)->toString ());
    }
    printFormat ("received %d/%d messages (%d%% packet loss)", totalReceived, totalSent, 
		 100 - (int)((double) 100 * (double) totalReceived / (double) totalSent));
  }
  
  if (node->isSwitch () && xbars)
  {
    printFormat ("switch %s:", node->getName ());
    printFormat ("the route from %s to %s is %s", myself->getName (), node->getName(), node->getRoute (0)->toString ());

    for (int i = 0; i < mt_Switch::NUM_PORTS; i++)
    {
      char*line = buffer;
      line += sprintf (line, "%2.2d   ", i);
    
      for (int j = 0; j < mt_Switch::NUM_PORTS; j++)
      {
	if (sent [i] [j])
	{
	  if (!received [i][j])
	    line += sprintf (line, "%s", "0 ");
	  else if (sent [i] [j] != received [i][j])
	    line += sprintf (line, "%s", "$ ");
	  else
	    line += sprintf (line, "%s", "1 ");
	}
	else line += sprintf (line, "%s", ". ");
      }
      printFormat ("%s", buffer);
    }
  }
 
  return 1;
  exception: return 0;
}

int pi_Counter::addSent (int from, int to)
{
  insist (this);
  insist (from >= 0 && from < mt_Switch::NUM_PORTS);
  insist (to >= 0 && to < mt_Switch::NUM_PORTS);

  insist (node);
  insist (node->isSwitch () || (!to && !from));
  sent [from] [to] ++;
  
  return 1;
  exception: return 0;
}

int pi_Counter::addReceived (int from, int to)
{
  insist (this);
  insist (from >= 0 && from < mt_Switch::NUM_PORTS);
  insist (to >= 0 && to < mt_Switch::NUM_PORTS);

  insist (node);
  insist (node->isSwitch () || (!from && !to)); 
  insist (received [from] [to] < sent [from] [to]);
  
  received [from] [to]++;
  
  return 1;
  exception: return 0;
  
}
int pi_Counter::setNode (mt_Node*n)
{
  insist (this);
  insist (n);  

  node = n;  
  return 1;
  exception: return 0;
}


mt_Node*pi_Counter::getNode ()
{
  insist (this);
  return node;
  exception: return 0;
}

int pi_Ping::fillFast (char*p, int length)
{
  insist (this);
  
  for (int i = 0; i < length; i++)
    p [i] = i & 0xff;
  return 1;
  exception: return 0;
}

int pi_Ping::fillSlow (char*p, int length)
{
  insist (this);
  
  mt_Component::srand ((int)&p);

  for (int i = 0; i < length; i++)
    p [i] = mt_Component::rand (255) & 0xff;

  return 1;
  exception: return 0;
}

  
