/*
  nw_Simulation.c
  nway
  finucane@myri.com (David Finucane)
*/

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "insist.h"
#include "nw_Simulation.h"
#include "sm_Host.h"
#include "mt_Queue.h"

class nw_Message
{
  public:
  int number;
  int from;
  int to;
  int priority;
  int ack;
  mt_Int64 startTime;
};

class nw_Ack : public mt_Queueable
{
  public:
  nw_Message*message;
};

nw_Job::~nw_Job ()
{
  if (acks) delete [] acks;
  if (receives) delete [] receives;
  if (sends) delete [] sends;
  if (hosts) delete hosts;
  if (trafficPositions)
  {
    free (trafficPositions);
    trafficPositions = 0;
  }
  if (trafficDestinations)
  {
    free (trafficDestinations);
    trafficDestinations = 0;
  }
  nw_Ack*a;
  while ((a = (nw_Ack*) ackQueue.get ()))
    delete a;
}

nw_Job::nw_Job (mt_Node*node, mt_Network*network, nw_Simulation *sim) : mt_Job (node, network)
{
  acks = 0;
  receives = 0;
  sends = 0;
  lastLatency = 0;
  
  this->sim = sim;
  
  mt_Graph*g = getNetwork ()->getGraph ();
  insist (g);

  numHosts = g->getNumHosts ();
  insist (numHosts > 0);

  me = node->getRouteIndex ();
  insist (me >= 0 && me < numHosts);
  
  receives = new int [numHosts];
  sends = new int [numHosts];
  acks = new int [numHosts];
  
  insistp (acks && receives && sends, ("nw_Job::nw_Job: alloc failed"));

  memset (acks, 0, numHosts);
  memset (receives, 0, numHosts);
  memset (sends, 0, numHosts);
  hosts = 0;
  
  if (sim->numLevels)
  {
    hosts = new mt_Node* [numHosts];
    insistp (hosts, ("nw_Job::nw_Job:: alloc failed"));

    network->getGraph ()->fillLevels (node, hosts, levels, hostsPerLevel, sim->numLevels + 1);
  }  
  if (*sim->trafficFile && !trafficDestinations)
    parseTrafficFile (sim->trafficFile, g);
  
  exception: return;
}

int nw_Job::start ()
{
  numSends = 0;
  numReceives = 0;
  to = 0;
  mt_Address address;
  char name [mt_Network::HOSTNAME_LENGTH + 1];
  int type;
  lastIncrement = 0;
 
  for (int i = 0; i < numHosts; i++)
    acks [i] = 1;
  
  if (!getNetwork ()->open (this, &address, name, &type, 0))
    return 0;

  this->sendAfterDelay ();
  
  return 1;
}

void nw_Job::receive (int event, char*p, int length)
{
  insist (this);

  switch (event)
  {
    case mt_Network::RECEIVE:
      {
	insist (!getNeverReceive (me));
      
	insist ((unsigned)length >= sizeof (nw_Message));
	nw_Message*m;
	m = (nw_Message*) p;
	insist (m->to == me);
	insist (m->to >= 0 && m->to < numHosts);

	if (m->ack)
	{
	  acks [m->from] = 1;
	  send ();
	}
	else
	{
	  insist(m->priority==0 || m->priority==1);
	  mt_Int64 latency = getNetwork()->getCurrentTime() - m->startTime;

	  lastLatency = latency;
	  insist (lastLatency > 0);
      
	  sim->latencyTotal[m->priority] += (double)latency;
	  sim->latencyNum[m->priority]++;
	  int bin;
	  bin = latency / nw_Simulation::BIN_SIZE;
	  insist(bin>=0);
	  if (bin >= nw_Simulation::NUM_BINS)
	    bin = nw_Simulation::NUM_BINS-1;
	  sim->lbinNum[m->priority][bin]++;

	  numReceives++;
	  receives [m->from]++;

	  if (sim->sendAcks)
	  {
	    ackQueueAppend (m->from);
	    ackQueueFlush ();
	  }
	  
	}	
	break;
      }
    case mt_Network::SEND_DONE:
      ackQueueFlush ();
      sendAfterDelay ();
      break;
    case mt_Network::TIMEOUT:
      send ();
      break;
    default:
      insist (0);
  }
  exception: return;
}

void nw_Job::sendAfterDelay()
{
  int delay;

  if (sim->meanDelay == 0)
    delay = 0;
  else
    delay = (int)(-(sim->meanDelay)*log(((double)::rand())/((double)RAND_MAX)));

  insist (delay >= 0);

  if (delay > 0)
    getNetwork()->setTimer (this, delay);
  else
    send();

  exception: return;
}

int nw_Job:: getNeverSend (int i)
{
  
  mt_Graph*g = getNetwork ()->getGraph ();
  insist (g);

  insist (numHosts == g->getNumHosts ());
  
  insist (i >= 0 && i < numHosts);
  return ((sm_Host*) g->getHost (i))->getNeverSend ();
  
  exception: return 0;
}

int nw_Job:: getNeverReceive (int i)
{
  
  mt_Graph*g = getNetwork ()->getGraph ();
  insist (g);

  insist (numHosts == g->getNumHosts ());
  
  insist (i >= 0 && i < numHosts);
  return ((sm_Host*) g->getHost (i))->getNeverReceive ();
  
  exception: return 0;
}

int nw_Job::pickHostFromLevel (int*host, int*delay, int level)
{
  insist (this);
  insist (host && delay);
  insist (sim->numLevels && hosts);
  insist (level >= 0 && level <= sim->numLevels);
  insist (hosts);
  insist (levels [level] >= &hosts [0] && levels [level] <= &hosts [numHosts]);

  *delay = 0;
  *host = 0;

  if (!hostsPerLevel [level])
  {
    if (lastLatency)
      *delay = lastLatency;
    else
      *delay = level * 100;
    *host = -1;
    return 1;
  }
  
  int r;
  r = mt_Component::rand (hostsPerLevel [level]);
  insist (r >= 0 && r < hostsPerLevel [level]);

  mt_Node*n;
  
  n = *(levels [level] + r);
  insist (n);
  insist (n->getDistance (getNode()) == level || level == sim->numLevels);

  *host = n->getRouteIndex ();
  return 1;
  exception: return 0;
}


int nw_Job::getNextDestination (int*to, int*delay)
{
  insist (this);
  insist (to && delay);

  if (hasTraffic ())
  {
    *to = getTrafficDestination (me);
    if (*to < 0)
      return 0;
    return 1;
  }
  
  if (sim->numLevels)
  {
    double r = (double) rand (10000) / (double)10000;

    double total = 0;

    int level;
    for (level = 0; level < sim->numLevels; level++)
    {
      total += sim->levelPercents [level];
      insist (total >= 0 && total <= 1.0);

      if (r <= total)
	break;
    }
    insist (level >= 0 && level < sim->numLevels);

    return pickHostFromLevel (to, delay, level);
  }
  else
  {
    *to = rand (numHosts);
    *delay = 0;
    while (*to == numHosts ||
	   (*to == me && !sim->selfSend || getNeverReceive (*to)))
      *to =  rand (numHosts);

    insist (*to >=0 && *to < numHosts);
    insist (!getNeverReceive (*to));
  }
  
  return 1;
  exception: return 0;
}



void nw_Job::send ()
{
  mt_Route route;
  int delay = 0;
  int to;
  
  insist (this);

  if (getNeverSend (me) || numHosts == 1 && (!sim->selfSend || ! getNeverReceive (me)))
    return;
  
  if (!getNextDestination (&to, &delay))
    return;
  
  if (delay > 0)
  {
    getNetwork()->setTimer (this, delay);
    return;
  }
  
  if (!acks [to])
  {
    getNetwork()->setTimer (this, 1);
    return;
  }
  
  nw_Message m;
  mt_Network*n;
  n = getNetwork ();
  insist (n);

  int len;
  
  int priority;
  if ((::rand() % 100) < sim->highPriorityPercent)
  {
    priority = 1;
    len = 32;
  }
  else
  {
    priority = 0;
    len = sim->packetLength;
  }

  int routeIndex, numRoutes, maxPriority;;

  numRoutes = n->getNumRoutes (me, to);
  insist (numRoutes > 0);
  
  //  numRoutes = 1;
  
  if (sim->shuffleRoutes)
  {
    routeIndex = mt_Component::rand (numRoutes);
  }
  else
  {
  
    routeIndex = priority;    
    maxPriority = numRoutes - 1;
    if (routeIndex > maxPriority)
      routeIndex = maxPriority;

    routeIndex = numSends % numRoutes;
  }
  
  insist (routeIndex >= 0 && routeIndex < numRoutes);
  
  int r;
  r = n->getRoute (me, to, routeIndex, &route);
  insist (r);
  insist (route.getLength () > 0);
  if (sim->longestRoute >= 0)
  {
    if (route.getLength () > sim->longestRoute)
    {
      printFormat ("route from %s to %s is longer than %d",
		   n->getGraph ()->getHost (me)->getName (),
		   n->getGraph ()->getHost (to)->getName (),
		   sim->longestRoute);
    }
  }
  
  m.to = to;
  m.from = me;
  m.number = numSends++;
  m.startTime = n->getCurrentTime ();
  m.ack = 0;
  
  insist (m.startTime >= 0);
  m.priority = priority;
  
  if ((unsigned) len < sizeof (m))
    len = sizeof (m);
  
  if (!n->sendWithPadding (this, &route, (char*) &m, sizeof (m), len))
  {
  }
  

  exception: return;
}

int nw_Job::ackQueueAppend (int to)
{
  nw_Ack*a = new nw_Ack;
  insist (a);
  a->message = new nw_Message;
  insist (a->message);
  
  a->message->to = to;
  a->message->from = me;
  a->message->number = 0;
  a->message->startTime = getNetwork ()->getCurrentTime ();
  a->message->ack = 1;
  
  return ackQueue.put (a);
  exception: return 0;
}

int nw_Job::ackQueueFlush ()
{
  nw_Ack*a;

  mt_Network*n = getNetwork ();
  mt_Route route;
  
  while ((a = (nw_Ack*) ackQueue.peek ()))
  {
    int r = n->getRoute (a->message->from, a->message->to, 0, &route);
    insist (r);

    if (!n->send (this, &route, (char*) a->message, sizeof (nw_Message)))
      return 1;
    else
      ackQueue.get ();
  }
  return 1;
  exception: return 0;
}

void nw_Job::dump (FILE*fp)
{
  insist (this);
  insist (fp);

  fprintf (fp, "; %s %d/%d\n", getNode()->getName (), numSends, numReceives);
  
  exception: return;
}

int nw_Job::willWait ()
{
  return 0;
}

mt_Job*nw_Simulation::newJob (mt_Node*node, mt_Network*network, mt_Calculator*)
{  
  insist (this);
  insist (node && node->isHost ());
  insist (network);
  
  return new nw_Job (node, network, this);
  exception: return 0;
}
void nw_Simulation::usage ()
{
  printFormat ("-simulation-args:");
  printFormat ("-self-send");
  printFormat ("-packet-length <bytes>");
  printFormat ("-mean-delay <microseconds>");
  printFormat ("-high-priority <percentage>");
  printFormat ("-histogram-file <filename>");
  printFormat ("-mesh-traffic <num levels> <level1 percent> <level2 percent> ...");
  printFormat ("-longest-route <length>");
  printFormat ("-shuffle-routes");
  printFormat ("-round-robin-routes <length>");
  printFormat ("-send-acks");
  printFormat ("-traffic-file <filename>");

}

nw_Simulation::nw_Simulation()
{
  packetLength = 1000;
  meanDelay = 0;
  selfSend = 0;
  highPriorityPercent = 0;
  *histFile = 0;
  longestRoute = -1;
  shuffleRoutes = 0;
  *trafficFile = 0;
  sendAcks = 0;
  
  numLevels = 0;
  
  for (int p = 0; p < 2; p++)
  {
    latencyTotal[p] = 0.0;
    latencyNum[p] = 0;
    for (int i = 0; i < NUM_BINS; i++)
      lbinNum[p][i] = 0;
  }
}

void nw_Simulation::dump (mt_Network*network)
{
  insist (this);
  insist (network);
  
  if (!*histFile)
    return;

  FILE *fp;
  fp = fopen (histFile,"w");
  if (!fp || ferror(fp))
    return;

  int priorities;

  if (highPriorityPercent == 0)
    priorities = 1;
  else
    priorities = 2;

  int bottom, top;
  
  bottom = NUM_BINS;
  top = 0;
  int b;
  int t;

  for (int p=0; p < priorities; p++)
  {
    b = 0;
    while (lbinNum[p][b] == 0 && b < NUM_BINS)
      b++;
    t = NUM_BINS - 1;
    while(lbinNum[p][t]==0 && t>top+1)
      t--;

    if (b<bottom)
      bottom = b;
    if (t>top)
      top = t;
  }

  fprintf(fp,"%d\n",priorities);
  for(int p=0; p<priorities; p++)
  {
    fprintf(fp,"%d\n",top-bottom);

    for(int i=bottom; i<top; i++)
      fprintf(fp,"%d %d\n", i * BIN_SIZE, lbinNum [p] [i]);
  }

  fclose(fp);
  exception: return;
}

void nw_Simulation::simpleDump (mt_Network*network, FILE *fp)
{
  insist (this);
  insist (network);
  
  fprintf(fp, "packet length: %d\n",packetLength);
  fprintf(fp, "mean delay between packets: %d microseconds\n",meanDelay);
  fprintf(fp, "hosts send to self: %s\n",selfSend?"yes":"no");
  fprintf(fp, "high priority messages: %d%%\n",highPriorityPercent);
  fprintf(fp, "shuffle-routes is %s\n", shuffleRoutes ? "true" : "false");

  
  fprintf (fp, "levels: ");  
  for (int i = 0; i < numLevels; i++)
    fprintf (fp, "%2.2f ", levelPercents [i]);
  fprintf (fp, "\n");

  fprintf (fp, "longestRoute:");
  if (longestRoute >= 0)
    fprintf (fp, "%d", longestRoute);
  fprintf (fp, "\n");
 
  
  for (int i = 0; i < 2; i++)
  {
    if (latencyNum [i] > 0)
      fprintf (fp, "average latency (priority %d): %d\n", i, (int) (latencyTotal [i]/ latencyNum [i]));
  }
  exception: return;
}

int nw_Simulation::parseArgs (mt_Args*args)
{
  int argc;  
  char**argv = args->getArgs (mt_Args::SIMULATION, &argc);

  for (int i = 0; i < argc; i++)
  {
    insist (argv[i]);
    if (!strcmp (argv[i], "-high-priority"))
    {
      if (++i == argc || !*argv[i])
      {
	printFormat ("sm_Simulator::parseArgs: missing high priority percentage");
	return 0;
      }
      if (argv[i][strlen(argv[i])-1]=='%')
	argv[i][strlen(argv[i])-1] = 0;
      highPriorityPercent = atoi (argv[i]);
      if (highPriorityPercent<0 || highPriorityPercent>100)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad percentage", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-packet-length"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing packet length");
	return 0;
      }
      packetLength = atoi (argv[i]);
      if (packetLength <= 0)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad packet length", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-longest-route"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing longest route");
	return 0;
      }
      longestRoute = atoi (argv[i]);
      if (longestRoute < 0)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad longest route", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-histogram-file"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing file name ");
	return 0;
      }
      strncpy(histFile,argv[i],255);
      histFile[255] = 0;
    }
    else if (!strcmp (argv[i], "-traffic-file"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing traffic file name ");
	return 0;
      }
      strncpy (trafficFile, argv[i], mt_File::FILENAME_LENGTH);
    }
    else if (!strcmp (argv[i], "-self-send"))
    {
      selfSend = 1;
    }
    else if (!strcmp (argv[i], "-shuffle-routes"))
    {
      shuffleRoutes = 1;
    }
    else if (!strcmp (argv[i], "-send-acks"))
    {
      sendAcks = 1;
    }
    else if (!strcmp (argv[i], "-mean-delay"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing mean delay");
	return 0;
      }
      meanDelay = atoi (argv[i]);
      if (meanDelay < 0)
      {
	printFormat ("sm_Simulator::parseArgs: %d is a bad number for mean delay", meanDelay);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-mesh-traffic"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing numLevels");
	return 0;
      }
      numLevels = atoi (argv [i]);

      if (numLevels < 0 || numLevels > MAX_LEVELS - 1)
      {
	printFormat ("sm_Simulator::parseArgs: bad numLevels");
	return 0;
      }
      if (numLevels == 0)
	continue;
      
      double total = 0;
      
      for (int j = 0; j < numLevels; j++)
      {
	if (++i == argc)
	{
	  printFormat ("sm_Simulator::parseArgs: missing level coefficient");
	  return 0;
	}

	float f;
	
	if (1 != sscanf (argv [i], "%f", &f))
	{
	  printFormat ("sm_Simulator::couldn't sscanf %s", argv [i]);
	  return 0;
	}
	levelPercents [j] = (double) f;
	
	total += levelPercents [j];
      
	if (levelPercents [j] < 0)
	{
	  printFormat ("sm_Simulator::parseArgs: bad level coefficient");
	  return 0;
	}
      }

      if (total <= 0)
      {
	printFormat ("sm_Simulator::parseArgs: bad level coefficient");
	return 0;
      }

      for (int j = 0; j < numLevels; j++)
	levelPercents [j] /= total;

      total = 0;

      for (int j = 0; j < numLevels; j++)
	total += levelPercents [j];
      
      if (total > 1.0)
	levelPercents [numLevels - 1] -= (1.0 - total);

      if (levelPercents [numLevels - 1] <= 0)
      {
	printFormat ("sm_Simulator::parseArgs: bad level coefficient");
	return 0;
      }
    }
    else
    {
      printFormat ("sm_Simulator::parseArgs: bad argument %s", argv [i]);
      return 0;
    }
  }
    
  args = args;
  return 1;

  exception:
  return 0;
}


int*nw_Job::trafficPositions;
int*nw_Job::trafficDestinations;

int nw_Job::hasTraffic ()
{
  insist (this);
  return trafficPositions != 0;
  exception: return 0;
}

int nw_Job::getTrafficDestination (int source)
{
  insist (this);
  insist (trafficPositions && trafficDestinations);
  
  insist (source >= 0 && source < getNetwork ()->getGraph ()->getNumHosts ());
  insist (trafficPositions [source + 1] >= 0);
  insist (trafficPositions [source] >= 0 && trafficPositions [source + 1] >= trafficPositions [source]);

  int numDestinations;
  numDestinations = trafficPositions [source + 1] - trafficPositions [source];
  insist (numDestinations >= 0);

  if (!numDestinations)
    return -1;
  
  int increment;
  
  if (sim->shuffleRoutes)
    increment = mt_Component::rand (numDestinations);
  else
    increment = (lastIncrement + 1) % numDestinations;
  insist (increment >= 0 && increment < numDestinations);
  lastIncrement = increment;
  
  return trafficDestinations [trafficPositions [source] + increment];
  exception: return -1;
}

int nw_Job::parseTrafficFile (char*trafficFile, mt_Graph*g)
{
  mt_Tokenizer pass1 (trafficFile, "\n\t ", 200, "#;");
  mt_Tokenizer pass2 (trafficFile, "\n\t ", 200, "#;");
    
  insistf (*trafficFile);
  insistf (g);
  
  char*h1,*h2;

  trafficPositions = (int*) calloc (sizeof (int),  (g->getNumHosts () + 1));
  insistp (trafficPositions, ("malloc failed"));

  int numDestinations;
  numDestinations = 0;
  while ((h1 = pass1.getNext ()))
  {
    if (!(h2 = pass1.getNext ()))
    {
      printFormat ("expected dest host after %s", h1);
      return 0;
    }
    
    mt_Node*n1,*n2;
    
    if (!(n1 = g->getNode (h1)))
    {
      printFormat ("%s not in graph", h1);
      return 0;
    }
    
    if (!(n2 = g->getNode (h2)))
    {
      printFormat ("%s not in graph", h2);
      return 0;
    }

    insistf (n1->getTypeIndex () >= 0 && n1->getTypeIndex () < g->getNumHosts ());
    trafficPositions [n1->getTypeIndex ()]++;
    numDestinations++;
  }
  
  if (!numDestinations)
  {
    printFormat ("no hosts found in %s", trafficFile);
    return 0;
  }

  trafficDestinations = (int*) malloc (sizeof (int) * numDestinations);
  insistp (trafficDestinations, ("malloc failed"));
  numDestinations = 0;
  
  while ((h1 = pass2.getNext ()))
  {
    if (!(h2 = pass2.getNext ()))
    {
      printFormat ("expected dest host after %s", h1);
      return 0;
    }
    
    mt_Node*n1,*n2;
    n1 = g->getNode (h1);
    n2 = g->getNode (h2);
    insistf (n1 && n2);
    insistf (n1->getTypeIndex () >= 0 && n1->getTypeIndex () < g->getNumHosts ());

    trafficDestinations [numDestinations++] = n2->getTypeIndex ();
  }
  
  insistf (numDestinations);

  int total;
  total = 0;
  for (int i = 0; i < g->getNumHosts (); i++)
  {
    total += trafficPositions [i];
    trafficPositions [i] = total - trafficPositions [i];
  }
  trafficPositions [g->getNumHosts ()] = total;

  insistf (total == numDestinations);

  return numDestinations;
  exception: return 0;
}
