/*
  mm_Switch.c
  myrinet mapper
  finucane@myri.com (David Finucane)
*/


#include <stdio.h>
#include "insist.h"
#include "mm_Switch.h"

mm_Switch::~mm_Switch ()
{
}

int mm_Switch::getNodeType ()
{
  return mt_Node::SWITCH;
}

mm_Switch::mm_Switch (char*name,  mt_Route*route, int id, int inPort, int numPorts) : mm_Node (name, "-")
{
  first = 0;
  setRoute (0, *route);
  alignment = 0;
  duplicate = 0;
  this->id = id;
  this->numPorts = numPorts;
  this->inPort = inPort;
  explored = 0;
  
  insist (numPorts >= 0 && numPorts <= NUM_PORTS);
  insist (inPort >= 0 && inPort <= NUM_PORTS);
  
  for (int i = 0; i < NUM_PORTS; i++)
  {
    nodes [i] = 0;
    opposites [i] = -1;
  }
  exception: return;
}


int mm_Switch::getId ()
{
  insist (this);
  return id;
  exception: return 0;
}

int mm_Switch::getExplored ()
{
  insist (this);
  return explored;
  exception: return 0;
}

void mm_Switch::setExplored (int explored)
{
  insist (this);
  this->explored = explored;
  exception: return;
}

int mm_Switch::getNumPorts ()
{
  insist (this);
  return numPorts;
  exception: return 0;
}

int mm_Switch::getInPort ()
{
  insist (this);
  return inPort;
  exception: return 0;
}

void mm_Switch::disconnect (int p)
{
  insist (this);
  insist (p >= MIN_PORT && p < MAX_PORT);
  
  nodes [p - MIN_PORT] = mm_Node::disconnected;
  
  exception: return;
}

int mm_Switch::getNumHosts ()
{
  insist (this);

  int numHosts;
  numHosts = 0;
  
  for (int i = 0; i < NUM_PORTS; i++)
    if (nodes [i] && nodes [i] != mm_Node::disconnected && nodes [i]->isHost ())
      numHosts++;
  
  return numHosts;
  exception: return 0;
}

int mm_Switch::getNumConnections ()
{
  insist (this);

  int numConnections;
  numConnections = 0;
  
  for (int i = 0; i < NUM_PORTS; i++)
    if (nodes [i] && nodes [i] != mm_Node::disconnected)
      numConnections++;
  
  return numConnections;
  exception: return 0;
}

int mm_Switch::getNumDefined ()
{
  insist (this);

  int numDefined;
  numDefined = 0;
  
  for (int i = 0; i < NUM_PORTS; i++)
    if (nodes [i])
      numDefined++;
  
  return numDefined;
  exception: return 0;
}

void mm_Switch::finish ()
{
  first = -1;

  for (int i = 0; i < NUM_PORTS; i++)
  {
    if (!nodes [i])
      nodes [i] = mm_Node::disconnected;
    else if (first == -1 && nodes [i] != mm_Node::disconnected)
      first = i;
  }
}

void mm_Switch::mConnect (int p, int o, mm_Node*n)
{
  insist (this);
  insist (n);
  insist (p >= MIN_PORT && p < MAX_PORT);
  insist (o >= MIN_PORT && o < MAX_PORT);
  
  nodes [p - MIN_PORT] = (mm_Node*)n;
  opposites [p - MIN_PORT] = o;
  
  exception: return;
}

mm_Node*mm_Switch::mGetNode (int p)
{
  insist (this);
  insist (p >= MIN_PORT && p < MAX_PORT);
  insist (p - MIN_PORT >= 0 && p - MIN_PORT < NUM_PORTS);
  mm_Node*n;
  n = nodes [p - MIN_PORT];
  return n;
  
  exception: return 0;
}

int mm_Switch::mGetOpposite (int p)
{
  insist (this);
  insist (p >= MIN_PORT && p < MAX_PORT);
  insist (p - MIN_PORT >= 0 && p - MIN_PORT < NUM_PORTS);
  return opposites [p - MIN_PORT];
  exception: return -1;
}

int mm_Switch::getPort (mm_Node*n)
{
  insist (this);
  insist (n);

  for (int i = 0; i < NUM_PORTS; i++)
    if (nodes [i] == n)
      return i + MIN_PORT;
  
  exception: return NO_PORT;
}

int mm_Switch::getMaxNodes ()
{
  return MAX_PORT;
}

int mm_Switch::getFirst ()
{
  return first;
}

void mm_Switch::setFirst (int first)
{
  insist (first >= 0 && first < NUM_PORTS);
  
  this->first = first;
  exception:;
}

void mm_Switch::connectMissingHost (int p, mt_Node*h)
{
  insist (p >= MIN_PORT && p < MAX_PORT);
  insist (h->isHost ());
  insist (p + first >= 0 && p + first < NUM_PORTS);  

  nodes [p + first] = (mm_Node*) h;
  opposites [p + first] = 0;
  h->connect (0, p + MIN_PORT, this);

  if (p < 0)
    first += p;

  exception:;
}

void mm_Switch::connect (int p, int o, mt_Node*n)
{
  insist (this);
  insist (p >= 0 && p < MAX_PORT);
  insist (first + p >= 0 && first + p < NUM_PORTS);

  mm_Node*m;
  m = (mm_Node*)n;
  
  nodes [first + p] = m;
  opposites [first + p] = o + m->getFirst ();
  
  exception: return;
}

mt_Node*mm_Switch::getNode (int p)
{
  insist (this);
  if (p < 0 || p >= MAX_PORT)
    return 0;
  
  if (first + p < 0 || first + p >= NUM_PORTS)
    return 0;

  mm_Node*n;
  n = nodes [first + p];
  return n && n->isConnected () ? n : 0;
  
  exception: return 0;
}

int mm_Switch::getOpposite (int p)
{
  insist (this);
  
  if (p < 0 || p >= MAX_PORT)
    return 0;
  
  if (first + p < 0 || first + p >= NUM_PORTS)
    return - 1;
  
  mm_Node*n;
  n = nodes [first + p];

  if (!n)
    return -1;
  
  return n->shift (opposites [first + p]);
  exception: return -1;
}

int mm_Switch::inheritHosts (mm_Switch*s, int alignment)
{
  insist (this);
  insist (s);

  for (int i = 0; i < NUM_PORTS; i++)
  {
    mm_Node*h = s->nodes [i];
    if (h && h->isHost () && !getHost (h->getAddress ()))
    {
      int o = h->mGetOpposite (0);
      
      insist (o == i + MIN_PORT);
      
      int p = o + alignment;
      insist (p >= MIN_PORT && p < MAX_PORT);
      insist (!nodes [p - MIN_PORT] || !nodes [p - MIN_PORT]->isConnected ());
      mConnect (p, 0, h);
      h->mConnect (0, p, this);
      s->disconnect (o);
    }
  }
  exception: return 0;
}  


int mm_Switch::shift (int n)
{
  insist (this);
  
  return n - (MIN_PORT + first);
  exception: return -1;
}

void mm_Switch::setDuplicate (mm_Switch*duplicate, int alignment)
{
  insist (this);
  insist (duplicate && duplicate->isSwitch ());
  
  this->duplicate = duplicate;
  this->alignment = alignment;
  duplicate->duplicate = this;
  duplicate->alignment = -alignment;
  
  exception: return;
}

int mm_Switch::sever (int alignment, mm_Switch*original, mm_Switch*ghost)
{
  insist (this);
  insist (ghost && original);
  insist (original != this && ghost != this && ghost != original);
  
  for (int i = mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT; i++)
  {  
    mm_Node*n = mGetNode (i);
    
    if (n == ghost)
    {
      insist (n->isSwitch ());
      n->disconnect (mGetOpposite (i));
      disconnect (i);      
      insist (!original->mGetNode (i - alignment));
      original->mConnect (i - alignment, i, this);
      this->mConnect (i, i - alignment, original);
    }
  }
  return 1;
  exception: return 0;
}


mm_Switch*mm_Switch::merge (mt_Graph*graph, mt_Queue*queue)
{
  insist (this);
  insist (graph && queue);
  
  insist (duplicate && duplicate->isSwitch ());
  mm_Switch*d;
  d = duplicate;
  duplicate = 0;
  
  for (int i = mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT; i++)
  {
    int a = i + alignment;
    if (a < mm_Switch::MIN_PORT || a >= mm_Switch::MAX_PORT)
    {
      insist (!mGetNode (i) || !mGetNode (i)->isConnected ());
      continue;
    }
    
    mm_Node*n1 = mGetNode (i);
    mm_Node*n2 = d->mGetNode (a);

    if (n1 == n2)
      continue;
    
    if (n1 && n1->isConnected ())
    {
      if (!n2 || !n2->isConnected ())
	continue;

      insist (n2->isSwitch () && n1->isSwitch ());
      insist (n1->mGetNode (mGetOpposite (i)) == this);
      insist (n2->mGetNode (d->mGetOpposite (a)) == d);
      
      n2->disconnect (d->mGetOpposite (a));
      d->disconnect (a);

      mm_Switch*s = (mm_Switch*) n2;
      
      if (!s->duplicate && !s->getNumConnections ())
      {
	if (queue->isQueued (s))
	  queue->remove (s);
	graph->remove (s);
      }
      
    }
    else if (n2 && n2->isConnected ())
    {
      insist (n1 != n2);
      int o = d->mGetOpposite (a);
      mConnect (i, o, n2);
      n2->mConnect (o, i, this);
    }
  }
  return d;
  exception: return 0;
}
