/*
  Histogram.java
  Merlin
  (c) 1998 Myricom, Inc.
  dmazzoni@myri.com (Dominic Mazzoni)
*/

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;

class HistogramCanvas extends Canvas
{
  int priorities=0;
  int numBins=0;
  int bin[][];
  int max;
  int block=1;
  boolean good;
  Label leftLabel;
  Label rightLabel;

  public HistogramCanvas(Label left, Label right)
  {
    leftLabel = left;
    rightLabel = right;
    good = false;
  }

  public void read(String fn)
  {
    FileInputStream s = null;
    try
    {
      good = false;
      s = new FileInputStream(fn);
      FileReader reader = new FileReader (s.getFD());
      StreamTokenizer tokens = new StreamTokenizer (reader);
      tokens.parseNumbers ();
      if (tokens.nextToken() != tokens.TT_NUMBER)
	return;
      priorities = (int)tokens.nval;
      if (tokens.nextToken() != tokens.TT_NUMBER)
	return;
      numBins = (int)tokens.nval;
      tokens.pushBack();
      bin = new int[priorities][numBins];

      for(int p=0; p<priorities; p++)
      {
	if (tokens.nextToken() != tokens.TT_NUMBER)
	  return;
	if (numBins != (int)tokens.nval)
	  return;
	for(int i=0; i<numBins; i++)
	{
	  if (tokens.nextToken() != tokens.TT_NUMBER)
	    return;
	  int ns = (int)tokens.nval;
	  if (i==0)
	    leftLabel.setText(""+ns);
	  if (i==numBins-1)
	    rightLabel.setText(""+ns);
	  if (tokens.nextToken() != tokens.TT_NUMBER)
	    return;
	  int value = (int)tokens.nval;
	  bin[p][i] = value;
	}
      }
      FixLabels();
      block = 0;
      good = true;
    }
    catch (Exception e)
    {
      System.out.println (e);
    }
    repaint();
  }

  private void FixLabels()
  {
    Font f = leftLabel.getFont();
    FontMetrics fm = leftLabel.getFontMetrics(f);
    int wid = fm.stringWidth(leftLabel.getText());
    Dimension size = leftLabel.getSize();
    leftLabel.setSize(wid+4, size.height);

    f = rightLabel.getFont();
    fm = rightLabel.getFontMetrics(f);
    wid = fm.stringWidth(rightLabel.getText());
    size = rightLabel.getSize();
    rightLabel.setSize(wid+4, size.height);
  }

  private void CalcMax()
  {
    max = 1;
    int numColumns = numBins / block;

    for(int p=0; p<priorities; p++)
      for(int i=0; i<numColumns; i++)
      {
	int value=0;
	for(int b=0; b<block; b++)
	  if (i*block+b < numBins)
	    value += bin[p][i*block+b];
	if (value > max)
	  max = value;
      }
  }

  private void CalcBlock()
  {
    Dimension size = getSize();

    block = 1;

    while(block < numBins && (size.width * block / numBins / priorities) < 4)
    {
      block++;
    }
  }

  private void setColorIndex(Graphics g, int index)
  {
    switch(index)
    {
      case 0:
	g.setColor(Color.red);
	break;
      case 1:
	g.setColor(Color.blue);
	break;
      case 2:
	g.setColor(Color.green);
	break;
      default:
	g.setColor(Color.black);
	break;
    }
  }

  public void more()
  {
    if (block<numBins)
    {
      block++;
      CalcMax();
      repaint();
    }
  }
  public void fewer()
  {
    if (block>1)
    {
      block--;
      CalcMax();
      repaint();
    }
  }

  public void paint(Graphics g)
  {
    Dimension size = getSize();

    g.setColor(Color.white);
    g.fillRect(0,0,size.width,size.height);

    if (!good || numBins==0 || priorities==0)
    {
      g.setColor(Color.black);
      g.drawString("No Histogram",2,20);
      return;
    }

    if (block==0)
    {
      CalcBlock();
      CalcMax();
    }

    int numColumns = numBins / block;

    int wid = size.width / numColumns / priorities;
    if (wid==0)
      wid = 1;

    for(int p=0; p<priorities; p++)
      for(int i=0; i<numColumns; i++)
      {
	int x = (i*priorities+p) * size.width / numColumns / priorities;
	int value=0;
	for(int b=0; b<block; b++)
	  if (i*block+b < numBins)
	    value += bin[p][i*block+b];
	int y = value * size.height / max;
	
	setColorIndex(g,p);
	g.fillRect(x,size.height-y,wid,y);
      }

    int h = 20;
    if (priorities>1)
      for(int p=0; p<priorities; p++)
      {	
	setColorIndex(g,p);
	String s = "Priority "+p;
	int w = g.getFontMetrics().stringWidth(s);
	g.drawString(s,size.width - 10 - w, h);
	h += 20;
      }
  }
}

class Histogram extends Panel implements ActionListener
{
  private Button closeButton;
  private Button moreButton;
  private Button fewerButton;
  private Label label;
  private HistogramCanvas canvas;

  public Histogram ()
  {
    setLayout (new BorderLayout());

    Panel bars = new Panel();
    bars.setLayout(new GridLayout(2,1));

    Panel labels = new Panel();
    labels.setLayout (new BorderLayout());
      
    Label leftLabel;
    Label rightLabel;
    labels.add("West",leftLabel = new Label(""));
    labels.add("Center",new Label("Latency (ns)",Label.CENTER));
    labels.add("East",rightLabel = new Label(""));

    bars.add (labels);

    Panel bottom = new Panel();
    bottom.setLayout (new BorderLayout());

    Panel twobuttons = new Panel();
    twobuttons.setLayout (new GridLayout(1,2));
    twobuttons.add (fewerButton = new Button ("<-"));
    twobuttons.add (moreButton = new Button ("->"));
    
    bottom.add (twobuttons);

    bars.add (bottom);

    this.add("South", bars);

    canvas = new HistogramCanvas(leftLabel, rightLabel);
    this.add ("Center", canvas);

    moreButton.addActionListener (this);
    fewerButton.addActionListener (this);
  }

  public void read(String fn)
  {
    canvas.read(fn);
  }

  public void actionPerformed (ActionEvent e)
  {
    Object object = e.getSource ();

    if (object == moreButton)
    {
      canvas.more();
    }
    else if (object == fewerButton)
    {
      canvas.fewer();
    }
  }
}
