/*
  Anatomy.java
  Merlyn
  (c) 1998 Myricom, Inc.
  finucane@myri.com (David Finucane)
*/

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.awt.image.*;
import GraphicsUtil;

class Anatomy
{
  public static final int MAX_NODES = 10000;
  public static final int MAX_EDGES = Switch.MAX_EDGES * MAX_NODES;
  private int numNodes;
  private int numEdges;
  private Node nodes[];
  private Edge edges[];
  private AllTest allTest;
  private boolean positioned = true;
  private Vector groups;
  private Node editLabelNode = null;
  private EditableLabel editLabel = null;

  private class Group
  {
    private int name;
    private int count = 0;
    private boolean collapsed = false;
    
    public Group (int name)
    {
      this.name = name;
    }
    public void setName (int name)
    {
      this.name = name;
    }
    public boolean empty ()
    {
      return count <= 0;
    }
    public void add ()
    {
      count++;
    }
    public void remove ()
    {
      count--;
    }
    public String toString ()
    {
      return "Group " + name;
    }
    public String forFile ()
    {
      return "" + name;
    }
  };
  
  public Anatomy ()
  {
    edges = new Edge [MAX_EDGES];
    nodes = new Node [MAX_NODES];
    groups = new Vector ();

    setEditLabel(null);
  }
  public Anatomy (String filename)
  {
    edges = new Edge [MAX_EDGES];
    nodes = new Node [MAX_NODES];
    groups = new Vector ();

    fromFile (filename);
    setEditLabel(null);
  }
  public Anatomy (InputStream stream)
  {
    edges = new Edge [MAX_EDGES];
    nodes = new Node [MAX_NODES];
    groups = new Vector ();

    try
      {
	fromReader (new InputStreamReader (stream));
      }
    catch (Exception x)
      {
      }
    setEditLabel(null);
  }
  private void ungroupNode (Node node)
  {
    groupNode (node, null);
  }
  public Node [] getNodes ()
  {
    Node ns [] = new Node [numNodes];
    
    for (int i = 0; i < numNodes; i++)
      ns [i] = nodes [i];
    return ns;
  }  
  public int getNumNodes ()
  {
    return numNodes;
  }
  private void setGroupNames ()
  {
    for (int i = 0; i < groups.size(); i++)
      ((Group)groups.elementAt (i)).setName (i);
  }
  private void groupNode (Node node, Group group)
  {
    Group g = (Group)node.getGroup ();
    if (g != null)
    {
      g.remove ();
      if (g.empty ())
      {
	groups.removeElement (g);
	setGroupNames ();
      }
    }
    node.setGroup (group);
    if (group != null)
      group.add ();
  }
  public static Node [] subtract (Node a[], Node b[])
  {
    if (a == null || b == null)
      return null;
    Vector s = new Vector();
    for (int i = 0; i < a.length; i++)
    {
      int j;
      for (j = 0; j < b.length; j++)
	if (a[i] == b[j])
	  break;
      if (j == b.length)
	s.addElement (a[i]);
    }
    Node ns [] = new Node [s.size()];
    
    for (int i = 0; i < s.size(); i++)
      ns [i] = (Node)s.elementAt (i);
    return ns;
  }

  private void groupNodes (Node nodes [], Group group)
  {
     if (nodes == null)
      return;

     for (int i = 0; i < nodes.length; i++)
       groupNode (nodes[i], group);
  }
  public void groupNodes (Node nodes [])
  {
    if (nodes == null)
      return;

    ungroupNodes (nodes);
    groupNodes (nodes, newGroup());
  }
  private Group newGroup ()
  {
    Group group = new Group (groups.size());
    groups.addElement (group);
    return group;
  }
  public void ungroupNodes (Node nodes [])
  {
    groupNodes (nodes, null);
  }
  public Node [] getGroupedNodes (Object group)
  {
    Vector nv = new Vector();
    for (int i = 0; i < numNodes; i++)
      if (group != null && nodes[i].getGroup() == group)
	nv.addElement (nodes[i]);

    Node ans[] = new Node [nv.size()];
    for (int i = 0; i < ans.length; i++)
      ans[i] = (Node)nv.elementAt (i);
    return ans;
  }
  public void autoGroupNodes (Node a[], Node b[])
  {
    Node c[];
    while ((c = matchForGroup (a, b)) != null)
    {
      groupNodes (c, newGroup ());
      b = subtract (b, c);
    }
  }
  private Node[] matchForGroup (Node a[], Node b[])
  {
    if (a == null || b == null)
      return null;
    
    for (int i = 0; i < a.length; i++)
    {
      for (int j = 0; j < b.length; j++)
      {
	clearMatches (a);
	clearMatches (b);

	if (matchNode (a[i], b[j]))
	{
	  Vector matches = new Vector ();
	  int cx = b[j].getX() - b[j].getMatch().getX();
	  int cy = b[j].getY() - b[j].getMatch().getY();
	  
	  for (int k = 0; k < b.length; k++)
	  {
	    Node n = b[k].getMatch ();
	    if (n != null)
	    {
	      //System.out.println (n.getName() + " matched with " + b[k].getName ());
	      b[k].moveGroup (n.getX() + cx, n.getY() + cy);
	      matches.addElement (b[k]);
	    }
	  }
	  Node rn [] = new Node [matches.size()];
	  for (int k = 0; k < matches.size(); k++)
	  {
	    rn[k] = (Node)matches.elementAt (k);
	    //System.out.println (rn[k].getName ());
	  }
	  return rn;
	}
      }
    }
    return null;
  }

  public static boolean matchHosts (Anatomy a, Anatomy b)
  {
    /*match hosts. a < b*/

    for (int i = 0; i < a.numNodes; i++)
    {
      if (a.nodes[i] instanceof Host)
      {
	Node n = b.getNode(a.nodes[i].getName());
	if (n != null && n instanceof Host)
	  a.nodes[i].setMatch (n);
	else
	  return false;
      }
    }
    return true;
  }

  private static boolean matchNodes (Node a[], Node b[])
  {
    if (a.length > b.length)
      return false;

    for (int i = 0; i < a.length; i++)
    {
      int j;
      for (j = 0; j < b.length; j++)
      {
	if (matchNode (a[i], b[j]))
	{
	  //System.out.println ("matchNodes:trying to match "+ a[i].getName() + " " + b[j].getName());
	  break;
	}
	//System.out.println ("matchNodes:failed to match " + a[i].getName() + " " + b[j].getName());
      }
      
      if (j == b.length)
      {
	//System.out.println ("matchNodes:failed to match " + a[i].getName());
	return false;
      }
    }
    //System.out.println ("matchNodes:matched a bunch of nodes");
    return true;
  }
  private static boolean matchNode (Node a, Node b)
  {
    if (a.getClass () != b.getClass ())
      return false;

    if (a.getHostCount () > b.getHostCount () ||
	a.getSwitchCount () > b.getSwitchCount ())
      return false;

    Node am = a.getMatch ();
    Node bm = b.getMatch ();

    if ((am == null) && (bm == null))
    {
      /*make a guess match, which will hold if the children match*/
      a.setMatch (b);
      //System.out.println ("matchNode:guessing that " + a.getName() + " matches " + b.getName());
      
      if (!matchNodes (a.getNodes (), b.getNodes ()))
      {
	a.setMatch (null);
	b.setMatch (null);
	//System.out.println ("matchNode: " + a.getName() + " doesn't match " + b.getName());
	return false;
      }

      //System.out.println ("matchNode: " + a.getName() + " matches " + b.getName());
      return true;
    }
    return am == b && bm == a;
  }

  public void bringSelectedToFront()
  {
    Node temp[] = new Node[numNodes];
    int i,j;

    j = 0;
    for(i=0; i<numNodes; i++)
      if (!nodes[i].getSelected())
	temp[j++] = nodes[i];
    for(i=0; i<numNodes; i++)
      if (nodes[i].getSelected())
	temp[j++] = nodes[i];
    for(i=0; i<numNodes; i++)
      nodes[i]=temp[i];

    setEditLabel(null);
  }
  public void sendSelectedToBack()
  {
    Node temp[] = new Node[numNodes];
    int i,j;

    j = 0;
    for(i=0; i<numNodes; i++)
      if (nodes[i].getSelected())
	temp[j++] = nodes[i];
    for(i=0; i<numNodes; i++)
      if (!nodes[i].getSelected())
	temp[j++] = nodes[i];
    for(i=0; i<numNodes; i++)
      nodes[i]=temp[i];

    setEditLabel(null);
  }
  public void setSelectedSwitchSize(int newSize)
  {
    Node sw[] = getSelectedSwitches();

    if (sw!=null)
    {
      for(int i=0; i<sw.length; i++)
	((Switch)sw[i]).setNumEdges(newSize);
    }
  }
  public void flipNodes (Node nodes [])
  {
     if (nodes == null)
      return;

     for (int i = 0; i < nodes.length; i++)
       {
	 if (nodes[i] instanceof Switch)
	   ((Switch)nodes[i]).flip();
       }
  }
  public void eraseRoutes ()
  {
    for (int i = 0; i < numEdges; i++)
      edges[i].setRouteCount (0);
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i] instanceof Host)
      {
	((Host)nodes[i]).setRouteOn (false);
	((Host)nodes[i]).setRedSquare (false);
      }
    }
  }
  private int getUniqueNumber (Node nodes[])
  {
    if (nodes == null)
      return 0;
    int largest = -1;
    for (int i = 0; i < nodes.length; i++)
    {
      int n = nodes[i].getNameNumber();
      if (n > largest)
	largest = n;
    }
    return largest + 1;
  }
  String getUniqueSwitchName ()
  {
    return "s" + getUniqueNumber (getSwitches());
  }
  String getUniqueHostName ()
  {
    return "h" + getUniqueNumber (getHosts());
  }
  int getSwitchCount ()
  {
    int c = 0;
    
    for (int i = 0; i < numNodes; i++)
      if (nodes[i] instanceof Switch)
	c++;
    return c;
  }
  int getCloudCount ()
  {
    int c = 0;
    
    for (int i = 0; i < numNodes; i++)
      if (nodes[i] instanceof Cloud)
	c++;
    return c;
  }
  int getHostCount ()
  {
    int c = 0;
    
    for (int i = 0; i < numNodes; i++)
      if (nodes[i] instanceof Host)
	c++;
    return c;
  }
  Node [] getSwitches ()
  {
    int numSwitches = getSwitchCount ();

    if (numSwitches == 0)
      return null;
    
    Node switches[] = new Node [numSwitches];
    
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i] instanceof Switch)
      {
	switches [numSwitches - 1] = nodes[i];
	numSwitches--;
      }
    }
    return switches;
  }

  Node [] getSwitchesAndClouds ()
  {
    int numSwitches = getSwitchCount () + getCloudCount ();

    if (numSwitches == 0)
      return null;
    
    Node switches[] = new Node [numSwitches];
    
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i] instanceof Switch || nodes[i] instanceof Cloud)
      {
	switches [numSwitches - 1] = nodes[i];
	numSwitches--;
      }
    }
    return switches;
  }
  
  Node [] getHosts ()
  {
    int numHosts = getHostCount ();

    if (numHosts == 0)
      return null;
    
    Node hosts[] = new Node [numHosts];
    
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i] instanceof Host)
      {
	hosts [numHosts - 1] = nodes[i];
	numHosts--;
      }
    }
    return hosts;
  }
  public Node duplicateSelectedNodes ()
  {
    Node ns[] = getSelectedNodes();
    if (ns.length == 0)
      return null;
    
    Node newNodes[] = new Node [ns.length];
    Hashtable hash = new Hashtable (ns.length);

    int delta = 40;
    Node selectedNode = null;
    
    for (int i = 0; i < ns.length; i++)
    {
      Node n;
      
      if (ns[i] instanceof Host)
	n = new Host (ns[i].getX()+delta, ns[i].getY()+delta, getUniqueHostName());
      else
	n = new Switch (ns[i].getNumEdges(), ns[i].getX()+delta, ns[i].getY()+delta, getUniqueSwitchName());
      newNodes[i] = n;
      hash.put (ns[i], n);
      add (n);
      selectedNode = n;
      n.setSelected (true);
      ns[i].setSelected (false);
    }
    for (int i = 0; i < ns.length; i++)
    {
      Node ao = ns[i];
      Node an = newNodes[i];
      
      for (int j = 0; j < an.getNumEdges(); j++)
      {
	Node bo, bn;
	Edge eo;
	if (an.getEdge (j) != null ||
	    (eo = ao.getEdge (j)) == null ||
	    (bo = eo.getOther (ao)) == null ||
	    (bn = (Node)hash.get (bo)) == null)
	  continue;

	Edge e = new Edge (an, bn, j, bo.getPort (eo));
	an.connect (e, j);
	bn.connect (e, bo.getPort (eo));
	add (e);
      }
    }
    setEditLabel(null);

    return selectedNode;
  }

  public Node getOneSelectedNode ()
  {
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i].getSelected())
	return nodes[i];
    }
    return null;
  }
  

  public Node [] getSelectedSwitches ()
  {
    Node ns[] = new Node [numNodes];

    int n = 0;
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i] instanceof Switch && nodes[i].getSelected())
	ns[n++] = nodes[i];
    }
    Node ans[] = new Node [n];
    for (int i = 0; i < n; i++)
      ans[i] = ns[i];
    return ans;
  }


  public Node [] getSelectedSwitchesAndClouds ()
  {
    Node ns[] = new Node [numNodes];

    int n = 0;
    for (int i = 0; i < numNodes; i++)
    {
      if ((nodes[i] instanceof Switch ||
	   nodes[i] instanceof Switch ) && nodes[i].getSelected())
	ns[n++] = nodes[i];
    }
    Node ans[] = new Node [n];
    for (int i = 0; i < n; i++)
      ans[i] = ns[i];
    return ans;
  }
  
  public Node [] getGroupedNodes ()
  {
    Node ns[] = new Node [numNodes];

    int n = 0;
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i].getGroup() != null)
	ns[n++] = nodes[i];
    }
    Node ans[] = new Node [n];
    for (int i = 0; i < n; i++)
      ans[i] = ns[i];
    return ans;
  }
  public Node [] getSelectedHosts ()
  {
    Node ns[] = new Node [numNodes];

    int n = 0;
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i] instanceof Host && nodes[i].getSelected())
	ns[n++] = nodes[i];
    }
    Node ans[] = new Node [n];
    for (int i = 0; i < n; i++)
      ans[i] = ns[i];
    return ans;
  }
  Node [] getSelectedNodes (Class c)
  {
    Node ns[] = new Node [numNodes];

    int n = 0;
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i].getClass() == c && nodes[i].getSelected())
	ns[n++] = nodes[i];
    }
    Node ans[] = new Node [n];
    for (int i = 0; i < n; i++)
      ans[i] = ns[i];
    return ans;
  }
  
  public Node [] getSelectedNodes ()
  {
    Node ns[] = new Node [numNodes];

    int n = 0;
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i].getSelected())
	ns[n++] = nodes[i];
    }
    Node ans[] = new Node [n];
    for (int i = 0; i < n; i++)
      ans[i] = ns[i];
    return ans;
  }
  public void moveSelected (int dx, int dy)
  {
    for (int i = 0; i < numNodes; i++)
      if (nodes[i].getSelected ())
	nodes[i].move (nodes[i].getX() + dx, nodes[i].getY() + dy);
  }
  
  private void removeEdges (Node node)
  {
    for (int i = 0; i < numEdges; i++)
    {
      if (edges[i].connectedTo (node))
      {
	Edge e = edges[i];
	node.disconnect (e);
	e.getOther (node).disconnect (e);
	 
	for (int j = i; j < numEdges - 1; j++)
	  edges[j] = edges[j + 1];
	
	numEdges--;
	i--;
      }
    }
  }
  public Node getSelectedNode ()
  {
    for (int i = 0; i < numNodes; i++)
      if (nodes[i].getSelected())
	return nodes[i];
    return null;
  }
  public int getNumSelectedNodes ()
  {
    int n = 0;
    for (int i = 0; i < numNodes; i++)
      if (nodes[i].getSelected())
	n++;
    return n;
  }
  
  public void removeSelectedNodes ()
  {
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i].getSelected())
      {
	removeEdges (nodes[i]);
	for (int j = i; j < numNodes - 1; j++)
	  nodes[j] = nodes[j + 1];
	numNodes --;
	i--;
      }
    }

    setEditLabel(null);
  }
  public void remove (Node node)
  {
    if (node instanceof Switch)
      ungroupNode (node);
    
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i] == node)
      {
	removeEdges (nodes[i]);
	for (int j = i; j < numNodes - 1; j++)
	  nodes[j] = nodes[j + 1];
	numNodes --;
	i--;
	return;
      }
    }
    setEditLabel(null);
  }
  public void remove (Edge e)
  {
    for (int i = 0; i < numEdges; i++)
    {
      if (edges[i] == e)
      {
	for (int j = i; j < numEdges - 1; j++)
	  edges[j] = edges[j + 1];
	numEdges--;
	Node a = e.getNode ();
	Node b = e.getOther (a);
	a.disconnect (e);
	b.disconnect (e);
	
	return;
      }
    }
    setEditLabel(null);
  }
  public void add (Node node)
  {
    nodes[numNodes++] = node;
    setEditLabel(null);
  }
  public void add (Edge edge)
  {
    edges [numEdges++] = edge;
    setEditLabel(null);
  }
  public boolean isSwitchSelected()
  {
    Node nodes[] = getSelectedSwitches ();

    return (nodes.length>0);
  }
  public String getSelectedSwitchesString()
  {
    String s="";

    Node[] sw = getSelectedSwitches ();
    s += sw.length;    
    for(int i=0; i<sw.length; i++)
      s += " " + sw[i].getName();

    return s;    
  }
  
  public String sssToString ()
  {

    Node nodes[] = getGroupedNodes();
    Node hosts[] = getSelectedHosts();
    if (hosts.length == 0)
      hosts = getHosts();

    boolean grouped = true;
    if (nodes.length == 0)
    {
      nodes = getSelectedSwitches ();
      grouped = false;
    }
    
    if (nodes.length == 0 || hosts.length == 0)
      return null;
   
    if (!computeRoutes (hosts[0]))
      return null;
    
    String s = "\"" + hosts[0].getName()  + "\"\n";

    for (int i = 0; i < nodes.length; i++)
    {
      s += "" + (grouped ? ((Group)nodes[i].getGroup()).forFile() : "0") + " " +
	"\"" + nodes[i].getName() + "\" " + nodes[i].getRoute().toString() + "\n";
    }
    
     return s;
  }
  public String toString ()
  {
    String s = "";
    for (int i = 0; i < numNodes; i++)
      s += nodes[i].toString() + "\n";
    return s;
  }
  public Node getNode (String name)
  {
    for (int i = 0; i < numNodes; i++)
      if (name.equals (nodes[i].getName ()))
        return nodes[i];
    return null;
  }
  public boolean editLabelVisible()
  {
    return (editLabelNode != null);
  }
  public void setEditLabel (Node n)
  {
    if (editLabelNode != null && n != editLabelNode)
      {
	editLabelNode.setText(editLabel.getText());
      }

    editLabelNode = n;

    if (editLabelNode != null)
    {
      editLabel = new EditableLabel(editLabelNode.getLabelRect(),
				    editLabelNode.getText());      
    }
  }
  public boolean handleDrag(int x, int y)
  {
    if (editLabelNode != null)
      return editLabel.mouseDrag(x, y);
    else
      return false;
  }
  public boolean handleClick(int x, int y)
  {
    if (editLabelNode != null)
      return editLabel.mouseClick(x, y);
    else
      return false;
  }
  public boolean handleKeyPressed(KeyEvent e)
  {
    if (editLabelNode != null)
      {
	editLabel.handleKeyPressed(e);
	return true;
      }
    else
      return false;
  }
  public boolean handleKeyTyped(KeyEvent e)
  {
    if (editLabelNode != null)
      {
	char c = e.getKeyChar ();

	if (c == '\n' || c == '\r')
	  {
	    setEditLabel(null);
	  }
	else
	  editLabel.handleKeyTyped(e);
	return true;
      }
    else
      return false;
  }
  public void paint (Graphics g, boolean hideHosts)
  {
    for (int i = 0 ; i < numNodes; i++)
      if (nodes[i] instanceof Switch || nodes [i] instanceof Cloud)
        nodes[i].paint (g);
    for (int i = 0; i < numEdges; i++)
    {
      Node n1 = edges [i].getNode ();
      Node n2 = edges [i].getOther (n1);
      if (!hideHosts || (n1 instanceof Switch && n2 instanceof Switch))
	edges[i].paint (g);
    }
    if (!hideHosts)
    {
      for (int i = 0; i < numNodes; i++)
	if (nodes[i] instanceof Host)
	  nodes[i].paint (g);
    }
    if (Node.showLabels())
      {
	for (int i = 0 ; i < numNodes; i++)
	  if (nodes[i] != editLabelNode)
	    nodes[i].paintLabel (g);
	if (editLabelNode != null)
	{
	  Rectangle r;
	  r = editLabelNode.getLabelRect();

	  if (editLabel != null)
	    editLabel.paint(g);
	}
      }
  }
  public Node[] showNodes (String names[])
  {
    Node ns[] = new Node [names.length];
    Node pair [] = new Node [2];
    
    int nn = 0;
    for (int i = 0; i < names.length; i++)
    {
      ns [nn] = getNode (names[i]);
      if (ns[nn] != null)
      {
	int j;
	for (j = 0; j < nn; j++)
	  if (ns[j] == ns [nn])
	    break;
	if (j == nn)
	  nn++;
      }
    }

    if (nn == 0)
      return null;
    
    unselect ();
    
    for (int i = 0; i < nn; i++)
    {
      swapNodes (ns[i], numNodes - 1 - i);
      ns[i].setSelected (true);
    }
    pair [0] = ns [0];
    pair [1] = nn >= 2 ? ns [1] : null;
    
    return pair;
  }
  private void swapNodes (Node node, int pos)
  {
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes [i] == node)
      {
	Node n = nodes [pos];
	nodes [pos] = node;
	nodes [i] = n;
	return;
      }
    }
  }
  public void straighten ()
  {
    for (int i = 0; i < numEdges; i++)
      edges[i].straighten ();
  }
  public void selectAll()
  {
    for (int i = 0; i < numNodes; i++)
      nodes[i].setSelected (true);
  }
  public void selectHosts()
  {
    for (int i = 0; i < numNodes; i++)
      nodes[i].setSelected (nodes[i] instanceof Host);
  }
  public void selectSwitches()
  {
    for (int i = 0; i < numNodes; i++)
      nodes[i].setSelected (nodes[i] instanceof Switch);
  }
  public Node select (Rectangle r)
  {
    Node n = null;
    for (int i = 0; i < numNodes; i++)
    {
      if (nodes[i].intersects (r))
      {
	n = nodes [i];
	nodes[i].setSelected (true);
      }
    }
    return n;
  }
  public void unselect ()
  {
    for (int i = 0; i < numNodes; i++)
      nodes[i].setSelected (false);
  }
  public void clearMatches ()
  {
    clearMatches (nodes, numNodes);
  }
  
  public static void clearMatches (Node nodes[])
  {
    clearMatches (nodes, nodes.length);
  }
  public static void clearMatches (Node nodes[], int numNodes)
  {
    for (int i = 0; i < numNodes; i++)
    {
      nodes[i].setMatch (null);
      nodes[i].setInPort (-1);
    }
  }
  public void markDead ()
  {
    for (int i = 0; i < numNodes; i++)
      nodes [i].dead = nodes[i].getMatch () == null;
  }
  public void makeMatch(Node a, Node b)
  {
    /* a < b */

    b.setMatch(a);
    a.move(b.getX(),b.getY());
    a.setText(b.getText());
  }
  
  public static boolean match (Anatomy a, Anatomy b)
  {
    a.clearMatches();
    b.clearMatches();

    //System.out.println ("TRYING TO MATCH");
    
    if (!matchHosts (a, b))
      return false;

    Node first = null;
    for (int i = 0; i < a.numNodes; i++)
    {
      if (a.nodes[i].getMatch () != null && a.nodes[i].getUnmatchedNodes().length > 0)
      {
	first = a.nodes[i];
	break;
      }
    }
    return  (first == null) || matchNodes (first.getNodes (), first.getMatch ().getNodes ());
  }
  public int moveMatches ()
  {
    int nm = 0;
    
    for (int i = 0; i < numNodes; i++)
    {
      Node n = nodes[i].getMatch ();
      if (n != null)
      {
	nodes[i].move (n.getX(), n.getY());
	nm++;
      }
    }
    return nm;
  }
  
  public Node getNodeLabel (int x, int y)
  {
    for (int i = numNodes - 1; i >= 0; i--)
      if (nodes[i] instanceof Host && nodes[i].labelContains (x, y))
        return nodes[i];
    
    for (int i = numNodes - 1; i >= 0; i--)
      if (nodes[i] instanceof Switch && nodes[i].labelContains (x, y))
        return nodes[i];
    
    return null;
  }
  public Node getNode (int x, int y)
  {
    for (int i = numNodes - 1; i >= 0; i--)
      if (nodes[i] instanceof Host && nodes[i].contains (x, y))
        return nodes[i];
    
    for (int i = numNodes - 1; i >= 0; i--)
      if (nodes[i].contains (x, y))
        return nodes[i];
    
    return null;
  }
  public Edge getEdge (int x, int y)
  {
    Edge e = null;
    double low = .6;
    
    for (int i = 0; i < numEdges; i++)
    {
      double d = edges[i].getDistance (x,y);
      if (d < low)
      {
	low = d;
	e = edges[i];
      }
    }
    return e;
  }
  private void circle (Node nodes[], double x, double y, double r, double start, double stop)
  {
    if (nodes == null) return;
    
    double a = (stop - start)/ nodes.length;

    for (int i = 0; i < nodes.length; i++)
    {
      if (nodes[i] == null)
	continue;
      double dx = r * Math.cos (i * a + start);
      double dy = r * Math.sin (i * a + start);
      nodes[i].move ((int)(dx + x), (int)(dy + y));
    }
    positioned = true;
  }
  private void circle (Node switches[])
  {
    if (switches == null)
      return;
    
    for (int i = 0; i < switches.length; i++)
      ((Switch)switches[i]).cleanUpHosts();
  }
  public void cleanUpHosts ()
  {
    circle (getSwitches());
  }

  public void levels(Dimension d)
  {
    Node switches[] = getSwitches();
    Node hosts[] = getHosts();

    int numSwitches = switches.length;
    int numHosts = hosts.length;

    for(int i=0; i<numHosts; i++)
      hosts[i].setRow(0);

    for(int i=0; i<numSwitches; i++)
      switches[i].setRow(-1);
      
    Node queue[] = new Node[numSwitches];
    int head = 0;
    int tail = 0;

    int maxRow = 0;

    for(int i=0; i<numHosts; i++)
    {
      Node nbors[] = hosts[i].getSwitches();
      for(int j=0; j<nbors.length; j++)
	if (nbors[j] != null)
	{
	  if (nbors[j].getRow() == -1)
	  {
	    nbors[j].setRow(1);
	    maxRow = 1;
	    queue[tail++] = nbors[j];
	  }
	}
    }

    while(head<tail)
    {
      Node pop = queue[head++];

      Node nbors[] = pop.getSwitches();
      
      for(int j=0; j<nbors.length; j++)
	if (nbors[j] != null)
	{
	  if (nbors[j].getRow() == -1)
	  {
	    nbors[j].setRow(pop.getRow() + 1);
	    if (nbors[j].getRow() > maxRow)
	      maxRow = nbors[j].getRow();
	    queue[tail++] = nbors[j];
	  }
	}
    }

    int ystep = (d.height-20) / (maxRow+1);
    if (ystep < 30)
      ystep = 30;

    int ypos = 10;

    for(int row = maxRow; row>=1; row--)
    {
      int rowSize = 0;
      for(int i=0; i<numSwitches; i++)
	if (switches[i].getRow() == row)
	  rowSize++;
      int xstep = (d.width-20) / (rowSize);
      if (xstep < 10)
	xstep = 10;
      
      int xpos = xstep/2;

      for(int i=0; i<numSwitches; i++)
	if (switches[i].getRow() == row)
	{
	  switches[i].move(xpos, ypos);

	  if (row==1 && switches[i].getHostCount()>0)
	  {
	    int hxpos = xpos - xstep/4;

	    int hxstep = xstep / switches[i].getHostCount();
	    Node nbors[] = switches[i].getHosts();
	    if (nbors != null)
	    {
	      for(int j=0; j<nbors.length; j++)
		if (nbors[j] != null)
		{
		  nbors[j].move(hxpos, ypos + ystep);
		  hxpos += hxstep;
		}
	    }
	  }
	  xpos += xstep;
	  if (maxRow == 1)
	    ypos += 10;
	}

      ypos += ystep;
    }
    positioned = true;
  }

  public void circle (Dimension d)
  {

    Node selected[] = getSelectedSwitchesAndClouds();
    if (selected.length > 0)
    {
      circle (selected);
    }
    else
    {
      Node switches[] = getSwitchesAndClouds();
      double r = Math.min (d.height, d.width) / 2 - 20; 
      
      int numSwitches = switches.length;
      int numHosts = getHostCount();
      int maxDegree = 0;

      for(int i=0; i<numSwitches; i++)
      {
	int itsDegree = switches[i].getHostCount();
	if (itsDegree > maxDegree)
	  maxDegree = itsDegree;
      }

      for(int i=0; i<numSwitches; i++)
      {

	double newX = 0.70 * r * Math.cos (i * 2 * Math.PI / numSwitches);
	double newY = 0.70 * r * Math.sin (i * 2 * Math.PI / numSwitches);

	switches[i].move((int)(r + newX), (int)(r + newY));
	if (switches [i] instanceof Switch)
	{ 
	  if (newY > 0)
	    ((Switch)switches[i]).flip(1);
	  else
	    ((Switch)switches[i]).flip(0);
	}
 
	Node hosts[] = switches[i].getHosts();

	int j=-1;
	if (hosts != null)
	{ 
	  for (int k=0; k < hosts.length; k++)
	  {
	    if (hosts[k] != null)
	    {
	      int pos;

	      j++;
	    
	      if (newY > 0)
		pos = (i * maxDegree + maxDegree - 1 - j) - maxDegree/2;
	      else
		pos = (i * maxDegree + j) - maxDegree/2;
	      
	      double hX = 0.95 * r * Math.cos 
		((i * maxDegree + j) * 2 * Math.PI / numSwitches / maxDegree);
	      double hY = 0.95 * r * Math.sin 
		((i * maxDegree + j) * 2 * Math.PI / numSwitches / maxDegree);
	      
	      hosts[k].move((int)(r + hX), (int)(r + hY));
	    }
	  }
	}
	positioned = true;
      }
    }
  }

  private void zeroRows ()
  {
    for (int i = 0; i < numNodes; i++)
      nodes[i].setRow (0);
  }

  private boolean computeRoutes (Node root)
  {
    Node nodes[] = new Node [numNodes];
    int head = 0;
    int tail = 0;
    int nFound = 0;
    
    for (int i = 0; i < numNodes; i++)
      this.nodes[i].setRoute (null);

    root.setIn (0);
    root.setRoute (new Route ());
    nFound ++;
    
    if (root instanceof Host)
    {
      Edge e = root.getEdge (0);
      Node t = e.getOther (root);
      t.setIn (t.getPort (e));
      
      root = t;
      root.setRoute (new Route ());
      nFound++;
    }
    
    nodes[tail++] = root;
    
    while (head != tail)
    {
      Node n = nodes [head++];
      int in = n.getIn();

      Edge edges[] = n.getEdges();

      for (int i = 0; i < edges.length; i++)
      {
	if (edges [i] == null)
	  continue;

	Node o = edges[i].getOther(n);
	if (o.getRoute() != null)
	  continue;
	
	o.setRoute (new Route (n.getRoute(), i - in));
	nFound++;
	if (o instanceof Host)
	  continue;
	o.setIn (o.getPort (edges[i]));
	nodes[tail++] = o;
      }
    }
    return nFound == numNodes;
  }
  
  private Node [] treeExplore (Node startNodes[], Dimension d)
  {
    zeroRows();

    int head = 0;
    int tail = 0;
    
    Node switches[] = new Node [getSwitchCount() + getCloudCount ()];

    for (int i = 0; i < startNodes.length; i++)
    {
      switches [tail++] = startNodes[i];
      startNodes[i].setRow (1);
    }
    
    while (head != tail)
    {
      Node n = switches [head++];
      Node ss[] = n.getSwitchesAndClouds ();
      int sslen = ss.length;

      for (int i = 0; i < sslen; i++)
      {
	Node s = ss[sslen-1-i];
	if (s.getRow () > 0)
	  continue;
	switches [tail++] = s;
	s.setRow (n.getRow() + 1);
      }
    }
    return switches;
  }
  public void tree (Node startNodes[], Dimension d)
  {
    if (startNodes == null || startNodes.length == 0)
      return;
    
    Node switches[] = treeExplore (startNodes, d);
    if (switches.length == 0)
      return;

    int widths[] = new int [switches.length];
    for (int i = 0; i < switches.length; i++)
      widths [i] = 0;

    int rowCount = 0;

    for (int i = 0; i < switches.length; i++)
    {
      int row = switches [i].getRow();
      if (row > rowCount)
	rowCount = row;
      widths [row - 1]++;
    }
    double r = Math.min (d.height, d.width) / 8;
    int s = 0;
    for (int i = 0; i < rowCount; i++)
    {
      for (int j = 0; j < widths[i]; j++, s++)
      {
	Node n = switches [s];
	n.move ( (j + 1) * d.width/ (widths [i] + 1), i * d.height / rowCount);
	Node hosts[] = n.getHosts ();
	int hlen = hosts.length;
	Node hostsbw[] = new Node[hlen];
	for(int h=0; h<hlen; h++)
	  hostsbw[h] = hosts[hlen-1-h];
	circle (hostsbw, n.getX(), n.getY(), r, Math.PI/4, 2 * Math.PI/4);
      }
    }
  }

  public boolean fromSimulation (String filename) throws Exception
  { 
    FileInputStream s = new FileInputStream(filename);
    FileReader reader = new FileReader (s.getFD());
    
    StreamTokenizer tokens = new StreamTokenizer (reader);
    tokens.parseNumbers ();
    tokens.quoteChar ('"');
    tokens.whitespaceChars ('(','(');
    tokens.whitespaceChars (')',')');
    tokens.whitespaceChars (',',',');
    tokens.whitespaceChars ('/','/');
    tokens.commentChar('#');
    tokens.commentChar(';');

    tokens.nextToken ();
    if (tokens.ttype != tokens.TT_NUMBER)
    {
      System.out.println ("line " + tokens.lineno() + ":error reading simulation file");
      return false;
    }
    int numSent = (int) tokens.nval;

    tokens.nextToken ();
    if (tokens.ttype == tokens.TT_EOF || tokens.ttype != tokens.TT_NUMBER)
    {
      System.out.println ("line " + tokens.lineno() + ":error reading simulation file.");
      return false;
    }
    int numExpected = (int) tokens.nval;

    while (tokens.nextToken () != tokens.TT_EOF)
    {
      int nodeType;
      
      while(tokens.ttype == tokens.TT_EOL)
	tokens.nextToken();

      if (tokens.ttype != tokens.TT_WORD)
	break;
      
      nodeType = parseNodeType (tokens.sval);
      
      if (nodeType == Node.NONE)
	break;
      
      do {tokens.nextToken();}
      while(tokens.ttype == tokens.TT_EOL);
      if (tokens.ttype != tokens.TT_WORD && tokens.ttype != '"')
	break;
      String name = new String (tokens.sval);

      Node n = getNode (name);
      if (n == null)
        break;

      do {tokens.nextToken();}
      while(tokens.ttype == tokens.TT_EOL);
      if (tokens.ttype != tokens.TT_NUMBER)
	break;
      int np = (int) tokens.nval;
      
      int i;
      for (i = 0; i < np; i++)
      {
	do {tokens.nextToken();}
	while(tokens.ttype == tokens.TT_EOL);

        if (tokens.ttype != tokens.TT_NUMBER)
          break;
        int p = (int) tokens.nval;
        
	do {tokens.nextToken();}
	while(tokens.ttype == tokens.TT_EOL);

        if (tokens.ttype != tokens.TT_NUMBER)
          break;
        double d = tokens.nval;
	//System.out.println ("p " + p + " d" + d);

	do {tokens.nextToken();}
	while(tokens.ttype == tokens.TT_EOL);
	
        n.setBandwidth (p, d);
      }
      if (i != np)
        break;
    }
    if (tokens.ttype != tokens.TT_EOF)
    {
      String ending;
      if (tokens.ttype == tokens.TT_WORD)
	ending = "(word) " + tokens.sval;
      else if (tokens.ttype == tokens.TT_NUMBER)
	ending = "(numerical) " + tokens.nval;
      else
	ending = "type "+tokens.ttype;
      System.out.println ("line " + tokens.lineno() + ":error reading simulation at token " + ending);
    }

    return numSent != numExpected;
  }
  public boolean readAllTest (String filename) throws Exception
  {
    allTest = new AllTest (filename);
    
    clearErrors();
    int hostCount = allTest.getHostCount();

    if (hostCount == 0)
      return false;
      
    for (int i = 0; i < hostCount; i++)
    {
      Host from = (Host)getNode (allTest.getName (i));
	
      for (int j = 0; j < hostCount; j++)
      {
	Host to = (Host)getNode (allTest.getName (j));	
	if (!addError (from, to, allTest.getErrorCount (i, j)))
	  return false;
      }
    }
    setHighestErrorFigure ();
    return true;
  }
  
  public void setHighestErrorFigure ()
  {    
    double d = 0;
    for (int i = 0; i < numEdges; i++)
    {
      double t = edges[i].getErrorFigure ();
      if (t > d)
	d = t;
    }
    Edge.setHighestErrorFigure (d);
  }

  private void clearErrors ()
  {
    for (int i = 0; i < numEdges; i++)
      edges[i].clearError ();
  }
  

  private boolean addError (Host from, Host to, int error)
  {
    Route r = from.getRoute (to);
    if (r == null)
    {
      System.out.println ("missing route.");
      return false;
    }
    int hops[] = r.getHops();
    int nHops = r.getLength();
    
    Node n = from;
    Edge e = from.getEdge (0);
    e.addError (n, error);
    n = e.getOther (n);
    
    for (int i = 0; i < nHops; i++)
    {
      int p = n.getPort (e);
      e = n.getEdge (p + hops[i]);
      e.addError (n, error);
      n = e.getOther(n);
    }
    return true;
  }
  public boolean readRoutes (String filename) throws Exception
  {
   
    Hashtable hostPot = new Hashtable(numNodes*13-1, (float)0.1);
    for (int i = 0; i < numNodes; i++)
      hostPot.put((Object)nodes[i].getName(), (Object)nodes[i]);    
    
    FileReader reader = new FileReader (filename);
    StreamTokenizer tokens = new StreamTokenizer (reader);

    tokens.parseNumbers ();
    tokens.quoteChar ('"');
    tokens.whitespaceChars ('(','(');
    tokens.whitespaceChars (')',')');
    tokens.whitespaceChars (',',',');
    tokens.wordChars ('.','.');
    tokens.eolIsSignificant(true);
    tokens.commentChar('#');
    tokens.commentChar(';');
    tokens.lowerCaseMode(false);

    tokens.nextToken();
    if (tokens.ttype != tokens.TT_NUMBER)
    {
      System.out.println("readRoutes: missing host count");
      return false;
    }
    int numHosts = (int)tokens.nval;

    tokens.nextToken();
    if (tokens.ttype != tokens.TT_NUMBER)
    {
      System.out.println("readRoutes: missing max route count");
      return false;
    }
    int maxRoutes = (int)tokens.nval;

    tokens.nextToken();
    if (tokens.ttype != tokens.TT_EOL)
    {
      System.out.println("readRoutes: missing header line");
      return false;
    }

    tokens.nextToken();
    while (tokens.ttype != tokens.TT_EOF)
    { 
      if (tokens.ttype != tokens.TT_WORD && tokens.ttype != '"')
	break;
      String name = new String (tokens.sval);

      tokens.nextToken();
      if (tokens.ttype != tokens.TT_NUMBER)
	break;

      int nRoutes = (int) tokens.nval;

      tokens.nextToken();
      if (tokens.ttype != tokens.TT_NUMBER)
	break;

      int nHops = (int) tokens.nval;

      Host from = (Host)hostPot.get((Object)name);
      if (from == null)
	break;
      from.clearRoutes();
      
      tokens.nextToken();
      if (tokens.ttype != tokens.TT_EOL)
	break;

      tokens.nextToken();

      Host lastTo = null;
      int index=0;

      int i, c;
      int hops[] = new int[100];
      for (i = 0; i < nRoutes; i++)
      {
	if (tokens.ttype != tokens.TT_WORD && tokens.ttype != '"')
	  break;

	Host to = (Host)hostPot.get((Object)tokens.sval);
	if (to == null)
	  break;

	c = 0;
	for (tokens.nextToken();
	     tokens.ttype != tokens.TT_EOL && tokens.ttype != tokens.TT_EOF;
	     tokens.nextToken(), c++)
	{
	  if (tokens.ttype != tokens.TT_NUMBER)
	    break;
	  hops[c] = (int) tokens.nval;
	}
	if (tokens.ttype != tokens.TT_EOL && tokens.ttype != tokens.TT_EOF)
	  break;
	tokens.nextToken();

	if (to != lastTo)
	  index = 0;
	else
	  index++;

	Route r = new Route (to, c, hops);
	from.setRoute (to, index, r);

      }
      if (i != nRoutes)
	break;
    }	  
    //    s.close();

    if (tokens.ttype != tokens.TT_EOF)
    {
      String ending = (tokens.ttype == tokens.TT_WORD ? "(text) " + tokens.sval : "(number) " + tokens.nval);
      System.out.println ("line " + tokens.lineno() + ":error reading routes at token" + ending);
      return false;
    }
    return true;
  }

  /* stolen from Sun's GraphLayout demo*/
  public synchronized void relax (Dimension d, Node anchor, boolean relaxEdges)
  {
    if (numNodes == 0)
      return;

    if (anchor == null)
      anchor = nodes [0];
    
    int hostLength = (int) Math.min (d.width, d.height) / 5;
    int switchLength = (int) Math.min (d.width, d.height) / 3;

    if (relaxEdges)
    {
      
      for (int i = 0 ; i < numEdges ; i++)
      {
	Edge e = edges [i];
	Node a = e.getNode ();
	Node b = e.getOther (a);
	
	double vx = a.getX() - b.getX();
	double vy = a.getY() - b.getY();
	double len = Math.sqrt(vx * vx + vy * vy);
	double myLen = a instanceof Host || b instanceof Host ? hostLength : switchLength;
	
	double f = (myLen - len) / (len * 3) ;
	double dx = f * vx;
	double dy = f * vy;
	
	a.dx += dx;
	a.dy += dy;
	b.dx += -dx;
	b.dy += -dy;
      }
    }
    
    for (int i = 0 ; i < numNodes ; i++)
    {
      Node n1 = nodes [i];
      double dx = 0;
      double dy = 0;

      for (int j = 0 ; j < numNodes ; j++)
      {
	if (i == j)
	  continue;
	Node n2 = nodes [j];
	double vx = n1.getX() - n2.getX();
      	double vy = n1.getY() - n2.getY();
       	double len = vx * vx + vy * vy;
	if (len == 0)
        {
	  dx += Math.random();
	  dy += Math.random();
	}
        else if (len < 100*100)
        {
	  dx += vx / len;
	  dy += vy / len;
	}
      }
      double dlen = dx * dx + dy * dy;
      if (dlen > 0)
      {
	dlen = Math.sqrt(dlen) / 2;
	n1.dx += dx / dlen;
	n1.dy += dy / dlen;
      }
    }

    for (int i = 0 ; i < numNodes ; i++)
    {
      Node n = nodes [i];

      if (n == anchor)
        continue;

      
      int x = n.getX() + Math.max(-5, Math.min(5, n.dx));
      int y = n.getY() + Math.max(-5, Math.min(5, n.dy));

      if (x < 0)
	x = 0;
      else if (x > d.width)
	x = d.width;
      if (y  < 0)
	y = 0;
      else if (y > d.height)
	y = d.height;

      n.dx /= 2;
      n.dy /= 2;
      n.move (x, y);
    }
  }
  public boolean getPositioned ()
  {
    return positioned;
  }
  public void fromString (String string)
  {
    try
      {
	fromReader (new StringReader (string));
      }
    catch (Exception e)
      {
	System.out.println (e);
      }
  }
  
  private void fromFile (String filename)
  {
    FileInputStream s = null;
    try
      {
	s = new FileInputStream(filename);
	FileReader reader = new FileReader (s.getFD());
	fromReader (reader);
	s.close ();
      }
    catch (Exception e)
      {
	System.out.println (e);
      }
  }

  private void fromReader (Reader reader) throws Exception
  {
    StreamTokenizer tokens = new StreamTokenizer (reader);

    tokens.parseNumbers ();
    tokens.quoteChar ('"');
    tokens.wordChars ('.','.');
    tokens.commentChar('#');
    tokens.commentChar(';');

    parseMapFile (tokens);
  }
  
  private int parsePortCount (int nodeType, String type)
  {
    int numPorts = 0;
      
    if (nodeType == Node.HOST)
      numPorts = 1;
    else
    {
      try
      {
	numPorts = Integer.parseInt(type);
      }
      catch (Exception ex)
      {
	if (nodeType != Node.SWITCH)
	  return 0;
	
	numPorts = Switch.MAX_EDGES;
      }
      if (nodeType == Node.SWITCH &&
	  (numPorts < 1 || numPorts > Switch.MAX_EDGES))
	numPorts = Switch.MAX_EDGES;
      }
    return numPorts;
  }
  
  private int parseNodeType (String s)
  {
    if (s.equals ("h"))
      return Node.HOST;
    else if (s.equals ("s") || s.equals ("s8"))
      return Node.SWITCH;
    else if (s.equals ("cloud"))
      return Node.CLOUD;
    else return Node.NONE;
  }
  
  public void parseMapFile(StreamTokenizer tokens) throws Exception
  {
    
    if (tokens.nextToken () != tokens.TT_EOF &&
	tokens.ttype == tokens.TT_NUMBER &&
	tokens.nextToken () != tokens.TT_EOF &&
	tokens.ttype == tokens.TT_NUMBER)
    {
      tokens.nextToken ();
    }
    
    for (;tokens.ttype != tokens.TT_EOF; tokens.nextToken ())
    {
      String type;
      String name;
      String label;
      int numPorts;
      Node n;

      if (tokens.ttype != tokens.TT_WORD)
	break;

      int nodeType = parseNodeType (tokens.sval);
      
      tokens.nextToken();
      if (tokens.ttype == tokens.TT_NUMBER)
	type = Integer.toString((int)tokens.nval);
      else if (tokens.ttype != tokens.TT_EOF)
	type = tokens.sval;
      else break;

      numPorts = parsePortCount (nodeType, type);
      if (numPorts == 0)
	break;
      
      tokens.nextToken();
      if (tokens.ttype == tokens.TT_NUMBER)
	name = Integer.toString((int)tokens.nval);
      else if (tokens.ttype != tokens.TT_EOF)
	name = tokens.sval;
      else break;

      if (tokens.nextToken () != tokens.TT_NUMBER)
	break;
      int numEdges = (int)tokens.nval;

      Node node = getNode (name);

      if (node == null)
      {
	switch (nodeType)
	{  
	  case Node.HOST:
	    node = new Host (0, 0, name);
	    break;
	  case Node.SWITCH:
	    node = new Switch (numPorts, 0, 0, name);
	    break;
	  case Node.CLOUD:
	    node = new Cloud (numPorts, 0, 0, name);
	    break;
	}
	add (node);
      }
      
      Point points [] = new Point [Edge.NUM_POINTS];
      for (int i = 0; i < Edge.NUM_POINTS; i++)
        points[i] = new Point (0, 0);
      
      //System.out.println ("parsing edges on " + name);
      
      int i;
      for (i = 0; i < numEdges; i++)
      {
	//System.out.println ("parsing edge " + i);
	
	if (tokens.nextToken () != tokens.TT_NUMBER)
	  break;
	int nodePort = (int)tokens.nval;
	
	//System.out.println ("nodePort is " + nodePort);
	
	if (tokens.nextToken () != tokens.TT_WORD)
	  break;
	
	nodeType = parseNodeType (tokens.sval);
	if (nodeType == Node.NONE)
	  break;
      
	//System.out.println ("nodeType is " + nodeType);
	
	tokens.nextToken();
	if (tokens.ttype == tokens.TT_NUMBER)
	  type = Integer.toString((int)tokens.nval);
	else if (tokens.ttype != tokens.TT_EOF)
	  type = tokens.sval;
	else break;
	
	//System.out.println ("type string is " + type);
	
	numPorts = parsePortCount (nodeType, type);
	if (numPorts == 0)
	  break;
      
	//System.out.println ("numPorts is " + numPorts);
	
	tokens.nextToken();
	if (tokens.ttype == tokens.TT_WORD ||
	    tokens.ttype == '"')
	  name = tokens.sval;
	else if (tokens.ttype == tokens.TT_NUMBER)
	  name = Integer.toString((int)tokens.nval);
	else break;

	if (tokens.nextToken() != tokens.TT_NUMBER)
	  break;
	int oppositePort = (int)tokens.nval;

	//System.out.println ("opposite port is " + oppositePort);
	
	if (nodeType == Node.HOST && oppositePort != 0 ||
	    nodeType != Node.HOST && oppositePort >= numPorts)
	  break;

	//System.out.println ("opposite port check ok");
	
	Node opposite = getNode (name);
	if (opposite == null)
	{
	  switch (nodeType)
	  {  
	    case Node.HOST:
	      opposite = new Host (0, 0, name);
	      break;
	    case Node.SWITCH:
	      opposite = new Switch (numPorts, 0, 0, name);
	      break;
	    case Node.CLOUD:
	      opposite = new Cloud (numPorts, 0, 0, name);
	      break;
	  }
	  //System.out.println ("adding node named " + name);
	  
	  add (opposite);
	}
	else
	{
	  Edge oppositeEdge;

	  oppositeEdge = opposite.getEdge(oppositePort);

	  if (oppositeEdge != null)
	    if (!(oppositeEdge.connectedTo(opposite) &&
		  oppositeEdge.connectedTo(node)))
	      {
		System.out.println("Inconsistency with node "+node.getName()+
				   " port "+nodePort+":");
		System.out.println("Can't connect to node "+name+
				   " port "+oppositePort+".");
		
		break;
	      }
	}

        points[0] = node.getPoint (nodePort);
        points[Edge.NUM_POINTS - 1] = opposite.getPoint (oppositePort);

	Edge e = node.getEdge (nodePort);
	if (e == null)
	{
	  Edge oppositeEdge = opposite.getEdge(oppositePort);
	  if (oppositeEdge != null)
	  {
	    System.out.println("Inconsistency with node "+node.getName()+
			       " port "+nodePort+":");
	    System.out.println("Can't connect an edge to "+name+
			       " port "+oppositePort);
	    System.out.println("because this is already connected to "+
			       oppositeEdge.getOther(opposite).getName()+
	          " port "+
		  oppositeEdge.getOther(opposite).getPort(oppositeEdge));
	  }

	  e = new Edge (node, opposite, points);
          node.connect (e, nodePort);
	  opposite.connect (e, oppositePort);
	  add (e);
	}
	else
	{
	  Node other = e.getOther(node);

	  if (other!=opposite)
	    {
	      System.out.println("Inconsistency with node "+node.getName()+
				 " port "+nodePort+":");
	      System.out.println("Already connected to "+other.getName());
	      
	      break;
	    }

	  if (other.getPort(e) != oppositePort)
	  {
	    System.out.println("Inconsistency with node "+node.getName()+
			       " port "+nodePort+":");
	    System.out.println("Already connected to "+other.getName()+
			       " on port "+other.getPort(e)+
			       ", not port "+oppositePort+".");
	    
	  }
	}
      }
      if (i != numEdges)
	break;
     
      
      String extraName;
      String extraValue;

      while (tokens.nextToken () != tokens.TT_EOF)
	{
	  if (tokens.ttype == tokens.TT_NUMBER)
	    extraName = Integer.toString((int)tokens.nval);
	  else if (tokens.ttype != tokens.TT_EOF)
	    extraName = tokens.sval;
	  else break;

	  if (extraName==null || parseNodeType (extraName) != Node.NONE)
	    {
	      tokens.pushBack();
	      break;
	    }

	  tokens.ordinaryChars('0','9');
	  tokens.wordChars('0','9');

	  tokens.nextToken ();
	  
	  if (tokens.ttype == tokens.TT_NUMBER)
	    extraValue = Integer.toString((int)tokens.nval);
	  else if (tokens.ttype != tokens.TT_EOF)
	    extraValue = tokens.sval;
	  else break;

	  tokens.parseNumbers();

	  node.addExtra(extraName,extraValue);

	}
      node.finish();

      if (tokens.ttype != tokens.TT_WORD)
	break;
    }

    if (tokens.ttype != tokens.TT_EOF)
    {
      String ending = (tokens.ttype == tokens.TT_WORD ?
		       "(text) " + tokens.sval :
		       "(number) " + tokens.nval);
      System.out.println ("Line " + tokens.lineno() +
			  ": Error reading graph at or just before token " +
			  ending);
    }

  }

  public void parseOldMapFile(StreamTokenizer tokens) throws Exception
  {
    boolean pointed = false;

    tokens.whitespaceChars (',',',');
    tokens.whitespaceChars ('(','(');
    tokens.whitespaceChars (')',')');

    while (tokens.nextToken () != tokens.TT_EOF)
    {
      boolean isHost;
      boolean dead = false;
      int description = 0;
      int switchSize = 0;

      if (tokens.ttype != tokens.TT_WORD)
	break;
      if (tokens.sval.equals ("h"))
	isHost = true;
      else if (tokens.sval.equals ("s") ||
               tokens.sval.equals ("s8"))
      {
        switchSize = 8;
        isHost = false;
      }
      else if (tokens.sval.equals ("s16"))
      {
        switchSize = 16;
        isHost = false;
      }
      else break;

      tokens.nextToken();
      if (tokens.ttype == tokens.TT_WORD && tokens.sval.equals ("d"))
      {
 	dead = true;
	tokens.nextToken();
      }

      if (isHost)
      {
	if (tokens.ttype != tokens.TT_NUMBER)
	  break;
	description = (int)tokens.nval;
	tokens.nextToken ();
      }

      if (tokens.ttype != tokens.TT_WORD && tokens.ttype != '"')
	break;
      String name = new String (tokens.sval);

      tokens.nextToken ();
      if (tokens.ttype != tokens.TT_WORD && tokens.ttype != '"')
	break;
      String label = tokens.sval;

      tokens.nextToken ();
      if (tokens.ttype != tokens.TT_NUMBER)
	break;
      int x = (int)tokens.nval;

      if (x != 0)
	positioned = true;

      tokens.nextToken ();
      if (tokens.ttype != tokens.TT_NUMBER)
	break;
      int y = (int)tokens.nval;

      if (y != 0)
	positioned = true;

      tokens.nextToken ();
      if (tokens.ttype != tokens.TT_NUMBER)
	break;
      int ec = (int)tokens.nval;

      Node node = getNode (name);

      if (node == null)
      {
	if (isHost)
	  node = new Host (x, y, name);
	else
	  node = new Switch (switchSize, x, y, name);
	add (node);
      }
      else
	node.move (x, y);
      
      node.setText (label);
      
      Point points [] = new Point [Edge.NUM_POINTS];
      for (int i = 0; i < Edge.NUM_POINTS; i++)
        points[i] = new Point (0, 0);
      
      int i;
      for (i = 0; i < ec; i++)
      {
	boolean dead_edge = false;
	int load = 0;
        
	tokens.nextToken ();
	if (tokens.ttype != tokens.TT_NUMBER)
	  break;
	int np = (int)tokens.nval;

	tokens.nextToken();
	if (tokens.ttype == tokens.TT_WORD && tokens.sval.equals ("d"))
	{
	  dead_edge = true;
	  tokens.nextToken();
	}
	if (tokens.ttype == tokens.TT_WORD && tokens.sval.equals ("[x"))
	{
	  tokens.nextToken ();
	  while (tokens.ttype != tokens.TT_WORD || !tokens.sval.equals ("x]"))
	    tokens.nextToken ();
	  tokens.nextToken ();
	}
        if (tokens.ttype == tokens.TT_WORD && tokens.sval.equals ("[p"))
	{
          pointed = true;
	  tokens.nextToken ();
          int p = 1;
	  while (tokens.ttype != tokens.TT_WORD || !tokens.sval.equals ("p]"))
	  {
            if (tokens.ttype != tokens.TT_NUMBER)
              break;
            points[p].x = (int)tokens.nval;
            tokens.nextToken ();
            
            if (tokens.ttype != tokens.TT_NUMBER)
              break;
            points[p].y = (int)tokens.nval;
            tokens.nextToken ();
            p++;
          }
          if (p != Edge.NUM_POINTS - 1)
            break;
	  tokens.nextToken ();
	}
        
	if (tokens.ttype == tokens.TT_NUMBER)
	{
	  load = (int)tokens.nval;
	  tokens.nextToken();
	}
	if (tokens.ttype != tokens.TT_WORD)
	  break;
	if (tokens.sval.equals ("h"))
	  isHost = true;
        else if (tokens.sval.equals ("s") ||
                 tokens.sval.equals ("s8"))
        {
          switchSize = 8;
          isHost = false;
        }
        else if (tokens.sval.equals ("s16"))
        {
          switchSize = 16;
          isHost = false;
        }
	else break;

	tokens.nextToken();
	if (tokens.ttype != tokens.TT_WORD  && tokens.ttype != '"')
	  break;
	String rn = tokens.sval;

	tokens.nextToken ();
	if (tokens.ttype != tokens.TT_NUMBER)
	  break;
	int rp = (int)tokens.nval;

	if (isHost && rp != 0 || !isHost && rp >= switchSize)
	  break;

	Node r = getNode (rn);
	if (r == null)
	{
	  if (isHost)
	    r = new Host (x, y, rn);
	  else
	    r = new Switch (switchSize, x, y, rn);
	  add (r);
	}
	else
	{
	  if (!(r.getName().equals(rn)))
	    break;
	  
	  Edge rpe = r.getEdge(rp);

	  if (rpe != null)
	    if (!(rpe.connectedTo(r) && rpe.connectedTo(node)))
	      {
		System.out.println("Inconsistency with node "+node.getName()+
				   " port "+np+":");
		System.out.println("Can't connect to node "+rn+
				   " port "+rp+".");
		
		break;
	      }
	}

	/*
	System.out.println(node.getName()+" "+np+" -> "+
			   r.getName()+" "+rp);
	*/

        points[0] = node.getPoint (np);
        points[Edge.NUM_POINTS - 1] = r.getPoint (rp);

	Edge e = node.getEdge (np);
	if (e == null)
	{
	  Edge oe = r.getEdge(rp);
	  if (oe != null)
	  {
	    System.out.println("Inconsistency with node "+node.getName()+
			       " port "+np+":");
	    System.out.println("Can't connect an edge to "+rn+
			       " port "+rp);
	    System.out.println("because this is already connected to "+
			       oe.getOther(r).getName()+
			       " port "+oe.getOther(r).getPort(oe));
	  }

	  e = new Edge (node, r, points);
          node.connect (e, np);
	  r.connect (e, rp);
	  add (e);
	}
	else
	{
	  Node other = e.getOther(node);

	  if (other!=r)
	    {
	      System.out.println("Inconsistency with node "+node.getName()+
				 " port "+np+":");
	      System.out.println("Already connected to "+other.getName());
	      
	      break;
	    }

	  if (other.getPort(e) != rp)
	  {
	    System.out.println("Inconsistency with node "+node.getName()+
			       " port "+np+":");
	    System.out.println("Already connected to "+other.getName()+
			       " on port "+other.getPort(e)+
			       ", not port "+rp+".");
	    
	  }
	}
      }
      if (i != ec)
	break;
     
      
      String extraName;
      String extraValue;

      while (tokens.nextToken () != tokens.TT_EOF)
	{
	  if (tokens.ttype == tokens.TT_WORD)
	    {
	      if (tokens.sval.equals ("h") ||
		  tokens.sval.equals ("s") ||
		  tokens.sval.equals ("s8") ||
		  tokens.sval.equals ("s16"))
		{
		  tokens.pushBack();
		  break;
		}
	      extraName = tokens.sval;
	    }
	  else if (tokens.ttype == tokens.TT_NUMBER)
	    {
	      extraName = Double.toString(tokens.nval);
	    }
	  else
	    break;

	  tokens.nextToken ();
	  if (tokens.ttype == tokens.TT_WORD)
	    extraValue = tokens.sval;
	  else if (tokens.ttype == tokens.TT_NUMBER)
	    extraValue = Double.toString(tokens.nval);
	  else
	    break;

	  /*
	  System.out.println("Extra data: "+extraName+", "+extraValue);
	  */

	  node.addExtra(extraName,extraValue);

	}
      if (tokens.ttype != tokens.TT_WORD)
	break;
    }
    if (!pointed)
      straighten ();
    if (tokens.ttype != tokens.TT_EOF)
    {
      String ending = (tokens.ttype == tokens.TT_WORD ?
		       "(text) " + tokens.sval :
		       "(number) " + tokens.nval);
      System.out.println ("Line " + tokens.lineno() +
			  ": Error reading graph at or just before token " +
			  ending);
    }
  }
  
}
