/*
  mapgen.c
  Set of functions to aid in generating .map files
  (c) 1998 Myricom, Inc.
  dmazzoni@myri.com (Dominic Mazzoni)
*/

#include <stdio.h>

struct Node
{
  int x,y;
  int gx, gy;
  int type;          // 0 = switch, 1 = host
  int number;
  int ports;
  int portnode[16];  // what node each port is connected to
  int portport[16];  // what port each port is connected to
  int edgenumber[16];
  char name[32];
};

struct Node *node;
int numNodes;
int numHosts;
int numSwitches;
int numDeadHosts;
int numDeadSwitches;

int currentEdge=0;
int printEdgeNumbers=0;

void Init(struct Node *n, int index)
{
  int p;

  n->x = 0;
  n->y = 0;
  n->gx = -1;
  n->gy = -1;
  n->type = 0;
  n->ports = 8;
  n->number = -1;
  for(p=0; p<16; p++)
    {
      n->portnode[p] = -1;
      n->portport[p] = -1;
      n->edgenumber[p] = -1;
    }
  sprintf(n->name,"n%d",index);
}

void NumberEdge(int n1, int n2)
{
  int p;

  while(n1<0)
    n1 += numSwitches;
  while(n2<0)
    n2 += numSwitches;

  n1 = n1%numSwitches;
  n2 = n2%numSwitches;

  printEdgeNumbers=1;
  for(p=0; p<node[n1].ports; p++)
  {
    if (node[n1].portnode[p] == n2 && node[n1].edgenumber[p] == -1)
    {
      node[n1].edgenumber[p] = currentEdge++;
      return;
    }
  }
  fprintf(stderr,"Error! %d and %d are not connected or already numbered.\n",
	  n1, n2);
  exit(0);
}

void ConnectPorts(int n1, int p1, int n2, int p2)
{
  node[n1].portnode[p1] = n2;
  node[n1].portport[p1] = p2;
  node[n2].portnode[p2] = n1;
  node[n2].portport[p2] = p1;
}

void ConnectNodesMinPN(int n1, int n2, int minPN)
{
  int i,j;

  if (node[n1].type==-1 || node[n2].type==-1)
    return;

  i = minPN;
  while (i<node[n1].ports && node[n1].portnode[i] != -1)
    i++;
  if (i==node[n1].ports)
    {
      fprintf(stderr,"*** Error: Src Node %d full!  Dest %d\n",n1,n2);
      exit(0);
    }
  j = minPN;
  while (j<node[n2].ports && node[n2].portnode[j] != -1)
    j++;
  if (j==node[n2].ports)
    {
      fprintf(stderr,"*** Error: Dest Node %d full!  Src %d\n",n2,n1);
      exit(0);
    }
  ConnectPorts(n1,i,n2,j);
}

void ConnectNodes(int n1, int n2)
{
  ConnectNodesMinPN(n1,n2,0);
}

// Creates an oct-switch out of nodes (first), (first+1), ..., (first+7)
void OctSwitch(int first)
{
  int i,j;

  for(i=0; i<4; i++)
    for(j=4; j<8; j++)
      {
	ConnectPorts(i+first,j,j+first,i);
      }
}

// new map file format
Output()
{
  int n, p, n2, c;
  int pass;

  printf("%d %d\n",numHosts-numDeadHosts,numSwitches-numDeadSwitches);
  for(pass=0; pass<2; pass++)
    for(n=0; n<numNodes; n++)
    {
      if (node[n].type==pass || node[n].type==-1)
     	continue;
      
      if (node[n].type == 1)
	printf("h - ");
      else
	{
	  printf("s %d ",node[n].ports);
	}
      printf("\"%s\"\n",node[n].name);

      // Print how many ports are in use
      
      c = 0;
      for(p=0; p<node[n].ports; p++)
	if (node[n].portnode[p]!=-1)
	  c++;
      printf("%d\n",c);

      // Print port information

      for(p=0; p<node[n].ports; p++)
	{
	  n2 = node[n].portnode[p];
	  if (n2!=-1)
	    {
	      printf("%d ",p);
	      if (node[n2].type == 1)
		printf("h - ");
	      else
		{
		  printf("s %d ",node[n2].ports);
		}
	      printf("\"%s\" %d\n",node[n2].name, node[n].portport[p]);
	    }
	}
      printf("x %d\n",node[n].x);
      printf("y %d\n",node[n].y);
      if (node[n].number >= 0)
	printf("number %d\n",node[n].number);
      if (printEdgeNumbers)
      {
	printf("edgenumbers \"");
	for(p=0; p<node[n].ports; p++)
	  printf("%d%s",
		 node[n].edgenumber[p],
		 p<node[n].ports-1 ? "," : "" );
	printf("\"\n");
      }

      printf("\n");
    }
}

void OutputGuide()
{
  int n, n2, p, c;

  for(n=0; n<numNodes; n++)
  {
    if (node[n].type!=0)
      continue;
      
    printf("S %d %d \"%s\"\n",node[n].gx, node[n].gy, node[n].name);

    // Print how many ports are in use
      
    c = 0;
    for(p=0; p<node[n].ports; p++)
      if (node[n].portnode[p]!=-1)
	c++;

    c -= 8;

    printf("%d\n",c);

    for(p=0; p<node[n].ports; p++)
    {
      n2 = node[n].portnode[p];
      if (n2!=-1 && node[n2].type==0)
	printf("%d %d %d %d \"%s\"\n",p, node[n2].gx, node[n2].gy,
	       node[n].portport[p], node[n2].name);
    }
    printf("\n");
  }
}
