/*
  sm_Host.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "insist.h"
#include "mt_Network.h"
#include "sm_Host.h"
#include "sm_Packet.h"
#include "sm_Simulator.h"
#include "mt_Message.h"

sm_Host::sm_Host (char*name, char*type, int queueSize, int queueLatency) :
  sm_Node (name, type), sendQueue (queueSize), receiveQueue (queueSize)
{
  gmId = 0;
  sleepTime = 0;
  dropCount = 0;
  hostType = 0;
  replyOption = mt_Message::PACKED_ROUTES_OPTION;
  
  insist (queueLatency >= 0);
  int b;
  b = mt_Network::MTU + mt_Route::MAX_ROUTE;
  definePort (0, b, b, sm_Node::TIME_FOR_BYTE);  
  numDrops = 0;
  this->queueLatency = queueLatency;
  neverSend = neverReceive = 0;
  
  exception: return;
}

sm_Host::~sm_Host ()
{
  sm_Packet*p;
  while ((p = (sm_Packet*) sendQueue.get ()))
    delete p;
  while ((p = (sm_Packet*) receiveQueue.get ()))
    delete p;
  if (job) delete job;
}

int sm_Host::getHostType ()
{
  insist (this);
  return hostType;
  exception: return 0;
}

int sm_Host::getGmId ()
{
  insist (this);
  return gmId;
  exception: return 0;
}

sm_Time sm_Host::getSleepTime ()
{
  insist (this);
  return this->sleepTime;
  exception: return 0;
}

sm_Time sm_Host::getTimeUtilized()
{
  return getOutput (0)->getTimeUtilized();
}

int sm_Host::addSleepTime (sm_Time sleepTime)
{
  insist (this);
  insist (sleepTime >= 0);
  
  this->sleepTime += sleepTime;
  
  return 1;
  exception: return 0;
}

int sm_Host::isNull ()
{
  return 0;
}

int sm_Host::getNodeType ()
{
  return mt_Node::HOST;
}

int sm_Host::getMaxNodes ()
{
  return 1;
}

int sm_Host::setOption (char*option, char*value)
{
  if (!sm_Node::setOption (option, value))
    return 0;
  
  if (!strcmp (option, "address"))
  {
    if (!address.fromString (value))
    {
      printFormat ("sm_Host::setOption: %s is not a valid myrinet address", value);
      return 0;
    }
  }
  else if (!strcmp (option, "hostType"))
    hostType = atoi (value);
  else if (!strcmp (option, "gmId"))
    gmId = atoi (value);
  else if (!strcmp (option, "replyOption"))
    replyOption = atoi (value);
  else if (!strcmp (option, "limit"))
  {
    if (!strcmp (value, "neverSend"))
      neverSend = 1;
    else if (!strcmp (value, "neverReceive"))
      neverReceive = 1;
    else if (!strcmp (value, "none"))
    {
      printFormat ("sm_Host::setOption: bad limit value %s", value);
      return 0;
    }
  }
  return 1;
}

int sm_Host::getReplyOption ()
{
  insist (this);
  return replyOption;
  exception: return 0;
}

int sm_Host::getNeverSend ()
{
  insist (this);
  return neverSend;
  exception: return 0;
}

int sm_Host::getNeverReceive ()
{
  insist (this);
  return neverReceive;
  exception: return 0;
}

sm_Port*sm_Host::getInput (int p)
{
  insist (this);
  insist (!p);

  return &input;
  
  exception: return 0;
}

sm_Port*sm_Host::getOutput (int p)
{
  insist (this);
  insist (!p);

  return &output;

  exception: return 0;
}

void sm_Host::definePort (int p, int inSize, int outSize, int timeForByte)
{
  insist (!p);
  insist (inSize > 0 && outSize > 0);

  input  = sm_Port (this, p, inSize, 0, timeForByte);
  output = sm_Port (this, p, outSize, 0, timeForByte);
  exception: return;
}

int sm_Host::send (sm_EventList*list, mt_Route*route, char*p, int dataLength,
		   int totalLength)
{
  insist (this);
  insist (list);
  insist (route && p);
  insist (!(totalLength % 4) && totalLength>=0);
  insist (dataLength<=totalLength && dataLength>=0);
  insist (job && job->getOpened ());
  
  if (sendQueue.isFull ())
    return 0;

  int wasEmpty;
  wasEmpty = sendQueue.isEmpty ();
  
  sm_Packet*k;

  int extra;
  if ((extra = totalLength - dataLength) < 0)
    extra = 0;
  
  k = new sm_Packet (route, p, dataLength, extra, list->getCurrentTime(), simulator->getLetPacketsDie ());
  insistp (k, ("sm_Host::send: alloc failed"));

  sendQueue.put (k);

  if (wasEmpty)
    addEvent (list, queueLatency, sm_EventList::SEND, 0);
  
  return 1;
  exception: return 0;
}

int sm_Host::send (sm_EventList*list)
{
  insist (this);
  insist (list);
  insist (job && job->getOpened ());
  
  sm_Packet*p;
  p = (sm_Packet*) sendQueue.get ();
  if (p)
  {
    p->setHeadPort (&output);
    p->setTailPort (&output);
    p->setSpeed (output.getTimeForByte ());
    int hp;
    hp = output.getSize () - 1;
    int tp;
    tp = hp - p->getLength () + 1;
    insist (tp >= 0);
    p->setHeadPosition (hp);
    p->setTailPosition (tp);
    output.setPacket (p);

    p->addEvent (list, 0, sm_EventList::UNBLOCK, 0);
  }
  
  exception: return 0;
}

void sm_Host::notifyJob (int event, char*p, int length)
{
  if (!simulator->waitNotify (job, event, p, length))
    job->receive (event, p, length);
}
  
void sm_Host::call (sm_EventList*list, int type, long data)
{
  insist (this);
  insist (job);
  insist (list);
  
  sm_Packet*p;
  p = (sm_Packet*) data;

  switch (type)
  {
    case sm_EventList::ARRIVAL:
      insist (p && p->getTailPort () == &input && p->getHeadPort () == &input);
      
      if (!p->isGoodForHost ())
      {
	insist (p->canDie ());
	delete p;
	numDrops++;
	break;
      }
      
      if (receiveQueue.isFull ())
      {
	numDrops++;
	delete p;
      }
      else
      {
	receiveQueue.put (p);	
	addEvent (list, queueLatency, sm_EventList::RECEIVE, 0);
      }
      break;
    case sm_EventList::DEPARTURE:
      insist (p && p->getTailPort () != &output);
      perr ("message %d left %s.", p->getId(), getName ());
      if (!sendQueue.isEmpty ())
	addEvent (list, queueLatency, sm_EventList::SEND, 0);
      notifyJob (mt_Network::SEND_DONE);
      break;
    case sm_EventList::SEND:
    {
      sm_Time delay = 0;
      int hostDelay = simulator->getHostDelay ();
      
      if (!job->willWait ())
	delay = hostDelay ? (sm_Time) rand (hostDelay) : 0;
      
      addEvent (list, delay * sm_EventList::MICROS_PER_TICK, sm_EventList::TO_NETWORK, 0);
      
      break;
    }
    case sm_EventList::TO_NETWORK:
      send (list);
      break;
    case sm_EventList::TO_HOST:
    {
      p = (sm_Packet*) receiveQueue.get ();
      p->received(list->getCurrentTime());
      insist (p);
      insist (dropCount >= 0);
      
      int period = simulator->getHostDropPeriod ();
      int type = simulator->getHostDropType ();

      if (period && ::rand () % period == 0 && dropCount && (type == -1 || type == ((mt_Message*)p->getMessage ())->subtype))
	dropCount--;
      else
	notifyJob (mt_Network::RECEIVE, p->getMessage(), p->getMessageLength ());
      
      simulator->addLatency(p->getLatency());
      delete p;
      break;
    }
    case sm_EventList::RECEIVE:
    {
      sm_Time delay;
      int hostDelay = simulator->getHostDelay ();
      int dropPeriod = simulator->getHostDropPeriod ();
      
      if (!hostDelay || !dropPeriod || ::rand () % dropPeriod)
	delay = 0;
      else
	delay = (sm_Time) rand (hostDelay);
			   
      addEvent (list, delay * sm_EventList::MICROS_PER_TICK, sm_EventList::TO_HOST, 0);
    }
    break;
    case sm_EventList::TIMEOUT:
      notifyJob (mt_Network::TIMEOUT);
    default: break;
  }
  exception: return;
}
 
mt_Address*sm_Host::getAddress ()
{
  insist (this);
  return &this->address;
  exception: return 0;
}

void sm_Host::setAddress (mt_Address*address)
{
  insist (this);
  insist (address);
  
  this->address = *address;

  exception: return;
}
