/*
  sm_Cloud.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "insist.h"
#include "mt_Network.h"
#include "sm_Cloud.h"
#include "sm_Packet.h"
#include "sm_Simulator.h"
#include "mt_Message.h"

sm_Cloud::sm_Cloud (char*name, char*type, int queueSize, int queueLatency) :
  sm_Node (name, type), sendQueue (queueSize), receiveQueue (queueSize)
{
  job = 0;
  inputs = outputs = 0;
  
  insist (name && type && *name && *type);
  numNodes = atoi (type);
  insistp (numNodes > 0, ("sm_Cloud::sm_Cloud: %d is not a good cloud node count", type));

  inputs = new sm_Port [numNodes];
  insistp (inputs, ("sm_Cloud::sm_Cloud: alloc of %d ports failed\n", numNodes));
  
  outputs = new sm_Port [numNodes];
  insistp (outputs, ("sm_Cloud::sm_Cloud: alloc of %d ports failed\n", numNodes));
  
  int b;
  b = mt_Network::MTU + mt_Route::MAX_ROUTE;

  for (int i = 0; i < numNodes; i++)
  {
    inputs [i] = sm_Port (this, i, b, b, sm_Node::TIME_FOR_BYTE);
    outputs [i] = sm_Port (this, i, b, b, sm_Node::TIME_FOR_BYTE);
  }
  
  insist (queueLatency >= 0);
  numDrops = 0;
  this->queueLatency = queueLatency;
  
  exception: return;
}

sm_Cloud::~sm_Cloud ()
{
  sm_Packet*p;
  while ((p = (sm_Packet*) sendQueue.get ()))
    delete p;
  while ((p = (sm_Packet*) receiveQueue.get ()))
    delete p;
  
  if (inputs) delete inputs;
  if (outputs) delete outputs;
  
  if (job) delete job;
}
sm_Time sm_Cloud::getTimeUtilized()
{
  return getOutput (0)->getTimeUtilized();
}

int sm_Cloud::isNull ()
{
  return 0;
}

int sm_Cloud::getNodeType ()
{
  return mt_Node::CLOUD;
}

int sm_Cloud::getMaxNodes ()
{
  return numNodes;
}

int sm_Cloud::setOption (char*option, char*value)
{
  if (!sm_Node::setOption (option, value))
    return 0;

  if (!strcmp (option, "address"))
  {
    if (!address.fromString (value))
    {
      printFormat ("sm_Clouds::setOption: %s is not a valid myrinet address", value);
      return 0;
    }
  }
  return 1;
}

sm_Port*sm_Cloud::getInput (int p)
{
  insist (this);
  insist (p >= 0 && p < numNodes);

  return &inputs [p];
  
  exception: return 0;
}

sm_Port*sm_Cloud::getOutput (int p)
{
  insist (this);
  insist (p >= 0 && p < numNodes);

  return &outputs [p];

  exception: return 0;
}

int sm_Cloud::send (sm_EventList*list, mt_Route*route, char*p, int dataLength,
		   int totalLength)
{
  insist (this);
  insist (list);
  insist (route && p);
  insist (!(totalLength % 4) && totalLength>=0);
  insist (dataLength <= totalLength && dataLength>=0);
  insist (job && job->getOpened ());
  
  if (sendQueue.isFull ())
    return 0;

  int wasEmpty;
  wasEmpty = sendQueue.isEmpty ();
  
  sm_Packet*k;

  int extra;
  if ((extra = totalLength - dataLength) < 0)
    extra = 0;
  
  k = new sm_Packet (route, p, dataLength, extra, list->getCurrentTime(), simulator->getLetPacketsDie ());
  insistp (k, ("sm_Cloud::send: alloc failed"));

  sendQueue.put (k);

  if (wasEmpty)
    addEvent (list, queueLatency, sm_EventList::SEND, 0);
  
  return 1;
  exception: return 0;
}

int sm_Cloud::send (sm_EventList*list)
{
  insist (this);
  insist (list);
  insist (job && job->getOpened ());
  
  sm_Packet*p;
  p = (sm_Packet*) sendQueue.get ();
  if (p)
  {
    p->setHeadPort (&outputs [0]);
    p->setTailPort (&outputs [0]);
    p->setSpeed (outputs[0].getTimeForByte ());
    int hp;
    hp = outputs[0].getSize () - 1;
    int tp;
    tp = hp - p->getLength () + 1;
    insist (tp >= 0);
    p->setHeadPosition (hp);
    p->setTailPosition (tp);
    outputs[0].setPacket (p);

    p->addEvent (list, 0, sm_EventList::UNBLOCK, 0);
  }
  
  exception: return 0;
}

void sm_Cloud::notifyJob (int event, char*p, int length)
{
  if (!simulator->waitNotify (job, event, p, length))
    job->receive (event, p, length);
}
  
void sm_Cloud::call (sm_EventList*list, int type, long data)
{
  insist (this);
  insist (job);
  insist (list);
  
  sm_Packet*p;
  p = (sm_Packet*) data;

  switch (type)
  {
    case sm_EventList::ARRIVAL:
      insist (p && p->getTailPort () == &inputs[0] && p->getHeadPort () == &inputs[0]);
      
      if (!p->isGoodForHost ())
      {
	insist (p->canDie ());
	delete p;
	numDrops++;
	break;
      }
      
      if (receiveQueue.isFull ())
      {
	numDrops++;
	delete p;
      }
      else
      {
	receiveQueue.put (p);
	addEvent (list, queueLatency, sm_EventList::RECEIVE, 0);
      }
      break;
    case sm_EventList::DEPARTURE:
      insist (p && p->getTailPort () != &outputs[0]);
      perr ("message %d left %s.", p->getId(), getName ());
      if (!sendQueue.isEmpty ())
	addEvent (list, queueLatency, sm_EventList::SEND, 0);
      notifyJob (mt_Network::SEND_DONE);
      break;
    case sm_EventList::SEND:
    {
      sm_Time delay;
      int hostDelay = simulator->getHostDelay ();
      
      delay = hostDelay ? (sm_Time) rand (hostDelay) : 0;
      addEvent (list, delay * sm_EventList::MICROS_PER_TICK, sm_EventList::TO_NETWORK, 0);
      
      break;
    }
    case sm_EventList::TO_NETWORK:
      send (list);
      break;
    case sm_EventList::TO_HOST:
    {
      p = (sm_Packet*) receiveQueue.get ();
      p->received(list->getCurrentTime());
      insist (p);
      
      notifyJob (mt_Network::RECEIVE, p->getMessage (), p->getMessageLength ());
      
      simulator->addLatency (p->getLatency ());
      delete p;
      break;
    }
    case sm_EventList::RECEIVE:
    {
      sm_Time delay;
      int hostDelay = simulator->getHostDelay ();
      int dropPeriod = simulator->getHostDropPeriod ();
      
      if (!hostDelay || !dropPeriod || ::rand () % dropPeriod)
	delay = 0;
      else
	delay = (sm_Time) rand (hostDelay);
			   
      addEvent (list, delay * sm_EventList::MICROS_PER_TICK, sm_EventList::TO_HOST, 0);
    }
    break;
    case sm_EventList::TIMEOUT:
      notifyJob (mt_Network::TIMEOUT);
    default: break;
  }
  exception: return;
}
 
void sm_Cloud::definePort (int p, int inSize, int outSize, int timeForByte)
{
  insist (this);
  
  insist (p >=0 && p < numNodes);
  insist (inSize > 0 && outSize > 0);

  inputs [p].define (inSize, 0, timeForByte);
  outputs[p].define (outSize, 0, timeForByte);
  exception: return;
}


mt_Address*sm_Cloud::getAddress ()
{
  insist (this);
  return &this->address;
  exception: return 0;
}

void sm_Cloud::setAddress (mt_Address*address)
{
  insist (this);
  insist (address);
  
  this->address = *address;

  exception: return;
}
