/*
  mt_Node.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "insist.h"
#include "mt_Node.h"
#include "mt_Follower.h"

mt_Node::mt_Node ()
{
  *name = 0;
  *type = 0;
  clonePointer = 0;
  mark = _NONE;
  typeIndex = -1;
  nodeIndex = -1;
  mapVersion = 0;
  number = 0;
  height = 0;
  in = 0;
  meshX = meshY = -1;
  meshZ = 0;

  lastHost = -1;
}

mt_Node::mt_Node (const char*name, const char*type)
{
  insist (name && *name);
  insist (type && *type);
  strncpy (this->name, name, NAME_LENGTH);
  strncpy (this->type, type, NAME_LENGTH);
  clonePointer = 0;
  mark = _NONE;
  typeIndex = -1;
  nodeIndex = -1;
  mapVersion = 0;
  number = 0;
  height = 0;
  in = 0;
  meshX = meshY = -1;
  meshZ = 0;

  exception: return;
}

mt_Node::~mt_Node ()
{
}

int mt_Node::getRouteIndex ()
{
  insist (this);
  return routeIndex;
  exception: return -1;
}

void mt_Node::setRoute (int n, const mt_Route&r)
{
  insist (this);
  if (!n) route = r;
  exception: return;  
}

void mt_Node::setRouteIndex (int routeIndex)
{
  insist (this);
  insist (routeIndex >= 0);
  this->routeIndex = routeIndex;
  exception: return;
}

char*mt_Node::getType ()
{
  insist (this);
  return type;
  exception: return 0;
}
void mt_Node::setName (char*name)
{
  insist (this);
  strncpy (this->name, name, NAME_LENGTH);
  exception:;
}

char*mt_Node::getName ()
{
  insist (this);
  return name;
  exception: return 0;
}

int mt_Node::isHost ()
{
  insist (getNodeType () > mt_Node::NODE && getNodeType () < mt_Node::NUM_TYPES);
  return getNodeType () == mt_Node::HOST;
  exception: return 0;
}

int mt_Node::isSwitch ()
{
  insist (getNodeType () > mt_Node::NODE && getNodeType () < mt_Node::NUM_TYPES);
  return getNodeType () == mt_Node::SWITCH;
  exception: return 0;
}
  
int mt_Node::isCloud ()
{
  insist (getNodeType () > mt_Node::NODE && getNodeType () < mt_Node::NUM_TYPES);
  return getNodeType () == mt_Node::CLOUD;
  exception: return 0;
}

mt_Node*mt_Node::getMatch ()
{
  insist (this);
  return match;
  exception: return 0;
}

void mt_Node::setMatch (mt_Node*m)
{
  insist (this);
  match = m;
  exception:;
} 

void mt_Node::startClone (mt_Node*n)
{
  insist (this);
  insist (n);
  insist (this->isHost () == n->isHost ());

  int maxNodes;
  maxNodes = getMaxNodes ();

  if (isHost ())
    setAddress (n->getAddress ());
  
  for (int i = 0; i < maxNodes; i++)
    if (n->getNode (i))
	connect (i, n->getOpposite (i), n->getNode (i));
  
  n->clonePointer = this;
  exception: return;
}

void mt_Node::finishClone ()
{
  insist (this);

  int maxNodes;
  maxNodes = getMaxNodes ();
  for (int i = 0; i < maxNodes; i++)
  {
    mt_Node*n = getNode (i);
    if (n)
    {
      insist (n->clonePointer);
      insist (n->clonePointer->isHost () == n->isHost ());
      connect (i, getOpposite (i), n->clonePointer);
    }
  }
  exception: return;
}

int mt_Node::getMaxStringLength ()
{
  insist (this);
  int maxNodes;
  maxNodes = getMaxNodes ();
  return (maxNodes + 1) * (NAME_LENGTH + 2 + 6 * 3);
  exception: return 0;
}

int mt_Node::getNumNodes ()
{
  insist (this);

  int maxNodes;
  maxNodes = getMaxNodes ();

  int c;
  c = 0;
  for (int i = 0; i < maxNodes; i++)
   if (getNode (i))
     c++;

  return c;
  exception: return 0;
}

int mt_Node::getNumHosts ()
{
  insist (this);

  int maxNodes;
  maxNodes = getMaxNodes ();

  int c;
  c = 0;
  for (int i = 0; i < maxNodes; i++)
   if (getNode (i) && getNode (i)->isHost ())
     c++;

  return c;
  exception: return 0;
}

int mt_Node::follow (mt_Route*r, mt_Node**n, int*in, mt_Follower*follower)
{
  insist (this);
  insist (r && n && in);
  insist (isHost ());
  
  int i;
  int length;
  length = r->getLength();
  char*hops;
  hops = r->getHops();
  
  *n = this->getNode (0);
  if (!*n)
    return 0;
  
  *in = getOpposite (0);

  if (follower) follower->step (this, 0);
    
  for (i = 0; i < length; i++)
  {
    if (!*n)
      return 0;

    if (follower) follower->step (*n, *in);
     
    int p = *in + hops[i];

    *in = (*n)->getOpposite (p);

    if (*in < 0)
      return 0;
    
    *n = (*n)->getNode (p);
  }
  if (follower && n) follower->step (*n, *in);
  
  return 1;
  exception: return 0;
}

 
mt_Node*mt_Node::getNode (mt_Route*r)
{
  insist (this);
  insist (r);
  
  mt_Node*n;
  int in;
  if (!follow (r, &n, &in))
    return 0;
  return n;
  exception: return 0;
}

mt_Node*mt_Node::getHost (mt_Address*a)
{
  insist (this);
  insist (a);

  int maxNodes;
  maxNodes = getMaxNodes ();
  
  for (int i = 0; i < maxNodes; i++)
  {
    mt_Node*n = getNode (i);
    if (n && n->isHost () && !a->compare (n->getAddress ()))
      return n;
  }
  exception: return 0;
}

mt_Node*mt_Node::getAnyHost ()
{
  insist (this);

  int maxNodes;
  maxNodes = getMaxNodes ();
  double numHosts;
  numHosts = getNumHosts ();

  if (numHosts == 0)
    return 0;
 
  if (++lastHost >= numHosts)
    lastHost = 0;
  
  insist (lastHost >= 0 && lastHost < numHosts);
  int c;
  c = 0;
  
  for (int i = 0; i < maxNodes; i++)
  {
    mt_Node*n = getNode (i);
    if (n && n->isHost () && ++c > lastHost)
      return n;  
  }
  insist (0);
  exception: return 0;
}

void mt_Node::setIn (int in)
{
  this->in = in;
}

int mt_Node::getIn ()
{
  return in;
}

int mt_Node::getId ()
{
  insist (0);
  exception: return 0;
}

int mt_Node::getControl ()
{
  insist (0);
  exception: return 0;
}

int mt_Node::setControl (int)
{
  insist (0);
  exception: return 0;
}

int mt_Node::getIn (mt_Route*r)
{
  insist (this);
  insist (r);
  
  mt_Node*n;
  int in;
  if (!follow (r, &n, &in))
    return 0;
  return in;
  exception: return 0;
}

mt_Address*mt_Node::getAddress ()
{
  insist (0);
  exception: return 0;
}

void mt_Node::setAddress (mt_Address*)
{
  insist (0);
  exception: return;
}

int mt_Node::getHostType ()
{
  insist (0);
  exception: return 0;
}

int mt_Node::getGmId ()
{
  insist (0);
  exception: return 0;
}

void mt_Node::setGmId (int)
{
  insist (0);
  exception: return;
}


void mt_Node::setMapVersion (int mapVersion)
{
  insist (this);
  this->mapVersion = mapVersion;
  exception: return;
}

int mt_Node::getMapVersion ()
{
  insist (this);
  return mapVersion;
  exception: return 0;
}

mt_Route*mt_Node::getRoute (int n)
{
  insist (this);
  return n == 0 ? &route : 0;
  exception: return 0;
}

int mt_Node::isMarked ()
{
  insist (this);
  return mark != _NONE;
  exception: return 0;
}

void mt_Node::setMark (const int mark)
{
  insist (this);
  
  insist (mark == _NONE ||
	  mark == UP ||
	  mark == DOWN);

  this->mark = mark;
  exception: return;
}

int mt_Node::getMark ()
{
  insist (this);
  return mark;
  exception: return 0;
}

void mt_Node::setTypeIndex (int typeIndex)
{
  insist (this);
  insist (typeIndex >= 0);
  
  this->typeIndex = typeIndex;
  
  exception: return;
}
void mt_Node::setNodeIndex (int nodeIndex)
{
  insist (this);
  insist (nodeIndex >= 0);

  this->nodeIndex = nodeIndex;
  
  exception: return;
}
int mt_Node::getTypeIndex ()
{
  insist (this);
  return typeIndex;
  exception: return -1;
}
int mt_Node::getNodeIndex ()
{
  insist (this);
  return nodeIndex;
  exception: return -1;
}

int mt_Node::setOption (char*option, char*value)
{
  insist (this);
  insist (option && value);
  insist (*option);
  
  if (!strcmp (option, "number"))
  {
    int n = atoi (value);
    if (n < 0)
      return 0;
    this->number = n;
    
    return 1;
  }
  else if (!strcmp (option, "meshX"))
  {
    meshX = atoi (value);
    
    if (meshX < 0)
    {
      printFormat ("mt_Node::setOption: %s is not a valid meshX", value);
      return 0;
    }
  }
  else if (!strcmp (option, "meshY"))
  {
    meshY = atoi (value);
    
    if (meshY < 0)
    {
      printFormat ("mt_Node::setOption: %s is not a valid meshY", value);
      return 0;
    }
  }
  else if (!strcmp (option, "meshZ"))
  {
    meshZ = atoi (value);
    
    if (meshZ < 0)
    {
      printFormat ("mt_Node::setOption: %s is not a valid meshZ", value);
      return 0;
    }
  }

  return 1;
  exception: return 0;
}

char* mt_Node::toString (char*s)
{
  insist (this);
  insist (s);
  char*t;
  t = s;

  int numNodes;
  numNodes = getNumNodes ();
  int maxNodes;
  maxNodes = getMaxNodes ();
  
  s += sprintf (s, "%s %s \"%s\"\n", isHost () ? "h" : isSwitch () ? "s" : "cloud", getType(), name);
  s += sprintf (s, "%d\n", numNodes);
  
  for (int i = 0; i < maxNodes; i++)
  {
    mt_Node*n = getNode (i);
    if (n)
    {
      s += sprintf (s, "%d %s %s \"%s\" %d\n",
		    i,
		    n->isHost () ? "h" : n->isSwitch () ? "s" : "cloud",
		    n->getType (),
		    n->getName (),
		    getOpposite (i));
    }
  }

  /*why not indeed?*/
  s += sprintf (s, "number %d\n", number);
  return t;
  exception: return 0;
}

int mt_Node::getPort (mt_Node*n, int k)
{
  insist (this);
  insist (n);
  insist (k >= -1);
  
  int maxNodes;
  maxNodes = getMaxNodes ();

  insist (k < maxNodes);
  
  int numPorts;
  numPorts = getNumPorts (n);
  
  insist (numPorts > 0 && numPorts <= maxNodes);
  int p;
  
  if (k < 0)
    p = rand (numPorts);
  else p = k;
  
  insist (p >= 0 && p < maxNodes);
  
  int c;
  c = 0;
  
  for (int i = 0; i < maxNodes; i++)
    if (n == getNode (i) && c++ == p)
      return i;
  
  insist (0);
  exception: return -1;
}


int mt_Node::getNumPorts (mt_Node*n)
{
  insist (this);
  insist (n);

  int maxNodes;
  maxNodes = getMaxNodes ();
  int c;
  c = 0;
  
  for (int i = 0; i < maxNodes; i++)
    if (n == getNode (i))
      c++;

  return c;
  exception: return 0;
}


int mt_Node::getMeshX ()
{
  insist (this);
  return meshX;
  exception: return -1;
}

int mt_Node::getMeshY ()
{
  insist (this);
  return meshY;
  exception: return -1;
}

int mt_Node::getMeshZ ()
{
  insist (this);
  return meshZ;
  exception: return -1;
}

int mt_Node::getDistance (mt_Node*n)
{
  insist (this);
  insist (n);

  insist (meshX >= 0 && meshY >= 0 && meshZ >= 0);
  insist (n->meshX >= 0 && n->meshY >= 0 && n->meshZ >= 0);
  
  return mt_abs (meshX - n->meshX) + mt_abs (meshY - n->meshY) + mt_abs (meshZ - n->meshZ);
  
  exception: return -1;
}

mt_Node*mt_Node::relativeToAbsolute (int numHops, char*hops)
{
  insist (this);
  insist (isHost ());
  insist (numHops >= 0 && numHops <= mt_Route::MAX_ROUTE);
  insist (hops);
  
  int in;
  mt_Node*n;
  in = this->getOpposite (0);
  n = this->getNode (0);
  insist (n);

  for (int i = 0; i < numHops; i++)
  {
    int port = in + hops [i];
    hops [i] = port;
    in = n->getOpposite (port);
    n = n->getNode (port);
    insist (n);
  }
  insist (n && n->isHost ());
  return n;
  exception: return 0;
}

mt_Node*mt_Node::absoluteToRelative (int numHops, char*hops)
{
  insist (this);
  insist (isHost ());
  insist (numHops >= 0 && numHops <= mt_Route::MAX_ROUTE);
  insist (hops);
  
  int in;
  mt_Node*n;
  in = this->getOpposite (0);
  n = this->getNode (0);
  insist (n);

  for (int i = 0; i < numHops; i++)
  {
    int port = hops [i];
    hops [i] -= in;
    in = n->getOpposite (port);
    n = n->getNode (port);
    insist (n);
  }
  //insist (n && n->isHost ());
  return n;
  exception: return 0;
}

void mt_Node::setHostType (int)
{
  insist (0);
  exception:;
}

void mt_Node::setMissing (int)
{
  insist (this);
  insist (0);
  exception:;
}

int mt_Node::getMissing ()
{
  insist (this);
  exception: return 0;
}

void mt_Node::connectMissingHost (int, mt_Node*)
{
  insist (0);
  exception:;
}

int mt_Node::getReplyOption ()
{
  insist (0);
  exception: return 0;
}

int mt_Node::setHeight (int height)
{
  insist (this);
  this->height = height;
  exception: return 0;
}

int mt_Node::getHeight ()
{
  insist (this);
  return height;
  exception: return 0;
}

