/*
  mt_MapperOptions.c
  map tools
  finucane@myri.com (David Finucane)
*/


#include <string.h>
#include <stdlib.h>

#include "insist.h"
#include "mt_MapperOptions.h"
#include "mt_Message.h"

mt_MapperOptions::mt_MapperOptions ()
{
  *mapperName = 0;
  *mapFile = 0;
  *hostFile = 0;
  *countersFile = 0;
  *routesFile = 0;
  mapOnce = 0;
  configureOnce = 0;
  neverConfigure = 0;
  neverEnd = 0;
  expectedNodes = 0;
  expectedHosts = 0;
  maxConfigureRoutes = mt_Message::MAX_CONFIGURE_ROUTES;
  maxConfigureByteStream = mt_ConfigureMessage::MAX_BYTE_STREAM;
  scoutTries = 3;
  probeTries = 3;
  configureTries = 3;
  makeHosts = 0;
  findLoops = 0;
  unit = 0;
  level = 0;
  verify = 0;
  verifyHosts = 0;
  compareSwitches = 0;
  rememberNodes = 0;
  soreLoser = 0;
  sendManyAtOnce = 0;
  sendManyCompares = 0;
  sendManyConfigures = 0;
  timeoutIncrease = 0;
  timeoutDecrease = 0;
  scoutMicroseconds = 1000;
  probeMicroseconds = 1000;
  configureMicroseconds = 1000;
  passiveMilliseconds = 5000;
  activeMilliseconds = 100;
  measureLatency = 0;
  measureConfigureLatency = 0;
  verifyProbeTries = 3;
  verifyScoutTries = 3;
  verifyProbeMicroseconds = 100;
  verifyScoutMicroseconds = 1000;
  
  verbose = 0;
  veryVerbose = 0;
  neverConfigureOthers = 0;
  justSetHostname = 0;
  neverSetHostname = 0;
  maxDepth = 0;
  probeSwitches = 0;
  interleaveConfigures = 0;
  resendConfiguresInOrder = 0;
  *bootHost = 0;
  numChangedMaps = 0;
  numUnchangedMaps = 0;
  fishGuts = 0;
  uniqueHostFile = 0;
  shortHostnames = 0;
  addMissingHosts = 0;
  configureMissingHosts = 0;
  testDeadlock = 0;
  warnings = 0;
  numReceiveBuffers = mt_Switch::NUM_PORTS * 2;
  growConfigureTimeouts = 0;
  clearRoutes = 0;
  configureOnlyWhenThereAreUnconfiguredHosts = 0;
  
  thrash = 0;

  badScoutPhases = 0;
  badProbePhases = 0;
  minScoutTime = scoutMicroseconds;
  maxScoutTime = scoutMicroseconds;
  minProbeTime = probeMicroseconds;
  maxProbeTime = probeMicroseconds;  

  maxTimeout = MAX_MICROSECONDS;
  fillGmIdGaps = 0;
}

void mt_MapperOptions::usage ()
{
  printFormat
  (
   "-mapper-args:\n"
   "-map-file <filename>\n"
   "-counters-file <filename>\n"
   "-host-file <filename>\n"
   "-unique-host-file\n"
   "-short-host-names\n"
   "-routes-file <filename>\n"
   "-unit <unit>\n"
   "-make-hosts\n"
   "-level <integer>\n"
   "-map-once\n"
   "-configure-once\n"
   "-never-configure\n"
   "-interleave-configures\n"
   "-resend-configures-in-order\n"
   "-configure-in-order\n"
   "-expected-nodes <integer>\n"
   "-expected-hosts <integer>\n"
   "-never-configure-others\n"
   "-just-set-hostname\n"
   "-never-set-hostname\n"
   "-never-end\n"
   "-verify\n"
   "-verify-hosts\n"
   "-remember-nodes\n"
   "-add-missing-hosts\n"
   "-configure-missing-hosts\n"
   "-sore-loser\n"
   "-compare-switches\n"
   "-send-many-at-once\n"
   "-send-many-compares\n"
   "-send-many-configures\n"
   "-find-loops\n"
   "-max-depth <integer>\n"
   "-scout-tries <integer>\n"
   "-probe-tries <integer>\n"
   "-verify-scout-tries <integer>\n"
   "-verify-probe-tries <integer>\n"
   "-configure-tries <integer>\n"
   "-timeout-increase <integer>\n"
   "-timeout-decrease <integer>\n"
   "-measure-latency\n"
   "-measure-configure-latency\n"
   "-scout-time <microseconds>\n"
   "-probe-time <microseconds>\n"
   "-verify-scout-time <microseconds>\n"
   "-verify-probe-time <microseconds>\n"
   "-configure-time <microseconds>\n"
   "-active-time <milliseconds>\n"
   "-passive-time <milliseconds>\n"
   "-verbose\n"
   "-very-verbose\n"
   "-mapper-name <name>\n"
   "-boot-host <name>\n"
   "-fish-guts\n"
   "-max-configure-routes <integer>\n"
   "-unchanged-maps\n"
   "-probe-switches\n"
   "-thrash\n"
   "-test-deadlock\n"
   "-warnings\n"
   "-receive-buffers <integer>\n"
   "-grow-configure-timeouts\n"
   "-clear-routes\n"
   "-max-timeout\n"
   "-fill-gm-id-gaps\n"
   "-configure-only-when-there-are-unconfigured-hosts\n"
   );
}

int mt_MapperOptions::parseArgs (mt_Args*args)
{
  int argc;  
  char**argv = args->getArgs (mt_Args::MAPPER, &argc);
  
  for (int i = 0; i < argc; i++)
  {
    insist (argv [i]);
    
    if (!strcmp (argv [i], "-map-file"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing map filename");
	return 0;
      }
      strncpy (mapFile, argv [i], mt_File::FILENAME_LENGTH);
      mt_File::getFullPath (mapFile, mapFile);
    }
    else if (!strcmp (argv [i], "-counters-file"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing counter filename");
	return 0;
      }
      strncpy (countersFile, argv [i], mt_File::FILENAME_LENGTH);
      mt_File::getFullPath (countersFile, countersFile);
    }
    else if (!strcmp (argv [i], "-host-file"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing host filename");
	return 0;
      }
      strncpy (hostFile, argv [i], mt_File::FILENAME_LENGTH);
      mt_File::getFullPath (hostFile, hostFile);
    }
    else if (!strcmp (argv [i], "-unique-host-file"))
    {
      uniqueHostFile = 1;
    }
    else if (!strcmp (argv [i], "-routes-file"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing routes filename");
	return 0;
      }
      strncpy (routesFile, argv [i], mt_File::FILENAME_LENGTH);
      mt_File::getFullPath (routesFile, routesFile);
    }
    else if (!strcmp (argv [i], "-unit"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing unit number");
	return 0;
      }
   
      unit = atoi (argv [i]);
      if (unit < 0 || unit > MAX_UNIT)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal unit number", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-make-hosts"))
    {
      makeHosts = 1;
    }
    else if (!strcmp (argv [i], "-level"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing level");
	return 0;
      }
      level = atoi (argv [i]);
    }
    else if (!strcmp (argv [i], "-max-depth"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing max depth");
	return 0;
      }
      maxDepth = atoi (argv [i]);
      if (maxDepth <= 0)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a valid max depth",
		     argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-max-configure-routes"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing max configure routes");
	return 0;
      }
      maxConfigureRoutes = atoi (argv [i]);
      if (maxConfigureRoutes <= 0 || maxConfigureRoutes > mt_Message::MAX_CONFIGURE_ROUTES)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a valid max configure routes",
		     argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-max-configure-byte-stream"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing max configure byte stream");
	return 0;
      }
      maxConfigureByteStream = atoi (argv [i]);
      if (maxConfigureByteStream <= 0 || maxConfigureByteStream > mt_ConfigureMessage::MAX_BYTE_STREAM)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a valid max configure byte stream",
		     argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-expected-nodes"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing expected nodes");
	return 0;
      }
      expectedNodes = atoi (argv [i]);
      if (expectedNodes <= 0)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a valid expected nodes",
		     argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-expected-hosts"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing expected hosts");
	return 0;
      }
      expectedHosts = atoi (argv [i]);
      if (expectedHosts <= 0)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a valid expected hosts", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-configure-only-when-there-are-unconfigured-hosts"))
    {
      configureOnlyWhenThereAreUnconfiguredHosts = 1;
    }
    else if (!strcmp (argv [i], "-map-once"))
    {
      mapOnce = 1;
    }
    else if (!strcmp (argv [i], "-configure-once"))
    {
      configureOnce = 1;
    }
    else if (!strcmp (argv [i], "-clear-routes"))
    {
      clearRoutes = 1;
    }
    else if (!strcmp (argv [i], "-never-configure"))
    {
      neverConfigure = 1;
    }
    else if (!strcmp (argv [i], "-interleave-configures"))
    {
      interleaveConfigures = 1;
    }
    else if (!strcmp (argv [i], "-resend-configures-in-order"))
    {
      resendConfiguresInOrder = 1;
    }
    else if (!strcmp (argv [i], "-short-host-names"))
    {
      shortHostnames = 1;
    }
    else if (!strcmp (argv [i], "-boot-host"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing boot node name");
	return 0;
      }
      strncpy (bootHost, argv [i], mt_Node::NAME_LENGTH);
    }
    else if (!strcmp (argv [i], "-unchanged-maps"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing unchanged maps count");
	return 0;
      }
      numUnchangedMaps = atoi (argv [i]);
      if (numUnchangedMaps <= 0)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is a bad unchanged maps count", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-changed-maps"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing changed maps count");
	return 0;
      }
      numChangedMaps = atoi (argv [i]);
      if (numChangedMaps <= 0)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is a bad changed maps count", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-receive-buffers"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing receive-buffers count");
	return 0;
      }
      numReceiveBuffers = atoi (argv [i]);
      if (numReceiveBuffers <= 2)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is a bad receive-buffers count", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-fish-guts"))
    {
      fishGuts = 1;
    }
    else if (!strcmp (argv [i], "-never-end"))
    {
      neverEnd = 1;
    }
    else if (!strcmp (argv [i], "-compare-switches"))
    {
      compareSwitches = 1;
    }
    else if (!strcmp (argv [i], "-thrash"))
    {
      thrash = 1;
    }
    else if (!strcmp (argv [i], "-test-deadlock"))
    {
      testDeadlock = 1;
    }
    else if (!strcmp (argv [i], "-warnings"))
    {
      warnings = 1;
    }
    else if (!strcmp (argv [i], "-send-many-at-once"))
    {
      sendManyAtOnce = 1;
    }
    else if (!strcmp (argv [i], "-send-many-compares"))
    {
      sendManyCompares = 1;
    }
    else if (!strcmp (argv [i], "-send-many-configures"))
    {
      sendManyConfigures = 1;
    }
    else if (!strcmp (argv [i], "-remember-nodes"))
    {
      rememberNodes = 1;
    }
    else if (!strcmp (argv [i], "-add-missing-hosts"))
    {
      addMissingHosts = 1;
    }
    else if (!strcmp (argv [i], "-configure-missing-hosts"))
    {
      configureMissingHosts = 1;
    }
    else if (!strcmp (argv [i], "-sore-loser"))
    {
      soreLoser = 1;
    }
    else if (!strcmp (argv [i], "-verbose"))
    {
      verbose = 1;
    }
    else if (!strcmp (argv [i], "-very-verbose"))
    {
     verbose = veryVerbose = 1;
    }
    else if (!strcmp (argv [i], "-verify"))
    {
      verify = 1;
    }
    else if (!strcmp (argv [i], "-verify-hosts"))
    {
      verifyHosts = 1;
    }
    else if (!strcmp (argv [i], "-never-configure-others"))
    {
      neverConfigureOthers = 1;
    }
    else if (!strcmp (argv [i], "-just-set-hostname"))
    {
      justSetHostname = 1;
    }
    else if (!strcmp (argv [i], "-never-set-hostname"))
    {
      neverSetHostname = 1;
    }
    else if (!strcmp (argv [i], "-find-loops"))
    {
      findLoops = 1;
    }
    else if (!strcmp (argv [i], "-scout-tries"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing scout tries");
	return 0;
      }
   
      scoutTries = atoi (argv [i]);
      if (scoutTries < 0 || scoutTries > MAX_TRIES)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal scout tries", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-probe-tries"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing probe tries");
	return 0;
      }
   
      probeTries = atoi (argv [i]);
      if (probeTries < 0 || probeTries > MAX_TRIES)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal probe tries", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-verify-scout-tries"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing verify scout tries");
	return 0;
      }
   
      verifyScoutTries = atoi (argv [i]);
      if (verifyScoutTries < 0 || verifyScoutTries > MAX_TRIES)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal verify scout tries", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-verify-probe-tries"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing verify probe tries");
	return 0;
      }
   
      verifyProbeTries = atoi (argv [i]);
      if (verifyProbeTries < 0 || verifyProbeTries > MAX_TRIES)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal verify probe tries", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-configure-tries"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing configure tries");
	return 0;
      }
   
      configureTries = atoi (argv [i]);
      if (configureTries < 0 || configureTries > MAX_TRIES)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal configure tries", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-measure-latency"))
    {
      measureLatency = 1; 
    }
    else if (!strcmp (argv [i], "-grow-configure-timeouts"))
    {
      growConfigureTimeouts = 1; 
    }
    else if (!strcmp (argv [i], "-measure-configure-latency"))
    {
      measureConfigureLatency = 1; 
    }
    else if (!strcmp (argv [i], "-timeout-increase"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing timeout increase");
	return 0;
      }
   
      timeoutIncrease = atoi (argv [i]);
      if (timeoutIncrease < 0 || timeoutIncrease > 100)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal timeout-increase", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-timeout-decrease"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing timeout decrease");
	return 0;
      }
   
      timeoutDecrease = atoi (argv [i]);
      if (timeoutDecrease < 0 || timeoutDecrease > 100)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal timeout-decrease", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-max-timeout"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing max time");
	return 0;
      }
   
      maxTimeout = atoi (argv [i]);
      if (maxTimeout < 0 || maxTimeout > MAX_MICROSECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal max timeout", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-fill-gm-id-gaps"))
    {
      fillGmIdGaps = 1;
    }
    else if (!strcmp (argv [i], "-scout-time"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing scout time");
	return 0;
      }
   
      scoutMicroseconds = atoi (argv [i]);
      if (scoutMicroseconds < 0 || scoutMicroseconds > MAX_MICROSECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal scout time", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-probe-time"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing probe time");
	return 0;
      }
   
      probeMicroseconds = atoi (argv [i]);
      if (probeMicroseconds < 0 || probeMicroseconds > MAX_MICROSECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal probe time", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-verify-scout-time"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing verify scout time");
	return 0;
      }
   
      verifyScoutMicroseconds = atoi (argv [i]);
      if (verifyScoutMicroseconds < 0 || verifyScoutMicroseconds > MAX_MICROSECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal verify scout time", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-verify-probe-time"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing verify probe time");
	return 0;
      }
   
      verifyProbeMicroseconds = atoi (argv [i]);
      if (verifyProbeMicroseconds < 0 || verifyProbeMicroseconds > MAX_MICROSECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal verify probe time", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-configure-time"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing configure time");
	return 0;
      }
   
      configureMicroseconds = atoi (argv [i]);
      if (configureMicroseconds < 0 || configureMicroseconds > MAX_MICROSECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal configure time", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-active-time"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing active time");
	return 0;
      }
   
      activeMilliseconds = atoi (argv [i]);
      if (activeMilliseconds < 0 || activeMilliseconds > MAX_MILLISECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal active time", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-passive-time"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing passive time");
	return 0;
      }
   
      passiveMilliseconds = atoi (argv [i]);
      if (passiveMilliseconds < 0 || passiveMilliseconds > MAX_MILLISECONDS)
      {
	printFormat ("mt_MapperOptions::parseArgs: %s is not a legal passive time", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-mapper-name"))
    {
      if (++i == argc)
      {
	printFormat ("mt_MapperOptions::parseArgs: missing mapper name");
	return 0;
      }
      strncpy (mapperName, argv [i], mt_Node::NAME_LENGTH);
    }
    else if (!strcmp (argv [i], "-probe-switches"))
      probeSwitches = 1;
    else 
    {
      printFormat ("mt_MapperOptions::parseArgs: bad option \"%s\"", argv [i]);
      return 0;
    }
  }

  minScoutTime = 1;
  maxScoutTime = maxTimeout;
  
  minProbeTime = 1;
  maxProbeTime = maxTimeout;

  if (scoutMicroseconds > maxTimeout)
    scoutMicroseconds = maxTimeout;
  
  if (probeMicroseconds > maxTimeout)
    probeMicroseconds = maxTimeout;

  if (configureMicroseconds > maxTimeout)
    configureMicroseconds = maxTimeout;

  return 1;
  exception: return 0;
}

int mt_MapperOptions::decreaseTimeouts ()
{
  insist (this);
  
  int changed;
  changed = 0;
  
  insist (scoutMicroseconds >= minScoutTime && scoutMicroseconds <= maxScoutTime);
  insist (probeMicroseconds >= minProbeTime && probeMicroseconds <= maxProbeTime);
  insist (minScoutTime > 0 && maxScoutTime <= maxTimeout);
  insist (minProbeTime > 0 && maxProbeTime <= maxTimeout);
  
  if (badScoutPhases == 0)
  {
    changed = (scoutMicroseconds != minScoutTime) && timeoutDecrease;
    
    scoutMicroseconds -= (int) (scoutMicroseconds * ((double) timeoutDecrease / (double) 100));
    if (scoutMicroseconds < minScoutTime)
      scoutMicroseconds = minScoutTime;
  }
  if (badProbePhases == 0)
  {
    changed |= (probeMicroseconds != minProbeTime) && timeoutDecrease;
    
    probeMicroseconds -= (int) (probeMicroseconds * ((double) timeoutDecrease / (double) 100));
    if (probeMicroseconds < minProbeTime)
      probeMicroseconds = minProbeTime;
  }
  badScoutPhases = badProbePhases = 0;
  return changed;
  exception: return 0;
}

int mt_MapperOptions::increaseTimeouts (int type)
{
  int maxTime = 0;
  int*timeout = 0;
  int changed = 0;
  
  insist (type >= 0 && type < mt_Message::NUM_TYPES);  
  insist (scoutMicroseconds >= minScoutTime && scoutMicroseconds <= maxScoutTime);
  insist (probeMicroseconds >= minProbeTime && probeMicroseconds <= maxProbeTime);
  insist (minScoutTime > 0 && maxScoutTime <= maxTimeout);
  insist (minProbeTime > 0 && maxProbeTime <= maxTimeout);

  switch (type)
  {
    case mt_Message::REPLY:
    case mt_Message::CLOUD_QUERY_REPLY:
      timeout = &scoutMicroseconds;
      maxTime = maxScoutTime;
      badScoutPhases++;
    break;

    case mt_Message::PROBE:
    case mt_Message::COMPARE:
    case mt_Message::ID_PROBE:
      badProbePhases++;
      timeout = &probeMicroseconds;
      maxTime = maxProbeTime;
    break;
  }

  if (timeout)
  {
    int increase;
    insist (maxTime);
    
    increase = (int) ((double) *timeout * (double) timeoutIncrease / (double) 100);

    if (timeoutIncrease && increase == 0)
      increase = 1;

    changed = *timeout != maxTime && increase;
    
    *timeout += increase;
    
    if (*timeout > maxTime)
      *timeout = maxTime;
  }    
  return changed;
  exception: return 0;
}

int mt_MapperOptions::setTimeouts (int scout, int probe, int configure)
{
  insist (this);
  
  if (scout >= minScoutTime && scout <= maxScoutTime)
    scoutMicroseconds = scout;
  
  if (probe >= minProbeTime && scout <= maxProbeTime)
    probeMicroseconds = probe;
  
  if (configure >= 1 && configure <= maxTimeout)
    configureMicroseconds = configure;
  
  return 1;
  exception: return 0;
}

