/*
  mt_Mapper.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <string.h>
#include <stdlib.h>
#include "mt_htonl.h"

#include "insist.h"
#include "mt_Mapper.h"
#include "mt_Network.h"
#include "mt_FileWriter.h"
#include "mt_MapFile.h"
#include "mt_Queue.h"
#include "mt_Switch.h"
#include "de_Deadlock.h"

mt_Mapper::Counters::Counters ()
{
  for (int i = 0; i < mt_Message::NUM_TYPES; i++)
    receives [i] = sends [i] = 0;
  
  badTypes = 0;
  badLengths = 0;
  gmIdConflicts = 0;
  badContents = 0;
  badPhases = 0;
  mapVersion = 0;
  activeMapper = 0;
  missingSwitches = 0;
  missingHosts = 0;
  newSwitches = 0;
  newHosts = 0;
  hostCount = 0;
  switchCount = 0;
}

void mt_Mapper::Counters::toFile (FILE*fp)
{
  insist (this);
  insist (fp);

  for (int i = 0; i < mt_Message::NUM_TYPES; i++)
  {
    fprintf (fp,"%s sent %d\n", mt_Message::names [i], sends [i]);
    fprintf (fp,"%s received %d\n", mt_Message::names [i], receives [i]);
  }
  
  fprintf (fp, "bad types %d\n", badTypes);
  fprintf (fp, "bad lengths %d\n", badLengths);
  fprintf (fp, "bad contents %d\n", badContents);
  fprintf (fp, "bad phases %d\n", badPhases);
  fprintf (fp, "map version %d\n", mapVersion);
  fprintf (fp, "active mapper %d\n", activeMapper);
  fprintf (fp, "missing ports %d\n", missingSwitches);
  fprintf (fp, "missing hosts %d\n", missingHosts);
  fprintf (fp, "new ports %d\n", newSwitches);
  fprintf (fp, "new hosts %d\n", newHosts);
  fprintf (fp, "id conflicts %d\n", gmIdConflicts);
  fprintf (fp, "host count %d\n", hostCount);
  fprintf (fp, "switch count %d\n", switchCount);

  exception: return;
}

void mt_Mapper::writeCounters (char*filename)
{    
  insist (this);
  insist (filename);
  static int count;
  
  if (*filename)
  {
    mt_mapper_c (("writing counters file \"%s\"", filename));
  
    mt_FileWriter w (filename);
    if (w.fp)
    {
      fprintf (w.fp, ";%d\n\n", count++);
      counters.toFile (w.fp);
    }
  }
  exception: return;
}


void mt_Mapper::writeMap (mt_Graph*graph, char*filename)
{    
  insist (this);
  insist (graph && filename);
  static int count;
  
  if (*filename)
  {
    mt_mapper_c (("writing map file \"%s\"", filename));
  
    mt_FileWriter w (filename);
    if (w.fp)
    {
      fprintf (w.fp, ";%d\n\n", count++);
      mt_MapFile m (w.fp);
      m.write (graph);
    }
  }
  exception: return;
}

int mt_Mapper::equalGraphs (mt_Graph*g1, mt_Graph*g2)
{
  mt_Node*h;
  insist (this);
  insist (g1 && g2);

  if ((h = g1->unequalMapVersions (g2)))
  {
    mt_mapper_c (("host %s has a new mapVersion", h->getName()));
    return 0;
  }
  else if (!g1->equals (g2))
  {
    mt_mapper_c (("the graphs are not equal."));
    return 0;
  }
  return 1;
  exception: return 0;
}

void mt_Mapper::writeHostTable (int makeHosts, char*filename)
{    
  insist (this);
  insist (filename);

  if (makeHosts && *filename)
  {
    mt_mapper_c (("writing host file \"%s\"", filename));
  
    mt_FileWriter w (filename);
    if (w.fp)
      hostTable->toFile (w.fp);
  }
  exception: return;
}

int mt_Mapper::makeHostTable (int gmId, int hostType, mt_Address*address, char*hostname, int makeHosts, char*filename)
{
  insist (this);
  insist (address && hostname && filename);

  insist (!hostTable);
  
  if (makeHosts && *filename)
    hostTable = new mt_HostTable (filename);
  else
    hostTable = new mt_HostTable ();
  
  insistp (hostTable, ("mt_Mapper::makeHostTable: alloc failed"));

  if (!hostTable->resolve (gmId, hostType, address))
    counters.gmIdConflicts++;
  
  if (*hostname)
    hostTable->setHostname (address, hostname);
    
  return 1;
  exception: return 0;
}

int mt_Mapper::makeHostTable (char*filename, mt_Graph*graph)
{
  insist (this);
  insist (graph);

  insist (!hostTable);
  
  hostTable = filename ? new mt_HostTable (filename) : new mt_HostTable ();
  
  insistp (hostTable, ("mt_Mapper::makeHostTable: alloc failed"));

  hostTable->assignAddresses (graph);
  
  int numHosts;
  numHosts = graph->getNumHosts ();
  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*n = graph->getHost (i);
    insist (n);
    
    if (!hostTable->resolve (n->getGmId (), n->getHostType(), n->getAddress ()))
    {
      mt_mapper_w (("inconsistent gm id for node %s", n->getName ()));
    }
    hostTable->setHostname (n->getAddress (), n->getName ());
    hostTable->setOption (n->getAddress (), mt_Message::PACKED_ROUTES_OPTION | mt_Message::LONG_HOSTNAME_OPTION);
  }
  return options.fillGmIdGaps ? hostTable->assignGmIdsByFillingInGaps (graph) : hostTable->assignGmIds (graph);

  exception: return 0;
}


mt_Mapper::mt_Mapper (mt_Node*node, mt_Network*network, mt_Calculator*calculator, mt_MapperOptions&options) : mt_Job (node, network)
{
  acks = 0;
  times = 0;
  
  hostTable = 0;
  this->calculator = calculator;
  for (int i = 0; i < mt_Message::NUM_TYPES; i++)
    clearHandler (i);

  this->options = options;
  lastIncreaseTimeoutsPhase = -1;
}

mt_Mapper::~mt_Mapper ()
{
  if (hostTable)
    delete hostTable;
}

void mt_Mapper::dump (FILE*)
{
}


void mt_Mapper::setHandler (int type, mt_MessageHandler handler)
{
  insist (this);
  insist (handler != 0);
  insist (type >= 0 && type < mt_Message::NUM_TYPES);
  
  handlers [type] = handler;
  
  exception: return;
}

void mt_Mapper::clearHandler (int type)
{
  insist (this);
  insist (type >= 0 && type < mt_Message::NUM_TYPES);
  
  handlers [type] = 0;

  exception: return;
}

void mt_Mapper::callMyHandler (int type, mt_Message*m, int length)
{
  (this->*handlers [type])(m, length);
}

int mt_Mapper::wait (int microseconds)
{
  insist (this);

  this->getNetwork ()->setTimer (this, microseconds);
  return wait ();
  
  exception: return 0;
}

int mt_Mapper::wait ()
{
  insist (this);
  
  char*p;
  int length;

  mt_mapper_v (("waiting for timeout"));
  wait (mt_Network::TIMEOUT, 0, 0, &p, &length);
  exception: return 0;
}

void mt_Mapper::printPacket (char*p, int length)
{
  char buffer [mt_Network::MTU / 10];
  int n = 0;
  
  printFormat ("dumping packet of length %d\n", length);
  
  for (int i = 0; i < length && n < mt_Network::MTU / 10 - 8; i++)
    n += sprintf (buffer + n, "%02x%c", (unsigned char) (0xff & p[i]), i % 8 == 7 ? '\n' : ' ');

  printFormat ("%s", buffer);
}

	
int mt_Mapper::wait (int event, int subtype, int phase, char**p, int*length)
{
  insist (this);
  insist (p && length);

  mt_Network*network;
  network = getNetwork ();
  insist (network);

  mt_mapper_v (("waiting for event type %d, message type %d", event, subtype));
  
  int e;
  while ((e = network->wait (p, length)))
  {
    if (e == mt_Network::RECEIVE)
    {
      mt_mapper_v (("got a receive event (event type %d)", e));

      insist (*p);
      insist (*length);
      
      mt_Message*m;
      m = (mt_Message*)*p;
      
      int mtype;
      int stype;
      int mphase;

      mtype = mt_htons (m->type);
      stype = mt_htons (m->subtype);
      mphase = mt_htonl (m->phase);

      if (options.veryVerbose)
	printPacket (*p, *length);
	
      if (mtype != mt_Message::GM_TYPE || stype < 0 || stype >= mt_Message::NUM_TYPES)
      {
	mt_mapper_v (("got message with bad type %d", mtype));
	counters.badTypes++;
	
	network->freePacket (*p);
	continue;
      }
      counters.receives [stype]++;

      if (event == mt_Network::RECEIVE && stype == subtype)
      {
	mt_mapper_v (("got expected message"));
	if (phase == 0 || mphase == phase)
	  return 1;

	mt_mapper_v (("late message"));
	counters.badPhases++;
	network->freePacket (*p);
	increaseTimeouts (stype, mphase);
	
	continue;
      }
      else
      {
	mt_mapper_v (("calling handler for unexpected message type %d", stype));

	if (handlers [stype])
	  callMyHandler (stype, m, *length);
	else
	  callHandler (stype, m, *length);
      }
      
      network->freePacket (*p);
    }
    else if (e == event)
    {
      mt_mapper_v (("got event we were waiting for"));
      return 1;
    }
    if (e == mt_Network::TIMEOUT)
    {
      mt_mapper_v (("timed out"));
      return 0;
    }
  }
  mt_mapper_w (("that's strange. I ran out of events."));
  mt_mapper_w (("At least I should have got a timeout"));
  exception: return 0;
}

int mt_Mapper::send (mt_Route*route, char*p, int length)
{
  insist (this);
  insist (length > 0);
  insist (!(length % 4));
  insist (length <= mt_Network::MTU);
  
  mt_Network*network;
  network = getNetwork ();
  insist (network);
  
  mt_mapper_v (("sending with route %s", route->toString ()));

  if (options.veryVerbose)
    printPacket (p, length);

  while (!network->isAborted () && !network->send (this, route, p, length))
  {
    char*px;
    int lengthx;

    mt_mapper_v (("send failed. waiting for send done event"));    
    if (!wait (mt_Network::SEND_DONE, 0, 0, &px, &lengthx))
    {
      mt_mapper_w (("serious error in sending: didn't get a expected send done"));
      return 0;
    }
    
    mt_mapper_v (("got send done event"));
  }
  mt_mapper_v (("sent message"));
  return 1;
  exception: return 0;
}


void mt_Mapper::receive (int, char*, int)
{
  insist (this);
  insist (0);
  exception: return;
}
  
int mt_Mapper::isConfigured (mt_Graph*graph, int mapVersion)
{
  insist (this);
  insist (graph);
  insist (hostTable);
  
  int configured;
  configured = 1;

  int numHosts;
  numHosts = graph->getNumHosts ();

  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*h = graph->getHost (i);
    insist (h && h->isHost ());
    
    int state = hostTable->getState (h->getAddress ());
    insist (state);
    insist (h->getNode (0));
    if (h->getNode (0)->getNodeType () == mt_Node::CLOUD && h->getOpposite (0))
    {
      insist (state == mt_HostTable::CONFIGURED);
      mt_mapper_c (("not going to configure %s (it is a cloud host)", h->getName ()));
      continue;
    }
    if (state == mt_HostTable::INCONSISTENT || hostTable->getGmId (h->getAddress ()) != h->getGmId ())
    {
      mt_mapper_w (("%s has an id conflict", h->getName ()));
      hostTable->setState (h->getAddress (), mt_HostTable::INCONSISTENT);
      configured = 0;
    }
    else if (!h->getGmId ())
    {
      mt_mapper_w (("%s was not assigned a gm id. It cannot be configured.", h->getName ()));
      mt_mapper_w (("perhaps the makeHosts option was disabled?"));
      
      hostTable->setState (h->getAddress (), mt_HostTable::INCONSISTENT);
      configured = 0;
    }
    else if (state != mt_HostTable::FIRST_SEEN && h->getMapVersion () != mapVersion)
    {
      mt_mapper_c (("%s needs configuring", h->getName ()));
      
      hostTable->setState (h->getAddress (), mt_HostTable::NOT_CONFIGURED);
      configured = 0;
    }
    else if (state != mt_HostTable::CONFIGURED)
      configured = 0;
  }
  return configured;
  exception: return 0;
}


int mt_Mapper::higherPriority (mt_Address*hisAddress, int hisLevel,
			       mt_Address*myAddress, int myLevel)
{

  insist (this);
  insist (hisAddress && myAddress);
  //insist (hisAddress->compare (myAddress) || hisAddress == myAddress);
  
  if (hisLevel > myLevel)
    return 1;
  if (hisLevel < myLevel)
    return 0;
  
  return hisAddress->compare (myAddress) > 0;
  
  exception: return 0;
}    
      
int mt_Mapper::verifyHost (mt_Route*r, int*phase, int tries, int microseconds, mt_Address*address,
			   mt_Address*mapperAddress, int level, int*mapVersion, char*hostname)
{
  mt_Address a;
  
  insist (this);
  insist (r && phase && mapVersion);
  insist (address && !address->isZero ());
  insist (mapperAddress);
  insist (hostname);
  insist (tries > 0);
  insist (microseconds > 0);
  
  {
    mt_Route b (*r);
    b.invert ();
    
    mt_Network*network = getNetwork ();
    insist (network);
    
    for (int i = 0; i < tries; i++)
    {
      mt_ScoutMessage m (0, ++(*phase), &b, address, level);
      int size = m.getSize ();
      m.swap ();
      counters.sends [mt_Message::SCOUT]++;

      if (!send (r, (char*) &m, size))
	return 0;
      
      counters.sends [mt_Message::SCOUT]++;
      network->setTimer (this, microseconds);

      char*p;
      int length;
  
      if (wait (mt_Network::RECEIVE, mt_Message::REPLY, *phase, &p, &length))
      {
	network->clearTimer (this);
	insist (p);
      
	mt_ReplyMessage*rm;
	rm = (mt_ReplyMessage*) p;

	if (length < (int) sizeof (mt_ReplyMessage) - mt_Message::HOSTNAME_LENGTH)
	{
	  counters.badLengths++;
	  mt_mapper_w (("bad length"));
	  network->freePacket (p);
	  return 0;
	}
	rm->swap ();
	insist (rm->phase == *phase);
      
	*mapVersion = rm->mapVersion;
	strncpy (hostname, rm->getHostname (), mt_Network::HOSTNAME_LENGTH);
	a.fromBytes (rm->address);
	mapperAddress->fromBytes (rm->mapperAddress);
	
	mt_mapper_c (("got a (verify) reply from %s (address %s)", hostname, a.toString ()));
	mt_mapper_c (("mapper address %s   mapVersion %d", 
		      mapperAddress->toString (), rm->mapVersion));

	int verified = !higherPriority (&a, rm->level, address, level);

	mt_mapper_c (("his priority is %s than mine", verified ? "not higher" : "higher"));
	network->freePacket (p);
	return verified;
      }    
    }
  }

  exception: return 0;
}

int mt_Mapper::verifySwitch (mt_Route*r, int*phase, int tries, int microseconds)
{
  insist (this);
  insist (r &&  phase);
  insist (tries > 0);
  insist (microseconds > 0);
  
  mt_Network*network;
  network = getNetwork ();
  insist (network);
  
  for (int i = 0; i < tries; i++)
  {
    mt_VerifyMessage m (++(*phase));
    m.swap ();
    counters.sends [mt_Message::VERIFY]++;

    if (!send (r, (char*) &m, sizeof (mt_VerifyMessage)))
      return 0;
    
    counters.sends [mt_Message::VERIFY]++;
    network->setTimer (this, microseconds);

    char*p;
    int length;
    
    if (wait (mt_Network::RECEIVE, mt_Message::VERIFY, *phase, &p, &length))
    {
      network->clearTimer (this);
      insist (p);
      network->freePacket (p);
      return 1;
    }
  }
  exception: return 0;
}

int mt_Mapper::verify (mt_Graph*graph, mt_Node*root, int*phase, int mapVersion,
		       mt_Address*address, int level, mt_Route*change, int*isHost,
		       char*hostname)
{
  mt_Queue queue;
  
  insist (this);
  insist (graph && root);
  insist (phase);  
  insist (change && isHost);
  insist (hostname);
  
  graph->clearMarks ();
  
  mt_Node*n;

  int numSeen;
  numSeen = 0;

  insist (root->isHost ());
  
  root->setMark (mt_Node::UP);
  numSeen++;
  
  mt_Node*first;
  first = root->getNode (0);

  if (!first)
    return 0;
  
  queue.put (first);
  first->setMark (mt_Node::UP);
  numSeen++;
  
  while ((n = (mt_Node*) queue.get ()))
  {
    insist (n->isMarked ());
    
    int maxNodes = n->getMaxNodes ();

    mt_Route r1 (*n->getRoute (0));
    mt_Route r2 (r1);
    r2.invert ();
    
    if (n->isHost ())
    {
      int v;
      mt_Address a;
      
      mt_mapper_c (("verifying host %s", n->getName ()));
      if (!verifyHost (&r1, phase, options.verifyScoutTries, options.verifyScoutMicroseconds,
		       address, &a, level, &v, hostname)
	  || v != mapVersion || a.compare (address) || *hostname && strcmp (n->getName (), hostname))
      {
	mt_mapper_c (("host gone or changed. aborting verify"));
	mt_mapper_c (("map version was %d (expected %d)", v, mapVersion));
	mt_mapper_c (("mapper address was %s", a.toString ()));
	mt_mapper_c (("(expected %s)", address->toString ()));
	mt_mapper_c (("hostname was %s (expected %s)", hostname, n->getName ()));
	
	*change = r1;
	*isHost = 1;
	counters.missingHosts++;
	queue.empty ();
	return 0;
      }
      else continue;  
    }

    int in = root->getIn (&r1);

    if (!options.verifyHosts)
    {
      mt_Route r3 (&r1, 0, &r2);      
      
      mt_mapper_c (("verifying switch %s", n->getName ()));
      
      if (!verifySwitch (&r3, phase, options.verifyProbeTries, options.verifyProbeMicroseconds))
      {
	mt_mapper_c (("switch gone. aborting verify"));
	
	*change = r1;
	*isHost = 0;
	counters.missingSwitches++;
	queue.empty ();
	return 0;
      }
    }

    for (int i = 1 - maxNodes; i < maxNodes; i++)
    {
      mt_Node*t = i < 0 ? 0 :  n->getNode (i);
      
      int h = i - in;

      if (h < mt_Route::MIN_HOP || h > mt_Route::MAX_HOP)
	continue;
      
      mt_Route r4 (&r1, h);
      mt_Route r5 (&r1, h, 0, -h, &r2);
      
      if (!t && !options.verifyHosts)
      {
	int v;
	mt_Address a;
	
	if (verifyHost (&r4, phase, options.scoutTries, options.scoutMicroseconds,
			address, &a, level, &v, hostname))
	{
	  mt_mapper_c (("new host found. aborting verify"));
      
	  *change = r4;
	  *isHost = 1;
	  counters.newHosts++;
	  queue.empty ();
	  return 0;
	}
	if (verifySwitch (&r5, phase, options.probeTries, options.probeMicroseconds))
	{
	  mt_mapper_c (("new switch found. aborting verify"));
      
	  *change = r4;
	  *isHost = 0;
	  counters.newSwitches++;
	  queue.empty ();
	  return 0;
	}
      }
      else if (t && !t->isMarked ())
      {
	numSeen++;
	t->setMark (mt_Node::UP);
	queue.put (t);
      }
    }
  }
  
  insist (numSeen == graph->getNumNodes ());
  
  return 1;
  exception: return 0;
}


int mt_Mapper::configureLocal (mt_Graph*graph, int from,  mt_Calculator*calculator)
{
  insist (this);
  insist (hostTable);
  insist (graph);
  insist (calculator);

  mt_mapper_c (("configuring myself"));
  
  mt_Network*network;
  network = getNetwork ();
  insist (network);

  int numHosts;
  numHosts = graph->getNumHosts ();
  insist (from >= 0 && from < numHosts);
  
  mt_Node*n;
  n = graph->getHost (from);
  insist (n);

  mt_Address*a;
  a = n->getAddress ();
  
  int state;
  state = hostTable->getState (a);

  if (state == mt_HostTable::CONFIGURED || state == mt_HostTable::INCONSISTENT)
  {
    mt_mapper_c (("on second thought, I'm not going to configure myself."));
    return state == mt_HostTable::CONFIGURED;
  }
  
  mt_mapper_c (("clearing routes"));

  if (options.clearRoutes)  
    network->clearRoutes (this);
  
  for (int i = 0; i < numHosts; i++)
  {
    mt_Route r;
    mt_Address*ai = graph->getHost (i)->getAddress();
    insist (ai);
    
    int gmId = hostTable->getGmId (ai);

    if (gmId > 0)
    {
      if (!calculator->getRoute (from, i, 0, &r) && numHosts != 2)
	return 0;

      insist (graph->getNumNodes (mt_Node::CLOUD) ||
	      (i == from && graph->getNumSwitches () == 0) ||
	      graph->getHost (from)->getNode (&r) == graph->getHost (i));

      if (graph->getHost (i)->getHostType () != graph->getHost (from)->getHostType ())
      {
	mt_mapper_c (("ignoring route to %s (nodeType %d is not ours, which is %d)",
		      graph->getHost (i)->getName (),
		      graph->getHost (i)->getHostType (),
		      graph->getHost (from)->getHostType ()));
	continue;
      }
      mt_mapper_v (("setting local route %d (%s) to %s", gmId, hostTable->getHostname (ai), r.toString ()));
      
      network->setRoute (this, gmId, &r);
      network->setAddress (this, gmId, ai);
      network->setHostname (this, gmId, hostTable->getHostname (ai));
    }
  }

  hostTable->setState (a, mt_HostTable::CONFIGURED);
  network->setGmId (this, hostTable->getGmId (a));
  
  mt_mapper_c (("I configured myself."));
  mt_mapper_c (("So it hasn't been all infinite loops and kludges."));

  return 1;
  exception: return 0;
}



int mt_Mapper::clearConflict (mt_Node*host, int*phase,  int level)
{
  insist (this);
  insist (host && phase);

  mt_Route*r;
  r = host->getRoute (0);
  insist (r);
  
  {
    mt_Route b (*r);
    b.invert ();
    
    mt_Network*network = getNetwork ();
    insist (network);
    
    for (int i = 0; i < options.scoutTries; i++)
    {
      mt_ScoutMessage m (0, ++(*phase), &b, host->getAddress (), level, mt_ScoutMessage::RESET);
      int size = m.getSize ();
      m.swap ();

      mt_mapper_c (("sending reset message to %s because it has an inconsistent gmId", host->getName ()));
      
      if (!send (r, (char*) &m, size))
	return 0;
      
      counters.sends [mt_Message::SCOUT]++;
      network->setTimer (this, options.scoutMicroseconds);

      char*p;
      int length;
  
      if (wait (mt_Network::RECEIVE, mt_Message::REPLY, *phase, &p, &length))
      {
	network->clearTimer (this);
	insist (p);
      
	mt_ReplyMessage*rm;
	rm = (mt_ReplyMessage*) p;

	if (length < (int) sizeof (mt_ReplyMessage) - mt_Message::HOSTNAME_LENGTH)
	{
	  counters.badLengths++;
	  mt_mapper_w (("bad length"));
	  network->freePacket (p);
	  return 0;
	}
	rm->swap ();
	insist (rm->phase == *phase);

	mt_Address a (rm->address);
	
	mt_mapper_c (("got a (reset) reply from %s (address %s)", rm->getHostname (), a.toString ()));

	if (rm->gmId)
	{
	  mt_mapper_c (("but it wasn't really reset."));
	  return 0;
	}
	
	network->freePacket (p);
	host->setGmId (0);
	hostTable->setState (host->getAddress (), mt_HostTable::NOT_CONFIGURED);
	mt_mapper_w (("cleared conflict on  %s (address %s)", rm->getHostname (), a.toString ()));
	
	return 1;
      }
    }
  }
  exception: return 0;
}

void mt_Mapper::handleConfigureReply (mt_Message*m, int length)
{
  mt_Address a;
  
  insist (this);
  insist (m && length);
  insist (acks && graph);
  insist (acksPerHost > 0);
  
  mt_ConfigureReplyMessage*n;
  n = (mt_ConfigureReplyMessage*)m;

  mt_mapper_c (("handling a configure reply"));
  
  if (length < (int) sizeof (mt_ConfigureReplyMessage))
  {
    counters.badLengths++;
    return;
  }
  n->swap ();
  if (n->phase != this->phase)
  {
    mt_mapper_w (("late message"));
    
    counters.badPhases++;
    increaseTimeouts (n->subtype, n->phase);
    return;
  }

  int host;
  host = n->port;
  
  a.fromBytes (n->address);
  if (a.isZero () ||
      host < 0 || host > graph->getNumHosts () ||
      a.compare (graph->getHost (host)->getAddress ()) ||
      n->hostSection < 0 ||
      n->hostSection >= calculator->getNumSections (host) ||
      n->hostSection + calculator->getFirstSection (host) > numAcks)
  {
    mt_mapper_w (("bad contents"));
    counters.badContents++;
    return;
  }
  mt_mapper_c (("got a reply from address %s, hostSection %d", a.toString (), n->hostSection));

  int start;
  start = calculator->getFirstSection (host);

  acks [start + n->hostSection] = 1;
  times [start + n->hostSection] = getNetwork ()->getCurrentTime () - times [start + n->hostSection];

  insist (acks [n->hostSection + start] == 1);
  
  graph->getHost (host)->setMapVersion (mapVersion);

  exception: return;
}

int mt_Mapper::configureRemote (char*acks, int phase, int host, int hostSection, int mapVersion, mt_Address*address,
				mt_Graph*graph, mt_ConfigureMessage*message)
{

  int start, stop, size;
  
  insist (this && acks && calculator && address && message);
  insist (host >= 0 && host < calculator->getNumHosts ());
  insist (hostSection >= 0 && hostSection < calculator->getNumSections (host));
  insist (graph);
  
  mt_Node*h;
  mt_Address*a;
  h = graph->getHost (host);
  insist (h);

  a = h->getAddress ();
  insist (a);
  
  calculator->getSection (host, hostSection, &start, &stop);

  insist (stop > start && stop - start <= calculator->getNumHosts ());

  {    
    mt_Route route (*h->getRoute (0));
    route.invert ();
    
    int r;
    r =  message->setHeader (phase, host, hostSection, a, address, hostTable->getGmId (a),
			     calculator->getNumHosts (), mapVersion, stop - start, &route);
    insist (r);
    
    for (int i = start; i < stop; i++)
    {
      int numRoutes = calculator->getNumRoutes (host, i);
      insist (numRoutes >= 0);
    
      mt_Node*n = calculator->getHost (i);
      insist (n);
      a = n->getAddress ();
    
      r = message->append (a,
			   hostTable->getGmId (a),
			   hostTable->getHostType (a),
			   0,
			   hostTable->getHostname (a),
			   numRoutes);
      insist (r);
      
      for (int j = 0; j < numRoutes; j++)
      {
	mt_Route rr;
	calculator->getRoute (host, i, j, &rr);
	r = message->append (&rr);
	insist (r);
      }
    }

    size = message->getSize ();
    message->swap ();

    mt_mapper_c (("sending configure message with routes from %d up to %d (%d bytes) to host %s with mapVersion %d",
		  start, stop, size, h->getName (), mapVersion));
    
    counters.sends [mt_Message::CONFIGURE]++;
  
    return send (h->getRoute (0), (char*) message, size);
  }
  
  exception: return 0;
}

int mt_Mapper::clearConflicts (mt_Graph*graph, int mapper, int*phase, int*numClears)
{
  int numConflicts = 0;

  
  insist (this);
  insist (graph);
  insist (phase);
  insist (numClears);
  
  *numClears = 0;
  
  
  int numHosts;
  numHosts = graph->getNumHosts ();
  insist (numHosts >= 0);

  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*n = graph->getHost (i);
    insist (n);

    if (i != mapper && hostTable->getState (n->getAddress ()) == mt_HostTable::INCONSISTENT)
    {
      numConflicts++;
      (*numClears) += clearConflict (n, phase, options.level);
    }
  }
  return numConflicts;
  exception: return 0;
}

int mt_Mapper::configure (mt_Graph*graph, int mapVersion, int mapper, int*phase)
{
  if (options.neverConfigure)
    return 1;

  this->mapVersion = mapVersion;
  
  mt_mapper_w (("configuring"));
  
  mt_mapper_c (("configuring hosts requiring obsolete messages"));

  old_configure (graph, mapVersion, mapper, phase);

  mt_mapper_c (("configuring hosts with regular configure messages"));
  
  times = 0;
  acks = 0;
  mt_ConfigureMessage*message = 0;
  int numConflicts = 0;
  int numClears = 0;
  int localConfigured = 0;
  int more = 0;
  
  insist (this);
  insist (graph);
  insist (calculator);
  insist (phase);
  insist (options.maxConfigureRoutes > 0);
  insist (options.maxConfigureRoutes <= mt_Message::MAX_CONFIGURE_ROUTES);  
  
  this->graph = graph;
  
  if (*options.bootHost)
  {
    bootHost = graph->getHostIndex (options.bootHost);
    insistp (bootHost >= 0 && bootHost < graph->getNumHosts(),
	     ("mt_Mapper:configure (bootHost option): host %s not found in graph", options.bootHost));
    mt_mapper_c (("configuring with bootHost %s", options.bootHost));
  }

  {
    mt_Network*network = getNetwork ();
    insist (network);
    insist (network->getNumReceiveBuffers () >= 2);
	    
    int numHosts = graph->getNumHosts ();
    insist (numHosts > 0);

    mt_mapper_c (("configuring %d hosts", numHosts));
    
    mt_Node*root = graph->getHost (mapper);
    insist (root && root->isHost ());
    mt_Node*first = root->getNode (0);

    if (!calculator->initialize (graph, first && first->isSwitch () ? first : 0))
      return 0;
    
    if (options.testDeadlock)
    {
      mt_mapper_c (("testing routes for deadlocks"));
      
      de_Deadlock d;
      if (!d.clone (calculator))
	return 0;
      
      if (d.deadlockable (calculator))
      {
	mt_mapper_w (("possible deadlock detected on this cycle:"));
	if (options.warnings)
	  d.print ();
	mt_mapper_w (("try disabling -shortest-path."));
	mt_mapper_w (("anyway, I'm not going to configure the network with these scary routes."));
	return 0;
      }
      mt_mapper_c (("these routes cannot deadlock."));
    }
    
    numAcks = calculator->computeSections (hostTable, mt_ConfigureMessage::SIZE_PER_HOST, mt_ConfigureMessage::SIZE_PER_ROUTE, options.maxConfigureByteStream);
    
    mt_mapper_c (("numAcks is %d", numAcks));

    numConflicts = clearConflicts (graph, mapper, phase, &numClears);

    if (numConflicts && numConflicts == numClears)
    {
      mt_mapper_w (("cleared some conflicts. reassigning GM ids"));
      isConfigured (graph, mapVersion);
      numConflicts = 0;
    }

    acks = (char*) calloc (numAcks, sizeof (char));
    insistp (acks, ("mt_Mapper::configure: calloc failed\n"));    
        
    times = (mt_Int64*) calloc (numAcks, sizeof (mt_Int64));
    insistp (times, ("mt_Mapper::configure: alloc failed"));

    if (!(localConfigured = configureLocal (graph, mapper, calculator)))
    {
      mt_mapper_c (("didn't configure myself"));
    }
    
    message = new mt_ConfigureMessage ();
    insistp (message, (("mt_Mapper::configure: couldn't allocate mt_ConfigureMessage\n")));
    
    for (int i = 0; i < numHosts; i++)
    {
      mt_Node*n = graph->getHost (i);
      insist (n);
      int state, option;
      state = hostTable->getState (n->getAddress ());
      option = hostTable->getOption (n->getAddress ());
      
      if (!(option & mt_Message::PACKED_ROUTES_OPTION) ||
	  state == mt_HostTable::CONFIGURED ||
	  state == mt_HostTable::INCONSISTENT ||
	  options.neverConfigureOthers && n != root ||
	  (!options.configureMissingHosts && n->getMissing ()))
      {
	mt_mapper_c (("I'm not going to configure %s", graph->getHost (i)->getName ()));
	insist (calculator->getFirstSection (i) + calculator->getNumSections (i) <= numAcks);
	memset (acks + calculator->getFirstSection (i), 1, calculator->getNumSections (i));
      }
    }
    
    setHandler (mt_Message::CONFIGURE_REPLY, mt_fp (mt_Mapper::handleConfigureReply));
    
    for (int i = 0;  (i < options.configureTries) && (!i || more); i++)
    {
      more = 0;
      this->phase = ++(*phase);
      
      int host, hostSection, pause, numSent, maxNumSections;
      host = hostSection = numSent = pause = maxNumSections = 0;
      
      if (i && options.growConfigureTimeouts)
      {
	int t = options.configureMicroseconds * 2;
	if (t > options.maxTimeout) t = options.maxTimeout;
	
	mt_mapper_c (("increasing configure timeout to %d microseconds", t));
	options.setTimeouts (options.scoutMicroseconds, options.probeMicroseconds, t);	
      }
      
      for (;;)
      {
	insist (host >= 0 && host < numHosts);
	int numSections = calculator->getNumSections (host);

	insist (hostSection < numSections || options.interleaveConfigures);
	
	if (maxNumSections < numSections)
	  maxNumSections = numSections;
 
	if (hostSection < numSections)
	{
	  int ack, correction;
	  ack = hostSection + calculator->getFirstSection (host);
	  insist (ack >= 0 && ack < numAcks);
	  insist (acks [ack] == 0 || acks [ack] == 1);
	  correction = 0;
	  
	  if (!acks [ack])
	  {
	    mt_mapper_c (("configuring for ack %d", ack));
	    
	    if (options.resendConfiguresInOrder)
	    {
	      int numMissing = 0;
	      for (numMissing = 0; numMissing < hostSection && !acks [ack - 1 - numMissing]; numMissing++);

	      if (numMissing)
	      {
		mt_mapper_c (("decrementing hostSection by %d to resend in order.", numMissing));
		correction = -numMissing;
	      }
	    }  
	    more = 1;  
	    
	    if (options.sendManyConfigures &&
		(pause || hostSection == 1 && (!options.interleaveConfigures || !host)))
	    {
	      mt_mapper_c (("pausing before new round of messages"));

	      network->setTimer (this, options.configureMicroseconds);
	      pause = 0;
	      wait ();
	    }

	    times [ack] = network->getCurrentTime ();
	    
	    if (!configureRemote (acks, *phase, host, hostSection + correction, mapVersion,
				  root->getAddress (), graph, message))
	    {
	      mt_mapper_c (("configure remote failed. breaking out of inner configure loop"));
	      break;
	    }
	    network->setTimer (this, options.sendManyConfigures && (++numSent % (network->getNumReceiveBuffers () - 1)) ?
			       options.probeMicroseconds : options.configureMicroseconds);
	    wait ();
	  }
	}
	if (options.interleaveConfigures)
	{
	  if (host == numHosts - 1 && hostSection == maxNumSections - 1)
	  {
	    pause = 1;
	    break;
	  }
	  
	  if (++host == numHosts)
	  {
	    host = 0;
	    hostSection++;
	    pause = 1;
	  }
 	}
	else
	{
	  pause = 1;
	  if (++hostSection == numSections)
	  {
	    if (++host == numHosts)
	      break;
	    hostSection = 0;
	  }
	}
      }
      network->setTimer (this, options.configureMicroseconds);
      wait ();
    }    

    clearHandler (mt_Message::CONFIGURE_REPLY);
    int configured;
    configured = updateHostTable (acks, calculator);
    //insist (!configured && more || configured && !more);
 
    if (*options.routesFile)
    {
      mt_mapper_c (("writing routes to %s", options.routesFile));
      int r = calculator->writeRoutes (options.routesFile);
      mt_mapper_c (("routes %s", r ? "written" : "not written -- there was an error"));
    }
    
    if (options.measureConfigureLatency)
    {
      mt_mapper_c (("looking at latencies"));
      
      int smallest = -1;
      int biggest = -1;

      for (int i = 0; i < numAcks; i++)
      {
	if (acks [i] && times [i])
	{
	  int t = (int) times [i];
	  
	  if (biggest < 0 || t > times [biggest])
	    biggest = i;

	  if (smallest < 0 || t < times [smallest])
	    smallest = i;
	}
      }
      if (biggest != -1)
      {
	mt_mapper_c (("configure ack latency ranged from %d (ack %d) to %d (ack %d)", (int) times [smallest], smallest, (int) times [biggest], biggest));
	mt_mapper_c (("setting configure timeout to %d", (int) times [biggest] * 2));
	options.setTimeouts (options.scoutMicroseconds, options.probeMicroseconds, (int) times [biggest] * 2);
      }
    }
    
    calculator->cleanup ();
    
    delete [] acks;
    delete message;
    return localConfigured && configured && !numConflicts;
  }
  
  exception:
  if (acks) free (acks);
  if (times) free (times);
  if (message) delete message;
  if (configures) free (configures);

  return 0;
}

void mt_Mapper::printVerifyProblem (mt_Graph*graph, mt_Node*root, mt_Route*route, int isHost, char*hostname)
{
  insist (this);
  insist (graph && root && route);
  insist (!isHost || hostname);
  
  char buffer [mt_Route::MAX_ROUTE * 10];
  
  mt_Node*n;
  n = root->getNode (route);

  if (n && n->isHost () && !isHost)
    printFormat ("new switch at %s, replacing host %s",
		 route->toString (buffer), n->getName ());
  else if (n && !n->isHost () && isHost)
    printFormat ("new host at %s (%s), replacing switch",
		 hostname, route->toString (buffer));
  else if (n && n->isHost () && isHost)
    printFormat ("changed or outdated host %s at %s",
		 hostname, route->toString (buffer));
  else if (!n && isHost)
    printFormat ("new host %s at %s", hostname,  route->toString (buffer));
  else if (!n && !isHost)
    printFormat ("new switch at %s",  route->toString (buffer));
  else
    printFormat ("unknown verify failure at %s", route->toString (buffer));
  
  exception: return;
}

int mt_Mapper::formatHostname (char*hostname, int unit)
{
  insist (this);
  insist (hostname);
  insist (unit >= 0);

  char*s;
  
  if (options.shortHostnames)
  {
    s = strrchr (hostname, ':');
    if (s && !unit)
    {
      int d = 0;
      if (sscanf (s + 1, "%x", &d) != 1)
	d = 0;
      unit = d;
    }
    
    s = strchr (hostname, '.');
    if (s) *s = 0;
  }
  
  if (unit > 0)
  { 
    s = strrchr (hostname, ':');
    if (s) *s = 0;
    
    char newhostname[mt_Network::HOSTNAME_LENGTH + 4];
    sprintf (newhostname,"%s:%x", hostname, unit);
    strncpy (hostname, newhostname, mt_Network::HOSTNAME_LENGTH);
  }
 
  return 1;
  exception: return 0;
}

int mt_Mapper::willWait ()
{
  return 1;
}

int mt_Mapper::addCloudHost (mt_Graph*graph, mt_Node*cloud, mt_CloudHost*ch, int hostType, int p)
{
  mt_Address a;

  insist (this);
  insist (graph && cloud && ch);
  insist (p > 0 && p < cloud->getMaxNodes ());
  
  a.fromBytes (ch->address);
  insist (!a.isZero ());
  
  if (!hostTable->resolve (0, hostType, &a))
  {
    mt_mapper_c (("host table problem with cloud host %s", a.toString ()));
    counters.gmIdConflicts++;
    return 0;
  }

  mt_Node*h;
  h = graph->getHost (&a);
  if (h)
  {
    if (h->getNode (0) != cloud)
    {
      mt_mapper_c (("inconsistent already seen cloud node"));
      return 0;
    }
  }    
     
  hostTable->setHostname (&a, a.toString ());
  hostTable->setState (&a, mt_HostTable::CONFIGURED);
  
  mt_mapper_c (("new cloud host %s type %d", a.toString (), hostType));

  h = newNode (mt_Node::HOST, a.toString (), "-");
  insistp (h, ("mt_Mapper::addCloudHost:couldn't allocate new host"));

  h->setAddress (&a);
  h->setGmId (hostTable->getGmId (&a));
  h->setMapVersion (cloud->getMapVersion ());
  h->setHostType (hostType);
  
  mt_mapper_c (("setting %s mapVersion to %d", h->getName (), h->getMapVersion ())); 

  mt_Route*r;
  r = cloud->getRoute (0);
  
  h->setRoute (0, *r);
  h->setControl (ch->control);  

  mt_mapper_c (("connecting host to cloud port %d", p));

  h->connect (0, p, cloud);
  cloud->connect (p, 0, h);

  graph->add (h);
  return 1;
  
  exception: return 0;
}


int mt_Mapper::addCloudHosts (mt_Graph*graph, mt_Node*cloud, mt_CloudReplyMessage*m)
{
  insist (this);
  insist (graph && cloud && m);
  
  int numHosts;
  numHosts = m->numHosts;
  insist (numHosts >= 0 && numHosts == cloud->getMaxNodes () - 1);
  
  if (numHosts > mt_Message::MAX_CLOUD_HOSTS)
    numHosts = mt_Message::MAX_CLOUD_HOSTS;
  
  int hostType;
  hostType = cloud->getHostType ();
  
  for (int i = 0; i < numHosts; i++)
    if (!addCloudHost (graph, cloud, &m->hosts [i], hostType, i + 1))
      return 0;
    
  return 1;
  exception: return 0;
}

int mt_Mapper::addCloudHosts (mt_Graph*graph, mt_Node*cloud, mt_CloudQueryReplyMessage*m)
{
  insist (this);
  insist (graph && cloud && m);
  
  int numHosts;
  numHosts = m->numHosts;
  insist (numHosts >= 0 && numHosts < cloud->getMaxNodes ());
  insist (numHosts <= mt_Message::MAX_CLOUD_HOSTS);
  
  int hostType;
  hostType = 0;

  for (int i = 0; i < numHosts; i++)
    if (!addCloudHost (graph, cloud, &m->hosts [i], hostType, m->first + i + 1))
      return 0;
    
  return 1;
  exception: return 0;
}

void mt_Mapper::handleCloudQueryReply (mt_Message*m, int length)
{
  mt_Address a;

  insist (this);
  insist (m && length);
  insist (graph);

  if (length < (int) sizeof (mt_Message))
  {
    mt_mapper_w (("bad length"));
    counters.badLengths++;
    return;
  }

  mt_CloudQueryReplyMessage*rm;
  rm = (mt_CloudQueryReplyMessage*) m;
  rm->swap ();

  a.fromBytes (rm->address);
  
  mt_mapper_c (("got a cloud query reply from %s", a.toString ()));

  if (rm->phase != this->phase)
  {
    mt_mapper_w (("late message"));
    counters.badPhases++;
    increaseTimeouts (rm->subtype, rm->phase);

    return;
  }

  int cloudIndex;
  cloudIndex = m->port;
  insist (cloudIndex >= 0 && cloudIndex < graph->getNumNodes (mt_Node::CLOUD));

  mt_Node*h;
  h = graph->getNode (mt_Node::CLOUD, cloudIndex);
  if (a.compare (h->getAddress ()))
  {
    mt_mapper_c (("mismatched address in cloud query reply"));
    mt_mapper_c (("%s versus", a.toString ()));
    mt_mapper_c (("%s", h->getAddress()->toString ()));
    
    return;
  }
  
  addCloudHosts (graph, h, rm);  
  exception: return;
}


int mt_Mapper::queryClouds (mt_Graph*graph, mt_Network*network, int*phase)
{
  insist (this);
  insist (graph && network);
  insist (phase);

  this->phase = ++(*phase);
  
  int numClouds;
  numClouds = graph->getNumNodes (mt_Node::CLOUD);
  insist (numClouds >= 0 && numClouds <= graph->getNumNodes (mt_Node::NODE));
  
  mt_mapper_c (("%d %s to query.", numClouds, numClouds == 1 ? "cloud" : "clouds"));
  mt_CloudQueryMessage*m;
  m = new mt_CloudQueryMessage ();
  insistp (m, ("mt_Mapper::queryClouds: alloc failed"));

  this->graph = graph;
  setHandler (mt_Message::CLOUD_QUERY_REPLY, mt_fp (mt_Mapper::handleCloudQueryReply));

  int t;
  for (t = 0; t < options.scoutTries; t++)
  {
    int sent;
    sent = 0;
    
    for (int i = 0; i < numClouds; i++)
    {
      mt_Node*c = graph->getNode (mt_Node::CLOUD, i);
      insist (c);

      mt_Route a (*c->getRoute (0));
      mt_Route r (a);
      r.invert ();

      int first = c->getNumNodes ();
      insist (first > 0);

      int maxNodes;
      maxNodes = c->getMaxNodes ();
      
      if (c->getMaxNodes () - c->getNumNodes ())
      {
	for (int j = 0; j < maxNodes;)
	{
	  if (c->getNode (j))
	  {
	    j++;
	    continue;
	  }
	  
	  m->subtype = mt_Message::CLOUD_QUERY;
	  m->type = mt_Message::GM_TYPE;
	  m->port = i;
	  m->phase = *phase;
	  m->routeLength = r.getLength ();
	  r.copyOut (m->route);
	  m->first = j - 1;
	  
	  insist (m->first >= 0);
	  
	  int remainder;
	  remainder = maxNodes - j;
	
	  m->numHosts = remainder < mt_Message::MAX_CLOUD_HOSTS ? remainder : mt_Message::MAX_CLOUD_HOSTS;

	  j += m->numHosts;
	  
	  mt_mapper_c (("querying cloud %s for hosts %d through %d", c->getName (), m->first, m->first + m->numHosts - 1));
	  
	  m->swap ();
	
	  if (!network->send (this, &a, (char*) m, sizeof (*m)))
	  {
	    delete m;
	    clearHandler (mt_Message::CLOUD_QUERY_REPLY);
	    return 0;
	  }
	  sent = 1;
	}

      }    
      if (sent)
      {
	network->setTimer (this, options.scoutMicroseconds);
	wait ();
      }
      else break;
    }
  }
  
  clearHandler (mt_Message::CLOUD_QUERY_REPLY);
  
  delete m;
  return 1;
  exception: return 0;
}

int mt_Mapper::addMissingHosts (mt_Graph*newGraph, mt_Graph*oldGraph)
{
  insist (this);
  insist (newGraph && oldGraph);
  
  int numOldHosts;
  numOldHosts = oldGraph->getNumHosts ();
  
  for (int i = 0; i < numOldHosts; i++)
  {
    mt_Node*oldHost = oldGraph->getHost (i);
    insist (oldHost);
    
    if (!newGraph->getHost (oldHost->getAddress ()))
    {
      mt_mapper_c (("host %s (address %s) is missing.", oldHost->getName (), oldHost->getAddress ()->toString ()));

      mt_Node*n = newGraph->getRoot ();
      int port = 0;
      mt_Route*r = oldHost->getRoute (0);
      insist (r);
      
      int length = r->getLength ();
      if (length)
      {
	mt_Route shorter (r->getHops (), length - 1);

	if (!n->follow (&shorter, &n, &port))
	{
	  mt_mapper_c (("missing link. can't get to old host %s using old route %s. it will stay missing.", oldHost->getName (), r->toString ()));
	  continue;
	}
	port += r->getLastHop ();
      }
      insist (n);
      
      mt_Node*o;
      
      if (port >= 0 && (o = n->getNode (port)))
      {
	mt_mapper_c (("new node %s at old node %s's position. it will stay missing.", o->getName (), oldHost->getName ()));
	continue;
      }
      
      insist (!port || n->isSwitch ());
      insist (port < 0 || !n->getNode (port));
      
      mt_mapper_c (("adding old host %s to new graph at port %d of node %s", oldHost->getName (), port, n->getName ()));
      
      mt_Node*h = newGraph->newNode (oldHost->getNodeType (), oldHost->getName (), "-");
      insist (h);
      
      h->setMissing (1);
      h->setAddress (oldHost->getAddress ());
      h->setGmId (oldHost->getGmId ());
      h->setMapVersion (oldHost->getMapVersion ());
      h->setRoute (0, *r);

      n->connectMissingHost (port, h);
      newGraph->add (h);
    }
  }
  
  exception: return 0;
}

int mt_Mapper::increaseTimeouts (int type, int phase)
{
  insist (this);
  
  mt_mapper_c (("trying to increase timeout for type %d", type));
  
  if (phase != lastIncreaseTimeoutsPhase && options.increaseTimeouts (type))
  {
    mt_mapper_c (("increased timeout"));
  }
  else
  {
    mt_mapper_c (("didn't increase timeout"));
  }
  mt_mapper_c (("probe timeout is %d microseconds, scout %d.", options.probeMicroseconds, options.scoutMicroseconds));
  
  lastIncreaseTimeoutsPhase = phase;
  exception: return 0;
}

int mt_Mapper::decreaseTimeouts ()
{
  insist (this);
 
  if (options.decreaseTimeouts ())
  {
    mt_mapper_c (("decreased one or more timeouts"));
  }
  else
  {
    mt_mapper_c (("did not decrease timeouts"));
  }
  mt_mapper_c (("probe timeout is %d microseconds, scout %d.", options.probeMicroseconds, options.scoutMicroseconds));

  exception: return 0;
}

int mt_Mapper::updateHostTable (char*acks, mt_Calculator*calculator)
{
  insist (this);
  insist (acks && calculator);

  int numHosts;
  numHosts = calculator->getNumHosts ();
  int configured;
  configured = 1;

  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*n = calculator->getHost (i);
    insist (n);

    int numSections = calculator->getNumSections (i);
    int j;
    for (j = 0; j < numSections && acks [calculator->getFirstSection (i) + j]; j++);
    
    if (j == numSections)
    {
      mt_mapper_c (("marking %s as configured", n->getName ()));
      hostTable->setState (n->getAddress (), mt_HostTable::CONFIGURED);
    }
    else
    {
      mt_mapper_w (("marking %s as not configured", n->getName ()));
      hostTable->setState (n->getAddress (), mt_HostTable::NOT_CONFIGURED);
      configured = 0;
    }
  }
  
  return configured;
  exception: return 0;
}



int mt_Mapper::getHostLatency (mt_Route*r, mt_Address*address, int level, int tries, int*phase)
{
  insist (this);
  insist (r && tries > 0);
  insist (phase);
  
  mt_Network*network;
  network = getNetwork ();
  insist (network);

  mt_Int64 start;
  start = network->getCurrentTime ();
  
  for (int i = 0; i < tries; i++)
  {
    mt_Route b (*r);
    b.invert ();
  
    mt_ScoutMessage m (0, ++(*phase), &b, address, level);
    int size = m.getSize ();
    m.swap ();

    if (!send (r, (char*) &m, size))
      return 0;    
    
    counters.sends [mt_Message::SCOUT]++;
    
    network->setTimer (this, options.MAX_MICROSECONDS);
    
    char*p;
    p = 0;
    int length;
    
    if (!wait (mt_Network::RECEIVE, mt_Message::REPLY, *phase, &p, &length))
      return 0;

    insist (p);
    
    network->freePacket (p);
    network->clearTimer (this);    
  }

  return (int) (network->getCurrentTime () - start) / tries;
  
  exception: return 0;
}

int mt_Mapper::getSwitchLatency (mt_Route*r, int tries, int*phase)
{
  insist (this);
  insist (r && tries > 0 && phase);
  
  mt_Network*network;
  network = getNetwork ();
  insist (network);

  mt_Int64 start;
  start = network->getCurrentTime ();

  for (int i = 0; i < tries; i++)
  {
    mt_Route r2 (*r);
    r2.invert ();
    mt_Route r3 (r, 0, &r2);

    mt_ProbeMessage m (0, ++(*phase));
    m.swap ();

    if (!send (&r3, (char*) &m, sizeof (mt_VerifyMessage)))
      return 0;
    
    counters.sends [mt_Message::PROBE]++;
    network->setTimer (this, options.MAX_MICROSECONDS);

    char*p;
    p = 0;
    int length;
    
    if (!wait (mt_Network::RECEIVE, mt_Message::PROBE, *phase, &p, &length))
      return 0;
    
    network->clearTimer (this);
    insist (p);
    network->freePacket (p);
  }

  return (int) (network->getCurrentTime () - start) / tries;
    
  exception: return 0;
}
  







/*obsolete functions to support old-style config messages*/

void mt_Mapper::old_handleConfigureReply (mt_Message*m, int length)
{
  mt_Address a;
  insist (this);
  insist (m && length);
  insist (acks && graph);
  insist (acksPerHost > 0);
  
  mt_ReplyMessage*n;
  n = (mt_ReplyMessage*)m;

  mt_mapper_c (("handling an obsolete configure reply"));
  
  if (length < (int) sizeof (mt_ReplyMessage) - mt_Message::HOSTNAME_LENGTH)
  {
    mt_mapper_w (("bad length"));
    counters.badLengths++;
    return;
  }
  n->swap ();
  if (n->phase != this->phase)
  {
    mt_mapper_w (("late message"));
    counters.badPhases++;
    increaseTimeouts (n->subtype, n->phase);
    return;
  }
  
  a.fromBytes (n->address);
  if (a.isZero () || n->port < 0 || n->port >= numAcks ||
      a.compare (graph->getHost (n->port / acksPerHost)->getAddress ()))
  {
    mt_mapper_w (("bad contents"));
    counters.badContents++;
    return;
  }
  mt_mapper_c (("got a reply from address %s, ack %d", a.toString (), n->port));

  a.fromBytes (n->mapperAddress);
  mt_mapper_c (("mapper address was %s, mapVersion was %d", a.toString (), n->mapVersion));
  mt_mapper_c (("previous mapVersion was %d",
		graph->getHost (n->port / acksPerHost)->getMapVersion()));
  
  int host;
  host = n->port / acksPerHost;
  int ack;
  ack = n->port % acksPerHost;
  
  mt_mapper_c (("recording config acks for %d for host %d", ack, host));
  
  for (int i = ack; i <= ack; i++)
    acks [host * acksPerHost + i] = 1;
  
  insist (acks [n->port] == 1);
  
  graph->getHost (host)->setMapVersion (mapVersion);
  
  exception: return;
}
int mt_Mapper::old_setConfigureMessage (mt_Graph*graph, mt_Address*mapperAddress, mt_Calculator*calculator,
					mt_OldConfigureMessage*message, int ack, int serial,
					int from, int start, int stop)
{
  insist (this);
  insist (hostTable);
  insist (graph);
  insist (calculator);
  insist (message);
  insist (from > 0 && start < stop);
  insist (serial >= 0);
  insist (mapperAddress && !mapperAddress->isZero ());
  
  {
    int numItems = 0;

    mt_Node*n = graph->getHost (from);
    insist (n);
    
    mt_Address*a = n->getAddress ();
    insist (a);

    mt_mapper_c (("forming configure message for %s (%s) with %d entries", a->toString (), hostTable->getHostname (a), 
		  stop - start));
    
    int gmId = hostTable->getGmId (a);

    insist (gmId > 0);
    
    mt_Route r (*n->getRoute (0));
    r.invert ();

    message->setGmId (gmId);
    message->setAddress (a);
    message->setMapperAddress (mapperAddress);
    message->setRoute (&r);
    message->setPart (ack);
    message->setSerial (serial);
    message->setNumHosts (graph->getNumHosts ());
 
    if (bootHost != -1 && from != bootHost)
    {
      if (bootHost < start || bootHost >= stop)
	return 0;
      
      start = bootHost;
      stop = bootHost + 1;
      message->setNumHosts (1);
    }

    mt_mapper_v (("gmId %d address %s serial %d part %d", gmId, a->toString (), serial, ack));
    mt_mapper_v (("mapper address %s", mapperAddress->toString ()));

    for (int i = start; i < stop; i++)
    {
      mt_Node*n = graph->getHost (i);
      insist (n);

      mt_Address*a = n->getAddress ();
      insist (a);

      int gmId = hostTable->getGmId (a);
      
      if (gmId <= 0)
      {
	mt_mapper_w (("skipping entry because of gm id inconsistency for address %s", a->toString ()));
	continue;
      }
      
      int hostType = hostTable->getHostType (a);
      mt_Route r;

      if (!calculator->getRoute (from, i, 0, &r))
	return -1;
      
      mt_mapper_v (("gmId %d hostType : %d", gmId, hostType));
    
      message->set (numItems, a, gmId, hostType, &r, hostTable->getHostname (a));
      mt_mapper_v (("route %d is %s", gmId, r.toString ()));
      
      numItems++;
    }

    mt_mapper_c (("configure message has %d entries with routes from index %d to indicies %d - %d",
		  numItems, from, start, stop));
    
    message->setNumItems (numItems);

    return numItems;
  }
  
  exception: return -1;
}

int mt_Mapper::old_configureRemote (int ack, char*acks, int acksPerHost, int mapVersion, mt_Address*mapperAddress, mt_Graph*graph,
				    mt_Calculator*calculator, mt_OldConfigureMessage*message)
{
  insist (this);
  insist (graph);
  insist (calculator);
  insist (acks);
  insist (ack > 0 && acksPerHost > 0);
  insist (mapperAddress && !mapperAddress->isZero ());
  
  {
    mt_Network*network = getNetwork ();
    insist (network);

    int numHosts = graph->getNumHosts ();
    int from = ack / acksPerHost;
    insist (from >= 0 && from < numHosts);

    int pos = ack % acksPerHost;
    int start = pos * options.maxConfigureRoutes;
    int stop = start + options.maxConfigureRoutes;
    if (stop > numHosts)
    {
      insist (pos == acksPerHost - 1);
      stop = numHosts;
    }

    int numItems = old_setConfigureMessage (graph, mapperAddress, calculator, message, ack, pos, from, start, stop);
    insist (numItems >= 0 && numItems <= stop - start);

    mt_Node*h = graph->getHost (from);
    insist (h);

    int state = hostTable->getState (h->getAddress ());
    insist (state);
    
    message->setMapVersion (/*state == mt_HostTable::FIRST_SEEN ? 0 : */mapVersion); 
    if (!numItems)
      acks [ack] = 1;
    else
    {
      mt_mapper_c (("sending obsolete configure message with %d items to to host %s with mapVersion %d",
		    numItems, h->getName (), mapVersion));
      counters.sends [mt_Message::OLD_CONFIGURE]++;
      if (send (h->getRoute (0), (char*) message, message->sizeOf (numItems)))
      {
	configures [ack / acksPerHost]++;
	return 1;
      }
      return 0;
    }
    
    return 1;
  }
  
  exception: return 0;
}

int mt_Mapper::old_startAcks (mt_Graph*graph, int mapper, char*acks, int acksPerHost)
{
  insist (this);
  insist (hostTable);
  insist (graph);
  insist (acks);
  insist (acksPerHost > 0);
  
  {
    int numHosts = graph->getNumHosts ();
    mt_Node*root = graph->getHost (mapper);
    insist (numHosts > 0);
    insist (root);
    
    memset (acks, 0, acksPerHost * numHosts);

    for (int i = 0; i < numHosts; i++)
    {
      mt_Node*n = graph->getHost (i);
      mt_Address*a = n->getAddress (); 
      insist (n);
      int state, option;
      state = hostTable->getState (a);
      option = hostTable->getOption (a);

      if ((option & mt_Message::PACKED_ROUTES_OPTION) ||
	  state == mt_HostTable::CONFIGURED ||
	  state == mt_HostTable::INCONSISTENT ||
	  options.neverConfigureOthers && n != root ||
	  (!options.configureMissingHosts && n->getMissing ()))
      {
	mt_mapper_c (("I'm not going to configure %s with obsolete messages", graph->getHost (i)->getName ()));
	memset (acks + i * acksPerHost, 1, acksPerHost);
      }
    }
    return 1;
  }
  exception: return 0;
}

int mt_Mapper::old_updateHostTable (char*acks, int acksPerHost, mt_Graph*graph)
{
  insist (this);
  insist (acks && graph);
  insist (acksPerHost >= 1);

  int numHosts;
  numHosts = graph->getNumHosts ();
  int configured;
  configured = 1;

  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*n = graph->getHost (i);
    insist (n);

    int j;
    for (j = acksPerHost * i; j < acksPerHost * (i + 1) && acks [j]; j++);

    if (hostTable->getOption (n->getAddress ()) & mt_Message::PACKED_ROUTES_OPTION)
    {
      mt_mapper_c (("not marking %s either way", n->getName ()));
      configured = 0;
    }
    else if (j == acksPerHost * (i + 1))
    {
      mt_mapper_c (("marking %s as configured", n->getName ()));
      hostTable->setState (n->getAddress (), mt_HostTable::CONFIGURED);
    }
    else
    {
      mt_mapper_w (("marking %s as not configured", n->getName ()));
      hostTable->setState (n->getAddress (), mt_HostTable::NOT_CONFIGURED);
      configured = 0;
    }
  }
  
  return configured;
  exception: return 0;
}

int mt_Mapper::old_configure (mt_Graph*graph, int mapVersion, int mapper, int*phase)
{
  mt_OldConfigureMessage*message = 0;
  acks = 0;
  configures = 0;
  bootHost = -1;
  int localConfigured;
  
  insist (this);
  insist (graph);
  insist (calculator);
  insist (phase);
  insist (options.maxConfigureRoutes > 0);
  insist (options.maxConfigureRoutes <= mt_Message::MAX_CONFIGURE_ROUTES);  
  
  if (*options.bootHost)
  {
    bootHost = graph->getHostIndex (options.bootHost);
    insistp (bootHost >= 0 && bootHost < graph->getNumHosts(),
	     ("mt_Mapper:configure (bootHost option): host %s not found in graph", options.bootHost));
    mt_mapper_c (("configuring with bootHost %s", options.bootHost));
  }

  {
    mt_Network*network = getNetwork ();
    insist (network);
  
    int numHosts = graph->getNumHosts ();
    insist (numHosts > 0);

    mt_mapper_c (("configuring %d hosts", numHosts));
    
    mt_Node*root = graph->getHost (mapper);
    insist (root && root->isHost ());
    mt_Node*first = root->getNode (0);

    if (!calculator->initialize (graph, first && first->isSwitch () ? first : 0))
      return 0;
    
    if (!(localConfigured = configureLocal (graph, mapper, calculator)))
    {
      mt_mapper_c (("didn't configure myself"));
    }
    
    int max = options.maxConfigureRoutes;
    int acksPerHost;
    
    if (numHosts < max)
      acksPerHost = 1;
    else
      acksPerHost = numHosts / max + (numHosts % max ? 1 : 0);

    numAcks = acksPerHost * numHosts;

    mt_mapper_c (("num acks is %d and acksPerHost is %d", numAcks, acksPerHost));

    acks = new char [numAcks];
    insistp (acks, ("mt_Mapper::configure: alloc failed"));
    
    int dontcare;
    
    int numConflicts = clearConflicts (graph, mapper, phase, &dontcare);
    old_startAcks (graph, mapper, acks, acksPerHost);

    configures = new int [numHosts];
    insistp (configures, ("mt_Mapper::configure alloc failed"));
    
    memset (configures, 0, numHosts * sizeof (int));

    int ack = 0;
    int more = 0;
    int j = 0;
    
    message = new (numHosts < max? numHosts : max) mt_OldConfigureMessage ();
    insistp (message, ("mt_Mapper::configure: alloc failed"));

    this->acksPerHost = acksPerHost;
    this->graph = graph;
    setHandler (mt_Message::REPLY, mt_fp (mt_Mapper::old_handleConfigureReply));

    for (int i = 0;  (i < options.configureTries)
	   && (!i || more); i++)
    {
      more = 0;
      message->setPhase (++(*phase));
      this->phase = *phase;
      
      for (j = 0; j < numAcks; j++)
      {
	if (options.interleaveConfigures)
	{
	  int n = j * acksPerHost;
	  ack = n % numAcks + n / numAcks;
	}
	else ack = j;
	
	insist (acks [ack] == 0 || acks [ack] == 1);
	
	if (!acks [ack])
	{
	  mt_mapper_c (("configuring for ack %d", ack));
	  
	  if (options.resendConfiguresInOrder)
	  {
	    int section = ack % acksPerHost;
	    if (section && !acks [ack - 1])
	    {
	      mt_mapper_c (("decrementing ack index to resend in order."));
	      ack--;
	    }
	  }  
	  more = 1;
	  insist (ack / acksPerHost != mapper);
	  if (!old_configureRemote (ack, acks, acksPerHost, mapVersion, root->getAddress (), graph, calculator, message))
	  {
	    mt_mapper_c (("configure remote failed. breaking out of inner configure loop"));
	    break;
	  }
	  
	  network->setTimer (this, options.sendManyConfigures ?
			     options.probeMicroseconds : options.configureMicroseconds);
	  wait ();
	}
      }
      network->setTimer (this, options.configureMicroseconds);
      wait ();
    }

    network->setTimer (this, options.configureMicroseconds);
    wait ();    

    clearHandler (mt_Message::REPLY);
    
    old_updateHostTable (acks, acksPerHost, graph);
 
    if (*options.routesFile)
    {
      mt_mapper_c (("writing routes to %s", options.routesFile));
      int r = calculator->writeRoutes (options.routesFile);
      mt_mapper_c (("routes %s", r ? "written" : "not written -- there was an error"));
    }
    
    calculator->cleanup ();
    
    for (int i = 0; i < numHosts; i++)
    {
      mt_mapper_c (("%s was sent %d obsolete configure %s",
		    graph->getHost (i)->getName (),
		    configures [i], configures [i] == 1 ? "message" : "messages"));
    }
      
    delete [] acks;
    delete message;
    delete configures;
    return localConfigured && j == numAcks && !more && !numConflicts;
  }
  
  exception:
  if (acks) delete [] acks;
  if (message) delete message;
  if (configures) delete configures;

  return 0;
}
