/*
  mt_Gm.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <string.h>

#include "insist.h"
#include "gm_internal.h"

#include "mt_Gm.h"
#include "mt_NetworkQueue.h"
#include "mt_Switch.h"
#include "mt_Message.h"
#include "mt_Mapper.h"

#if GM_OS_OSF1 || GM_CPU_mips || GM_OS_SOLARIS || GM_OS_LINUX || GM_OS_FREEBSD
extern "C" int gethostname (char *, size_t);
#endif

#define mt_max_length_for_size(size) (size < 3? 0 : (1 << size) - 8)

static gm_port_t *gmPort;
static gm_alarm_t gmAlarm;

static const int NUM_SEND_TOKENS = GM_NUM_SEND_TOKENS;
static const int NUM_BIG_TOKENS = 2;
static const int NUM_SMALL_TOKENS = NUM_SEND_TOKENS - NUM_BIG_TOKENS;
static const int NUM_BIG_BUFFERS = 2*NUM_BIG_TOKENS;
static const int NUM_SMALL_BUFFERS = 2*NUM_SMALL_TOKENS;
static const int BIG_GM_SIZE = 12;
static const int BIG_SIZE  mt_max_length_for_size (BIG_GM_SIZE);
static const int SMALL_GM_SIZE = 11;
static const int SMALL_SIZE mt_max_length_for_size (SMALL_GM_SIZE);

mt_Gm::mt_Gm () : mt_Network (), 
  smallSends (NUM_SMALL_BUFFERS + 1),
  smallBuffers (NUM_SMALL_BUFFERS + 1),
  bigSends (NUM_BIG_BUFFERS + 1),
  bigBuffers (NUM_BIG_BUFFERS + 1)
{
  openCount = 0;
  numBadLengths = 0;
  sentDoneLength = 0;
}


mt_Gm::~mt_Gm ()
{
}
int mt_Gm::open (mt_Job*job, mt_Address*address, char*hostname, int*type, int unit, int numReceiveBuffers)
{
  insist (this);
  insist (job && address && hostname && type);
  this->job = job;

  insist (openCount >= 0);
  
  if (!numReceiveBuffers)
    numReceiveBuffers = GM_NUM_RECV_TOKENS;

  this->numReceiveBuffers = numReceiveBuffers;
  
  insist (numReceiveBuffers > 2 && numReceiveBuffers <= GM_NUM_RECV_TOKENS);
  
  if (!openCount)
  {
    gm_status_t r;
    r = _gm_mapper_open (&gmPort, unit, GM_API_VERSION_1_0);
    if (r != GM_SUCCESS)
    {
      gm_perror("gm_mapper_port open failed", r);
    }
    insistp (r == GM_SUCCESS, ("gm_mapper_port open failed on unit %d", unit));

    gm_initialize_alarm (&gmAlarm);
  }
  
  char unique[6];
  gm_get_unique_board_id (gmPort, unique);
  address->fromBytes (unique);

  *hostname = 0;
  if (gm_get_host_name (gmPort, hostname) != GM_SUCCESS || (!*hostname))
    gethostname (hostname, mt_Network::HOSTNAME_LENGTH);

  if (gm_get_node_type (gmPort, type) != GM_SUCCESS)
    return 0;
  
  if (!openCount)
  {
    int r;
    
    bigTokens = NUM_BIG_TOKENS;
    smallTokens = NUM_SMALL_TOKENS;

    smallStart = (char *) gm_dma_malloc (gmPort, SMALL_SIZE * NUM_SMALL_BUFFERS);
    smallEnd = (char *) smallStart + SMALL_SIZE * NUM_SMALL_BUFFERS;
    insistp (smallStart, (("gm_dma_malloc failed")));

    bigStart = (char *) gm_dma_malloc (gmPort, BIG_SIZE * NUM_BIG_BUFFERS);
    bigEnd = (char *) bigStart + BIG_SIZE * NUM_BIG_BUFFERS;
    insistp (bigStart, (("gm_dma_malloc failed")));

    char *p;
    int i;

    for (i = 0, p = smallStart; i < NUM_SMALL_BUFFERS; i++, p += SMALL_SIZE)
    {
      r = smallBuffers.put (p, SMALL_SIZE);
      insist (r);
    }

    for (i = 0, p = bigStart; i < NUM_BIG_BUFFERS; i++, p += BIG_SIZE)
    {
      r = bigBuffers.put (p, BIG_SIZE);
      insist (r);
    }

    for (i = 0; i < numReceiveBuffers; i++)
    {
      p = (char *)gm_dma_malloc (gmPort, SMALL_SIZE);
      insist (p);
      //memset (p, 0, SMALL_SIZE);
      _gm_provide_raw_receive_buffer (gmPort, p);
    }
  }
  
  openCount++;
  
  return 1;
  exception: return 0;
}

void mt_Gm::close (mt_Job*job)
{
  insist (this);
  insist (openCount >= 0); 
  insist (job == this->job);
  
  if (openCount == 1)
  {
    gm_close (gmPort);
    smallSends.empty ();
    smallBuffers.empty ();
    bigSends.empty ();
    bigBuffers.empty ();
  }
  openCount--;
  exception: return;
}

int mt_Gm::send (mt_Job*job, mt_Route*route, char*p, int length)
{
  insist (this);
  insist (job == this->job);
  insist (length >= 0);

  void*m;
  mt_NetworkQueue*q,*qs;
  int*c;
  int routeLength;
  routeLength= route->getLength ();
  
  if (GM_DMA_ROUNDUP (u32, length + routeLength) > (unsigned) SMALL_SIZE)
  {
    insist ((unsigned)BIG_SIZE >= GM_DMA_ROUNDUP (u32, length + routeLength));
    q = &bigBuffers;
    qs = &bigSends;
    c = &bigTokens;
  }
  else
  {
    q = &smallBuffers;
    qs = &smallSends;
    c = &smallTokens;
  }

  int n;
  
  if (!(n = q->get (&m)))
    return 0;  

  insist (m && n >= (int) GM_DMA_ROUNDUP (u32, length + routeLength));

  char *t;
  t = route->press ((char*)m);
  memcpy (t, p, length);

  if (*c)
  {
    insist (m);
    _gm_raw_send (gmPort, m, length + routeLength, routeLength);
    (*c)--;
  }
  else
  {  
    int a = qs->put (m, length, routeLength);
    insist (a);
  }
  return 1;
  exception: return 0;

}


int mt_Gm::sendWithPadding (mt_Job*job, mt_Route*route, char*p, int length, int totalLength)
{
  insist (job == this->job);
  insist (route);
  insist (p);
  insist (length >= 0);
  insist (totalLength >= 0);
  exception: return 0;
}

void mt_Gm::reclaimSend (void*p)
{
  insist (this);
  insist (p);

  int r;
  r = 0;

  if (smallStart <= p && p < smallEnd)
  {
    smallTokens++;
    r = smallBuffers.put (p, SMALL_SIZE);
    insist (smallTokens > 0 && smallTokens <= NUM_SMALL_TOKENS);
  }
  else if (bigStart <= p && p < bigEnd)
  {
    bigTokens++;
    r = bigBuffers.put (p, BIG_SIZE);
    insist (bigTokens > 0 && bigTokens <= NUM_BIG_TOKENS);
  }

  insist (r);
  exception: return;
}

void mt_Gm::flushSends ()
{
  void *p;
  int length;
  int routeLength;
  int index;

  while (smallTokens && (length = smallSends.get (&p, &routeLength)))
  {
    insist (p);
    insist (routeLength >= 0 && routeLength <= mt_Route::MAX_ROUTE);
    
    _gm_raw_send (gmPort, p, length + routeLength, routeLength);
    smallTokens--;
  }
  while (bigTokens && (length = bigSends.get (&p, &routeLength, &index)))
  {
    insist (p);
    insist (routeLength >= 0 && routeLength <= mt_Route::MAX_ROUTE);
    
    _gm_raw_send (gmPort, p, length + routeLength, routeLength);
    bigTokens--;
  }
  exception: return;
}

void mt_Gm::freePacket (char*m)
{
  delete [] m;
}

int mt_Gm::receivePacket (int isRaw, void*buffer, void*mp, int mlength, char**p, int*length)
{
  insist (this);
  insist (mlength && buffer && mp);
  insist (p && length);
  insist (this->job);

  char*t;
  t = 0;
  *p = 0;
  *length = 0;
  
  /*have to give the buffer back, so make a copy*/
  
  if (mlength < (int) sizeof (mt_Message))
    numBadLengths++;
  else
  {
    t = new char [mlength];
    insist (t);
    memcpy (t, mp, mlength);
  }
  
  if (isRaw)
  {
    //memset (buffer, 0, SMALL_SIZE);
    _gm_provide_raw_receive_buffer (gmPort, buffer);
  }
  else
  {
    insist (0);
    gm_provide_receive_buffer (gmPort, buffer , BIG_GM_SIZE, GM_LOW_PRIORITY);
  }
  
  if (mlength < (int)sizeof (mt_Message))
    return 0;

  *p = t;
  *length = mlength;
  return 1;
  
  exception: return 0;
}

int mt_Gm::receive (char**p, int*length)
{
  insist (this);
  
  gm_up_n_t *k;
  *p = 0;
  *length = 0;
  
  gm_recv_event_t*event;
#ifdef BUSY_WAIT
#warning BUSY_WAITing
  event = gm_receive (gmPort);
#else
  event = gm_blocking_receive_no_spin (gmPort);
#endif
  
  //  printFormat ("event %d", GM_RECV_EVENT_TYPE (event));
      
  switch (GM_RECV_EVENT_TYPE (event))
  {
    case GM_SENT_EVENT:
      sentDoneLength = 0;
      for (k = (gm_up_n_t *) gm_ntohp (event->sent.message_list);
	   gm_ntohp (*k);
	   k++)
      {
	reclaimSend (gm_ntohp (*k));
	sentDoneLength++;
      }
      flushSends ();
      return mt_Network::SEND_DONE;
    case GM_RAW_RECV_EVENT:
      if (!receivePacket (1, gm_ntohp (event->recv.buffer),
			  gm_ntohp (event->recv.buffer),
			  gm_ntoh_u32 (event->recv.length),
			  p, length))
	return 0;
      return mt_Network::RECEIVE;
      break;
    case GM_FAST_RECV_EVENT:
    case GM_FAST_HIGH_RECV_EVENT:
    case GM_FAST_PEER_RECV_EVENT:
    case GM_FAST_HIGH_PEER_RECV_EVENT:
      if (!receivePacket (0, gm_ntohp (event->recv.buffer),
			  gm_ntohp (event->recv.message),
			  gm_ntoh_u32 (event->recv.length),
			  p, length))
	return 0;
      return mt_Network::RECEIVE;
    case GM_RECV_EVENT:
    case GM_HIGH_RECV_EVENT:
    case GM_PEER_RECV_EVENT:
    case GM_HIGH_PEER_RECV_EVENT:
      if (!receivePacket (0, gm_ntohp (event->recv.buffer),
			  gm_ntohp (event->recv.buffer),
			  gm_ntoh_u32 (event->recv.length),
			  p, length))
	return 0;
      return mt_Network::RECEIVE;
    case GM_NO_RECV_EVENT:    
      flushSends ();
      return 0;
    case GM_ALARM_EVENT:
      alarmed = 0;
      gm_unknown (gmPort, event);
      if (alarmed)
      {
	alarmed = 0;
	return mt_Network::TIMEOUT;
      }
      return 0;
  case GM_BAD_SEND_DETECTED_EVENT:
  case GM_SEND_TOKEN_VIOLATION_EVENT:
    {
      printFormat ("bad sent event");
      return 0;
    }
    break;
    
    default:
      
      gm_unknown (gmPort, event);
      return 0;
  }
  exception: return 0;
}


int mt_Gm::wait (char**p, int*length)
{
  insist (this);
  insist (p && length);
  int event;
  
  while (!(event = receive (p, length)));

  return event;
  exception: return 0;
}

static void alarmHook (void*p)
{
  ((mt_Gm*)p)->alarmed = 1;
}

void mt_Gm::setTimer (mt_Job*job, int microseconds)
{
  insist (this);
  insist (job == this->job);
  gm_set_alarm (gmPort, &gmAlarm, microseconds, alarmHook, this);
  exception: return;
}

void mt_Gm::clearTimer (mt_Job*job)
{
  insist (this);
  insist (job == this->job);
  gm_cancel_alarm (&gmAlarm);
  exception: return;
}

void mt_Gm::setRoute (mt_Job*job, int gmId, mt_Route*route)
{
  insist (this);
  insist (job == this->job);
  
  insist (gmId >= 0);
  {
    int length = route->getLength ();
    char buffer [mt_Route::MAX_ROUTE];
    mt_Route::physical (route->getHops (), buffer, length);
  
    int r = _gm_set_route (gmPort, gmId, length, buffer);

    if (r != GM_SUCCESS)
      printFormat ("_gm_set_route () failed. GM IDs might be too high.");
    
    insist (r == GM_SUCCESS);
  }
  
  exception: return;
}

void mt_Gm::setAddress (mt_Job*job, int gmId, mt_Address*address)
{
  insist (this);
  insist (job && address && gmId > 0);
  insist (job == this->job);
  
  int r;
  r = _gm_set_unique_id (gmPort, (unsigned) gmId, address->getBytes ());
  
  if (r != GM_SUCCESS)
    printFormat ("_gm_set_unique_id () failed. GM IDs might be too high.");
    
  insist (r == GM_SUCCESS);
  exception: return;
}

void mt_Gm::setHostname (mt_Job*job, int gmId, char*hostname)
{
  insist (this);
  insist (job);
  insist (gmId >= 0);
  insist (hostname);
  insist (job == this->job);
  
  
  int r;
  r = _gm_set_host_name (gmPort, gmId, hostname);

  if (r != GM_SUCCESS)
    printFormat ("_gm_set_host_name () failed. GM IDs might be too high.");

  insist (r == GM_SUCCESS);
  exception: return;
}

void mt_Gm::setGmId (mt_Job*job, int gmId)
{
  insist (this);
  insist (gmId > 0);
  insist (job == this->job);
  
  _gm_set_node_id (gmPort, gmId);
  exception: return;
}

int mt_Gm::getGmId (mt_Job*job)
{
  insist (this);
  insist (job);
  insist (job == this->job);
  
  unsigned int id;
  
  if (gm_get_node_id (gmPort, &id) != GM_SUCCESS)
    return 0;
  
  return id;
  exception: return 0;
}

void mt_Gm::clearRoutes (mt_Job*)
{
  insist (this);
  
  int r;
  r = _gm_clear_all_routes (gmPort);
  insist (r == GM_SUCCESS);

  exception: return;
}

mt_Int64 mt_Gm::getCurrentTime ()
{
  return gm_ticks (gmPort);
}

mt_Graph*mt_Gm::getGraph ()
{
  return 0;
}

int mt_Gm::getRoute (int, int, int, mt_Route*)
{
  insist (0);
  exception: return 0;
}

int mt_Gm::getNumRoutes (int, int)
{
  insist (0);
  exception: return 0;
}

int mt_Gm::parseArgs (mt_Args*args)
{
  insist (this);
  insist (args);
  exception: return 1;
}
void mt_Gm::usage ()
{
}
void mt_Gm::dump ()
{
}

void mt_Gm::run ()
{
  insist (0);
  exception: return;
}

void mt_Gm::setLevel (mt_Job*job, int level)
{
  insist (this);
  insist (job == this->job);
  insist (level >= 0);
  
  _gm_set_mapper_level (gmPort, level);
  
  exception: return;
}

mt_Network*mt_getNetwork ()
{
  return new mt_Gm ();
}
int mt_Gm::getNumReceiveBuffers ()
{
  return numReceiveBuffers;
}
int mt_Gm::getReceiveMtu ()
{
  return SMALL_SIZE;
}

int mt_Gm::writeRegister (unsigned int offset, unsigned int value)
{
#if GM_OS_VXWORKS
  printFormat ("mt_GM::writeRegister -- Not implemented for vxworks!");
  return GM_SUCCESS;
#else /* !GM_OS_VXWORKS */
  return _gm_user_write_lanai_register (gmPort, offset, value) == GM_SUCCESS;
#endif /* GM_OS_VXWORKS */
}

int mt_Gm::enableCrc ()
{
  int r;
  insist (this);

  r = _gm_cop_wakeup (gmPort);

  insist (r == GM_SUCCESS);
  
  r = _gm_cop_send (gmPort, 0x2F);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0x1a);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0x00);
  insist (r == GM_SUCCESS);
  r = _gm_cop_end (gmPort);
  insist (r == GM_SUCCESS);

  r = _gm_cop_wakeup (gmPort);
  insist (r == GM_SUCCESS);

  r = _gm_cop_send (gmPort, 0x22);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0xC7);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0xf0);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0x0f);
  insist (r == GM_SUCCESS);
  r = _gm_cop_end (gmPort);
  insist (r == GM_SUCCESS);

  return 1;
  exception: return 0;
}

int mt_Gm::disableCrc ()
{
  int r;
  insist (this);
 
  r = _gm_cop_wakeup (gmPort);
  insist (r == GM_SUCCESS);
  
  r = _gm_cop_send (gmPort, 0x2F);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0x1a);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0x10);
  insist (r == GM_SUCCESS);
  r = _gm_cop_end (gmPort);
  insist (r == GM_SUCCESS);

  r = _gm_cop_wakeup (gmPort);
  insist (r == GM_SUCCESS);

  r = _gm_cop_send (gmPort, 0x22);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0xC7);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0xc0);
  insist (r == GM_SUCCESS);
  r = _gm_cop_send (gmPort, 0x0f);
  insist (r == GM_SUCCESS);
  r = _gm_cop_end (gmPort);
  insist (r == GM_SUCCESS);

  return 1;
  exception: return 0;
}

int mt_Gm::getSentDoneLength ()
{
  return sentDoneLength;
}
