/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

#include "gm_types.h"

#define	LAN7_SIG_BIT	0x00800000
#define	LAN6_SIG_BIT	0x00400000
#define	LAN5_SIG_BIT	0x00200000
#define	LAN4_SIG_BIT	0x00100000
#define	LAN3_SIG_BIT	0x00080000
#define	LAN2_SIG_BIT	0x00040000
#define	LAN1_SIG_BIT	0x00020000
#define	LAN0_SIG_BIT	0x00010000
#define WORD_RDY_BIT    0x00008000
#define HALF_RDY_BIT    0x00004000
#define SEND_RDY_BIT    0x00002000
#define LINK_INT_BIT	0x00001000
#define NRES_INT_BIT	0x00000800
#define WAKE_INT_BIT	0x00000400
#define OFF_BY_2_BIT	0x00000200
#define OFF_BY_1_BIT	0x00000100
#define TAIL_INT_BIT    0x00000080
#define WDOG_INT_BIT	0x00000040
#define TIME_INT_BIT	0x00000020
#define DMA_INT_BIT	0x00000010
#define SEND_INT_BIT 	0x00000008
#define BUFF_INT_BIT 	0x00000004
#define RECV_INT_BIT 	0x00000002
#define BYTE_RDY_BIT    0x00000001

enum GM_STATE_bits
{
  /* 0x00000001 */ FREE_RECV_CHUNK = BYTE_RDY_BIT,
/* 0x00000002 *//* BUFF_INT_BIT */
/* 0x00000004 *//* RECV_INT_BIT */
/* 0x00000008 *//* SEND_INT_BIT */
/* 0x00000010 *//* DMA_INT_BIT  */
/* 0x00000020 *//* TIME_INT_BIT */
  /* 0x00000040 */ FREE_SEND_CHUNK = WDOG_INT_BIT,
  /* 0x00000080 */ ACK_PENDING = TAIL_INT_BIT,
  /* 0x00000100 */ RDMA_PENDING = OFF_BY_1_BIT,
  /* 0x00000200 */ RDMAING = OFF_BY_2_BIT,
  /* 0x00000400 */ RECEIVING = WAKE_INT_BIT,
  /* 0x00000800 */ SENDING = NRES_INT_BIT,
  /* 0x00001000 */ SDMAING = LINK_INT_BIT,
  /* 0x00002000 */ SEND_PENDING = SEND_RDY_BIT,
  /* 0x00004000 */ SDMA_PENDING = HALF_RDY_BIT,
};

#define BIGGEST_STATE_BIT	SDMA_PENDING

enum gm_event_index
{
  POLL_EVENT = 0,

  START_SDMA_EVENT = 4,
  FINISH_SDMA_EVENT = 8,

  START_SEND_EVENT = 12,
  FINISH_SEND_EVENT = 16,
  SEND_ACK_EVENT = 20,

  START_RECV_MESSAGE_EVENT = 24,
  FINISH_RECV_MESSAGE_EVENT = 28,
  RECV_BUFFER_OVERFLOW_EVENT = 32,

  START_RDMA_EVENT = 36,
  FINISH_RDMA_EVENT = 40,

  TIMER_EVENT = 44,

  FAIR_RDMA_EVENT = 48,
  FAIR_SDMA_EVENT = 52,
  FAIR_SDMA_RDMA_EVENT = 56,
};
char *event_name[] = {
  "POLL_EVENT",

  "START_SDMA_EVENT",
  "FINISH_SDMA_EVENT",

  "START_SEND_EVENT",
  "FINISH_SEND_EVENT",
  "SEND_ACK_EVENT",

  "START_RECV_MESSAGE_EVENT",
  "FINISH_RECV_MESSAGE_EVENT",
  "RECV_BUFFER_OVERFLOW_EVENT",

  "START_RDMA_EVENT",
  "FINISH_RDMA_EVENT",

  "TIMER_EVENT",

  "FAIR_RDMA_EVENT",
  "FAIR_SDMA_EVENT",
  "FAIR_SDMA_RDMA_EVENT",
  0
};

#define NUM_EVENT_TYPES 15

extern gm_lanai_globals_t *lg;

void
verify (unsigned i, unsigned value)
{
  if (lg->event_index_table[i] != value)
    {
      printf ("lg->event_index_table[0x%x] != %s", i, event_name[value / 4]);
      fflush (stdout);
    }
}

void
test_tab ()
{
  unsigned i;

  for (i = 0; i < (BIGGEST_STATE_BIT << 1); i++)
    {
      /* Highest priority so queues will be rewound when needed.
         Timer events happen only when no acks are pending. */

      if (i & TIME_INT_BIT)
	{
	  verify (i, lg->event_index_table[i] == TIMER_EVENT);
	  continue;
	}

      if (i & DMA_INT_BIT && i & RDMAING)
	{
	  verify (i, lg->event_index_table[i] == FINISH_RDMA_EVENT);
	  continue;
	}

      /* receive events */

      if (i & RECV_INT_BIT && i & RECEIVING)
	{
	  verify (i, lg->event_index_table[i] == FINISH_RECV_MESSAGE_EVENT);
	  continue;
	}
      if (i & BUFF_INT_BIT && i & RECEIVING)
	{
	  verify (i, lg->event_index_table[i] == RECV_BUFFER_OVERFLOW_EVENT);
	  continue;
	}
      if (~i & RECEIVING && i & FREE_RECV_CHUNK)
	{
	  verify (i, lg->event_index_table[i] == START_RECV_MESSAGE_EVENT);
	  continue;
	}

      /* send events */

      if (i & SEND_INT_BIT && i & SENDING)
	{
	  verify (i, lg->event_index_table[i] == FINISH_SEND_EVENT);
	  continue;
	}
      if (i & SEND_INT_BIT && ~i & SENDING && i & ACK_PENDING)
	{
	  verify (i, lg->event_index_table[i] == SEND_ACK_EVENT);
	  continue;
	}
      if (i & SEND_INT_BIT && ~i & SENDING && i & SEND_PENDING)
	{
	  verify (i, lg->event_index_table[i] == START_SEND_EVENT);
	  continue;
	}

      /* SDMA completion events */

      if (i & DMA_INT_BIT && i & SDMAING)
	{
	  verify (i, lg->event_index_table[i] == FINISH_SDMA_EVENT);
	  continue;
	}

      /* Interlocked SDMA/RDMA state machine fairness events */

      {
	int bits;

	bits == 0;

	if (i & DMA_INT_BIT
	    && ~i & SDMAING
	    && ~i & RDMAING && i & FREE_SEND_CHUNK && i & SDMA_PENDING)
	  bits |= SDMA_PENDING;

	if (i & DMA_INT_BIT
	    && ~i & SDMAING && ~i & RDMAING && i & RDMA_PENDING)
	  bits |= RDMA_PENDING;

	switch (bits)
	  {
	  case (SDMA_PENDING + RDMA_PENDING):
	    verify (i, lg->event_index_table[i] == FAIR_SDMA_RDMA_EVENT);
	    continue;
	  case (SDMA_PENDING):
	    verify (i, lg->event_index_table[i] == FAIR_SDMA_EVENT);
	    continue;
	  case (RDMA_PENDING):
	    verify (i, lg->event_index_table[i] == FAIR_RDMA_EVENT);
	    continue;
	  case (0):
	    verify (i, lg->event_index_table[i] == POLL_EVENT);
	    continue;
	  default:
	    gm_assert (0);
	  }
      }
    }
}

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
