/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file implements the extremely limited amount of file system
   support required to make the freeBSD printf() code work in the
   LANai. */

#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/errno.h>

#include "gm_compiler.h"
#include "gmcp.h"

/****************
 * errno
 ****************/

int errno;

/****************
 * isatty
 ****************/

int
isatty (int fd)
{
  GM_PARAMETER_MAY_BE_UNUSED (fd);
  
  /* We don't support ttys. */
  
  return 0;
}

/****************
 * fstat
 ****************/

int
fstat (int fildes, struct stat *buf)
{
  GM_PARAMETER_MAY_BE_UNUSED (fildes);
  GM_PARAMETER_MAY_BE_UNUSED (buf);
  
  /* We don't have a filesystem. */

  errno = ENOENT;
  return -1;
}

/****************
 * read
 ****************/

ssize_t
read (int fildes, void *buf, size_t nbyte)
{
  GM_PARAMETER_MAY_BE_UNUSED (fildes);
  GM_PARAMETER_MAY_BE_UNUSED (buf);
  GM_PARAMETER_MAY_BE_UNUSED (nbyte);
  
  errno = EBADF;
  return -1;
}

/****************
 * lseek
 ****************/

off_t
lseek (int fildes, off_t offset, int whence)
{
  GM_PARAMETER_MAY_BE_UNUSED (fildes);
  GM_PARAMETER_MAY_BE_UNUSED (offset);
  GM_PARAMETER_MAY_BE_UNUSED (whence);
  
  errno = ESPIPE;
  return (off_t) - 1;
}

/****************
 * write
 ****************/

ssize_t
write (int fildes, const void *buf, size_t nbyte)
{
  if (fildes != 1)
    {
      errno = EBADF;
      return -1;
    }

  prepare_to_interrupt ("printf");
  gm.interrupt.write.length = nbyte;
  gm.interrupt.write.buffer = (gm_lp_t) buf;
  gm_interrupt (GM_WRITE_INTERRUPT);

  /* block until the data has been printed */

  await_interrupt_completion ("printf wait");
  return nbyte;
}

/****************
 * close
 ****************/

int
close (int fildes)
{
  GM_PARAMETER_MAY_BE_UNUSED (fildes);
  
  errno = EIO;
  return -1;
}

/****************
 * getpagesize
 ****************/

int
getpagesize ()
{
  return GM_PAGE_LEN;
}

/****************
 * sbrk
 ****************/

char *
sbrk (int incr)
{
  static char *end;

  if (!end)
    end = gm.malloc_ram;

  if ((char *) gm.malloc_ram <= end + incr
      && end + incr <= (char *) gm.malloc_ram + sizeof (gm.malloc_ram))
    {
      char *ret;

      ret = end;
      end += incr;
      return ret;
    }

  errno = ENOMEM;
  return (char *) -1;
}

/****************************************************************
 * math hooks
 ****************************************************************/

long long
__negdi2 (long long x)
{
  union
  {
    unsigned long long whole;
    struct
    {
      unsigned long high;
      unsigned long low;
    }
    parts;
  }
  a, ret;

  a.whole = x;
  ret.parts.low = ~a.parts.low + 1;
  ret.parts.high = ~a.parts.high;
  if (ret.parts.low < ~a.parts.low)
    ret.parts.high += 1;
  return ret.whole;
}

unsigned long
__udivsi3 (unsigned long a, unsigned long b)
{
  unsigned long i, m, l = 1, q = 0;

  for (i = 0; i <= 64; i++)
    {
      if (b << i > a)
	break;
    }

  /* perform long division */

  do
    {
      m = b << i;
      if (m <= a)
	{
	  a -= m;
	  q += l << i;
	}
    }
  while (i--);

  return (q);
}

long
__divsi3 (long a, long b)
{
  if (a < 0)
    {
      if (b < 0)
	return __udivsi3 (-a, -b);
      else
	return -__udivsi3 (-a, b);
    }
  else
    {
      if (b < 0)
	return -__udivsi3 (a, -b);
      else
	return __udivsi3 (a, b);
    }
}

/****************************************************************
 * double word support
 ****************************************************************/

static unsigned long long
di_udivmod (unsigned long long a, unsigned long long b,
	    unsigned long long *remainder)
{
  unsigned long long i, m, l = 1ULL, q = 0;

  /* count the bits in b */

  for (i = 0; i <= 64; i++)
    {
      if (b << i > a)
	break;
    }

  /* perform long division */

  do
    {
      m = b << i;
      if (m <= a)
	{
	  a -= m;
	  q += l << i;
	}
    }
  while (i--);

  if (remainder)
    *remainder = a;
  return (q);
}

unsigned long long
__udivdi3 (unsigned long long a, unsigned long long b)
{
  return di_udivmod (a, b, 0);
}

unsigned long long
__umoddi3 (unsigned long long a, unsigned long long b)
{
  unsigned long long ret;

  di_udivmod (a, b, &ret);
  return ret;
}

long long
__divdi3 (long long a, long long b)
{
  if (a < 0)
    {
      if (b < 0)
	return __udivdi3 (-a, -b);
      else
	return -__udivdi3 (-a, b);
    }
  else
    {
      if (b < 0)
	return -__udivdi3 (a, -b);
      else
	return __udivdi3 (a, b);
    }
}

unsigned long long
__ashldi3 (unsigned long long _a, unsigned int shift)
{
  union
  {
    unsigned long long whole;
    struct
    {
      unsigned long high;
      unsigned long low;
    }
    parts;
  }
  a, ret;

  a.whole = _a;

  if (shift == 0)
    return a.whole;
  else if (shift < 32U)
    {
      ret.parts.low = a.parts.low << shift;
      ret.parts.high = a.parts.high << shift | a.parts.low >> (32U - shift);
    }
  else if (shift == 32U)
    {
      ret.parts.low = 0;
      ret.parts.high = a.parts.low;
    }
  else
    {
      ret.parts.low = 0;
      ret.parts.high = a.parts.low << (shift - 32U);
    }
  return ret.whole;
}

unsigned long long
__lshrdi3 (unsigned long long _a, unsigned int shift)
{
  union
  {
    unsigned long long whole;
    struct
    {
      unsigned long high;
      unsigned long low;
    }
    parts;
  }
  a, ret;

  a.whole = _a;

  if (shift == 0)
    return a.whole;
  else if (shift < 32U)
    {
      ret.parts.high = a.parts.high >> shift;
      ret.parts.low = a.parts.low >> shift | a.parts.high << (32U - shift);
    }
  else if (shift == 32U)
    {
      ret.parts.high = 0;
      ret.parts.low = a.parts.high;
    }
  else
    {
      ret.parts.high = 0;
      ret.parts.low = a.parts.high >> (shift - 32U);
    }
  return ret.whole;
}

/****************************************************************
 * Misc
 ****************************************************************/

char *
memcpy (char *s, const char *ct, unsigned int n)
{
  char *a = s;
  const char *b = ct;
  unsigned int i;

  for (i = 0; i < n; i++)
    *a++ = *b++;
  return s;
}

unsigned int
__umulsi3 (unsigned int a, unsigned int b)
{
  register unsigned int result = 0;
  do
    {
      if (a & 1)
	{
	  result += b;
	}
      a >>= 1;
      b <<= 1;
    }
  while (a);
  return result;
}

int
__mulsi3 (int a, int b)
{
  if (a < 0)
    {
      if (b < 0)
	{
	  return (__umulsi3 (-a, -b));
	}
      else
	{
	  return (-__umulsi3 (-a, b));
	}
    }
  else
    {
      if (b < 0)
	{
	  return (-__umulsi3 (a, -b));
	}
      else
	{
	  return (__umulsi3 (a, b));
	}
    }
}

void
__main (void)
{
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
