/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

#if L4
#define OVERRUN(isr) (((isr)>>8)&3)
#elif L5 | L6 | L7 | L8 | L9
#define OVERRUN(isr) (((isr)>>8)&7)
#else
#error bogus GM_LANAI_MAJOR_VERSION
#endif


/***********************
** Recv state machine **
***********************/

				/* Handle receiving a message that fits */
				/* entirely within a chunk. */
MARK_LABEL (L_recv__got_chunk_, LZERO)
{
  const gm_packet_header_t *p;
  char *rmp;
  unsigned int bytes_received;
  GM_CRC_TYPE crc;
#if GM_ENABLE_CRC32
  gm_u8_t *crc32;
#endif
  void *_RMP;
  gm_u32_t isr;
  enum gm_packet_type p__type;
  enum gm_packet_subtype p__subtype;
  unsigned int p__sender_node_id;
  unsigned int max_node_id;

  GM_LOG_EVT (GM_GOT_CHUNK);
  gm_assert (NOTICED (FREE_RECV_CHUNK));
  gm_assert (NOTICED (RECEIVING));
  gm_assert (get_ISR () & RECV_INT_BIT);
  ASSERT_HANDLER (FINISH_RECV_PACKET_EVENT, L_recv__got_chunk_, LZERO);
  ASSERT_HANDLER (RECV_BUFFER_OVERFLOW_EVENT, L_recv__discard_overflow_,
		  LZERO);

  gm_puts ("gm_recv: received a packet.\n");
  GM_INCR_PACKET_CNT (gm.netrecv_cnt);

  isr = get_ISR ();
  /* pre */ _RMP = RMP;
  p = &gm.recv_chunk[LZERO].packet.as_gm.header;
  p__type = p->type;
  rmp = (char *) _RMP - OVERRUN (isr) - sizeof (GM_CRC_TYPE);
#if GM_ENABLE_CRC32
  crc32 = rmp - 4;
#endif
  crc = *(GM_CRC_TYPE *) rmp;
  bytes_received = rmp - (char *) p;
  p__sender_node_id = p->sender_node_id;
  max_node_id = gm.max_node_id;

  /* Check 8-bit CRC, not assuming that it will be aligned on any
     particular boundary. */

  gm_assert (ORUN2_INT_BIT == 0x200);
  gm_assert (ORUN1_INT_BIT == 0x100);
  if (crc
#if GM_ENABLE_CRC32
      || (((unsigned int) crc32) & 0x3) || *(int *) crc32
#endif
    )
    {
      /*
#warning feldy
      gm_printf("BAD CRC  bytesrecv=%d crc = 0x%02x  Pcrc32=%p crc32 = 0x%08x",
                bytes_received, (unsigned char)crc, crc32,
	        *(unsigned int *)crc32);
      fflush(stdout);
      gm_hex_dump(p,((char *)RMP-(char *)p)+8);
      fflush(stdout);
      */

      handle_bad_crc (p, bytes_received);
      RMP = (void *) p;
      set_RML (&gm.recv_chunk[LZERO].end);
      DISPATCH (42, "bad CRC received.");
    }
#if 0
#warning nelson debug stuff
     {
       int i;
       gm_printf("Received a packet: \n");
       for (i=0; i < 32; i++ ) {
	 gm_printf("%02x ", gm.recv_chunk[LZERO].packet.as_bytes[i] & 0xff );
	 if( i % 16 == 15 )
	   {
	     gm_printf("\n");
	   }
       }
       gm_printf("\n");
       fflush(stdout);  
     }
#endif

  switch (p__type)
    {
      /*********************
       * GM packet
       *********************/
    case GM_PACKET_TYPE:
      {
	unsigned int p__target_node_id;
	unsigned int gm_this_node_id;

	p__target_node_id = p->target_node_id;
	gm_this_node_id = gm.this_node_id;
	p__subtype = p->subtype;

	/* Check the legitimacy of the GM header */
	if ((bytes_received < sizeof (gm_packet_header_t))	/* length */
	    || (gm_galvantech_check_header_checksum (p) != GM_SUCCESS)
	    || (p__sender_node_id > max_node_id)	/* compatibility */
	  )
	  {
	    /* Drop the packet. */
	    RMP = (void *) p;
	    set_RML (&gm.recv_chunk[LZERO].end);
	    GM_INCR_ERROR_CNT (gm.drop_cnt);
	    GM_INCR_ERROR_CNT (gm.bad_header_cnt);

	    DISPATCH (43, "bad header detected");
	  }

	/* Verify that the packet was routed to the correct node */
	if (p__target_node_id != gm_this_node_id)
	  {
	    /* record the routing error for the connection */
	    if (++gm_connection[p__sender_node_id].misrouted_packet_error_cnt
		== 0)
	      gm_connection[p__sender_node_id].misrouted_packet_error_cnt =
		~0;
	    /* Drop the packet */
	    RMP = (void *) p;
	    set_RML (&gm.recv_chunk[LZERO].end);
	    GM_INCR_ERROR_CNT (gm.misrouted_cnt);
	    GM_INCR_ERROR_CNT (gm.drop_cnt);

	    gm_printf_p ("Misrouted GM packet.\n");
	    DISPATCH (44, "misrouted packet detected");
	  }
      }

      /* BAD: should use dispatch table here */
      /* since we know the bounds checking */
      /* is not needed. */
      switch (p__subtype)
	{
	  /********************
	   * GM_NACK_CLOSE_CONNECTION_TYPE
	   ********************/

	case GM_NACK_CLOSE_CONNECTION_SUBTYPE:
	  {
	    gm_connection_t *c;
	    c = &gm_connection[p__sender_node_id];
	    GM_INCR_ERROR_CNT (gm.nack_received_cnt);
	    GM_INCR_ERROR_CNT (gm.nack_receive_close_connection_cnt);
	    /* Ignore duplicate close requests. */
	    if ((rtc64 () - (c->close_time + GM_CONNECTION_TIMEOUT)) > 0)
	      {
		c->close_time = rtc64 ();
		/* Record new sexno to use, indicating connection is
		   closed */
		c->send_sexno.parts.sesno = 0;
		c->send_sexno.parts.seqno = 1;
		rewind_connection (c);
	      }
	    else
	      {
		GM_INCR_ERROR_CNT (gm.nack_ignore_close_connection_cnt);
		GM_INCR_ERROR_CNT (gm.drop_cnt);
	      }
	    /* Reuse the receive buffer */
	    RMP = (void *) p;
	    set_RML (&gm.recv_chunk[LZERO].end);
	    DISPATCH (45, "handled connection close");
	  }

	  /********************
	   * GM_NACK_OPEN_CONNECTION_TYPE
	   ********************/

	case GM_NACK_OPEN_CONNECTION_SUBTYPE:
	  {
	    gm_connection_t *c;

	    GM_INCR_ERROR_CNT (gm.nack_received_cnt);
	    GM_INCR_ERROR_CNT (gm.nack_receive_open_connection_cnt);
	    c = &gm_connection[p__sender_node_id];
	    /* open a connection, if not already open */
	    if (c->send_sexno.parts.sesno == 0)
	      {
		/* Record new sexno to use */
		c->send_sexno.whole = p->sexno.whole;
		rewind_connection (c);
	      }
	    else
	      {
		GM_INCR_ERROR_CNT (gm.nack_ignore_open_connection_cnt);
		GM_INCR_ERROR_CNT (gm.drop_cnt);
	      }

	    /* Reuse the receive buffer */
	    RMP = (void *) p;
	    set_RML (&gm.recv_chunk[LZERO].end);
	    DISPATCH (46, "handled connection open");
	  }

	  /*********************
	   * (N)ACK packet
	   *********************/
	case GM_ACK_SUBTYPE:
	case GM_NACK_SUBTYPE:
	case GM_NACK_DOWN_SUBTYPE:
	case GM_NACK_REJECT_SUBTYPE:
	  {
	    /* Packet is a (n)ack. */
	    unsigned int acked_seqno;
	    gm_s32_t acked_sexno;
	    gm_ack_packet_t *ap;
	    gm_send_record_t *sent;
	    gm_connection_t *c;
	    gm_s32_t delay_until;
	    gm_s32_t rtc;

	    ap = (gm_ack_packet_t *) p;

	    if (ap->subtype != GM_ACK_SUBTYPE)
	      {
		GM_INCR_ERROR_CNT (gm.nack_received_cnt);
	      }

	    c = &gm_connection[p__sender_node_id];
	    rtc = RTC;

	    gm_assert (p__sender_node_id <= max_node_id);

#if GM_DEBUG
	    if (p__subtype == GM_ACK_SUBTYPE)
	      gm_printf_p (" Got ACK (%d,%d).\n",
			   ap->sexno.parts.sesno, ap->sexno.parts.seqno);
	    else
	      {
		gm_printf_p (" Got a NACK (%d,%d).\n",
			     ap->sexno.parts.sesno, ap->sexno.parts.seqno);
	      }
#endif


	    acked_seqno = ap->sexno.parts.seqno;
	    acked_sexno = ap->sexno.whole;

	    /* note that the connection is alive. */
	    c->known_alive_time = rtc;

	    /* Retransmit after a delay between 0 */
	    /* and the specified number of */
	    /* microseconds. */
	    delay_until = gmcp_rand_mod (ap->usecs_delay) + rtc;

	    /* Now that we have read all the info */
	    /* from the packet, start receiving */
	    /* the next packet in the same buffer */
	    RMP = (void *) p;
	    GM_STBAR ();
	    set_RML (&gm.recv_chunk[LZERO].end);

	    sent = c->first_send_record;

	    /* If there is no packet to be freed */
	    /* or resent by the ack, make an early */
	    /* exit. */
	    if (!sent)
	      {
		gm_printf_p ("none to (n)ack.\n");
		DISPATCH (47, "(N)ACK procesed (none to ack)");
	      }

	    /*** Process acked send records, if any */

	    if (SEQ_CMP16 (sent->sexno.parts.seqno, acked_seqno) < 0)
	      {
		/* Below in this conditional, SENT points to an ACKed
		   send record, and NEXT points to the next send
		   record. */

		gm_send_record_t *next = sent->next;
		gm_subport_t *sp;

		do
		  {
		    sp = sent->send_token->common.subport;

		    /* Record when to next send for the acked subport */

		    sp->delay_until = delay_until;

		    /* Note that progress was made for the subport. */

		    sp->progress_time = rtc;

		    /* Report the finished send if this was the last
		       send for a packet */
		    if (sent->before_len <= GM_MTU)
		      {
			unsigned sp_id;
			gm_send_token_t *st;

			st = sent->send_token;
			gm_assert (st);
			gm_assert (st->common.subport);

			sp_id = sp->id;
			gm_assert (st->common.type != GM_ST_DATAGRAM);
			remove_first_send_token_from_send_queue (st, sp_id);

			pass_sent_token_to_port_and_free
			  (st, &gm_port[GM_SUBPORT_PORT (sp_id)],
			   0, GM_SUCCESS);
		      }
#if GM_DEBUG
		    sent->send_token = 0;
#endif

		    /* If acked all sent packets for the connection... */
		    if (!next)
		      {
			/* ... then we're done. */

			/* Move the acked send records from the send
			   record list to the free send record list. */
			sent->next = gm.free_send_records;
			gm.free_send_records = c->first_send_record;
			c->first_send_record = 0;

			gm_printf_p ("(n)acked all.\n");
			DISPATCH (48, "(N)ACKed all");
		      }

		    if (SEQ_CMP16 (next->sexno.parts.seqno, acked_seqno) >= 0)
		      break;

		    sent = next;
		    next = sent->next;
		  }
		while (1);

		/* Move the acked send records from
		   the send record list to the free
		   send record list. */
		sent->next = gm.free_send_records;
		gm.free_send_records = c->first_send_record;
		c->first_send_record = next;

		/* If the packet was an ack, we're done. */
		if (p__subtype == GM_ACK_SUBTYPE)
		  {
		    gm_printf_p ("Acked some.\n");
		    DISPATCH (49, "ACKed some, but not all");
		  }

		sent = next;
	      }
	    else
	      {
		/* No packet was acked. */
		/* If the packet was an ack, we're done. */
		if (p__subtype == GM_ACK_SUBTYPE)
		  {
		    gm_printf_p ("Acked none.\n");
		    DISPATCH (50, "ACKed none");
		  }
		if (SEQ_CMP16 (sent->sexno.parts.seqno, acked_seqno) != 0)
		  {
		    GM_INCR_ERROR_CNT (gm.nack_ignored_cnt);
		    gm_printf_p ("Stale NACK.\n");
		    DISPATCH (51, "ACKed none (stale NACK)");
		  }
	      }

	    /************
	     * NACKs
	     ************/

	    /* If the session number of the NACK is less than or equal
	       to the current send session number, ignore the nack,
	       since we've already rewound the send queue
	       appropriately, and doing so again (if the sesno's are equal)
	       can result in protocol confusion. */

	    if (SEQ_CMP16 (acked_sexno >> 16, c->send_sexno.whole >> 16) <= 0)
	      {
		GM_INCR_ERROR_CNT (gm.nack_ignored_cnt);
		DISPATCH (52, "NACK ignored (stale sesno)");
	      }

	    /* BAD: The following ignores the distinction between the
	       different sorts of NACKS.  For GM_NACK_DOWN_TYPE and
	       GM_NACK_REJECT_TYPE, the failed sends should be
	       returned to the sender, not retransmitted. */

	    /* The packet was a nack. */
	    gm_assert (p__subtype == GM_NACK_SUBTYPE
		       || p__subtype == GM_NACK_DOWN_SUBTYPE
		       || p__subtype == GM_NACK_REJECT_SUBTYPE);

	    /* From here on, SENT points to a send to be NACKED */

	    /* There are unacked sends for the connection, all of
	       which are to be NACKED.  SENT points to the send record
	       for the first of them. */
	    gm_assert (sent);
	    gm_assert (SEQ_CMP16 (sent->sexno.parts.seqno, acked_seqno) == 0);

	    /* Delay the nacked subport by the specified amount */
#if 0
	    if ((delay_until - RTC) > 0)
	      {
#endif
		sent->send_token->common.subport->delay_until = delay_until;
		gm_printf_p ("Delaying subport %d.\n",
			     sent->send_token->common.subport->id);
#if 0
	      }
#endif

	    gm_assert ((acked_sexno & 0xffff)
		       == ((int) acked_seqno & 0xffff));

	    /* Start resending with the expected sequence number
	       specified by the receiver. */

	    c->send_sexno.whole = acked_sexno;

	    gm_assert (sent);

	    {
	      gm_send_token_t *st;
	      gm_subport_t *sp;
	      gm_port_protected_lanai_side_t *port;
	      unsigned int port__open;
	      gm_s32_t progt;

	      /* Rewind the send queue for the connection and compute port */

	      st = sent->send_token;
	      gm_assert (st);
	      rewind_send_tokens (sent);
	      sp = st->ackable.subport;
	      gm_assert (sp);
	      port = &gm_port[GM_SUBPORT_PORT (sp->id)];

	      /* pre */ progt = sp->progress_time;
	      /* pre */ port__open = port->open;
	      /* pre */ rtc = RTC;


	      if (p__subtype == GM_NACK_SUBTYPE
		  && (SEQ_CMP32 ((progt + GM_FATAL_SEND_TIMEOUT), rtc) > 0)
		  && port__open)
		{
		  /* Handle a normal NACK for open port, unless a timout
		     has occurred. */

		  GM_INCR_ERROR_CNT (gm.nack_normal_cnt);

		  /* fairly allow the next subport to be the first to send. */
		  c->first_active_send_port = sp->next;
		}
	      else if (!port->enable_nack_down_flag &&
			(p__subtype == GM_NACK_DOWN_SUBTYPE)
		  	&& (SEQ_CMP32 ((progt + GM_FATAL_SEND_TIMEOUT), rtc) > 0)
		  	&& port__open)
		{
		  /* Handle a normal NACK for open port, unless a timout
		     has occurred. */

		  GM_INCR_ERROR_CNT (gm.nack_down_cnt);

		  /* fairly allow the next subport to be the first to send. */
		  c->first_active_send_port = sp->next;
		}
	      else
		{
		  /* Received some sort of NACK indicating an error. */
		  /* report the error to the user */
#if 0
		  gm_printf ("(0x%x + 0x%x)=0x%x > 0x%x cmp=%d open=%d\n",
			     progt, GM_FATAL_SEND_TIMEOUT,
			     (progt + GM_FATAL_SEND_TIMEOUT), rtc,
			     (SEQ_CMP32 ((progt + GM_FATAL_SEND_TIMEOUT),
					 rtc)
			      > 0),
			     port__open);
		  fflush (stdout);
#endif
		  handle_send_error
		    (sp, (p__subtype == GM_NACK_REJECT_SUBTYPE
			  ? GM_SEND_REJECTED
			  : p__subtype == GM_NACK_DOWN_SUBTYPE
			  ? GM_SEND_TARGET_PORT_CLOSED : GM_SEND_TIMED_OUT),
			(int) (c - gm.connection));
		}

	      /* Free all the send records for the connection. */

	      c->last_send_record->next = gm.free_send_records;
	      gm.free_send_records = c->first_send_record;
	      c->first_send_record = 0;

	      gm_printf_p ("Requeued nacked packets.\n");
	      DISPATCH (53, "NACKed some");
	    }
	  }
	  gm_always_assert (0);	/* never get here */

#if GM_ENABLE_DATAGRAMS
	  /*********************
	   * Datagrams
	   *********************/
	case GM_DATAGRAM_SUBTYPE_0...GM_DATAGRAM_SUBTYPE_31:
	  /* Fall through */
#endif
#if GM_ENABLE_DIRECTED_SEND
	  /*********************
	   * Reliable directed sends
	   *********************/
	case GM_DIRECTED_DATA_SUBTYPE:
	case GM_DIRECTED_HEAD_DATA_SUBTYPE:
	case GM_DIRECTED_BODY_DATA_SUBTYPE:
	case GM_DIRECTED_TAIL_DATA_SUBTYPE:
	  /* Fall through */
#endif
	  /*********************
	   * Reliable ordered delivery
	   *********************/
	case GM_RELIABLE_DATA_SUBTYPE_0...GM_RELIABLE_DATA_SUBTYPE_31:
	  case
	GM_RELIABLE_HEAD_DATA_SUBTYPE_13...GM_RELIABLE_HEAD_DATA_SUBTYPE_31:
	case GM_RELIABLE_BODY_DATA_SUBTYPE:
	case GM_RELIABLE_TAIL_DATA_SUBTYPE:

	  gm_assert (p__sender_node_id <= max_node_id);
	  gm_assert (GM_SUBPORT_PRIORITY (p->target_subport_id)
		     <= GM_MAX_PRIORITY);

	  /* Verify packet was not truncated */

	  if (bytes_received < sizeof (gm_packet_header_t) + p->length)
	    {
	      /* Drop the packet. */
	      RMP = (void *) p;
	      set_RML (&gm.recv_chunk[LZERO].end);
	      GM_INCR_ERROR_CNT (gm.short_packet_cnt);
	      GM_INCR_ERROR_CNT (gm.drop_cnt);

	      gm_printf_p ("Bad length for GM data packet.\n");
	      DISPATCH (54, "Bad length for GM receive");
	    }

#ifndef START_DEFAULT
#define START_DEFAULT(dispatch)						\
	  if (--gm.free_recv_chunk_cnt)					\
	    {								\
	      RMP = gm.recv_chunk[LONE].packet.as_bytes;		\
	      set_RML (&gm.recv_chunk[LONE].end);			\
	      gm_assert (NOTICED (RECEIVING));			\
	      NOTICE (RDMA_PENDING);					\
	      SET_HANDLER (FINISH_RECV_PACKET_EVENT,			\
			   L_recv__got_chunk_, LONE);			\
	      SET_HANDLER (RECV_BUFFER_OVERFLOW_EVENT,			\
			   L_recv__discard_overflow_, LONE);		\
	      gm_printf_p						\
		("Received monolithic packet. "				\
		 "Started next receive.\n");				\
	      dispatch;							\
	    }								\
									\
	  NOTICE (RDMA_PENDING);					\
	  NOTICE_NOT (FREE_RECV_CHUNK | RECEIVING);			\
	  SET_HANDLER (START_RECV_PACKET_EVENT,				\
		       L_recv__start_receiving_chunk_, LONE);		\
	  gm_printf_p							\
	    ("Received monolithic packet. "				\
	     "No more free recv buffs.\n");
#endif /* START_DEFAULT */

	  START_DEFAULT
	    (DISPATCH (55, "received packet (out of recv chunks)"));
	     DISPATCH (56, "received packet (more recv chunks)");

	  /****************
	   * Drop unrecognized GM packet subtypes.
	   ****************/

	  /* Unrecognized subtypes may be received if some nodes on
	     the network support GM extensions that are not supported
	     by other nodes on the netowrk.  For example, if a datagram
	     is sent to a node without datagram support. */

	default:
	  gm_printf_p ("received unrecognized subtype %d\n",
		       (int) p__subtype);

	  RMP = (void *) p;
	  set_RML (&gm.recv_chunk[LZERO].end);
	  GM_INCR_ERROR_CNT (gm.drop_cnt);
	  DISPATCH (57, "unrecognized GM subtype dropped");
	}


    case GM_MAPPING_PACKET_TYPE:
      /*********************
       * GM mapping packet.
       *********************/
      {
	gm_mapper_packet_t *mp;

	/* Drop mapping packets that are too short to have a valid
	   subtype. */
	if (bytes_received < 4)
	  {
	    /* Drop the packet. */
	    RMP = (void *) p;
	    set_RML (&gm.recv_chunk[LZERO].end);
	    GM_INCR_ERROR_CNT (gm.short_mapper_packet_cnt);
	    GM_INCR_ERROR_CNT (gm.drop_cnt);

	    gm_printf_p ("truncated mapper packet\n");
	    DISPATCH (58, "truncated mapper packet dropped");
	  }

	mp = (gm_mapper_packet_t *) p;
	switch (mp->common.subtype)
	  {
	    /****
	     * config packets
	     ****/

	  case GM_MAPPER_CONFIG_PACKET_SUBTYPE:

	    gm_assert (*(volatile gm_s16_t *) &mp->common.subtype
		       == GM_MAPPER_CONFIG_PACKET_SUBTYPE);

	    /* Drop truncated, out-of-order, and misrouted packets */
	    if (bytes_received < sizeof (gm_mapper_config_packet_t)
		|| (gm.this_node_id
		    && (SEQ_CMP32 (mp->config.map_version,
				   (gm.mapper_state.scout_reply.packet.
				    map_version)) < 0)
		    && mp->config.map_version)
		|| ether_addr_cmp (gm.mapper_state.scout_reply.packet.address,
				   mp->config.address))
	      {

		/* Queue a reply to the config packet, in case the
		   previous reply was lost. */
		queue_mapper_config_reply (&gm.mapper_state.config_reply, mp);
		/* Drop the packet. */
		RMP = (void *) p;
		set_RML (&gm.recv_chunk[LZERO].end);
		GM_INCR_ERROR_CNT (gm.short_mapper_config_packet_cnt);
		GM_INCR_ERROR_CNT (gm.drop_cnt);

		gm_printf_p ("truncated mapper config packet\n");
		DISPATCH (59, "truncated mapper config packet dropped");
	      }

	    gm_assert (*(volatile gm_s16_t *) &mp->common.subtype
		       == GM_MAPPER_CONFIG_PACKET_SUBTYPE);

	    /* Update the map version number (stored in the mapper
	       reply packets) as specified in the config packet. */

	    /* If the map version changed, or the map version is zero,
	       or the mapper node changed,
	       record the new version and clear all routes */

	    if ((gm.mapper_state.scout_reply.packet.map_version
		 != mp->config.map_version)
		|| (!mp->config.map_version)
		|| ether_addr_cmp (mp->config.mapper_address,
				   (gm.mapper_state.scout_reply.packet.
				    mapper_address)))
	      {
		gm.mapper_state.scout_reply.packet.map_version
		  = mp->config.map_version;
		{
		  int index;
		  for (index = 0; index < 6; index++)
		    {
		      gm.mapper_state.scout_reply.packet.
			mapper_address[index] =
			gm.mapper_state.unique_id[index] =
			mp->config.mapper_address[index];
		    }
		}

		/* clear our count of the number of new routes seen.
		   this count is used to tell us when we have seen all
		   the routes for this config, so we know when to
		   clear any old routes */

#if GM_CLEAR_ROUTES_BEFORE_CONFIGURING

		prepare_to_interrupt ("gm_recv.h: clear_tables interrupt \n");
		gm_interrupt (GM_CLEAR_TABLES_INTERRUPT);
		/* no need to wait until interrupt completes, we have
                   lots of work to do */
		for (c = &gm_connection[0];
		     c <= &gm_connection[gm.max_node_id]; c++)
		  {
		    GM_CONNECTION_CLEAR_ROUTE (c);
		  }
#else
		gm.mapper_state.config_reply.num_routes = 0;
		gm_mark_routes_as_retired ();
#endif
	      }

	    gm_assert (*(volatile gm_s16_t *) &mp->common.subtype
		       == GM_MAPPER_CONFIG_PACKET_SUBTYPE);

	    /* Update connection error counting state iff the node
	       ID has changed */

	    if (!gm.this_node_id && mp->config.id &&
			(gm.this_node_id != mp->config.id))
	      {
		gm_connection_t *c;

		gm.this_node_id
		  = gm.mapper_state.scout_reply.packet.gm_id = mp->config.id;
		for (c = &gm_connection[0];
		     c <= &gm_connection[gm.max_node_id]; c++)
		  {
		    c->probable_crc_error_cnt = 0;
		    c->misrouted_packet_error_cnt = 0;
		    c->ack_packet.sender_node_id = gm.this_node_id;
		  }
	      }

#if 0
	    gm_assert (*(volatile gm_s16_t *) &mp->common.subtype
		       == GM_MAPPER_CONFIG_PACKET_SUBTYPE);
#endif

#if GM_CLEAR_ROUTES_BEFORE_CONFIGURING

	    /* now make sure that the interrupt has completed */
	    _await_interrupt_completion
	      ("gm_recv.h: await finish of clear_tables interrupt \n");

#endif

	    /* Set all the routes as specified by the routing
	       packet */

	    if ((unsigned) mp->config.num_bytes < bytes_received)
	      {
		gm_u8_t *ptr;

		ptr = (gm_u8_t *) & mp->config.bytes[0];
		/*skip over return route length and route itself */
		ptr += 1 + *ptr;

		while (ptr < ((gm_u8_t *) & mp->config.bytes[0]
			      + mp->config.num_bytes))
		  {

		    gm_u16_t id;
		    gm_u16_t node_type;
		    gm_u8_t *address;
		    /*gm_u8_t name[GM_MAX_HOST_NAME_LEN + 1];*/
		    gm_u8_t name[GM_MAX_HOST_NAME_LEN + 1 + GM_DMA_GRANULARITY];
		    gm_u8_t *_name;
		    int len;
		    int num_routes;
		    gm_u8_t *r;
		    gm_connection_t *c;

		    /* get address */
		    address = ptr;
		    ptr += 6;

		    /* get id */
		    gm_bcopy (ptr, &id, 2);
		    ptr += 2;

		    /* get node type */
		    gm_bcopy (ptr, &node_type, 2);
		    ptr += 2;

		    /* skip control */
		    ptr += 2;

		    len = (*ptr <= GM_MAX_HOST_NAME_LEN
			   ? *ptr : GM_MAX_HOST_NAME_LEN);

		    /* get name */
		    /* name needs to be 64 bit aligned for the lanai 5
		       or the DMA of the name to the host won't work as
		       expected - nelson */
		    _name = (gm_u8_t *)GM_ROUNDUP(lp, name, GM_DMA_GRANULARITY);
		    gm_bcopy (ptr + 1, _name, len);
		    _name[len] = 0;
		    ptr += 1 + *ptr;

		    /* get num_routes */
		    num_routes = *ptr;
		    ptr++;

		    /* get first route length */
		    len = *ptr;
		    ptr++;

		    /* get first route */
		    r = ptr;
		    ptr += len;

		    if (num_routes == 0)
		      break;

		    /* skip over remaining routes */
		    while (--num_routes)
		      ptr += 1 + *ptr;

		    /* Verify ID not too big */
		    if (id > gm.max_node_id)
		      continue;

		    /* only accept route if it is for a host with the
		       same node type as us */

		    if (node_type
			== gm.mapper_state.scout_reply.packet.node_type)
		      {
			c = &gm_connection[id];

			/* bring route out of retirement */
			c->retired = 0;

#if !GM_CLEAR_ROUTES_BEFORE_CONFIGURING

			/* we have seen another new route */
			gm.mapper_state.config_reply.num_routes++;

#endif
			/* Record route length and copy the route */
			GM_CONNECTION_SET_ROUTE (c, len, r);

			/* Record the ethernet address for the GM ID */
			gm_ethernet_addr_table_set (id, address);

			/* Record the hostname for the GM ID */
			gm_name_table_set (id, _name);

			/* Reset the error statistics for the connection. */
			c->probable_crc_error_cnt = 0;
			c->misrouted_packet_error_cnt = 0;
			c->ack_packet.sender_node_id = gm.this_node_id;

			if (id > gm.max_node_id_inuse)
			  {
			    gm.max_node_id_inuse = id;
			    if (gm.max_node_id_inuse > gm.max_node_id)
			      {
				gm.max_node_id_inuse = gm.max_node_id;
			      }
			  }
		      }
		  }
	      }

#if !GM_CLEAR_ROUTES_BEFORE_CONFIGURING

	    if (gm.mapper_state.config_reply.num_routes
		>= (unsigned) mp->config.num_hosts)
	      {
		gm_clear_retired_routes ();
	      }
#endif
	    gm_assert (*(volatile gm_s16_t *) &mp->common.subtype
		       == GM_MAPPER_CONFIG_PACKET_SUBTYPE);
	    /* Queue a reply to the config packet. */
	    queue_mapper_config_reply (&gm.mapper_state.config_reply, mp);
	    /* Drop the processed packet (do not forward it to the host). */
	    RMP = (void *) p;
	    set_RML (&gm.recv_chunk[LZERO].end);
	    /* GM_INCR_ERROR_CNT (gm.drop_cnt); */

	    gm_printf_p ("Handled config packet.\n");
	    DISPATCH (60, "handled config packet");

	    /****
	     * Scout
	     ****/

	  case GM_MAPPER_SCOUT_PACKET_SUBTYPE:

	    if (bytes_received <
		GM_OFFSETOF (gm_mapper_scout_packet_t, extended_route))
	      {

		/* Drop the packet. */
		RMP = (void *) p;
		set_RML (&gm.recv_chunk[LZERO].end);
		GM_INCR_ERROR_CNT (gm.short_mapper_scout_packet_cnt);
		GM_INCR_ERROR_CNT (gm.drop_cnt);
		gm_printf_p ("truncated mapper scout packet\n");
		DISPATCH (61, "truncated mapper scout packet dropped");
	      }

	    /* Reply to the scout message */
	    gm_assert (*(volatile gm_s16_t *) &mp->common.subtype
		       == GM_MAPPER_SCOUT_PACKET_SUBTYPE);


	    {
	      gm_port_protected_lanai_side_t *port;
	      gm_u32_t p;
	      volatile gm_u32_t can_reset = 1;
	      gm_mapper_scout_reset_packet_t *mp_reset = 
					(gm_mapper_scout_reset_packet_t *)mp;

	      if (mp_reset->command == 0x0ded /* REBOOT_NODE */)
		{
		  /* this is a magic packet for harware customers */
		  /* to use to reboot a LANai node */
		  gm_mcp_reboot_node();
		  /* should never get here ?? */
		}
	      else if (mp_reset->command == 1 /* RESET */)
		{
		  for (p = 0; p < GM_NUM_PORTS; p++)
		    {
		      port = &gm_port[p];
		      if (port->open) 
			{
			  can_reset = 0;
			  break;
			}
		    }

		  if (can_reset) 
		    {
		      printf("mapper is resetting my gmID\n");
		      fflush(stdout);
		      gm.this_node_id
				= gm.mapper_state.scout_reply.packet.gm_id = 0;
		    }
		  else
		    {
		      /* Drop the packet. */

		      printf("Can NOT reset my gmID port=%d is open",p);
		      printf("Node will probably be unusable - id conflict");
		      fflush(stdout);
		    }
		}
	    }
	
	    queue_mapper_scout_reply (&gm.mapper_state.scout_reply, mp);

	    /* Pass the scout packet to the mapper. */
	    /* fall through */
	  default:
	  }
      }

      /* fall through */
    default:
      /***
       * Raw packets
       ***/

      /* Drop everything but mapping and ethernet packets. */

#ifdef  DELETE_ME
      if (p__type != GM_MAPPING_PACKET_TYPE
	  && p__type != GM_ETHERNET_PACKET_TYPE)
	{
	  RMP = (void *) p;
	  set_RML (&gm.recv_chunk[LZERO].end);
	  GM_INCR_ERROR_CNT (gm.bad_type_cnt);
	  GM_INCR_ERROR_CNT (gm.drop_cnt);
	  gm_printf_p ("drop everything not mapper or ethernet\n");
	  DISPATCH (62, "drop everything not mapper or ethernet");
	}
#endif /* DELETE_ME */

      /* All unrecognized packets are forwarded to the RDMA state
         machine in case it wants "raw" packets as well as recognized
         packets.  Raw packets are first hacked up such that their
         "type" field (when interpreted as a GM packet) is
         GM_RAW_HACK_TYPE, allowing the RDMA state machine to
         efficiently switch on the GM type field instead of performing a
         special check for non-GM packets. */

      gm.recv_chunk[LZERO].raw_type_HACK
	= gm.recv_chunk[LZERO].packet.as_gm.header.subtype;
      if (p__type == GM_ETHERNET_PACKET_TYPE)
	{
	  gm_puts ("gm_recv: Got an ethernet packet.\n");

	  gm.recv_chunk[LZERO].packet.as_gm.header.subtype
	    = GM_ETHERNET_HACK_SUBTYPE;
	}
      else
	gm.recv_chunk[LZERO].packet.as_gm.header.subtype =
	  GM_RAW_HACK_SUBTYPE;

      {
	/* Record the length of the raw receive. */
	gm.recv_chunk[LZERO].raw_length = bytes_received;

	/* Catch zero-length receives. */

	if (bytes_received <= 0)
	  {
	    gm_puts ("gm_recv: Caught zero-length receive.\n");

	    RMP = (void *) p;
	    set_RML (&gm.recv_chunk[LZERO].end);
	    GM_INCR_ERROR_CNT (gm.zero_len_cnt);
	    GM_INCR_ERROR_CNT (gm.drop_cnt);
	    DISPATCH (63, "zero-length recv packet dropped");
	  }
      }

      START_DEFAULT
	(DISPATCH (64, "raw packet received (no more recv chunks)"));
	 DISPATCH (65, "raw packet received (more recv chunks)");
    }
}
GM_END_HANDLER;

MARK_LABEL (L_recv__discard_overflow_, LZERO)
{
  gm_assert (NOTICED (FREE_RECV_CHUNK));
  gm_assert (NOTICED (RECEIVING));
  SET_HANDLER (FINISH_RECV_PACKET_EVENT, L_recv__done_discarding_overflow_,
	       LZERO);
  ASSERT_HANDLER (RECV_BUFFER_OVERFLOW_EVENT, L_recv__discard_overflow_,
		  LZERO);

  RMP = gm.recv_chunk[LZERO].packet.as_bytes;
  set_RML (&gm.recv_chunk[LZERO].end);
  gm_printf_p ("Discarded a chunk.\n");
  DISPATCH (66, "starting to drop overlarge recv (buff_int)");
}
GM_END_HANDLER;

MARK_LABEL (L_recv__done_discarding_overflow_, LZERO)
{
  gm_assert (NOTICED (FREE_RECV_CHUNK));
  gm_assert (NOTICED (RECEIVING));
  ASSERT_HANDLER (FINISH_RECV_PACKET_EVENT, L_recv__done_discarding_overflow_,
		  LZERO);
  ASSERT_HANDLER (RECV_BUFFER_OVERFLOW_EVENT, L_recv__discard_overflow_,
		  LZERO);

  RMP = gm.recv_chunk[LZERO].packet.as_bytes;
  set_RML (&gm.recv_chunk[LZERO].end);

  SET_HANDLER (FINISH_RECV_PACKET_EVENT, L_recv__got_chunk_, LZERO);
  gm_printf_p ("Discarded last chunk.\n");
  DISPATCH (67, "done dropping overlarge recv");
}
GM_END_HANDLER;

MARK_LABEL (L_recv__start_receiving_chunk_, LZERO)
{
  gm_assert (NOTICED (FREE_RECV_CHUNK));
  gm_assert (NOTICED_NOT (RECEIVING));

  RMP = gm.recv_chunk[LZERO].packet.as_bytes;
  set_RML (&gm.recv_chunk[LZERO].end);

  SET_HANDLER (FINISH_RECV_PACKET_EVENT, L_recv__got_chunk_, LZERO);
  SET_HANDLER (RECV_BUFFER_OVERFLOW_EVENT, L_recv__discard_overflow_, LZERO);

  NOTICE (RECEIVING);
  gm_printf_p ("Set up next recv.\n");
  DISPATCH (68, "set up recv (no longer out of recv chunks)");
}
GM_END_HANDLER;

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
