/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* This table implements LANai access to the page hash table.  The
   page hash table stores page dma addresses keyed on
   host-virtual-addr/port-number pairs.  This allows the interface to
   translate host virtual addresses to DMA addresses.

   The page table is large, requiring about 1% of host physical memory.
   Consequently, it is stored in the host where there is guaranteed to
   be enough memory for it.  Entries are cached in LANai SRAM in a much
   smaller hash table.  This cache stores the most recently used hash
   table entries.

   The page table may not be contiguous in DMA space.  Consequently,
   the lanai assumes the page table is broken into page-sized "pieces"
   and stores a pointer to each piece in the GM_PAGE_HASH_PIECE array,
   which is large enough to hold the worst case
   GM_MAX_NUM_HOST_HASH_TABLE_PIECES references, which is a power of
   two.  Although the host may allocate a much smaller number of
   pieces to hold the page hash table, it always allocates a power of
   two pieces and initializes the GM_PAGE_HASH_PIECE array to cyclicly
   refer to these pieces, such that the LANai can assume there are
   GM_MAX_NUM_HOST_HASH_TABLE_PIECES pieces.  This results in faster
   access for almost no cost.

   For example, if GM_MAX_NUM_HOST_HASH_TABLE_PIECES were 8 but there
   were only 2 host hash table pieces, the GM_PAGE_HASH_PIECE array
   would be initialized as follows:

    GM_PAGE_HASH_PIECE array
   +-----+-----+-----+-----+-----+-----+-----+-----+
   |     |     |     |     |     |     |     |     |
   +-----+-----+-----+-----+-----+-----+-----+-----+
      |     |     |     |     |     |     |     |
      |,----|-----'-----|-----'-----|-----'     |
      |     |           |           |           |
      |     `-----------`-----------`----------.|
      |                                         |
      V                                         V
     +-----------------------------------------+----------------- - - ---+
     |   first piece                           | second piece            |
     +-----------------------------------------+------------------ - - --+
       Host-side page hash table

   */

#include "gmcp.h"
#include "gm_debug.h"
#include "gm_debug_lanai_tlb.h"
#include "gm_enable_galvantech_workaround.h"
#include "gm_enable_lmru_cache.h"
#include "gm_types.h"
#include "gm_page_hash.h"
#include "gm_bootstrap.h"
#include "gm_lanai_def.h"
#include "gm_page_hash.h"
#include "gm_pte.h"
#include "gm_rand.h"

/* CSPI L5 hack */
#if L5

#ifdef DMA_INT_BIT
#error DMA_INT_BIT already defined?
#endif

#define DMA_INT_BIT (E2L_INT_BIT | L2E_INT_BIT)
#endif /*L5 */

#if GM_PAGE_LEN == 0
/* #error Should not build this file for MCP for host without VM support. */
#else /* GM_PAGE_LEN != 0 */

/* Flush the page hash table cache. This is called (via a host/lanai
   synchronization mechanism) any time pages are removed from the host
   page hash table to prevent DMAs to or from the removed pages. */
void
gm_init_page_table ()
{
  int i;
  for (i = 0; i <= GM_MAX_PAGE_HASH_CACHE_INDEX + 1; i++)
    gm.page_hash.cache.entry[i].pte.page_port = 0;
#if GM_ENABLE_LMRU_CACHE
  gm.page_hash.cache.root.younger
    = gm.page_hash.cache.root.older = &gm.page_hash.cache.root;
#endif
  gm.page_hash.cache.cnt = 0;
}

/* Load an uncached page hash table entry into the cache and return
   the dma address for the page.  If there is no matching table entry,
   return 0. */

static void
check_cache_integrity (int line)
{
  if (GM_DEBUG)
    {
      unsigned int i;
      unsigned int cnt;
      cnt = 0;
      for (i = 0; i <= GM_MAX_PAGE_HASH_CACHE_INDEX; i++)
	{
	  if (gm.page_hash.cache.entry[i].pte.page_port != 0)
	    {
	      cnt += 1;
	    }
	}
      /*  gm_always_assert (cnt == gm.page_hash.cache.cnt); */
      if (cnt != gm.page_hash.cache.cnt)
	{
	  gm_printf ("LANai PTE cache integrity has been comprised\n"
		     "cnt=%d gm.page_hash.cache.cnt=%d line=%d\n",
		     cnt, gm.page_hash.cache.cnt, line);
	  gm_always_assert (0);
	}
    }
}

#define CACHE_CHECK() check_cache_integrity(__LINE__)

gm_dp_t
get_uncached_dma_page_entry (gm_up_t page_port)
{
  gm_cached_pte_t *oldest, *entry;
  unsigned index;

  if (GM_DEBUG_PAGE_HASH)
    {
      gm_printf ("get_uncached_dma_page_entry(0x%qx) called.\n",
		 (gm_u64_t) page_port);
      fflush (stdout);
    }

  CACHE_CHECK ();
  /*** If needed, make room to load a new cache entry. */

  gm_assert (gm.page_hash.cache.root.older);
  gm_assert (gm.page_hash.cache.root.younger);
  gm_assert (gm.page_hash.cache.cnt <= GM_MAX_PAGE_HASH_CACHE_INDEX / 2);

  if (gm.page_hash.cache.cnt == GM_MAX_PAGE_HASH_CACHE_INDEX / 2)
    {
      gm.page_hash.cache.cnt -= 1;

#if GM_ENABLE_LMRU_CACHE
      /*** Remove the oldest cache entry */

      oldest = gm.page_hash.cache.root.younger;
      gm.page_hash.cache.root.younger = oldest->younger;
      gm.page_hash.cache.root.younger->older = &gm.page_hash.cache.root;
      oldest->pte.page_port = 0;
      entry = oldest + 1;
#else
      /*** Remove a random cache entry */
     if (GM_DEBUG_PAGE_HASH)
       {
         gm_printf ("get_uncached_dma_page_entry: Remove a random cache entry.\n");
       }

      do
	{
	  entry = &gm.page_hash.cache.entry[gmcp_rand ()
					    & GM_MAX_PAGE_HASH_CACHE_INDEX];
	}
      while (!entry->pte.page_port);
      entry->pte.page_port = 0;
      entry++;
#endif


      /*** Remove and reinsert all the immediately following entries
	in the hash table to remove any gap we may have introduced. */

      while (1)
	{
	  gm_cached_pte_t *new_entry;
	  gm_up_t entry__page_port;

	  entry__page_port = entry->pte.page_port;
	  if (!entry__page_port)
	    {
	      /* Handle wraparound and done cases */
	      if (entry
		  > &gm.page_hash.cache.entry[GM_MAX_PAGE_HASH_CACHE_INDEX])
		{
		  entry = &gm.page_hash.cache.entry[0];
		  continue;
		}
	      break;
	    }

      /*** Find out where this entry should be located now that we
	    have removed a preceeding entry. */

	  gm_assert (GM_POWER_OF_TWO (GM_MAX_PAGE_HASH_CACHE_INDEX + 1));
	  new_entry
	    = &gm.page_hash.cache.entry[(GM_HASH_PAGE_PORT (entry__page_port)
					 & GM_MAX_PAGE_HASH_CACHE_INDEX)];

	  /* Scan for an an available slot before the old location. */

	  while (1)
	    {
	      if (new_entry->pte.page_port == 0)
		{
		  if (new_entry >
		      &gm.page_hash.cache.entry[GM_MAX_PAGE_HASH_CACHE_INDEX])
		    {
		      new_entry = &gm.page_hash.cache.entry[0];
		      continue;
		    }
		  break;
		}
	      if (new_entry == entry)
		break;

	      new_entry++;
	    }

	  /* Move the entry to its new location, if one was found. */

	  if (new_entry != entry)
	    {
	      /* Copy hash table info */
	      new_entry->pte.page_port = entry__page_port;
	      new_entry->pte._packed_dma_addr = entry->pte._packed_dma_addr;
	      gm_assert (gm_pte_get_port_id (&new_entry->pte) < GM_NUM_PORTS);

	      /* Mark old entry as free */
	      entry->pte.page_port = 0;

#if GM_ENABLE_LMRU_CACHE
	      {
		gm_cached_pte_t *younger, *older;

		/* Update age queue to reflect the
		   moved entry. */
		younger = entry->younger;
		older = entry->older;
		younger->older = new_entry;
		older->younger = new_entry;
		new_entry->older = older;
		new_entry->younger = younger;
	      }
#endif
	    }

	  entry++;
	}
    }


    if (GM_DEBUG_PAGE_HASH)
      {
        gm_printf ("get_uncached_dma_page_entry: Compute table entry.\n");
      }

  /* Compute where to cache the table entry we are about to fetch from
     host memory.  (Ugly because optimized to generate fast assembly). */

  entry = (&gm.page_hash.cache.entry[(GM_HASH_PAGE_PORT (page_port)
				      & GM_MAX_PAGE_HASH_CACHE_INDEX)] - 1);
  while (1)
    {
      entry++;
      if (entry->pte.page_port != 0)
	continue;
      if (entry <= &gm.page_hash.cache.entry[GM_MAX_PAGE_HASH_CACHE_INDEX])
	break;
      entry = &gm.page_hash.cache.entry[0] - 1;
    }

  /*** DMA the hash entry from the host table, scanning for the right
       entry. */

  index = GM_HASH_PAGE_PORT (page_port) & GM_PAGE_HASH_MAX_INDEX;

  /* Wait for DMA engine to be available */
  while (!(get_ISR () & DMA_INT_BIT));

  do
    {
      gm_fill_pte_from_host (entry, index);

      gm_assert (gm_pte_get_port_id (&entry->pte) < GM_NUM_PORTS);

      if (entry->pte.page_port == 0)
	{
	  if (GM_DEBUG_PAGE_HASH)
	    {
	      gm_printf ("Could not find PTE.\n");
	      fflush (stdout);
	    }

	  if (GM_DEBUG_LANAI_TLB)
	    {
	      GM_PANIC (("index=0x%x, page_port=0x%qx\n",
			 index, (gm_u64_t) page_port));
	    }
	  return 0;
	}

      index = (index + 1) & GM_PAGE_HASH_MAX_INDEX;
    }
  while (entry->pte.page_port != page_port);

  /* gm_puts ("Caching a new PTE.\n"); */

#if GM_ENABLE_LMRU_CACHE
  /*** Add the new entry to the age list as the youngest entry. */

  gm_assert (gm.page_hash.cache.root.older);
  gm_assert (gm.page_hash.cache.root.younger);
  entry->older = gm.page_hash.cache.root.older;
  entry->younger = &gm.page_hash.cache.root;
  entry->older->younger = entry;
  gm.page_hash.cache.root.older = entry;
#endif

  /*** record that we added a new entry */

  gm.page_hash.cache.cnt++;

  gm_assert (gm_pte_get_dma_page (&entry->pte));

  {
    gm_dp_t ret;

    ret = gm_pte_get_dma_page (&entry->pte);
    if (GM_DEBUG_PAGE_HASH)
      {
	gm_printf ("get_uncached_dma_page_entry() returning 0x%qx\n",
		   (gm_u64_t) ret);
	fflush (stdout);
      }
    return ret;
  }
}

#endif /* GM_PAGE_LEN != 0 */

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
